﻿#include "ntd-test-libc.h"
#include <errno.h>
#include <stddef.h>
#include <stdio.h>
#include <stdlib.h>
#include <dirent.h>
#include <unistd.h>
#include <sys/stat.h>

#if __NX__
#define tempdirname "host:/dirent_list"
static const char *files[] = {
    "host:/dirent_list/file_a",
    "host:/dirent_list/file_b",
    "host:/dirent_list/file_c",
    "host:/dirent_list/file_1",
    "host:/dirent_list/file_2",
    "host:/dirent_list/file_10"
};
#else
#define tempdirname "dirent_list"
static const char *files[] = {
    "dirent_list/file_a",
    "dirent_list/file_b",
    "dirent_list/file_c",
    "dirent_list/file_1",
    "dirent_list/file_2",
    "dirent_list/file_10"
};
#endif

static const char *file_names[] = {
    "file_1",
    "file_10",
    "file_2",
    "file_a",
    "file_b",
    "file_c"
};

static const char *file_names_ver[] = {
    "file_a",
    "file_b",
    "file_c",
    "file_1",
    "file_2",
    "file_10"
};

static void setup_test_list()
{
    int result = mkdir(tempdirname, 0777);
    TESTCASE_MESSAGE((result == 0 || (result == -1 && errno == EEXIST)),
        "mkdir('%s') result = %d errno = %d", tempdirname, result, errno);
    int i;
    for (i = 0; i < sizeof(files) / sizeof(files[0]); i++) {
        FILE *f = fopen(files[i],"w");
        TESTCASE(f != NULL);
        if (f != NULL) {
            int j;
            for (j = 0; j < 1000; j++) {
                fprintf(f, "%04d:%08x\n", j, (i << 16) + j);
            }
            fclose(f);
        }
    }
}

static void test_list (const char *name)
{
    DIR *dirp;
    struct dirent *entp;
    int matched_files = 0;
    int num_files = sizeof(file_names) / sizeof(file_names[0]);

    dirp = opendir (name);
    TESTCASE(dirp != NULL);
    if (dirp == NULL) {
        perror ("opendir");
        return;
    }

    errno = 0;
    do {
        entp = readdir(dirp);
        if (entp != NULL) {
            if (entp->d_name[0] != '.') {
                /* test if this is one of the expected files */
                int i;
                for (i = 0; i < num_files; i++) {
                    if (strcmp(entp->d_name, file_names[i]) == 0) {
                        matched_files++;
                    }
                }
            }
        }
    } while (entp != NULL);
    TESTCASE_MESSAGE(matched_files == num_files, "matched_files = %d, num_files = %d", matched_files, num_files);
    TESTCASE(errno == 0);
    if (errno) perror ("readdir");

    int result = closedir(dirp);
    TESTCASE(result == 0);
    if (result < 0) perror ("closedir");
}

static struct dirent **release_namelist(struct dirent **namelist, int count)
{
    int i;

    if (namelist == NULL) return NULL;
    for (i = 0; i < count; i++) {
        if (namelist[i] != NULL) {
            free(namelist[i]);
            namelist[i] = NULL;
        }
    }
    free(namelist);
    return NULL;
}

static int filter_fileprefix(const struct dirent *dirent)
{
    if (strncmp(dirent->d_name, "file_", 5) == 0) return 1;
    return 0;
}

// register namelist for leaks checking
static void register_namelist(struct dirent **namelist, int count)
{
    int i;
    NTD_TEST_REGISTER_ALLOCATION(namelist);
    /* read all files sorted by alphasort */
    for (i = 0; i < count; i++) {
        NTD_TEST_REGISTER_ALLOCATION(namelist[i]);

    }

}

static void test_scandir(const char *name)
{
    int i;
    int j;
    struct dirent **namelist = NULL;
    int num_files = (sizeof(file_names) / sizeof(file_names[0]));
    j = 0;
    int count = scandir(name, &namelist, NULL, alphasort);
    TESTCASE(count > 0);
    TESTCASE(namelist != NULL);
    register_namelist(namelist, count);

    /* read all files sorted by alphasort */
    for (i = 0; i < count; i++) {
        TESTCASE(namelist[i] != NULL);
        if (namelist[i]->d_name[0] != '.') {
            TESTCASE_MESSAGE(strcmp(namelist[i]->d_name, file_names[j]) == 0,
                "got '%s', expected '%s'", namelist[i]->d_name, file_names[j]);
            j++;
        }
    }
    namelist = release_namelist(namelist, count);

    /* read files with prefix "file_" sorted by alphasort */
    count = scandir(name, &namelist, filter_fileprefix, alphasort);
    TESTCASE(count == num_files);
    TESTCASE(namelist != NULL);
    register_namelist(namelist, count);
    for (i = 0; i < count; i++) {
        TESTCASE(namelist[i] != NULL);
        if (namelist[i]->d_name[0] != '.') {
            TESTCASE_MESSAGE(strcmp(namelist[i]->d_name, file_names[i]) == 0,
                "%d: got '%s', expected '%s'", i, namelist[i]->d_name, file_names[i]);
        }
    }
    namelist = release_namelist(namelist, count);

    /* read files with prefix "file_" sorted by versionsort */
    count = scandir(name, &namelist, filter_fileprefix, versionsort);
    TESTCASE(count == num_files);
    TESTCASE(namelist != NULL);
    register_namelist(namelist, count);
    for (i = 0; i < count; i++) {
        TESTCASE(namelist[i] != NULL);
        if (namelist[i]->d_name[0] != '.') {
            TESTCASE_MESSAGE(strcmp(namelist[i]->d_name, file_names_ver[i]) == 0,
                "%d: got '%s', expected '%s'", i, namelist[i]->d_name, file_names_ver[i]);
        }
    }
    namelist = release_namelist(namelist, count);

}

void ntd_dirent_list(void)
{
    setup_test_list();
    test_list(tempdirname);
    test_scandir(tempdirname);
}
