﻿/*******************************************************************************

  This file contains various tests for the function rename()

*******************************************************************************/

#include "ntd_extended_test.h"
#include <errno.h>
#include <string.h>
#include <sys/stat.h>

static void test_written_content_remains_after_rename()
{
    FILE *file;
    int rename_ret_val; /* Return value from rename() */
    const char *filename;
    const char *new_filename;
    const char *fwrite_text = "text written to file";
    char *fread_buffer;
    filename = HOST_FILENAME("rename_before.txt");
    new_filename = HOST_FILENAME("rename_after.txt");

    DELETE_IF_EXISTS(filename);
    DELETE_IF_EXISTS(new_filename);

    CREATE_FILE_TEST(filename, fwrite_text);

    rename_ret_val = rename(filename, new_filename);
    TESTCASE_MESSAGE(rename_ret_val == 0, "rename(\"%s\", \"%s\") returned %d expected 0",
        filename, new_filename, rename_ret_val);

    /* test that attempting to rename over an existing file is successful */
    CREATE_FILE_TEST(filename, fwrite_text);
    rename_ret_val = rename(filename, new_filename);
    HORIZON_KNOWN_FAILURE_MESSAGE("RYNDA-329", rename_ret_val == 0,
        "rename(\"%s\", \"%s\") returned %d expected 0 (override existing file) errno = %d (%s)",
        filename, new_filename, rename_ret_val, errno, strerror(errno));

    file = FOPEN_TEST(new_filename, "r");
    FREAD_FILE_TEST(&fread_buffer, file);
    TESTCASE_STRINGS_MATCH(fread_buffer, fwrite_text);
    FCLOSE_TEST(file, new_filename);
    free(fread_buffer);
    file = NULL;
}

static void test_open_old_filename()
{
    FILE *file;
    int rename_ret_val; /* Return value from rename() */
    const char *filename;
    const char *new_filename;
#if __NX__
    filename = "host:/rename_open_old.txt";
    new_filename = "host:/rename_open_new.txt";
#else
    filename = "rename_open_old.txt";
    new_filename = "rename_open_new.txt";
#endif /* if __NX__ */

    file = FOPEN_TEST(filename, "w");
    FCLOSE_TEST(file, filename);
    file = NULL;

    rename_ret_val = rename(filename, new_filename);
    TESTCASE_MESSAGE(rename_ret_val == 0, "rename(\"%s\", \"%s\") returned %d expected 0",
        filename, new_filename, rename_ret_val);

    file = fopen(filename, "r");
    TESTCASE_MESSAGE(file == NULL, "fopen(\"%s\", \"r\") returned %p expected NULL",
        filename, file);
    FCLOSE_TEST(file, filename);
    file = NULL;

    /* Try to open the file with the new filename. Should succeed */
    file = FOPEN_TEST(new_filename, "r");
    FCLOSE_DELETE_TEST(file, new_filename);
    file = NULL;
}

static void test_delete_old_filename()
{
    FILE *file;
    int fileio_ret_val; /* Return value from rename() and remove() */
    const char *filename;
    const char *new_filename;
#if __NX__
    filename = "host:/rename.c_test_delete_old_filename.txt";
    new_filename = "host:/rename.c_test_delete_old_filename--renamed.txt";
#else
    filename = "rename.c_test_delete_old_filename.txt";
    new_filename = "rename.c_test_delete_old_filename--renamed.txt";
#endif /* if __NX__ */

    file = FOPEN_TEST(filename, "w");
    FCLOSE_TEST(file, filename);
    file = NULL;

    fileio_ret_val = rename(filename, new_filename);
    TESTCASE_MESSAGE(fileio_ret_val == 0, "rename(\"%s\", \"%s\") returned %d expected 0",
        filename, new_filename, fileio_ret_val);

    fileio_ret_val = remove(filename);
    TESTCASE_MESSAGE(fileio_ret_val != 0, "remove(\"%s\",) returned 0 expected failure",
        filename);

    fileio_ret_val = remove(new_filename);
    TESTCASE_MESSAGE(fileio_ret_val == 0, "remove(\"%s\",) returned %d expected 0",
        filename, fileio_ret_val);
}

static void test_same_names()
{
    FILE *file;
    int rename_ret_val; /* Return value from rename() */
    const char *filename;
#if __NX__
    filename = "host:/same_names.txt";
#else
    filename = "same_names.txt";
#endif /* if __NX__ */

    file = FOPEN_TEST(filename, "w");
    FCLOSE_TEST(file, filename);
    file = NULL;

    rename_ret_val = rename(filename, filename);
    HORIZON_KNOWN_FAILURE_MESSAGE("RYNDA-301", rename_ret_val == 0,
        "rename('%s', '%s') returned %d, expected 0",
        filename, filename, rename_ret_val);

    file = FOPEN_TEST(filename, "r");
    FCLOSE_DELETE_TEST(file, filename);
    file = NULL;
}

static void test_multiple_calls()
{
    FILE *file;
    int rename_ret_val; /* Return value from rename() */
    const char *filename1;
    const char *filename2;
    const char *filename3;
#if __NX__
    filename1 = "host:/subsequent_rename_1.txt";
    filename2 = "host:/subsequent_rename_2.txt";
    filename3 = "host:/subsequent_rename_3.txt";
#else
    filename1 = "subsequent_rename_1.txt";
    filename2 = "subsequent_rename_2.txt";
    filename3 = "subsequent_rename_3.txt";
#endif /* if __NX__ */

    file = FOPEN_TEST(filename1, "w");
    FCLOSE_TEST(file, filename1);
    file = NULL;

    rename_ret_val = rename(filename1, filename2);
    TESTCASE_MESSAGE(rename_ret_val == 0, "rename('%s', '%s') returned %d, expected 0",
        filename1, filename2, rename_ret_val);

    file = FOPEN_TEST(filename2, "w");
    FCLOSE_TEST(file, filename2);
    file = NULL;

    rename_ret_val = rename(filename2, filename3);
    TESTCASE_MESSAGE(rename_ret_val == 0, "rename('%s', '%s') returned %d, expected 0",
        filename2, filename3, rename_ret_val);

    /* fopen("r") name2 (should fail) */
    file = fopen(filename2, "r");
    TESTCASE_MESSAGE(file == NULL, "fopen(\"%s\", \"r\") returned %p expected NULL",
        filename2, file);
    FCLOSE_TEST(file, filename2);
    file = NULL;

    /* fopen("r") name3 (should succeed) */
    file = FOPEN_TEST(filename3, "r");
    FCLOSE_DELETE_TEST(file, filename3);
    file = NULL;
}

static void test_rename_non_existent_file()
{
    FILE *file;
    int rename_ret_val; /* Return value from rename() */
    const char *filename;
    const char *new_filename;
#if __NX__
    filename = "host:/rename_non_existent_file.txt";
    new_filename = "host:/non_existent_file_after.txt";
#else
    filename = "rename_non_existent_file.txt";
    new_filename = "rename_non_existent_file.txt";
#endif /* if __NX__ */

    /* Make sure the file doesn't exist */
    DELETE_IF_EXISTS(filename);

    rename_ret_val = rename(filename, new_filename);
    TESTCASE_MESSAGE(rename_ret_val != 0, "rename('%s', '%s') returned 0 expected failure",
        filename, new_filename);
}

static void test_rename_directory()
{
    FILE *file;
    int rename_ret_val; /* Return value from rename() */
    const char *dirname;
    const char *new_dirname;
#if __NX__
    dirname = "host:/test_rename_dir";
    new_dirname = "host:/test_rename_dir_new";
    mkdir("host:/test_rename_dir", 0666);
#else
    dirname = "test_rename_dir";
    new_dirname = "test_rename_dir_new";
    mkdir("test_rename_dir", 0666);
#endif /* if __NX__ */

    /* Make sure the new directory doesn't exist */
    DELETE_IF_EXISTS(new_dirname);

    rename_ret_val = rename(dirname, new_dirname);
    TESTCASE_MESSAGE(rename_ret_val == 0, "rename('%s', '%s') returned %d expected 0 errno = %d (%s)",
        dirname, new_dirname, rename_ret_val, errno, strerror(errno));
}

int ntd_extended_rename(void)
{
    NTD_TEST_GROUP_START("rename", 2);
    test_written_content_remains_after_rename();
    test_open_old_filename();
    test_delete_old_filename();
    test_same_names();
    test_multiple_calls();
    test_rename_non_existent_file();
    test_rename_directory();
    return NTD_TEST_GROUP_END("rename", 2);
}
