﻿/*******************************************************************************

  This file contains a series of tests for freopen()

  This file doesn't test using freopen() to redirect stdout to a file
  (see function 'test_redirect_stdout()' or the freopen() sample code on
   cplusplus.com) since doing so breaks printf() and t_error()

*******************************************************************************/

#include "ntd_extended_test.h"
#include <string.h>
#include <errno.h>
#include <stdlib.h>

static void test_all_modes_helper(const char *mode)
{
    FILE *fopen_ptr;
    FILE *freopen_ptr;
    const char *filename;
    filename = HOST_FILENAME("freopenAMH.txt");

    fopen_ptr = FOPEN_TEST(filename, "w");
    freopen_ptr = FREOPEN_TEST(filename, mode, fopen_ptr);
    FCLOSE_DELETE_TEST(freopen_ptr, filename);
}

/* Test passing all modes into freopen() in 2 ways: Calling freopen() on the
   same fopen_ptr, and calling it on separate, newly-created pointers via
   the helper function above */
static void test_all_modes()
{
    FILE *fopen_ptr;
    FILE *freopen_ptr;
    const char *filename;
    int i; /* Loop counter */
    filename = HOST_FILENAME("freopenAM.txt");


    const int MODES_ARRAY_SIZE       = 12; /*<--- Change this if elements are added to array below */
    char *modes_array[] = { "w", "wb", "w+b", "wb+",
                          "r", "rb", "r+b", "rb+",
                          "a", "ab", "a+b", "ab+" };

    fopen_ptr = FOPEN_TEST(filename, "w");
    for (i = 0; i < MODES_ARRAY_SIZE; ++i)
    {
        freopen_ptr = FREOPEN_TEST(filename, modes_array[i], fopen_ptr);

        test_all_modes_helper(modes_array[i]);
    }

    FCLOSE_DELETE_TEST(freopen_ptr, filename);
}

static void test_all_x_modes_helper(const char *mode)
{
    FILE *fopen_ptr;
    FILE *freopen_ptr;
    const char *filename;
    filename = HOST_FILENAME("freopenXM.txt");

    fopen_ptr = FOPEN_TEST(filename, mode);
    freopen_ptr = freopen(filename, mode, fopen_ptr);

    TESTCASE_MESSAGE(freopen_ptr == NULL, "freopen(\"%s\", \"%s\", fopen_ptr(%p)) expected to fail",
        filename, mode, fopen_ptr);

    FCLOSE_DELETE_TEST(freopen_ptr, filename);
    errno = 0;
}

static void test_all_x_modes()
{
    test_all_x_modes_helper("wx");
    test_all_x_modes_helper("w+x");
    test_all_x_modes_helper("wbx");
    test_all_x_modes_helper("w+bx");
    test_all_x_modes_helper("wb+x");

    test_all_x_modes_helper("ax");
    test_all_x_modes_helper("a+x");
    test_all_x_modes_helper("abx");
    test_all_x_modes_helper("a+bx");
    test_all_x_modes_helper("ab+x");
}

static void test_freopen_write()
{
    FILE *fopen_ptr;
    FILE *freopen_ptr;
    char *fread_buffer;
    const char *fwrite_text;
    const char *filename;
    filename = HOST_FILENAME("freopenW.txt");

    fwrite_text = "123";

    /* Create the file if it doesn't already exist */
    CREATE_FILE_TEST(filename, "test_freopen_write\n");

    fopen_ptr = FOPEN_TEST(filename, "r");
    freopen_ptr = FREOPEN_TEST(filename, "w", fopen_ptr);
    if (freopen_ptr == NULL) return;

    FWRITE_STRING_TEST(fwrite_text, freopen_ptr);

    rewind(fopen_ptr);
    TESTCASE_MESSAGE(errno == 0 && ferror(fopen_ptr) == 0,
        "rewind(%p) failed ferror(fopen_ptr) = %d freopen_ptr = %p",
        fopen_ptr, ferror(fopen_ptr), freopen_ptr);

    size_t fwrite_text_size = strlen(fwrite_text);
    fread_buffer = (char*)malloc(fwrite_text_size);
    if (fread_buffer == NULL) return;

    /* fread should fail as file is open only for write */
    size_t read_size = fread(fread_buffer, 1, fwrite_text_size, fopen_ptr);
    TESTCASE_MESSAGE(read_size == 0, "fread(%p, 1, %d, %p) returned %d expected 0",
        fread_buffer, fwrite_text_size, fopen_ptr);
    free(fread_buffer);

    FCLOSE_DELETE_TEST(freopen_ptr, filename);
    freopen_ptr = NULL;
    fopen_ptr = NULL;
}

static void test_freopen_read()
{
    FILE *fopen_ptr;
    FILE *freopen_ptr;
    const char *filename;
    const char *fwrite_text;
    char *fread_buffer;
    size_t fwrite_result; /* Return value from fwrite() */
    filename = HOST_FILENAME("freopenR.txt");

    fwrite_text = "fwrite text";

    fopen_ptr = FOPEN_TEST(filename, "w");
    FWRITE_STRING_TEST("abc", fopen_ptr);

    freopen_ptr = FREOPEN_TEST(filename, "r", fopen_ptr);
    if (freopen_ptr == NULL) return;

    /* Try reading from the file. The function should display errors if there's
     a problem */
    FREAD_FILE_TEST(&fread_buffer, fopen_ptr);
    free(fread_buffer);

    /* Attempt to call fwrite() to ensure that it fails */
    fwrite_result = fwrite(fwrite_text, sizeof(char), strlen(fwrite_text), fopen_ptr);
    TESTCASE_MESSAGE(fwrite_result == 0, "fwrite(%p, %d, %d, %p) returned %d, expected failure",
        fwrite_text, sizeof(char), strlen(fwrite_text), fopen_ptr, fwrite_result);

    FCLOSE_DELETE_TEST(freopen_ptr, filename);
    errno = 0;
}

static void test_char_array()
{
    FILE *fopen_ptr;
    FILE *freopen_ptr;
    const char *filename;
    int i;          /* Loop counter */
    int array_size; /* Size of 'modes' array */
    filename = HOST_FILENAME("freopenCA");

    char *modes  [5] = { "w", "r", "rb", "a+", "w+"};
    array_size =  5;

    /* Run through the test cases above */
    for (i = 0; i < array_size; ++i)
    {
        fopen_ptr = FOPEN_TEST(filename, "w");
        freopen_ptr = FREOPEN_TEST(filename, modes[i], fopen_ptr);
        FCLOSE_TEST(freopen_ptr, filename);
        freopen_ptr = NULL;
    }

    DELETE_IF_EXISTS(filename);
}

static void test_freopen_ferror_reset()
{
    FILE *fopen_ptr;
    FILE *freopen_ptr;
    const char *filename;
    filename = HOST_FILENAME("freopenER.txt");

    /* Create file if it doesn't already exist */
    fopen_ptr = FOPEN_TEST(filename, "w");
    FCLOSE_TEST(fopen_ptr, filename);
    fopen_ptr = NULL;

    /* trigger error by open for read but try to write */
    fopen_ptr = FOPEN_TEST(filename, "r");
    size_t result = fwrite("a", 1, 1, fopen_ptr);
    TESTCASE_MESSAGE(result <= 0, "fwrite(\"a\", 1, 1, %p) return %d",
        fopen_ptr, result);
    TESTCASE_MESSAGE(ferror(fopen_ptr) != 0, "ferror(%p) returned %d",
        fopen_ptr, ferror(fopen_ptr));

    /* freopen() should clear the error */
    freopen_ptr = freopen(filename, "r", fopen_ptr);
    TESTCASE_MESSAGE(ferror(freopen_ptr) == 0, "ferror(%p) returned %d",
        fopen_ptr, ferror(freopen_ptr));

    FCLOSE_DELETE_TEST(freopen_ptr, filename);
    freopen_ptr = NULL;
    fopen_ptr = NULL;
}

static void test_freopen_feof_reset()
{
    FILE *fopen_ptr;
    FILE *freopen_ptr;
    const char *filename;
    int result;
    int c;
    filename = HOST_FILENAME("freopenEOF.txt");

    /* Create and write content to it so it's not empty when fgetc() is called */
    fopen_ptr = FOPEN_TEST(filename, "w");
    FWRITE_STRING_TEST("text", fopen_ptr);
    FCLOSE_TEST(fopen_ptr, filename);

    /* read past EOF to set feof() flag */
    fopen_ptr = FOPEN_TEST(filename, "r");
    result = fseek(fopen_ptr, 0, SEEK_END);
    TESTCASE_MESSAGE(result == 0, "fseek(%p, 0, SEEK_END) returned %d",
        fopen_ptr, result);
    c = fgetc(fopen_ptr);
    TESTCASE_MESSAGE(c == EOF && feof(fopen_ptr) != 0,
        "fgetc(%p) returned %d expected EOF(%d), feof() == %d",
        fopen_ptr, c, EOF, feof(fopen_ptr));

    /* Call freopen() to reset the EOF indicator */
    freopen_ptr = FREOPEN_TEST(filename, "r", fopen_ptr);
    if (freopen_ptr == NULL) return;
    TESTCASE_MESSAGE(feof(fopen_ptr) == 0, "feof(%p) returned %d",
        fopen_ptr, feof(fopen_ptr));

    FCLOSE_DELETE_TEST(freopen_ptr, filename);
}

static void test_pointers_equal()
{
    FILE *fopen_ptr;
    FILE *freopen_ptr;
    const char *filename;
    filename = HOST_FILENAME("freopenPE.txt");

    fopen_ptr = FOPEN_TEST(filename, "w");
    freopen_ptr = FREOPEN_TEST(filename, "r", fopen_ptr);
    if (freopen_ptr == NULL) return;

    TESTCASE_MESSAGE(fopen_ptr == freopen_ptr,
        "freopen(\"%s\", \"r\", %p) returned %p expected %p",
        filename, fopen_ptr, freopen_ptr, fopen_ptr);

    FCLOSE_DELETE_TEST(freopen_ptr, filename);

}

static void test_freopen_same_mode()
{
    FILE *fopen_ptr;
    FILE *freopen_ptr;
    const char *filename;
    const char *fwrite_text;
    char *fread_buffer;
    filename = HOST_FILENAME("freopenSM");

    fwrite_text = "second fwrite";

    fopen_ptr = FOPEN_TEST(filename, "w");
    FWRITE_STRING_TEST("first fwrite", fopen_ptr);

    freopen_ptr = FREOPEN_TEST(filename, "w", fopen_ptr);
    if (freopen_ptr == NULL) return;
    FWRITE_STRING_TEST(fwrite_text, fopen_ptr);

    FCLOSE_TEST(freopen_ptr, filename);
    freopen_ptr = NULL;
    fopen_ptr = NULL;

    /* Open the file again for reading and make sure the right content was written */
    fopen_ptr = FOPEN_TEST(filename, "r");
    FREAD_FILE_TEST(&fread_buffer, fopen_ptr);
    TESTCASE_STRINGS_MATCH(fwrite_text, fread_buffer);
    free(fread_buffer);

    FCLOSE_DELETE_TEST(fopen_ptr, filename);
    fopen_ptr = NULL;
}

static void test_mismatched_filename_and_stream()
{
    FILE *fopen_ptr;
    FILE *freopen_ptr;
    const char *filename1;
    const char *filename2;
    filename1 = HOST_FILENAME("freopenMM1.txt");
    filename2 = HOST_FILENAME("freopenMM2.txt");

    fopen_ptr = FOPEN_TEST(filename1, "w");
    freopen_ptr = FREOPEN_TEST(filename2, "w", fopen_ptr);
    FCLOSE_DELETE_TEST(freopen_ptr, filename2);
    DELETE_IF_EXISTS(filename1);
}

static void test_empty_filename()
{
    FILE *fopen_ptr;
    FILE *freopen_ptr;
    const char *filename;
    filename = HOST_FILENAME("freopenEF.txt");

    fopen_ptr = FOPEN_TEST(filename, "w");
    freopen_ptr = freopen("", "a", fopen_ptr);
    TESTCASE_MESSAGE(freopen_ptr == NULL && errno != 0,
        "freopen(\"\", \"a\", %p) returned %p expected NULL",
        fopen_ptr, freopen_ptr);
    FCLOSE_DELETE_TEST(freopen_ptr, filename);
}

#define STRING_1    "fwrite1"
#define STRING_2    "append text"
static void test_multiple_freopens()
{
    FILE *fopen_ptr;
    FILE *freopen_ptr;
    const char *filename = HOST_FILENAME("freopenMFO.txt");
    const char *fwrite1 = STRING_1;
    const char *append  = STRING_2;
    char *test_string   = STRING_1 STRING_2;
    size_t test_string_size;
    char *fread_buffer;
    size_t fread_result;

    test_string_size = strlen(test_string) + 1;

    fopen_ptr = FOPEN_TEST(filename, "w");
    FWRITE_STRING_TEST(fwrite1, fopen_ptr);

    freopen_ptr = FREOPEN_TEST(filename, "r", fopen_ptr);
    if (freopen_ptr == NULL) return;

    freopen_ptr = FREOPEN_TEST(filename, "a", fopen_ptr);
    if (freopen_ptr == NULL) return;
    FWRITE_STRING_TEST(append, fopen_ptr);

    /* Try reading from the file even though it shouldn't be possible */
    fread_buffer = (char*)malloc(test_string_size);
    if (fread_buffer == NULL) return;
    fread_result = fread(fread_buffer, 1, test_string_size, fopen_ptr);
    TESTCASE_MESSAGE(fread_result == 0,
        "fread(%p, 1, %d, %p) returned %d expected 0",
        fread_buffer, test_string_size, fopen_ptr, fread_result);
    free(fread_buffer);

    /* Now freopen() for read and make sure the right content was written */
    freopen_ptr = FREOPEN_TEST(filename, "r", fopen_ptr);
    if (freopen_ptr == NULL) return;
    FREAD_FILE_TEST(&fread_buffer, fopen_ptr);
    TESTCASE_STRINGS_MATCH(test_string, fread_buffer);
    free(fread_buffer);

    FCLOSE_DELETE_TEST(freopen_ptr, filename);
    errno = 0;
}
/*------------------------------------------------------------------------}}}}*/



int ntd_extended_freopen(void)
{
    /* Before running the tests, open a file that will be left open while the
     other tests run, then close it at the end to test leaving a file open
     longer-term */
    FILE *fopen_ptr;
    FILE *freopen_ptr;
    const char *filename = HOST_FILENAME("freopenL.txt");

    NTD_TEST_GROUP_START("freopen", 2);

    fopen_ptr = FOPEN_TEST(filename, "w");
    freopen_ptr = freopen(filename, "r", fopen_ptr);
    if (freopen_ptr != NULL) {
        test_all_modes();
        test_all_x_modes();
        test_freopen_write();
        test_freopen_read();
        test_char_array();
        test_freopen_ferror_reset();
        test_freopen_feof_reset();
        test_pointers_equal();
        test_freopen_same_mode();
        test_mismatched_filename_and_stream();
        test_empty_filename();
        test_multiple_freopens();

        /* Commented out this test since running it breaks printf() and t_error() */
        /*test_redirect_stdout("test_redirect_stdout");*/ /* <-- This test must run last since it messes up printf() */


        /* No need to test garbage input for the 'mode' parameter in freopen()
         (i.e., fopen(const char *filename, const char *mode, FILE *stream) )
         since the situation is similar to fopen.c in that if 'mode' doesn't
         match exactly one of the known modes (i.e., "w", "rb", etc.) then
         freopen()'s behavior becomes undefined and implementation specific */

        FCLOSE_DELETE_TEST(freopen_ptr, filename);
    }

    return NTD_TEST_GROUP_END("freopen", 2);
}
