﻿/*******************************************************************************

  This file contains a series of tests for fopen()

*******************************************************************************/

#include "ntd_extended_test.h"
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <errno.h>
#include <fcntl.h>
#include <sys/stat.h>  // for definition of mkdir
#include <unistd.h>

static char gcharNonConstName[255];

static void test_append()
{
    FILE *file;
    char *fread_buffer;
    const char *filename;
    size_t count;
    int result;
    fread_buffer = NULL;
    filename = HOST_FILENAME("fopen.c_test_append.txt");

    DELETE_IF_EXISTS(filename);

    /* Write some data to the file */
    file = FOPEN_TEST(filename, "w");
    count = fwrite("abc", sizeof(char), 3, file);
    TESTCASE_MESSAGE(count == 3, "fwrite(\"abc\", 1, 3, file) failed count = %d", count);
    result = fclose(file);
    TESTCASE_MESSAGE(result == 0, "fclose(%p) failed result = %d", file, result);
    file = NULL;

    /* Try to append data to the file */
    file = FOPEN_TEST(filename, "a");
    count = fwrite("def", sizeof(char), 3, file);
    TESTCASE_MESSAGE(count == 3, "fwrite(\"def\", 1, 3, file) failed count = %d", count);
    result = fclose(file);
    TESTCASE_MESSAGE(result == 0, "fclose(%p) failed result = %d", file, result);
    file = NULL;

    file = FOPEN_TEST(filename, "r");
    if (file) {
        FREAD_FILE_TEST(&fread_buffer, file);

        TESTCASE_STRINGS_MATCH(fread_buffer, "abcdef");

        free(fread_buffer);
        result = FCLOSE_DELETE_TEST(file, filename);
    }
}


static void test_fopen_with_different_modes()
{
    FILE *file;
    const char *filename = HOST_FILENAME("fopen_with_different_modes.txt");
    int i;
    int array_size;
    int result;

    char *modes  [5] = { "w", "r", "rb", "a+", "w+b"};
    array_size =  5;

    /* Run through the test cases above */
    for (i = 0; i < array_size; ++i)
    {
        file = FOPEN_TEST(filename, modes[i]);
        result = fclose(file);
        TESTCASE_MESSAGE(result == 0, "fclose(%p) failed result = %d mode = \"%s\"",
            file, result, modes[i]);
        file = NULL;
    }

    DELETE_IF_EXISTS(filename);
}

static void test_write_to_read_file_helper(const char *mode)
{
    FILE *file;
    const char *filename = HOST_FILENAME("write_to_read.txt");
    size_t fwrite_ret_val;
    int result;

    CREATE_FILE_TEST(filename, "test_write_to_read_file_helper\n");

    file = FOPEN_TEST(filename, mode);

    fwrite_ret_val = fwrite("123", 1, 3, file);

    TESTCASE_MESSAGE(fwrite_ret_val == 0 && ferror(file),
        "fwrite(\"123\", 1, 3, %p) returned %d, ferror(%p) = %d expected failure (%s mode = %s)",
        file, fwrite_ret_val, file, ferror(file), filename, mode);

    FCLOSE_DELETE_TEST(file, filename);
}

static void test_write_to_read_file()
{
  test_write_to_read_file_helper("r");
  test_write_to_read_file_helper("rb");
}


static void test_write_non_exist_file_helper(const char *mode)
{
    FILE *file;
    const char *filename = HOST_FILENAME("write_non_exist_file");

    DELETE_IF_EXISTS(filename);

    file = FOPEN_TEST(filename, mode);
    FCLOSE_DELETE_TEST(file, filename);
}

static void test_write_non_exist_file()
{
  test_write_non_exist_file_helper("w");
  test_write_non_exist_file_helper("wb");
  test_write_non_exist_file_helper("a");
  test_write_non_exist_file_helper("ab");
}

static void test_read_non_exist_file_helper(const char *mode)
{
    FILE *file;
    const char *filename = HOST_FILENAME("fopen.c_non-existent_file.txt");

    file = fopen(filename, mode);
    TESTCASE_MESSAGE(file == NULL, "fopen('%s', '%s') return %p expected NULL for a non-existent file",
        filename, mode, file);
    if (file != NULL) {
        fclose(file);
        return;
    }
    FCLOSE_DELETE_TEST(file, filename);
}

static void test_read_non_exist_file()
{
  test_read_non_exist_file_helper("r");
  test_read_non_exist_file_helper("rb");
  test_read_non_exist_file_helper("r+");
  test_read_non_exist_file_helper("rb+");
  test_read_non_exist_file_helper("r+b");
}

static void test_read_overflow()
{
    int res;
    const char *filename = HOST_FILENAME("fopen.c_read_overflow.txt");

    int fd = open(filename, O_CREAT | O_RDWR);
    TESTCASE_MESSAGE(fd != 0, "open('%s') return %d expected non-zero for open()",
        filename, fd);
    lseek(fd, 1024, SEEK_SET);
    char buf[10];
    ssize_t size;
    size = read(fd, buf, 10);
    TESTCASE_MESSAGE(size == 0, "read('%s') return %d expected 0 for an overflow file",
        filename, size );
    res = close(fd);
    TESTCASE_MESSAGE(res == 0, "close('%s') return %d expected 0 for close()",
        filename, res);

}

static void test_discard_contents_on_write()
{
    FILE *file;
    char *fwrite_buffer;
    char *fread_buffer;
    size_t fwrite_ret_val;
    size_t fread_ret_val;
    long int ftell_ret_val;
    int i;
    const char *filename = HOST_FILENAME("fopen.c_discard_contents_test.txt");


    fwrite_buffer = "abc";

    file = FOPEN_TEST(filename, "wb");
    fwrite_ret_val = FWRITE_TEST(fwrite_buffer, sizeof(char), 3, file);

    FCLOSE_TEST(file, filename);
    file = NULL;

    file = FOPEN_TEST(filename, "wb");
    fwrite_buffer = "xyz";
    FWRITE_STRING_TEST(fwrite_buffer, file);
    FCLOSE_TEST(file, filename);
    file = NULL;

    /* Now test to make sure only the content from the 2nd fwrite() is in the file */

    file = FOPEN_TEST(filename, "rb");
    FREAD_FILE_TEST(&fread_buffer, file);
    TESTCASE_STRINGS_MATCH(fread_buffer, fwrite_buffer);

    FCLOSE_DELETE_TEST(file, filename);
    file = NULL;
    free(fread_buffer);
    fread_buffer = NULL;
}

static void test_open_x_write_modes_helper(const char *mode)
{
    FILE *file;
    const char *filename = HOST_FILENAME("fopen.c_test_open_x_modes.txt");

    DELETE_IF_EXISTS(filename);
    file = FOPEN_TEST(filename, mode);
    FCLOSE_TEST(file, filename);
    file = NULL;

    /* Now that the file exists, try re-opening it with the same mode and make
     * sure that it fails to do so
     */
    file = fopen(filename, mode);
    TESTCASE_MESSAGE(file == NULL, "fopen(\"%s\", \"%s\") returned %p expected NULL",
        filename, mode, file);
    FCLOSE_DELETE_TEST(file, filename);
}

/*
 * Test opening a file in the writing modes that use "x" (i.e., 'wx', 'w+x',
 * 'w+bx', etc.). The 'x' modes should prevent from writing to existing files
 */
static void test_open_x_write_modes()
{
  test_open_x_write_modes_helper("wx");
  test_open_x_write_modes_helper("w+x");
  test_open_x_write_modes_helper("wbx");
  test_open_x_write_modes_helper("wb+x");
  test_open_x_write_modes_helper("w+bx");
  test_open_x_write_modes_helper("ax");
  test_open_x_write_modes_helper("a+x");
  test_open_x_write_modes_helper("abx");
  test_open_x_write_modes_helper("ab+x");
  test_open_x_write_modes_helper("a+bx");
}

static void test_open_same_file()
{
    FILE *file_ptr_w;
    FILE *file_ptr_r;
    FILE *file_ptr_r2;
    FILE *file_ptr_a;
    char *fread_buffer = NULL;
    const char *fwrite_text = "text";
    const char *append_text = "append";
    char *strcmp_text;
    char *str_ret_val;
    const char *filename = HOST_FILENAME("open_twice.txt");

    asprintf(&strcmp_text, "%s%s", fwrite_text, append_text);
    TESTCASE_MESSAGE(strcmp_text != NULL, "asprintf() of \"%s\", \"%s\" returned %p",
        fwrite_text, append_text);
    if (!strcmp_text) return;

    file_ptr_w = FOPEN_TEST(filename, "w");
#if __NX__
    file_ptr_r = fopen(filename, "r");
    TESTCASE_MESSAGE(file_ptr_r == NULL && errno == EBUSY,
       "fopen(\"%s\",\"r\") returns %p file is already open for write "
       "errno = %d (%s) expected EBUSY on horizon",
       filename, file_ptr_r, errno, strerror(errno));
#else
    file_ptr_r = FOPEN_TEST(filename,"r");
#endif
    if (file_ptr_r == NULL)
    {
        FCLOSE_DELETE_TEST(file_ptr_w, filename);
        free(strcmp_text);
        return;
    }

    FWRITE_STRING_TEST(fwrite_text, file_ptr_w);

    rewind(file_ptr_w);

    file_ptr_r2 = FOPEN_TEST(filename, "r");
    if (file_ptr_r2 == NULL)
    {
        FCLOSE_TEST(file_ptr_r, filename);
        FCLOSE_DELETE_TEST(file_ptr_w, filename);
        free(strcmp_text);
        return;
    }

    FCLOSE_TEST(file_ptr_r, filename);

    FREAD_FILE_TEST(&fread_buffer, file_ptr_r2);
    TESTCASE_STRINGS_MATCH(fread_buffer, fwrite_text);
    free(fread_buffer);

    file_ptr_a = FOPEN_TEST(filename, "a");
    if (file_ptr_a == NULL)
    {
        FCLOSE_TEST(file_ptr_r2, filename);
        FCLOSE_DELETE_TEST(file_ptr_w, filename);
        free(strcmp_text);
        return;
    }

    FCLOSE_TEST(file_ptr_w, filename);

    FWRITE_STRING_TEST(append_text, file_ptr_a);

    rewind(file_ptr_a);

    file_ptr_r = FOPEN_TEST(filename, "r");
    if (file_ptr_r == NULL)
    {
        FCLOSE_TEST(file_ptr_r2, filename);
        FCLOSE_DELETE_TEST(file_ptr_w, filename);
        free(strcmp_text);
        return;
    }

    FCLOSE_TEST(file_ptr_r2, filename);

    FREAD_FILE_TEST(&fread_buffer, file_ptr_r);
    TESTCASE_STRINGS_MATCH(fread_buffer, strcmp_text);
    free(fread_buffer);
    free(strcmp_text);

    FCLOSE_TEST(file_ptr_a, filename);
    file_ptr_a = NULL;
    FCLOSE_DELETE_TEST(file_ptr_r, filename);
    file_ptr_r = NULL;
}

static void test_open_same_file_in_w_mode()
{
    FILE *file_ptr_w;
    FILE *file_ptr_w2;
    FILE *file_ptr_r;
    char *fread_buffer = NULL;
    const char *fwrite_text = "text";
    const char *filename = HOST_FILENAME("open_twice.txt");

    file_ptr_w = FOPEN_TEST(filename, "w");
    fprintf(file_ptr_w, "%s", fwrite_text);
    FCLOSE_TEST(file_ptr_w, filename);

    file_ptr_r = FOPEN_TEST(filename, "r");
    if (file_ptr_r == NULL)
    {
        return;
    }
    FREAD_FILE_TEST(&fread_buffer, file_ptr_r);
    TESTCASE_STRINGS_MATCH(fread_buffer,fwrite_text);
    FCLOSE_TEST(file_ptr_w, filename);

    file_ptr_w2 = FOPEN_TEST(filename, "w");
    long size = ftell(file_ptr_w2);
    size_t read_size = fread(fread_buffer, 1, size, file_ptr_w2);
    NTD_TESTCASE_MESSAGE(read_size == 0, __FILE__, __LINE__, "read_size == 0",
        "fread(\"%p\", 1, %d, %p) returned %d",
        fread_buffer, size, filename, read_size);
    FCLOSE_TEST(file_ptr_w2, filename);
    free(fread_buffer);
    file_ptr_w = NULL;
    file_ptr_w2 = NULL;
    file_ptr_r = NULL;
}

static void test_open_readonly_file_in_w_mode()
{
    FILE *file_ptr_w;
    const char *filename = HOST_FILENAME("readonly.txt");

    if (access(filename, F_OK)) {
        printf("NOTE: Please create a readonly file %s for test %s %s()\n", filename, __FILE__, __FUNCTION__);
        return;
    }

    file_ptr_w = fopen(filename, "w");
    TESTCASE_MESSAGE(file_ptr_w == NULL && errno == EPERM,
       "fopen(\"%s\",\"r\") returns %p file is a read-only file "
       "errno = %d (%s) expected EPERM on horzion",
       filename, file_ptr_w, errno, strerror(errno));
    FCLOSE_TEST(file_ptr_w, filename);
    file_ptr_w = NULL;
}



static void test_empty_string_filename()
{
    FILE *file;
    int errno_val; /* To store value of errno right after error happens */
    file = fopen("", "w");

    errno_val = errno;

    TESTCASE_MESSAGE(file == NULL, "open of empty string should fail");
    TESTCASE_MESSAGE(errno_val == ENOENT,
        "errno = %d after attempting to open a empty string should be ENOENT(%d)",
        errno_val,
        ENOENT);
}

static void test_long_string_filename()
{
    FILE *file;
    int errno_val; /* To store value of errno right after error happens */
    const char *filename = HOST_FILENAME("thisisalooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooongfilename.txt");
    file = fopen(filename, "w");

    errno_val = errno;

    TESTCASE_MESSAGE(file == NULL, "open of a longer string than EntryNameLengthMax should fail");
    TESTCASE_MESSAGE(errno_val == ENAMETOOLONG,
        "errno = %d after attempting to open a longer string than EntryNameLengthMax should be ENAMETOOLONG(%d)",
        errno_val,
        ENOENT);
}

static void test_open_non_const_string_filename()
{
    FILE *file;
    const char *filename = HOST_FILENAME("fopen_non_const_string_filename.txt");
    char charNonConstName[255];
    char charNonConstName2[255] = HOST_FILENAME("fopen_local_non_const_string_filename.txt");


    strcpy(charNonConstName, filename);
    strcpy(gcharNonConstName, filename);
    int i;
    int array_size;
    int result;

    char *modes  [5] = { "w", "r", "rb", "a+", "w+b"};
    array_size =  5;

    /* Run through the test cases above for a filename resides in a local array */
    for (i = 0; i < array_size; ++i)
    {
        file = FOPEN_TEST(charNonConstName, modes[i]);
        result = fclose(file);
        TESTCASE_MESSAGE(result == 0, "fclose(%p) failed result = %d mode = \"%s\"",
            file, result, modes[i]);
        file = NULL;
    }

    /* Run through the test cases above for a filename resides in a global array */
    for (i = 0; i < array_size; ++i)
    {
        file = FOPEN_TEST(gcharNonConstName, modes[i]);
        result = fclose(file);
        TESTCASE_MESSAGE(result == 0, "fclose(%p) failed result = %d mode = \"%s\"",
            file, result, modes[i]);
        file = NULL;
    }

    /* Run through the test cases above for a filename resides in a global array */
    for (i = 0; i < array_size; ++i)
    {
        file = FOPEN_TEST(charNonConstName2, modes[i]);
        result = fclose(file);
        TESTCASE_MESSAGE(result == 0, "fclose(%p) failed result = %d mode = \"%s\"",
            file, result, modes[i]);
        file = NULL;
    }

   DELETE_IF_EXISTS(filename);
   DELETE_IF_EXISTS(charNonConstName2);
   DELETE_IF_EXISTS(gcharNonConstName);
}

static void test_access()
{
    FILE *file_ptr;
    const char *filename = HOST_FILENAME("test_access.txt"), *dirname = HOST_FILENAME("test_access_dir");
    int ret;

    file_ptr = fopen(filename, "w");
    FCLOSE_TEST(file_ptr, filename);
    ret = access(filename, F_OK);
    TESTCASE_MESSAGE(ret == 0,
        "access(\"%s\",\"r\") returns %d "
        "errno = %d (%s)",
        filename, ret, errno, strerror(errno));
    file_ptr = NULL;

    mkdir(HOST_FILENAME("test_access_dir"), 0777);

    ret = access(dirname, F_OK);
    TESTCASE_MESSAGE(ret == 0,
        "access(\"%s\",\"r\") returns %d"
        "errno = %d (%s)",
        dirname, ret, errno, strerror(errno));
}

static void test_read_from_console()
{
    char buf[10];
    ssize_t size;
    int ret;

#if __NX__
    /* horizon does not support console read */
    size = read(STDIN_FILENO, buf, 10);
    TESTCASE_MESSAGE(size == -1, "read(STDIN_FILENO) return %d expected -1",
        size);
    size = read(STDOUT_FILENO, buf, 10);
    TESTCASE_MESSAGE(size == -1 && errno == EBADF,
        "read(STDOUT_FILENO) return %d expected -1 errno = %d (%s) expected EBADF on horizon",
        size, errno, strerror(errno));
    size = read(STDERR_FILENO, buf, 10);
    TESTCASE_MESSAGE(size == -1 && errno == EBADF,
        "read(STDERR_FILENO) return %d expected -1 errno = %d (%s) expected EBADF on horizon",
        size, errno, strerror(errno));

    ret = close(STDIN_FILENO);
    TESTCASE_MESSAGE(ret == 0, "close(STDIN_FILENO) return %d expected 0",
        ret);
    ret = close(STDOUT_FILENO);
    TESTCASE_MESSAGE(ret == 0, "close(STDOUT_FILENO) return %d expected 0",
        ret);
    ret = close(STDERR_FILENO);
    TESTCASE_MESSAGE(ret == 0, "close(STDERR_FILENO) return %d expected 0",
        ret);
#endif
}


int ntd_extended_fopen(void)
{
    FILE *file;
    #if __NX__
    const char *filename = HOST_FILENAME("long_term_file.txt");
    #else
    const char *filename = "long_term_file.txt";
    #endif

    NTD_TEST_GROUP_START("fopen", 2);

    file = fopen(filename, "w");

    if (file != NULL) {

        test_append();
        test_fopen_with_different_modes();
        test_write_to_read_file();
        test_write_non_exist_file();
        test_read_non_exist_file();
        test_read_overflow();
        test_discard_contents_on_write();
        test_open_x_write_modes();
        test_open_same_file();
        test_open_same_file_in_w_mode();
        test_empty_string_filename();
        test_long_string_filename();
        test_open_non_const_string_filename();
        test_access();
        test_read_from_console();

#if 0 /* Remove this test for now since it requires a pre-existing read-only file */
        test_open_readonly_file_in_w_mode();
#endif


        /* No need to test garbage input for the 'mode' parameter in fopen()
         (i.e., fopen(const char *filename, const char *mode) )
         since the C Standard specifies that if the 'mode' string doesn't match
         exactly one of the recognized modes (ex. "r", "wb", etc.) then fopen's
         behavior becomes undefined and implementation-specific */

        FCLOSE_DELETE_TEST(file, filename);
    }

    return NTD_TEST_GROUP_END("fopen", 2);
}
