﻿//==============================================================================
//
//  Main entry for MtStepTest
//
//==============================================================================

#include <nn/nn_Log.h>
#include "nn\nn_Result.h"
#include "nn\nn_TimeSpan.h"
#include "nn\os\os_Mutex.h"
#include "nn\os\os_Thread.h"
#include "nn\fs.h"
#include "nn\fs\fs_FileSystem.h"

const unsigned int THREAD_STACK_SIZE        = 4096; //Must be a multiple of 4096?
const unsigned int THREAD_STACK_ALIGNMENT   = 4096;
const unsigned int NUMBER_OF_THREADS        = 3;
const unsigned int LENGTH_OF_SLEEP          = 0;
const unsigned int RECURSION_DEPTH          = 0;

int Global = 0;

void RecursiveSleep( nn::TimeSpan ts, int Recursion )
{
    if( Recursion )
    {
        RecursiveSleep( ts, Recursion - 1 );
    }
    else
    {
        nn::os::SleepThread( ts );
    }
}

void* MyThreadFunction( void* p )
{
    int64_t ThreadIndex = (int64_t)p;

    for(;;)
    {
        NN_LOG( "ThreadIndex: %lld\n", ThreadIndex );
        RecursiveSleep( nn::TimeSpan::FromSeconds( LENGTH_OF_SLEEP ), RECURSION_DEPTH );

        if( ThreadIndex & 0x01 ) // For manual testing, place breakpoint here, remove it when hit, then begin stepping
        {
            Global += 1; // MtStepTest.01: For automated testing with 3 threads, place breakpoint here to always stop with ThreadIndex == 1
        }
        else if( ThreadIndex & 0x02 )
        {
            Global += 2;
        }
        else if( ThreadIndex & 0x04 )
        {
            Global += 4;
        }
    }
    NN_LOG( "End thread with ThreadIndex: %lld\n", ThreadIndex );

    return nullptr;
}

#ifdef NN_NINTENDO_SDK
extern "C" void nnMain( int argc, char **argv )
#else
int main( int argc, char **argv )
#endif
{
    NN_LOG( "Multi-Thread Step Test\n\n" );

#ifndef NX32
    for( int ArgIndex = 1; ArgIndex < argc; ArgIndex++ )
    {
        NN_LOG( "Command line arg %d:  %s\n", ArgIndex, argv[ArgIndex] );
    }
#endif

    nn::Result          TheResult = nn::ResultSuccess();
    nn::os::ThreadType* pThread[ NUMBER_OF_THREADS ];
    char*               pStack[  NUMBER_OF_THREADS ];

    static // This is required for CreateThread to work
    char StackBuffer[ NUMBER_OF_THREADS * THREAD_STACK_SIZE ] __attribute__((__aligned__(THREAD_STACK_ALIGNMENT)));

    //Create and start the threads.
    for( int i = 0; i < NUMBER_OF_THREADS; i++ )
    {
        pThread[ i ] = new nn::os::ThreadType;
        pStack[  i ] = &StackBuffer[ i * THREAD_STACK_SIZE ];

        NN_LOG( "Create thread %d on core %d with pThread:%p and pStack:%p\n", i, i%4, pThread[i], pStack[i] );

        //TheResult = nn::os::CreateThread( pThread[i], (nn::os::ThreadFunction)MyThreadFunction, (void*)i, pStack[i], THREAD_STACK_SIZE, 1 ); // Single-core version
        TheResult = nn::os::CreateThread( pThread[i], (nn::os::ThreadFunction)MyThreadFunction, (void*)i, pStack[i], THREAD_STACK_SIZE, 1, i%4 ); // Multi-core version
    }

    for(int i = 0; i < NUMBER_OF_THREADS; i++ )
    {
        nn::os::StartThread( pThread[i] );
    }

    for(int i = 0; i < NUMBER_OF_THREADS; i++)
    {
        nn::os::WaitThread( pThread[i] );
    }

    //Wait for each thread to finish, then destroy and free.
    for( unsigned int i = 0; i < NUMBER_OF_THREADS; i++ )
    {
        nn::os::DestroyThread( pThread[i] );
        delete pThread[i];
    }

    NN_LOG( "Global = %d\n", Global );
}
