﻿// ==========================================================================
//  Copyright (C) Deja Tools, LLC.  All rights reserved.
// ==========================================================================

#pragma once

#include <stdint.h>

namespace StressTestsCommon
{
    // --------------------------------------------------------------------------

    const char *ApplicationName{ "StressTestsAppName" };
    const char *PortName{ "StressTestsPort" };

    enum : int32_t
    {
        MINIMUM_NUMBER_OF_BYTES_TO_SEND = (1024 * 64),
        MAXIMUM_NUMBER_OF_BYTES_TO_SEND = (1024 * 1024 * 64),

        SMALL_MINIMUM_NUMBER_OF_BYTES_TO_SEND = (1),
        SMALL_MAXIMUM_NUMBER_OF_BYTES_TO_SEND = (1024 * 1),
    };

    // --------------------------------------------------------------------------
    // Commands to send
    enum class Command : int32_t
    {
        // Invalid.
        Invalid = 0,
        // Add new Commands below this value.
        // ----------------------------------

        // Set up a test run.  This is always sent from the Client to the Server.
        InitializeServerTest,

        // Requests the test results from the Server.  This is always sent from the Client to the Server.
        RequestServerTestResults,

        // The Test result response Sever.  This is always sent from the Server to the Client.
        ServerTestResults,

        // The Server will send data to the Client.
        Send,

        // The Client will send data to the Server.
        Receive,

        // The Server's comparison result of the Receive.  Will be followed by ReceiveResult.
        ReceiveServerResult,

        // This session is over.
        Close,

        // Close the server.
        CloseServer,

        // Sent from Client to Server.  Runs a set of Thread tests.  Will be followed by struct ThreadsTest.
        ThreadsTest,

        // Sent from Server to Client.  Will be followed by struct ThreadsTestResponse.
        ThreadsTestResponse,

        // ----------------------------------
        // Add new Commands above this value.
        NumberOfCommands
    };

    // Packet definition.
#pragma pack(push)
#pragma pack(1)
    struct Header
    {
        Command m_Command;
        int32_t m_SequenceNumber;
        int32_t m_DataLength;
    };

    // For use with Command::InitializeTest.
    struct InitializeTest
    {
        int32_t m_ExpectedNumberOfCommands;        // Sum of all commands.
        int32_t m_ExpectedNumberOfSendCommands;    // 0: Undetermined count (random).  Non-0: The expected number.
        int32_t m_ExpectedNumberOfReceiveCommands; // 0: Undetermined count (random).  Non-0: The expected number.
    };

    // For use with Command::FinalizeTest.
    struct FinalizeTest
    {
        int32_t m_ExpectedNumberOfCommands;        // This will be a copy of the InitializeTest values.
        int32_t m_ExpectedNumberOfSendCommands;    //
        int32_t m_ExpectedNumberOfReceiveCommands; //
        int32_t m_ActualNumberOfCommands;          // The "actual" number of commands received.
        int32_t m_ActualNumberOfSendCommands;      // The "actual" number of send commands received.
        int32_t m_ActualNumberOfReceiveCommands;   // The "actual" number of receive commands received.
        int32_t m_NumberOfBytesTransferred;        // The total number of bytes send/received (command data, not header data).
        int32_t m_NumberOfSendBytes;               // The total number of bytes sent to the Client.
        int32_t m_NumberOfReceiveBytes;            // The total number of bytes received from the Client.
    };

    struct ReceiveResult
    {
        int32_t m_PatternMatched;                  // 1: The pattern matched.  0: The pattern did not match.
        int32_t m_DiscrepancyIndex;                // Only valid when m_PatternMatched == 0: The index where the mismatch occurred.
        int32_t m_ActualValue;                     // Only valid when m_PatternMatched == 0: The actual byte value.
        int32_t m_ExpectedValue;                   // Only valid when m_PatternMatched == 0: The expected byte value.
    };

    // Sent from Client to Server.
    struct ThreadsTest
    {
        int32_t m_NumberOfSocketThreads;           // The number of socket threads to create.
    };
    // Sent from Server to Client.
    struct ThreadsTestResponse
    {
        int32_t m_Result;                          // 0: Sockets and Threads are ready.  1: Failure.
    };
#pragma pack(pop)

    int32_t WritePatternFill( uint8_t *pDataBuffer, int32_t Count )
    {
        int32_t Result{ 0 };

        // Write the buffer pattern.
        if( Result == 0 )
        {
            // Pattern fill:
            // - int32_t bytes (m_dataLength >= 4):
            //   - 0, 1, 2, 3, ... PatternFillCount.
            // - int8_t bytes (m_dataLength & 0x3):
            //   - 1, 2, 3.
            int32_t PatternFillCount{ Count / static_cast<int32_t>(sizeof(Count)) };
            int32_t LeftOverPattern{ Count & (static_cast<int32_t>(sizeof(Count) - 1)) };
            int32_t *pInteger{ reinterpret_cast<int32_t *>(pDataBuffer) };
            for( int32_t Index = 0; Index < PatternFillCount; Index++ )
            {
                *pInteger = Index;
                pInteger++;
            }

            // Left over bytes.
            for( int32_t Index = 0; Index < LeftOverPattern; Index++ )
            {
                const int32_t LeftOverIndex{ Count - LeftOverPattern + Index };
                pDataBuffer[LeftOverIndex] = static_cast<uint8_t>(Index + 1);
            }
        }

        return Result;
    }

    int32_t ValidatePatternFill( const uint8_t *pDataBuffer, int32_t Count, int32_t& DiscrepancyIndex, uint8_t& ActualBufferValue, uint8_t& ExpectedBufferValue )
    {
        int32_t Result{ 0 };

        // Default the out parameters.
        DiscrepancyIndex    = -1;
        ActualBufferValue   = 0;
        ExpectedBufferValue = 0;

        if( Result == 0 )
        {
            // Pattern fill:
            // - int32_t bytes (m_dataLength >= 4):
            //   - 0, 1, 2, 3, ... PatternFillCount.
            // - int8_t bytes (m_dataLength & 0x3):
            //   - 1, 2, 3.
            int32_t       PatternFillCount{ Count / static_cast<int32_t>(sizeof(Count)) };
            int32_t       LeftOverPattern{ Count & (static_cast<int32_t>(sizeof(Count) - 1)) };
            const int32_t *pInteger{ reinterpret_cast<const int32_t *>(pDataBuffer) };
            for( int32_t Index = 0; (Index < PatternFillCount) && (Result == 0); Index++ )
            {
                if( *pInteger != Index )
                {
                    // Find the exact byte that does not match.
                    const uint8_t *pIndex{ reinterpret_cast<uint8_t *>(&Index) };
                    const uint8_t *pData{ reinterpret_cast<const uint8_t *>(pInteger) };
                    for (int32_t detailedIndex = 0; (detailedIndex < sizeof(Index)) && (DiscrepancyIndex == -1); detailedIndex++ )
                    {
                        if( *(pIndex + detailedIndex) != *(pData + detailedIndex) )
                        {
                            DiscrepancyIndex    = (pData + detailedIndex) - pDataBuffer;
                            ActualBufferValue   = *(pData + detailedIndex);
                            ExpectedBufferValue = *(pIndex + detailedIndex);
                        }
                    }
                    Result = EXIT_FAILURE;
                }
                pInteger++;
            }

            // Test the "leftover bytes".
            for( int32_t Index = 0; (Index < LeftOverPattern) && (Result == 0); Index++ )
            {
                const int32_t LeftOverIndex{ Count - LeftOverPattern + Index };
                if (pDataBuffer[LeftOverIndex] != (Index + 1))
                {
                    DiscrepancyIndex    = LeftOverIndex;
                    ActualBufferValue   = pDataBuffer[LeftOverIndex];
                    ExpectedBufferValue = static_cast<uint8_t>(Index + 1);
                    Result = EXIT_FAILURE;
                }
            }
        }

        return Result;
    }
}
