﻿//========================================================================================
// HTCSDirectory.cpp
//========================================================================================

#include <stdio.h>
#include <tchar.h>
#include "HTCSDirectory.h"
#include "string.h"

//==============================================================================
// Our directory server particulars
const char* DIRECTORY_SERVER_IP     = "127.0.0.1";
const int   DIRECTORY_PORT_NUMBER   =  8003;

//========================================================================================

htcs_directory_entry::htcs_directory_entry( char* pPeerName, char* pPortName, char* pIPAddress, char* pPort )
{
    strcpy_s( m_PeerName, pPeerName );
    strcpy_s( m_PortName, pPortName );
    strcpy_s( m_IPAddress, pIPAddress );
    strcpy_s( m_Port, pPort );
    m_PortNumber = atoi(pPort);
}

//========================================================================================

htcs_directory_entry::~htcs_directory_entry()
{
}

//========================================================================================
// Accessors

char* htcs_directory_entry::GetPeerName (){ return m_PeerName;}
char* htcs_directory_entry::GetPortName (){ return m_PortName;}
char* htcs_directory_entry::GetIPAddress(){ return m_IPAddress;}
int   htcs_directory_entry::GetPort     (){ return m_PortNumber;}

void htcs_directory_entry::GetTcpAddress( char* pIPAddress, int* pPort )
{
    strcpy_s( pIPAddress, 64, m_IPAddress );
    *pPort = m_PortNumber;
}

void htcs_directory_entry::GetHTCSAddress( char* pServerName, char* pPortName )
{
    strcpy_s( pServerName, 64, m_PeerName );
    strcpy_s( pPortName, 64, m_PortName );
}

//========================================================================================

char* htcs_directory_entry::Describe( char* pToString, size_t SizeOfStringBuffer )
{
    sprintf_s( pToString, SizeOfStringBuffer, "%s:%s is at %s:%s", m_PeerName, m_PortName, m_IPAddress, m_Port );
    return pToString;
}

//========================================================================================

bool htcs_directory_entry::Is( const char* pTargetServer, const char* pTargetPort )
{
    // Are they looking for a particular peer name?
    if( strlen( pTargetServer ) > 0 )
    {
        if( strcmp( pTargetServer, m_PeerName) != 0 )
        {
            return false;
        }
    }

    return ( strcmp( m_PortName, pTargetPort ) == 0 );
}

//========================================================================================

bool htcs_directory_entry::Is( const char* pIPAddress, int Port )
{
    if( m_PortNumber == Port )
    {
        return ( strcmp( m_IPAddress, pIPAddress ) == 0 );
    }

    return false;
}

//========================================================================================

//========================================================================================
// Directory stream.  Isolates our server communications from the data
//========================================================================================

class htcs_directory_stream
{
public:
    htcs_directory_stream();
    ~htcs_directory_stream();

    bool                                IsAvailable ();
    bool                                HasError    ();
    void                                ReportError ();
    int                                 Read        ( char*     pBuffer,
                                                      size_t    SizeOfBuffer );

    int                                 Write       ( char*     pBuffer,
                                                      size_t    SizeOfData );

private:
    char                                m_ErrorMessage[1024];
    bool                                m_IsAvailable;
    SOCKET m_connectingSocket;
};

//========================================================================================

htcs_directory_stream::htcs_directory_stream()
{
    memset( m_ErrorMessage, 0, sizeof(m_ErrorMessage) );
    m_IsAvailable = false;
    m_connectingSocket = socket( AF_INET, SOCK_STREAM, 0 );

    //==============================================================================
    // Do our connect
    struct sockaddr_in ServerAddress;

    memset( &ServerAddress, 0, sizeof(ServerAddress) );
    ServerAddress.sin_family      = AF_INET;
    ServerAddress.sin_port        = htons( DIRECTORY_PORT_NUMBER );

    //==============================================================================
    //  Set the remote IP address
    InetPton( AF_INET, DIRECTORY_SERVER_IP, &ServerAddress.sin_addr.s_addr );

    //==============================================================================
    // connect() to the server
    int Res = connect( m_connectingSocket, (struct sockaddr *) &ServerAddress, sizeof(ServerAddress) );

    if ( Res < 0 )
    {
        strcpy_s(m_ErrorMessage, "htcs_directory_stream: Error calling connect()\n");
    }
    else
    {
        m_IsAvailable = true;
    }

}
htcs_directory_stream::~htcs_directory_stream()
{
    closesocket( m_connectingSocket );
}

//========================================================================================

bool htcs_directory_stream::IsAvailable () { return m_IsAvailable; }
bool htcs_directory_stream::HasError    () { return strlen( m_ErrorMessage ) != 0; }

//========================================================================================

void htcs_directory_stream::ReportError()
{
    printf( "ERROR:  %s\n", m_ErrorMessage );
}

//========================================================================================

int htcs_directory_stream::Read( char* pBuffer, size_t SizeOfBuffer )
{
    size_t AmountRead;
    for ( AmountRead = 1; AmountRead < SizeOfBuffer; AmountRead++ )
    {
        char c;
        int Res = recv( m_connectingSocket, &c, 1, 0 );
        if ( Res == 1 )
        {
            *pBuffer++ = c;
            if ( c == '\n' )
            {
                *pBuffer++ = 0;
                break;
            }
        }
        else if ( Res == 0 )
        {
            if ( AmountRead == 1 )
            {
                return 0;
            }
            else
            {
                break;
            }
        }
        else
        {
            if( WSAGetLastError() == WSAEINTR )
            {
                continue;
            }

            return -1;
        }
    }

    *pBuffer = 0;
    return AmountRead;
}

//========================================================================================

int htcs_directory_stream::Write( char* pBuffer, size_t SizeOfData )
{
    int nwritten = 0;
    size_t nleft  = SizeOfData;

    while ( nleft > 0 )
    {
        nwritten = send( m_connectingSocket, pBuffer, nleft, 0 );

        if ( nwritten  <= 0 )
        {
            if( WSAGetLastError() == WSAEINTR )
            {
                nwritten = 0;
            }
            else
            {
                return -1;
            }
        }

        nleft  -= nwritten;
        pBuffer += nwritten;
    }

    return nwritten;
}

//========================================================================================

//========================================================================================
// Directory class
//========================================================================================

htcs_directory::htcs_directory()
{
    m_pStream = new htcs_directory_stream();
    FetchDirectoryListing();
}

//========================================================================================

htcs_directory::~htcs_directory()
{
    for ( std::list<htcs_directory_entry*>::iterator it = m_Entries.begin(); it != m_Entries.end(); ++it)
    {
        htcs_directory_entry* pEntry = *it;
        delete pEntry;
    }

    m_Entries.clear();
    delete m_pStream;
}

//========================================================================================

bool                                htcs_directory::IsAvailable         () { return m_pStream->IsAvailable(); }
bool                                htcs_directory::HasError            () { return m_pStream->HasError(); }
std::list<htcs_directory_entry*>&   htcs_directory::GetDirectoryListing () { return m_Entries; }

//========================================================================================

void htcs_directory::ReportError()
{
    m_pStream->ReportError();
}

//========================================================================================

bool htcs_directory::RegisterEntry( const char* pServerName, const char* pPortName, const char* pIPAddress, const char* pPort, const char* pRequestName )
{
    char pBuffer[4096];

    // Build our XML
    strcpy_s( pBuffer, "<RegisterPortMapCommand>" );
        strcat_s( pBuffer, "<PortMapItem>" );

            strcat_s( pBuffer, "<HtcsPeerName>" );
                strcat_s( pBuffer, pServerName );
            strcat_s( pBuffer, "</HtcsPeerName>" );

            strcat_s( pBuffer, "<HtcsPortName>" );
                strcat_s( pBuffer, pPortName );
            strcat_s( pBuffer, "</HtcsPortName>" );

            strcat_s( pBuffer, "<IPAddress>" );
                strcat_s( pBuffer, pIPAddress );
            strcat_s( pBuffer, "</IPAddress>" );

            strcat_s( pBuffer, "<TcpPortNumber>" );
                strcat_s( pBuffer, pPort );
            strcat_s( pBuffer, "</TcpPortNumber>" );

        strcat_s( pBuffer, "</PortMapItem>" );
        if( pRequestName != NULL )
        {
            strcat_s( pBuffer, "<RequestName>" );
                strcat_s( pBuffer, pRequestName );
            strcat_s( pBuffer, "</RequestName>" );
        }
    strcat_s( pBuffer, "</RegisterPortMapCommand>" );

    //NOTICE that this HAS to end with a new line or HTC won't read it.
    strcat_s( pBuffer, "\n" );

    if( m_pStream->Write( pBuffer, strlen(pBuffer) ) > 0 )
    {
        //===============================================================
        // When we send something to the HTCS server, it sends a result.
        return CheckCommandResult( NULL );
    }
    return false;
}

//========================================================================================

bool htcs_directory::UnRegisterEntry ( const char* pServerName, const char* pPortName,  const char* pRequestName )
{
    char pBuffer[4096];

    // Build our XML
    strcpy_s( pBuffer, "<UnregisterPortMapCommand>" );
        strcat_s( pBuffer, "<HtcsPeerName>" );
            strcat_s( pBuffer, pServerName );
        strcat_s( pBuffer, "</HtcsPeerName>" );
        strcat_s( pBuffer, "<HtcsPortName>" );
            strcat_s( pBuffer, pPortName );
        strcat_s( pBuffer, "</HtcsPortName>" );
        if( pRequestName != NULL )
        {
            strcat_s( pBuffer, "<RequestName>" );
                strcat_s( pBuffer, pRequestName );
            strcat_s( pBuffer, "</RequestName>" );
        }
    strcat_s( pBuffer, "</UnregisterPortMapCommand>" );

    //NOTICE that this HAS to end with a new line or HTC won't read it.
    strcat_s( pBuffer, "\n" );

    if( m_pStream->Write( pBuffer, strlen(pBuffer) ) > 0 )
    {
        return CheckCommandResult( NULL );
    }
    return false;
}

//========================================================================================

bool htcs_directory::FetchCommandResult( char* pBuffer, size_t SizeOfBuffer )
{
    while( m_pStream->Read( pBuffer, SizeOfBuffer )  > 0 )
    {
        //For us, the first 3 bytes are the Byte Order Mark (BOM).
        // Skip past that to do our parsing.
        char* pReadAt = pBuffer;// + 1;// + 3;
        //========================================================================================
        // The format is like this:
        // <CommandResult>
        //      <RequestName>MyRequest1</RequestName>
        //      <Value>0</Value>
        //  </CommandResult>
        //========================================================================================
        char* pEnd = pReadAt + strlen( pReadAt );
        char* pCommandResStart = DirectoryXMLFind( pReadAt, pEnd, "CommandResult" );
        if( pCommandResStart != NULL )
        {
            return true;
        }
    }
    return false;
}

//========================================================================================

bool htcs_directory::CheckCommandResult( const char* pRequestName )
{
    char pReadBuffer[4096];
    if( FetchCommandResult( pReadBuffer, sizeof(pReadBuffer) ) )
    {
        //For us, the first 3 bytes are the Byte Order Mark (BOM).
        // Skip past that to do our parsing.
        char* pReadAt = pReadBuffer + 1;// + 3;
        //========================================================================================
        // The format is like this:
        // <CommandResult>
        //      <RequestName>MyRequest1</RequestName>
        //      <Value>0</Value>
        //  </CommandResult>
        //========================================================================================
        char* pEnd = pReadAt + strlen( pReadAt );
        char ValueBuffer[1024];
        if( DirectoryXMLGet( pReadAt, pEnd, "Value", ValueBuffer ) )
        {
            return ( (ValueBuffer[0] == '0') && (ValueBuffer[1] == 0) );
        }
    }

    return false;
}

//========================================================================================

void htcs_directory::FetchDirectoryListing()
{
    char pReadBuffer[4096*16];
    if( m_pStream->Read( pReadBuffer, sizeof(pReadBuffer) )  > 0 )
    {
        // What we really want is ALL the ports, including the system ports.
        // Tell the server and let's try again
        char pWriteBuffer[1024];

        // Build our XML - NOTICE that this HAS to end with a new line or HTC won't read it.
        strcpy_s( pWriteBuffer, "<RequestSystemPortMapping></RequestSystemPortMapping>\n" );

        if( m_pStream->Write( pWriteBuffer, strlen(pWriteBuffer) ) > 0 )
        {
            //========================================================================================
            // The response SHOULD be all our ports, including the "system" ports
            //========================================================================================
            if( m_pStream->Read( pReadBuffer, sizeof(pReadBuffer) )  > 0 )
            {
                char* pReadAt = pReadBuffer;// + 3;
                //========================================================================================
                //The format is like this:
                // <PortMapItem>
                //         <HtcsPeerName>Target1</HtcsPeerName>
                //         <HtcsPortName>TestPortName1</HtcsPortName>
                //         <IPAddress>127.0.0.1</IPAddress>
                //         <TcpPortNumber>50867</TcpPortNumber>
                // </PortMapItem>
                //========================================================================================
                char* pPortMapStart;
                char* pPortMapEnd;
                char* pEnd = pReadAt + strlen( pReadAt );
                while( DirectoryXMLFindPortMap( pReadAt, pEnd, &pPortMapStart, &pPortMapEnd ) )
                {
                    DirectoryXMLParse( pPortMapStart, pPortMapEnd );
                    pReadAt = pPortMapEnd;
                }
            }
        }
    }
}

//========================================================================================

char* htcs_directory::DirectoryXMLFind( char* pPortMapStart, char* pPortMapEnd, char* pTag )
{
    char SearchBuffer[1024];

    //Create our search tag
    strcpy_s(SearchBuffer, "<");
    strcat_s(SearchBuffer, pTag);
    strcat_s(SearchBuffer, ">");
    char *pRet = strstr( pPortMapStart, SearchBuffer );
    if( pRet != NULL )
    {
        if( pRet >= pPortMapEnd )
        {
            pRet = NULL;
        }
        else
        {
            pRet += strlen(SearchBuffer);
        }
    }
    return pRet;
}

//========================================================================================

bool htcs_directory::DirectoryXMLGet( char* pStart, char* pEnd, char* pTag, char* pContentBuffer )
{
    char* pContent = DirectoryXMLFind( pStart, pEnd, pTag );

    if( pContent != NULL )
    {
        char* pWriteTo = pContentBuffer;
        char* pReadFrom = pContent;
        while( *pReadFrom && *pReadFrom != '<')
        {
            *pWriteTo = *pReadFrom;
            pWriteTo += 1;
            pReadFrom += 1;
        }
        *pWriteTo = 0;

        return true;
    }
    return false;
}

//========================================================================================

bool htcs_directory::DirectoryXMLFindPortMap ( char *pReadAt, char* pEnd, char** ppPortMapStart, char** ppPortMapEnd )
{
    *ppPortMapEnd = pEnd;
    *ppPortMapStart = DirectoryXMLFind( pReadAt, pEnd, "PortMapItem" );
    if( *ppPortMapStart != NULL )
    {
        //Find the closing bracket.
        char SearchBuffer[1024];

        //Create our search tag
        strcpy_s( SearchBuffer, "</PortMapItem>");
        char *pRet = strstr( *ppPortMapStart, SearchBuffer );
        if( pRet != NULL )
        {
            pRet += strlen(SearchBuffer);
            *ppPortMapEnd = pRet;
        }
        return true;
    }

    return false;
}

//========================================================================================

void htcs_directory::DirectoryXMLParse( char *pPartMapStart, char* pPortMapEnd )
{

    //========================================================================================
    //The format is like this:
    // <PortMapItem>
    //         <HtcsPeerName>Target1</HtcsPeerName>
    //         <HtcsPortName>TestPortName1</HtcsPortName>
    //         <IPAddress>127.0.0.1</IPAddress>
    //         <TcpPortNumber>50867</TcpPortNumber>
    // </PortMapItem>
    //========================================================================================
    char HtcsPeerName[1024];
    if( DirectoryXMLGet( pPartMapStart, pPortMapEnd, "HtcsPeerName", HtcsPeerName ) )
    {
        char HtcsPortName[1024];
        if( DirectoryXMLGet( pPartMapStart, pPortMapEnd, "HtcsPortName", HtcsPortName ) )
        {
            char IPAddress[1024];
            if( DirectoryXMLGet( pPartMapStart, pPortMapEnd, "IPAddress", IPAddress ) )
            {
                char TcpPortNumber[1024];
                if( DirectoryXMLGet( pPartMapStart, pPortMapEnd, "TcpPortNumber", TcpPortNumber ) )
                {
                    m_Entries.push_back( new htcs_directory_entry( HtcsPeerName, HtcsPortName, IPAddress, TcpPortNumber ) );
                }
            }
        }
    }
}

//========================================================================================
// Resolution functionality
//========================================================================================

bool htcs_directory::ResolveTcpAddress( const char* pServerName, const char* pPortName, char* pIPAddress, int* pPort )
{
    for ( std::list<htcs_directory_entry*>::iterator it = m_Entries.begin(); it != m_Entries.end(); ++it)
    {
        htcs_directory_entry* pEntry = *it;
        if( pEntry->Is( pServerName, pPortName ) )
        {
            pEntry->GetTcpAddress( pIPAddress, pPort );
            return true;
        }
    }
    return false;
}

//========================================================================================

bool htcs_directory::ResolveHTCSAddress( const char* pIPAddress, int Port, char* pServerName, char* pPortName )
{
    for ( std::list<htcs_directory_entry*>::iterator it = m_Entries.begin(); it != m_Entries.end(); ++it)
    {
        htcs_directory_entry* pEntry = *it;
        if( pEntry->Is( pIPAddress, Port ) )
        {
            pEntry->GetHTCSAddress( pServerName, pPortName );
            return true;
        }
    }
    return false;
}

//========================================================================================
