﻿
#include <stdint.h>
#include <nn/nn_Macro.h>
#include <nn/nn_Log.h>
#include <nn/nn_Assert.h>
#include <nn/os.h>

//-----------------------------------------------------------------------------

namespace {

const size_t           ThreadStackSize = 8192;              // スレッド操作スレッドのスタックサイズ
NN_OS_ALIGNAS_THREAD_STACK char  g_ThreadStack1[ ThreadStackSize ];   // 1つ目のスレッドのスタック
NN_OS_ALIGNAS_THREAD_STACK char  g_ThreadStack2[ ThreadStackSize ];   // 2つ目のスレッドのスタック

nn::os::ThreadType  g_Thread1;
nn::os::ThreadType  g_Thread2;

}   // namespace

//-----------------------------------------------------------------------------

//
//  スレッド情報を表示
//
void    PrintThreadInfo(const char* name, nn::os::ThreadType* thread)
{
    NN_LOG("[%s]\n", name);

    const char* threadName = nn::os::GetThreadNamePointer( thread );
    NN_LOG("name     : %s\n", threadName);
    NN_LOG("priority : %d\n", nn::os::GetThreadPriority( thread ));
    NN_LOG("core No. : %d\n", nn::os::GetCurrentCoreNumber());
    NN_LOG("\n");
}


//
//  1 つ目のスレッド
//
void ThreadFunction1(void *arg)
{
    NN_UNUSED(arg);

    PrintThreadInfo( "thread1", &g_Thread1 );
}


//
//  2 つ目のスレッド
//
void ThreadFunction2(void *arg)
{
    NN_UNUSED(arg);

    // スレッド 1 が終了するのを待つ
    nn::os::WaitThread( &g_Thread1 );

    PrintThreadInfo( "thread2", &g_Thread2 );

    // スレッド 2 の名前を変更
    NN_LOG("Change name of thread2\n");
    nn::os::SetThreadName( &g_Thread2, "Thread2_NewName" );

    // スレッド 2 の優先度を変更
    NN_LOG("Change priority of thread2 to 5\n");
    nn::os::ChangeThreadPriority( &g_Thread2, 5 );

    NN_LOG("\n");
    PrintThreadInfo( "thread2", &g_Thread2 );
}


//
//  メイン関数です。
//
extern "C" void nnMain()
{
    nn::Result      result;

    // スレッドを生成する
    result = nn::os::CreateThread( &g_Thread1, ThreadFunction1, NULL, g_ThreadStack1, ThreadStackSize, nn::os::DefaultThreadPriority );
    NN_ASSERT( result.IsSuccess(), "Cannot create g_Thread1." );

    result = nn::os::CreateThread( &g_Thread2, ThreadFunction2, NULL, g_ThreadStack2, ThreadStackSize, nn::os::DefaultThreadPriority );
    NN_ASSERT( result.IsSuccess(), "Cannot create g_Thread2." );

    // スレッドの実行を開始する
    nn::os::StartThread( &g_Thread1 );
    nn::os::StartThread( &g_Thread2 );

    // スレッドが終了するのを待つ
    nn::os::WaitThread( &g_Thread1 );
    nn::os::WaitThread( &g_Thread2 );

    // スレッドを破棄する
    nn::os::DestroyThread( &g_Thread1 );
    nn::os::DestroyThread( &g_Thread2 );
}

