﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nnt.h>
#include <nnt/viUtil/testVi_Fixture.h>
#include <nn/vi.private.h>
#include <nn/ae.h>

typedef nnt::vi::util::Fixture SetDisplayMode;

TEST_F(SetDisplayMode, Default_Any)
{
    nn::vi::Initialize();

    nn::vi::Display* pDisplay;
    NNT_ASSERT_RESULT_SUCCESS(nn::vi::OpenDefaultDisplay(&pDisplay));

    nn::vi::DisplayModeInfo mode;
    NNT_ASSERT_RESULT_SUCCESS(nn::vi::GetDisplayMode(&mode, pDisplay));

    nn::vi::DisplayModeInfo modes[nn::vi::DisplayModeCountMax];
    int modeCount = nn::vi::ListDisplayModes(modes, sizeof(modes) / sizeof(modes[0]), pDisplay);
    ASSERT_NE(0, modeCount);

    for( int i = 0; i < modeCount; ++i )
    {
        NNT_ASSERT_RESULT_SUCCESS(nn::vi::SetDisplayMode(pDisplay, modes + i));

        nn::vi::DisplayModeInfo mode;
        std::memset(&mode, 0, sizeof(mode));

        NNT_ASSERT_RESULT_SUCCESS(nn::vi::GetDisplayMode(&mode, pDisplay));

        EXPECT_EQ(modes[i].width, mode.width);
        EXPECT_EQ(modes[i].height, mode.height);
        EXPECT_EQ(modes[i].refreshRate, mode.refreshRate);
        EXPECT_EQ(modes[i].mode, mode.mode);
    }

    // revert back to original mode
    NNT_ASSERT_RESULT_SUCCESS(nn::vi::SetDisplayMode(pDisplay, &mode));

    nn::vi::CloseDisplay(pDisplay);
}

TEST_F(SetDisplayMode, Default_InvalidMode_Any)
{
    nn::vi::Initialize();

    nn::vi::Display* pDisplay;
    NNT_ASSERT_RESULT_SUCCESS(nn::vi::OpenDefaultDisplay(&pDisplay));

    nn::vi::DisplayModeInfo current;
    NNT_ASSERT_RESULT_SUCCESS(nn::vi::GetDisplayMode(&current, pDisplay));

    nn::vi::DisplayModeInfo mode{ 0, 0, 0.f, nn::vi::StereoMode_None };
    NNT_ASSERT_RESULT_FAILURE(nn::vi::ResultNotSupported, nn::vi::SetDisplayMode(pDisplay, &mode));

    NNT_ASSERT_RESULT_SUCCESS(nn::vi::GetDisplayMode(&mode, pDisplay));
    EXPECT_EQ(current.width, mode.width);
    EXPECT_EQ(current.height, mode.height);
    EXPECT_EQ(current.refreshRate, mode.refreshRate);
    EXPECT_EQ(current.mode, mode.mode);

    nn::vi::CloseDisplay(pDisplay);
}

TEST_F(SetDisplayMode, External_InvalidMode_IcosaCopper)
{
    if( nn::ae::GetOperationMode() == nn::ae::OperationMode_Console )
    {
        nn::vi::Initialize();

        nn::vi::Display* pDisplay;
        NNT_ASSERT_RESULT_SUCCESS(nn::vi::OpenDisplay(&pDisplay, "External"));

        nn::vi::DisplayModeInfo current;
        NNT_ASSERT_RESULT_SUCCESS(nn::vi::GetDisplayMode(&current, pDisplay));

        nn::vi::DisplayModeInfo mode{ 0, 0, 0.f, nn::vi::StereoMode_None };
        NNT_ASSERT_RESULT_FAILURE(nn::vi::ResultNotSupported, nn::vi::SetDisplayMode(pDisplay, &mode));

        NNT_ASSERT_RESULT_SUCCESS(nn::vi::GetDisplayMode(&mode, pDisplay));
        EXPECT_EQ(current.width, mode.width);
        EXPECT_EQ(current.height, mode.height);
        EXPECT_EQ(current.refreshRate, mode.refreshRate);
        EXPECT_EQ(current.mode, mode.mode);

        nn::vi::CloseDisplay(pDisplay);
    }
}

TEST_F(SetDisplayMode, External_Console_IcosaCopper)
{
    if( nn::ae::GetOperationMode() == nn::ae::OperationMode_Console )
    {
        nn::vi::Initialize();

        nn::vi::Display* pDisplay;
        NNT_ASSERT_RESULT_SUCCESS(nn::vi::OpenDisplay(&pDisplay, "External"));

        nn::vi::DisplayModeInfo mode;
        NNT_ASSERT_RESULT_SUCCESS(nn::vi::GetDisplayMode(&mode, pDisplay));

        nn::vi::DisplayModeInfo modes[nn::vi::DisplayModeCountMax];
        int modeCount = nn::vi::ListDisplayModes(modes, sizeof(modes) / sizeof(modes[0]), pDisplay);

        for( int i = 0; i < modeCount; ++i )
        {
            NNT_ASSERT_RESULT_SUCCESS(nn::vi::SetDisplayMode(pDisplay, modes + i));

            nn::vi::DisplayModeInfo mode;
            std::memset(&mode, 0, sizeof(mode));

            NNT_ASSERT_RESULT_SUCCESS(nn::vi::GetDisplayMode(&mode, pDisplay));

            EXPECT_EQ(modes[i].width, mode.width);
            EXPECT_EQ(modes[i].height, mode.height);
            EXPECT_EQ(modes[i].refreshRate, mode.refreshRate);
            EXPECT_EQ(modes[i].mode, mode.mode);
        }

        // revert back to original mode
        NNT_ASSERT_RESULT_SUCCESS(nn::vi::SetDisplayMode(pDisplay, &mode));

        nn::vi::CloseDisplay(pDisplay);
    }
}

TEST_F(SetDisplayMode, External_Handheld_IcosaCopper)
{
    if( nn::ae::GetOperationMode() == nn::ae::OperationMode_Handheld )
    {
        nn::vi::Initialize();

        nn::vi::Display* pDisplay;
        NNT_ASSERT_RESULT_SUCCESS(nn::vi::OpenDisplay(&pDisplay, "External"));

        nn::vi::DisplayModeInfo mode{ 720, 480, 60.00f, nn::vi::StereoMode_None };
        NNT_ASSERT_RESULT_FAILURE(nn::vi::ResultDisplayDisconnected, nn::vi::SetDisplayMode(pDisplay, &mode));

        nn::vi::CloseDisplay(pDisplay);
    }
}

TEST_F(SetDisplayMode, Edid)
{
    nn::vi::Initialize();

    nn::vi::Display* pDisplay;
    NNT_ASSERT_RESULT_SUCCESS(nn::vi::OpenDisplay(&pDisplay, "Edid"));

    nn::vi::DisplayModeInfo mode{ 720, 480, 60.00f, nn::vi::StereoMode_None };
    NNT_EXPECT_RESULT_FAILURE(nn::vi::ResultNotSupported, nn::vi::SetDisplayMode(pDisplay, &mode));

    nn::vi::CloseDisplay(pDisplay);
}

TEST(SetDisplayModePreconditions, ViNotInitialized)
{
    nn::vi::Display* pDisplay = reinterpret_cast<nn::vi::Display*>(0x1234);

    nn::vi::DisplayModeInfo mode;
    ASSERT_DEATH_IF_SUPPORTED(nn::vi::SetDisplayMode(pDisplay, &mode), "");
}

TEST(SetDisplayModePreconditions, pModeNull)
{
    nn::vi::Initialize();

    nn::vi::Display* pDisplay = nullptr;
    NNT_EXPECT_RESULT_SUCCESS(nn::vi::OpenDefaultDisplay(&pDisplay));

    if( pDisplay != nullptr )
    {
        EXPECT_DEATH_IF_SUPPORTED(nn::vi::SetDisplayMode(pDisplay, nullptr), "");

        nn::vi::CloseDisplay(pDisplay);
    }

    nn::vi::Finalize();
}

TEST(SetDisplayModePreconditions, pDisplayNull)
{
    nn::vi::Initialize();

    nn::vi::DisplayModeInfo mode;
    ASSERT_DEATH_IF_SUPPORTED(nn::vi::SetDisplayMode(nullptr, &mode), "");

    nn::vi::Finalize();
}

TEST(SetDisplayModePreconditions, pDisplayInvalidHandle)
{
    nn::vi::Initialize();

    nn::vi::Display* pDisplay = nullptr;
    NNT_EXPECT_RESULT_SUCCESS(nn::vi::OpenDefaultDisplay(&pDisplay));

    if( pDisplay != nullptr )
    {
        nn::vi::CloseDisplay(pDisplay);

        nn::vi::DisplayModeInfo mode;
        EXPECT_DEATH_IF_SUPPORTED(nn::vi::SetDisplayMode(pDisplay, &mode), "");
    }

    nn::vi::Finalize();
}
