﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cmath>
#include <nnt.h>
#include <nnt/viUtil/testVi_Fixture.h>
#include <nn/vi.private.h>
#include <nn/ae.h>

typedef nnt::vi::util::Fixture ListDisplayModes;

namespace
{
    nn::vi::DisplayModeInfo g_ValidModes[] =
    {
        {
            720,
            480,
            60.00f,
            nn::vi::StereoMode_None
        },

        {
            1280,
            720,
            60.00f,
            nn::vi::StereoMode_None
        },

        {
            1920,
            1080,
            60.00f,
            nn::vi::StereoMode_None
        },
    };

    bool operator==(const nn::vi::DisplayModeInfo& lhs, const nn::vi::DisplayModeInfo& rhs)
    {
        return lhs.width == rhs.width &&
               lhs.height == rhs.height &&
               std::abs(lhs.refreshRate - rhs.refreshRate) <= .001f &&
               lhs.mode == rhs.mode;
    }

    bool AreAllModesValid(const nn::vi::DisplayModeInfo* modes, int modeCount)
    {
        for( int i = 0; i < modeCount; ++i )
        {
            bool isFound = false;
            for( int j = 0; !isFound && j < sizeof(g_ValidModes) / sizeof(g_ValidModes[0]); ++j )
            {
                isFound = modes[i] == g_ValidModes[j];
            }

            if( !isFound )
            {
                return false;
            }
        }

        return true;
    }
}

// TODO: Either allow nullptr or update documentation to include precondition
TEST_F(ListDisplayModes, DISABLED_pOutModesNull_Any)
{
    nn::vi::Initialize();

    nn::vi::Display* pDefaultDisplay;
    NNT_ASSERT_RESULT_SUCCESS(nn::vi::OpenDefaultDisplay(&pDefaultDisplay));

    nn::vi::Display* pExternalDisplay;
    NNT_ASSERT_RESULT_SUCCESS(nn::vi::OpenDisplay(&pExternalDisplay, "External"));

    nn::vi::Display* pEdidDisplay;
    NNT_ASSERT_RESULT_SUCCESS(nn::vi::OpenDisplay(&pEdidDisplay, "Edid"));

    EXPECT_EQ(0, nn::vi::ListDisplayModes(nullptr, 1, pDefaultDisplay));
    EXPECT_EQ(0, nn::vi::ListDisplayModes(nullptr, 1, pExternalDisplay));
    EXPECT_EQ(0, nn::vi::ListDisplayModes(nullptr, 1, pEdidDisplay));

    nn::vi::CloseDisplay(pEdidDisplay);
    nn::vi::CloseDisplay(pExternalDisplay);
    nn::vi::CloseDisplay(pDefaultDisplay);
}

TEST_F(ListDisplayModes, ZeroCount_Any)
{
    nn::vi::Initialize();

    nn::vi::Display* pDefaultDisplay;
    NNT_ASSERT_RESULT_SUCCESS(nn::vi::OpenDefaultDisplay(&pDefaultDisplay));

    nn::vi::Display* pExternalDisplay;
    NNT_ASSERT_RESULT_SUCCESS(nn::vi::OpenDisplay(&pExternalDisplay, "External"));

    nn::vi::Display* pEdidDisplay;
    NNT_ASSERT_RESULT_SUCCESS(nn::vi::OpenDisplay(&pEdidDisplay, "Edid"));

    nn::vi::DisplayModeInfo mode { 0, 0, 0.f, nn::vi::StereoMode_None };
    EXPECT_EQ(0, nn::vi::ListDisplayModes(&mode, 0, pDefaultDisplay));
    EXPECT_EQ(0, mode.width);
    EXPECT_EQ(0, mode.height);
    EXPECT_EQ(0.f, mode.refreshRate);
    EXPECT_EQ(nn::vi::StereoMode_None, mode.mode);

    EXPECT_EQ(0, nn::vi::ListDisplayModes(&mode, 0, pExternalDisplay));
    EXPECT_EQ(0, mode.width);
    EXPECT_EQ(0, mode.height);
    EXPECT_EQ(0.f, mode.refreshRate);
    EXPECT_EQ(nn::vi::StereoMode_None, mode.mode);

    EXPECT_EQ(0, nn::vi::ListDisplayModes(&mode, 0, pEdidDisplay));
    EXPECT_EQ(0, mode.width);
    EXPECT_EQ(0, mode.height);
    EXPECT_EQ(0.f, mode.refreshRate);
    EXPECT_EQ(nn::vi::StereoMode_None, mode.mode);

    nn::vi::CloseDisplay(pEdidDisplay);
    nn::vi::CloseDisplay(pExternalDisplay);
    nn::vi::CloseDisplay(pDefaultDisplay);
}

TEST_F(ListDisplayModes, Default_Any)
{
    nn::vi::Initialize();

    nn::vi::Display* pDisplay;
    NNT_ASSERT_RESULT_SUCCESS(nn::vi::OpenDefaultDisplay(&pDisplay));

    nn::vi::DisplayModeInfo modes[nn::vi::DisplayModeCountMax];
    int modeCount = nn::vi::ListDisplayModes(modes, sizeof(modes) / sizeof(modes[0]), pDisplay);
    ASSERT_NE(0, modeCount);
    EXPECT_TRUE(AreAllModesValid(modes, modeCount));

    nn::vi::CloseDisplay(pDisplay);
}

TEST_F(ListDisplayModes, External_Console_IcosaCopper)
{
    if( nn::ae::GetOperationMode() == nn::ae::OperationMode_Console )
    {
        nn::vi::Initialize();

        nn::vi::Display* pDisplay;
        NNT_ASSERT_RESULT_SUCCESS(nn::vi::OpenDisplay(&pDisplay, "External"));

        nn::vi::DisplayModeInfo modes[nn::vi::DisplayModeCountMax];
        int modeCount = nn::vi::ListDisplayModes(modes, sizeof(modes) / sizeof(modes[0]), pDisplay);
        ASSERT_NE(0, modeCount);
        EXPECT_TRUE(AreAllModesValid(modes, modeCount));

        nn::vi::CloseDisplay(pDisplay);
    }
}

TEST_F(ListDisplayModes, External_Handheld_IcosaCopper)
{
    if( nn::ae::GetOperationMode() == nn::ae::OperationMode_Handheld )
    {
        nn::vi::Initialize();

        nn::vi::Display* pDisplay;
        NNT_ASSERT_RESULT_SUCCESS(nn::vi::OpenDisplay(&pDisplay, "External"));

        nn::vi::DisplayModeInfo modes[nn::vi::DisplayModeCountMax];
        EXPECT_EQ(0, nn::vi::ListDisplayModes(modes, sizeof(modes) / sizeof(modes[0]), pDisplay));

        nn::vi::CloseDisplay(pDisplay);
    }
}

TEST_F(ListDisplayModes, Edid_IcosaCopper)
{
    nn::vi::Initialize();

    nn::vi::Display* pDisplay;
    NNT_ASSERT_RESULT_SUCCESS(nn::vi::OpenDisplay(&pDisplay, "Edid"));

    nn::vi::DisplayModeInfo modes[nn::vi::DisplayModeCountMax];
    int modeCount = nn::vi::ListDisplayModes(modes, sizeof(modes) / sizeof(modes[0]), pDisplay);
    ASSERT_NE(0, modeCount);
    EXPECT_TRUE(AreAllModesValid(modes, modeCount));

    nn::vi::CloseDisplay(pDisplay);
}

TEST(ListDisplayModesPreconditions, ViNotInitialized_Any)
{
    nn::vi::DisplayModeInfo modes[nn::vi::DisplayModeCountMax];
    nn::vi::Display* pDisplay = reinterpret_cast<nn::vi::Display*>(0x1234);
    ASSERT_DEATH_IF_SUPPORTED(nn::vi::ListDisplayModes(modes, sizeof(modes) / sizeof(modes[0]), pDisplay), "");
}

TEST(ListDisplayModesPreconditions, NegativeCount_Any)
{
    nn::vi::Initialize();

    nn::vi::Display* pDisplay = nullptr;
    NNT_EXPECT_RESULT_SUCCESS(nn::vi::OpenDefaultDisplay(&pDisplay));

    if( pDisplay != nullptr )
    {
        nn::vi::DisplayModeInfo modes[nn::vi::DisplayModeCountMax];
        EXPECT_DEATH_IF_SUPPORTED(nn::vi::ListDisplayModes(modes, -1, pDisplay), "");

        nn::vi::CloseDisplay(pDisplay);
    }

    nn::vi::Finalize();
}

TEST(ListDisplayModesPreconditions, pDisplayNull_Any)
{
    nn::vi::Initialize();

    nn::vi::DisplayModeInfo modes[nn::vi::DisplayModeCountMax];
    EXPECT_DEATH_IF_SUPPORTED(nn::vi::ListDisplayModes(modes, sizeof(modes) / sizeof(modes[0]), nullptr), "");

    nn::vi::Finalize();
}

TEST(ListDisplayModesPreconditions, pDisplayInvalidHandle_Any)
{
    nn::vi::Initialize();

    nn::vi::Display* pDisplay = nullptr;
    NNT_EXPECT_RESULT_SUCCESS(nn::vi::OpenDefaultDisplay(&pDisplay));

    if( pDisplay != nullptr )
    {
        nn::vi::CloseDisplay(pDisplay);

        nn::vi::DisplayModeInfo modes[nn::vi::DisplayModeCountMax];
        EXPECT_DEATH_IF_SUPPORTED(nn::vi::ListDisplayModes(modes, sizeof(modes) / sizeof(modes[0]), pDisplay), "");
    }

    nn::vi::Finalize();
}
