﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "MemoryPoolManager.h"

namespace framework{
    extern nns::gfx::GraphicsFramework g_Gfw;
}

namespace framework{ namespace detail{

    MemoryPoolHolder::MemoryPoolHolder() NN_NOEXCEPT
        : m_PoolType(nns::gfx::GraphicsFramework::MemoryPoolType::MemoryPoolType_CommandBuffer)
        , m_PoolSize(0)
        , m_TotalAllocatedSize(0)
        , m_TotalAllocatedSizeMax(0)
    {
    }

    void MemoryPoolHolder::Initialize(nns::gfx::GraphicsFramework::MemoryPoolType poolType, size_t poolSize) NN_NOEXCEPT
    {
        m_PoolType = poolType;
        m_PoolSize = poolSize;
        m_TotalAllocatedSize = 0;
        m_TotalAllocatedSizeMax = 0;

        UpdateAllocatedSize();
    }

    void MemoryPoolHolder::Finalize() NN_NOEXCEPT
    {
    }

    void MemoryPoolHolder::UpdateAllocatedSize() NN_NOEXCEPT
    {
        if(m_PoolSize > 0)
        {
            auto pAllocator = g_Gfw.GetMemoryPoolAllocator(m_PoolType);
            m_TotalAllocatedSize = m_PoolSize - pAllocator->GetTotalFreeSize();
            if(m_TotalAllocatedSize > m_TotalAllocatedSizeMax)
            {
                m_TotalAllocatedSizeMax = m_TotalAllocatedSize;
            }
        }
    }

    ptrdiff_t MemoryPoolHolder::Allocate(size_t size, size_t alignment) NN_NOEXCEPT
    {
        auto offset = g_Gfw.AllocatePoolMemory(m_PoolType, size, alignment);
        if(offset < 0)
        {
            return -1;
        }

        UpdateAllocatedSize();
        return offset;
    }

    void MemoryPoolHolder::Free(ptrdiff_t offset) NN_NOEXCEPT
    {
        g_Gfw.FreePoolMemory(m_PoolType, offset);

        UpdateAllocatedSize();
    }

    nn::gfx::MemoryPool* MemoryPoolHolder::GetPool() NN_NOEXCEPT
    {
        return g_Gfw.GetMemoryPool(m_PoolType);
    }

    size_t MemoryPoolHolder::GetTotalAllocatedSize() const NN_NOEXCEPT
    {
        return m_TotalAllocatedSize;
    }

    size_t MemoryPoolHolder::GetTotalAllocatedSizeMax() const NN_NOEXCEPT
    {
        return m_TotalAllocatedSizeMax;
    }

    //------------------------------------------------------------------------------

    void MemoryPoolManager::Initialize(const FrameworkMemoryPoolParameter& param) NN_NOEXCEPT
    {
        m_TextureMemoryPoolHolder       .Initialize(nns::gfx::MemoryPoolType_RenderTarget  , param.texturePoolSize       );
        m_CommandBufferMemoryPoolHolder .Initialize(nns::gfx::MemoryPoolType_CommandBuffer , param.commandBufferPoolSize );
        m_ConstantBufferMemoryPoolHolder.Initialize(nns::gfx::MemoryPoolType_ConstantBuffer, param.constantBufferPoolSize);
        m_ShaderMemoryPoolHolder        .Initialize(nns::gfx::MemoryPoolType_Shader        , param.shaderPoolSize        );
        m_DataMemoryPoolHolder          .Initialize(nns::gfx::MemoryPoolType_Data          , param.dataPoolSize          );
    }

    void MemoryPoolManager::Finalize() NN_NOEXCEPT
    {
        m_TextureMemoryPoolHolder       .Finalize();
        m_CommandBufferMemoryPoolHolder .Finalize();
        m_ConstantBufferMemoryPoolHolder.Finalize();
        m_ShaderMemoryPoolHolder        .Finalize();
        m_DataMemoryPoolHolder          .Finalize();
    }

    MemoryPoolHolder* MemoryPoolManager::GetTextureMemoryPoolHolder       () NN_NOEXCEPT{ return &m_TextureMemoryPoolHolder       ; }
    MemoryPoolHolder* MemoryPoolManager::GetCommandBufferMemoryPoolHolder () NN_NOEXCEPT{ return &m_CommandBufferMemoryPoolHolder ; }
    MemoryPoolHolder* MemoryPoolManager::GetConstantBufferMemoryPoolHolder() NN_NOEXCEPT{ return &m_ConstantBufferMemoryPoolHolder; }
    MemoryPoolHolder* MemoryPoolManager::GetShaderMemoryPoolHolder        () NN_NOEXCEPT{ return &m_ShaderMemoryPoolHolder        ; }
    MemoryPoolHolder* MemoryPoolManager::GetDataMemoryPoolHolder          () NN_NOEXCEPT{ return &m_DataMemoryPoolHolder          ; }

}}
