﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "Hid.h"

#include <nn/nn_Log.h>
#include <nn/hid/hid_Npad.h>
#include <nn/hid/hid_NpadJoy.h>
#include <nn/hid/hid_DebugPad.h>

//#define FRAMEWORK_LOG_HID(...) NN_LOG("[hid]" __VA_ARGS__)

#ifndef FRAMEWORK_LOG_HID
#define FRAMEWORK_LOG_HID(...)
#endif

namespace {
    nn::hid::NpadIdType g_NpadIds[] = {
        nn::hid::NpadId::No1,
        nn::hid::NpadId::No2,
        nn::hid::NpadId::No3,
        nn::hid::NpadId::No4,
        nn::hid::NpadId::Handheld
    };
    const int NpadIdCountMax = sizeof(g_NpadIds) / sizeof(nn::hid::NpadIdType);

    HidButtonHandleParameter g_ButtonHandleParameter ={};

    HidButtons<int> g_ButtonPressedFrameCounts = {};
    HidButtons<int> g_PrevButtonInput = {};

    HidButtonState  g_CurrentButtonState = {};


    //----------------------------------------------------------------------------------------------
    // 押されたボタンを取得
    // いずれかのコントローラで押されたボタンは 1、それ以外は 0 になる
    template<typename TButtonIndex, typename TState>
    void ExtractButtonInput(HidButtons<int>* pOutValue, const TState& state) NN_NOEXCEPT
    {
#define FRAMEWORK_HID_EXTRACT_BUTTON_INPUT(VAR, BUTTON) pOutValue->VAR |= state.Test(TButtonIndex::BUTTON::Index) ? 1 : 0;
        FRAMEWORK_HID_EXTRACT_BUTTON_INPUT(A    , A    );
        FRAMEWORK_HID_EXTRACT_BUTTON_INPUT(B    , B    );
        FRAMEWORK_HID_EXTRACT_BUTTON_INPUT(X    , X    );
        FRAMEWORK_HID_EXTRACT_BUTTON_INPUT(Y    , Y    );
        FRAMEWORK_HID_EXTRACT_BUTTON_INPUT(L    , L    );
        FRAMEWORK_HID_EXTRACT_BUTTON_INPUT(L    , ZL   );
        FRAMEWORK_HID_EXTRACT_BUTTON_INPUT(R    , R    );
        FRAMEWORK_HID_EXTRACT_BUTTON_INPUT(R    , ZR   );
        FRAMEWORK_HID_EXTRACT_BUTTON_INPUT(Left , Left );
        FRAMEWORK_HID_EXTRACT_BUTTON_INPUT(Up   , Up   );
        FRAMEWORK_HID_EXTRACT_BUTTON_INPUT(Right, Right);
        FRAMEWORK_HID_EXTRACT_BUTTON_INPUT(Down , Down );
#undef FRAMEWORK_HID_EXTRACT_BUTTON_INPUT
    }

    template<typename TState>
    void ExtractNpadInput(HidButtons<int>* pOutValue, const TState& state) NN_NOEXCEPT
    {
        ExtractButtonInput<nn::hid::NpadJoyButton>(pOutValue, state.buttons);
    }

    template<typename TState>
    void ExtractDebugPadInput(HidButtons<int>* pOutValue, const TState& state) NN_NOEXCEPT
    {
        ExtractButtonInput<nn::hid::DebugPadButton>(pOutValue, state.buttons);
    }

    void ExtractCurrentInput(HidButtons<int>* pOutValue) NN_NOEXCEPT
    {
        // Npad
        for(int i = 0; i < NpadIdCountMax; i++)
        {
            //現在有効な操作形態(NpadStyleSet)を取得
            nn::hid::NpadStyleSet style = nn::hid::GetNpadStyleSet(g_NpadIds[i]);
            // フルキー操作
            if (style.Test<nn::hid::NpadStyleFullKey>() == true)
            {
                nn::hid::NpadFullKeyState state = {};
                nn::hid::GetNpadState(&state, g_NpadIds[i]);
                ExtractNpadInput(pOutValue, state);
            }
            // ジョイコン操作
            else if (style.Test<nn::hid::NpadStyleJoyDual>() == true)
            {
                nn::hid::NpadJoyDualState state = {};
                nn::hid::GetNpadState(&state, g_NpadIds[i]);
                ExtractNpadInput(pOutValue, state);
            }
            // 携帯機コントローラー操作
            else if (style.Test<nn::hid::NpadStyleHandheld>() == true)
            {
                nn::hid::NpadHandheldState state = {};
                nn::hid::GetNpadState(&state, g_NpadIds[i]);
                ExtractNpadInput(pOutValue, state);
            }
        }

        // DebugPad
        {
            nn::hid::DebugPadState state = {};
            nn::hid::GetDebugPadState(&state);
            ExtractDebugPadInput(pOutValue, state);
        }
    }

    //----------------------------------------------------------------------------------------------

    // 押され続けているフレームを加算
    void UpdatePressedFrameCount(HidButtons<int>* pFrameCount, const HidButtons<int>& currentInput) NN_NOEXCEPT
    {
#define FRAMEWORK_HID_UPDATE_PRESSED_FRAME_COUNT(BUTTON) pFrameCount->BUTTON = (pFrameCount->BUTTON + currentInput.BUTTON) * currentInput.BUTTON;
        FRAMEWORK_HID_UPDATE_PRESSED_FRAME_COUNT(A);
        FRAMEWORK_HID_UPDATE_PRESSED_FRAME_COUNT(B);
        FRAMEWORK_HID_UPDATE_PRESSED_FRAME_COUNT(X);
        FRAMEWORK_HID_UPDATE_PRESSED_FRAME_COUNT(Y);
        FRAMEWORK_HID_UPDATE_PRESSED_FRAME_COUNT(L);
        FRAMEWORK_HID_UPDATE_PRESSED_FRAME_COUNT(R);
        FRAMEWORK_HID_UPDATE_PRESSED_FRAME_COUNT(Left);
        FRAMEWORK_HID_UPDATE_PRESSED_FRAME_COUNT(Up);
        FRAMEWORK_HID_UPDATE_PRESSED_FRAME_COUNT(Right);
        FRAMEWORK_HID_UPDATE_PRESSED_FRAME_COUNT(Down);
#undef FRAMEWORK_HID_UPDATE_PRESSED_FRAME_COUNT
    }

    //----------------------------------------------------------------------------------------------

    void GetCurrentButtonStateImpl(bool* pOutIsPressed, bool* pOutIsTriggered, bool* pOutIsDown, bool* pOutIsUp, int pressedCount, bool currentInput, bool prevInput) NN_NOEXCEPT
    {
        *pOutIsPressed = currentInput;
        *pOutIsDown = currentInput && !prevInput;
        *pOutIsUp   = !currentInput && prevInput;

        int begCount = g_ButtonHandleParameter.repeatTriggerFrameBeginCount;
        int interval = g_ButtonHandleParameter.repeatTriggerFrameInterval;

        *pOutIsTriggered = false;
        if(pressedCount == 1)
        {
            *pOutIsTriggered = true;
        }
        else if(begCount > 0)
        {
            if(pressedCount >= begCount &&
                ((pressedCount - begCount) % interval) == 0
                )
            {
                *pOutIsTriggered = true;
            }
        }
    }

    //----------------------------------------------------------------------------------------------

    void UpdateCurrentButtonState(HidButtonState* pState, const HidButtons<int>& frameCounts, const HidButtons<int>& currentInput, const HidButtons<int>& prevInput) NN_NOEXCEPT
    {
        HidButtonState state = {};
#define FRAMEWORK_HID_GET_CURRENT_BUTTON_STATE(BUTTON) \
    GetCurrentButtonStateImpl(                         \
        &state.isPressed.BUTTON, \
        &state.isTriggered.BUTTON, \
        &state.isDown.BUTTON, \
        &state.isUp.BUTTON, \
        frameCounts.BUTTON, \
        currentInput.BUTTON != 0, \
        prevInput.BUTTON != 0 \
    );
        FRAMEWORK_HID_GET_CURRENT_BUTTON_STATE(A);
        FRAMEWORK_HID_GET_CURRENT_BUTTON_STATE(B);
        FRAMEWORK_HID_GET_CURRENT_BUTTON_STATE(X);
        FRAMEWORK_HID_GET_CURRENT_BUTTON_STATE(Y);
        FRAMEWORK_HID_GET_CURRENT_BUTTON_STATE(L);
        FRAMEWORK_HID_GET_CURRENT_BUTTON_STATE(R);
        FRAMEWORK_HID_GET_CURRENT_BUTTON_STATE(Left);
        FRAMEWORK_HID_GET_CURRENT_BUTTON_STATE(Up);
        FRAMEWORK_HID_GET_CURRENT_BUTTON_STATE(Right);
        FRAMEWORK_HID_GET_CURRENT_BUTTON_STATE(Down);
#undef FRAMEWORK_HID_GET_CURRENT_BUTTON_STATE
        *pState = state;
    }

    //----------------------------------------------------------------------------------------------
}

void Hid::Initialize() NN_NOEXCEPT
{
    nn::hid::InitializeNpad();
    nn::hid::SetSupportedNpadStyleSet(nn::hid::NpadStyleFullKey::Mask | nn::hid::NpadStyleJoyDual::Mask | nn::hid::NpadStyleHandheld::Mask);
    nn::hid::SetSupportedNpadIdType(g_NpadIds, NpadIdCountMax);

    nn::hid::InitializeDebugPad();
}

void Hid::Finalize() NN_NOEXCEPT
{
}

void Hid::SetParameter(const HidButtonHandleParameter& param) NN_NOEXCEPT
{
    g_ButtonHandleParameter = param;
}

void Hid::Update() NN_NOEXCEPT
{
    HidButtons<int> currentInput = {};
    ExtractCurrentInput(&currentInput);
    UpdatePressedFrameCount(&g_ButtonPressedFrameCounts, currentInput);
    UpdateCurrentButtonState(&g_CurrentButtonState, g_ButtonPressedFrameCounts, currentInput, g_PrevButtonInput);
    g_PrevButtonInput = currentInput;
}


HidButtonState Hid::GetButtonState() NN_NOEXCEPT
{
    return g_CurrentButtonState;
}


