﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <vector>
#include <string>
#include <sstream>
#include <nn/nn_Assert.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_ScopeExit.h>
#include <nnt.h>

#include "testVi_Context.h"
#include "testVi_Macro.h"
#include "testVi_MemoryManagement.h"
#include "../common/testVi_Native.h"

namespace {

    uint8_t GetChannelValue(int x, int length, uint8_t colorMax) NN_NOEXCEPT
    {
        NN_ASSERT_GREATER(length, 0);
        float v = std::floorf(static_cast<float>(colorMax) * static_cast<float>(x) / static_cast<float>(length - 1));
        if(v < 0)
        {
            return v;
        }
        if(v > 255)
        {
            return 255;
        }
        return static_cast<uint8_t>(v);
    }

    std::vector<uint8_t> GetTestImageBuffer(int width, int height, nn::vi::ImageTransform transform, uint8_t colorMax, uint8_t alpha) NN_NOEXCEPT
    {
        std::vector<uint8_t> out;
        out.resize(4 * width * height);

        uint8_t b = 0;

        for(int y = 0; y < height; y++)
        {
            uint8_t g = GetChannelValue(
                (transform == nn::vi::ImageTransform_YFlip) ? (height - 1 - y) : y,
                height,
                colorMax
            );

            for(int x = 0; x < width; x++)
            {
                uint8_t r = GetChannelValue(x, width, colorMax);

                uint8_t* p = out.data() + 4 * (x + (width * y));
                p[0] = r;
                p[1] = g;
                p[2] = b;
                p[3] = alpha;
            }
        }

        return out;
    }

    void CheckImageBuffer(
        const std::vector<uint8_t>& a,
        const std::vector<uint8_t>& b,
        int width,
        int height,
        int tolerance
    ) NN_NOEXCEPT
    {
        NN_ABORT_UNLESS_EQUAL(a.size(), 4 * width * height);
        NN_ABORT_UNLESS_EQUAL(b.size(), 4 * width * height);

        bool isOk = true;
        for(size_t i = 0; i < a.size(); i++)
        {
            if(std::labs(a[i] - b[i]) > tolerance)
            {
                isOk = false;
                NN_LOG("check img buffer: diff = %d (i=%lld)\n", static_cast<int>(a[i] - b[i]), i);
                break;
            }
        }

        if(!isOk)
        {
            static const int ThumbDeltaX = 100;
            static const int ThumbDeltaY = 100;
            auto addChar = [](std::stringstream& ss, uint8_t r, uint8_t g)
            {
                if(r < 128)
                {
                    if(g < 128)
                    {
                        ss << ".";
                    }
                    else
                    {
                        ss << "|";
                    }
                }
                else
                {
                    if(g < 128)
                    {
                        ss << "-";
                    }
                    else
                    {
                        ss << "+";
                    }
                }
            };

            NN_LOG("expected image:\n");
            for(int y = 0; y < height; y += ThumbDeltaY)
            {
                std::stringstream ss;
                for(int x = 0; x < width; x += ThumbDeltaX)
                {
                    auto* p = a.data() + 4 * (x + y * width);
                    addChar(ss, p[0], p[1]);
                }

                auto str = ss.str();
                NN_LOG("  %s\n", str.c_str());
            }

            NN_LOG("actual image:\n");
            for(int y = 0; y < height; y += ThumbDeltaY)
            {
                std::stringstream ss;
                for(int x = 0; x < width; x += ThumbDeltaX)
                {
                    auto* p = b.data() + 4 * (x + y * width);
                    addChar(ss, p[0], p[1]);
                }

                auto str = ss.str();
                NN_LOG("  %s\n", str.c_str());
            }
            FAIL();
        }
        else
        {
            SUCCEED();
        }
    }

    void CheckContentParameter(
        ContextExt& context,
        nn::vi::fbshare::SharedBufferHandle hBuffer,
        int index,
        nn::vi::LayerStackFlagType expectedStacks,
        const nn::vi::CropRegion& expectedCrop,
        uint32_t expectedTransform,
        int32_t expectedPresentInterval
    ) NN_NOEXCEPT
    {
        nn::vi::LayerStackFlagType stacks = 0;
        nn::vi::CropRegion crop = {};
        int32_t scalingMode = 0;
        uint32_t transform = 0;
        int32_t presentInterval = 0;
        NN_ABORT_UNLESS_RESULT_SUCCESS(context.GetSharedFrameBufferContentParameter(&stacks, &crop, &scalingMode, &transform, &presentInterval, hBuffer, index));
        EXPECT_EQ(expectedStacks, stacks);
        if(std::memcmp(&expectedCrop, &crop, sizeof(nn::vi::CropRegion)) != 0)
        {
            NN_LOG("  expected crop = {%d, %d; %d, %d}\n", expectedCrop.x, expectedCrop.y, expectedCrop.width, expectedCrop.height);
            NN_LOG("  actual   crop = {%d, %d; %d, %d}\n", crop.x, crop.y, crop.width, crop.height);
            EXPECT_EQ(expectedCrop.x, crop.x);
            EXPECT_EQ(expectedCrop.y, crop.y);
            EXPECT_EQ(expectedCrop.width, crop.width);
            EXPECT_EQ(expectedCrop.height, crop.height);
        }
        EXPECT_EQ(0, scalingMode);
        EXPECT_EQ(expectedTransform, transform);
        EXPECT_EQ(expectedPresentInterval, presentInterval);
    }
}

// テストの実行関数
namespace {

    struct TestSetImage
    {
        static const int Width = nn::vi::fbshare::SharedFrameBufferWidth;
        static const int Height = nn::vi::fbshare::SharedFrameBufferHeight;
        static const int FrameBufferCount = 4;
        static const int SlotIndex = 0;
        static const nn::vi::LayerStackFlagType LayerStacks = nn::vi::ValidLayerStackFlags;
        static const int DestTextureIndex = 0;

        struct Parameter
        {
        public:
            Parameter() NN_NOEXCEPT
            {
                this->srcTransform = nn::vi::ImageTransform_None;
                this->dstTransform = nn::vi::ImageTransform_None;

                this->srcAlpha     = 255;
                this->alphaOption  = nn::vi::fbshare::SharedTextureAlphaOption_Opaque;
                this->colorOption  = nn::vi::fbshare::SharedTextureColorOption_None;
            }

        public:
            nn::vi::ImageTransform srcTransform;
            nn::vi::ImageTransform dstTransform;

            uint8_t srcAlpha;
            nn::vi::fbshare::SharedTextureAlphaOption alphaOption;
            nn::vi::fbshare::SharedTextureColorOption colorOption;
        };

        static void Test(Parameter& param) NN_NOEXCEPT
        {
            ContextExt context0({"server"});
            context0.ConnectService();

            // setup SharedBuffer
            auto hBuffer = context0.CreateSharedBuffer(SlotIndex, FrameBufferCount);

            auto srcImg    = GetTestImageBuffer(Width, Height, param.srcTransform, 255, param.srcAlpha);
            auto expectImg = GetTestImageBuffer(
                Width,
                Height,
                nn::vi::ImageTransform_None, // 最終的に読込む画像は左上原点になる
                (param.colorOption == nn::vi::fbshare::SharedTextureColorOption_PreMultipledAlpha) ? param.srcAlpha : 255,
                (param.alphaOption == nn::vi::fbshare::SharedTextureAlphaOption_Opaque) ? 255 : param.srcAlpha
            );

            // 書込み
            nn::vi::fbshare::SharedTextureOption dstOption = {};
            dstOption.alphaOption = param.alphaOption;
            dstOption.colorOption = param.colorOption;
            dstOption.transform   = param.dstTransform;
            dstOption.stacks      = LayerStacks;
            nn::vi::ImageTransform srcTransform = {};
            srcTransform = param.srcTransform;
            NN_ABORT_UNLESS_RESULT_SUCCESS(context0.SetDetachedSharedBufferImage(hBuffer, DestTextureIndex, srcImg.data(), srcImg.size(), dstOption, srcTransform));

            // 読み出し
            std::vector<uint8_t> dstImg;
            dstImg.resize(srcImg.size());
            size_t imgSize = 0;
            NN_ABORT_UNLESS_RESULT_SUCCESS(context0.GetDetachedSharedBufferImage(&imgSize, dstImg.data(), dstImg.size(), hBuffer, DestTextureIndex));

            // 許容誤差
            int tol = 0;
            if(param.colorOption == nn::vi::fbshare::SharedTextureColorOption_PreMultipledAlpha)
            {
                tol = 1;
            }

            // チェック
            CheckImageBuffer(expectImg, dstImg, Width, Height, tol);
            CheckContentParameter(context0, hBuffer, DestTextureIndex, LayerStacks, { 0, 0, 0, 0 }, param.dstTransform, 1);

            context0.DisconnectService();
            context0.CleanSharedBuffer(SlotIndex);
        }
    };
}

NNT_VI_TEST_FUNCTIONAL(SetImage_Default)
{
    TestSetImage::Parameter p = {};
    TestSetImage::Test(p);
}

NNT_VI_TEST_FUNCTIONAL(SetImage_SourceYFlip)
{
    TestSetImage::Parameter p = {};
    p.srcTransform = nn::vi::ImageTransform_YFlip;
    TestSetImage::Test(p);
}

NNT_VI_TEST_FUNCTIONAL(SetImage_DestinationYFlip)
{
    TestSetImage::Parameter p = {};
    p.dstTransform = nn::vi::ImageTransform_YFlip;
    TestSetImage::Test(p);
}

NNT_VI_TEST_FUNCTIONAL(SetImage_SourceDestinationYFlip)
{
    TestSetImage::Parameter p = {};
    p.srcTransform = nn::vi::ImageTransform_YFlip;
    p.dstTransform = nn::vi::ImageTransform_YFlip;
    TestSetImage::Test(p);
}

NNT_VI_TEST_FUNCTIONAL(SetImage_SourceAlpha)
{
    TestSetImage::Parameter p = {};
    p.srcAlpha = 100;
    p.alphaOption = nn::vi::fbshare::SharedTextureAlphaOption_None;
    TestSetImage::Test(p);
}

NNT_VI_TEST_FUNCTIONAL(SetImage_Opaque)
{
    TestSetImage::Parameter p = {};
    p.srcAlpha = 100;
    p.alphaOption = nn::vi::fbshare::SharedTextureAlphaOption_Opaque;
    TestSetImage::Test(p);
}

NNT_VI_TEST_FUNCTIONAL(SetImage_SourceColor)
{
    TestSetImage::Parameter p = {};
    p.srcAlpha = 100;
    p.colorOption = nn::vi::fbshare::SharedTextureColorOption_None;
    TestSetImage::Test(p);
}

NNT_VI_TEST_FUNCTIONAL(SetImage_PremultipledAlpha)
{
    TestSetImage::Parameter p = {};
    p.srcAlpha = 100;
    p.colorOption = nn::vi::fbshare::SharedTextureColorOption_PreMultipledAlpha;
    TestSetImage::Test(p);
}

