﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_ScopeExit.h>

#include "testVi_Context.h"
#include "testVi_Macro.h"
#include "testVi_MemoryManagement.h"
#include "../common/testVi_Native.h"
#include "testVi_NvnUtility.h"

static NN_ALIGNAS(4096) char g_CommandStorage[10 * 4096];
static NN_ALIGNAS(4096) char g_ControlStorage[4096];

static const int DefaultLoopCount  = 10;
static const int DefaultFrameCount = 10;

NNT_VI_TEST_FUNCTIONAL(ImportSharedBuffer)
{
    static const int LoopCount = DefaultLoopCount;
    static const int SlotIndex = 0;
    static const int FrameBufferCount = 4;
    static const int FrameCount = DefaultFrameCount;
    NN_LOG("ImportSharedBuffer-------------------------------\n");

    for(int i = 0; i < LoopCount; i++)
    {
        ContextExt context0({"server"});
        Context context1({"client"});

        context0.ConnectService();
        context1.ConnectService();

        // setup SharedLowLevelLayer
        auto layerId = context0.GetDefaultLayerId();
        NN_ABORT_UNLESS_RESULT_SUCCESS(context0.BindLowLevelLayerToManagedLayer(layerId));

        // setup SharedBuffer
        auto hBuffer = context0.CreateSharedBuffer(SlotIndex, FrameBufferCount);
        NN_ABORT_UNLESS_RESULT_SUCCESS(context0.ConnectLowLevelLayerToSharedBuffer(layerId, hBuffer));

        // setup SharedLayer
        nn::vi::fbshare::SharedLayerHandle hLayer = {};
        nn::vi::LayerStackFlagType stacks = (1 << nn::vi::LayerStack_Default);
        NN_ABORT_UNLESS_RESULT_SUCCESS(context0.CreateSharedLayer(&hLayer, nn::applet::GetAppletResourceUserId()));
        NN_ABORT_UNLESS_RESULT_SUCCESS(context0.SetSharedLayerLayerStack(hLayer, stacks));

        // attach
        NN_ABORT_UNLESS_RESULT_SUCCESS(context0.AttachSharedLayerToLowLevelLayer(hLayer, layerId, 0, 1, 2, 3));

        // open SharedLayer
        NN_ABORT_UNLESS_RESULT_SUCCESS(context1.OpenSharedLayer(hLayer));
        NN_ABORT_UNLESS_RESULT_SUCCESS(context1.ConnectSharedLayer(hLayer));
        nn::os::SystemEventType acquirableEvent = {};
        NN_ABORT_UNLESS_RESULT_SUCCESS(context1.GetSharedFrameBufferAcquirableEvent(&acquirableEvent, hLayer));
        NN_UTIL_SCOPE_EXIT{ nn::os::DestroySystemEvent(&acquirableEvent); };

        // export SharedBuffer to me
        NN_ABORT_UNLESS_RESULT_SUCCESS(context0.RegisterSharedBufferImporterAruid(hBuffer, nn::applet::GetAppletResourceUserId()));

        // get hMem id
        nn::vi::native::NativeMemoryHandleId memId = {};
        size_t memSize = 0;
        nn::vi::fbshare::SharedMemoryPoolLayout layout = {};
        NN_ABORT_UNLESS_RESULT_SUCCESS(context1.GetSharedBufferMemoryHandleId(&memId, &memSize, &layout, hBuffer));

        NN_LOG("importing memory:\n");
        NN_LOG("  id   : %u\n", memId);
        NN_LOG("  size : %llu(%lluKB)\n", memSize, memSize / 1024);
        NN_LOG("  count: %d\n", layout.count);
        for(int i = 0; i < layout.count; i++)
        {
            auto& e = layout.entries[i];
            NN_LOG("    texture[%d]: %dx%d %llu-%llu(%llu bytes)\n", i, e.width, e.height, e.offset, e.offset + e.size, e.size);
        }

        // import
        NvRmMemHandle hMem;
        uint64_t hMemValue = 0;
        NN_ABORT_UNLESS_EQUAL(NvRmMemHandleFromFd(static_cast<int>(memId._data), &hMem), NvSuccess);
        std::memcpy(&hMemValue, &hMem, std::min(sizeof(hMem), sizeof(hMemValue)));

        uint64_t importSize = NvRmMemGetSize(hMem);

        NN_LOG("imported memory:\n");
        NN_LOG("  hMem : %u\n", hMem);
        NN_LOG("  size : %llu(%lluKB)\n", importSize, importSize / 1024);

        // init nvn
        NNT_VI_SCOPED_DEVICE(device);
        NNT_VI_SCOPED_COMMANDMEMORYPOOL(commandPool, device, g_CommandStorage, sizeof(g_CommandStorage));
        NNT_VI_SCOPED_QUEUE(queue, device);
        NNT_VI_SCOPED_COMMANDBUFFER(commandBuffer, device);
        NNT_VI_LOAD_PRIVATE_PROC(device);

        // create physical pool
        NN_LOG("Creating imported memory pool\n");
        NVNmemoryPool importedPool;
        {
            NVNmemoryPoolBuilder builder;
            nvnMemoryPoolBuilderSetDevice(&builder, &device);
            nvnMemoryPoolBuilderSetDefaults(&builder);
            nvnMemoryPoolBuilderSetFlags(&builder,
                NVN_MEMORY_POOL_FLAGS_CPU_NO_ACCESS_BIT |
                NVN_MEMORY_POOL_FLAGS_PHYSICAL_BIT |
                NVN_MEMORY_POOL_FLAGS_GPU_NO_ACCESS_BIT);
            nvnMemoryPoolBuilderSetNativeHandleNVX(&builder, hMemValue);
            NN_ABORT_UNLESS(nvnMemoryPoolInitialize(&importedPool, &builder));
        }

        // create virtual pool
        NN_LOG("Creating virtual memory pool\n");
        NVNmemoryPool virtualPool;
        {
            NVNtextureBuilder texBuilder;
            nvnTextureBuilderSetDevice(&texBuilder, &device);
            nvnTextureBuilderSetDefaults(&texBuilder);
            nvnTextureBuilderSetFlags(&texBuilder,
                NVN_TEXTURE_FLAGS_DISPLAY_BIT |
                NVN_TEXTURE_FLAGS_IMAGE_BIT);
                //NVN_TEXTURE_FLAGS_COMPRESSIBLE_BIT |
                //NVN_TEXTURE_FLAGS_LINEAR_BIT |
            nvnTextureBuilderSetFormat(&texBuilder, NVN_FORMAT_RGBA8_SRGB);

            NVNmemoryPoolBuilder builder;
            nvnMemoryPoolBuilderSetDevice(&builder, &device);
            nvnMemoryPoolBuilderSetFlags(&builder,
                NVN_MEMORY_POOL_FLAGS_CPU_NO_ACCESS_BIT |
                NVN_MEMORY_POOL_FLAGS_GPU_CACHED_BIT |
                NVN_MEMORY_POOL_FLAGS_VIRTUAL_BIT);
            nvnMemoryPoolBuilderSetStorage(&builder, nullptr, memSize);
            nvnMemoryPoolInitialize(&virtualPool, &builder);

            NVNmappingRequest req;
            req.physicalPool = &importedPool;
            req.physicalOffset = 0;
            req.virtualOffset = 0;
            req.size = static_cast<uint64_t>(memSize);
            req.storageClass = nvnTextureBuilderGetStorageClass(&texBuilder);

            NN_ABORT_UNLESS(nvnMemoryPoolMapVirtual(&virtualPool, 1, &req));
        }

        // create texture
        NVNtexture texture[FrameBufferCount];
        for(int i = 0; i < FrameBufferCount; i++)
        {
            NN_LOG("Creating texture[%d]\n", i);
            NVNtextureBuilder texBuilder;
            nvnTextureBuilderSetDevice(&texBuilder, &device);
            nvnTextureBuilderSetDefaults(&texBuilder);
            nvnTextureBuilderSetFlags(&texBuilder,
                NVN_TEXTURE_FLAGS_DISPLAY_BIT |
                NVN_TEXTURE_FLAGS_IMAGE_BIT);
                //NVN_TEXTURE_FLAGS_COMPRESSIBLE_BIT |
                //NVN_TEXTURE_FLAGS_LINEAR_BIT |
            nvnTextureBuilderSetFormat(&texBuilder, NVN_FORMAT_RGBA8_SRGB);
            nvnTextureBuilderSetWidth(&texBuilder, nn::vi::fbshare::SharedFrameBufferWidth);
            nvnTextureBuilderSetHeight(&texBuilder, nn::vi::fbshare::SharedFrameBufferHeight);
            nvnTextureBuilderSetTarget(&texBuilder, NVN_TEXTURE_TARGET_2D);
            NN_ABORT_UNLESS_GREATER_EQUAL(nn::vi::fbshare::SharedFrameBufferSize, nvnTextureBuilderGetStorageSize(&texBuilder));
            nvnTextureBuilderSetStorage(&texBuilder, &virtualPool, nn::vi::fbshare::SharedFrameBufferSize * i);
            nvnTextureInitialize(&texture[i], &texBuilder);
        }

        // create texture view
        NVNtextureView texView;
        {
            NN_LOG("Creating textureView\n");
            nvnTextureViewSetDefaults(&texView);
            nvnTextureViewSetFormat(&texView, NVN_FORMAT_RGBA8_SRGB);
            nvnTextureViewSetTarget(&texView, NVN_TEXTURE_TARGET_2D);
        }

        for(int frame = 0; frame < FrameCount; frame++)
        {
            NN_LOG("Frame %d\n", frame);
            nn::os::WaitSystemEvent(&acquirableEvent);
            int index = -1;
            nn::vi::native::NativeSync dispSync = {};
            nn::vi::native::NativeSync appSync = {};
            nn::vi::fbshare::SharedLayerTextureIndexList indexList;
            NN_ABORT_UNLESS_RESULT_SUCCESS(context1.AcquireSharedFrameBuffer(&index, &dispSync, &indexList, hLayer));
            NN_ABORT_UNLESS_RANGE(index, 0, FrameBufferCount);
            NN_LOG("  texture index = %d\n", index);
            nnt::vi::WaitSync(dispSync);

            NVNtexture* pTargetTexture = &texture[index];

            nvnCommandBufferAddCommandMemory(&commandBuffer, &commandPool, 0, sizeof(g_CommandStorage));
            nvnCommandBufferAddControlMemory(&commandBuffer, g_ControlStorage, sizeof(g_ControlStorage));
            nvnCommandBufferBeginRecording(&commandBuffer);

            // clear
            {
                float angle = 0.5f * 2.f * 3.14f * frame / 60.f;
                float r = 0.5f * std::cos(angle               ) + 0.5f;
                float g = 0.5f * std::cos(angle + 3.14f       ) + 0.5f;
                float b = 0.5f * std::cos(angle + 3.14f * 0.5f) + 0.5f;
                float clearColor[4] = { r, g, b, 1.f };
                NVNcopyRegion clearRegion;
                clearRegion.xoffset = 0;
                clearRegion.yoffset = 0;
                clearRegion.zoffset = 0;
                clearRegion.width  = nn::vi::fbshare::SharedFrameBufferWidth;
                clearRegion.height = nn::vi::fbshare::SharedFrameBufferHeight;
                clearRegion.depth  = 1;
                nvnCommandBufferClearTexture(&commandBuffer, pTargetTexture, &texView, &clearRegion, clearColor, NVN_CLEAR_COLOR_MASK_RGBA);
            }

            // clear
            {
                float origX = nn::vi::fbshare::SharedFrameBufferWidth / 2;
                float origY = nn::vi::fbshare::SharedFrameBufferHeight / 2;
                float length = 300;
                float edge = 50;
                float angle = 0.5f * 2.f * 3.14f * frame / 60.f;
                float x = length * std::cos(angle) + origX;
                float y = length * std::sin(angle) + origY;
                float clearColor[4] = { 1.f, 1.f, 1.f, 1.f };
                NVNcopyRegion clearRegion;
                clearRegion.xoffset = x - edge * 0.5f;
                clearRegion.yoffset = y - edge * 0.5f;
                clearRegion.zoffset = 0;
                clearRegion.width  = edge;
                clearRegion.height = edge;
                clearRegion.depth  = 1;
                nvnCommandBufferClearTexture(&commandBuffer, pTargetTexture, &texView, &clearRegion, clearColor, NVN_CLEAR_COLOR_MASK_RGBA);
            }


            auto hCommand = nvnCommandBufferEndRecording(&commandBuffer);
            nvnQueueSubmitCommands(&queue, 1, &hCommand);
            nvnQueueFinish(&queue);

            NN_ABORT_UNLESS_RESULT_SUCCESS(context1.PresentSharedFrameBuffer(hLayer, index, appSync));
        }


        for(int i = 0; i < FrameBufferCount; i++)
        {
            nvnTextureFinalize(&texture[i]);
        }
        nvnMemoryPoolFinalize(&virtualPool);
        nvnMemoryPoolFinalize(&importedPool);

        NvRmMemHandleFree(hMem);

        context0.DisconnectService();
        context1.DisconnectService();
        context0.CleanSharedBuffer(SlotIndex);
    }
}// NOLINT(impl/function_size)
