﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstdlib>
#include <nn/vi.private.h>
#include <nn/nn_Assert.h>
#include <nn/nn_Abort.h>
#include <nn/nn_Log.h>
#include <nn/os.h>
#include <nv/nv_MemoryManagement.h>
#include <nn/vi/vi_Content.h>
#include <nn/vi/vi_DisplayConfig.h>
#include "testVi_Graphics.h"

static void* NvAllocate(size_t size, size_t alignment, void* userPtr)
{
    NN_UNUSED(userPtr);
    return std::aligned_alloc(alignment, size);
}

static void NvFree(void* ptr, void* userPtr)
{
    NN_UNUSED(userPtr);
    std::free(ptr);
}

static void* NvReallocate(void* ptr, size_t size, void* userPtr)
{
    NN_UNUSED(userPtr);
    return std::realloc(ptr, size);
}

extern "C" void nnMain()
{
    const size_t GraphicsSystemMemorySize = 8 * 1024 * 1024;
    nv::SetGraphicsAllocator(NvAllocate, NvFree, NvReallocate, nullptr);
    nv::SetGraphicsDevtoolsAllocator(NvAllocate, NvFree, NvReallocate, nullptr);
    nv::InitializeGraphics( std::malloc( GraphicsSystemMemorySize ), GraphicsSystemMemorySize );

    // This function is a temporary workaround and will be removed at a later date.
    // Eventually, nn::vi::Initialize will handle all use cases.
    nn::vi::Initialize();

    nn::vi::Display* pDisplay;

    {
        nn::Result result = nn::vi::OpenDefaultDisplay(&pDisplay);
        NN_ASSERT(result.IsSuccess(), "Failed to open default display.");
    }

    // These values are decided by system policy and may differ between displays.
    int zOrderCountMin = nn::vi::GetZOrderCountMin(pDisplay);
    int zOrderCountMax = nn::vi::GetZOrderCountMax(pDisplay);
    NN_LOG("Minimum Z-Order: %d\n", zOrderCountMin);
    NN_LOG("Maximum Z-Order: %d\n", zOrderCountMax);

    nn::vi::Layer* pLayer;

    const int LayerWidth = 760;
    const int LayerHeight = 410;

    nn::vi::LayerSettings settings;
    nn::vi::SetLayerSettingsDefaults(&settings);
    // disabling autoscaling to current resolution
    settings.Reset<nn::vi::LayerFlags::Fullscreen>();

    {
        nn::Result result = nn::vi::CreateLayer(&pLayer, pDisplay, settings);
        NN_ASSERT(result.IsSuccess(), "Failed to create layer.");
    }

    int width;
    int height;

    nn::vi::GetDisplayLogicalResolution(&width, &height, pDisplay);

    // since fullscreen flag is not set, layer size and position will not be correct
    // and must be set manually
    nn::vi::SetLayerSize(pLayer, LayerWidth, LayerHeight);
    // centering layer
    nn::vi::SetLayerPosition(pLayer, (width - LayerWidth) / 2, (height - LayerHeight) / 2);

    Graphics g;

    g.RegisterLayer(pLayer, LayerWidth, LayerHeight, Color(1.f, 0.f, 0.f));

    nn::vi::SetLayerZ(pLayer, zOrderCountMax);

    nn::vi::Display* pLcd;
    nn::vi::OpenDisplay(&pLcd, "Internal");

    nn::os::Tick start = nn::os::GetSystemTick();

    float alpha[] = { 1.f, .95f, .9f, .85f, .8f, .75f, .7f, .65f, .6f, .55f, .5f, .45f, .4f, .35f, .3f, .25f, .2f, .15f, .1f, .05f, 0.f };

    for (int i = 0, j = 1;;)
    {
        g.Update();

        nn::os::Tick end = nn::os::GetSystemTick();
        if( (end - start).ToTimeSpan().GetMilliSeconds() >= 64 )
        {
            NN_ABORT_UNLESS_RESULT_SUCCESS(nn::vi::SetDisplayAlpha(pLcd, alpha[i]));
            start = nn::os::GetSystemTick();

            if (i == 0)
            {
                j = 1;
            }
            else
            {
                if (i == sizeof(alpha) / sizeof(alpha[0]) - 1)
                {
                    j = -1;
                }
            }

            i += j;
        }
    }

    nn::vi::Finalize();
}
