﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "utils.h"

#include <curl/curl.h>
#include <nn/os.h>

#include <algorithm>
#include <cstring>
#include <sstream>

size_t FileReaderCallback(char *buffer, size_t size, size_t nitems, void *instream)
{
    FileReaderInfo *frInfo = reinterpret_cast<FileReaderInfo*>(instream);
    size_t bufferSize = size * nitems;

    size_t bytesWrittenToBuffer;
    nn::Result nResult = nn::fs::ReadFile(&bytesWrittenToBuffer, frInfo->handle, frInfo->bytesRead, buffer, bufferSize);
    if (nResult.IsFailure())
        return CURL_READFUNC_ABORT;

    frInfo->bytesRead += bytesWrittenToBuffer;
    return bytesWrittenToBuffer;
}

size_t FileWriterCallback(char *ptr, size_t size, size_t nmemb, void *userdata)
{
    FileWriterInfo *fwInfo = reinterpret_cast<FileWriterInfo*>(userdata);
    size_t bytesToWrite = size * nmemb;

    nn::Result nResult = nn::fs::WriteFile(fwInfo->handle, fwInfo->bytesWritten, ptr, bytesToWrite, nn::fs::WriteOption::MakeValue(nn::fs::WriteOptionFlag_Flush));
    if (nResult.IsFailure())
    {
        return 0;
    }

    fwInfo->bytesWritten += bytesToWrite;
    return bytesToWrite;
}

size_t NullReaderCallback(char *buffer, size_t size, size_t nitems, void *instream)
{
    size_t *fileSize = reinterpret_cast<size_t*>(instream);
    size_t bufferSize = size * nitems;

    size_t bytesToFill = std::min(bufferSize, *fileSize);

    memset(buffer, 0, bytesToFill);

    *fileSize -= bytesToFill;
    return bytesToFill;
}

size_t NullWriterCallback(char *ptr, size_t size, size_t nmemb, void *userdata)
{
    NN_UNUSED(ptr);
    NN_UNUSED(userdata);

    // Don't do anything, just return a success value
    return size * nmemb;
}

size_t StringReaderCallback(char *buffer, size_t size, size_t nitems, void *instream)
{
    StringReaderInfo *srInfo = reinterpret_cast<StringReaderInfo*>(instream);
    size_t bufferSize = size * nitems;

    size_t bytesToFill = std::min(bufferSize, srInfo->dataSize - srInfo->offset);

    memcpy(buffer, srInfo->pData + srInfo->offset, bytesToFill);

    srInfo->offset += bytesToFill;
    return bytesToFill;
}

size_t StringWriterCallback(char *ptr, size_t size, size_t nmemb, void *userdata)
{
    std::stringstream *stream = reinterpret_cast<std::stringstream*>(userdata);
    size_t bytesToWrite = size * nmemb;

    stream->write(ptr, bytesToWrite);
    if (!stream->good())
    {
        return 0;
    }

    return bytesToWrite;
}

nn::Result CreateFileForWriting(nn::fs::FileHandle *handle, const char *filename)
{
    nn::Result nResult;

    // Check to see if the file already exists
    nn::fs::DirectoryEntryType fileEntry;
    nResult = nn::fs::GetEntryType(&fileEntry, filename);
    if (nn::fs::ResultPathNotFound::Includes(nResult))
    {
        // File does not exist, attempt to create it
        nResult = nn::fs::CreateFile(filename, 0);
        if (nResult.IsFailure())
        {
            return nResult; // Couldn't create new file
        }
    }
    else if (nResult.IsSuccess())
    {
        // Filename already exists, see if it's a directory
        if (fileEntry == nn::fs::DirectoryEntryType_Directory)
        {
            return nn::fs::ResultAlreadyExists(); // Directory with that name already exists
        }
    }
    else
    {
        return nResult; // Something went very wrong, you figure it out
    }

    // Open the (now-existant) file for writing/appending
    nn::fs::FileHandle internalHandle;
    nResult = nn::fs::OpenFile(&internalHandle, filename, nn::fs::OpenMode_Write | nn::fs::OpenMode_AllowAppend);
    if (nResult.IsFailure())
    {
        return nResult; // Failed opening file for writing
    }

    // Truncate opened file
    nResult = nn::fs::SetFileSize(internalHandle, 0);
    if (nResult.IsFailure())
    {
        nn::fs::CloseFile(internalHandle);
        return nResult; // Failed truncating file to 0 bytes
    }

    // File successfully opened
    *handle = internalHandle;
    return nn::ResultSuccess();
}

nn::Result OpenFileForReading(nn::fs::FileHandle *handle, const char *filename)
{
    nn::Result nResult;

    // Attempt to open the requested file
    nn::fs::FileHandle internalHandle;
    nResult = nn::fs::OpenFile(&internalHandle, filename, nn::fs::OpenMode_Read);
    if (nResult.IsFailure())
        return nResult; // Something went wrong, probably a nonexistant file

    // File successfully opened
    *handle = internalHandle;
    return nn::ResultSuccess();
}

nn::Result LoadCaCertIntoContext(nn::ssl::Context *ctx)
{
    nn::Result nResult;
    nn::fs::FileHandle certFile;

    // Read certificate from ROM into memory
    nResult = nn::fs::OpenFile(&certFile, "rom:/libcurlTestingCA.crt", nn::fs::OpenMode_Read);
    if (nResult.IsFailure())
    {
        return nResult; // Could not open certificate for reading
    }

    int64_t certSize;
    nResult = nn::fs::GetFileSize(&certSize, certFile);
    if (nResult.IsFailure())
    {
        nn::fs::CloseFile(certFile);
        return nResult; // Couldn't get size of certificate file
    }

    unsigned int certSize_unsigned = static_cast<unsigned int>(certSize);
    char *certBuffer = new char[certSize_unsigned];
    if (certBuffer == nullptr)
    {
        nn::fs::CloseFile(certFile);
        return nn::os::ResultOutOfMemory();
    }

    nResult = nn::fs::ReadFile(certFile, 0, certBuffer, certSize_unsigned);
    if (nResult.IsFailure())
    {
        delete[] certBuffer;
        nn::fs::CloseFile(certFile);
        return nResult; // Couldn't read certificate file
    }

    // Load the certificate into the provided context
    nn::ssl::CertStoreId caCertId;
    nResult = ctx->ImportServerPki(&caCertId, certBuffer, static_cast<uint32_t>(certSize), nn::ssl::CertificateFormat_Pem);
    if (nResult.IsFailure())
    {
        delete[] certBuffer;
        nn::fs::CloseFile(certFile);
        return nResult; // Failed to import certificate into SSL context
    }

    // Clean up and finish
    delete[] certBuffer;
    nn::fs::CloseFile(certFile);

    return nn::ResultSuccess();
}

ReadFileResult ReadXmlDocFromFile(pugi::xml_document *xmlDoc, const char *filename)
{
    nn::Result nResult;

    // Open the requested file
    nn::fs::FileHandle xmlFile;
    nResult = nn::fs::OpenFile(&xmlFile, filename, nn::fs::OpenMode_Read);
    if (nResult.IsFailure())
    {
        return ReadFileResult_FileOpenError;
    }

    // Get the size of the file
    int64_t xmlFileSize;
    nResult = nn::fs::GetFileSize(&xmlFileSize, xmlFile);
    if (nResult.IsFailure())
    {
        nn::fs::CloseFile(xmlFile);
        return ReadFileResult_FileSizeError;
    }

    // Create a memory buffer and read the XML data
    unsigned int xmlFileSize_unsigned = static_cast<unsigned int>(xmlFileSize);
    char *xmlBuffer = new char[xmlFileSize_unsigned];
    if (xmlBuffer == nullptr)
    {
        nn::fs::CloseFile(xmlFile);
        return ReadFileResult_OutOfMemoryError;
    }

    nResult = nn::fs::ReadFile(xmlFile, 0, xmlBuffer, xmlFileSize_unsigned);
    if (nResult.IsFailure())
    {
        delete[] xmlBuffer;
        nn::fs::CloseFile(xmlFile);
        return ReadFileResult_FileReadError;
    }

    // Close the file handle
    nn::fs::CloseFile(xmlFile);

    // Parse the XML data from memory into the provided document
    pugi::xml_parse_result pResult;
    pResult = xmlDoc->load_buffer(xmlBuffer, xmlFileSize_unsigned);
    if (pResult.status != pugi::status_ok)
    {
        delete[] xmlBuffer;
        return ReadFileResult_FileParseError; // Failed parsing XML data
    }

    // XML doc copied the buffer, so release it
    delete[] xmlBuffer;

    return ReadFileResult_Success;
}

ReadFileResult ReadJsonDocFromFile(rapidjson::Document *jsonDoc, const char *filename)
{
    nn::Result nResult;

    // Obtain a handle to the file
    nn::fs::FileHandle fHandle;
    nResult = nn::fs::OpenFile(&fHandle, filename, nn::fs::OpenMode_Read);
    if (nResult.IsFailure())
    {
        return ReadFileResult_FileOpenError;
    }

    // Compute the size of the file
    int64_t fileSize;
    nResult = nn::fs::GetFileSize(&fileSize, fHandle);
    if (nResult.IsFailure())
    {
        nn::fs::CloseFile(fHandle);
        return ReadFileResult_FileSizeError;
    }

    // Allocate a buffer for the file data
    char *fileBuffer = new char[fileSize];
    if (fileBuffer == nullptr)
    {
        nn::fs::CloseFile(fHandle);
        return ReadFileResult_OutOfMemoryError;
    }

    // Read the file into memory
    nResult = nn::fs::ReadFile(fHandle, 0, fileBuffer, fileSize);
    if (nResult.IsFailure())
    {
        delete[] fileBuffer;
        nn::fs::CloseFile(fHandle);
        return ReadFileResult_FileReadError;
    }

    // Close the file handle
    nn::fs::CloseFile(fHandle);

    // Parse the JSON data
    jsonDoc->Parse(fileBuffer);
    if (jsonDoc->HasParseError())
    {
        delete[] fileBuffer;
        return ReadFileResult_FileParseError;
    }

    // Clean up
    delete[] fileBuffer;

    return ReadFileResult_Success;
}
