﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Abort.h>
#include <nn/nn_Log.h>
#include <nn/nn_Macro.h>
#include <nn/hid/hid_Npad.h>
#include <nn/hid/hid_NpadJoy.h>
#include <nn/hid/system/hid_Nfc.h>
#include <nn/xcd/xcd.h>
#include "util.h"

namespace nnt { namespace xcd {

void InitializeNpad() NN_NOEXCEPT
{
    nn::hid::InitializeNpad();

    // フルコンと右コンを接続可能
    nn::hid::SetSupportedNpadStyleSet(
        nn::hid::NpadStyleHandheld::Mask |
        nn::hid::NpadStyleFullKey::Mask |
        nn::hid::NpadStyleJoyRight::Mask);
    nn::hid::SetSupportedNpadIdType(SupportedNpadIds, SupportedNpadIdCount);

    for (const auto& npadId : SupportedNpadIds)
    {
        nn::hid::SetNpadJoyAssignmentModeSingle(npadId);
    }
}

nn::xcd::DeviceHandle GetXcdHandle(const nn::hid::NpadIdType& npadId) NN_NOEXCEPT
{
    nn::xcd::DeviceHandle handle;
    NN_ABORT_UNLESS_RESULT_SUCCESS(
        nn::hid::system::GetXcdHandleForNpadWithNfc(&handle, npadId));

    return handle;
}

nn::hid::NpadButtonSet GetButtonState(const nn::hid::NpadIdType& npadId) NN_NOEXCEPT
{
    auto style = nn::hid::GetNpadStyleSet(npadId);
    if (style.Test<nn::hid::NpadStyleFullKey>())
    {
        nn::hid::NpadFullKeyState state;
        nn::hid::GetNpadState(&state, npadId);
        return state.buttons;
    }
    else
    {
        nn::hid::NpadJoyRightState state;
        nn::hid::GetNpadState(&state, npadId);
        return state.buttons;
    }
}

const char* GetProtocolName(nn::xcd::NfcProtocol protocol) NN_NOEXCEPT
{
    switch (protocol)
    {
    case nn::xcd::NfcProtocol_TypeA:     return "NFC-A";
    case nn::xcd::NfcProtocol_TypeB:     return "NFC-B";
    case nn::xcd::NfcProtocol_TypeF:     return "NFC-F";
    case nn::xcd::NfcProtocol_Type15693: return "ISO-15693";
    default: return "Unknown";
    }
}

const char* GetTagTypeName(nn::xcd::NfcTagType type) NN_NOEXCEPT
{
    switch (type)
    {
    case nn::xcd::NfcTagType_Type1:    return "Type1";
    case nn::xcd::NfcTagType_Type2:    return "Type2";
    case nn::xcd::NfcTagType_Type3:    return "Type3";
    case nn::xcd::NfcTagType_Type4A:   return "Type4A";
    case nn::xcd::NfcTagType_Type4B:   return "Type4B";
    case nn::xcd::NfcTagType_Iso15693: return "ISO15693";
    case nn::xcd::NfcTagType_Mifare:   return "MIFARE";
    default: return "Unknown";
    }
}

const char* GetEventReasonName(nn::xcd::NfcEventReason reason) NN_NOEXCEPT
{
    switch (reason)
    {
    case nn::xcd::NfcEventReason_Detected:               return "Detected";
    case nn::xcd::NfcEventReason_Deactivated:            return "Deactivated";
    case nn::xcd::NfcEventReason_ReadFinish:             return "ReadFinish";
    case nn::xcd::NfcEventReason_WriteFinish:            return "WriteFinish";
    case nn::xcd::NfcEventReason_PassThruResult:         return "PassThruResult";
    case nn::xcd::NfcEventReason_MifareKeyWriteFinish:   return "MifareKeyWriteFinish";
    case nn::xcd::NfcEventReason_MifareResult:           return "MifareResult";
    case nn::xcd::NfcEventReason_ControllerDisconnected: return "ControllerDisconnected";
    case nn::xcd::NfcEventReason_Error:                  return "Error";
    default: return "Unknown";
    }
}

}}  // nnt::xcd
