﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <algorithm>
#include <cstdlib>
#include <cstring>

#include <nn/nn_Abort.h>
#include <nn/nn_Log.h>
#include <nn/nn_Macro.h>
#include <nn/nn_TimeSpan.h>
#include <nn/os.h>
#include <nn/hid/hid_Npad.h>
#include <nn/hid/hid_NpadJoy.h>
#include <nn/hid/system/hid_Nfc.h>
#include <nn/xcd/xcd.h>
#include "NpadController.h"
#include "util.h"

namespace
{

bool g_IsDeviceConnected = false;
bool g_IsNfcMonitoring   = false;
bool g_IsNfcEventBound   = false;

nn::hid::NpadIdType         g_ConnectedDeviceNpadId;

nn::os::SystemEventType     g_NfcBindEvent;
nn::os::SystemEventType     g_NfcEvent;
nn::os::SystemEventType     g_NfcDetectEvent;
nn::os::TimerEventType      g_TerminateEvent;

nn::os::MultiWaitHolderType g_NfcBindHolder;
nn::os::MultiWaitHolderType g_NfcHolder;
nn::os::MultiWaitHolderType g_NfcDetectHolder;
nn::os::MultiWaitHolderType g_TerminateHolder;

nn::os::MultiWaitType       g_MultiWait;

nn::os::ThreadType g_Thread;
NN_OS_ALIGNAS_THREAD_STACK char g_ThreadStack[nn::os::ThreadStackAlignment * 2];

nn::os::Tick g_StartTick;

nnt::xcd::INpadController* g_pController = nullptr;

// テストの状態
enum TestSequence
{
    WaitingForConnect,  // デバイス接続待ち
    StateTransition,    // ステート遷移処理
    StartingTest,       // テスト開始処理
    CommandWaiting,     // コマンド入力待ち
    TagDetection,       // タグ検出待ち
    RunningTest,        // テスト実行中
    Finished            // テスト終了
};

// テストのコマンド
enum TestCommand
{
    None,
    RegisterKey,
    ClearKey,
    Read,
    Write
};

TestSequence g_CurrentSequence = TestSequence::WaitingForConnect;
TestCommand  g_CurrentCommand  = TestCommand::None;

/*
 * 処理中判定
 */
bool IsBusy() NN_NOEXCEPT
{
    return g_CurrentCommand != TestCommand::None;
}

/*
 * NFC 機能のセットアップ
 */
void SetupNfc(const nn::hid::NpadIdType& npadId) NN_NOEXCEPT
{
    // NFC のアクティブ化完了を通知するイベント
    nn::os::SystemEventType activateEvent;
    nn::hid::system::BindNfcActivateEvent(
        g_ConnectedDeviceNpadId,
        &activateEvent,
        nn::os::EventClearMode_ManualClear);
    nn::os::ClearSystemEvent(&activateEvent);

    // アクティブ化待ち
    nn::hid::system::ActivateNfc(g_ConnectedDeviceNpadId);
    nn::os::WaitSystemEvent(&activateEvent);
    nn::os::DestroySystemEvent(&activateEvent);

    auto handle = nnt::xcd::GetXcdHandle(npadId);

    NN_ABORT_UNLESS_RESULT_SUCCESS(
        nn::xcd::SetDataFormat(nn::xcd::PeriodicDataFormat_MCU, handle));
    NN_ABORT_UNLESS_RESULT_SUCCESS(
        nn::xcd::SetNfcEvent(&g_NfcEvent, &g_NfcDetectEvent, handle));
    NN_LOG("[TeraMifareTest] NFC events are registered\n");

    nn::os::InitializeMultiWaitHolder(&g_NfcHolder, &g_NfcEvent);
    nn::os::InitializeMultiWaitHolder(&g_NfcDetectHolder, &g_NfcDetectEvent);
    nn::os::LinkMultiWaitHolder(&g_MultiWait, &g_NfcHolder);
    nn::os::LinkMultiWaitHolder(&g_MultiWait, &g_NfcDetectHolder);
    g_IsNfcEventBound = true;

    NN_LOG("[TeraMifareTest] Set state to NFC\n");
    NN_ABORT_UNLESS_RESULT_SUCCESS(
        nn::xcd::SetMcuState(nn::xcd::McuState_Nfc, handle));
}

/*
 * テストの終了
 */
void TerminateTest() NN_NOEXCEPT
{
    if (g_IsDeviceConnected)
    {
        NN_LOG("[TeraMifareTest] Set state to Standby\n");

        auto handle = nnt::xcd::GetXcdHandle(g_ConnectedDeviceNpadId);
        NN_ABORT_UNLESS_RESULT_SUCCESS(
            nn::xcd::SetMcuState(nn::xcd::McuState_Standby, handle));
        nn::os::StartOneShotTimerEvent(&g_TerminateEvent, nn::TimeSpan::FromSeconds(1));
    }

    g_CurrentSequence = TestSequence::Finished;
}

/*
 * タグの検出開始
 */
void StartDetection(nn::xcd::DeviceHandle handle) NN_NOEXCEPT
{
    NN_LOG("[TeraMifareTest] Start discovery\n");
    nn::xcd::NfcDiscoveryParameter param = {};
    param.pollingMask       = nn::xcd::NfcPollingMask_All;
    param.activationTimeout = 0;    // タイムアウトしない
    param.discoveryPeriod   = 300;  // 300 msec 間隔でポーリング

    NN_ABORT_UNLESS_RESULT_SUCCESS(
        nn::xcd::StartNfcDiscovery(param, handle));

    NN_LOG("Please touch a tag!\n");
}

/*
 * タグの検出終了
 */
void StopDetection(nn::xcd::DeviceHandle handle) NN_NOEXCEPT
{
    NN_LOG("[TeraMifareTest] Stop discovery\n");

    NN_ABORT_UNLESS_RESULT_SUCCESS(
        nn::xcd::StopNfcDiscovery(handle));
}

/*
 * MIFARE 読み込み開始
 */
void StartMifareRead(const nn::xcd::NfcTagInfo& tagInfo, nn::xcd::DeviceHandle xcdHandle) NN_NOEXCEPT
{
    NN_LOG("[TeraMifareTest] Start mifare read\n");

    // XXX: パラメータはとりあえず固定
    nn::xcd::MifareReadParameter parameter = {};
    parameter.timeoutMsec = 5000;
    parameter.keyFormat   = nn::xcd::MifareKeyValueFormat_Raw;
    parameter.tagId       = tagInfo.tagId;
    parameter.blockCount  = nn::xcd::MifareReadBlockCountMax;

    for (decltype(parameter.blockCount) i = 0; i < parameter.blockCount; i++)
    {
        auto& block = parameter.blocks[i];
        block.key.type     = nn::xcd::MifareKeyType_A;
        block.blockAddress = static_cast<uint8_t>(i);

        // 鍵は All 0xFF
        std::memset(block.key.raw.value, 0xFF, nn::xcd::MifareKeyLength);
    }
    NN_ABORT_UNLESS_RESULT_SUCCESS(
        nn::xcd::StartMifareRead(parameter, xcdHandle));

    g_StartTick = nn::os::GetSystemTick();
}

/*
 * MIFARE 書き込み開始
 */
void StartMifareWrite(const nn::xcd::NfcTagInfo& tagInfo, nn::xcd::DeviceHandle xcdHandle) NN_NOEXCEPT
{
    NN_LOG("[TeraMifareTest] Start mifare write\n");

    // XXX: パラメータはとりあえず固定
    nn::xcd::MifareWriteParameter parameter = {};
    parameter.timeoutMsec = 5000;
    parameter.keyFormat   = nn::xcd::MifareKeyValueFormat_Raw;
    parameter.tagId       = tagInfo.tagId;
    parameter.blockCount  = nn::xcd::MifareWriteBlockCountMax;

    uint8_t writeValue  = 0;
    uint8_t nextAddress = 1;
    for (decltype(parameter.blockCount) i = 0; i < parameter.blockCount; i++)
    {
        auto& block = parameter.blocks[i];
        block.key.type     = nn::xcd::MifareKeyType_A;
        block.blockAddress = nextAddress;

        // 鍵は All 0xFF
        std::memset(block.key.raw.value, 0xFF, nn::xcd::MifareKeyLength);

        for (int j = 0; j < nn::xcd::MifareBlockBytes; j++)
        {
            block.data[j] = static_cast<char>(writeValue);
            writeValue = (writeValue + 1) & 0xFF;
        }

        nextAddress++;
        if (nextAddress % 4 == 3)
        {
            // Sector trailer には書き込まない
            nextAddress++;
        }
    }

    NN_ABORT_UNLESS_RESULT_SUCCESS(
        nn::xcd::StartMifareWrite(parameter, xcdHandle));

    g_StartTick = nn::os::GetSystemTick();
}

/*
 * タグ検出・喪失時の処理
 */
void ProcessNfcDetectEvent() NN_NOEXCEPT
{
    if (!g_IsDeviceConnected)
    {
        return;
    }

    NN_LOG("[TeraMifareTest] NFC detect event!!\n");

    auto xcdHandle = nnt::xcd::GetXcdHandle(g_ConnectedDeviceNpadId);

    nn::xcd::NfcInfo info;
    NN_ABORT_UNLESS_RESULT_SUCCESS(
        nn::xcd::GetNfcInfo(&info, xcdHandle));
    NN_LOG("[TeraMifareTest] reason: %d (%s)\n", info.reason, nnt::xcd::GetEventReasonName(info.reason));

    switch (info.reason)
    {
    case nn::xcd::NfcEventReason_Detected:
        {
            const auto& tagInfo = info.tagInfo;
            NN_LOG("[TeraMifareTest] Detected tag info:\n");
            NN_LOG("  Protocol: %d (%s)\n", tagInfo.protocol, nnt::xcd::GetProtocolName(tagInfo.protocol));
            NN_LOG("  Tag type: %d (%s)\n", tagInfo.tagType, nnt::xcd::GetTagTypeName(tagInfo.tagType));
            NN_LOG("  UID: ");
            for (int i = 0; i < tagInfo.tagId.length; ++i)
            {
                NN_LOG("%02X ", static_cast<nn::Bit8>(tagInfo.tagId.uid[i]));
            }
            NN_LOG("\n");

            // MIFARE アクセス
            switch (g_CurrentCommand)
            {
            case TestCommand::Read:
                {
                    StartMifareRead(tagInfo, xcdHandle);
                }
                break;
            case TestCommand::Write:
                {
                    StartMifareWrite(tagInfo, xcdHandle);
                }
                break;
            default:
                NN_UNEXPECTED_DEFAULT;
            }

            g_CurrentSequence = TestSequence::RunningTest;
        }
        break;
    case nn::xcd::NfcEventReason_Deactivated:
        {
            NN_LOG("[TeraMifareTest] Tag released\n");
            StopDetection(xcdHandle);
            TerminateTest();
        }
        break;
    default:
        NN_UNEXPECTED_DEFAULT;
    }
}

/*
 * MIFARE 鍵書き込みの完了処理
 */
void ProcessMifareKeyWriteFinish() NN_NOEXCEPT
{
    auto diffTime = (nn::os::GetSystemTick() - g_StartTick).ToTimeSpan();

    switch (g_CurrentCommand)
    {
    case TestCommand::RegisterKey:
        {
            NN_LOG("MIFARE keys are registered!!\n");
            break;
        }
    case TestCommand::ClearKey:
        {
            NN_LOG("MIFARE keys are cleared!!\n");
            break;
        }
    default:
        NN_UNEXPECTED_DEFAULT;
    }

    NN_LOG("  Elapsed time: %lld msec\n", diffTime.GetMilliSeconds());
}

/*
 * MIFARE の完了処理
 */
void ProcessMifareResult(const nn::xcd::NfcMifareData& mifareData) NN_NOEXCEPT
{
    auto diffTime = (nn::os::GetSystemTick() - g_StartTick).ToTimeSpan();

    switch (g_CurrentCommand)
    {
    case TestCommand::Read:
        {
            NN_LOG("Read finished!!\n");

            // 読み込んだデータを表示
            NN_LOG("  Addr : Data\n");
            for (int i = 0; i < mifareData.dataCount; i++)
            {
                const auto& block = mifareData.readDataBlocks[i];
                NN_LOG("    %02X : ", block.blockAddress);
                for (int j = 0; j < nn::xcd::MifareBlockBytes; j++)
                {
                    NN_LOG("%02X ", block.data[j]);
                }
                NN_LOG("\n");
            }
            NN_LOG("\n");
            break;
        }
    case TestCommand::Write:
        {
            NN_LOG("Write finished!!\n");
            break;
        }
    default:
        NN_UNEXPECTED_DEFAULT;
    }

    NN_LOG("  Elapsed time: %lld msec\n", diffTime.GetMilliSeconds());
}

/*
 * NFC の検出・喪失以外のイベントに関する処理
 */
void ProcessNfcGeneralEvent() NN_NOEXCEPT
{
    if (!g_IsDeviceConnected)
    {
        return;
    }

    NN_LOG("[TeraMifareTest] NFC event!!\n");

    auto xcdHandle = nnt::xcd::GetXcdHandle(g_ConnectedDeviceNpadId);

    nn::xcd::NfcInfo info = {};
    NN_ABORT_UNLESS_RESULT_SUCCESS(
        nn::xcd::GetNfcInfo(&info, xcdHandle));
    NN_LOG("[TeraMifareTest] reason: %d (%s)\n", info.reason, nnt::xcd::GetEventReasonName(info.reason));

    switch (info.reason)
    {
    case nn::xcd::NfcEventReason_MifareKeyWriteFinish:
        {
            ProcessMifareKeyWriteFinish();
        }
        break;
    case nn::xcd::NfcEventReason_MifareResult:
        {
            ProcessMifareResult(info.mifareData);
        }
        break;
    case nn::xcd::NfcEventReason_Error:
        {
            NN_LOG("[TeraMifareTest] An error has occurred\n");
            NN_LOG("  Result code: %d\n", info.errorInfo.resultCode);
        }
        break;
    default:
        NN_UNEXPECTED_DEFAULT;
    }

    nn::xcd::StopNfcDiscovery(xcdHandle);

    g_CurrentCommand  = TestCommand::None;
    g_CurrentSequence = TestSequence::StartingTest;
}

/*
 * 終了コマンド入力処理
 */
void ProcessExitCommandInput(const nn::hid::NpadIdType& npadId) NN_NOEXCEPT
{
    // R+ZR で終了
    if (g_pController->IsTriggered(
        nn::hid::NpadButton::R::Mask | nn::hid::NpadButton::ZR::Mask))
    {
        auto handle = nnt::xcd::GetXcdHandle(npadId);
        StopDetection(handle);
        TerminateTest();
    }
}

/*
 * コマンド入力処理
 */
void ProcessCommandInput(const nn::hid::NpadIdType& npadId) NN_NOEXCEPT
{
    if (IsBusy())
    {
        ProcessExitCommandInput(npadId);
        return;
    }

    auto xcdHandle = nnt::xcd::GetXcdHandle(npadId);

    if (g_pController->IsTriggered(nn::hid::NpadButton::A::Mask))
    {
        NN_LOG("<Command> Register key\n");
        g_CurrentCommand = TestCommand::RegisterKey;

        // とりあえずキーを All 0xFF にする
        nn::xcd::MifareKeyWriteParameter parameter = {};
        parameter.timeoutMsec = 5000;
        parameter.keyCount    = nn::xcd::MifareKeyCountMax;
        for (int i = 0; i < parameter.keyCount; i++)
        {
            std::memset(parameter.keys[i].value, 0xFF, nn::xcd::MifareEncryptedKeyLength);
        }
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::xcd::RegisterMifareKey(parameter, xcdHandle));
        g_StartTick = nn::os::GetSystemTick();
        g_CurrentSequence = TestSequence::RunningTest;
    }
    else if (g_pController->IsTriggered(nn::hid::NpadButton::B::Mask))
    {
        NN_LOG("<Command> Clear key\n");
        g_CurrentCommand = TestCommand::ClearKey;

        nn::xcd::MifareKeyClearParameter parameter = {};
        parameter.timeoutMsec = 5000;
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::xcd::ClearMifareKey(parameter, xcdHandle));
        g_StartTick = nn::os::GetSystemTick();
        g_CurrentSequence = TestSequence::RunningTest;
    }
    else if (g_pController->IsTriggered(nn::hid::NpadButton::X::Mask))
    {
        NN_LOG(
            "<Command> Read\n"
            "  + : Cancel\n");
        g_CurrentCommand  = TestCommand::Read;
        g_CurrentSequence = TestSequence::TagDetection;
        StartDetection(xcdHandle);
    }
    else if (g_pController->IsTriggered(nn::hid::NpadButton::Y::Mask))
    {
        NN_LOG(
            "<Command> Write\n"
            "  + : Cancel\n");
        g_CurrentCommand  = TestCommand::Write;
        g_CurrentSequence = TestSequence::TagDetection;
        StartDetection(xcdHandle);
    }
    else
    {
        ProcessExitCommandInput(npadId);
        return;
    }
}

/*
 * 現在のシーケンスに応じた処理
 */
void ProcessSequence() NN_NOEXCEPT
{
    if (g_pController != nullptr)
    {
        g_pController->Update();
    }

    switch (g_CurrentSequence)
    {
    case TestSequence::StateTransition:
        {
            if (!g_IsDeviceConnected)
            {
                return;
            }

            auto xcdHandle = nnt::xcd::GetXcdHandle(g_ConnectedDeviceNpadId);
            nn::xcd::McuState state;
            nn::xcd::GetMcuState(&state, xcdHandle);
            if (state == nn::xcd::McuState_Nfc)
            {
                // NFC ステートに遷移したらテストを開始
                g_CurrentSequence = TestSequence::StartingTest;
            }
            return;
        }
    case TestSequence::StartingTest:
        {
            if (!g_IsDeviceConnected)
            {
                return;
            }

            g_CurrentSequence = TestSequence::CommandWaiting;

            NN_LOG(
                "\n<Test commands>\n"
                "  A    : Register key\n"
                "  B    : Clear key\n"
                "  X    : Read\n"
                "  Y    : Write\n"
                "  R+ZR : Exit\n");
            return;
        }
    case TestSequence::CommandWaiting:
        {
            ProcessCommandInput(g_ConnectedDeviceNpadId);
            return;
        }
    case TestSequence::TagDetection:
        {
            if (g_pController->IsTriggered(nn::hid::NpadButton::Plus::Mask))
            {
                auto xcdHandle = nnt::xcd::GetXcdHandle(g_ConnectedDeviceNpadId);
                StopDetection(xcdHandle);

                NN_LOG("Command is cancelled.\n");
                g_CurrentCommand  = TestCommand::None;
                g_CurrentSequence = TestSequence::StartingTest;
            }
            return;
        }
    case TestSequence::RunningTest:
        {
            ProcessExitCommandInput(g_ConnectedDeviceNpadId);
            return;
        }
    case TestSequence::WaitingForConnect:
    case TestSequence::Finished:
        {
            // 何もしない
            return;
        }
    default:
        NN_UNEXPECTED_DEFAULT;
    }
}

/*
 * NFC デバイスの接続状態の更新
 */
void UpdateNfcDeviceConnection() NN_NOEXCEPT
{
    // デバイスのハンドルを取得
    int count = nn::hid::system::GetNpadsWithNfc(&g_ConnectedDeviceNpadId, 1);
    if (count > 0)
    {
        g_IsDeviceConnected = true;

        // XXX: NFC デバイス接続イベント通知直後はスタイルが反映されていない場合があるので、
        //      反映されるのを待つための W/A
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(50));

        // 接続されたデバイスに応じたコントローラクラスを作成
        g_pController = nnt::xcd::CreateNpadController(g_ConnectedDeviceNpadId);
        g_pController->Initialize();

        SetupNfc(g_ConnectedDeviceNpadId);
        g_CurrentSequence = TestSequence::StateTransition;

        NN_LOG("NFC device is connected.\n");
    }
    else
    {
        g_IsDeviceConnected = false;

        if (g_pController != nullptr)
        {
            g_pController->~INpadController();
            g_pController = nullptr;
        }

        g_CurrentSequence = TestSequence::WaitingForConnect;
        g_CurrentCommand  = TestCommand::None;

        NN_LOG("NFC device is disconnected.\n");
    }
}

/*
 * シグナルされたイベントに応じた処理
 */
void HandleEvent(nn::os::MultiWaitHolderType* pHolder) NN_NOEXCEPT
{
    NN_ABORT_UNLESS_NOT_NULL(pHolder);

    if (pHolder == &g_NfcBindHolder)
    {
        // デバイスの接続・切断
        nn::os::ClearSystemEvent(&g_NfcBindEvent);
        UpdateNfcDeviceConnection();
    }
    else if (pHolder == &g_NfcDetectHolder)
    {
        // タグの検知・喪失
        nn::os::ClearSystemEvent(&g_NfcDetectEvent);
        ProcessNfcDetectEvent();
    }
    else if (pHolder == &g_NfcHolder)
    {
        // 他の NFC 関連イベント
        nn::os::ClearSystemEvent(&g_NfcEvent);
        ProcessNfcGeneralEvent();
    }
}

/*
 * NFC のイベントモニタスレッド
 */
void NfcMonitoringThread(void* arg) NN_NOEXCEPT
{
    NN_UNUSED(arg);

    while (g_IsNfcMonitoring)
    {
        // デバイス情報の更新を待機
        auto pHolder = nn::os::WaitAny(&g_MultiWait);

        if (pHolder == &g_TerminateHolder)
        {
            g_IsNfcMonitoring = false;
        }
        else if (g_CurrentSequence != TestSequence::Finished)
        {
            HandleEvent(pHolder);
        }
    }
}

/*
 * 初期化処理
 */
void InitializeTest() NN_NOEXCEPT
{
    nnt::xcd::InitializeNpad();

    nn::os::InitializeMultiWait(&g_MultiWait);

    // 中断用イベント
    nn::os::InitializeTimerEvent(&g_TerminateEvent, nn::os::EventClearMode_ManualClear);
    nn::os::InitializeMultiWaitHolder(&g_TerminateHolder, &g_TerminateEvent);
    nn::os::LinkMultiWaitHolder(&g_MultiWait, &g_TerminateHolder);

    // NFC 用イベント
    nn::hid::system::BindNfcDeviceUpdateEvent(&g_NfcBindEvent, nn::os::EventClearMode_ManualClear);
    nn::os::InitializeMultiWaitHolder(&g_NfcBindHolder, &g_NfcBindEvent);
    nn::os::LinkMultiWaitHolder(&g_MultiWait, &g_NfcBindHolder);

    // 最初から接続済みの場合があるため、一度確認しておく
    UpdateNfcDeviceConnection();

    g_IsNfcMonitoring = true;
    nn::os::CreateThread(&g_Thread, NfcMonitoringThread, nullptr, g_ThreadStack, sizeof(g_ThreadStack), nn::os::DefaultThreadPriority);
    nn::os::SetThreadNamePointer(&g_Thread, "TeraMifareTestThread");
    nn::os::StartThread(&g_Thread);
}

/*
 * 終了処理
 */
void FinalizeTest() NN_NOEXCEPT
{
    if (g_IsDeviceConnected)
    {
        auto handle = nnt::xcd::GetXcdHandle(g_ConnectedDeviceNpadId);

        // 後始末
        nn::xcd::SetDataFormat(nn::xcd::PeriodicDataFormat_Basic, handle);
        nn::hid::system::DeactivateNfc(g_ConnectedDeviceNpadId);

        g_IsDeviceConnected = false;
    }

    nn::os::WaitThread(&g_Thread);
    nn::os::DestroyThread(&g_Thread);

    if (g_IsNfcEventBound)
    {
        nn::os::UnlinkMultiWaitHolder(&g_NfcDetectHolder);
        nn::os::UnlinkMultiWaitHolder(&g_NfcHolder);
        nn::os::FinalizeMultiWaitHolder(&g_NfcDetectHolder);
        nn::os::FinalizeMultiWaitHolder(&g_NfcHolder);
        nn::os::DestroySystemEvent(&g_NfcDetectEvent);
        nn::os::DestroySystemEvent(&g_NfcEvent);
        g_IsNfcEventBound = false;
    }

    nn::os::UnlinkMultiWaitHolder(&g_NfcBindHolder);
    nn::os::UnlinkMultiWaitHolder(&g_TerminateHolder);
    nn::os::FinalizeMultiWaitHolder(&g_NfcBindHolder);
    nn::os::FinalizeMultiWaitHolder(&g_TerminateHolder);
    nn::os::DestroySystemEvent(&g_NfcBindEvent);
    nn::os::FinalizeTimerEvent(&g_TerminateEvent);

    nn::os::FinalizeMultiWait(&g_MultiWait);
}

}  // anonymous

extern "C" void nnMain()
{
    NN_LOG("XCD TeraMifareManualTest tool starts.\n");
    NN_LOG("\n");

    InitializeTest();

    while (g_IsNfcMonitoring)
    {
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(15));

        ProcessSequence();
    }

    FinalizeTest();

    NN_LOG("Finished\n");
}
