﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/hid/hid_Npad.h>
#include <nn/hid/hid_NpadJoyRight.h>
#include <nn/xcd/xcd.h>

namespace nnt { namespace xcd {

/**
 * @brief   右コン操作に関する定義 (NpadController クラスのテンプレート引数として利用します)
 */
struct NpadRightJoy
{
    typedef nn::hid::NpadStyleJoyRight Style;
    typedef nn::hid::NpadJoyRightState State;
};

/**
 * @brief   フルキー操作に関する定義 (NpadController クラスのテンプレート引数として利用します)
 */
struct NpadFullKey
{
    typedef nn::hid::NpadStyleFullKey Style;
    typedef nn::hid::NpadFullKeyState State;
};

/**
 * @brief   携帯機操作に関する定義 (NpadController クラスのテンプレート引数として利用します)
 */
struct NpadHandheld
{
    typedef nn::hid::NpadStyleHandheld Style;
    typedef nn::hid::NpadHandheldState State;
};

/**
 * @brief   コントローラーに対する処理を扱うインターフェースクラスです。
 */
class INpadController
{
    NN_DISALLOW_COPY(INpadController);
    NN_DISALLOW_MOVE(INpadController);

public:
    INpadController() NN_NOEXCEPT {}
    virtual ~INpadController() NN_NOEXCEPT {}

    virtual void Initialize() NN_NOEXCEPT = 0;
    virtual void Finalize() NN_NOEXCEPT = 0;
    virtual void Update() NN_NOEXCEPT = 0;

    /**
     * @brief   指定されたボタンが押されているか
     */
    virtual bool IsPressed(nn::hid::NpadButtonSet buttons) NN_NOEXCEPT = 0;

    /**
     * @brief   指定されたボタンが新たに押されたか
     */
    virtual bool IsTriggered(nn::hid::NpadButtonSet buttons) NN_NOEXCEPT = 0;

    /**
     * @brief   指定されたボタンが離されたか
     */
    virtual bool IsReleased(nn::hid::NpadButtonSet buttons) NN_NOEXCEPT = 0;
};


/**
 * @brief   操作形態ごとの処理を記述した、 INpadController の派生クラスです。
 */
template<typename Npad>
class NpadController : public INpadController
{
    NN_DISALLOW_COPY(NpadController);
    NN_DISALLOW_MOVE(NpadController);

public:
    explicit NpadController(nn::hid::NpadIdType npadId) NN_NOEXCEPT :
        m_NpadId(npadId),
        m_CurrentButtons(),
        m_PreviousButtons() {}
    virtual ~NpadController() NN_NOEXCEPT
    {
        Finalize();
    }

    virtual void Initialize() NN_NOEXCEPT
    {
    }

    virtual void Finalize() NN_NOEXCEPT
    {
    }

    virtual void Update() NN_NOEXCEPT
    {
        UpdateButtons();
    }

    virtual bool IsPressed(nn::hid::NpadButtonSet buttons) NN_NOEXCEPT
    {
        return (m_CurrentButtons & buttons) == buttons;
    }

    virtual bool IsTriggered(nn::hid::NpadButtonSet buttons) NN_NOEXCEPT
    {
        return (m_CurrentButtons & buttons) == buttons &&
            (m_PreviousButtons & buttons) != buttons;
    }

    virtual bool IsReleased(nn::hid::NpadButtonSet buttons) NN_NOEXCEPT
    {
        return (m_CurrentButtons & buttons) != buttons &&
            (m_PreviousButtons & buttons) == buttons;
    }

private:
    void UpdateButtons() NN_NOEXCEPT
    {
        m_PreviousButtons = m_CurrentButtons;

        typename Npad::State state;
        nn::hid::GetNpadState(&state, m_NpadId);
        m_CurrentButtons = state.buttons;
    }

private:
    const nn::hid::NpadIdType m_NpadId;             //!< Npad ID
    nn::hid::NpadButtonSet    m_CurrentButtons;     //!< 現在のボタン情報
    nn::hid::NpadButtonSet    m_PreviousButtons;    //!< 前回のボタン情報
};

/**
 * @brief   指定された NpadIdType のスタイルに対応する NpadController を作成します。
 */
INpadController* CreateNpadController(const nn::hid::NpadIdType& npadId) NN_NOEXCEPT;

}}  // nnt::xcd
