﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "MomentModeTestTask.h"
#include <test_ConsoleFrameworkOpenCv.h>
#include <test_CvUtil.h>
#include <nn/xcd/xcd.h>

namespace
{

    const CvScalar ColorRed = {36, 28, 255};
    const CvScalar ColorWhite    = { 255,255,255, 255 };
    const CvScalar ColorGray    = { 128,128,128, 255 };
    const CvScalar ColorCyan     = { 239,174,  0, 255 };
    const CvScalar ColorMagenta     = { 255,0,  255, 255 };

    void DrawLine(CvArr* pImage, int x1, int y1, int x2, int y2, CvScalar color, int width)
    {
        CvPoint p1 = cvPoint(x1, y1);
        CvPoint p2 = cvPoint(x2, y2);
        cvLine(pImage, p1, p2, color, width);
    }

    void DrawCircle(CvArr* pImage, int ox, int oy, int radius, CvScalar color, int width)
    {
        CvPoint p = cvPoint(ox, oy);
        cvCircle(pImage, p, radius, color, width);
    }

    const char* OUTPUT_IMAGE_FILE_NAME = "%s\\%s_%04d%02d%02d_%02d%02d%02d_%04d.png";
    const char* OUTPUT_MOMENT_FILE_NAME = "%s\\%s_%s.csv";
    FILE* fp;
    const size_t FILE_NAME_SIZE = 256;
    const size_t LINE_SIZE = 4096;

}

const char* MomentModeTestTask::MENU_ITEM_MOMENT_NAMES[MENU_ITEM_MOMENT_NUM] =
{
    "WOI",
    "PreProcess",
    "Threshold"
};

//============================================================
void MomentModeTestTask::DoInitializeCore()
{
    auto trialCounter = 0;
    const int TrialCountMax = 50;
    while (!m_pDriver->RequestModeSet(nn::xcd::IrProcessorType::Moment))
    {
        NN_SDK_ASSERT_LESS(trialCounter, TrialCountMax);
        nn::os::SleepThread(nn::TimeSpanType::FromMilliSeconds(15));
        trialCounter++;
    }
    m_ImageHeader.irProcessorType = nn::xcd::IrProcessorType::Moment;
    // 内部パラメータの初期化
    SetDefaultMomentConfig(m_Config);
    SetDefaultMomentConfig(m_CurrentConfig);

} //MomentModeTestTask::DoInitializeCore()

//============================================================
void MomentModeTestTask::DoFinalizeCore()
{
    m_pDriver->SetSixAxisSampling(false);
    m_pDriver->SetLraSending(false);
    auto trialCounter = 0;
    const int TrialCountMax = 50;
    while (!m_pDriver->RequestModeSet(nn::xcd::IrProcessorType::Ready))
    {
        NN_SDK_ASSERT_LESS(trialCounter, TrialCountMax);
        nn::os::SleepThread(nn::TimeSpanType::FromMilliSeconds(15));
        trialCounter++;
    }
    m_ImageHeader.irProcessorType = nn::xcd::IrProcessorType::Ready;
} //MomentModeTestTask::DoFinalizeCore()

//============================================================
void MomentModeTestTask::DoCalcCore(IplImage* pImage, IplImage* pDstImage)
{
    NN_UNUSED(pImage);
    NN_UNUSED(pDstImage);
    m_SamplingResult = m_pDriver->GetMomentData(&m_ImageHeader, &m_State);
} //MomentModeTestTask::DoCalc()

//============================================================
void MomentModeTestTask::DoDrawCore(IplImage* pImage)
{
    cvSetZero(pImage);
    DrawMoment(m_State, pImage);
    if (m_IsLogEnabled)
    {
        SaveData(pImage, m_State);
    }
} //MomentModeTestTask::DoDraw()


void MomentModeTestTask::DrawMoment( nn::xcd::IrMomentProcessorState& momentProcessorState, IplImage* pResultImage)
{
    int roiStartX = 0;
    int roiStartY = 0;
    int roiWidth = 320;
    int roiHeight = 240;
    int blockDivX = nn::xcd::IrMomentModeBlockCountWidth;
    int blockDivY = nn::xcd::IrMomentModeBlockCountHeight;
    int blockWidth = roiWidth / blockDivX;
    int blockHeight = roiHeight / blockDivY;

    // 各ブロックの重心
    for (int y = 0; y < blockDivY; ++y)
    {
        for (int x = 0; x < blockDivX; ++x)
        {
            const nn::xcd::IrMomentStatistic* pBlock = &momentProcessorState.blocks[y * blockDivX + x];
            CvPoint startRectPoint = cvPoint(roiStartX + blockWidth * x, roiStartY + blockHeight * y);
            CvPoint endRectPoint = cvPoint(startRectPoint.x + blockWidth, startRectPoint.y + blockHeight);
            uint8_t avgY = static_cast<uint8_t>(pBlock->averageIntensity);
            cvRectangle(pResultImage, startRectPoint, endRectPoint, CV_RGB(avgY, avgY, avgY), -1);
            int lineWidth = 2;
            int lineHeight = static_cast<int>(blockHeight * pBlock->averageIntensity / 255.0);
            int startX = roiStartX + blockWidth * (x + 1) - 1 - lineWidth / 2;
            int startY = roiStartY + blockHeight * (y + 1) - 1;
            int endX = startX;
            int endY = startY - lineHeight;
            DrawLine(
                pResultImage,
                startX,
                startY,
                endX,
                endY,
                ColorMagenta, lineWidth);
            DrawCircle(
                pResultImage,
                static_cast<int>(pBlock->centroid.x),
                static_cast<int>(pBlock->centroid.y),
                3, ColorCyan, -1);
        }
    }

    // 全体の重心
    nn::xcd::IrMomentStatistic totalData;
    CalculateMomentArea(momentProcessorState, &totalData);
    CvScalar brightness = { 255, 0, totalData.averageIntensity};
    DrawCircle(
        pResultImage,
        static_cast<int>(totalData.centroid.x),
        static_cast<int>(totalData.centroid.y),
        3, brightness, -1);

    // ブロックの枠線
    for (int y = 0; y < blockDivY; ++y)
    {
        DrawLine(
            pResultImage,
            roiStartX,
            roiStartY + blockHeight * y,
            roiStartX + roiWidth - 1,
            roiStartY + blockHeight * y,
            ColorGray, 1);
    }
    DrawLine(
        pResultImage,
        roiStartX,
        roiStartY + blockHeight * blockDivY - 1,
        roiStartX + roiWidth,
        roiStartY + blockHeight * blockDivY - 1,
        ColorGray, 1);

    for (int x = 0; x < blockDivX; ++x)
    {
        DrawLine(
            pResultImage,
            roiStartX + blockWidth * x,
            roiStartY,
            roiStartX + blockWidth * x,
            roiStartY + roiHeight - 1,
            ColorGray, 1);
    }
    DrawLine(
        pResultImage,
        roiStartX + blockWidth * blockDivX - 1,
        roiStartY,
        roiStartX + blockWidth * blockDivX - 1,
        roiStartY + roiHeight - 1,
        ColorGray, 1);

}

//============================================================
void MomentModeTestTask::DoDrawMenuCore(IplImage* pImage)
{
    const CvSize FONT_SIZE = cvSize(16, 16);
    //const char* WOI_NAMES[] = {"QVGA", "QQVGA", "QQQVGA"};
    const char* PREPROCESS_NAMES[] = {"CUT OFF", "BINALIZE"};

    for (s32 i = MENU_ITEM_NUM; i < (MENU_ITEM_NUM + MENU_ITEM_MOMENT_NUM); i++)
    {
        s32 tx = 2 * FONT_SIZE.width;
        s32 ty = i * FONT_SIZE.height;

        m_TextWriter.SetFontColor((m_Cursor == i) ? test::CONST_COLOR_RED : test::CONST_COLOR_WHITE);
        m_TextWriter.PutText(pImage, 0, ty, "%s", (m_Cursor == i) ? ">" : " ");

        //if (i == (MENU_ITEM_MOMENT_WOI + MENU_ITEM_NUM))
        //{
        //    m_TextWriter.PutText(pImage, tx, ty, "%s : %s", MENU_ITEM_MOMENT_NAMES[MENU_ITEM_MOMENT_WOI], SIZE_FORMAT_NAMES[static_cast<int>(m_Format)]);
        //}
        if (i == (MENU_ITEM_MOMENT_PREPROCESS + MENU_ITEM_NUM))
        {
            m_TextWriter.PutText(pImage, tx, ty, "%s : %s", MENU_ITEM_MOMENT_NAMES[MENU_ITEM_MOMENT_PREPROCESS], PREPROCESS_NAMES[static_cast<int>(m_Config.preProcess)]);
        }
        if (i == (MENU_ITEM_MOMENT_THRESHOLD + MENU_ITEM_NUM))
        {
            m_TextWriter.PutText(pImage, tx, ty, "%s : %d", MENU_ITEM_MOMENT_NAMES[MENU_ITEM_MOMENT_THRESHOLD], m_Config.threshold);
        }
    }
} //MomentModeTestTask::DoDraw()


//============================================================
void MomentModeTestTask::DoCalcMenuCore(int* pMenuCnt)
{
    s32 keycode = GetKeycode();
    if (m_Cursor == (MENU_ITEM_MOMENT_WOI + MENU_ITEM_NUM))
    {
        //if (keycode == KEY_CODE_LEFT)
        //{
        //    m_Config.woi = static_cast<SizeFormat>(m_Format - 1);
        //    if (m_Format < 0)
        //    {
        //        m_Format = static_cast<SizeFormat>(SIZE_FORMAT_MAX_NUM - 1);
        //    }
        //}
        //if (keycode == KEY_CODE_RIGHT)
        //{
        //    m_Format= static_cast<SizeFormat>(m_Format + 1);
        //    if (m_Format >= SIZE_FORMAT_MAX_NUM)
        //    {
        //        m_Format = static_cast<SizeFormat>(0);
        //    }
        //}
    }
    if (m_Cursor == (MENU_ITEM_MOMENT_PREPROCESS + MENU_ITEM_NUM))
    {
        if (keycode == KEY_CODE_LEFT)
        {
            m_Config.preProcess = static_cast<Preprocess>(m_Config.preProcess - 1);
            if (m_Config.preProcess < 0)
            {
                m_Config.preProcess = static_cast<Preprocess>(MOMENT_PREPROCESS_NUM - 1);
            }
        }
        if (keycode == KEY_CODE_RIGHT)
        {
            m_Config.preProcess = static_cast<Preprocess>(m_Config.preProcess + 1);
            if (m_Config.preProcess >= MOMENT_PREPROCESS_NUM)
            {
                m_Config.preProcess = static_cast<Preprocess>(0);
            }
        }
    }
    if (m_Cursor == (MENU_ITEM_MOMENT_THRESHOLD + MENU_ITEM_NUM))
    {
        if (keycode == KEY_CODE_LEFT)
        {
            m_Config.threshold--;
            if (m_Config.threshold < 0)
            {
                m_Config.threshold = 0;
            }
        }
        if (keycode == KEY_CODE_RIGHT)
        {
            m_Config.threshold++;
            if (m_Config.threshold >= 255)
            {
                m_Config.threshold = 255;
            }
        }
    }
    *pMenuCnt = MENU_ITEM_MOMENT_NUM;
}

void MomentModeTestTask::DoWriteRegSettingCore(nn::xcd::IrWriteRegisterSetting& setting, int& index)
{
    //if (m_Config.woi != m_CurrentConfig.woi)
    //{
    //    //WOI 変更の修正 (TO BE IMPLEMENTED)
    //    setting.registerBlock[index].bankId = 1;
    //    setting.registerBlock[index].address = 0x30;
    //    setting.registerBlock[index].data = 0x00;
    //    index++;
    //    setting.registerCount = index;
    //}
    if (m_Config.preProcess != m_CurrentConfig.preProcess)
    {
        setting.registerBlock[index].bankId = 0;
        setting.registerBlock[index].address = 0x2A;
        uint8_t data = 0;
        if (m_Config.preProcess == MOMENT_PREPROCESS_CUTOFF)
        {
            data = 0x00;
        }
        else
        {
            data = 0x01;
        }
        setting.registerBlock[index].data = data;
        index++;
        setting.registerCount = index;
    }
    if (m_Config.threshold != m_CurrentConfig.threshold)
    {
        setting.registerBlock[index].bankId = 0;
        setting.registerBlock[index].address = 0x20;
        setting.registerBlock[index].data = static_cast<uint8_t>(m_Config.threshold);
        index++;
        setting.registerCount = index;
    }
    m_CurrentConfig = m_Config;
}

void MomentModeTestTask::DoSetDefaultConfigCore(ProcessorConfig& config)
{
    config.digitalGain = GAIN_X8;
    config.exposureTimeUsec = 300;
    config.irLedMode = IR_LED_BOTH;
    config.irLedIntensity = 16;
    config.postProcess = POSTPROCESS_NONE;
    config.isOnOffSubtractionEnabled = true;
    config.isHdrEnabled = false;
    config.isLensShadingEnabled = true;
    config.fps = 133;
}

void MomentModeTestTask::SetDefaultMomentConfig(MomentConfig& config)
{
    config.woi.x = 0;
    config.woi.y = 0;
    config.woi.width = 320;
    config.woi.height = 240;
    config.threshold = 80;
    config.preProcess = MOMENT_PREPROCESS_CUTOFF;
}

void MomentModeTestTask::SaveData(IplImage* pImage, nn::xcd::IrMomentProcessorState& state)
{
    char path[1024];

    SYSTEMTIME st;
    GetLocalTime(&st);

    m_TotalFrameCount++;
    bool isCaptureEnabled = false;
    if (m_SkipFrame == 0 || (m_TotalFrameCount % m_SkipFrame == 0))
    {
        m_CaptureFrameCount++;
        isCaptureEnabled = true;
    }

    if (isCaptureEnabled)
    {
        if (m_IsVisualLogEnabled && m_SamplingResult.IsSuccess())
        {
            sprintf_s(path, OUTPUT_IMAGE_FILE_NAME, m_DirectoryPath, GetName(), st.wYear, st.wMonth, st.wDay, st.wHour, st.wMinute, st.wSecond, st.wMilliseconds);
            s32 success = cvSaveImage(path, pImage);
            if (success == 0)
            {
                TEST_PRINTF("Failed to save an image: %s\n", path);
                return;
            }
        }

        // データログ
        if (fp != NULL)
        {
            char data[LINE_SIZE];
            sprintf_s(data, "%04d%02d%02d_%02d%02d%02d_%04d", st.wYear, st.wMonth, st.wDay, st.wHour, st.wMinute, st.wSecond, st.wMilliseconds);
            if (m_SamplingResult.IsSuccess())
            {
                sprintf_s(data, "%s,%d,%d", data, state.samplingNumber, state.diffTimeStampCount);
                // 全体のデータを計算
                nn::xcd::IrMomentStatistic totalData;
                CalculateMomentArea(state, &totalData);
                sprintf_s(data, "%s,%f,%f,%f", data, totalData.averageIntensity, totalData.centroid.x, totalData.centroid.y);
                for (int i = 0; i < nn::xcd::IrMomentProcessorBlockCount; i++)
                {
                    sprintf_s(data, "%s,%f,%f,%f",
                        data,
                        state.blocks[i].averageIntensity,
                        state.blocks[i].centroid.x,
                        state.blocks[i].centroid.y
                    );
                }
                sprintf_s(data, "%s\n", data);
            }
            else
            {
                sprintf_s(data, "%s,SamplingFailed,%x\n", data, m_SamplingResult);
            }
            fputs(data, fp);
        }
    }
}

void MomentModeTestTask::DoCreateLogFileCore()
{
    char path[FILE_NAME_SIZE];
    sprintf_s(path, OUTPUT_MOMENT_FILE_NAME, m_DirectoryPath, GetName(), m_DirectoryPath);
    if ((fp = fopen(path, "w")) == NULL)
    {
        TEST_PRINTF("Failed to create file\n");
    }
    char data[LINE_SIZE];
    sprintf_s(data, "time,samplingNumber,deltaTime,total.averageIntensity,total.centroidX,total.centroidY");
    for (int i = 0; i< nn::xcd::IrMomentProcessorBlockCount; i++)
    {
        sprintf_s(data, "%s,blocks[%d].averageIntensity,blocks[%d].centroidX,blocks[%d].centoridY", data, i, i, i);
    }
    sprintf_s(data, "%s\n", data);
    fputs(data, fp);
}

void MomentModeTestTask::DoCloseLogFileCore()
{
    fclose(fp);
}

void MomentModeTestTask::CalculateMomentArea(nn::xcd::IrMomentProcessorState& state, nn::xcd::IrMomentStatistic* pStatistic)
{
    double moment00 = 0;
    double moment01 = 0;
    double moment10 = 0;
    const int MomentProcessorBlockColumnCount = 8;
    const int MomentProcessorBlockRowCount = 6;
    int startRow = 0;
    int startColumn = 0;
    int rowCount = MomentProcessorBlockRowCount;
    int columnCount = MomentProcessorBlockColumnCount;

    auto blockWidth = 320 / MomentProcessorBlockColumnCount;
    auto blockHeight = 240 / MomentProcessorBlockRowCount;

    for (auto i = startRow; i < startRow + rowCount; ++i)
    {
        for (auto j = startColumn; j < startColumn + columnCount; ++j)
        {
            auto block = state.blocks[j + i * MomentProcessorBlockColumnCount];
            auto tempMoment00 = block.averageIntensity * blockWidth * blockHeight;
            moment00 += tempMoment00;
            moment10 += block.centroid.x * tempMoment00;
            moment01 += block.centroid.y * tempMoment00;
        }
    }

    auto blockCount = rowCount * columnCount;
    pStatistic->averageIntensity = static_cast<float>(moment00 / (blockCount * blockWidth * blockHeight));
    if (moment00 != 0)
    {
        pStatistic->centroid.x = static_cast<float>(moment10 / moment00);
        pStatistic->centroid.y = static_cast<float>(moment01 / moment00);
    }
    else
    {
        pStatistic->centroid.x = 0.0f;
        pStatistic->centroid.y = 0.0f;
    }
}
