﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Abort.h>
#include <nn/nn_Log.h>
#include <nn/nn_Macro.h>
#include <nn/os/os_Tick.h>
#include <nn/os/os_Thread.h>
#include <nn/nn_TimeSpan.h>
#include <nn/xcd/xcd.h>
#include <nn/applet/applet_Types.h>
#include <nn/applet/applet_Apis.h>
#include <nn/hid.h>
#include <nn/hid/system/hid_Nfc.h>
#include <nn/hid/system/hid_Irsensor.h>
#include <nn/hid/hid_Npad.h>
#include <nn/hid/hid_NpadJoy.h>

#include "TeraIrsensorDriver.h"

#if defined (USE_OPENCV)
#include "visualizer/ReadyModeTestTask.h"
#include "visualizer/MomentModeTestTask.h"
#include "visualizer/ClusteringModeTestTask.h"
#include "visualizer/ImageTransferModeTestTask.h"
#include "visualizer/DpdModeTestTask.h"
#include "visualizer/HandAnalysisModeTestTask.h"

#include "framework/test_Common.h"
#include "framework/test_ConsoleFrameworkOpenCv.h"
#include "framework/test_ConsoleMenu.h"
#include "framework/test_TaskFactory.h"
//============================================================
// Linking OpenCV libraries
//============================================================
#pragma warning(push)
#pragma warning(disable: 4668)
#include <opencv2/opencv.hpp>
#pragma warning(pop)

#define CV_VERSION_STR CVAUX_STR(CV_MAJOR_VERSION) CVAUX_STR(CV_MINOR_VERSION) CVAUX_STR(CV_SUBMINOR_VERSION)

#ifdef _DEBUG
#define CV_EXT_STR "d.lib"
#else
#define CV_EXT_STR ".lib"
#endif

#pragma comment(lib, "opencv_core"       CV_VERSION_STR CV_EXT_STR)
#pragma comment(lib, "opencv_highgui"    CV_VERSION_STR CV_EXT_STR)
#pragma comment(lib, "opencv_imgproc"    CV_VERSION_STR CV_EXT_STR)
#endif

namespace
{
const nn::hid::NpadIdType NpadIds[] =
{
    nn::hid::NpadId::No1
};

const char* Version = "V2.1";

const size_t NpadIdCountMax = sizeof(NpadIds) / sizeof(NpadIds[0]);

const auto CommandInterval = ::nn::TimeSpan::FromMilliSeconds(15);
const auto InitializeInterval = ::nn::TimeSpan::FromMilliSeconds(500);
const auto DeviceAttachTimeout = ::nn::TimeSpan::FromSeconds(5);

nn::xcd::DeviceHandle g_Handle;
bool g_LoopFlag = true;

enum IrLedPattern
{
    LedOn,
    LedWide,
    LedNarrow,
    LedOff,
    LedOnAlways,
    LedWideAlways,
    LedNarrowAlways,
    LedPatternCountMax,
};
IrLedPattern g_IrLedPattern = LedOn;
nn::xcd::IrWriteRegisterBlock g_IrEnableRegister[LedPatternCountMax] =
{
    {0x00, 0x10, 0x00},   // Led On
    {0x00, 0x10, 0x10},   // Led Wide
    {0x00, 0x10, 0x20},   // Led Narrow
    {0x00, 0x10, 0x30},   // Led Off
    {0x00, 0x10, 0x01},   // Led On
    {0x00, 0x10, 0x11},   // Led Wide
    {0x00, 0x10, 0x21},   // Led Narrow
};

#if defined (USE_OPENCV)
static test::TaskFactoryContainer TASK_FACTORY_TABLE[] =
{
    {test::TaskFactoryFunc<ReadyModeTestTask >, "ReadyModeTestTask" , NULL},
    {test::TaskFactoryFunc<MomentModeTestTask    >, "MomentModeTestTask"    , NULL},
    {test::TaskFactoryFunc<ClusteringModeTestTask>, "ClusteringModeTestTask", NULL},
    {test::TaskFactoryFunc<ImageTransferModeTestTask >, "ImageTransferModeTestTask" , NULL},
    {test::TaskFactoryFunc<DpdModeTestTask>, "DpdModeTestTask", NULL},
    {test::TaskFactoryFunc<HandAnalysisModeTestTask>, "HandAnalysisModeTestTask", NULL},
};
static const s32 TASK_FACTORY_NUM = sizeof(TASK_FACTORY_TABLE) / sizeof(test::TaskFactoryContainer);

#endif

/*
 * アプリ側のボタン処理
 */
void UpdatePad(nnt::XcdIrSensorDriver& driver) NN_NOEXCEPT
{
    static nn::hid::NpadJoyRightState prevState;
    nn::hid::NpadJoyRightState currentState;
    nn::hid::NpadButtonSet buttonDownSetJoy;

    //現在有効な操作形態(NpadStyleSet)を取得
    nn::hid::NpadStyleSet style = nn::hid::GetNpadStyleSet(NpadIds[0]);

    if (style.Test<nn::hid::NpadStyleJoyRight>() == true)
    {
        // 押されたボタンを検出
        nn::hid::GetNpadState(&currentState, NpadIds[0]);
        buttonDownSetJoy = currentState.buttons & ~prevState.buttons;
        prevState = currentState;
    }

    if ((buttonDownSetJoy & nn::hid::NpadJoyButton::RightSR::Mask).IsAnyOn())
    {
        // 終了処理
        if (driver.RequestMcuFinalize())
        {
            g_LoopFlag = false;
        }
    }

    if ((buttonDownSetJoy & nn::hid::NpadJoyButton::RightSL::Mask).IsAnyOn())
    {
        // WriteRegister
        g_IrLedPattern = static_cast<IrLedPattern>((g_IrLedPattern + 1) % IrLedPattern::LedPatternCountMax);
        nn::xcd::IrWriteRegisterSetting setting;
        setting.registerBlock[0] = g_IrEnableRegister[g_IrLedPattern];
        setting.registerCount = 1;
        driver.RequestWriteRegister(setting);
    }

    if ((buttonDownSetJoy & nn::hid::NpadJoyButton::X::Mask).IsAnyOn())
    {
        // Ready に遷移
        driver.RequestModeSet(nn::xcd::IrProcessorType::Ready);
    }

    if ((buttonDownSetJoy & nn::hid::NpadJoyButton::A::Mask).IsAnyOn())
    {
        // Momentに遷移
        driver.RequestModeSet(nn::xcd::IrProcessorType::Moment);
    }

    if ((buttonDownSetJoy & nn::hid::NpadJoyButton::B::Mask).IsAnyOn())
    {
        // Clustering に遷移
        driver.RequestModeSet(nn::xcd::IrProcessorType::Clustering);
    }

    if ((buttonDownSetJoy & nn::hid::NpadJoyButton::Y::Mask).IsAnyOn())
    {
        // ImageTrnasfer に遷移
        driver.RequestModeSet(nn::xcd::IrProcessorType::ImageTransfer);
    }

    if ((buttonDownSetJoy & nn::hid::NpadJoyButton::R::Mask).IsAnyOn())
    {
        // Dpd に遷移
        driver.RequestModeSet(nn::xcd::IrProcessorType::Dpd);
    }

    if ((buttonDownSetJoy & nn::hid::NpadJoyButton::ZR::Mask).IsAnyOn())
    {
        // HandAnalysis に遷移
        driver.RequestModeSet(nn::xcd::IrProcessorType::TeraPlugin);
    }

    if ((buttonDownSetJoy & nn::hid::NpadJoyButton::Plus::Mask).IsAnyOn())
    {
        // ReadRegister
        driver.RequestReadRegister();
    }
}

/*
 * 初期化処理
 */
int Initialize() NN_NOEXCEPT
{
    nn::hid::InitializeNpad();

    // RightJoy のみ使用するモードに設定
    nn::hid::SetSupportedNpadStyleSet(nn::hid::NpadStyleJoyRight::Mask);

    // Npad の初期化
    nn::hid::SetSupportedNpadIdType(NpadIds, NpadIdCountMax);
    for (const auto& id : NpadIds)
    {
        nn::hid::SetNpadJoyAssignmentModeSingle(id);
    }
    nn::hid::NpadIdType npadId;

    // デバイスの接続を待つ
    // ぺアリング部分は hid の機能を使用する。
    auto startTick = nn::os::GetSystemTick();
    auto diffTime = (nn::os::GetSystemTick() - startTick).ToTimeSpan();

    while (nn::hid::system::GetNpadsWithNfc(&npadId, 1) == 0)
    {
        diffTime = (nn::os::GetSystemTick() - startTick).ToTimeSpan();
        if (diffTime.GetMilliSeconds() > DeviceAttachTimeout.GetMilliSeconds())
        {
            // 接続がなくタイムアウトが続いた場合はエラーをかえす。
            NN_LOG("Controller is not connected\n");
            nn::os::SleepThread(nn::TimeSpan::FromSeconds(2));
            return -1;
        }
        nn::os::SleepThread(CommandInterval);
    }

    nn::os::SystemEventType completionEvent;

    nn::hid::system::BindIrSensorEvent(npadId, &completionEvent, nn::os::EventClearMode_ManualClear);

    nn::hid::system::ActivateIrSensor(npadId, nn::applet::AppletResourceUserId::GetInvalidId()).IsSuccess();

    while (!nn::os::TryWaitSystemEvent(&completionEvent))
    {
        nn::os::SleepThread(CommandInterval);
    }

    nn::hid::system::GetXcdHandleForNpadWithIrSensor(
        &g_Handle,
        npadId).IsSuccess();

    return 0;
}

/*
 * 終了処理
 */
void Finalize() NN_NOEXCEPT
{
}

}  // anonymous

extern "C" void nnMain()
{
    NN_LOG("XCD TeraIrsensorManualTest tool %s starts.\n", Version);
    NN_LOG("\n");
#if !defined (USE_OPENCV)
    NN_LOG("A: Moment B: Clustering Y: ImageTransfer R:Dpd ZR: HandAnalysis X: Ready\n");
    NN_LOG("SL: WriteRegister +: ReadRegister\n");
    NN_LOG("SR: Finalize Application\n");
    NN_LOG("\n");
#endif

    if(Initialize())
    {
        // コントローラの接続がなかった場合は終了
        return;
    }
    ::nnt::XcdIrSensorDriver xcdIrDriver;
    xcdIrDriver.Initialize(g_Handle);

    auto trialCounter = 0;
    const int TrialCountMax = 50;
    while (!xcdIrDriver.RequestMcuInitialize())
    {
        NN_SDK_ASSERT_LESS(trialCounter, TrialCountMax);
        nn::os::SleepThread(nn::TimeSpanType::FromMilliSeconds(15));
        trialCounter++;
    }

#if defined (USE_OPENCV)
    test::ConsoleFrameworkOpenCv fw;
    fw.Initialize();

    for(auto i = 0; i< TASK_FACTORY_NUM; i++)
    {
        // 引数に xcdIrdriver を渡す
        TASK_FACTORY_TABLE[i].arg = reinterpret_cast<uptr>(&xcdIrDriver);
    }

    test::ConsoleMenu::CreateConfig createConfig;
    createConfig.pTaskFactoryTable = TASK_FACTORY_TABLE;
    createConfig.taskNum = TASK_FACTORY_NUM;
    createConfig.firstIndex = 0;
    test::TaskBase* pTask = new test::ConsoleMenu("ConsoleMenu", reinterpret_cast<uptr>(&createConfig));
    fw.Run(pTask);

    delete pTask;
    fw.Finalize();
    xcdIrDriver.RequestMcuFinalize();
#else
    xcdIrDriver.RequestMcuInitialize();
    nn::os::SleepThread(InitializeInterval);

    while(g_LoopFlag)
    {
        UpdatePad(xcdIrDriver);

        nn::os::SleepThread(CommandInterval);
    }
#endif
    xcdIrDriver.Finalize();
    Finalize();

    NN_LOG("Finished\n");
}
