﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Assert.h>
#include <nn/os/os_MultipleWait.h>
#include <nn/os/os_SystemEvent.h>
#include <nn/hid.h>
#include <nn/hid/hid_Npad.h>
#include <nn/xcd/xcd.h>
#include <nnt/nntest.h>

#include "testXcd_NfcPassThruAutoCommon.h"
#include "testXcd_NfcPassThruAutoHelper.h"

/**
 * @brief   テスト単位の開始・終了処理を行うためのフィクスチャクラス
 */
class XcdNfcBase : public ::testing::Test
{
protected:
    /**
     * @brief   発生したイベントの種類
     */
    enum class EventType
    {
        None,       //!< イベントなし (タイムアウト)
        Detect,     //!< タグ検出・喪失イベント
        General     //!< タグ検出・喪失以外のイベント
    };

protected:
    /**
     * @brief   共通開始処理。テスト開始時に実行される
     */
    virtual void SetUp() NN_NOEXCEPT NN_OVERRIDE;

    /**
     * @brief   テストの終了処理。テスト関数を抜けるときに実行される
     */
    virtual void TearDown() NN_NOEXCEPT NN_OVERRIDE;

    /**
     * @brief   テストの開始処理
     *
     * @retval  true    成功
     * @retval  false   失敗
     */
    bool SetupTest() NN_NOEXCEPT;

    /**
     * @brief   直近の NfcInfo を取得
     *
     * @param[out]  pOutInfo    NfcInfo の取得先
     */
    void GetNfcInfo(nn::xcd::NfcInfo* pOutInfo) NN_NOEXCEPT
    {
        NN_ASSERT_NOT_NULL(pOutInfo);
        *pOutInfo = m_LastNfcInfo;
    }

    /**
     * @brief   タグ検出処理
     *
     * @param[in]   parameter   検出パラメータ
     *
     * @retval  true    成功
     * @retval  false   失敗
     */
    bool DetectTag(const nn::xcd::NfcDiscoveryParameter& parameter) NN_NOEXCEPT;

    /**
     * @brief   タグの検出停止
     *
     * @retval  true    成功
     * @retval  false   失敗
     */
    bool StopTagDetection() NN_NOEXCEPT;

    bool SendRawData(
        nn::xcd::NfcPassThruData* pOutReceivedData,
        const nn::xcd::NfcPassThruParameter& parameter) NN_NOEXCEPT;

private:
    bool WaitDeviceConnection() NN_NOEXCEPT;

    /**
     * @brief   NFC デバイスのハンドルを取得
     *
     * @param[out]  pOutHandle      XCD のデバイスハンドル
     *
     * @retval  true    成功
     * @retval  false   失敗
     *
     * @pre
     *   - @a pOutHandle != nullptr
     */
    bool GetNfcDeviceHandle(nn::xcd::DeviceHandle* pOutHandle) NN_NOEXCEPT;

    /**
     * @brief   イベントの発生を待つ
     *
     * @return  発生したイベント (@ref EventType)
     */
    EventType WaitEvent() NN_NOEXCEPT;

    /**
     * @brief   リトライ処理 + ウェイト
     */
    void RetryAndWait() NN_NOEXCEPT;

    /**
     * @brief   汎用的なタグ検出処理
     *
     * @retval  true    成功
     * @retval  false   失敗
     */
    bool DetectTagBasic() NN_NOEXCEPT;

protected:

    nn::hid::NpadIdType         m_DeviceNpadId;         //!< デバイスの NpadId
    nn::os::SystemEventType     m_DeviceUpdateEvent;    //!< NFC デバイスの状態更新イベント
    nn::os::SystemEventType     m_GeneralEvent;         //!< 通常の NFC イベント
    nn::os::SystemEventType     m_DetectEvent;          //!< タグ検出・喪失イベント
    nn::os::MultiWaitType       m_MultiWait;            //!< NFC イベント待ち用
    nn::os::MultiWaitHolderType m_GeneralEventHolder;   //!< 通常の NFC イベント用 MultiWaitHolder
    nn::os::MultiWaitHolderType m_DetectEventHolder;    //!< 検出・喪失イベント用 MUltiWaitHolder

private:
    bool                        m_IsNfcInitialized;     //!< NFC 初期化済みフラグ
    nn::xcd::NfcInfo            m_LastNfcInfo;          //!< 直近の NfcInfo
};

