﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstdlib>
#include <cstdio>
#include <memory>
#include <nn/nn_Log.h>

#include <nn/settings/fwdbg/settings_SettingsSetterApi.h>
#include <nn/nifm.h>
#include <nn/socket/socket_Api.h>
#include <nn/socket/socket_Types.h>
#include <nn/socket/netinet/ip.h>
#include <nn/socket/netinet/ip_icmp.h>
#include <nn/socket/net/ethernet.h>
#include <nn/bsdsocket/cfg/cfg_Types.h>
#include <nn/bsdsocket/cfg/cfg_ClientApi.h>

#include <nn/wlan/wlan_Types.h>
#include <nn/wlan/wlan_Result.h>
#include <nn/wlan/wlan_InfraApi.h>
#include <nn/wlan/wlan_LocalApi.h>
#include <nn/wlan/wlan_SocketApi.h>
#include <nn/wlan/wlan_DetectApi.h>
#include <nn/btm/btm.h>
#include <nn/btm/btm_Result.h>
#include <nn/bluetooth/bluetooth_Api.h>
#include <array>

#include "TA_display.h"
#include "Async.h"
#include "TestAgent.h"

namespace TestAgent {

const char* APP_NAME = "TestAgent";
const char* APP_VERSION = "1.0.5";

const int KLocalConcurrencyLimit = 4;
const int DETECT_FRAME_HASH_SIZE   = 8;
const int DETECT_FRAME_HEADER_SIZE = 18;
const int DETECT_FRAME_HASH_OFFSET = 10;
const int DETECT_FRAME_SEQNO_OFFSET= 18;
const int RECV_FRAME_COUNT_MAX   = 1000;
const int HASH_FILTER_COUNT_MAX  = 800;
const int AGENT_COMMAND_MAX      = 100;

NN_ALIGNAS(4096) uint8_t SocketMemoryPoolBuffer[nn::socket::DefaultSocketMemoryPoolSize];
uint8_t  g_rxBufferAf[2048]; // ActionFrame受信用バッファ
uint8_t  g_txBuf[1400];
const size_t ThreadStackSize = 4096;              // スレッドのスタックサイズ
NN_OS_ALIGNAS_THREAD_STACK char  g_ThreadStackRxAf[ ThreadStackSize ];   // ActionFrame受信用スレッドスタック

uint32_t rxIdAf; // 受信エントリ番号受取り用変数
nn::os::ThreadType  g_ThreadRxAf;
static bool g_ExitFlagRxAf = false;

nn::os::MutexType m_RxBuffLock;
nn::os::MutexType m_CmdListLock;
nn::os::MutexType m_ContTxLock;
DetectFrameInfo_t RecvFrameList[RECV_FRAME_COUNT_MAX];
uint64_t HashFilterList[HASH_FILTER_COUNT_MAX];
TestAgentCmd_t AgentCmdList[AGENT_COMMAND_MAX];
int RecvFrameCount;
int HashFilterCount;
int CurrentCmdIdx;
int IsAgentTesting;
int SendFrameCount;
nn::os::EventType g_rxAfEv;
static nn::os::Tick g_startTick;

nn::wlan::ScanParameters g_scanParam = {
        nn::wlan::ScanType_Passive,
        {1, 6, 11, 36, 40, 44, 48},
        0,
        120,
        0,
        NULL,
        0,
        nn::wlan::MacAddress::CreateBroadcastMacAddress()
};

void RxThreadFuncAf(void* arg);
void IncreaseBufValue(uint8_t* pBuf, int idx);
DetectContTxPara_t ContTxPara;

//------------------------------------------------------------------------------
// グラフィックスシステム用メモリ割り当て・破棄関数
//------------------------------------------------------------------------------
#if defined( NN_BUILD_CONFIG_OS_SUPPORTS_HORIZON ) && defined( NN_BUILD_CONFIG_SPEC_NX )
void* NvAllocate(size_t size, size_t alignment, void* userPtr) NN_NOEXCEPT
{
    NN_UNUSED(userPtr);
    return aligned_alloc(alignment, nn::util::align_up(size, alignment));
}

void NvFree(void* addr, void* userPtr) NN_NOEXCEPT
{
    NN_UNUSED(userPtr);
    free(addr);
}

void* NvReallocate(void* addr, size_t newSize, void* userPtr) NN_NOEXCEPT
{
    NN_UNUSED(userPtr);
    return realloc(addr, newSize);
}
#endif

static unsigned short checksum(unsigned short *buf, int bufsz) NN_NOEXCEPT
{
    unsigned long sum = 0;

    while (bufsz > 1) {
        sum += *buf;
        buf++;
        bufsz -= 2;
    }

    if (bufsz == 1) {
        sum += *reinterpret_cast<unsigned char *>(buf);
    }

    sum = (sum & 0xffff) + (sum >> 16);
    sum = (sum & 0xffff) + (sum >> 16);

    return ~sum;
}

// nibble を ascii に
inline char NibbleToAscii(uint8_t nibble)
{
    nibble &= 0xf;
    return nibble < 0xa ? nibble + '0' : nibble - 0xa + 'A';
}

static const auto SocketReceiveCallback = [](int socket, int pollResult, void* userptr)
{
    const size_t AddressByteWidth = 2;
    const size_t BytesInLine = 16;
    const size_t LineWidth = AddressByteWidth * 2 + 1 + 3 * BytesInLine + 1 + BytesInLine + 1;

    if (pollResult & POLLHUP) {
        // 切断された
        NN_LOG(">socket_disconnected\n");
        nn::socket::Close(socket);
        return;
    } else if (pollResult & POLLNVAL) {
        // ソケット閉じた
        NN_LOG(">socket_closed\n");
        nn::socket::Close(socket);
        return;
    }

    NN_LOG(">recv\n");

    {
        char format[1 + 4 + 1 + BytesInLine * 3 + 1 + BytesInLine + 1 + 1];
        nn::util::SNPrintf(format, sizeof(format), ">%%-%ds +0 +1 +2 +3 +4 +5 +6 +7 +8 +9 +A +B +C +D +E +F 0123456789ABCDEF\n", AddressByteWidth * 2);

        NN_LOG(format, "addr");
    }

    std::array<char, LineWidth> lineBuffer;
    lineBuffer.back() = '\0';

    int totalReceivedCount = 0;
    char* lineHexWritePtr;
    char* lineAsciiWritePtr;

    uint8_t recvBuf[128];
    for (;;) {
        auto readPtr = recvBuf;
        auto readEnd = readPtr + nn::socket::Recv(socket, recvBuf, sizeof(recvBuf), 0);

        for (; readPtr < readEnd; ++readPtr) {
            if (!(totalReceivedCount & 0xf)) {
                auto lineWritePtr = lineBuffer.begin();
                lineHexWritePtr = lineWritePtr + AddressByteWidth * 2 + 1;
                lineAsciiWritePtr = lineHexWritePtr + 3 * BytesInLine;

                for (int i = AddressByteWidth * 2; i--;) {
                    *lineWritePtr++ = NibbleToAscii(totalReceivedCount >> (i * 4));
                }

                while (lineWritePtr < lineBuffer.end()) {
                    *lineWritePtr++ = ' ';
                }
            }

            *(lineHexWritePtr) = NibbleToAscii(*readPtr >> 4);
            *(lineHexWritePtr + 1) = NibbleToAscii(*readPtr);
            lineHexWritePtr += 3;

            *lineAsciiWritePtr++ = *readPtr >= 0x20 && *readPtr < 0x80 ? *readPtr : '.';

            if (!(~totalReceivedCount & 0xf)) {
                NN_LOG(">%s\n", lineBuffer.data());
            }

            ++totalReceivedCount;
        }

        int bufferedDataLength;
        nn::socket::Ioctl(socket, FIONREAD, &bufferedDataLength, sizeof(bufferedDataLength));
        if (bufferedDataLength == 0) {
            break;
        }
    }

    if (totalReceivedCount & 0xf) {
        NN_LOG(">%s\n", lineBuffer.data());
    }

    if (totalReceivedCount > 0) {
        Async::WaitRecv(socket, reinterpret_cast<void(*)(int, int, void*)>(userptr), userptr);
    } else {
        // RST 投げられたらここに来るようなので、たぶん RST
        NN_LOG(">socket_rst\n");
        nn::socket::Close(socket);
    }
};

int DetectStandAloneSetHashList() NN_NOEXCEPT
{
    int retval = 0;
    nn::Result result;

    if (HashFilterCount) {
        result = nn::wlan::Detect::SetHashList(&HashFilterList[0], HashFilterCount);
        NN_ABORT_UNLESS_RESULT_SUCCESS(result);
    }

    return retval;
}

int DetectStandAloneStart(int txInterval, int txCount, int idleCount,
                        int rxStart, int rxCount, uint8_t* hashData,
                        uint32_t dataSize, uint8_t channel) NN_NOEXCEPT
{
    nn::wlan::DetectPeriodicAfCycle FramePattern;
    nn::wlan::DetectHash FrameHash;
    nn::Result result;
    int retval = 0;

    FramePattern.txInterval   = txInterval;
    FramePattern.txCount      = txCount;
    FramePattern.idleCount    = idleCount;
    FramePattern.rxStartCount = rxStart;
    FramePattern.rxCount      = rxCount;

    // Openと同時にI/FもUpされる
    if (channel == 1)
        result = nn::wlan::Detect::OpenMode();
    else
        result = nn::wlan::Detect::OpenMode(channel);
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);

    // すれちがいスリープ中に受信したすれちがいパケットの累計を0に初期化
    result = nn::wlan::Detect::ClearTotalRecvCountInSleep();
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);

    // すれちがいスリープの予約
    nn::wlan::Detect::ReserveDetectSleep();

    // すれちがいスリープ用ActionFrameのセット
    memcpy(&FrameHash.hash[0], hashData, DETECT_FRAME_HASH_SIZE);
    for (uint32_t i = 0; i < sizeof(g_txBuf); i++) {
        g_txBuf[i] = static_cast<uint8_t>(i & 0xFF);
    }
    *((uint32_t*)&g_txBuf[0]) = 0;
    strcpy(reinterpret_cast<char*>(&g_txBuf[4]), "periodic action frame of stand alone mode.");
    result = nn::wlan::Detect::SetActionFrameForSleep(nn::wlan::ActionFrameType_Detect,
                                                      FrameHash,
                                                      g_txBuf,
                                                      dataSize - DETECT_FRAME_HEADER_SIZE);
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);
    NN_LOG("Set SA mode: Interval: %d, TxCnt: %d, IdleCnt: %d, RxStart: %d, RxCnt: %d\n",
           FramePattern.txInterval,
           FramePattern.txCount,
           FramePattern.idleCount,
           FramePattern.rxStartCount,
           FramePattern.rxCount);

    nn::wlan::Detect::SetPeriodicActionFrameCycle(FramePattern,
                                                  nn::wlan::DetectPeriodicAfCycleTarget_Sa);

    // ActionFrame受信開始
    uint16_t afTypes[] = {   // 受信したいActionFrameType）
            static_cast<uint16_t>(nn::wlan::ActionFrameType_Detect),
    };

    // ActionFrame用受信エントリ作成
    result = nn::wlan::Detect::CreateRxEntryForActionFrame(&rxIdAf, afTypes, sizeof(afTypes) / sizeof(uint16_t), 10);
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);

    // ActionFrame受信スレッド作成
    result = nn::os::CreateThread( &g_ThreadRxAf, RxThreadFuncAf, &rxIdAf, g_ThreadStackRxAf, ThreadStackSize, nn::os::DefaultThreadPriority );
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);
    nn::os::StartThread( &g_ThreadRxAf );

    result = nn::wlan::Detect::CancelGetActionFrame(rxIdAf);
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);
    g_startTick = nn::os::GetSystemTick();

    return retval;
}

int DetectStandAloneStop() NN_NOEXCEPT
{
    int retval = 0;
    nn::Result result;

    // ActionFrame受信停止と後始末
    // 受信のキャンセルと受信スレッドの破棄
    g_ExitFlagRxAf = false;
    nn::os::SignalEvent(&g_rxAfEv);
    nn::os::WaitThread( &g_ThreadRxAf );
    nn::os::DestroyThread( &g_ThreadRxAf );

    // 受信エントリ削除
    result = nn::wlan::Detect::DeleteRxEntryForActionFrame(rxIdAf);
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);

    result = nn::wlan::Detect::CloseMode();
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);

    return retval;
}

int DetectStandAloneStartFetchFrame() NN_NOEXCEPT
{
    nn::os::SignalEvent(&g_rxAfEv);

    return 0;
}

int DetectStandAloneStopFetchFrame() NN_NOEXCEPT
{
    nn::Result result;

    result = nn::wlan::Detect::CancelGetActionFrame(rxIdAf);
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);
    g_startTick = nn::os::GetSystemTick();

    return 0;
}

int GetRecvStandAloneFrameCount() NN_NOEXCEPT
{
    // スリープ中に受信したすれちがいActionFrame数を取得
    uint64_t count = 0;
    nn::Result result;

    result = nn::wlan::Detect::GetTotalRecvCountInSleep(&count);
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);

    return (int)count;
}

void ClearRecvStandAloneFrameCount() NN_NOEXCEPT
{
    nn::Result result;

    // すれちがいスリープ中に受信したすれちがいパケットの累計を0に初期化
    result = nn::wlan::Detect::ClearTotalRecvCountInSleep();
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);
}

int GetAgentStatus() NN_NOEXCEPT
{
    return IsAgentTesting;
}

int SetAgentStatus(int isTesting) NN_NOEXCEPT
{
    IsAgentTesting = isTesting > 0 ? 1 : 0;
    return 0;
}

int GetCmdListIdx() NN_NOEXCEPT
{
    return CurrentCmdIdx;
}

void* GetCmdListBuff() NN_NOEXCEPT
{
    return (void*)&AgentCmdList[0];
}

void ClearCmdList() NN_NOEXCEPT
{
    nn::os::LockMutex(&m_CmdListLock);
    CurrentCmdIdx = 0;
    memset(&AgentCmdList, 0, sizeof(TestAgentCmd_t) * AGENT_COMMAND_MAX);
    nn::os::UnlockMutex(&m_CmdListLock);
}

int SetCmdList(char* cmd) NN_NOEXCEPT
{
    nn::os::LockMutex(&m_CmdListLock);
    if (AgentCmdList[CurrentCmdIdx].cmd[0] != '\0') {
        if (CurrentCmdIdx + 1 == AGENT_COMMAND_MAX) CurrentCmdIdx = 0;
        else CurrentCmdIdx++;
    }
    if (strlen(cmd) > 63) {
        std::strcpy(&AgentCmdList[CurrentCmdIdx].cmd[0], cmd + 63);
        if (CurrentCmdIdx + 1 == AGENT_COMMAND_MAX) CurrentCmdIdx = 0;
        else CurrentCmdIdx++;
        std::strncpy(&AgentCmdList[CurrentCmdIdx].cmd[0], cmd, 63);
    } else
        std::strcpy(&AgentCmdList[CurrentCmdIdx].cmd[0], cmd);
    nn::os::UnlockMutex(&m_CmdListLock);

    return 0;
}

int GetHashFilterCount() NN_NOEXCEPT
{
    return HashFilterCount;
}

int SetHashFilter(uint8_t* hashData) NN_NOEXCEPT
{
    int retval = 0;

    if (HashFilterCount == HASH_FILTER_COUNT_MAX) return -1;

    memcpy(&HashFilterList[HashFilterCount], hashData, DETECT_FRAME_HASH_SIZE);
    HashFilterCount++;
    NN_LOG("Set Hash[%2d]: 0x%02X%02X%02X%02X%02X%02X%02X%02X\n", HashFilterCount,
           hashData[0], hashData[1], hashData[2], hashData[3],
           hashData[4], hashData[5], hashData[6], hashData[7]);

    return retval;
}

void ClearHashFilter() NN_NOEXCEPT
{
    HashFilterCount = 0;
    memset(&HashFilterList, 0, sizeof(uint64_t) * HASH_FILTER_COUNT_MAX);
}

int GetRecvDetectFrameCount() NN_NOEXCEPT
{
    return RecvFrameCount;
}

void* GetRecvDetectFrameBuff() NN_NOEXCEPT
{
    return (void*)&RecvFrameList[0];
}

void ClearRecvDetectFrameBuff() NN_NOEXCEPT
{
    nn::os::LockMutex(&m_RxBuffLock);
    RecvFrameCount = 0;
    memset(&RecvFrameList, 0, sizeof(DetectFrameInfo_t) * RECV_FRAME_COUNT_MAX);
    nn::os::UnlockMutex(&m_RxBuffLock);
}

int FindDetectFrameByHash(uint8_t* bssid, uint8_t* data, uint32_t dataSize)
{
    int recvTimes;
    int i;

    nn::os::LockMutex(&m_RxBuffLock);
    for (i = 0; i < RecvFrameCount; i++) {
        if ((memcmp(&RecvFrameList[i].bssid[0], bssid, nn::wlan::MacAddress::MacAddressSize) == 0) &&
            (memcmp(&RecvFrameList[i].dataHash[0], data, DETECT_FRAME_HASH_SIZE) == 0) &&
            (RecvFrameList[i].recvLength == dataSize)) {
            break;
        }
    }
    if (i == RecvFrameCount) {
        recvTimes = 0;
    } else {
        recvTimes = RecvFrameList[i].recvTimes;
    }
    nn::os::UnlockMutex(&m_RxBuffLock);

    return recvTimes;
}

static int FindExistDetectFrame(uint8_t* bssid, uint8_t* data, uint32_t dataSize)
{
    int idx;
    int i;

    for (i = 0; i < RecvFrameCount; i++) {
        if ((memcmp(&RecvFrameList[i].bssid[0], bssid, nn::wlan::MacAddress::MacAddressSize) == 0) &&
            (memcmp(&RecvFrameList[i].dataHash[0], data, DETECT_FRAME_HASH_SIZE) == 0) &&
            (RecvFrameList[i].recvLength == dataSize)) {
            break;
        }
    }
    if (i == RECV_FRAME_COUNT_MAX) {
        idx = RECV_FRAME_COUNT_MAX;
    } else if (i == RecvFrameCount) {
        idx = -1;
    } else {
        idx = i;
    }

    return idx;
}

void RxThreadFuncAf(void* arg)
{
    nn::Result result;
    g_ExitFlagRxAf = true;

    uint32_t* rxId = reinterpret_cast<uint32_t*>(arg);
    nn::wlan::MacAddress mac;
    char macStr[nn::wlan::MacAddress::MacStringSize];
    uint16_t channel;
    int16_t rssi;
    nn::os::Tick tick;

    while( g_ExitFlagRxAf )
    {
        size_t rxSize = 0;
        result = nn::wlan::Detect::GetActionFrame(&mac, g_rxBufferAf, sizeof(g_rxBufferAf), &rxSize, *rxId, &channel, &rssi, &tick); // ブロックされる
        if( result.IsSuccess() )
        {
            // ActionFrameを受信したタイミングのシステムチック値が取得出来るので、あらかじめ取得しておいたシステムチック値との差分を取れば、
            // 何秒後に受信していたのかが計算出来る。
            NN_LOG("Detect:%s %2d, %ddBm, %4dB, %02X%02X%02X%02X%02X%02X%02X%02X, %lldms, %d\n",
                   mac.GetString(macStr), channel, rssi, rxSize,
                   g_rxBufferAf[DETECT_FRAME_HASH_OFFSET],
                   g_rxBufferAf[DETECT_FRAME_HASH_OFFSET + 1],
                   g_rxBufferAf[DETECT_FRAME_HASH_OFFSET + 2],
                   g_rxBufferAf[DETECT_FRAME_HASH_OFFSET + 3],
                   g_rxBufferAf[DETECT_FRAME_HASH_OFFSET + 4],
                   g_rxBufferAf[DETECT_FRAME_HASH_OFFSET + 5],
                   g_rxBufferAf[DETECT_FRAME_HASH_OFFSET + 6],
                   g_rxBufferAf[DETECT_FRAME_HASH_OFFSET + 7],
                   (tick - g_startTick).ToTimeSpan().GetMilliSeconds(),
                   *((uint32_t*)&g_rxBufferAf[DETECT_FRAME_SEQNO_OFFSET]));
            nn::os::LockMutex(&m_RxBuffLock);
            int idx = FindExistDetectFrame(mac.GetMacAddressData(), &g_rxBufferAf[DETECT_FRAME_HASH_OFFSET], rxSize);
            if (idx == RECV_FRAME_COUNT_MAX) {
                NN_LOG("Recv Buffer is full\n");
            } else if (idx < 0) {
                DetectFrameInfo_t *pDetectFrame = &RecvFrameList[RecvFrameCount];
                memcpy(&pDetectFrame->bssid[0], mac.GetMacAddressData(), nn::wlan::MacAddress::MacAddressSize);
                memcpy(&pDetectFrame->dataHash[0], &g_rxBufferAf[DETECT_FRAME_HASH_OFFSET], DETECT_FRAME_HASH_SIZE);
                pDetectFrame->recvLength = rxSize;
                pDetectFrame->recvTimes = 1;
                RecvFrameCount++;
            } else {
                RecvFrameList[idx].recvTimes++;
            }
            nn::os::UnlockMutex(&m_RxBuffLock);
        }
        else if( result.GetDescription() == nn::wlan::ResultGetFrameCancelled().GetDescription() )
        {
            NN_LOG("GetFrame Cancelled.\n");
            NN_LOG("%s: Wait for the event to be signaled.\n", __FUNCTION__);
            nn::os::WaitEvent(&g_rxAfEv);
        }
        else
        {
            NN_ABORT("GetActionFrame Failed\n");
            nn::os::SleepThread(nn::TimeSpan::FromSeconds(1));
        }
    }
}

int DetectHostDrivenStart(uint32_t txInterval, int txCount, int idleCount,
                        int rxStart, int rxCount, uint8_t* hashData,
                        uint32_t dataSize, bool sendPeriodical,
                        uint8_t channel) NN_NOEXCEPT
{
    int retval = 0;
    nn::wlan::DetectPeriodicAfCycle FramePattern;
    nn::wlan::DetectHash FrameHash;
    nn::Result result;

    FramePattern.txInterval   = txInterval;
    FramePattern.txCount      = txCount;
    FramePattern.idleCount    = idleCount;
    FramePattern.rxStartCount = rxStart;
    FramePattern.rxCount      = rxCount;

    // Openと同時にI/FもUpされる
    if (channel == 1)
        result = nn::wlan::Detect::OpenMode();
    else
        result = nn::wlan::Detect::OpenMode(channel);
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);

    nn::wlan::WlanState wlanState;
    result = nn::wlan::Detect::GetState(&wlanState);
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);

    result = nn::wlan::Detect::StartCommunication();
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);

    // ActionFrame受信開始
    uint16_t afTypes[] = {   // 受信したいActionFrameType）
            static_cast<uint16_t>(nn::wlan::ActionFrameType_Detect),
    };

    g_startTick = nn::os::GetSystemTick();

    // ActionFrame用受信エントリ作成
    result = nn::wlan::Detect::CreateRxEntryForActionFrame(&rxIdAf, afTypes, sizeof(afTypes) / sizeof(uint16_t), 10);
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);

    // ActionFrame受信スレッド作成
    result = nn::os::CreateThread( &g_ThreadRxAf, RxThreadFuncAf, &rxIdAf, g_ThreadStackRxAf, ThreadStackSize, nn::os::DefaultThreadPriority );
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);
    nn::os::StartThread( &g_ThreadRxAf );

    if (sendPeriodical) {
        memcpy(&FrameHash.hash[0], hashData, DETECT_FRAME_HASH_SIZE);
        for (uint32_t i = 0; i < sizeof(g_txBuf); i++) {
            g_txBuf[i] = static_cast<uint8_t>(i & 0xFF);
        }
        *((uint32_t*)&g_txBuf[0]) = 0;
        strcpy(reinterpret_cast<char*>(&g_txBuf[4]), "periodic action frame of host driven mode.");
        nn::wlan::Detect::SetPeriodicActionFrameCycle(FramePattern,
                                                      nn::wlan::DetectPeriodicAfCycleTarget_Hd);
        NN_LOG("Set HD mode: Interval: %d, TxCnt: %d, IdleCnt: %d, RxStart: %d, RxCnt: %d\n",
               FramePattern.txInterval,
               FramePattern.txCount,
               FramePattern.idleCount,
               FramePattern.rxStartCount,
               FramePattern.rxCount);
        // 指定間隔で送信する
        result = nn::wlan::Detect::StartPeriodicActionFrame(nn::wlan::ActionFrameType_Detect,
                                                            FrameHash,
                                                            g_txBuf,
                                                            dataSize - DETECT_FRAME_HEADER_SIZE,
                                                            txInterval);
        NN_ABORT_UNLESS_RESULT_SUCCESS(result);
        NN_LOG("Start periodic action frame\n");
    }

    return retval;
}

int DetectHostDrivenStop() NN_NOEXCEPT
{
    int retval = 0;
    nn::Result result;

    // PeriodicActoinFrameの停止
    result = nn::wlan::Detect::CancelPeriodicActionFrame();
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);

    // ActionFrame受信停止と後始末
    // 受信のキャンセルと受信スレッドの破棄
    g_ExitFlagRxAf = false;
    result = nn::wlan::Detect::CancelGetActionFrame(rxIdAf);
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);
    nn::os::SignalEvent(&g_rxAfEv);
    nn::os::WaitThread( &g_ThreadRxAf );
    nn::os::DestroyThread( &g_ThreadRxAf );

    // 受信エントリ削除
    result = nn::wlan::Detect::DeleteRxEntryForActionFrame(rxIdAf);
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);

    // I/FがDOWNされる
    result = nn::wlan::Detect::StopCommunication();
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);

    result = nn::wlan::Detect::CloseMode();
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);

    return retval;
}

int DetectHostDrivenOneShot(uint8_t* hashData, uint32_t dataSize, uint8_t dataCmd,
                          uint8_t* mac, uint32_t seq) NN_NOEXCEPT
{
    int retval = 0;
    nn::Result result;
    nn::wlan::DetectHash FrameHash;
    nn::wlan::MacAddress bssid;

    bssid = nn::wlan::MacAddress(mac);

    // ActionFrameの中身作成 -------------------------
    memcpy(&FrameHash.hash[0], hashData, DETECT_FRAME_HASH_SIZE);
    for (uint32_t i = 0; i < sizeof(g_txBuf); i++) {
        g_txBuf[i] = static_cast<uint8_t>(i & 0xFF);
    }
    *((uint32_t*)&g_txBuf[0]) = seq;
    if (dataCmd == 1)
        strcpy(reinterpret_cast<char*>(&g_txBuf[4]), "one shot(P) action frame of host driven mode.");
    else if (dataCmd == 2)
        strcpy(reinterpret_cast<char*>(&g_txBuf[4]), "one shot action frame of host driven mode.");
    else return -1;

    nn::wlan::DetectHeader ndhp = {
            1, // major
            0, // minor
            dataCmd,
            0, // reserved
            FrameHash
    };
    result = nn::wlan::Detect::PutActionFrameOneShotEx(nn::wlan::MacAddress::CreateBroadcastMacAddress(),
                                                       bssid,
                                                       nn::wlan::ActionFrameType_Detect,
                                                       ndhp, g_txBuf,
                                                       dataSize - DETECT_FRAME_HEADER_SIZE, 0);

    if( nn::wlan::ResultNoMemory().Includes(result) == true )
    {
        NN_LOG("Put action frame one shot failed due to no memory\n");
    }
    else
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS(result);
    }

    return retval;
}

int GetSendDetectFrameCount() NN_NOEXCEPT
{
    return SendFrameCount;
}

uint8_t* GetCurrentHash() NN_NOEXCEPT
{
    return &ContTxPara.hash[0];
}

NN_ALIGNAS(4096) char  g_ContOneShotThreadStack[4096];
nn::os::ThreadType  g_ContOneShotThread;
static bool g_ExitFlagContOneShot = false;

void ContOneShotThreadFunc(void* arg)
{
    DetectContTxPara_t* pPara = static_cast<DetectContTxPara_t*>(arg);
    g_ExitFlagContOneShot = true;
    nn::os::Tick sysTick;

    SendFrameCount = 0;
    sysTick = nn::os::GetSystemTick();
    DetectHostDrivenOneShot(&pPara->hash[0], pPara->size, 1, &pPara->mac[0], pPara->seq);
    SendFrameCount++;
    while (g_ExitFlagContOneShot) {
        if((nn::os::GetSystemTick() - sysTick).ToTimeSpan().GetMilliSeconds() > pPara->interval) {
            sysTick = nn::os::GetSystemTick();
            pPara->seq++;
            if (pPara->isChange) {
                nn::os::LockMutex(&m_ContTxLock);
                IncreaseBufValue(&pPara->hash[0], 7);
                nn::os::UnlockMutex(&m_ContTxLock);
            }
            DetectHostDrivenOneShot(&pPara->hash[0], pPara->size, 1, &pPara->mac[0], pPara->seq);
            if (pPara->isChange) SendFrameCount = 1;
            else SendFrameCount++;
        } else {
            nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(1));
        }
    }
}

int DetectContTxThreadStart(uint8_t* hashData, uint32_t dataSize, uint8_t isChange,
                          uint8_t* mac, uint32_t seq, uint32_t txInterval) NN_NOEXCEPT
{
    nn::Result result;

    memcpy(&ContTxPara.hash[0], hashData, DETECT_FRAME_HASH_SIZE);
    memcpy(&ContTxPara.mac[0], mac, nn::wlan::MacAddress::MacAddressSize);
    ContTxPara.size = dataSize;
    ContTxPara.interval = txInterval ? txInterval : 1;
    ContTxPara.isChange = isChange;
    ContTxPara.seq = seq;

    /* AgentThreadスレッド作成 */
    result = nn::os::CreateThread(&g_ContOneShotThread, ContOneShotThreadFunc,
                                  &ContTxPara, g_ContOneShotThreadStack,
                                  sizeof(g_ContOneShotThreadStack),
                                  nn::os::DefaultThreadPriority);
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);

    nn::os::StartThread( &g_ContOneShotThread );

    return 0;
}

int DetectContTxThreadStop() NN_NOEXCEPT
{
    g_ExitFlagContOneShot = false;

    nn::os::WaitThread( &g_ContOneShotThread );
    nn::os::DestroyThread( &g_ContOneShotThread );

    return 0;
}

void IncreaseBufValue(uint8_t* pBuf, int idx)
{
    if (idx < 0) return;
    if (pBuf[idx] == 255) {
        pBuf[idx] = 0;
        IncreaseBufValue(&pBuf[0], idx - 1);
    } else {
        pBuf[idx]++;
    }
}

int DetectSimulatePeerTx(uint8_t* hashData, uint8_t* mac, uint32_t count,
                       uint32_t dataSize) NN_NOEXCEPT
{
    int retval = 0;
    nn::Result result;
    nn::wlan::DetectHash FrameHash;
    nn::wlan::MacAddress bssid;
    uint8_t tmpHashData[DETECT_FRAME_HASH_SIZE];
    uint8_t tmpMacData[nn::wlan::MacAddress::MacAddressSize];

    memcpy(&tmpHashData[0], hashData, DETECT_FRAME_HASH_SIZE);
    memcpy(&tmpMacData[0], mac, nn::wlan::MacAddress::MacAddressSize);

    // ActionFrameの中身作成 -------------------------
    for (uint32_t i = 0; i < sizeof(g_txBuf); i++) {
        g_txBuf[i] = static_cast<uint8_t>(i & 0xFF);
    }
    strcpy(reinterpret_cast<char*>(&g_txBuf[4]), "Simulate Peer Tx action frame.");

    for (int i = 0; i < count; i++) {
        *((uint32_t*)&g_txBuf[0]) = i;
        memcpy(&FrameHash.hash[0], &tmpHashData[0], DETECT_FRAME_HASH_SIZE);
        bssid = nn::wlan::MacAddress(&tmpMacData[0]);

        nn::wlan::DetectHeader ndhp = {
                1, // major
                0, // minor
                1,
                0, // reserved
                FrameHash
        };
        result = nn::wlan::Detect::PutActionFrameOneShotEx(nn::wlan::MacAddress::CreateBroadcastMacAddress(),
                                                           bssid,
                                                           nn::wlan::ActionFrameType_Detect,
                                                           ndhp, g_txBuf,
                                                           dataSize - DETECT_FRAME_HEADER_SIZE, 0);

        if( nn::wlan::ResultNoMemory().Includes(result) == true )
        {
            NN_LOG("Put action frame one shot failed due to no memory\n");
        }
        else
        {
            NN_ABORT_UNLESS_RESULT_SUCCESS(result);
        }
        IncreaseBufValue(&tmpHashData[0], 7);
        IncreaseBufValue(&tmpMacData[0], 5);
    }

    return retval;
}

int WlanDetectGetMacAddress(uint8_t *macAddr) NN_NOEXCEPT
{
    nn::Result result;
    nn::wlan::MacAddress myMacAddress;

    WlanDetectInitialize();

    result = nn::wlan::Detect::GetMacAddress(&myMacAddress);
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);

    memcpy(macAddr, myMacAddress.GetMacAddressData(),
           nn::wlan::MacAddress::MacAddressSize);

    WlanDetectUninitialize();

    return 0;
}

int SocketConnect(char* serverIp, int serverPort) NN_NOEXCEPT
{
    int retval = 0;

    Async::Initialize();
    int socket = nn::socket::Socket(AF_INET, SOCK_STREAM, 0);
    if (socket < 0) {NN_LOG("Socket create failed @ %d\n", __LINE__); return -1;}

    int opt = 1;
    retval = nn::socket::SetSockOpt(socket, SOL_TCP, TCP_NODELAY, &opt, sizeof(opt));
    if (retval < 0) {NN_LOG("failed to setting TCP_NODELAY flag\n");}

    // disable SACK/WindowScaling/TimeStamp
    opt = 1;
    retval = nn::socket::SetSockOpt(socket, SOL_TCP, TCP_NOOPT, &opt, sizeof(opt));
    if (retval < 0) {NN_LOG("failed to setting TCP_NODELAY flag\n");}

    opt = 4096;
    retval = nn::socket::SetSockOpt(socket, SOL_SOCKET, SO_RCVBUF, &opt, sizeof(opt));
    if (retval  < 0) {NN_LOG("failed to setting SO_RCVBUF value\n");}

    in_addr addr;
    retval = nn::socket::InetAton(serverIp, &addr);
    if (retval  < 0) {NN_LOG("failed to convert network address\n");}

    sockaddr_in destAddr = {};
    destAddr.sin_addr = addr;
    destAddr.sin_port = nn::socket::InetHtons(static_cast<uint16_t>(serverPort));
    destAddr.sin_family = AF_INET;

    retval = nn::socket::Connect(socket, reinterpret_cast<sockaddr*>(&destAddr), sizeof(destAddr));
    if (retval < 0) {
        NN_LOG("failed to connect socket\n");
        Async::Finalize();
        nn::socket::Close(socket);
        return retval;
    }

    Async::WaitRecv(socket, SocketReceiveCallback, reinterpret_cast<void*>(static_cast<void(*)(int, int, void*)>(SocketReceiveCallback)));

    retval = socket;

    return retval;
}

int SocketDisconnect(int socketId) NN_NOEXCEPT
{
    int retval = 0;

    retval = nn::socket::Close(socketId);
    Async::Finalize();
    if (retval < 0) {NN_LOG("failed @ %d\n", __LINE__); return -1;}

    return retval;
}

int WowlSetSocketInfo(int socketId) NN_NOEXCEPT
{
    int retval = 0;
    nn::socket::TcpInfo ti;
    nn::socket::SockLenT len;
    nn::bsdsocket::cfg::IfState state;

    nn::bsdsocket::cfg::GetIfState("wl0", &state);
    len = sizeof(ti);

    retval = nn::socket::GetSockOpt(socketId, nn::socket::Level::Sol_Tcp,
                                    nn::socket::Option::Tcp_Info,
                                    reinterpret_cast<void*>(&ti), &len);
    if (retval < 0) {NN_LOG(">GetSockOpt failed (error %d)\n", nn::socket::GetLastErrno()); return -1;}

    nn::socket::SockAddrIn saiClient;
    len = sizeof(saiClient);
    retval = nn::socket::GetSockName(socketId, reinterpret_cast<sockaddr*>(&saiClient), &len);
    if (retval < 0) {NN_LOG(">GetSockName failed (error %d)\n", nn::socket::GetLastErrno()); return -1;}
    if (saiClient.sin_addr.S_addr == nn::socket::InAddr_Any) {
        saiClient.sin_addr.S_addr = state.addr.S_addr;
    }

    nn::socket::SockAddrIn saiServer;
    len = sizeof(saiServer);
    retval = nn::socket::GetPeerName(socketId, reinterpret_cast<sockaddr*>(&saiServer), &len);
    if (retval < 0) {NN_LOG(">GetPeerName failed (error %d)\n", nn::socket::GetLastErrno()); return -1;}

    uint8_t remoteHardwareAddress[ETHER_ADDR_LEN];

    // FIXME: static routes
    nn::socket::SockAddrIn saiNext;
    if ((saiServer.sin_addr.S_addr & state.subnetMask.S_addr) == (state.gatewayAddr.S_addr & state.subnetMask.S_addr)) {
        saiNext = saiServer;
    } else {
        saiNext.sin_addr = state.gatewayAddr;
    }

    nn::Result result = nn::bsdsocket::cfg::LookupArpEntry(remoteHardwareAddress, ETHER_ADDR_LEN, saiNext.sin_addr);
    if (result.IsFailure()) {
        NN_LOG(">LookupArpEntry failed (result 0x%08x)\n", result.GetInnerValueForDebug());
        return -1;
    }

    NN_LOG(">       seq: % 16u[0x%08x]\n", ti.tcpi_snd_nxt, ti.tcpi_snd_nxt);
    NN_LOG(">       ack: % 16u[0x%08x]\n", ti.tcpi_rcv_nxt, ti.tcpi_rcv_nxt);
    NN_LOG(">      rwin: % 16u\n", ti.tcpi_rcv_space);
    NN_LOG(">     local: % 16s:%u\n", nn::socket::InetNtoa(saiClient.sin_addr), nn::socket::InetNtohs(saiClient.sin_port));
    NN_LOG(">    remote: % 16s:%u\n", nn::socket::InetNtoa(saiServer.sin_addr), nn::socket::InetNtohs(saiServer.sin_port));
    NN_LOG("> next addr: % 16s[%02x:%02x:%02x:%02x:%02x:%02x]\n",
        nn::socket::InetNtoa(saiNext.sin_addr),
        remoteHardwareAddress[0], remoteHardwareAddress[1], remoteHardwareAddress[2], remoteHardwareAddress[3], remoteHardwareAddress[4], remoteHardwareAddress[5]);

    nn::wlan::WlanIpv4Address src;
    nn::wlan::WlanIpv4Address dst;
    char srcIpStr[20];
    char dstIpStr[20];
    char* tok;

    std::strcpy(srcIpStr, nn::socket::InetNtoa(saiClient.sin_addr));
    std::strcpy(dstIpStr, nn::socket::InetNtoa(saiServer.sin_addr));
    tok = std::strtok(srcIpStr, ".");
    src.addr[0] = std::atoi(tok);
    for(int i = 1; i < 4; i++) {
        tok = std::strtok(NULL, ".");
        src.addr[i] = std::atoi(tok);
    }

    tok = std::strtok(dstIpStr, ".");
    dst.addr[0] = std::atoi(tok);
    for(int i = 1; i < 4; i++) {
        tok = std::strtok(NULL, ".");
        dst.addr[i] = std::atoi(tok);
    }

    nn::wlan::MacAddress dstMac(remoteHardwareAddress);
    result = nn::wlan::Infra::SetTcpSessionInformation(dstMac,
            src, dst, nn::socket::InetNtohs(saiClient.sin_port), nn::socket::InetNtohs(saiServer.sin_port),
            ti.tcpi_rcv_nxt, static_cast<uint16_t>(ti.tcpi_rcv_space));
    if (result.IsFailure()) {NN_LOG("SetTcpSessionInformation failed.\n"); return -1;}

    return retval;
}

int SendPing(char* IpAdder, int dataSize, int sendInterval, int sendCount) NN_NOEXCEPT
{
    int retval = 0;
    int recvCount = 0;
    int socket;
    int i;

    struct timeval tv;
    memset(&tv, 0, sizeof(tv));
    tv.tv_sec = 3;

    struct packet {
        icmp hdr;
        char data[4096];
    };
    packet senddata;
    int data_len = 0;

    if (0 < dataSize) {data_len = sizeof(senddata.hdr) + dataSize;}
    else {data_len = sizeof(senddata.hdr);}

    char recvBuffer[4096];
    struct icmp *icmpptr;
    struct ip *iphdrptr;

    in_addr addr;
    retval = nn::socket::InetAton(IpAdder, &addr);
    if (retval  < 0) {NN_LOG("failed to convert network address\n");}

    sockaddr_in destAddr = {};
    destAddr.sin_family = AF_INET;
    destAddr.sin_addr = addr;

    /* RAWソケットを作成します */
    socket = nn::socket::Socket(PF_INET, SOCK_RAW, IPPROTO_ICMP);
    if (socket < 0) {NN_LOG("Socket create failed @ %d\n", __LINE__); return -1;}

    // 受信タイムアウトを設定
    retval = nn::socket::SetSockOpt(socket, SOL_SOCKET, SO_RCVTIMEO, &tv, sizeof(tv));
    if (retval < 0) {
        NN_LOG("SetSockOpt error : %d\n", nn::socket::GetLastErrno());
        retval = nn::socket::Close(socket);
        if (retval < 0) {
            NN_LOG("Close error : %d\n", socket);
        }
        return retval;
    }

    memset(&senddata.hdr, 0, sizeof(senddata.hdr));

    /* ICMPヘッダ */
    senddata.hdr.icmp_type = ICMP_ECHO;
    senddata.hdr.icmp_code = 0;
    senddata.hdr.icmp_cksum = 0;
    senddata.hdr.icmp_hun.ih_idseq.icd_id = 0;

    /* Data */
    if (dataSize > 0) {
        for (i = 0; i < dataSize; i++)
            senddata.data[i] = i % 10;
    }

    /* ICMP ECHO REPLY受信部分 */
    memset(recvBuffer, 0, sizeof(recvBuffer));
    for (i = 0; i < sendCount; i++) {
        senddata.hdr.icmp_hun.ih_idseq.icd_seq = i;
        senddata.hdr.icmp_cksum = 0;
        /* ICMPヘッダのチェックサムを計算 */
        senddata.hdr.icmp_cksum = checksum(reinterpret_cast<unsigned short *>(&senddata.hdr), data_len);

        /* ICMPヘッダだけのICMPパケットを送信 */
        retval = nn::socket::SendTo(socket, (char *)&senddata, data_len, 0, reinterpret_cast<sockaddr*>(&destAddr), sizeof(destAddr));
        if (retval < 1) {NN_LOG("SendTo error : %d\n", nn::socket::GetLastErrno()); retval = -1;}

        /* 相手ホストからのICMP ECHO REPLYを待ち */
        retval = nn::socket::Recv(socket, recvBuffer, sizeof(recvBuffer), 0);
        if (retval < 1) {
            NN_LOG("Recv error : %d\n", nn::socket::GetLastErrno());
            continue;
        }

        /* 受信データからIPヘッダ部分へのポインタを取得 */
        iphdrptr = (struct ip *)recvBuffer;

        /* 受信データからICMPヘッダ部分へのポインタを取得 */
        icmpptr = (struct icmp *)(recvBuffer + (iphdrptr->ip_hl * 4));

        /* ICMPヘッダからICMPの種類を特定 */
        if (icmpptr->icmp_type == ICMP_ECHOREPLY) {
            recvCount++;
        }

        if (sendInterval > 0) {
            nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(sendInterval));
        }
    }

    /* 終了 */
    retval = nn::socket::Close(socket);
    if (retval < 0) {NN_LOG("Close error : %d\n", socket); return retval;}

    retval = recvCount;
    return retval;
}

int WlanStartIperf(char* ifName, char* para) NN_NOEXCEPT
{
    int retval = 0;

    return retval;
}

int WlanIfstate(char* ifName, char* buf) NN_NOEXCEPT
{
    nn::bsdsocket::cfg::IfState ifState;
    nn::Result result = nn::Result();
    char* pbuf;
    int len;
    int retval = 0;

    result = nn::bsdsocket::cfg::GetIfState(ifName, &ifState);
    if (result.IsSuccess()) {
        pbuf = buf;
        len = std::sprintf(pbuf, "mode,%s", (ifState.mode == nn::bsdsocket::cfg::IfIpAddrMode_Dhcp) ? "dhcp" : "static");
        pbuf += len;
        len = std::sprintf(pbuf, ",ip,%s", nn::socket::InetNtoa(ifState.addr));
        pbuf += len;
        len = std::sprintf(pbuf, ",subnet,%s", nn::socket::InetNtoa(ifState.subnetMask));
        pbuf += len;
        len = std::sprintf(pbuf, ",gw,%s", nn::socket::InetNtoa(ifState.gatewayAddr));
        pbuf += len;
        len = std::sprintf(pbuf, ",dns1,%s", nn::socket::InetNtoa(ifState.dnsAddrs[0]));
        pbuf += len;
        len = std::sprintf(pbuf, ",dns2,%s", nn::socket::InetNtoa(ifState.dnsAddrs[1]));
        pbuf += len;
        len = std::sprintf(pbuf, ",state,%s", ifState.u.modeDhcp.currentState);
    } else {
        NN_LOG("failed to get interface %s state - %d:%d\n",
               ifName, result.GetModule(), result.GetDescription());
        retval = -1;
    }

    return retval;
}

int WlanIfconfig(char* ifName, bool state, bool dhcp, char* ipAddress) NN_NOEXCEPT
{
    int retval = 0;
    nn::Result result;
    struct ifreq ifr;

    std::strcpy(ifr.ifr_name, ifName);
    ifr.ifr_flags = IFF_UP;
    int socket = nn::socket::Socket(PF_INET, SOCK_DGRAM, 0);
    if (socket < 0) {NN_LOG("Socket create failed @ %d\n", __LINE__); return -1;}
    nn::socket::Ioctl(socket, SIOCSIFFLAGS, &ifr, sizeof(struct ifreq));

    if (state) {
        nn::bsdsocket::cfg::IfSettings ifcfg;
        memset(&ifcfg, 0, sizeof(ifcfg));

        if (dhcp) {
            // dhcp
            ifcfg.mode = nn::bsdsocket::cfg::IfIpAddrMode_Dhcp;
            ifcfg.mtu  = 1500;
        } else {
            // static ip
            ifcfg.mode = nn::bsdsocket::cfg::IfIpAddrMode_Static;
            ifcfg.mtu  = 1500;
            nn::socket::InetAton(ipAddress, &ifcfg.u.modeStatic.addr);
            ifcfg.u.modeStatic.broadcastAddr.S_addr =
                    (ifcfg.u.modeStatic.addr.S_addr & ifcfg.u.modeStatic.subnetMask.S_addr) |
                    ~ifcfg.u.modeStatic.subnetMask.S_addr;
#if 0
            nn::socket::InetAton(GWIpAdder, &ifcfg.u.modeStatic.gatewayAddr);
            nn::socket::InetAton(STATIC_SUBNET_MASK, &ifcfg.u.modeStatic.subnetMask);
            nn::socket::InetAton(STATIC_DNS_1, &ifcfg.dnsAddrs[0]);
            nn::socket::InetAton(STATIC_DNS_2, &ifcfg.dnsAddrs[1]);
#endif
        }

        result = nn::bsdsocket::cfg::SetIfUp(const_cast<char*>(ifName), &ifcfg);
        if (result.IsFailure()) {
            NN_LOG("             failed to configure interface %s - %d:%d\n",
                   ifName,
                   result.GetModule(),
                   result.GetDescription());
            return -1;
        }
    } else {
        nn::bsdsocket::cfg::SetIfDown(const_cast<char*>(ifName));
    }
    nn::socket::Close(socket);

    return retval;
}

int WlanInfraConnect(char* ssid, nn::wlan::Security* security) NN_NOEXCEPT
{
    int count = 0;
    int retval = 0;
    nn::os::SystemEventType connectionEvent;
    nn::wlan::ConnectionStatus connectionStatus;
    nn::Result result;

    result = nn::wlan::Infra::GetConnectionEvent(&connectionEvent);
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);

    while(NN_STATIC_CONDITION(1)) {
        NN_LOG("WiFi Connect Start\n");
        result = nn::wlan::Infra::Connect(nn::wlan::Ssid(ssid),
                                          nn::wlan::MacAddress::CreateBroadcastMacAddress(),
                                          -1,
                                          *security,
                                          true,
                                          15);
        NN_ABORT_UNLESS_RESULT_SUCCESS(result);

        nn::os::WaitSystemEvent(&connectionEvent);
        NN_LOG("  WiFi WaitSystemEvent Signal\n");

        result = nn::wlan::Infra::GetConnectionStatus(&connectionStatus);
        NN_ABORT_UNLESS_RESULT_SUCCESS(result);

        if (connectionStatus.state == nn::wlan::ConnectionState_Connected) {
            NN_LOG("WiFi CONNECTED!!!\n");
            break;
        } else {
            NN_LOG("  NOT CONNECTED!!! STATE %d\n",connectionStatus.state);
            count ++;
            if (count >= 10) {
                nn::os::DestroySystemEvent(&connectionEvent);
                NN_LOG("Failed to Connect AP for 10 times\n");
                retval = -1;
                break;
            }
        }

        // 2秒待って再試行
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(2000));
    }
    nn::os::DestroySystemEvent(&connectionEvent);

    return retval;
}

int WlanInfraDisconnect() NN_NOEXCEPT
{
    int retval = 0;
    nn::os::SystemEventType connectionEvent;
    nn::wlan::ConnectionStatus connectionStatus;
    nn::Result result;

    result = nn::wlan::Infra::Disconnect();
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);

    result = nn::wlan::Infra::GetConnectionEvent(&connectionEvent);
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);

    while(NN_STATIC_CONDITION(1)) {
        if (nn::os::TimedWaitSystemEvent(&connectionEvent, nn::TimeSpan::FromMilliSeconds(16)) == true) {
            result = nn::wlan::Infra::GetConnectionStatus(&connectionStatus);
            NN_ABORT_UNLESS_RESULT_SUCCESS(result);
            if (connectionStatus.state != nn::wlan::ConnectionState_Connected) {
                NN_LOG("Success to disconnect with AP\n");
                break;
            }
        }
    }
    nn::os::DestroySystemEvent(&connectionEvent);

    return retval;
}

int WlanInfraInitialize() NN_NOEXCEPT
{
    nn::Result result;

    nn::bluetooth::InitializeBluetoothDriver();

    result = nn::socket::Initialize(reinterpret_cast<void*>(SocketMemoryPoolBuffer),
                                     nn::socket::DefaultSocketMemoryPoolSize,
                                     nn::socket::MinSocketAllocatorSize,
                                     KLocalConcurrencyLimit);
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);

    result = nn::wlan::InitializeSocketManager();
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);

    result = nn::wlan::InitializeInfraManager();
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);

    result = nn::wlan::Infra::OpenMode();
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);

    nn::btm::InitializeBtmInterface();

    return 0;
}

int WlanInfraUninitialize(char* ifName) NN_NOEXCEPT
{
    nn::Result result;

    nn::bsdsocket::cfg::SetIfDown(const_cast<char*>(ifName));

    nn::btm::FinalizeBtmInterface();

    result = nn::socket::Finalize();
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);

    result = nn::wlan::Infra::Disconnect();
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);

    result = nn::wlan::Infra::CloseMode();
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);

    result = nn::wlan::FinalizeInfraManager();
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);

    result = nn::wlan::FinalizeSocketManager();
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);

    nn::bluetooth::FinalizeBluetoothDriver();

    return 0;
}

bool SetBtmWlanMode(const nn::btm::WlanMode& wlanMode)
{
    nn::Result result;
    uint32_t count = 0;
    do {
        result = nn::btm::SetWlanMode(wlanMode);
        if (nn::btm::ResultBusy::Includes(result)) {
            NN_LOG(" btm is busy %d\n", count);
            nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(100));
            ++count;
        } else if (result.IsFailure()) {
            break;
        }

        // 通常 1 秒以内で終了するが、最悪ケースで 30 秒ほどかかることもある (参照:SIGLO-39875)
    } while(result.IsFailure() && count < 5 * 10);

    if (result.IsFailure()) {
        NN_LOG("  - failed : nn::btm::SetWlanMode(wlanMode = %d), Description = %d\n", wlanMode, result.GetDescription());
        return false;
    }

    // 状態変更が完了するまで待機します。
    nn::os::Tick tick = nn::os::GetSystemTick();
    nn::btm::DeviceConditionList deviceCondition;
    do {
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(50));
        nn::btm::GetConnectedDeviceCondition(&deviceCondition);
        if ((tick - nn::os::GetSystemTick()).ToTimeSpan().GetMilliSeconds() > 3000) {
            break;
        }
    } while(deviceCondition.wlanMode != wlanMode);

    return true;
}

int WlanLocalInitialize(int mode) NN_NOEXCEPT
{
    nn::Result result;

    nn::bluetooth::InitializeBluetoothDriver();

    result = nn::wlan::InitializeLocalManager();
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);

    result = nn::wlan::InitializeSocketManager();
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);

    result = nn::socket::Initialize(reinterpret_cast<void*>(SocketMemoryPoolBuffer),
                                    nn::socket::DefaultSocketMemoryPoolSize,
                                    nn::socket::MinSocketAllocatorSize,
                                    KLocalConcurrencyLimit);
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);

    nn::btm::InitializeBtmInterface();

    // デフォルトのジョイコン設定は０台
    if (SetBtmWlanMode(nn::btm::WlanMode_None) == false) {
        NN_LOG("failed @ %d\n", __LINE__);
        return -1;
    }

    switch (mode) {
    case WLAN_MODE_MASTER:
        result = nn::wlan::Local::OpenMasterMode();
        NN_ABORT_UNLESS_RESULT_SUCCESS(result);
        break;
    case WLAN_MODE_CLIENT:
        result = nn::wlan::Local::OpenClientMode();
        NN_ABORT_UNLESS_RESULT_SUCCESS(result);
        break;
    default:
        break;
    }

    return 0;
}

int WlanLocalUninitialize(int mode) NN_NOEXCEPT
{
    nn::Result result;

    nn::btm::FinalizeBtmInterface();

    result = nn::socket::Finalize();
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);

    switch (mode) {
    case WLAN_MODE_MASTER:
        result = nn::wlan::Local::CloseMasterMode();
        NN_ABORT_UNLESS_RESULT_SUCCESS(result);
        break;
    case WLAN_MODE_CLIENT:
        result = nn::wlan::Local::CloseClientMode();
        NN_ABORT_UNLESS_RESULT_SUCCESS(result);
        break;
    default:
        break;
    }

    result = nn::wlan::FinalizeSocketManager();
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);

    result = nn::wlan::FinalizeLocalManager();
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);

    nn::bluetooth::FinalizeBluetoothDriver();

    return 0;
}

int WlanDetectInitialize() NN_NOEXCEPT
{
    nn::Result result;

    nn::os::InitializeEvent(&g_rxAfEv, false, nn::os::EventClearMode_AutoClear);
    result = nn::wlan::InitializeDetectManager();
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);

    nn::wlan::WlanState wlanState;
    result = nn::wlan::Detect::GetState(&wlanState);
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);

    return 0;
}

int WlanDetectUninitialize() NN_NOEXCEPT
{
    nn::Result result;

    result = nn::wlan::FinalizeDetectManager();
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);
    nn::os::FinalizeEvent(&g_rxAfEv);

    return 0;
}

void Initialize() NN_NOEXCEPT
{
    nn::Result result;
    // デバッグ設定書き込み
    const bool isEnabled = true;

    nn::os::SetThreadNamePointer(nn::os::GetCurrentThread(), APP_NAME);
    nn::settings::fwdbg::SetSettingsItemValue("nifm",
                                              "is_communication_control_enabled_for_test", &isEnabled,
                                              sizeof(isEnabled));

    nn::nifm::Initialize();
    nn::nifm::SetWirelessCommunicationEnabledForTest(false);
    // nifmのwlan利用停止を確実に待つために1秒ほどwaitを入れておく
    nn::os::SleepThread(nn::TimeSpan::FromSeconds(1));

#if defined( NN_BUILD_CONFIG_OS_SUPPORTS_HORIZON ) && defined( NN_BUILD_CONFIG_SPEC_NX )
    // グラフィックスシステムのためのメモリ周りの初期化を行います。
    {
        const size_t GraphicsSystemMemorySize = 8 * 1024 * 1024;
        nv::SetGraphicsAllocator(NvAllocate, NvFree, NvReallocate, NULL);
        nv::SetGraphicsDevtoolsAllocator(NvAllocate, NvFree, NvReallocate, NULL);
        nv::InitializeGraphics(malloc(GraphicsSystemMemorySize), GraphicsSystemMemorySize);
    }
#endif
    nn::os::InitializeMutex(&m_RxBuffLock, false, 0);
    nn::os::InitializeMutex(&m_CmdListLock, false, 0);
    nn::os::InitializeMutex(&m_ContTxLock, false, 0);
    RecvFrameCount = 0;
    memset(&RecvFrameList, 0, sizeof(DetectFrameInfo_t) * RECV_FRAME_COUNT_MAX);
    HashFilterCount = 0;
    memset(&HashFilterList, 0, sizeof(uint64_t) * HASH_FILTER_COUNT_MAX);
    CurrentCmdIdx = 0;
    memset(&AgentCmdList, 0, sizeof(TestAgentCmd_t) * AGENT_COMMAND_MAX);
    IsAgentTesting = 0;
    SendFrameCount = 0;
}

void Uninitialize() NN_NOEXCEPT
{
    nn::os::FinalizeMutex(&m_ContTxLock);
    nn::os::FinalizeMutex(&m_CmdListLock);
    nn::os::FinalizeMutex(&m_RxBuffLock);
    return;
}
}
