﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <vector>
#include <cmath>
#include <sstream>
#include <iomanip>

#include "TA_value.h"
#include "TA_uicontrol.h"

namespace TestAgent {

/*!--------------------------------------------------------------------------*
 @brief        選択器

 *---------------------------------------------------------------------------*/
template<typename T>
class Selector: public UIControl {
public:
    Selector() {
        current_item = 0;
        cursor_position = 0;
        SetCursor('>');
        CursorSmoothness = 1.1;
        LineHeight = 32;
        FontSize = 24;
        Offset = 10;
        NextKey = Button::DOWN;
        BackKey = Button::UP;
    }

    virtual ~Selector() {
        // ##? Selector 内で new していないので delete もしない
        // while(!items.empty())
        // {
        //     delete &(items.back());
        //     items.pop_back();
        // }
    }

    virtual void Register(string name, T* item) {
        names.push_back(name);
        items.push_back(item);
    }

    virtual T* GetSelectingItem() {
        if (items.size() == 0) {
            return nullptr;
        }
        return items[current_item];
    }

    int Offset;
    int LineHeight;
    int FontSize;
    virtual void SetCursor(char cursor) {
        this->cursor[0] = cursor;
        this->cursor[1] = '\0';
    }

    double CursorSmoothness;
    Button NextKey;
    Button BackKey;

    virtual void ShowImpl(Display& display) {
        int available = Height / LineHeight;
        static int start = 0;

        if ((start + available - 1) < current_item) {
            start += current_item - (start + available - 1);
        }
        if (start > current_item) {
            start -= start - current_item;
        }

        if (start > 0) {
            display.DrawText(GetX(), GetY() - FontSize / 2, "^");
        }
        if (items.size() > start + available) {
            display.DrawText(GetX(),
                    GetY() + ((available - 1) * LineHeight) + FontSize / 2,
                    "v");
        }

        for (int i = start; i < items.size() && i < start + available; i++) {
            int x = GetX();
            int y = GetY() - start * LineHeight;

            //##?
            /*
             if(i == current_item)
             {
             double difference = abs(current_item * LineHeight - cursor_position);
             int sign = current_item * LineHeight < cursor_position ? -1 : 1;
             cursor_position += sign * ceil(difference / CursorSmoothness);
             display.SetFontSize(FontSize);
             display.DrawText(x, y + cursor_position, cursor);    // とりあえず、
             }

             display.DrawText(x + Offset, y + (i * LineHeight), GetItemName(i) );
             */
            // ##?
            if (i == current_item) {
                display.DrawText(x + Offset, y + (i * LineHeight),
                        "> " + GetItemName(i));
            } else {
                display.DrawText(x + Offset, y + (i * LineHeight),
                        "  " + GetItemName(i));
            }
        }
    }

    virtual void InputPad(Pad& key) {
        if (key.IsTrigger(BackKey)) {
            if (current_item == 0) {
                current_item = items.size() - 1;
            } else {
                current_item--;
            }
        } else if (key.IsHold(BackKey)) {
            if (current_item == 0) {
            } else {
                current_item--;
            }
        } else if (key.IsTrigger(NextKey)) {
            if (current_item == items.size() - 1) {
                current_item = 0;
            } else {
                current_item++;
            }
        } else if (key.IsHold(NextKey)) {
            if (current_item == items.size() - 1) {
            } else {
                current_item++;
            }
        }
    }

    virtual string GetItemName() {
        return GetItemName(current_item);
    }

    virtual string GetItemName(int index) {
        return names[index];
    }

    virtual void Clear() {
        items.clear();
        names.clear();
        current_item = 0;
        cursor_position = 0;
    }

protected:

    vector<T*> items;
    vector<string> names;
    int current_item;
    int cursor_position;
    char cursor[2];
};

/*!--------------------------------------------------------------------------*
 @brief        値選択器

 *---------------------------------------------------------------------------*/
class ValueSelector: public Selector<ISelectableValue> {
public:

    ValueSelector() {
        max_name_length = 0;
        NextValueKey = Button::RIGHT;
        BackValueKey = Button::LEFT;
    }

    virtual ~ValueSelector() {
    }

    virtual void Register(string name, ISelectableValue* value) {
        selector::Register(name, value);

        if (max_name_length < name.length()) {
            max_name_length = name.length();
        }
    }

    virtual void ShowImpl(Display& display) {
        int available = Height / LineHeight;
        static int start = 0;

        if ((start + available - 1) < current_item) {
            start += current_item - (start + available - 1);
        }
        if (start > current_item) {
            start -= start - current_item;
        }

        if (start > 0) {
            display.DrawText(GetX(), GetY() - FontSize / 2, "^");
        }
        if (items.size() > start + available) {
            display.DrawText(GetX(),
                    GetY() + ((available - 1) * LineHeight) + FontSize / 2,
                    "v");
        }

        for (int i = start; i < items.size() && i < start + available; ++i) {
            int x = GetX();
            int y = GetY() - start * LineHeight;

            //##?
            /*
             if(i == current_item)
             {
             double difference = abs(current_item * LineHeight - cursor_position);
             int sign = current_item * LineHeight < cursor_position ? -1 : 1;
             cursor_position += sign * ceil(difference / CursorSmoothness);
             display.SetFontSize(FontSize);
             display.DrawText(x, y + cursor_position, cursor);    // とりあえず、
             }

             display.DrawText(x + Offset, y + (i * LineHeight), GetItemName(i) );
             */
            // ##?
            if (i == current_item) {
                display.DrawText(x + Offset, y + (i * LineHeight),
                        "> " + GetItemName(i));
            } else {
                display.DrawText(x + Offset, y + (i * LineHeight),
                        "  " + GetItemName(i));
            }
        }
    }

    virtual void InputPad(Pad& key) {
        selector::InputPad(key);

        if (key.IsTrigger(NextValueKey)) {
            selector::items[selector::current_item]->Next(true);
        } else if (key.IsHold(NextValueKey)) {
            selector::items[selector::current_item]->Next();
        } else if (key.IsTrigger(BackValueKey)) {
            selector::items[selector::current_item]->Back(true);
        } else if (key.IsHold(BackValueKey)) {
            selector::items[selector::current_item]->Back();
        }
    }

    typedef Selector<ISelectableValue> selector;
    Button NextValueKey;
    Button BackValueKey;

protected:

    int max_name_length;
    virtual string GetItemName(int index) {
        ostringstream oss;
        oss << left << setw(max_name_length) << selector::names[index] << " : "
                << selector::items[index];

        return oss.str();
    }

};
}
