﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nnt.h>

#include <nn/settings/fwdbg/settings_SettingsSetterApi.h>
#include <nn/nifm.h>
#include <nn/nifm/nifm_ApiCommunicationControlForTest.h>

#include <nn/wlan/wlan_Types.h>
#include <nn/wlan/wlan_Ssid.h>
#include <nn/wlan/wlan_InfraApi.h>
#include <nn/wlan/wlan_ScanResultReader.h>
#include <nn/wlan/wlan_BssDescriptionReader.h>
#include <nn/wlan/wlan_Result.h>

namespace
{
    // スキャンバッファです。
    const size_t buffSize = 100 * 1024;
    NN_ALIGNAS(4096) char g_scanBuffer[ buffSize ];  // 100KB

    class InfraScan : public ::testing::Test
    {
    protected:
        nn::wlan::ScanParameters scanParam;

        virtual void SetUp() NN_OVERRIDE
        {
            // 基本となるスキャンパラメータです。
            scanParam.scanType = nn::wlan::ScanType_Passive;
            memset(scanParam.channelList, 0, sizeof(scanParam.channelList));
            scanParam.channelCount = 0;
            scanParam.channelScanTime = 120;
            scanParam.homeChannelTime = 0;
            scanParam.ssidList = NULL;
            scanParam.ssidCount = 0;
            scanParam.bssid = nn::wlan::MacAddress::CreateBroadcastMacAddress();

            // スキャンバッファを初期化します。
            memset(g_scanBuffer, 0, sizeof(g_scanBuffer));

            // インフラ通信モードを開始します。
            nn::wlan::InitializeInfraManager();
            nn::wlan::Infra::OpenMode();
        }

        virtual void TearDown() NN_OVERRIDE
        {
            // インフラ通信モードを終了します。
            nn::wlan::Infra::CloseMode();
            nn::wlan::FinalizeInfraManager();
        }

    public:
        void InfraScanLoop()  NN_NOEXCEPT
        {
            for( int i = 0; i < 100; i++ )
            {
                // スキャンバッファを初期化します。
                memset(g_scanBuffer, 0, sizeof(g_scanBuffer));

                // スキャンを開始します。
                NNT_ASSERT_RESULT_SUCCESS(nn::wlan::Infra::StartScan(g_scanBuffer, buffSize, scanParam));

                // スキャン結果からRSSI値を取り出し、0でないか評価します。
                nn::wlan::BeaconScanResultReader resultReader(g_scanBuffer);

                // 見付かったAPが0個だと意味がないので、ここで見付かった個数の評価を行います。
                ASSERT_NE(0, resultReader.GetCount());

                for( int j = 0; j < resultReader.GetCount(); j++ )
                {
                    nn::wlan::BeaconDescriptionReader beacon = resultReader.GetNextDescription();
                    ASSERT_NE(0, beacon.GetRssi());
                }
            }
        }
    };
}

TEST_F(InfraScan, RssiNonZeroPassive)
{
    InfraScanLoop();
}

TEST_F(InfraScan, RssiNonZeroActive)
{
    scanParam.scanType = nn::wlan::ScanType_Active;
    scanParam.channelScanTime = 40;

    InfraScanLoop();
}

//
// テストのエントリポイントです。
//
extern "C" void nnMain()
{
    int argc = nn::os::GetHostArgc();
    char **argv = nn::os::GetHostArgv();
    ::testing::InitGoogleTest(&argc, argv);

    // 無線スイッチをオフにします。
    {
        // デバッグ設定書き込み。
        const bool isEnabled = true;
        nn::settings::fwdbg::SetSettingsItemValue("nifm", "is_communication_control_enabled_for_test", &isEnabled, sizeof(isEnabled));

        nn::nifm::Initialize();
        nn::nifm::SetWirelessCommunicationEnabledForTest(false);
        // nifmのwlan利用停止を確実に待つために1秒ほどwaitを入れておきます。
        nn::os::SleepThread(nn::TimeSpan::FromSeconds(1));
    }

    // テストを実行します。
    int result = RUN_ALL_TESTS();
    nnt::Exit(result);
}
