﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#include <random>
#include <nnt.h>
#include <nn/os.h>
#include <nn/nn_Log.h>

#include <nn/settings/fwdbg/settings_SettingsSetterApi.h>
#include <nn/nifm.h>
#include <nn/nifm/nifm_ApiCommunicationControlForTest.h>

#include <nn/wlan/wlan_Types.h>
#include <nn/wlan/wlan_Ssid.h>
#include <nn/wlan/wlan_DetectApi.h>
#include <nn/wlan/wlan_Result.h>

using namespace nn::wlan;
namespace
{
    // 送信バッファです。
    const size_t txBuffSize = DetectActionFramePayloadMax;
    NN_ALIGNAS(4096) uint8_t g_txBuffer[ txBuffSize ];
    // 受信バッファです。
    const size_t rxBuffSize = OneShotActionFrameSizeMax;
    NN_ALIGNAS(4096) uint8_t g_rxBuffer[ rxBuffSize ];

    const DetectHash myHash = {
            {0x01, 0x23, 0x45, 0x67, 0x89, 0xab, 0xcd, 0xef }
    };

    nn::os::ThreadType g_rxThread;
    const size_t threadStackSize = 4096;
    NN_ALIGNAS(4096) char g_threadStack[ threadStackSize ];

    static bool g_ExitFlagRx = false;

    class DetectBasic : public ::testing::Test
    {
    protected:
        virtual void SetUp() NN_OVERRIDE
        {
            // Detect通信を開始します。
            InitializeDetectManager();
            NNT_ASSERT_RESULT_SUCCESS(Detect::OpenMode());
        }

        virtual void TearDown() NN_OVERRIDE
        {
            // Detect通信を終了します。
            NNT_ASSERT_RESULT_SUCCESS(Detect::CloseMode());
            FinalizeDetectManager();
        }

    public:
        void DetectGetState() NN_NOEXCEPT
        {
            WlanState state;
            NNT_ASSERT_RESULT_SUCCESS(Detect::GetState(&state));
            ASSERT_EQ(WlanState_DetectIdle, state);
        }

        void DetectGetMacAddress() NN_NOEXCEPT
        {
            MacAddress mac;
            NNT_ASSERT_RESULT_SUCCESS(Detect::GetMacAddress(&mac));
            ASSERT_NE(MacAddress::CreateBroadcastMacAddress(), mac);
            ASSERT_NE(MacAddress::CreateZeroMacAddress(), mac);
        }

        void DetectGetAllowedChannels() NN_NOEXCEPT
        {
            int16_t channels[WirelessChannelsCountMax];
            uint32_t count;
            NNT_ASSERT_RESULT_SUCCESS(Detect::GetAllowedChannels(channels, &count));
            ASSERT_NE(0, count);
        }

        void DetectStartStopCommunication() NN_NOEXCEPT
        {
            NNT_ASSERT_RESULT_SUCCESS(Detect::StartCommunication());
            NNT_ASSERT_RESULT_SUCCESS(Detect::StopCommunication());
        }

        void DetectPutActionFrameOneShot() NN_NOEXCEPT
        {
            GenerateTxFrame();
            NNT_ASSERT_RESULT_SUCCESS(Detect::StartCommunication());
            NNT_ASSERT_RESULT_SUCCESS(Detect::PutActionFrameOneShot(
                    MacAddress::CreateBroadcastMacAddress(),
                    ActionFrameType_Detect,
                    myHash,
                    g_txBuffer,
                    sizeof(g_txBuffer),
                    0
                    ));
            NNT_ASSERT_RESULT_SUCCESS(Detect::StopCommunication());
        }

        void DetectStartStopPeriodicActionFrame() NN_NOEXCEPT
        {
            GenerateTxFrame();
            NNT_ASSERT_RESULT_SUCCESS(Detect::StartCommunication());
            NNT_ASSERT_RESULT_SUCCESS(Detect::StartPeriodicActionFrame(
                    ActionFrameType_Detect,
                    myHash,
                    g_txBuffer,
                    sizeof(g_txBuffer),
                    40
                    ));
            NNT_ASSERT_RESULT_SUCCESS(Detect::CancelPeriodicActionFrame());
            NNT_ASSERT_RESULT_SUCCESS(Detect::StopCommunication());
        }

        void DetectGetActionFrame() NN_NOEXCEPT
        {
            uint32_t rxId;
            uint16_t afTypes[] = {
                    static_cast<uint16_t>(nn::wlan::ActionFrameType_Detect),
            };
            NNT_ASSERT_RESULT_SUCCESS(Detect::CreateRxEntryForActionFrame(
                    &rxId,
                    afTypes,
                    sizeof(afTypes) / sizeof(uint16_t),
                    10
                    ));
            NNT_ASSERT_RESULT_SUCCESS(nn::os::CreateThread(
                    &g_rxThread,
                    ReceiveActionFrame,
                    &rxId,
                    g_threadStack,
                    threadStackSize,
                    nn::os::DefaultThreadPriority
                    ));
            nn::os::StartThread(&g_rxThread);

            NNT_ASSERT_RESULT_SUCCESS(Detect::StartCommunication());
            NNT_ASSERT_RESULT_SUCCESS(Detect::StopCommunication());

            // 受信停止
            g_ExitFlagRx = true;
            NNT_ASSERT_RESULT_SUCCESS(Detect::CancelGetActionFrame(rxId));
            nn::os::WaitThread(&g_rxThread);
            nn::os::DestroyThread(&g_rxThread);
            NNT_ASSERT_RESULT_SUCCESS(Detect::DeleteRxEntryForActionFrame(rxId));
        }

        void DetectAddDeleteSubtype() NN_NOEXCEPT
        {
            uint32_t rxId;
            uint32_t outRxId;
            uint16_t afTypes[] = {
                    static_cast<uint16_t>(nn::wlan::ActionFrameType_Detect),
            };
            NNT_ASSERT_RESULT_SUCCESS(Detect::CreateRxEntryForActionFrame(
                    &rxId,
                    afTypes,
                    sizeof(afTypes) / sizeof(uint16_t),
                    10
                    ));
            NNT_ASSERT_RESULT_SUCCESS(Detect::AddSubtypeToRxEntryForActionFrame(
                    rxId,
                    ActionFrameType_Reserved1
                    ));
            outRxId = rxId;
            NNT_ASSERT_RESULT_SUCCESS(Detect::DeleteSubtypeFromRxEntryForActionFrame(
                    &outRxId,
                    ActionFrameType_Reserved1
                    ));
            ASSERT_EQ(outRxId, rxId);
            NNT_ASSERT_RESULT_SUCCESS(Detect::DeleteRxEntryForActionFrame(rxId));
        }

        void DetectReserveCancelSleep() NN_NOEXCEPT
        {
            NNT_ASSERT_RESULT_SUCCESS(Detect::ReserveDetectSleep());
            NNT_ASSERT_RESULT_SUCCESS(Detect::CancelDetectSleep());
        }

        void DetectRequestSleepWakeUpNormal() NN_NOEXCEPT
        {
            NNT_ASSERT_RESULT_SUCCESS(Detect::RequestSleep());
            NNT_ASSERT_RESULT_SUCCESS(Detect::RequestWakeUp());
        }

        void DetectRequestSleepWakeUp() NN_NOEXCEPT
        {
            GenerateTxFrame();
            NNT_ASSERT_RESULT_SUCCESS(Detect::SetActionFrameForSleep(
                    ActionFrameType_Detect,
                    myHash,
                    g_txBuffer,
                    sizeof(g_txBuffer)
                    ));
            NNT_ASSERT_RESULT_SUCCESS(Detect::ReserveDetectSleep());
            NNT_ASSERT_RESULT_SUCCESS(Detect::RequestSleep());
            NNT_ASSERT_RESULT_SUCCESS(Detect::RequestWakeUp());
        }

        void DetectSetHashList() NN_NOEXCEPT
        {
            uint64_t hashList[DetectSleepFilterMax];
            // ランダムなハッシュリストの作成
            std::random_device seed;
            std::mt19937_64 engine(seed());
            for( int i = 0; i < DetectSleepFilterMax; i++ )
            {
                hashList[i] = engine();
            }
            NNT_ASSERT_RESULT_SUCCESS(Detect::SetHashList(
                    hashList,
                    sizeof(hashList) / sizeof(hashList[0])
                    ));
        }

        void DetectSetPeriodicActionFrameCycle() NN_NOEXCEPT
        {
            DetectPeriodicAfCycle cycle1 = {
                    100,  // tx interval
                    6,    // tx count
                    1,    // idle count
                    2,    // rx start
                    2     // rx count
            };
            DetectPeriodicAfCycle cycle2 = {
                    40,   // tx interval
                    5,    // tx count
                    2,    // idle count
                    0,    // rx start
                    2     // rx count
            };
            DetectPeriodicAfCycle cycle3 = {
                    120,  // tx interval
                    10,   // tx count
                    3,    // idle count
                    4,    // rx start
                    3     // rx count
            };
            NNT_ASSERT_RESULT_SUCCESS(Detect::SetPeriodicActionFrameCycle(
                    cycle1,
                    DetectPeriodicAfCycleTarget_Both
                    ));
            NNT_ASSERT_RESULT_SUCCESS(Detect::SetPeriodicActionFrameCycle(
                    cycle2,
                    DetectPeriodicAfCycleTarget_Hd
                    ));
            NNT_ASSERT_RESULT_SUCCESS(Detect::SetPeriodicActionFrameCycle(
                    cycle3,
                    DetectPeriodicAfCycleTarget_Sa
                    ));
        }

        void DetectPutActionFrameOneShotEx() NN_NOEXCEPT
        {
            GenerateTxFrame();
            MacAddress mac(0x7c, 0xbb, 0x8a, 0x12, 0x34, 0x56);
            DetectHeader myDhp = {
                    0x1,     // major
                    0x0,     // minor
                    0x1,     // cmd
                    0,       // reserved
                    myHash   // hash
            };
            NNT_ASSERT_RESULT_SUCCESS(Detect::StartCommunication());
            NNT_ASSERT_RESULT_SUCCESS(Detect::PutActionFrameOneShotEx(
                    MacAddress::CreateBroadcastMacAddress(),
                    mac,
                    ActionFrameType_Detect,
                    myDhp,
                    g_txBuffer,
                    sizeof(g_txBuffer),
                    0
                    ));
            NNT_ASSERT_RESULT_SUCCESS(Detect::StopCommunication());
        }

    private:
        void GenerateTxFrame() NN_NOEXCEPT
        {
            uint8_t j = 0;
            for( int i = 0; i < sizeof(g_txBuffer); i++ )
            {
                g_txBuffer[i] = j;
                if( j == 0xFF )
                {
                    j = 0;
                }
                else
                {
                    j++;
                }
            }
        }

        static void ReceiveActionFrame(void* arg) NN_NOEXCEPT
        {
            nn::Result result;
            g_ExitFlagRx = false;

            uint32_t* rxId = reinterpret_cast<uint32_t*>(arg);
            nn::wlan::MacAddress mac;
            uint16_t channel;
            int16_t rssi;
            nn::os::Tick tick;

            while( g_ExitFlagRx != true )
            {
                size_t rxSize = 0;
                result = Detect::GetActionFrame(
                        &mac,
                        g_rxBuffer,
                        sizeof(g_rxBuffer),
                        &rxSize,
                        *rxId,
                        &channel,
                        &rssi,
                        &tick
                        );
                if( result.GetDescription() == nn::wlan::ResultGetFrameCancelled().GetDescription() )
                {
                    NN_LOG("canceled\n");
                }
                else
                {
                    ASSERT_EQ(true, result.IsSuccess());
                }
            }
        }
    };
}

TEST_F(DetectBasic, GetState)
{
    DetectGetState();
}

TEST_F(DetectBasic, GetMacAddress)
{
    DetectGetMacAddress();
}

TEST_F(DetectBasic, GetAllowedChannels)
{
    DetectGetAllowedChannels();
}

TEST_F(DetectBasic, StartStopCommunication)
{
    DetectStartStopCommunication();
}

TEST_F(DetectBasic, PutActionFrameOneShot)
{
    DetectPutActionFrameOneShot();
}

TEST_F(DetectBasic, StartStopPeriodicActionFrame)
{
    DetectStartStopPeriodicActionFrame();
}

TEST_F(DetectBasic, GetActionFrame)
{
    DetectGetActionFrame();
}

TEST_F(DetectBasic, AddDeleteSubtype)
{
    DetectAddDeleteSubtype();
}

TEST_F(DetectBasic, ReserveCancelSleep)
{
    DetectReserveCancelSleep();
}

TEST_F(DetectBasic, RequestSleepWakeUpNormal)
{
    DetectRequestSleepWakeUpNormal();
}

TEST_F(DetectBasic, RequestSleepWakeUp)
{
    DetectRequestSleepWakeUp();
}

TEST_F(DetectBasic, SetHashList)
{
    DetectSetHashList();
}

TEST_F(DetectBasic, SetPeriodicActionFrameCycle)
{
    DetectSetPeriodicActionFrameCycle();
}

TEST_F(DetectBasic, PutActionFrameOneShotEx)
{
    DetectPutActionFrameOneShotEx();
}

//
// テストのエントリポイントです。
//
extern "C" void nnMain()
{
    int argc = nn::os::GetHostArgc();
    char **argv = nn::os::GetHostArgv();
    ::testing::InitGoogleTest(&argc, argv);

    // 無線スイッチをオフにします。
    {
        // デバッグ設定書き込み。
        const bool isEnabled = true;
        nn::settings::fwdbg::SetSettingsItemValue("nifm", "is_communication_control_enabled_for_test", &isEnabled, sizeof(isEnabled));

        nn::nifm::Initialize();
        nn::nifm::SetWirelessCommunicationEnabledForTest(false);
        // nifmのwlan利用停止を確実に待つために1秒ほどwaitを入れておきます。
        nn::os::SleepThread(nn::TimeSpan::FromSeconds(1));
    }

    // テストを実行します。
    int result = RUN_ALL_TESTS();
    nnt::Exit(result);
}
