﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <memory>

#include <nnt.h>
#include <nnt/result/testResult_Assert.h>

#include "../../Common/testWlan_localApiClass.h"
#include "../../Common/testWlan_UnitTest.h"
#include "../../Common/testWlan_UnitTestCommon.h"

#include <nn/init.h>
#include <nn/settings/fwdbg/settings_SettingsSetterApi.h>

NN_ALIGNAS(4096) uint8_t  g_MallocBuffer[8 * 1024 * 1024];

extern "C" void nninitStartup()
{
    nn::init::InitializeAllocator(g_MallocBuffer, sizeof(g_MallocBuffer));
}

namespace WlanTest {

    /********************************************/
    // Class 定義
    /********************************************/
    class LocalMasterExceptionStateUnitTest : public ::testing::Test
    {
    protected:
        nn::Result Result;
        nn::wlan::Ssid ssid;
        nn::Bit8 ssdid_test[Wlantest_ssidlengthmax];

        nn::wlan::ClientStatus clientStatus[nn::wlan::ConnectableClientsCountMax];
        nn::Bit32 clientStatusBitMap = 0;

        nn::wlan::MacAddress Bssid;
        int32_t Rssi;
        nn::wlan::LinkLevel LinkLevel;

        nn::wlan::Security security = {
            nn::wlan::SecurityMode_Open,
            nn::wlan::SecurityMode_Open,
            0,
            "012345678",  // Key
        };

    protected:
        LocalMasterExceptionStateUnitTest()  NN_NOEXCEPT
        {
            memset(ssdid_test, 0, sizeof(ssdid_test));
            strcmp((char *)ssdid_test, "a");
            ssid.Set(ssdid_test, Ssidsizemax);
        }


        virtual void SetUp() NN_NOEXCEPT NN_OVERRIDE
        {
        }

        virtual void TearDown() NN_NOEXCEPT NN_OVERRIDE
        {
        }

    };

    class LocalMasterExceptionArgmentUnitTest : public ::testing::Test
    {
    protected:
        LocalApiClass localMaster;
        nn::Result Result;

    protected:
        virtual void SetUp() NN_NOEXCEPT NN_OVERRIDE
        {
            WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::InitializeLocalManager());
            WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::OpenMasterMode());
        }

        virtual void TearDown() NN_NOEXCEPT NN_OVERRIDE
        {
            nn::os::SleepThread(nn::TimeSpan::FromSeconds(3));
            WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::CloseMasterMode());
            WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::FinalizeLocalManager());
        }

    };

    TEST_F(LocalMasterExceptionArgmentUnitTest, OpenCloseClientMode_No5_No7)
    {
        Result = nn::wlan::Local::OpenClientMode();
        WLANTEST_EXPECT_TRUE(true == nn::wlan::ResultInvalidState::Includes(Result));
        Result = nn::wlan::Local::CloseClientMode();
        WLANTEST_EXPECT_TRUE(true == nn::wlan::ResultInvalidState::Includes(Result));
    }

    TEST_F(LocalMasterExceptionArgmentUnitTest, OpenCloseSpectatorMode_No9_No11)
    {
        Result = nn::wlan::Local::OpenSpectatorMode();
        WLANTEST_EXPECT_TRUE(true == nn::wlan::ResultInvalidState::Includes(Result));
        Result = nn::wlan::Local::CloseSpectatorMode();
        WLANTEST_EXPECT_TRUE(true == nn::wlan::ResultInvalidState::Includes(Result));
    }

    TEST_F(LocalMasterExceptionArgmentUnitTest, Connect_No20)
    {
        nn::wlan::Ssid ssid = nn::wlan::Ssid("LocalException");
        nn::wlan::Security security = {
            nn::wlan::SecurityMode_Open,
            nn::wlan::SecurityMode_Open,
            0,
            "012345678",  // Key
        };
        Result = nn::wlan::Local::Connect(ssid, nn::wlan::MacAddress::CreateBroadcastMacAddress()
                                        , -1, security, true, nn::wlan::BeaconIndication_Enable, 10);
        WLANTEST_EXPECT_RESULT_NOTSUCCESS(Result);
        WLANTEST_EXPECT_TRUE(true == nn::wlan::ResultInvalidState::Includes(Result));
    }

    TEST_F(LocalMasterExceptionArgmentUnitTest, Connect_No24)
    {
        Result = nn::wlan::Local::CancelConnect();
        WLANTEST_EXPECT_TRUE(true == nn::wlan::ResultInvalidState::Includes(Result));
    }

    TEST_F(LocalMasterExceptionArgmentUnitTest, CancelJoin_No30)
    {
        Result = nn::wlan::Local::CancelJoin();
        WLANTEST_EXPECT_TRUE(true == nn::wlan::ResultInvalidState::Includes(Result));
    }


    TEST_F(LocalMasterExceptionArgmentUnitTest, Disconnect_No35)
    {
        nn::wlan::DisconnectClient disClient;

        const nn::wlan::MasterBssParameters boundaryTestBss = {
            1, false, 30, false, nn::wlan::RateSetLegacy_11gMask, nn::wlan::RateSetLegacy_11gMask,
            { nn::wlan::SecurityMode_Open, nn::wlan::SecurityMode_Open, 0, "" }, nn::wlan::Ssid(LocalMasterTestSSID), 100
        };

        WLANTEST_EXPECT_TRUE(localMaster.SetupLocalMaster(boundaryTestBss, TestMatchInfo, 1));
        WLANTEST_EXPECT_TRUE(localMaster.LocalMasterConnectWait(localMaster.m_clientMacArray, WlanTestMasterWaitTimeOut, 1));
        disClient.clientMacAddress = nn::wlan::MacAddress(0x11, 0x11, 0x11, 0x11, 0x11, 0x11);
        disClient.reasonCode = nn::wlan::Dot11ReasonCode_Reserved;
        Result = nn::wlan::Local::Disconnect(nn::wlan::LocalCommunicationMode_Master, &disClient);
        WLANTEST_EXPECT_RESULT_NOTSUCCESS(Result);
        WLANTEST_EXPECT_TRUE(true == nn::wlan::ResultClientNotFound::Includes(Result));

        localMaster.LocalRelease();
    }

    TEST_F(LocalMasterExceptionArgmentUnitTest, GetClientStatus_No42)
    {
        nn::wlan::ClientStatus clientStatus[nn::wlan::ConnectableClientsCountMax];
        Result = nn::wlan::Local::GetClientStatus(clientStatus, NULL);
        WLANTEST_EXPECT_TRUE(true == nn::wlan::ResultInvalidArgument::Includes(Result));
    }

    TEST_F(LocalMasterExceptionArgmentUnitTest, GetMacAddress_No53)
    {
        Result = nn::wlan::Local::GetMacAddress(NULL);
        WLANTEST_EXPECT_TRUE(true == nn::wlan::ResultInvalidArgument::Includes(Result));
    }

    TEST_F(LocalMasterExceptionArgmentUnitTest, GetRssi_No55)
    {
        int32_t Rssi;
        Result = nn::wlan::Local::GetRssi(&Rssi);
        WLANTEST_EXPECT_TRUE(true == nn::wlan::ResultInvalidState::Includes(Result));
    }

    TEST_F(LocalMasterExceptionArgmentUnitTest, GetLinkLevel_No59)
    {
        nn::wlan::LinkLevel LinkLevel;
        Result = nn::wlan::Local::GetLinkLevel(&LinkLevel);
        WLANTEST_EXPECT_TRUE(true == nn::wlan::ResultInvalidState::Includes(Result));
    }

    TEST_F(LocalMasterExceptionArgmentUnitTest, SetMaxAssociationNumber_No123)
    {
        Result = nn::wlan::Local::SetMaxAssociationNumber(-1);
        WLANTEST_EXPECT_TRUE(true == nn::wlan::ResultInvalidArgument::Includes(Result));
    }

    extern "C" void nnMain()
    {
        int     argc = nn::os::GetHostArgc();
        char**  argv = nn::os::GetHostArgv();
        NN_LOG("argc=%d\n", argc);
        for (int i = 0; i < argc; i++)
        {
            NN_LOG("argv[%d]=%s\n", argc, argv[i]);
        }

        const bool isEnabled = true;

        WLANTEST_ASSERT_RESULT_SUCCESS(nn::nifm::Initialize());
        nn::settings::fwdbg::SetSettingsItemValue("nifm", "is_communication_control_enabled_for_test", &isEnabled, sizeof(isEnabled));
        nn::nifm::SetWirelessCommunicationEnabledForTest(false);
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(1000));

        ::testing::InitGoogleTest(&argc, argv);

        int result = RUN_ALL_TESTS();
        nnt::Exit(result);
    }

} // namespace WlanTest

