﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <memory>

#include "../../Common/testWlan_localApiClass.h"
#include "../../Common/testWlan_UnitTest.h"
#include "../../Common/testWlan_UnitTestCommon.h"

#include <nn/init.h>
#include <nn/settings/fwdbg/settings_SettingsSetterApi.h>

NN_ALIGNAS(4096) uint8_t  g_MallocBuffer[8 * 1024 * 1024];

extern "C" void nninitStartup()
{
    nn::init::InitializeAllocator(g_MallocBuffer, sizeof(g_MallocBuffer));
}

namespace {

    const uint32_t WlanTestClinetConnectTime = 30 * 1000;

    // CreateRxEntry
    const uint16_t boundaryethertypes[RxEntryCountmax] = { 0x88B7, 0x88B8, 0x88B9, 0x88BA, 0x88BB, 0x88BC, 0x88BD, 0x88BE, 0x88BF, 0x88C0, 0x88C1, 0x88C2, 0x88C3, 0x88C4, 0x88C5, 0x88C6 };

    const nn::wlan::ScanParameters boudaryDefaultScanParam = {
        nn::wlan::ScanType_Active,
        { 1, 6, 11 },
        0,
        -1,
        -1,
        nullptr,
        0,
        nn::wlan::MacAddress::CreateBroadcastMacAddress()
    };

    const nn::wlan::ScanIeMatchInfo boudaryDefaultscanIeMatch = {
        { 0x00,0x22,0xAA,0xBB,0xCC },
        5
    };

    const WlanTest::LocalClientConnectParameter BounaryConnectParam = {
        nn::wlan::Ssid(LocalMasterTestSSID), nn::wlan::MacAddress::CreateBroadcastMacAddress(), 1,
        { nn::wlan::SecurityMode_Open, nn::wlan::SecurityMode_Open, 0, "" }, true, nn::wlan::BeaconIndication_Enable, 10
    };
}

namespace WlanTest{

    /********************************************/
    // Class 定義
    /********************************************/
    class LocalClientBoundaryUnitTest : public ::testing::Test
    {
    protected:
        // StartScan
        size_t  size = BufferSize100k;
        std::unique_ptr<char[]>  pTestMemchar;
        nn::wlan::Ssid ssid;
        nn::Bit8 ssdid_test[WlanTest::Wlantest_ssidlengthmax];
        nn::wlan::ScanParameters scanParam;
        nn::wlan::ScanIeMatchInfo scanIeMatch;

        // Connect
        LocalApiClass localClient;
        nn::os::SystemEventType connectionEvent;
        const uint8_t LocalVendorIe[7] = {
                0x00, 0x22, 0xAA, 0x12, 0x34, 0x56, 0x78
        };

        nn::wlan::Ssid masterSsid;           // MASTERのSSID格納先
        nn::wlan::MacAddress masterBssid;    // MASTERのBSSID格納先
        int16_t masterChannel;               // MASTERのChannel格納先
        nn::Bit8 matchingOui[3] = {
                LocalVendorIe[0],
                LocalVendorIe[1],
                LocalVendorIe[2],
        };  // MASTERのVIEのOUI

        nn::wlan::ScanParameters scanParamConnect = {
            nn::wlan::ScanType_Passive,
            {1, 6, 11, 36, 40, 44, 48},
            7,  // チャンネルリストに記載したうちの何チャンネル分をスキャンするか
            120,
            0,
            nullptr,
            0,
            nn::wlan::MacAddress::CreateBroadcastMacAddress()
        };
        // 相手先のセキュリティ情報入力
        nn::wlan::Security security = {
                nn::wlan::SecurityMode_Open,//StaticAes,
                nn::wlan::SecurityMode_Open,//StaticAes,
                0,
                "",
        };
        // Disconnect
        nn::wlan::DisconnectClient disconnect = {
                nn::wlan::MacAddress::CreateBroadcastMacAddress(),
                nn::wlan::Dot11ReasonCode_Reserved
        };

        // GetFrameRaw
        std::unique_ptr<uint8_t[]> pTestMemGetFrameRaw;
        size_t     out_size = 0;

        // AddEthertypeToRxEntry type は 0x0600 以上
        uint16_t ethertypes_u16 = 0x0600;

        // DeleteEthertypeFromRxEntry
        uint32_t rxId = 0;

        // AddMatchingDataToRxEntry
        nn::wlan::ReceivedDataMatchInfo receiveMatch = {
                {0x7F,0x01, 0x02, 0x03, nn::wlan::ActionFrameType_Local, 0x01, 0x02, 0x03, 0x04, 0x05},
                10
        };

        // SetActionFrameWithBeacon
        uint8_t  txBuffer = 0x00;

        // GetActionFrame
        uint8_t  rxBufferAf[2048];
        size_t   OutSize=0;

        uint16_t afTypes[10] = {   // 受信したいActionFrameType）
                static_cast<uint16_t>(nn::wlan::ActionFrameType_Beacon),
                static_cast<uint16_t>(nn::wlan::ActionFrameType_Local),
                static_cast<uint16_t>(nn::wlan::ActionFrameType_Reserved1),
                static_cast<uint16_t>(nn::wlan::ActionFrameType_Detect),
                static_cast<uint16_t>(nn::wlan::ActionFrameType_Reserved3),
                static_cast<uint16_t>(nn::wlan::ActionFrameType_Reserved1),
                static_cast<uint16_t>(nn::wlan::ActionFrameType_Detect),
                static_cast<uint16_t>(nn::wlan::ActionFrameType_Reserved3),
                static_cast<uint16_t>(nn::wlan::ActionFrameType_End),
                static_cast<uint16_t>(nn::wlan::ActionFrameType_Reserved1)
        };
        // GetActionFrame
        nn::wlan::MacAddress macAddr;

    protected:
        LocalClientBoundaryUnitTest() NN_NOEXCEPT
        {
            std::unique_ptr<char[]>pTestMemchar_tmp(new char[size]);
            pTestMemchar = std::move(pTestMemchar_tmp);

            memset(ssdid_test, 0, sizeof(ssdid_test));
            ssid.Set(ssdid_test , Ssidsizemin - 1);

            std::unique_ptr<uint8_t[]>pTestMemGetFrameRaw_tmp(new uint8_t[size]);
            pTestMemGetFrameRaw = std::move(pTestMemGetFrameRaw_tmp);

            macAddr = nn::wlan::MacAddress::CreateBroadcastMacAddress();

            memcpy(ssdid_test, LocalMasterTestSSID, sizeof(LocalMasterTestSSID));
            masterSsid.Set(ssdid_test, Ssidsizemax);

            masterBssid = nn::wlan::MacAddress::CreateBroadcastMacAddress();
        }

        virtual void SetUp() NN_NOEXCEPT NN_OVERRIDE
        {
            scanParam = boudaryDefaultScanParam;
            scanParam.ssidList = &ssid;
            scanIeMatch = boudaryDefaultscanIeMatch;

            WLANTEST_ASSERT_TRUE(pTestMemchar.get() != nullptr);
            WLANTEST_ASSERT_TRUE(pTestMemGetFrameRaw.get() != nullptr);

            WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::InitializeLocalManager());
            WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::OpenClientMode());
        }

        virtual void TearDown() NN_NOEXCEPT NN_OVERRIDE
        {
            WLANTEST_EXPECT_TRUE(localClient.LocalClientDisconnect(WlanTestDisconnectTimeOut));
            localClient.LocalRelease();
            WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::CloseClientMode());
            WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::FinalizeLocalManager());
        }

        void SetUpConnect(nn::wlan::Ssid setMasterSsid, nn::wlan::Security& setSecurity, int beaconTimeout, bool isCreateEntry)  NN_NOEXCEPT
        {
            LocalClientConnectParameter connectParam;
            std::memcpy(&setSecurity.key[0], &StaticAesKey[0], 16);
            std::memcpy(&connectParam, &BounaryConnectParam, sizeof(LocalClientConnectParameter));

            connectParam.ssid               = setMasterSsid;
            connectParam.beaconLostTimeout  = beaconTimeout;
            connectParam.channel            = Channelmin;
            connectParam.security           = setSecurity;
            connectParam.autoKeepAlive      = true;
            connectParam.indication         = nn::wlan::BeaconIndication_Enable;

            if (isCreateEntry == true)
            {
                WLANTEST_ASSERT_TRUE(localClient.SetupLocalClient(connectParam.ssid, connectParam.security, TestMatchInfo, 1, true, true));
            }
            else
            {
                WLANTEST_ASSERT_TRUE(localClient.SetupLocalClient(connectParam.ssid, connectParam.security/*, TestMatchInfo, 1*/));
            }
            WLANTEST_ASSERT_TRUE(localClient.LocalClientConnect(connectParam, WlanTestClinetConnectTime));
        }

        void SetUpConnect(nn::wlan::Ssid setMasterSsid, nn::wlan::Security& setSecurity, uint16_t channel, int beaconTimeout, bool isCreateEntry)  NN_NOEXCEPT
        {
            LocalClientConnectParameter connectParam;
            std::memcpy(&setSecurity.key[0], &StaticAesKey[0], 16);
            std::memcpy(&connectParam, &BounaryConnectParam, sizeof(LocalClientConnectParameter));

            connectParam.ssid                   = setMasterSsid;
            connectParam.beaconLostTimeout      = beaconTimeout;
            connectParam.channel                = channel;
            connectParam.security               = setSecurity;
            connectParam.autoKeepAlive          = true;
            connectParam.indication             = nn::wlan::BeaconIndication_Enable;

            if (isCreateEntry == true)
            {
                WLANTEST_ASSERT_TRUE(localClient.SetupLocalClient(connectParam.ssid, connectParam.security, TestMatchInfo, 1, true, true));
            }
            else
            {
                WLANTEST_ASSERT_TRUE(localClient.SetupLocalClient(connectParam.ssid, connectParam.security/*, TestMatchInfo, 1*/));
            }
            WLANTEST_ASSERT_TRUE(localClient.LocalClientConnect(connectParam, WlanTestClinetConnectTime));
        }

        void SetUpConnect()  NN_NOEXCEPT
        {
            LocalClientConnectParameter connectParam;
            std::memcpy(&connectParam, &BounaryConnectParam, sizeof(LocalClientConnectParameter));

            std::memcpy(&security.key[0], &StaticAesKey[0], 16);

            connectParam.ssid               = masterSsid;
            connectParam.beaconLostTimeout  = Beaconlosttimemin;
            connectParam.channel            = Channelmin;
            connectParam.security           = security;
            connectParam.autoKeepAlive      = true;
            connectParam.indication         = nn::wlan::BeaconIndication_Enable;

            WLANTEST_ASSERT_TRUE(localClient.SetupLocalClient(connectParam.ssid, connectParam.security/*, TestMatchInfo, 1*/));
            WLANTEST_ASSERT_TRUE(localClient.LocalClientConnect(connectParam, WlanTestClinetConnectTime));
        }

        void DeleteSubtypeFromRxEntryForActionFrameInit()  NN_NOEXCEPT
        {
            WLANTEST_EXPECT_RESULT_SUCCESS(nn::wlan::Local::CreateRxEntryForActionFrame(&rxId, afTypes, sizeof(afTypes) / sizeof(uint16_t), 10));
        }

        void GetActionFrameInit()  NN_NOEXCEPT
        {
            WLANTEST_EXPECT_RESULT_SUCCESS(nn::wlan::Local::CreateRxEntryForActionFrame(&rxId, afTypes, sizeof(afTypes) / sizeof(uint16_t), 10));
        }
    };

    /*************************/
    // StartScan Test
    /*************************/
    TEST_F(LocalClientBoundaryUnitTest, StartScanTest_No30)
    {
        gResult = nn::wlan::Local::StartScan(pTestMemchar.get(), Scansizemin - 1, scanParam, &scanIeMatch);
        WLANTEST_EXPECT_TRUE(true == nn::wlan::ResultInvalidArgument::Includes(gResult));
    }

    TEST_F(LocalClientBoundaryUnitTest, StartScanTest_No31)
    {
        WLANTEST_EXPECT_RESULT_SUCCESS(nn::wlan::Local::StartScan(pTestMemchar.get(), Scansizemin, scanParam, &scanIeMatch));
    }

    TEST_F(LocalClientBoundaryUnitTest, StartScanTest_No34)
    {
        scanParam.scanType = static_cast<nn::wlan::ScanType>(nn::wlan::ScanType_Active - 1);
        gResult = nn::wlan::Local::StartScan(pTestMemchar.get(), size, scanParam, &scanIeMatch);
        WLANTEST_EXPECT_TRUE(true == nn::wlan::ResultInvalidArgument::Includes(gResult));
    }

    TEST_F(LocalClientBoundaryUnitTest, StartScanTest_No35)
    {
        scanParam.scanType = nn::wlan::ScanType_Active;
        WLANTEST_EXPECT_RESULT_SUCCESS(nn::wlan::Local::StartScan(pTestMemchar.get(), size, scanParam, &scanIeMatch));
    }

    TEST_F(LocalClientBoundaryUnitTest, StartScanTest_No36)
    {
        scanParam.scanType = static_cast<nn::wlan::ScanType>(nn::wlan::ScanType_Passive + 1);
        gResult = nn::wlan::Local::StartScan(pTestMemchar.get(), size, scanParam, &scanIeMatch);
        WLANTEST_EXPECT_TRUE(true == nn::wlan::ResultInvalidArgument::Includes(gResult));
    }

    TEST_F(LocalClientBoundaryUnitTest, StartScanTest_No37)
    {
        scanParam.scanType = nn::wlan::ScanType_Passive;
        WLANTEST_EXPECT_RESULT_SUCCESS(nn::wlan::Local::StartScan(pTestMemchar.get(), size, scanParam, &scanIeMatch));

        scanParam.scanType = nn::wlan::ScanType_Active;
    }

    TEST_F(LocalClientBoundaryUnitTest, StartScanTest_No39)
    {
        scanParam.channelCount = Channelcountsizemin;
        WLANTEST_EXPECT_RESULT_SUCCESS(nn::wlan::Local::StartScan(pTestMemchar.get(), size, scanParam, &scanIeMatch));
    }

    TEST_F(LocalClientBoundaryUnitTest, StartScanTest_No40)
    {
        for (int i = 0; i < (sizeof(scanParam.channelList) / sizeof(scanParam.channelList[0])); i++)
        {
            scanParam.channelList[i] = 1;
        }
        scanParam.channelCount = Channelcountsizemax + 1;
        gResult = nn::wlan::Local::StartScan(pTestMemchar.get(), size, scanParam, &scanIeMatch);
        WLANTEST_EXPECT_TRUE(true == nn::wlan::ResultInvalidArgument::Includes(gResult));
    }

    TEST_F(LocalClientBoundaryUnitTest, StartScanTest_No41)
    {
        for (int i = 0; i < (sizeof(scanParam.channelList) / sizeof(scanParam.channelList[0])); i++)
        {
            scanParam.channelList[i] = 1;
        }
        scanParam.channelCount = Channelcountsizemax;
        WLANTEST_EXPECT_RESULT_SUCCESS(nn::wlan::Local::StartScan(pTestMemchar.get(), size, scanParam, &scanIeMatch));
    }

    TEST_F(LocalClientBoundaryUnitTest, StartScanTest_No42)
    {
        scanParam.channelScanTime = Channelscantimedefault - 1;
        gResult = nn::wlan::Local::StartScan(pTestMemchar.get(), size, scanParam, &scanIeMatch);
        WLANTEST_EXPECT_TRUE(true == nn::wlan::ResultInvalidArgument::Includes(gResult));
    }

    TEST_F(LocalClientBoundaryUnitTest, StartScanTest_No43)
    {
        scanParam.channelScanTime = Channelscantimedefault;
        WLANTEST_EXPECT_RESULT_SUCCESS(nn::wlan::Local::StartScan(pTestMemchar.get(), size, scanParam, &scanIeMatch));
    }

    TEST_F(LocalClientBoundaryUnitTest, StartScanTest_No44)
    {
        scanParam.channelScanTime = Channelscantimedefault + 1;
        gResult = nn::wlan::Local::StartScan(pTestMemchar.get(), size, scanParam, &scanIeMatch);
        WLANTEST_EXPECT_TRUE(true == nn::wlan::ResultInvalidArgument::Includes(gResult));
    }

    TEST_F(LocalClientBoundaryUnitTest, StartScanTest_No45)
    {
        scanParam.channelScanTime = Channelscantimemin - 1;
        gResult = nn::wlan::Local::StartScan(pTestMemchar.get(), size, scanParam, &scanIeMatch);
        WLANTEST_EXPECT_TRUE(true == nn::wlan::ResultInvalidArgument::Includes(gResult));
    }

    TEST_F(LocalClientBoundaryUnitTest, StartScanTest_No46)
    {
        scanParam.channelScanTime = Channelscantimemin;
        WLANTEST_EXPECT_RESULT_SUCCESS(nn::wlan::Local::StartScan(pTestMemchar.get(), size, scanParam, &scanIeMatch));
    }

    TEST_F(LocalClientBoundaryUnitTest, StartScanTest_No47)
    {
        scanParam.channelScanTime = Channelscantimemax + 1;
        gResult = nn::wlan::Local::StartScan(pTestMemchar.get(), size, scanParam, &scanIeMatch);
        WLANTEST_EXPECT_TRUE(true == nn::wlan::ResultInvalidArgument::Includes(gResult));
    }

    TEST_F(LocalClientBoundaryUnitTest, StartScanTest_No48)
    {
        scanParam.channelScanTime = Channelscantimemax;
        WLANTEST_EXPECT_RESULT_SUCCESS(nn::wlan::Local::StartScan(pTestMemchar.get(), size, scanParam, &scanIeMatch));
    }

    TEST_F(LocalClientBoundaryUnitTest, StartScanTest_No49)
    {
        scanParam.homeChannelTime = Homechanneltimemin - 1;
        gResult = nn::wlan::Local::StartScan(pTestMemchar.get(), size, scanParam, &scanIeMatch);
        WLANTEST_EXPECT_RESULT_NOTSUCCESS(gResult);
        WLANTEST_EXPECT_TRUE(true == nn::wlan::ResultInvalidArgument::Includes(gResult));
    }

    TEST_F(LocalClientBoundaryUnitTest, StartScanTest_No50)
    {
        scanParam.homeChannelTime = Homechanneltimemin;
        WLANTEST_EXPECT_RESULT_SUCCESS(nn::wlan::Local::StartScan(pTestMemchar.get(), size, scanParam, &scanIeMatch));
    }

    TEST_F(LocalClientBoundaryUnitTest, StartScanTest_No51)
    {
        scanParam.homeChannelTime = Homechanneltimemax + 1;
        gResult = nn::wlan::Local::StartScan(pTestMemchar.get(), size, scanParam, &scanIeMatch);
        WLANTEST_EXPECT_TRUE(true == nn::wlan::ResultInvalidArgument::Includes(gResult));
    }

    TEST_F(LocalClientBoundaryUnitTest, StartScanTest_No52)
    {
        scanParam.homeChannelTime = Homechanneltimemax;
        WLANTEST_EXPECT_RESULT_SUCCESS(nn::wlan::Local::StartScan(pTestMemchar.get(), size, scanParam, &scanIeMatch));
    }

    TEST_F(LocalClientBoundaryUnitTest, StartScanTest_No54)
    {
        scanIeMatch.matchLength = MatchLengthmin;
        WLANTEST_EXPECT_RESULT_SUCCESS(nn::wlan::Local::StartScan(pTestMemchar.get(), size, scanParam, &scanIeMatch));
    }

    TEST_F(LocalClientBoundaryUnitTest, StartScanTest_No56)
    {
        scanIeMatch.matchLength = MatchLengthmax;
        WLANTEST_EXPECT_RESULT_SUCCESS(nn::wlan::Local::StartScan(pTestMemchar.get(), size, scanParam, &scanIeMatch));
    }

    /*************************/
    // Connect Test
    /*************************/
    TEST_F(LocalClientBoundaryUnitTest, ConnectTest_No57)
    {
        ssid.Set(ssdid_test, Ssidsizemin - 1);
        std::memcpy(&security.key[0], &StaticAesKey[0], 16);
        gResult = nn::wlan::Local::Connect(ssid, masterBssid, -1, security, true, nn::wlan::BeaconIndication_Enable, 10);
        WLANTEST_EXPECT_TRUE(true == nn::wlan::ResultInvalidArgument::Includes(gResult));
    }

    TEST_F(LocalClientBoundaryUnitTest, ConnectTest_No58)
    {
        ssid.Set(ssdid_test , Ssidsizemin);
        std::memcpy(&security.key[0], &StaticAesKey[0], 16);
        WLANTEST_EXPECT_RESULT_SUCCESS(nn::wlan::Local::Connect(ssid, masterBssid, -1, security, true, nn::wlan::BeaconIndication_Enable, 10));
        WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::Disconnect(nn::wlan::LocalCommunicationMode_ClientSpectator, nullptr));
    }

    TEST_F(LocalClientBoundaryUnitTest, ConnectTest_No59)
    {
        strcmp((char *)ssdid_test, Ssid_max);
        ssid.Set(ssdid_test , Ssidsizemax + 1);
        std::memcpy(&security.key[0], &StaticAesKey[0], 16);
        gResult = nn::wlan::Local::Connect(ssid, masterBssid, -1, security, true, nn::wlan::BeaconIndication_Enable, 10);
        WLANTEST_EXPECT_TRUE(true == nn::wlan::ResultInvalidArgument::Includes(gResult));
    }

    TEST_F(LocalClientBoundaryUnitTest, ConnectTest_No60)
    {
        ssid.Set(ssdid_test , Ssidsizemax);
        std::memcpy(&security.key[0], &StaticAesKey[0], 16);
        WLANTEST_EXPECT_RESULT_SUCCESS(nn::wlan::Local::Connect(ssid, masterBssid, -1, security, true, nn::wlan::BeaconIndication_Enable, 10));
        WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::Disconnect(nn::wlan::LocalCommunicationMode_ClientSpectator, nullptr));
    }

    TEST_F(LocalClientBoundaryUnitTest, ConnectTest_No61)
    {
        std::memcpy(&security.key[0], &StaticAesKey[0], 16);
        gResult = nn::wlan::Local::Connect(masterSsid, masterBssid, Channelmin - 1, security, true, nn::wlan::BeaconIndication_Enable, 10);
        WLANTEST_EXPECT_TRUE(nn::wlan::ResultInvalidArgument::Includes(gResult));
    }

    TEST_F(LocalClientBoundaryUnitTest, ConnectTest_No62)
    {
        std::memcpy(&security.key[0], &StaticAesKey[0], 16);
        WLANTEST_EXPECT_RESULT_SUCCESS(nn::wlan::Local::Connect(masterSsid, masterBssid, Channelmin, security, true, nn::wlan::BeaconIndication_Enable, 10));
        WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::Disconnect(nn::wlan::LocalCommunicationMode_ClientSpectator, nullptr));
    }

    TEST_F(LocalClientBoundaryUnitTest, ConnectTest_No63)
    {
        std::memcpy(&security.key[0], &StaticAesKey[0], 16);
        gResult = nn::wlan::Local::Connect(masterSsid, masterBssid, Channelmax + 1, security, true, nn::wlan::BeaconIndication_Enable, 10);
        WLANTEST_EXPECT_TRUE(true == nn::wlan::ResultInvalidArgument::Includes(gResult));
    }

    TEST_F(LocalClientBoundaryUnitTest, ConnectTest_No64)
    {
        std::memcpy(&security.key[0], &StaticAesKey[0], 16);
        WLANTEST_EXPECT_RESULT_SUCCESS(nn::wlan::Local::Connect(masterSsid, masterBssid, Channelmax, security, true, nn::wlan::BeaconIndication_Enable, 10));
        WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::Disconnect(nn::wlan::LocalCommunicationMode_ClientSpectator, nullptr));
    }

    TEST_F(LocalClientBoundaryUnitTest, ConnectTest_No65)
    {
        security.privacyMode = static_cast<nn::wlan::SecurityMode>(nn::wlan::SecurityMode_Open - 1);
        std::memcpy(&security.key[0], &StaticAesKey[0], 16);
        gResult = nn::wlan::Local::Connect(masterSsid, masterBssid, Channelmin, security, true, nn::wlan::BeaconIndication_Enable, 10);
        WLANTEST_EXPECT_TRUE(true == nn::wlan::ResultInvalidArgument::Includes(gResult));
    }

    TEST_F(LocalClientBoundaryUnitTest, ConnectTest_No66)
    {
        security.privacyMode = nn::wlan::SecurityMode_Open;
        std::memcpy(&security.key[0], &StaticAesKey[0], 16);
        WLANTEST_EXPECT_RESULT_SUCCESS(nn::wlan::Local::Connect(masterSsid, masterBssid, Channelmin, security, true, nn::wlan::BeaconIndication_Enable, 10));
        WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::Disconnect(nn::wlan::LocalCommunicationMode_ClientSpectator, nullptr));
    }

    TEST_F(LocalClientBoundaryUnitTest, ConnectTest_No67)
    {
        security.privacyMode = static_cast<nn::wlan::SecurityMode>(nn::wlan::SecurityMode_StaticAes + 1);
        std::memcpy(&security.key[0], &StaticAesKey[0], 16);
        gResult = nn::wlan::Local::Connect(masterSsid, masterBssid, Channelmin, security, true, nn::wlan::BeaconIndication_Enable, 10);
        WLANTEST_EXPECT_TRUE(true == nn::wlan::ResultInvalidArgument::Includes(gResult));
    }

    TEST_F(LocalClientBoundaryUnitTest, ConnectTest_No68)
    {
        security.privacyMode = nn::wlan::SecurityMode_StaticAes;
        std::memcpy(&security.key[0], &StaticAesKey[0], 16);
        WLANTEST_EXPECT_RESULT_SUCCESS(nn::wlan::Local::Connect(masterSsid, masterBssid, Channelmin, security, true, nn::wlan::BeaconIndication_Enable, 10));
        WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::Disconnect(nn::wlan::LocalCommunicationMode_ClientSpectator, nullptr));
    }

    TEST_F(LocalClientBoundaryUnitTest, ConnectTest_No73)
    {
        security.keyIdx = Keyinxmin - 1;
        std::memcpy(&security.key[0], &StaticAesKey[0], 16);
        WLANTEST_EXPECT_RESULT_SUCCESS(nn::wlan::Local::Connect(masterSsid, masterBssid, Channelmin, security, true, nn::wlan::BeaconIndication_Enable, 10));
        WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::Disconnect(nn::wlan::LocalCommunicationMode_ClientSpectator, nullptr));
    }

    TEST_F(LocalClientBoundaryUnitTest, ConnectTest_No74)
    {
        security.keyIdx = Keyinxmin;
        std::memcpy(&security.key[0], &StaticAesKey[0], 16);
        WLANTEST_EXPECT_RESULT_SUCCESS(nn::wlan::Local::Connect(masterSsid, masterBssid, Channelmin, security, true, nn::wlan::BeaconIndication_Enable, 10));
        WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::Disconnect(nn::wlan::LocalCommunicationMode_ClientSpectator, nullptr));
    }

    TEST_F(LocalClientBoundaryUnitTest, ConnectTest_No75)
    {
        security.keyIdx = Keyinxmax + 1;
        std::memcpy(&security.key[0], &StaticAesKey[0], 16);
        gResult = nn::wlan::Local::Connect(masterSsid, masterBssid, Channelmin, security, true, nn::wlan::BeaconIndication_Enable, 10);
        WLANTEST_EXPECT_RESULT_NOTSUCCESS(gResult);
        WLANTEST_EXPECT_TRUE(true == nn::wlan::ResultInvalidArgument::Includes(gResult));
    }

    TEST_F(LocalClientBoundaryUnitTest, ConnectTest_No76)
    {
        security.keyIdx = Keyinxmax;
        std::memcpy(&security.key[0], &StaticAesKey[0], 16);
        WLANTEST_EXPECT_RESULT_SUCCESS(nn::wlan::Local::Connect(masterSsid, masterBssid, Channelmin, security, true, nn::wlan::BeaconIndication_Enable, 10));
        WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::Disconnect(nn::wlan::LocalCommunicationMode_ClientSpectator, nullptr));
    }

    TEST_F(LocalClientBoundaryUnitTest, ConnectTest_No81)
    {
        std::memcpy(&security.key[0], &StaticAesKey[0], 16);
        gResult = nn::wlan::Local::Connect(masterSsid, masterBssid, Channelmin, security, true, nn::wlan::BeaconIndication_Enable, Beaconlosttimemin - 1);
        WLANTEST_EXPECT_TRUE(true == nn::wlan::ResultInvalidArgument::Includes(gResult));
    }

    TEST_F(LocalClientBoundaryUnitTest, ConnectTest_No82)
    {
        std::memcpy(&security.key[0], &StaticAesKey[0], 16);
        WLANTEST_EXPECT_RESULT_SUCCESS(nn::wlan::Local::Connect(masterSsid, masterBssid, Channelmin, security, true, nn::wlan::BeaconIndication_Enable, Beaconlosttimemin));
        WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::Disconnect(nn::wlan::LocalCommunicationMode_ClientSpectator, nullptr));
    }

    TEST_F(LocalClientBoundaryUnitTest, ConnectTest_No83)
    {
        std::memcpy(&security.key[0], &StaticAesKey[0], 16);
        gResult = nn::wlan::Local::Connect(masterSsid, masterBssid, Channelmin, security, true, nn::wlan::BeaconIndication_Enable, Beaconlosttimemax + 1);
        WLANTEST_EXPECT_TRUE(true == nn::wlan::ResultInvalidArgument::Includes(gResult));
    }

    TEST_F(LocalClientBoundaryUnitTest, ConnectTest_No84)
    {
        std::memcpy(&security.key[0], &StaticAesKey[0], 16);
        WLANTEST_EXPECT_RESULT_SUCCESS(nn::wlan::Local::Connect(masterSsid, masterBssid, Channelmin, security, true, nn::wlan::BeaconIndication_Enable, Beaconlosttimemax));
        WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::Disconnect(nn::wlan::LocalCommunicationMode_ClientSpectator, nullptr));
    }

    /*************************/
    // Disconnect Test
    /*************************/
    TEST_F(LocalClientBoundaryUnitTest, DisconnectTest_No105)
    {
        SetUpConnect(nn::wlan::Ssid(LocalMasterTestSSID), security, 10, true);
        gResult = nn::wlan::Local::Disconnect(static_cast<nn::wlan::LocalCommunicationMode>(nn::wlan::LocalCommunicationMode_Master - 1), nullptr);
        WLANTEST_EXPECT_TRUE(true == nn::wlan::ResultInvalidArgument::Includes(gResult));
        nn::wlan::Local::Disconnect(nn::wlan::LocalCommunicationMode_ClientSpectator, nullptr);
    }

    TEST_F(LocalClientBoundaryUnitTest, DisconnectTest_No106)
    {
        SetUpConnect(nn::wlan::Ssid(LocalMasterTestSSID), security, 10, true);
        WLANTEST_EXPECT_RESULT_NOTSUCCESS(nn::wlan::Local::Disconnect(nn::wlan::LocalCommunicationMode_Master, nullptr));
        nn::wlan::Local::Disconnect(nn::wlan::LocalCommunicationMode_ClientSpectator, nullptr);
    }

    TEST_F(LocalClientBoundaryUnitTest, DisconnectTest_No107)
    {
        SetUpConnect(nn::wlan::Ssid(LocalMasterTestSSID), security, 10, true);
        gResult = nn::wlan::Local::Disconnect(static_cast<nn::wlan::LocalCommunicationMode>(nn::wlan::LocalCommunicationMode_ClientSpectator + 1), nullptr);
        WLANTEST_EXPECT_TRUE(true == nn::wlan::ResultInvalidArgument::Includes(gResult));
        nn::wlan::Local::Disconnect(nn::wlan::LocalCommunicationMode_ClientSpectator, nullptr);
    }

    TEST_F(LocalClientBoundaryUnitTest, DisconnectTest_No108)
    {
        SetUpConnect(nn::wlan::Ssid(LocalMasterTestSSID), security, 10, true);
        WLANTEST_EXPECT_RESULT_SUCCESS(nn::wlan::Local::Disconnect(nn::wlan::LocalCommunicationMode_ClientSpectator, nullptr));
    }


    /*************************/
    // GetFrameRaw Test
    /*************************/
    TEST_F(LocalClientBoundaryUnitTest, GetFrameRawTest_No130)
    {
        SetUpConnect(nn::wlan::Ssid(LocalMasterTestSSID), security, 1, 10, true);
        WLANTEST_EXPECT_RESULT_NOTSUCCESS(nn::wlan::Local::GetFrameRaw(pTestMemGetFrameRaw.get(), 0, &out_size, localClient.m_entRxId));
    }

    TEST_F(LocalClientBoundaryUnitTest, GetFrameRawTest_No132)
    {
        SetUpConnect(nn::wlan::Ssid(LocalMasterTestSSID), security, 1, 10, true);
        WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::GetFrameRaw(pTestMemGetFrameRaw.get(), FrameSizemax, &out_size, localClient.m_entRxId));
    }

    TEST_F(LocalClientBoundaryUnitTest, GetFrameRawTest_No133)
    {
        nn::wlan::ConnectionStatus connectionStatus;

        rxId = RxIdmin;
        SetUpConnect(nn::wlan::Ssid(LocalMasterTestSSID), security, 1, 10, true);
        WLANTEST_EXPECT_RESULT_SUCCESS(nn::wlan::Local::GetConnectionStatus(&connectionStatus));
        NN_LOG("             LocalClient GetFrameRawTest_No133 State : %s(%d)\n", WlanConnectionStateStr[connectionStatus.state], connectionStatus.state);

        gResult = nn::wlan::Local::GetFrameRaw(pTestMemGetFrameRaw.get(), 0, &out_size, rxId);
        WLANTEST_EXPECT_RESULT_NOTSUCCESS(gResult);
        WLANTEST_EXPECT_TRUE(true == nn::wlan::ResultRxEntryNotFound::Includes(gResult));
    }

    TEST_F(LocalClientBoundaryUnitTest, GetFrameRawTest_No135)
    {
        rxId = RxIdmax;
        SetUpConnect(nn::wlan::Ssid(LocalMasterTestSSID), security, 1, 10, true);

        gResult = nn::wlan::Local::GetFrameRaw(pTestMemGetFrameRaw.get(), 0, &out_size, rxId);
        WLANTEST_EXPECT_RESULT_NOTSUCCESS(gResult);
        WLANTEST_EXPECT_TRUE(true == nn::wlan::ResultRxEntryNotFound::Includes(gResult));
    }

    /*************************/
    // CancelGetFrame Test
    /*************************/
    TEST_F(LocalClientBoundaryUnitTest, CancelGetFrameTest_No136)
    {
        NN_LOG("         CancelGetFrameTest_No136->START\n");
        gResult = nn::wlan::Local::CancelGetFrame(RxIdmin);
        NN_LOG("         CancelGetFrameTest_No136->END\n\n");
        WLANTEST_EXPECT_TRUE(true == nn::wlan::ResultRxEntryNotFound::Includes(gResult));
    }

    TEST_F(LocalClientBoundaryUnitTest, CancelGetFrameTest_No137)
    {
        gResult = nn::wlan::Local::CancelGetFrame(RxIdmax);
        WLANTEST_EXPECT_TRUE(true == nn::wlan::ResultRxEntryNotFound::Includes(gResult));
    }

    /*************************/
    // CreateRxEntry Test
    /*************************/
    TEST_F(LocalClientBoundaryUnitTest, CreateRxEntryTest_No139)
    {
        WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::CreateRxEntry(&rxId, boundaryethertypes, RxEntryCountmin, RxEntryCapacitymax));
        WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::DeleteRxEntry(rxId));
    }

    TEST_F(LocalClientBoundaryUnitTest, CreateRxEntryTest_No140)
    {
        WLANTEST_EXPECT_RESULT_NOTSUCCESS(nn::wlan::Local::CreateRxEntry(&rxId, boundaryethertypes, RxEntryCountmax + 1, RxEntryCapacitymax));
    }

    TEST_F(LocalClientBoundaryUnitTest, CreateRxEntryTest_No141)
    {
        WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::CreateRxEntry(&rxId, boundaryethertypes, RxEntryCountmax, RxEntryCapacitymax));
        WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::DeleteRxEntry(rxId));
    }

#if 0 // SIGLO-49573で仕様が確認できるまでは NT
    TEST_F(LocalClientBoundaryUnitTest, CreateRxEntryTest_No142)
    {
        WLANTEST_EXPECT_RESULT_NOTSUCCESS(nn::wlan::Local::CreateRxEntry(&rxId, boundaryethertypes, RxEntryCountmin, RxEntryCapacitymin - 1));
    }
#endif

    TEST_F(LocalClientBoundaryUnitTest, CreateRxEntryTest_No143)
    {
        WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::CreateRxEntry(&rxId, boundaryethertypes, RxEntryCountmin, RxEntryCapacitymin));
        WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::DeleteRxEntry(rxId));
    }

    TEST_F(LocalClientBoundaryUnitTest, CreateRxEntryTest_No144)
    {
        gResult = nn::wlan::Local::CreateRxEntry(&rxId, boundaryethertypes, RxEntryCountmin, RxEntryCapacitymax + 1);
        WLANTEST_EXPECT_RESULT_NOTSUCCESS(gResult);
        WLANTEST_EXPECT_TRUE(true == nn::wlan::ResultInvalidArgument::Includes(gResult));
    }

    TEST_F(LocalClientBoundaryUnitTest, CreateRxEntryTest_No145)
    {
        WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::CreateRxEntry(&rxId, boundaryethertypes, RxEntryCountmin, RxEntryCapacitymax));
        WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::DeleteRxEntry(rxId));
    }

    /*************************/
    // DeleteRxEntry Test
    /*************************/
    TEST_F(LocalClientBoundaryUnitTest, DeleteRxEntryTest_No146)
    {
        WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::CreateRxEntry(&rxId, boundaryethertypes, RxEntryCountmin, RxEntryCapacitymax));
        gResult = nn::wlan::Local::DeleteRxEntry(RxIdmin);
        WLANTEST_EXPECT_TRUE(true == nn::wlan::ResultRxEntryNotFound::Includes(gResult));

        nn::wlan::Local::DeleteRxEntry(rxId);
    }

    TEST_F(LocalClientBoundaryUnitTest, DeleteRxEntryTest_No147)
    {
        WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::CreateRxEntry(&rxId, boundaryethertypes, RxEntryCountmin, RxEntryCapacitymax));
        gResult = nn::wlan::Local::DeleteRxEntry(RxIdmax);
        WLANTEST_EXPECT_TRUE(true == nn::wlan::ResultRxEntryNotFound::Includes(gResult));

        nn::wlan::Local::DeleteRxEntry(rxId);
    }

    /*************************/
    // AddEthertypeToRxEntry Test
    /*************************/
    TEST_F(LocalClientBoundaryUnitTest, AddEthertypeToRxEntryTest_No148)
    {
        gResult = nn::wlan::Local::AddEthertypeToRxEntry(RxIdmin, ethertypes_u16);
        WLANTEST_EXPECT_TRUE(true == nn::wlan::ResultRxEntryNotFound::Includes(gResult));
    }

    TEST_F(LocalClientBoundaryUnitTest, AddEthertypeToRxEntryTest_No149)
    {
        gResult = nn::wlan::Local::AddEthertypeToRxEntry(RxIdmax, ethertypes_u16);
        WLANTEST_EXPECT_TRUE(true == nn::wlan::ResultRxEntryNotFound::Includes(gResult));
    }

    /*************************/
    // DeleteEthertypeFromRxEntry Test
    /*************************/
    TEST_F(LocalClientBoundaryUnitTest, DeleteEthertypeFromRxEntryTest_No154)
    {
        gResult = nn::wlan::Local::DeleteEthertypeFromRxEntry(&rxId, ethertypes_u16);
        WLANTEST_EXPECT_TRUE(true == nn::wlan::ResultRxEntryNotFound::Includes(gResult));
    }

    TEST_F(LocalClientBoundaryUnitTest, DeleteEthertypeFromRxEntryTest_No155)
    {
        rxId = 0xFFFFFFFF;
        gResult = nn::wlan::Local::DeleteEthertypeFromRxEntry(&rxId, ethertypes_u16);
        WLANTEST_EXPECT_TRUE(true == nn::wlan::ResultRxEntryNotFound::Includes(gResult));
    }

    /*************************/
    // AddMatchingDataToRxEntry Test
    /*************************/
    TEST_F(LocalClientBoundaryUnitTest, AddMatchingDataToRxEntryTest_No160)
    {
        gResult = nn::wlan::Local::AddMatchingDataToRxEntry(RxIdmin, receiveMatch);
        WLANTEST_EXPECT_TRUE(true == nn::wlan::ResultRxEntryNotFound::Includes(gResult));
    }

    TEST_F(LocalClientBoundaryUnitTest, AddMatchingDataToRxEntryTest_No161)
    {
        gResult = nn::wlan::Local::AddMatchingDataToRxEntry(RxIdmax, receiveMatch);
        WLANTEST_EXPECT_TRUE(true == nn::wlan::ResultRxEntryNotFound::Includes(gResult));
    }

    TEST_F(LocalClientBoundaryUnitTest, AddMatchingDataToRxEntryTest_No162)
    {
        memset(&receiveMatch.matchData[0], 0x00, sizeof(receiveMatch.matchData));
        WLANTEST_EXPECT_RESULT_SUCCESS(nn::wlan::Local::CreateRxEntry(&rxId, boundaryethertypes, Createrxcountmin, Createrxcapacitynormal));

        receiveMatch.matchLength = 0;
        WLANTEST_EXPECT_RESULT_SUCCESS(nn::wlan::Local::AddMatchingDataToRxEntry(rxId, receiveMatch));
        gResult = nn::wlan::Local::AddMatchingDataToRxEntry(rxId, receiveMatch);
        WLANTEST_EXPECT_RESULT_NOTSUCCESS(gResult);
        WLANTEST_EXPECT_TRUE(true == nn::wlan::ResultDataMatchInfoError::Includes(gResult));

        WLANTEST_EXPECT_RESULT_SUCCESS(nn::wlan::Local::DeleteRxEntry(rxId));
    }

    TEST_F(LocalClientBoundaryUnitTest, AddMatchingDataToRxEntryTest_No163)
    {
        receiveMatch.matchLength = 1;
        WLANTEST_EXPECT_RESULT_SUCCESS(nn::wlan::Local::CreateRxEntry(&rxId, boundaryethertypes, Createrxcountmin, Createrxcapacitynormal));

        WLANTEST_EXPECT_RESULT_SUCCESS(nn::wlan::Local::AddMatchingDataToRxEntry(rxId, receiveMatch));

        WLANTEST_EXPECT_RESULT_SUCCESS(nn::wlan::Local::DeleteRxEntry(rxId));
    }

#if 0 // 最小値の条件が仕様にないためNT
    TEST_F(LocalClientBoundaryUnitTest, AddMatchingDataToRxEntryTest_No164)
    {
        WLANTEST_EXPECT_RESULT_SUCCESS(nn::wlan::Local::CreateRxEntry(&rxId, boundaryethertypes, Createrxcountmin, Createrxcapacitynormal));

        receiveMatch.matchLength = nn::wlan::ReceivedDataMatchLengthMax + 1;
        for(int i = 0; i < nn::wlan::ReceivedDataMatchLengthMax; i++)
        {
            receiveMatch.matchData[i] = i;
        }

        gResult = nn::wlan::Local::AddMatchingDataToRxEntry(rxId, receiveMatch);
        WLANTEST_EXPECT_RESULT_NOTSUCCESS(gResult);
        WLANTEST_EXPECT_TRUE(true == nn::wlan::ResultInvalidArgument::Includes(gResult));

        WLANTEST_EXPECT_RESULT_SUCCESS(nn::wlan::Local::DeleteRxEntry(rxId));
    }
#endif

    TEST_F(LocalClientBoundaryUnitTest, AddMatchingDataToRxEntryTest_No165)
    {
        WLANTEST_EXPECT_RESULT_SUCCESS(nn::wlan::Local::CreateRxEntry(&rxId, boundaryethertypes, Createrxcountmin, Createrxcapacitynormal));

        receiveMatch.matchLength = nn::wlan::ReceivedDataMatchLengthMax;
        WLANTEST_EXPECT_RESULT_SUCCESS(nn::wlan::Local::AddMatchingDataToRxEntry(rxId, receiveMatch));

        WLANTEST_EXPECT_RESULT_SUCCESS(nn::wlan::Local::DeleteRxEntry(rxId));
    }

    /*************************/
    // RemoveMatchingDataFromRxEntry Test
    /*************************/
    TEST_F(LocalClientBoundaryUnitTest, RemoveMatchingDataFromRxEntryTest_No166)
    {
        gResult = nn::wlan::Local::RemoveMatchingDataFromRxEntry(RxIdmin, receiveMatch);

        WLANTEST_EXPECT_TRUE(true == nn::wlan::ResultRxEntryNotFound::Includes(gResult));
    }

    TEST_F(LocalClientBoundaryUnitTest, RemoveMatchingDataFromRxEntryTest_No167)
    {
        gResult = nn::wlan::Local::RemoveMatchingDataFromRxEntry(RxIdmax, receiveMatch);

        WLANTEST_EXPECT_TRUE(true == nn::wlan::ResultRxEntryNotFound::Includes(gResult));
    }

    TEST_F(LocalClientBoundaryUnitTest, RemoveMatchingDataFromRxEntryTest_No168)
    {
        receiveMatch.matchLength = 0;
        gResult = nn::wlan::Local::RemoveMatchingDataFromRxEntry(rxId, receiveMatch);
        WLANTEST_EXPECT_TRUE(true == nn::wlan::ResultRxEntryNotFound::Includes(gResult));
    }

    TEST_F(LocalClientBoundaryUnitTest, RemoveMatchingDataFromRxEntryTest_No169)
    {
        receiveMatch.matchLength = 1;
        gResult = nn::wlan::Local::RemoveMatchingDataFromRxEntry(rxId, receiveMatch);
        WLANTEST_EXPECT_TRUE(true == nn::wlan::ResultRxEntryNotFound::Includes(gResult));
    }

    TEST_F(LocalClientBoundaryUnitTest, RemoveMatchingDataFromRxEntryTest_No170)
    {
        receiveMatch.matchLength = nn::wlan::ReceivedDataMatchLengthMax + 1;
        gResult = nn::wlan::Local::RemoveMatchingDataFromRxEntry(rxId, receiveMatch);
        WLANTEST_EXPECT_TRUE(true == nn::wlan::ResultInvalidArgument::Includes(gResult));
    }

    TEST_F(LocalClientBoundaryUnitTest, RemoveMatchingDataFromRxEntryTest_No171)
    {
        receiveMatch.matchLength = nn::wlan::ReceivedDataMatchLengthMax;
        gResult = nn::wlan::Local::RemoveMatchingDataFromRxEntry(rxId, receiveMatch);
        WLANTEST_EXPECT_TRUE(true == nn::wlan::ResultRxEntryNotFound::Includes(gResult));
    }

    /*************************/
    // GetActionFrame Test
    /*************************/
    TEST_F(LocalClientBoundaryUnitTest, GetActionFrameTest_No188)
    {
        WLANTEST_EXPECT_RESULT_SUCCESS(nn::wlan::Local::CreateRxEntryForActionFrame(&rxId, afTypes, Createrxcountmin, Createrxcapacitynormal));

        SetUpConnect(nn::wlan::Ssid(LocalMasterTestSSID), security, 10, false);
        gResult = nn::wlan::Local::GetActionFrame(&macAddr, rxBufferAf, 0, &OutSize, rxId);
        WLANTEST_EXPECT_RESULT_NOTSUCCESS(gResult);
        WLANTEST_EXPECT_TRUE(true == nn::wlan::ResultInvalidArgument::Includes(gResult));
        WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::Disconnect(nn::wlan::LocalCommunicationMode_ClientSpectator, nullptr));
        nn::wlan::Local::DeleteRxEntryForActionFrame(rxId);
        localClient.LocalRelease();
    }

    TEST_F(LocalClientBoundaryUnitTest, GetActionFrameTest_No189)
    {
        WLANTEST_EXPECT_RESULT_SUCCESS(nn::wlan::Local::CreateRxEntryForActionFrame(&rxId, afTypes, Createrxcountmin, Createrxcapacitynormal));

        SetUpConnect(nn::wlan::Ssid(LocalMasterTestSSID), security, 10, false);
        gResult = nn::wlan::Local::GetActionFrame(&macAddr, rxBufferAf, Getactionsizemin, &OutSize, rxId);
        WLANTEST_EXPECT_RESULT_NOTSUCCESS(gResult);
        WLANTEST_EXPECT_TRUE(true == nn::wlan::ResultBufferTooShort::Includes(gResult));
        WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::Disconnect(nn::wlan::LocalCommunicationMode_ClientSpectator, nullptr));
        nn::wlan::Local::DeleteRxEntryForActionFrame(rxId);
        localClient.LocalRelease();
    }

    TEST_F(LocalClientBoundaryUnitTest, GetActionFrameTest_No192)
    {
        SetUpConnect(nn::wlan::Ssid(LocalMasterTestSSID), security, 10, false);
        gResult = nn::wlan::Local::GetActionFrame(&macAddr, rxBufferAf, 1024, &OutSize, RxIdmin);
        WLANTEST_EXPECT_TRUE(true == nn::wlan::ResultRxEntryNotFound::Includes(gResult));
        WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::Disconnect(nn::wlan::LocalCommunicationMode_ClientSpectator, nullptr));
        nn::wlan::Local::DeleteRxEntryForActionFrame(rxId);
        localClient.LocalRelease();
    }

    TEST_F(LocalClientBoundaryUnitTest, GetActionFrameTest_No193)
    {
        SetUpConnect(nn::wlan::Ssid(LocalMasterTestSSID), security, 10, false);
        gResult = nn::wlan::Local::GetActionFrame(&macAddr, rxBufferAf, 1024, &OutSize, RxIdmax);
        WLANTEST_EXPECT_TRUE(true == nn::wlan::ResultRxEntryNotFound::Includes(gResult));
        WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::Disconnect(nn::wlan::LocalCommunicationMode_ClientSpectator, nullptr));
        nn::wlan::Local::DeleteRxEntryForActionFrame(rxId);
        localClient.LocalRelease();
    }

    /*************************/
    // CancelGetActionFrame Test
    /*************************/
    TEST_F(LocalClientBoundaryUnitTest, CancelGetActionFrameTest_No194)
    {
        gResult =  nn::wlan::Local::CancelGetActionFrame(RxIdmin);
        WLANTEST_EXPECT_TRUE(true == nn::wlan::ResultRxEntryNotFound::Includes(gResult));
    }

    TEST_F(LocalClientBoundaryUnitTest, CancelGetActionFrameTest_No195)
    {
        gResult =  nn::wlan::Local::CancelGetActionFrame(RxIdmax);
        WLANTEST_EXPECT_TRUE(true == nn::wlan::ResultRxEntryNotFound::Includes(gResult));
    }

    /*************************/
    // CreateRxEntryForActionFrame Test
    /*************************/
    TEST_F(LocalClientBoundaryUnitTest, CreateRxEntryForActionFrameTest_No197)
    {
        WLANTEST_EXPECT_RESULT_SUCCESS(nn::wlan::Local::CreateRxEntryForActionFrame(&rxId, afTypes, Createrxcountmin, Createrxcapacitynormal));
        nn::wlan::Local::DeleteRxEntryForActionFrame(rxId);
    }

    TEST_F(LocalClientBoundaryUnitTest, CreateRxEntryForActionFrameTest_No198)
    {
        gResult = nn::wlan::Local::CreateRxEntryForActionFrame(&rxId, afTypes, Createrxcountmax + 1, Createrxcapacitynormal);
        WLANTEST_EXPECT_TRUE(true == nn::wlan::ResultInvalidArgument::Includes(gResult));
        nn::wlan::Local::DeleteRxEntryForActionFrame(rxId);
    }

    TEST_F(LocalClientBoundaryUnitTest, CreateRxEntryForActionFrameTest_No199)
    {
        WLANTEST_EXPECT_RESULT_SUCCESS(nn::wlan::Local::CreateRxEntryForActionFrame(&rxId, afTypes, Createrxcountmax, Createrxcapacitynormal));
        nn::wlan::Local::DeleteRxEntryForActionFrame(rxId);
    }

    TEST_F(LocalClientBoundaryUnitTest, CreateRxEntryForActionFrameTest_No200)
    {
        WLANTEST_EXPECT_RESULT_SUCCESS(nn::wlan::Local::CreateRxEntryForActionFrame(&rxId, afTypes, Createrxcountnormal, Createrxcapacitymin));
        nn::wlan::Local::DeleteRxEntryForActionFrame(rxId);
    }

    TEST_F(LocalClientBoundaryUnitTest, CreateRxEntryForActionFrameTest_No201)
    {
        // 最大値が暫定
        gResult = nn::wlan::Local::CreateRxEntryForActionFrame(&rxId, afTypes, Createrxcountnormal, Createrxcapacitymax + 1);
        WLANTEST_EXPECT_TRUE(true == nn::wlan::ResultInvalidArgument::Includes(gResult));
        nn::wlan::Local::DeleteRxEntryForActionFrame(rxId);
    }

    TEST_F(LocalClientBoundaryUnitTest, CreateRxEntryForActionFrameTest_No202)
    {
        WLANTEST_EXPECT_RESULT_SUCCESS(nn::wlan::Local::CreateRxEntryForActionFrame(&rxId, afTypes, Createrxcountnormal, Createrxcapacitymax));
        nn::wlan::Local::DeleteRxEntryForActionFrame(rxId);
    }

    /*************************/
    // DeleteRxEntryForActionFrame Test
    /*************************/
    TEST_F(LocalClientBoundaryUnitTest, DeleteRxEntryForActionFrameTest_No203)
    {
        WLANTEST_EXPECT_RESULT_SUCCESS(nn::wlan::Local::CreateRxEntryForActionFrame(&rxId, afTypes, Createrxcountnormal, Createrxcapacitymax));
        gResult =  nn::wlan::Local::DeleteRxEntryForActionFrame(RxIdmin);
        WLANTEST_EXPECT_TRUE(true == nn::wlan::ResultRxEntryNotFound::Includes(gResult));
        nn::wlan::Local::DeleteRxEntryForActionFrame(rxId);
    }

    TEST_F(LocalClientBoundaryUnitTest, DeleteRxEntryForActionFrameTest_No204)
    {
        WLANTEST_EXPECT_RESULT_SUCCESS(nn::wlan::Local::CreateRxEntryForActionFrame(&rxId, afTypes, Createrxcountnormal, Createrxcapacitymax));
        gResult =  nn::wlan::Local::DeleteRxEntryForActionFrame(RxIdmax);
        WLANTEST_EXPECT_TRUE(true == nn::wlan::ResultRxEntryNotFound::Includes(gResult));
        nn::wlan::Local::DeleteRxEntryForActionFrame(rxId);
    }

    /*************************/
    // AddSubtypeToRxEntryForActionFrame Test
    /*************************/
    TEST_F(LocalClientBoundaryUnitTest, AddSubtypeToRxEntryForActionFrameTest_No205)
    {
        gResult =  nn::wlan::Local::AddSubtypeToRxEntryForActionFrame(RxIdmin, nn::wlan::ActionFrameType_Beacon);
        WLANTEST_EXPECT_TRUE(true == nn::wlan::ResultRxEntryNotFound::Includes(gResult));
    }

    TEST_F(LocalClientBoundaryUnitTest, AddSubtypeToRxEntryForActionFrameTest_No206)
    {
        gResult =  nn::wlan::Local::AddSubtypeToRxEntryForActionFrame(RxIdmax, nn::wlan::ActionFrameType_Beacon);
        WLANTEST_EXPECT_TRUE(true == nn::wlan::ResultRxEntryNotFound::Includes(gResult));
    }

    TEST_F(LocalClientBoundaryUnitTest, AddSubtypeToRxEntryForActionFrameTest_No207)
    {
        uint16_t testAfTypes[1] = {
            static_cast<uint16_t>(nn::wlan::ActionFrameType_Local)
        };
        uint32_t afCount = sizeof(testAfTypes[0]) / sizeof(testAfTypes);

        WLANTEST_EXPECT_RESULT_SUCCESS(nn::wlan::Local::CreateRxEntryForActionFrame(&rxId, testAfTypes, afCount, Createrxcapacitymax));
        gResult = nn::wlan::Local::AddSubtypeToRxEntryForActionFrame(rxId, static_cast<nn::wlan::ActionFrameType>(nn::wlan::ActionFrameType_Beacon - 1));
        WLANTEST_EXPECT_TRUE(true == nn::wlan::ResultInvalidArgument::Includes(gResult));
        WLANTEST_EXPECT_RESULT_SUCCESS(nn::wlan::Local::DeleteRxEntryForActionFrame(rxId));
    }

    TEST_F(LocalClientBoundaryUnitTest, AddSubtypeToRxEntryForActionFrameTest_No208)
    {
        uint16_t testAfTypes[1] = {
            static_cast<uint16_t>(nn::wlan::ActionFrameType_Local)
        };
        uint32_t afCount = sizeof(testAfTypes[0]) / sizeof(testAfTypes);

        WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::CreateRxEntryForActionFrame(&rxId, testAfTypes, afCount, Createrxcapacitynormal));
        WLANTEST_EXPECT_RESULT_SUCCESS(nn::wlan::Local::AddSubtypeToRxEntryForActionFrame(rxId, static_cast<nn::wlan::ActionFrameType>(nn::wlan::ActionFrameType_Beacon)));
        WLANTEST_EXPECT_RESULT_SUCCESS(nn::wlan::Local::DeleteRxEntryForActionFrame(rxId));
    }

    TEST_F(LocalClientBoundaryUnitTest, AddSubtypeToRxEntryForActionFrameTest_No209)
    {
        uint16_t testAfTypes[1] = {
            static_cast<uint16_t>(nn::wlan::ActionFrameType_Local)
        };
        uint32_t afCount = sizeof(testAfTypes[0]) / sizeof(testAfTypes);

        WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::CreateRxEntryForActionFrame(&rxId, testAfTypes, afCount, Createrxcapacitynormal));
        gResult = nn::wlan::Local::AddSubtypeToRxEntryForActionFrame(rxId, static_cast<nn::wlan::ActionFrameType>(nn::wlan::ActionFrameType_End + 1));
        WLANTEST_EXPECT_TRUE(true == nn::wlan::ResultInvalidArgument::Includes(gResult));
        WLANTEST_EXPECT_RESULT_SUCCESS(nn::wlan::Local::DeleteRxEntryForActionFrame(rxId));
    }

    TEST_F(LocalClientBoundaryUnitTest, AddSubtypeToRxEntryForActionFrameTest_No210)
    {
        uint16_t testAfTypes[1] = {
            static_cast<uint16_t>(nn::wlan::ActionFrameType_Local)
        };
        uint32_t afCount = sizeof(testAfTypes[0]) / sizeof(testAfTypes);

        WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::CreateRxEntryForActionFrame(&rxId, testAfTypes, afCount, Createrxcapacitynormal));
        gResult = nn::wlan::Local::AddSubtypeToRxEntryForActionFrame(rxId, static_cast<nn::wlan::ActionFrameType>(nn::wlan::ActionFrameType_End));
        WLANTEST_EXPECT_TRUE(true == nn::wlan::ResultInvalidArgument::Includes(gResult));
        WLANTEST_EXPECT_RESULT_SUCCESS(nn::wlan::Local::DeleteRxEntryForActionFrame(rxId));
    }

    /*************************/
    // DeleteSubtypeFromRxEntryForActionFrame Test
    /*************************/
    TEST_F(LocalClientBoundaryUnitTest, DeleteSubtypeFromRxEntryForActionFrameTest_No211)
    {
        WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::CreateRxEntryForActionFrame(&rxId, afTypes, Createrxcountnormal, Createrxcapacitynormal));
        gResult = nn::wlan::Local::DeleteSubtypeFromRxEntryForActionFrame(&rxId, static_cast<nn::wlan::ActionFrameType>(nn::wlan::ActionFrameType_Beacon - 1));
        WLANTEST_EXPECT_TRUE(true == nn::wlan::ResultInvalidArgument::Includes(gResult));
        WLANTEST_EXPECT_RESULT_SUCCESS(nn::wlan::Local::DeleteRxEntryForActionFrame(rxId));
    }

    TEST_F(LocalClientBoundaryUnitTest, DeleteSubtypeFromRxEntryForActionFrameTest_No212)
    {
        WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::CreateRxEntryForActionFrame(&rxId, afTypes, Createrxcountnormal, Createrxcapacitynormal));
        nn::wlan::Local::CreateRxEntryForActionFrame(&rxId, afTypes, Createrxcountnormal, Createrxcapacitymax);
        WLANTEST_EXPECT_RESULT_SUCCESS(nn::wlan::Local::DeleteSubtypeFromRxEntryForActionFrame(&rxId, static_cast<nn::wlan::ActionFrameType>(nn::wlan::ActionFrameType_Beacon)));
        WLANTEST_EXPECT_RESULT_SUCCESS(nn::wlan::Local::DeleteRxEntryForActionFrame(rxId));
    }

    TEST_F(LocalClientBoundaryUnitTest, DeleteSubtypeFromRxEntryForActionFrameTest_No213)
    {
        WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::CreateRxEntryForActionFrame(&rxId, afTypes, Createrxcountnormal, Createrxcapacitynormal));
        gResult = nn::wlan::Local::DeleteSubtypeFromRxEntryForActionFrame(&rxId, static_cast<nn::wlan::ActionFrameType>(nn::wlan::ActionFrameType_End + 1));
        WLANTEST_EXPECT_TRUE(true == nn::wlan::ResultInvalidArgument::Includes(gResult));
        WLANTEST_EXPECT_RESULT_SUCCESS(nn::wlan::Local::DeleteRxEntryForActionFrame(rxId));
    }

    TEST_F(LocalClientBoundaryUnitTest, DeleteSubtypeFromRxEntryForActionFrameTest_No214)
    {
        WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::CreateRxEntryForActionFrame(&rxId, afTypes, Createrxcountnormal, Createrxcapacitynormal));
        WLANTEST_EXPECT_RESULT_NOTSUCCESS(nn::wlan::Local::DeleteSubtypeFromRxEntryForActionFrame(&rxId, nn::wlan::ActionFrameType_End));
        WLANTEST_EXPECT_RESULT_SUCCESS(nn::wlan::Local::DeleteRxEntryForActionFrame(rxId));
    }

    extern "C" void nnMain()
    {
        int     argc = nn::os::GetHostArgc();
        char**  argv = nn::os::GetHostArgv();
        NN_LOG("argc=%d\n", argc);
        for (int i = 0; i < argc; i++)
        {
            NN_LOG("argv[%d]=%s\n", argc, argv[i]);
        }

        const bool isEnabled = true;

        WLANTEST_ASSERT_RESULT_SUCCESS(nn::nifm::Initialize());
        nn::settings::fwdbg::SetSettingsItemValue("nifm", "is_communication_control_enabled_for_test", &isEnabled, sizeof(isEnabled));
        nn::nifm::SetWirelessCommunicationEnabledForTest(false);
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(1000));

        ::testing::InitGoogleTest(&argc, argv);

        int result = RUN_ALL_TESTS();
        nnt::Exit(result);
    }

} // namespace WlanTest
