﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#define WLANTEST_GTESTLIB  // gtest有効/無効切り替え

#include <memory>
#include "../../Common/testWlan_UnitTest.h"
#include "../../Common/testWlan_UnitTestCommon.h"
#include <nn/settings/fwdbg/settings_SettingsSetterApi.h>

namespace {

// 仕様確定後に有効
//#define BOUNDARY_NOT_TEST

// Bufer
const int Buffersizemin = sizeof(nn::wlan::ScanResultHeader) + 1;

// TestName
#define STARTSCAN_TESTNAME       "StartScan"
#define CONNECT_TESTNAME         "Connect"
#define CONVERTRSSI_TESTNAME     "ConvertRssiToLinkLevel"
#define CHANGRERX_TESTNAME       "ChangeRxAntenna"
#define CONNECT_WW_TESTNAME      "ConnectWithWps"
int32_t BoundaryTestNo = 1;

} // namespace

namespace WlanTest {
    nn::Result Result;
    nn::wlan::Ssid connectSsid(WtestSsid[0]);

// StartScan試験
TEST(InfraBoundaryUnitTest, StartScanTestCase) NN_NOEXCEPT
{
    size_t     size = BufferSize100k;
    std::unique_ptr<char[]>pTestMem(new char[size]);
    WLANTEST_EXPECT_TRUE(pTestMem.get() != nullptr);

    nn::wlan::Ssid ssid;
    nn::Bit8 ssdid_test[Wlantest_ssidlengthmax];
    memset(ssdid_test, 0, sizeof(ssdid_test));
    strcmp((char *)ssdid_test, "a");
    ssid.Set( ssdid_test , Ssidsizemin - 1 );
    nn::wlan::ScanParameters scanParam = {
            nn::wlan::ScanType_Active,
            {1, 6, 11},
            0,
            -1,
            -1,
            &ssid,
            0,
            nn::wlan::MacAddress::CreateBroadcastMacAddress()
    };

    // ***************************************************/
    //         第二引数 境界値試験
    // ***************************************************/
    // 最小値-1 境界値テスト：1
    WLANTEST_SET_TESTNAME( STARTSCAN_TESTNAME, &BoundaryTestNo );
    std::unique_ptr<char[]>pTestMem_min_1(new char[Buffersizemin - 1]);
    WLANTEST_EXPECT_TRUE(pTestMem_min_1.get() != nullptr);
    Result = nn::wlan::Infra::StartScan(pTestMem_min_1.get(), Buffersizemin - 1, scanParam);
    WLANTEST_EXPECT_TRUE(true == nn::wlan::ResultInvalidArgument::Includes(Result));

    // 最小値 境界値テスト：2
    WLANTEST_SET_TESTNAME( STARTSCAN_TESTNAME, &BoundaryTestNo );
    std::unique_ptr<char[]>pTestMem_min(new char[Buffersizemin]);
    WLANTEST_EXPECT_TRUE(pTestMem_min.get() != nullptr);
    WLANTEST_ASSERT_RESULT_SUCCESS( nn::wlan::Infra::StartScan(pTestMem_min.get(), Buffersizemin, scanParam) );

    // 最大値 境界値テスト：3
    // 試験対象外となったため削除
    BoundaryTestNo ++;

    // 最大値+1 境界値テスト：4
    // 試験対象外となったため削除
    BoundaryTestNo ++;

    // ***************************************************/
    //         第三引数 scanType 境界値試験
    // ***************************************************/
    // 最小値-1 境界値テスト：5
    BoundaryTestNo = 5;
    WLANTEST_SET_TESTNAME( STARTSCAN_TESTNAME, &BoundaryTestNo );
    scanParam.scanType = static_cast<nn::wlan::ScanType>(-1);
    Result = nn::wlan::Infra::StartScan(pTestMem.get(), size, scanParam);
    WLANTEST_EXPECT_TRUE(true == nn::wlan::ResultInvalidArgument::Includes(Result));

    // 最小値 境界値テスト：6
    WLANTEST_SET_TESTNAME( STARTSCAN_TESTNAME, &BoundaryTestNo );
    scanParam.scanType = nn::wlan::ScanType_Active;
    WLANTEST_ASSERT_RESULT_SUCCESS( nn::wlan::Infra::StartScan(pTestMem.get(), size, scanParam) );

    // 最大値 境界値テスト：7
    WLANTEST_SET_TESTNAME( STARTSCAN_TESTNAME, &BoundaryTestNo );
    scanParam.scanType = nn::wlan::ScanType_Passive;
    WLANTEST_ASSERT_RESULT_SUCCESS( nn::wlan::Infra::StartScan(pTestMem.get(), size, scanParam) );


    // 最大値+1 境界値テスト：8
    WLANTEST_SET_TESTNAME( STARTSCAN_TESTNAME, &BoundaryTestNo );
    scanParam.scanType = static_cast<nn::wlan::ScanType>(2);
    Result = nn::wlan::Infra::StartScan(pTestMem.get(), size, scanParam);
    WLANTEST_EXPECT_TRUE(true == nn::wlan::ResultInvalidArgument::Includes(Result));

    // scanType を戻す
    scanParam.scanType = nn::wlan::ScanType_Active;

    // ***************************************************/
    //         第三引数 channelCount 境界値試験
    // ***************************************************/
    // 最小値-1 境界値テスト：9
    BoundaryTestNo = 9;
    // 最小値は仕様上 0であり変数もuintとなるため、最小値-1の試験は不可
    BoundaryTestNo ++;

    // 最小値 境界値テスト：10
    WLANTEST_SET_TESTNAME( STARTSCAN_TESTNAME, &BoundaryTestNo );
    scanParam.channelCount = Channelcountsizemin;
    WLANTEST_ASSERT_RESULT_SUCCESS( nn::wlan::Infra::StartScan(pTestMem.get(), size, scanParam) );

    // 最大値 境界値テスト：11
    WLANTEST_SET_TESTNAME( STARTSCAN_TESTNAME, &BoundaryTestNo );
    scanParam.channelCount = Channelcountsizemax;
    for (int i = 3; i < Channelcountsizemax; i ++)
    {
        scanParam.channelList[i] = 1;
    }
    WLANTEST_ASSERT_RESULT_SUCCESS( nn::wlan::Infra::StartScan(pTestMem.get(), size, scanParam) );

    // 最大値+1 境界値テスト：12
    WLANTEST_SET_TESTNAME( STARTSCAN_TESTNAME, &BoundaryTestNo );//TestNo12
    scanParam.channelCount = Channelcountsizemax + 1;
    Result = nn::wlan::Infra::StartScan(pTestMem.get(), size, scanParam);
    WLANTEST_EXPECT_TRUE(true == nn::wlan::ResultInvalidArgument::Includes(Result));

    // channelCount を戻す
    scanParam.channelCount = 0;

    // ***************************************************/
    //         第三引数 channelScanTime 境界値試験
    // ***************************************************/
    // デフォルト値 -1 境界値テスト：13
    BoundaryTestNo = 13;
    WLANTEST_SET_TESTNAME( STARTSCAN_TESTNAME, &BoundaryTestNo );
    scanParam.channelScanTime = Channelscantimedefault - 2; // -2が最小値-1となる。-1はデフォルト値が指定される
    Result = nn::wlan::Infra::StartScan(pTestMem.get(), size, scanParam);
    WLANTEST_EXPECT_TRUE(true == nn::wlan::ResultInvalidArgument::Includes(Result));

    // デフォルト値 境界値テスト：14
    WLANTEST_SET_TESTNAME( STARTSCAN_TESTNAME, &BoundaryTestNo );
    scanParam.channelScanTime = Channelscantimedefault;
    WLANTEST_ASSERT_RESULT_SUCCESS( nn::wlan::Infra::StartScan(pTestMem.get(), size, scanParam) );

    // 最小値-1 境界値テスト：15
    WLANTEST_SET_TESTNAME( STARTSCAN_TESTNAME, &BoundaryTestNo );
    scanParam.channelScanTime = Channelscantimemin - 1;
    Result = nn::wlan::Infra::StartScan(pTestMem.get(), size, scanParam);
    WLANTEST_EXPECT_TRUE(true == nn::wlan::ResultInvalidArgument::Includes(Result));

    // 最小値 境界値テスト：16
    WLANTEST_SET_TESTNAME( STARTSCAN_TESTNAME, &BoundaryTestNo );
    scanParam.channelScanTime = Channelscantimemin;
    WLANTEST_ASSERT_RESULT_SUCCESS( nn::wlan::Infra::StartScan(pTestMem.get(), size, scanParam) );

    // 最大値 境界値テスト：17
    WLANTEST_SET_TESTNAME( STARTSCAN_TESTNAME, &BoundaryTestNo );
    scanParam.channelScanTime = Channelscantimemax;
    WLANTEST_ASSERT_RESULT_SUCCESS( nn::wlan::Infra::StartScan(pTestMem.get(), size, scanParam) );

    // 最大値+1 境界値テスト：18
    WLANTEST_SET_TESTNAME( STARTSCAN_TESTNAME, &BoundaryTestNo );//TestNo18
    scanParam.channelScanTime = Channelscantimemax + 1;
    Result = nn::wlan::Infra::StartScan(pTestMem.get(), size, scanParam);
    WLANTEST_EXPECT_TRUE(true == nn::wlan::ResultInvalidArgument::Includes(Result));

    // channelScanTime をデフォルト設定に戻す
    scanParam.channelScanTime =  - 1;

    // ***************************************************/
    //         第三引数 homeChannelTime 境界値試験
    // ***************************************************/
    // 最小値-1 境界値テスト：19
    BoundaryTestNo = 19;
    WLANTEST_SET_TESTNAME( STARTSCAN_TESTNAME, &BoundaryTestNo );
    scanParam.homeChannelTime = Homechanneltimedefault - 1; // -2が最小値-1となる。-1はデフォルト値が指定される
    Result = nn::wlan::Infra::StartScan(pTestMem.get(), size, scanParam);
    WLANTEST_EXPECT_TRUE(true == nn::wlan::ResultInvalidArgument::Includes(Result));

    // 最小値 境界値テスト：20
    WLANTEST_SET_TESTNAME( STARTSCAN_TESTNAME, &BoundaryTestNo );
    scanParam.homeChannelTime = Homechanneltimemin;
    WLANTEST_ASSERT_RESULT_SUCCESS( nn::wlan::Infra::StartScan(pTestMem.get(), size, scanParam) );

    // 最大値 境界値テスト：21
    WLANTEST_SET_TESTNAME( STARTSCAN_TESTNAME, &BoundaryTestNo );
    scanParam.homeChannelTime = Homechanneltimemax;
    WLANTEST_ASSERT_RESULT_SUCCESS( nn::wlan::Infra::StartScan(pTestMem.get(), size, scanParam) );

    WLANTEST_SET_TESTNAME( STARTSCAN_TESTNAME, &BoundaryTestNo );//TestNo22
    scanParam.homeChannelTime = Homechanneltimemax + 1;
    Result = nn::wlan::Infra::StartScan(pTestMem.get(), size, scanParam);
    WLANTEST_EXPECT_TRUE(true == nn::wlan::ResultInvalidArgument::Includes(Result));

    // homeChannelTime をデフォルト設定に戻す
    scanParam.homeChannelTime =  - 1;

    // ***************************************************/
    //         第三引数 Ssid 境界値試験
    // ***************************************************/
    // 最小値-1 境界値テスト：23
    BoundaryTestNo = 23;
    WLANTEST_SET_TESTNAME( STARTSCAN_TESTNAME, &BoundaryTestNo );
    memset( ssdid_test, 0, sizeof(ssdid_test));
    ssid.Set( ssdid_test , Ssidsizemin - 1 );
    scanParam.ssidList = &ssid;
    scanParam.ssidCount = 1;
    nn::Result Result = nn::wlan::Infra::StartScan(pTestMem.get(), size, scanParam);
    WLANTEST_EXPECT_TRUE( true == nn::wlan::ResultInvalidArgument::Includes(Result) );

    // 最小値 境界値テスト：24
    WLANTEST_SET_TESTNAME( STARTSCAN_TESTNAME, &BoundaryTestNo );
    memset( ssdid_test, 0, sizeof(ssdid_test));
    ssdid_test[0] = 0x31;
    ssdid_test[1] = NULL;
    ssid.Set( ssdid_test , Ssidsizemin );
    scanParam.ssidList = &ssid;
    WLANTEST_ASSERT_RESULT_SUCCESS( nn::wlan::Infra::StartScan(pTestMem.get(), size, scanParam) );

    // 最大値 境界値テスト：25
    WLANTEST_SET_TESTNAME( STARTSCAN_TESTNAME, &BoundaryTestNo );
    memset( ssdid_test, 1, Ssidsizemax );
    ssid.Set( ssdid_test, Ssidsizemax );
    scanParam.ssidList = &ssid;
    WLANTEST_ASSERT_RESULT_SUCCESS( nn::wlan::Infra::StartScan(pTestMem.get(), size, scanParam) );

    // 最大値+1 境界値テスト：26
    WLANTEST_SET_TESTNAME( STARTSCAN_TESTNAME, &BoundaryTestNo );//TestNo26
    ssid.Set( ssdid_test , Ssidsizemax + 1 );
    scanParam.ssidList = &ssid;
    WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Infra::StartScan(pTestMem.get(), size, scanParam));

    // ***************************************************/
    //         第三引数 ssidCount 境界値試験
    // ***************************************************/
    nn::Bit8 ssdid_test_max[10][32] = {{1},{1}};
    nn::wlan::Ssid ssidmax[10];
    for( int i = 0; i < 10; i++ )
    {
        ssidmax[i].Set( &ssdid_test_max[i][0] , 1 );
    }
    scanParam.ssidList = &ssidmax[0];

    // 最小値-1 境界値テスト：27
    BoundaryTestNo = 27;
    BoundaryTestNo ++;

    // 最小値 境界値テスト：28
    WLANTEST_SET_TESTNAME( STARTSCAN_TESTNAME, &BoundaryTestNo );
    scanParam.ssidCount = Ssidcountmin;
    WLANTEST_ASSERT_RESULT_SUCCESS( nn::wlan::Infra::StartScan(pTestMem.get(), size, scanParam) );

    // 最大値 境界値テスト：29
    WLANTEST_SET_TESTNAME( STARTSCAN_TESTNAME, &BoundaryTestNo );
    scanParam.ssidCount = Ssidcountmax;
    WLANTEST_ASSERT_RESULT_SUCCESS( nn::wlan::Infra::StartScan(pTestMem.get(), size, scanParam) );

    // 最大値+1 境界値テスト：30
    WLANTEST_SET_TESTNAME( STARTSCAN_TESTNAME, &BoundaryTestNo );
    scanParam.ssidCount = Ssidcountmax + 1;
    WLANTEST_ASSERT_RESULT_SUCCESS( nn::wlan::Infra::StartScan(pTestMem.get(), size, scanParam) );
} // NOLINT(impl/function_size)


// Connect試験
TEST(InfraBoundaryUnitTest, ConnectTestCase) NN_NOEXCEPT
{
    // Macアドレス取得
    nn::wlan::MacAddress macAddr;
    WLANTEST_EXPECT_RESULT_SUCCESS( nn::wlan::Infra::GetMacAddress(&macAddr) );

    // ConnectionEventをWLANプロセスのものにアタッチ
    nn::os::SystemEventType connectionEvent;
    WLANTEST_EXPECT_RESULT_SUCCESS( nn::wlan::Infra::GetConnectionEvent(&connectionEvent) );

    nn::wlan::ScanParameters scanParam = {
            nn::wlan::ScanType_Passive,
            {1, 6, 11},
            0,  // 全チャンネルスキャン
            120,
            0,
            NULL,
            0,
            nn::wlan::MacAddress::CreateBroadcastMacAddress()
    };

    size_t     size = BufferSize100k;
    std::unique_ptr<char[]>pTestMem(new char[size]);
    WLANTEST_EXPECT_TRUE(pTestMem.get() != nullptr);

    WLANTEST_EXPECT_RESULT_SUCCESS( nn::wlan::Infra::StartScan(pTestMem.get(), size, scanParam) );

    nn::wlan::BeaconScanResultReader resultReader(pTestMem.get());
    uint32_t bssCount = resultReader.GetCount();

    nn::wlan::Ssid Ssid(WtestSsid[0]);
    nn::wlan::MacAddress bssid;                   // 接続先のBSSID格納先 (SSIDが分かっていれば不要)
    int16_t channel;                              // 接続先のChannel格納先
    int16_t channel_bak;                          // 接続先のChannel格納先 バックアップ用

    for(uint32_t i = 0; i < bssCount; i++)
    {
        nn::wlan::BeaconDescriptionReader beacon = resultReader.GetNextDescription();
        if( Ssid == beacon.GetSsid() )
        {
            NN_LOG("WlanInfra: Found access point! ch:%d\n", beacon.GetChannel());
            bssid = beacon.GetBssid();
            channel = beacon.GetChannel();
            channel_bak = channel;
        }
    }

    // 相手先(SecurityMode_Open)のセキュリティ情報入力
    nn::wlan::Security security = {
            nn::wlan::SecurityMode_Open,
            nn::wlan::SecurityMode_Open,
            0,
            "",  // Key
    };

    // 相手先(SecurityMode_Wep64Shared)のセキュリティ情報入力
    nn::wlan::Security securitySharedWEP64 = {
        nn::wlan::SecurityMode::SecurityMode_Wep64Shared,
        nn::wlan::SecurityMode::SecurityMode_Wep64Shared,
        0,
        "1010101010",  // Key
    };

    // 相手先(SecurityMode_Wep128Shared)のセキュリティ情報入力
    nn::wlan::Security securitySharedWEP128 = {
        nn::wlan::SecurityMode::SecurityMode_Wep128Shared,
        nn::wlan::SecurityMode::SecurityMode_Wep128Shared,
        0,
        "31323334353637383930313234",  // Key
    };

    // 相手先(SecurityMode_Wpa2Aes)のセキュリティ情報入力
    nn::wlan::Security securityWpa2Aes = {
        nn::wlan::SecurityMode::SecurityMode_Wpa2Aes,
        nn::wlan::SecurityMode::SecurityMode_Wpa2Aes,
        0,
        "AbcdefghijklmnopqrstuvwxyZA",  // Key
    };

    nn::Bit8 ssdid_test[32] = {0};


    // ***************************************************/
    //         第一引数 Ssid 境界値試験
    // ***************************************************/
    BoundaryTestNo = 31;
    // 最小値-1 境界値テスト：31
    WLANTEST_SET_TESTNAME( CONNECT_TESTNAME, &BoundaryTestNo );
    Ssid.Set( ssdid_test , Ssidsizemin - 1 );

    Result = nn::wlan::Infra::Connect(Ssid, bssid, channel, security, true);
    WLANTEST_EXPECT_TRUE(true == nn::wlan::ResultInvalidArgument::Includes(Result));
    WLANTEST_EXPECT_RESULT_SUCCESS( nn::wlan::Infra::Disconnect() );

    // 最小値 境界値テスト：32
    WLANTEST_SET_TESTNAME( CONNECT_TESTNAME, &BoundaryTestNo );
    memcpy(ssdid_test, "1", sizeof("1"));
    Ssid.Set( ssdid_test , Ssidsizemin );

    WLANTEST_EXPECT_RESULT_SUCCESS(nn::wlan::Infra::Connect(Ssid, nn::wlan::MacAddress::CreateBroadcastMacAddress(),
        -1, { nn::wlan::SecurityMode_Open, nn::wlan::SecurityMode_Open, 0, "" }, true) );
    WLANTEST_EXPECT_RESULT_SUCCESS( nn::wlan::Infra::Disconnect() );

    // 最大値 境界値テスト：33
    WLANTEST_SET_TESTNAME( CONNECT_TESTNAME, &BoundaryTestNo );
    memset( ssdid_test, 1, Ssidsizemax );
    channel = 13;
    memcpy(ssdid_test, "1234567890123456789012345678901", Ssidsizemax);
    bssid = nn::wlan::MacAddress::CreateBroadcastMacAddress();
    Ssid.Set( ssdid_test, Ssidsizemax );

    WLANTEST_EXPECT_RESULT_SUCCESS( nn::wlan::Infra::Connect(Ssid, bssid, channel, security, true) );
    WLANTEST_EXPECT_RESULT_SUCCESS( nn::wlan::Infra::Disconnect() );

    // 正常系の試験で実施する。
    // 最大値+1 境界値テスト：34
    BoundaryTestNo++;

    // ***************************************************/
    //         第三引数 channel 境界値試験
    // ***************************************************/
    BoundaryTestNo = 35;
    // 最小値-1 境界値テスト：35
    WLANTEST_SET_TESTNAME( CONNECT_TESTNAME, &BoundaryTestNo );
    channel = Channelmin - 2; // -1は全てのチャンネルスキャン
    Result = nn::wlan::Infra::Connect(Ssid, bssid, channel, security, true);
    WLANTEST_EXPECT_TRUE(true == nn::wlan::ResultInvalidArgument::Includes(Result));
    WLANTEST_EXPECT_RESULT_SUCCESS( nn::wlan::Infra::Disconnect() );


    // 最小値 境界値テスト：36
    BoundaryTestNo = 36;
    WLANTEST_SET_TESTNAME( CONNECT_TESTNAME, &BoundaryTestNo );
    channel = 13;
    bssid = nn::wlan::MacAddress::CreateBroadcastMacAddress();

    memcpy(ssdid_test, "1", sizeof("1"));
    Ssid.Set(ssdid_test, Ssidsizemin);

    WLANTEST_EXPECT_RESULT_SUCCESS( nn::wlan::Infra::Connect(Ssid, bssid, channel, security, true) );
    WLANTEST_EXPECT_RESULT_SUCCESS( nn::wlan::Infra::Disconnect() );

    // 最大値 境界値テスト：37 未実装
#ifdef BOUNDARY_NOT_TEST
    WLANTEST_SET_TESTNAME( CONNECT_TESTNAME, &BoundaryTestNo );
    int16_t channellist[nn::wlan::WirelessChannelsCountMax];
    memset( channellist, 0, sizeof(channellist) );
    nn::wlan::Infra::GetAllowedChannels(channellist);

    uint32_t i;
    for(i = nn::wlan::WirelessChannelsCountMax; i > 0; i--)
    {
        if( 0 < channellist[i] )
        {
            break;
        }
    }

    channel = channellist[i];

    WLANTEST_EXPECT_RESULT_SUCCESS( nn::wlan::Infra::Connect(Ssid, bssid, channel, security, false) );
    WLANTEST_EXPECT_RESULT_SUCCESS( nn::wlan::Infra::Disconnect() );
#else
    BoundaryTestNo ++;
#endif

    // 最大値+1 境界値テスト：38　未実装
#ifdef BOUNDARY_NOT_TEST
    WLANTEST_SET_TESTNAME( CONNECT_TESTNAME, &BoundaryTestNo );
    channel = channellist[i] + 1;

    Result = nn::wlan::Infra::Connect(Ssid, bssid, channel, security, false);
    WLANTEST_EXPECT_TRUE(true == nn::wlan::ResultInvalidArgument::Includes(Result));
    WLANTEST_EXPECT_RESULT_SUCCESS( nn::wlan::Infra::Disconnect() );
#else
    BoundaryTestNo ++;
#endif

    // もとに戻す
    channel = channel_bak;
    // ***************************************************/
    //         第四引数 privacyMode 境界値試験
    // ***************************************************/
    BoundaryTestNo = 39;
    // 最小値-1 境界値テスト：39
    WLANTEST_SET_TESTNAME( CONNECT_TESTNAME, &BoundaryTestNo );
    security.privacyMode = static_cast<nn::wlan::SecurityMode>(-1);

    Result = nn::wlan::Infra::Connect(Ssid, bssid, channel, security, true);
    WLANTEST_EXPECT_TRUE(true == nn::wlan::ResultInvalidArgument::Includes(Result));
    WLANTEST_EXPECT_RESULT_SUCCESS( nn::wlan::Infra::Disconnect() );

    // 最小値 境界値テスト：40
    WLANTEST_SET_TESTNAME( CONNECT_TESTNAME, &BoundaryTestNo );
    channel = 13;
    security.privacyMode = nn::wlan::SecurityMode_Open;
    bssid = nn::wlan::MacAddress::CreateBroadcastMacAddress();

    memcpy(ssdid_test, "1", sizeof("1"));
    Ssid.Set(ssdid_test, Ssidsizemin);

    WLANTEST_EXPECT_RESULT_SUCCESS( nn::wlan::Infra::Connect(Ssid, bssid, channel, security, true) );
    WLANTEST_EXPECT_RESULT_SUCCESS( nn::wlan::Infra::Disconnect() );

    // 最大値 境界値テスト(securityWpa2Aes)：41

    BoundaryTestNo = 41;
    WLANTEST_SET_TESTNAME( CONNECT_TESTNAME, &BoundaryTestNo );

    channel = 64;
    bssid = nn::wlan::MacAddress::CreateBroadcastMacAddress();

    memcpy(ssdid_test, WtestSsid[8], sizeof(WtestSsid[8]));
    Ssid.Set(ssdid_test, sizeof(ssdid_test));

    WLANTEST_EXPECT_RESULT_SUCCESS( nn::wlan::Infra::Connect(Ssid, bssid, channel, securityWpa2Aes, true) );
    WLANTEST_EXPECT_RESULT_SUCCESS( nn::wlan::Infra::Disconnect() );

    // 最大値+1(SecurityMode_StaticAes) 境界値テスト：42
    WLANTEST_SET_TESTNAME( CONNECT_TESTNAME, &BoundaryTestNo );
    security.privacyMode = static_cast<nn::wlan::SecurityMode>(10);

    Result = nn::wlan::Infra::Connect(Ssid, bssid, channel, security, true);
    WLANTEST_EXPECT_TRUE(true == nn::wlan::ResultInvalidArgument::Includes(Result));
    WLANTEST_EXPECT_RESULT_SUCCESS( nn::wlan::Infra::Disconnect() );

    // もとに戻す
    security.privacyMode = nn::wlan::SecurityMode_Open;
    security.groupPrivacyMode = nn::wlan::SecurityMode_Open;

    // ***************************************************/
    //         第四引数 groupPrivacyMode 境界値試験
    // ***************************************************/
    BoundaryTestNo = 43;
    // 最小値-1 境界値テスト：43
    WLANTEST_SET_TESTNAME( CONNECT_TESTNAME, &BoundaryTestNo );
    security.groupPrivacyMode = static_cast<nn::wlan::SecurityMode>(-1);

    channel = 13;
    bssid = nn::wlan::MacAddress::CreateBroadcastMacAddress();

    memcpy(ssdid_test, "1", sizeof("1"));
    Ssid.Set(ssdid_test, Ssidsizemin);

    Result = nn::wlan::Infra::Connect(Ssid, bssid, channel, security, true);
    WLANTEST_EXPECT_TRUE(true == nn::wlan::ResultInvalidArgument::Includes(Result));
    WLANTEST_EXPECT_RESULT_SUCCESS( nn::wlan::Infra::Disconnect() );

    // 最小値 境界値テスト：44
    WLANTEST_SET_TESTNAME( CONNECT_TESTNAME, &BoundaryTestNo );
    security.privacyMode = nn::wlan::SecurityMode_Open;
    security.groupPrivacyMode = nn::wlan::SecurityMode_Open;

    channel = 13;
    bssid = nn::wlan::MacAddress::CreateBroadcastMacAddress();

    memcpy(ssdid_test, "1", sizeof("1"));
    Ssid.Set(ssdid_test, Ssidsizemin);

    WLANTEST_EXPECT_RESULT_SUCCESS( nn::wlan::Infra::Connect(Ssid, bssid, channel, security, true) );
    WLANTEST_EXPECT_RESULT_SUCCESS( nn::wlan::Infra::Disconnect() );
    // 最大値 境界値テスト：45

    WLANTEST_SET_TESTNAME( CONNECT_TESTNAME, &BoundaryTestNo );

    channel = 64;
    bssid = nn::wlan::MacAddress::CreateBroadcastMacAddress();

    memcpy(ssdid_test, WtestSsid[8], sizeof(WtestSsid[8]));
    Ssid.Set(ssdid_test, sizeof(ssdid_test));

    security.privacyMode = nn::wlan::SecurityMode_Open;
    security.groupPrivacyMode = nn::wlan::SecurityMode_StaticAes;
    WLANTEST_EXPECT_RESULT_SUCCESS( nn::wlan::Infra::Connect(Ssid, bssid, channel, security, true) );
    WLANTEST_EXPECT_RESULT_SUCCESS( nn::wlan::Infra::Disconnect() );

    // 最大値+1 境界値テスト(SecurityMode_StaticAes)：46
    WLANTEST_SET_TESTNAME( CONNECT_TESTNAME, &BoundaryTestNo );
    security.groupPrivacyMode = static_cast<nn::wlan::SecurityMode>(10);

    WLANTEST_EXPECT_RESULT_NOTSUCCESS(nn::wlan::Infra::Connect(Ssid, bssid, channel, security, true));
    WLANTEST_EXPECT_RESULT_SUCCESS( nn::wlan::Infra::Disconnect() );

    // もとに戻す
    security.privacyMode = nn::wlan::SecurityMode_Open;
    security.groupPrivacyMode = nn::wlan::SecurityMode_Open;

    // ***************************************************/
    //         第四引数 keyIdx 境界値試験
    // ***************************************************/
    BoundaryTestNo = 47;
    // 最小値-1 境界値テスト：47
    // 最小値は仕様上 0であり変数もuintとなるため、最小値-1の試験は不可
    BoundaryTestNo ++;//TestNo47

    // 最小値 境界値テスト：48
    WLANTEST_SET_TESTNAME( CONNECT_TESTNAME, &BoundaryTestNo );
    securitySharedWEP64.keyIdx = Keyinxmin;

    channel = 6;
    bssid = nn::wlan::MacAddress::CreateBroadcastMacAddress();

    memcpy(ssdid_test, WtestSsid[2], sizeof(WtestSsid[2]));
    Ssid.Set(ssdid_test, sizeof(ssdid_test));

    WLANTEST_EXPECT_RESULT_SUCCESS(nn::wlan::Infra::Connect(Ssid, bssid, channel, securitySharedWEP64, true));
    WLANTEST_EXPECT_RESULT_SUCCESS( nn::wlan::Infra::Disconnect() );

    // 最大値 境界値テスト：49
    WLANTEST_SET_TESTNAME( CONNECT_TESTNAME, &BoundaryTestNo );
    //security.keyIdx = Keyinxmax;
    securitySharedWEP128.keyIdx = Keyinxmin;//変更予定
    channel = 6;
    bssid = nn::wlan::MacAddress::CreateBroadcastMacAddress();

    memcpy(ssdid_test, WtestSsid[4], sizeof(WtestSsid[4]));
    Ssid.Set(ssdid_test, sizeof(ssdid_test));

    WLANTEST_EXPECT_RESULT_SUCCESS( nn::wlan::Infra::Connect(Ssid, bssid, channel, securitySharedWEP128, true) );
    WLANTEST_EXPECT_RESULT_SUCCESS( nn::wlan::Infra::Disconnect() );

    // 最大値+1 境界値テスト：50
    WLANTEST_SET_TESTNAME( CONNECT_TESTNAME, &BoundaryTestNo );
    securitySharedWEP128.keyIdx = Keyinxmax + 1;

    Result = nn::wlan::Infra::Connect(Ssid, bssid, channel, securitySharedWEP128, true);
    WLANTEST_EXPECT_TRUE(true == nn::wlan::ResultInvalidArgument::Includes(Result));

    // もとに戻す
    securitySharedWEP128.keyIdx =  Keyinxmin;
    // ***************************************************/
    //         第六引数 beaconLostTimeout 境界値試験
    // ***************************************************/
    BoundaryTestNo = 51;
    // 最小値-1 境界値テスト：51
    WLANTEST_SET_TESTNAME( CONNECT_TESTNAME, &BoundaryTestNo );
    Result = nn::wlan::Infra::Connect(Ssid, bssid, channel, security, true, Beaconlosttimemin - 1);
    WLANTEST_EXPECT_TRUE(true == nn::wlan::ResultInvalidArgument::Includes(Result));
    WLANTEST_EXPECT_RESULT_SUCCESS( nn::wlan::Infra::Disconnect() );

    // 最小値 境界値テスト：52
    WLANTEST_SET_TESTNAME( CONNECT_TESTNAME, &BoundaryTestNo );
    channel = 13;
    bssid = nn::wlan::MacAddress::CreateBroadcastMacAddress();

    memcpy(ssdid_test, "1", sizeof("1"));
    Ssid.Set(ssdid_test, sizeof(ssdid_test));

    WLANTEST_EXPECT_RESULT_SUCCESS( nn::wlan::Infra::Connect(Ssid, bssid, channel, security, true, Beaconlosttimemin) );
    WLANTEST_EXPECT_RESULT_SUCCESS( nn::wlan::Infra::Disconnect() );

    // 最大値 境界値テスト：53
    WLANTEST_SET_TESTNAME( CONNECT_TESTNAME, &BoundaryTestNo );
    WLANTEST_EXPECT_RESULT_SUCCESS( nn::wlan::Infra::Connect(Ssid, bssid, channel, security, true, Beaconlosttimemax) );
    WLANTEST_EXPECT_RESULT_SUCCESS( nn::wlan::Infra::Disconnect() );

    // 最大値+1 境界値テスト：54
    WLANTEST_SET_TESTNAME( CONNECT_TESTNAME, &BoundaryTestNo );//TestNo54
    Result = nn::wlan::Infra::Connect(Ssid, bssid, channel, security, true, Beaconlosttimemax + 1);
    WLANTEST_EXPECT_TRUE(true == nn::wlan::ResultInvalidArgument::Includes(Result));
    WLANTEST_EXPECT_RESULT_SUCCESS( nn::wlan::Infra::Disconnect() );
} // NOLINT(impl/function_size)

// ConvertRssiToLinkLevel試験
TEST(InfraBoundaryUnitTest, ConvertRssiToLinkLevelTestCase) NN_NOEXCEPT
{
    // 接続させる
    nn::wlan::Ssid Ssid2_4G(WtestSsid[1]);
    nn::wlan::ConnectionStatus connectionStatus;
    nn::os::SystemEvent connectionEvent;
    nn::wlan::Infra::GetConnectionEvent(connectionEvent.GetBase());
    NN_LOG("             Connect()成功まで待つ \n");

    WLANTEST_EXPECT_TRUE(InfraConnectWait(Ssid2_4G, nn::wlan::MacAddress::CreateBroadcastMacAddress(), -1,
        { nn::wlan::SecurityMode_Wep64Open, nn::wlan::SecurityMode_Wep64Open, 0, "3031323334" },
        true, WiFiBeaconLostTimeout, WiFiConnectTimeout));

    // 最小値-1 境界値テスト：55
    // 最小値-1は存在しない

    // 最小値 境界値テスト：56
    BoundaryTestNo = 56;
    WLANTEST_SET_TESTNAME(CONVERTRSSI_TESTNAME, &BoundaryTestNo);
    WLANTEST_EXPECT_TRUE((nn::wlan::LinkLevel::LinkLevel_0 == nn::wlan::Infra::ConvertRssiToLinkLevel(Covertrssitolinklevelmin)));

    // 最小値 境界値テスト：57
    WLANTEST_SET_TESTNAME(CONVERTRSSI_TESTNAME, &BoundaryTestNo);
    WLANTEST_EXPECT_TRUE((nn::wlan::LinkLevel::LinkLevel_0 == nn::wlan::Infra::ConvertRssiToLinkLevel(RssiThreshold1_2_4G - 1)));

    // 最小値 境界値テスト：58
    WLANTEST_SET_TESTNAME(CONVERTRSSI_TESTNAME, &BoundaryTestNo);
    WLANTEST_EXPECT_TRUE((nn::wlan::LinkLevel::LinkLevel_1 == nn::wlan::Infra::ConvertRssiToLinkLevel(RssiThreshold1_2_4G)));

    // 最小値 境界値テスト：59
    WLANTEST_SET_TESTNAME(CONVERTRSSI_TESTNAME, &BoundaryTestNo);
    WLANTEST_EXPECT_TRUE((nn::wlan::LinkLevel::LinkLevel_1 == nn::wlan::Infra::ConvertRssiToLinkLevel(RssiThreshold2_2_4G - 1)));

    // 最小値 境界値テスト：60
    WLANTEST_SET_TESTNAME(CONVERTRSSI_TESTNAME, &BoundaryTestNo);
    WLANTEST_EXPECT_TRUE((nn::wlan::LinkLevel::LinkLevel_2 == nn::wlan::Infra::ConvertRssiToLinkLevel(RssiThreshold2_2_4G)));

    // 最小値 境界値テスト：61
    WLANTEST_SET_TESTNAME(CONVERTRSSI_TESTNAME, &BoundaryTestNo);
    WLANTEST_EXPECT_TRUE((nn::wlan::LinkLevel::LinkLevel_2 == nn::wlan::Infra::ConvertRssiToLinkLevel(RssiThreshold3_2_4G - 1)));

    // 最小値 境界値テスト：62
    WLANTEST_SET_TESTNAME( CONVERTRSSI_TESTNAME, &BoundaryTestNo );
    WLANTEST_EXPECT_TRUE( (nn::wlan::LinkLevel::LinkLevel_3 == nn::wlan::Infra::ConvertRssiToLinkLevel(RssiThreshold3_2_4G)) );

    WLANTEST_EXPECT_RESULT_SUCCESS(nn::wlan::Infra::Disconnect());

    // 接続させる
    nn::wlan::Ssid Ssid5G(WtestSsid[8]);
    nn::wlan::Infra::GetConnectionEvent(connectionEvent.GetBase());
    NN_LOG("             Connect()成功まで待つ \n");

    WLANTEST_EXPECT_TRUE(InfraConnectWait(Ssid5G, nn::wlan::MacAddress::CreateBroadcastMacAddress(), -1,
        { nn::wlan::SecurityMode_Wpa2Aes, nn::wlan::SecurityMode_Wpa2Aes, 0, "AbcdefghijklmnopqrstuvwxyZ" }, true));

    // 最小値 境界値テスト：63
    WLANTEST_SET_TESTNAME(CONVERTRSSI_TESTNAME, &BoundaryTestNo);
    WLANTEST_EXPECT_TRUE((nn::wlan::LinkLevel::LinkLevel_0 == nn::wlan::Infra::ConvertRssiToLinkLevel(RssiThreshold1_5G - 1)));

    // 最小値 境界値テスト：64
    WLANTEST_SET_TESTNAME(CONVERTRSSI_TESTNAME, &BoundaryTestNo);
    WLANTEST_EXPECT_TRUE((nn::wlan::LinkLevel::LinkLevel_1 == nn::wlan::Infra::ConvertRssiToLinkLevel(RssiThreshold1_5G)));

    // 最小値 境界値テスト：65
    WLANTEST_SET_TESTNAME(CONVERTRSSI_TESTNAME, &BoundaryTestNo);
    WLANTEST_EXPECT_TRUE((nn::wlan::LinkLevel::LinkLevel_1 == nn::wlan::Infra::ConvertRssiToLinkLevel(RssiThreshold2_5G - 1)));

    // 最小値 境界値テスト：66
    WLANTEST_SET_TESTNAME(CONVERTRSSI_TESTNAME, &BoundaryTestNo);
    WLANTEST_EXPECT_TRUE((nn::wlan::LinkLevel::LinkLevel_2 == nn::wlan::Infra::ConvertRssiToLinkLevel(RssiThreshold2_5G)));

    // 最小値 境界値テスト：67
    WLANTEST_SET_TESTNAME(CONVERTRSSI_TESTNAME, &BoundaryTestNo);
    WLANTEST_EXPECT_TRUE((nn::wlan::LinkLevel::LinkLevel_2 == nn::wlan::Infra::ConvertRssiToLinkLevel(RssiThreshold3_5G - 1)));

    // 最小値 境界値テスト：68
    WLANTEST_SET_TESTNAME(CONVERTRSSI_TESTNAME, &BoundaryTestNo);
    WLANTEST_EXPECT_TRUE((nn::wlan::LinkLevel::LinkLevel_3 == nn::wlan::Infra::ConvertRssiToLinkLevel(RssiThreshold3_5G)));

    // 最大値 境界値テスト：69
    WLANTEST_SET_TESTNAME(CONVERTRSSI_TESTNAME, &BoundaryTestNo);
    WLANTEST_EXPECT_TRUE((nn::wlan::LinkLevel::LinkLevel_3 == nn::wlan::Infra::ConvertRssiToLinkLevel(Covertrssitolinklevelmax)));

    // 最大値+1 境界値テスト：70
    WLANTEST_SET_TESTNAME(CONVERTRSSI_TESTNAME, &BoundaryTestNo);
    WLANTEST_EXPECT_FALSE((nn::wlan::LinkLevel::LinkLevel_3 == nn::wlan::Infra::ConvertRssiToLinkLevel(Covertrssitolinklevelmax + 1)));

    WLANTEST_EXPECT_RESULT_SUCCESS(nn::wlan::Infra::Disconnect());

} // NOLINT(impl/function_size)

// ChangeRxAntenna試験
TEST(InfraBoundaryUnitTest, ChangeRxAntennaTestCase) NN_NOEXCEPT
{
    BoundaryTestNo = 71;
    nn::wlan::RxAntennaPattern rxAntenna;

    // 最小値-1 境界値テスト：71
    WLANTEST_SET_TESTNAME( CHANGRERX_TESTNAME, &BoundaryTestNo );
    rxAntenna = static_cast<nn::wlan::RxAntennaPattern>(0);
    nn::Result Result = nn::wlan::Infra::ChangeRxAntenna(rxAntenna);
    WLANTEST_EXPECT_TRUE( true == nn::wlan::ResultInvalidArgument::Includes(Result) );

    // 最小値 境界値テスト：72
    WLANTEST_SET_TESTNAME( CHANGRERX_TESTNAME, &BoundaryTestNo );
    rxAntenna = nn::wlan::RxAntennaPattern_0;
    WLANTEST_ASSERT_RESULT_SUCCESS( nn::wlan::Infra::ChangeRxAntenna(rxAntenna) );

    // 最大値 境界値テスト：73
    WLANTEST_SET_TESTNAME( CHANGRERX_TESTNAME, &BoundaryTestNo );
    rxAntenna = nn::wlan::RxAntennaPattern_Both;
    WLANTEST_ASSERT_RESULT_SUCCESS( nn::wlan::Infra::ChangeRxAntenna(rxAntenna) );

    // 最大値+1 境界値テスト：74
    WLANTEST_SET_TESTNAME( CHANGRERX_TESTNAME, &BoundaryTestNo );//TestNo62
    rxAntenna = static_cast<nn::wlan::RxAntennaPattern>(4);
    Result = nn::wlan::Infra::ChangeRxAntenna(rxAntenna);
    WLANTEST_EXPECT_TRUE( true == nn::wlan::ResultInvalidArgument::Includes(Result) );

} // NOLINT(impl/function_size)

// ConnectWithWps試験
TEST(InfraBoundaryUnitTest, ConnectWithWpsTestCase) NN_NOEXCEPT
{
    BoundaryTestNo = 75;
    char sPin[(nn::wlan::WpsPinLength + 1)];
    nn::wlan::Infra::GenerateWpsPin(&sPin[0], sizeof(sPin));

    // 最小値-1 境界値テスト：75
    WLANTEST_SET_TESTNAME(CONNECT_WW_TESTNAME, &BoundaryTestNo);
    nn::Result Result = nn::wlan::Infra::ConnectWithWps(nn::wlan::WpsMethod_Pin, &sPin[0], sizeof(sPin), 0);
    WLANTEST_EXPECT_TRUE(true == nn::wlan::ResultInvalidArgument::Includes(Result));

    // 最小値 境界値テスト：76
    WLANTEST_SET_TESTNAME(CONNECT_WW_TESTNAME, &BoundaryTestNo);
    WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Infra::ConnectWithWps(nn::wlan::WpsMethod_Pin, &sPin[0], sizeof(sPin), 1));

    // 最大値 境界値テスト：77
    WLANTEST_SET_TESTNAME(CONNECT_WW_TESTNAME, &BoundaryTestNo);
    WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Infra::ConnectWithWps(nn::wlan::WpsMethod_Pin, &sPin[0], sizeof(sPin), 30));

    // 最大値+1 境界値テスト：78
    WLANTEST_SET_TESTNAME(CONNECT_WW_TESTNAME, &BoundaryTestNo);
    Result = nn::wlan::Infra::ConnectWithWps(nn::wlan::WpsMethod_Pin, &sPin[0], sizeof(sPin), 31);
    WLANTEST_EXPECT_TRUE(true == nn::wlan::ResultInvalidArgument::Includes(Result));

} // NOLINT(impl/function_size)

// 境界値試験
extern "C" void nnMain()
{
    int     argc = nn::os::GetHostArgc();
    char**  argv = nn::os::GetHostArgv();
    NN_LOG("argc=%d\n", argc);
    for (int i = 0; i < argc; i++)
    {
        NN_LOG("argv[%d]=%s\n", argc, argv[i]);
    }

    ::testing::InitGoogleTest(&argc, argv);

    WLANTEST_STATE_SET(WlanTestState_Infra);
    WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::InitializeInfraManager());
    WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Infra::OpenMode());

    int result = RUN_ALL_TESTS();
    nnt::Exit(result);
}


} // namespace WlanTest
