﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "./Include/testWlan_LocalInfraStressTest.h"

#define STATIC_IP

NN_ALIGNAS(4096) uint8_t  g_MallocBuffer[8 * 1024 * 1024];

extern "C" void nninitStartup()
{
    nn::init::InitializeAllocator(g_MallocBuffer, sizeof(g_MallocBuffer));
}

extern "C"
{
    int testMain(int argc, char **argv);
    void cleanup();
}

namespace {
    const char IpAddressParam[]     = "-oct4=";
    int32_t g_IpOct4                = 5;

    const char ClientNum[]          = "-client=";
    const char AgingTime[]          = "-agingtime=";

    const char IpAddressParam2Ghz[] = "-ip2g=";
    const char IpAddressParam5Ghz[] = "-ip5g=";
    const char DefaultIp2Ghz[]      = "192.168.11.2";
    const char DefaultIp5Ghz[]      = "192.168.10.2";

    const char Ssid2Ghz[]           = "-ssid2g=";
    const char Ssid5Ghz[]           = "-ssid5g=";
    const char DefaultSsid2Ghz[]    = "WlanTest8";
    const char DefaultSsid5Ghz[]    = "WlanTest9";

    const char SubNetMask[]         = "255.255.255.0";

    WlanTest::StressTestParameter g_ParamList;

    void LogoutWlanVersion() NN_NOEXCEPT
    {
        const size_t verSize = 512;
        nn::Result result;
        std::unique_ptr<char[]> verBuffer(new char[verSize]);
        if (verBuffer == nullptr)
        {
            NN_LOG("             Get Version Buffer Allocate Failed\n");
            return;
        }

        result = nn::wlan::InitializeInfraManager();
        if (result.IsSuccess() != true)
        {
            NN_LOG("             Infra Initialize Failed : 0x%08x\n", result.GetInnerValueForDebug());
            return;
        }

        result = nn::wlan::Infra::OpenMode();
        if (result.IsSuccess() != true)
        {
            nn::wlan::FinalizeInfraManager();
            NN_LOG("             Infra OpenMode Failed : 0x%08x\n", result.GetInnerValueForDebug());
            return;
        }

        result = nn::wlan::Infra::GetFwVersion(verBuffer.get(), verSize);
        if (result.IsSuccess() == true)
        {
            NN_LOG("             WLAN VER : %s", verBuffer.get());
        }
        else
        {
            NN_LOG("             GetFwVersion Failed : 0x%08x\n", result.GetInnerValueForDebug());
        }

        result = nn::wlan::Infra::CloseMode();
        if (result.IsSuccess() != true)
        {
            NN_LOG("             Infra CloseMode Failed : 0x%08x\n", result.GetInnerValueForDebug());
        }

        result = nn::wlan::FinalizeInfraManager();
        if (result.IsSuccess() != true)
        {
            NN_LOG("             Infra Finalize Failed : 0x%08x\n", result.GetInnerValueForDebug());
        }
    }

};

namespace WlanTest{

    void TraceState(nn::wlan::WlanState oldState, nn::wlan::WlanState newState) NN_NOEXCEPT
    {
        NN_LOG("\n             **** WLAN STATE CHANGE EVENT *****\n");
        NN_LOG("             Previous State : %d\n", oldState);
        NN_LOG("             Current State  : %d\n", newState);
        NN_LOG("             **** WLAN STATE CHANGE EVENT *****\n\n");
    }

    void TraceState(LocalApiClass* plocal) NN_NOEXCEPT
    {
        NN_ABORT_UNLESS_NOT_NULL(plocal);

        nn::wlan::WlanState state;
        nn::Result result;

#ifdef WLAN_TEST_STATE_ASSER_STOP
        result = plocal->GetState(&state);
#else
        NN_UNUSED(plocal);
        result = nn::wlan::Local::GetState(&state);
#endif
        if (result.IsSuccess() == true)
        {
            NN_LOG("\n             **** WLAN STATE *****\n");
            NN_LOG("             Current State  : %d\n", state);
            NN_LOG("             **** WLAN STATE *****\n\n");
        }
        else
        {
            NN_LOG("             GetState Result failed : 0x%08x\n", result.GetInnerValueForDebug());
        }
    }

    // エージング時間取得
    int64_t GetAgintTime() NN_NOEXCEPT
    {
        int64_t resultTime = Time1h;
        if (g_ParamList.agingList.size() != 0)
        {
            int32_t pos = g_ParamList.agingPos;
            ++g_ParamList.agingPos %= g_ParamList.agingList.size();
            resultTime = g_ParamList.agingList[pos];
        }
        NN_LOG("             Stress Test Aging Time     : %lld msec\n", resultTime);
        return resultTime;
    }

    // クライアント台数取得関数
    int32_t GetClientsCount() NN_NOEXCEPT
    {
        if (g_ParamList.clientCount == 0)
        {
            g_ParamList.clientCount = 1;
        }

        if (g_ParamList.clientCount > ClientMaxConnect)
        {
            g_ParamList.clientCount = ClientMaxConnect;
        }

        NN_LOG("             Stress Test Client Count   : %d\n", g_ParamList.clientCount);

        return g_ParamList.clientCount;
    }

    int32_t GetOct4IpAddress() NN_NOEXCEPT
    {
        return g_ParamList.ipOct4;
    }

    std::string GetIpAddress2g() NN_NOEXCEPT
    {
        return g_ParamList.ip2g;
    }

    std::string GetIpAddress5g() NN_NOEXCEPT
    {
        return g_ParamList.ip5g;
    }

    std::string GetSsid2g() NN_NOEXCEPT
    {
        return g_ParamList.ssid2g;
    }

    std::string GetSsid5g() NN_NOEXCEPT
    {
        return g_ParamList.ssid5g;
    }

    nn::nifm::Security* GetEtherSecurity2g() NN_NOEXCEPT
    {
        return &g_ParamList.etherSecurity2g;
    }

    nn::nifm::Security* GetEtherSecurity5g() NN_NOEXCEPT
    {
        return &g_ParamList.etherSecurity5g;
    }

    nn::nifm::Security* GetWifiSecurity2g() NN_NOEXCEPT
    {
        return &g_ParamList.wifiSecurity2g;
    }

    nn::nifm::Security* GetWifiSecurity5g() NN_NOEXCEPT
    {
        return &g_ParamList.wifiSecurity5g;
    }

    const char* GetSubnetMaskString() NN_NOEXCEPT
    {
        return SubNetMask;
    }

    // カンマ区切りint64_t型分割関数
    void SplitInt64(std::vector<int64_t>&list, const char* pStrInt64list) NN_NOEXCEPT
    {
        if (pStrInt64list == nullptr)
        {
            return;
        }

        std::string strValue = pStrInt64list;
        char* pPos = const_cast<char*>(strValue.c_str());

        while (pPos != nullptr)
        {
            char* pSearch = strstr(pPos, ",");
            if (pSearch != nullptr)
            {
                *pSearch = '\0';
                int64_t value = strtoll(pPos, 0, 10);
                list.push_back(value);
                pPos = ++pSearch;
            }
            else
            {
                if (pPos != nullptr)
                {
                    int64_t value = strtoll(pPos, 0, 10);
                    list.push_back(value);
                }
                break;
            }
        }
    }

    void GetLdnNifmParam(char* arg) NN_NOEXCEPT
    {
        char* search;
        search = strstr(arg, IpAddressParam2Ghz);
        if (search != nullptr)
        {
            int offset = nn::util::Strnlen(IpAddressParam2Ghz, sizeof(IpAddressParam2Ghz));
            g_ParamList.ip2g = std::string(&search[offset]);
            NN_LOG("Get Paramter WiFi-AP 2.4Ghz ip : %s\n", g_ParamList.ip2g.c_str());
            return;
        }

        search = strstr(arg, IpAddressParam5Ghz);
        if (search != nullptr)
        {
            int offset = nn::util::Strnlen(IpAddressParam5Ghz, sizeof(IpAddressParam5Ghz));
            g_ParamList.ip5g = std::string(&search[offset]);
            NN_LOG("Get Paramter WiFi-AP 5Ghz ip : %s\n", g_ParamList.ip5g.c_str());
            return;
        }

        search = strstr(arg, Ssid2Ghz);
        if (search != nullptr)
        {
            int offset = nn::util::Strnlen(Ssid2Ghz, sizeof(Ssid2Ghz));
            g_ParamList.ssid2g = std::string(&search[offset]);
            NN_LOG("Get Paramter WiFi-AP 2.4Ghz ssid : %s\n", g_ParamList.ssid2g.c_str());
            return;
        }

        search = strstr(arg, Ssid5Ghz);
        if (search != nullptr)
        {
            int offset = nn::util::Strnlen(Ssid5Ghz, sizeof(Ssid5Ghz));
            g_ParamList.ssid5g = std::string(&search[offset]);
            NN_LOG("Get Paramter WiFi-AP 5Ghz ssid : %s\n", g_ParamList.ssid5g.c_str());
            return;
        }
    }

    void SettingSecuritys() NN_NOEXCEPT
    {
        g_ParamList.etherSecurity2g.authEncryption.authentication   = nn::nifm::Authentication_Open;
        g_ParamList.etherSecurity2g.authEncryption.encryption       = nn::nifm::Encryption_None;
        nn::util::Strlcpy(g_ParamList.etherSecurity2g.sharedKey.keyMaterial,
            "", g_ParamList.etherSecurity2g.sharedKey.KeyMaterialSize);

        g_ParamList.wifiSecurity2g.authEncryption.authentication = nn::nifm::Authentication_Open;
        g_ParamList.wifiSecurity2g.authEncryption.encryption = nn::nifm::Encryption_None;
        nn::util::Strlcpy(g_ParamList.wifiSecurity2g.sharedKey.keyMaterial,
            "", g_ParamList.wifiSecurity2g.sharedKey.KeyMaterialSize);

        g_ParamList.etherSecurity5g.authEncryption.authentication = nn::nifm::Authentication_Open;
        g_ParamList.etherSecurity5g.authEncryption.encryption = nn::nifm::Encryption_None;
        nn::util::Strlcpy(g_ParamList.etherSecurity5g.sharedKey.keyMaterial,
            "", g_ParamList.etherSecurity5g.sharedKey.KeyMaterialSize);

        g_ParamList.wifiSecurity5g.authEncryption.authentication = nn::nifm::Authentication_Open;
        g_ParamList.wifiSecurity5g.authEncryption.encryption = nn::nifm::Encryption_None;
        nn::util::Strlcpy(g_ParamList.wifiSecurity5g.sharedKey.keyMaterial,
            "", g_ParamList.wifiSecurity5g.sharedKey.KeyMaterialSize);
    }

    // コマンド引数取得
    void GetParam(int argc, char** argv) NN_NOEXCEPT
    {
        g_ParamList.ipOct4          = g_IpOct4;
        g_ParamList.agingPos        = 0;
        g_ParamList.clientCount     = 1;

        g_ParamList.ip2g            = DefaultIp2Ghz;
        g_ParamList.ip5g            = DefaultIp5Ghz;
        g_ParamList.ssid2g          = DefaultSsid2Ghz;
        g_ParamList.ssid5g          = DefaultSsid5Ghz;

        SettingSecuritys();

        for (int i = 0; i < argc; i++)
        {
            // 第四オクテッドIP取得
            char* search = strstr(argv[i], IpAddressParam);
            if (search != nullptr)
            {
                int offset = nn::util::Strnlen(IpAddressParam, sizeof(IpAddressParam));
                g_ParamList.ipOct4 = strtol(&search[offset], 0, 10);
                continue;
            }

            // エージング時間取得（単位ミリ秒）
            search = strstr(argv[i], AgingTime);
            if (search != nullptr)
            {
                int offset = nn::util::Strnlen(AgingTime, sizeof(AgingTime));
                SplitInt64(g_ParamList.agingList, &search[offset]);
                continue;
            }

            // クライアント待機台数（ローカル通信用）
            search = strstr(argv[i], ClientNum);
            if (search != nullptr)
            {
                int offset = nn::util::Strnlen(ClientNum, sizeof(ClientNum));
                g_ParamList.clientCount = strtol(&search[offset], 0, 10);
                continue;
            }

            // LDN NIFMパラメータ（ローカルインフラ通信用）
            GetLdnNifmParam(argv[i]);
        }
    }

    extern "C" void nnMain()
    {
        int     argc = nn::os::GetHostArgc();
        char**  argv = nn::os::GetHostArgv();

        GetParam(argc, argv);

        NN_LOG("*** gtest Parameter Info ***\n");
        for (int i = 0; i < argc; i++)
        {
            NN_LOG("    Parameter(%d) : %s\n", i, argv[i]);
        }
        NN_LOG("*** gtest Parameter Info ***\n\n");

        LogoutWlanVersion();
        ::testing::InitGoogleTest(&argc, argv);

        nnt::Exit(RUN_ALL_TESTS());
    }

} // namespace WlanTest
