﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <memory>

#include "../../Common/testWlan_localApiClass.h"
#include "../../Common/testWlan_UnitTest.h"
#include "../../Common/testWlan_UnitTestCommon.h"

#include <nn/bsdsocket/cfg/cfg_Types.h>
#include <nn/bsdsocket/cfg/cfg_ClientApi.h>
#include <nn/nifm/nifm_ApiForNetworkMiddleware.h>
#include <nn/socket/socket_TypesPrivate.h>

#define STATIC_IP

#include <nn/init.h>

#include <nn/settings/fwdbg/settings_SettingsSetterApi.h>

NN_ALIGNAS(4096) uint8_t  g_MallocBuffer[8 * 1024 * 1024];

extern "C" void nninitStartup()
{
    nn::init::InitializeAllocator(g_MallocBuffer, sizeof(g_MallocBuffer));
}

extern "C"
{
    int testMain(int argc, char **argv);
    void cleanup();
}

namespace {

#ifdef STATIC_IP
#define STATIC_GW_ADDR          "192.168.11.1"
#define STATIC_IP_ADDR          "192.168.11.25"
#define STATIC_CLIENT1_IP_ADDR  "192.168.11.80"
#define STATIC_CLIENT2_IP_ADDR  "192.168.11.81"
#define STATIC_SUBNET_MASK      "255.255.255.0"
#define STATIC_DNS_1            "8.8.8.8"
#define STATIC_DNS_2            "8.8.8.9"
#endif

    NN_ALIGNAS(4096) uint8_t SocketMemoryPoolBuffer[nn::socket::DefaultSocketMemoryPoolSize];

    nn::wlan::Ssid ssidList[] = { nn::wlan::Ssid(LocalMasterTestSSID) };

    const char IpAddressParam[] = "-oct4=";
    int32_t g_IpOct4 = 5;

    const char ClientNum[] = "-client=";
    int32_t g_ClientNum = 3;

    const char ScanTime[] = "-scantime=";
    int32_t g_ScanTime = WlanTestLdnScanTime;
    bool g_SetScanTime = false;

    const char JoyconPairing[]  = "-j";
    bool g_SetJoyconPairing     = false;

    bool g_IsShort = false;

    TestMasterClientInfo TestMasterClient[nn::wlan::ConnectableClientsCountMax + 1];

    struct TestScanParams
    {
        size_t size;
        nn::wlan::Ssid compareSsid;
        nn::wlan::ScanParameters params;
    };

    struct LocalTestIeInfo
    {
        nn::wlan::ManagementFrameType frameType;
        uint8_t vendrId[256];
        size_t vendrLength;
    };

    struct TestScanActionFrame
    {
        uint8_t buffer[1360];
        size_t bufferSize = sizeof(TestScanActionFrame::buffer);
    };

    struct TestScanConnectParam
    {
        uint32_t maxPool;
        uint32_t readPos;
        uint32_t writePos;
        uint32_t rxId;
        int32_t recvInterval;
        std::unique_ptr<TestScanActionFrame[]> actInfo;
        WlanTest::LocalAutoEvent recvEvent;
        bool isStop;
    };

    // TCP SERVER設定
#define WLAN_INTERFACE_NAME "wl0"

    const int kLocalConcurrencyLimit    = 4;

    const uint16_t  PortNo              = 5001;
    const int32_t echo_server_backlog   = 5;

    // StartScanテストパラメータ構造体
    const size_t TestScanMaxSize        = 100 * 1024;
    const uint32_t TestBufferMaxPool    = 100;

    // PLR基準値
    const float TestUnicastPlrPerRate   = 10.00;
    const float TestBroadcastPlrPerRate = 10.00;
    const int TestPlrPaketLostCount     = 4;

    // RTT基準値
    const int TestRttPlrRate            = 70;

    // Time定義
    const int Time1msec     = 1;
    const int Time10msec    = Time1msec * 10;
    const int Time100msec   = Time10msec * 10;
    const int Time300msec   = 300;
    const int Time1s        = 1000;
    const int Time3s        = 3 * Time1s;
    const int Time5s        = 5 * Time1s;
    const int Time10s       = Time1s * 10;
    const int Time15s       = Time1s * 15;
    const int Time30s       = 30 * Time1s;
    const int Time60s       = Time1s * 60;
    const int Time120s      = Time60s * 2;
    const int Time180s      = Time60s * 3;
    const int Time240s      = Time60s * 4;
    const int Time300s      = Time60s * 5;
    const int Time600s      = Time300s * 2;
    const int Time1800s     = Time60s * 30;
    const int Time1h        = Time60s * 60;
    const int Time4h        = Time1h * 4;
    const int Time6h        = Time1h * 6;
    const int Time8h        = Time1h * 8;
    const int Time12h       = Time6h * 2;
    const int Time1day      = Time12h * 2;
    const int PutInterval   = 22;
    const int Interval_1m   = 1;
    const int Interval_5m   = Interval_1m * 5;
    const int Interval_16m  = Interval_5m * 3 + Interval_1m;
    const int Interval_30m  = Interval_5m * 6;
    const int Interval_33m  = Interval_30m + 3;
    const int Interval_100m = 100;

    struct ScanTestPattern
    {
        uint32_t scanSize;
        nn::wlan::ScanParameters scanParam;
    };

    struct TestBufferMngRetail
    {
        size_t bufferSize;
        size_t dataSize;
        uint8_t* pData;
        GetTestFrameType frameType;
    };

    struct TestBufferMng
    {
        uint32_t maxCount;
        uint32_t writeIdx;
        uint32_t readIdx;
        TestBufferMngRetail info[TestBufferMaxPool];
        uint8_t* pBuffer;
        size_t bufferSize;
        size_t packetSize;
    };

    enum TestParsePacketEvent
    {
        EventType_Parse = 0,
        EventType_End
    };

    enum TestScanResultType
    {
        Type_ScanPass = 0,
        Type_ScanApiFail,
        Type_ScanActionFrameFail,
        Type_ConnectPass,
        Type_ConnectFail,
        Type_End
    };

    const uint32_t ClientMaxConnect     = nn::wlan::ConnectableClientsCountMax;
    const uint32_t ClientDynamicConnet  = nn::wlan::ConnectableClientsCountMax - 1;
    const uint32_t Client1SetBt         = ClientMaxConnect;
    const uint32_t Client2SetBt         = (1 + nn::wlan::ConnectableClientsCountMax) / 2 - 1;
    const size_t threadStackSize        = nn::os::ThreadStackAlignment * 4;
    NN_OS_ALIGNAS_THREAD_STACK char  threadStack[threadStackSize];

    const uint16_t EtherTypes[1] = { 0x88b7 };

    static const LocalTestIeInfo LocalTestIe[] = {
        { nn::wlan::ManagementFrameType_Beacon,  { 0x00, 0x22, 0xAA, 0x12, 0x34, 0x56, 0x78 }, 7 },
        { nn::wlan::ManagementFrameType_ProbeRes,{ 0x00, 0x22, 0xAA, 0x12, 0x34, 0x56, 0x79 }, 7 },
        { nn::wlan::ManagementFrameType_AssocRes,{ 0x00, 0x22, 0xAA, 0x12, 0x34, 0x56, 0x80 }, 7 },
        { nn::wlan::ManagementFrameType_AuthRes, { 0x00, 0x22, 0xAA, 0x12, 0x34, 0x56, 0x81 }, 7 },
        { nn::wlan::ManagementFrameType_ProbeReq,{ 0x00, 0x22, 0xAA, 0x12, 0x34, 0x56, 0x82 }, 7 }
    };
};

namespace WlanTest{

    void LogoutWlanVertion() NN_NOEXCEPT
    {
        const size_t verSize = 512;
        nn::Result result;
        std::unique_ptr<char[]> verBuffer(new char[verSize]);
        if (verBuffer == nullptr)
        {
            NN_LOG("             Get Vertion Buffer Allocate Failed\n");
            return;
        }

        result = nn::wlan::InitializeInfraManager();
        if (result.IsSuccess() != true)
        {
            NN_LOG("             Infra Initialize Failed : 0x%08x\n", result.GetInnerValueForDebug());
            return;
        }

        result = nn::wlan::Infra::OpenMode();
        if (result.IsSuccess() != true)
        {
            nn::wlan::FinalizeInfraManager();
            NN_LOG("             Infra OpenMode Failed : 0x%08x\n", result.GetInnerValueForDebug());
            return;
        }

        result = nn::wlan::Infra::GetFwVersion(verBuffer.get(), verSize);
        if (result.IsSuccess() == true)
        {
            NN_LOG("             WLAN VER : %s", verBuffer.get());
        }
        else
        {
            NN_LOG("             GetFwVersion Failed : 0x%08x\n", result.GetInnerValueForDebug());
        }

        result = nn::wlan::Infra::CloseMode();
        if (result.IsSuccess() != true)
        {
            NN_LOG("             Infra CloseMode Failed : 0x%08x\n", result.GetInnerValueForDebug());
        }

        result = nn::wlan::FinalizeInfraManager();
        if (result.IsSuccess() != true)
        {
            NN_LOG("             Infra Finalize Failed : 0x%08x\n", result.GetInnerValueForDebug());
        }
    }

    class TCPConnecter
    {
    protected:
        int sockOpen;
        int sockTcp;
        const int bufSize = 1024 * 10;
        std::unique_ptr<char[]> sendBuf;
        std::unique_ptr<char[]> recvBuf;

        nn::os::ThreadType severTcpId;
        nn::os::ThreadType clientTcpId;
        bool isTcpServerLoad;
        bool isTcpClientLoad;

        bool isTcpClientConnect;
        std::string strTcpIpAddress;

        nn::os::Tick logTick;

        const int64_t logOutTime = Time30s;

    public:
        TCPConnecter() NN_NOEXCEPT
        {
            std::unique_ptr<char[]> sendmessage(new char[bufSize]);
            sendBuf = std::move(sendmessage);

            std::unique_ptr<char[]> recvmessage(new char[bufSize]);
            recvBuf = std::move(recvmessage);

            isTcpServerLoad = false;
            isTcpClientLoad = false;
            isTcpClientConnect = false;
        }

        void ConfigureInterface(const char* interfaceName, const char* ipAddress, bool enable) NN_NOEXCEPT
        {
            nn::Result result;
            nn::socket::IfReq ifr = { WLAN_INTERFACE_NAME };
            ifr.ifr_flags = nn::socket::IfrFlag::Iff_Up;
            int socket = nn::socket::Socket(nn::socket::Family::Pf_Inet, nn::socket::Type::Sock_Dgram, nn::socket::Protocol::IpProto_Udp);
            NN_ABORT_UNLESS(socket >= 0);
            nn::socket::Ioctl(socket, static_cast<nn::socket::IoctlCommand>(nn::socket::IoctlCommandPrivate::SiocSIfFlags), &ifr, sizeof(nn::socket::IfReq));

            if (enable)
            {
                nn::bsdsocket::cfg::IfSettings ifcfg;
                memset(&ifcfg, 0, sizeof(ifcfg));

#ifdef STATIC_IP
                // static ip
                ifcfg.mode = nn::bsdsocket::cfg::IfIpAddrMode_Static;
                ifcfg.mtu = 1500;
                nn::socket::InetAton(ipAddress, &ifcfg.u.modeStatic.addr);
                nn::socket::InetAton(STATIC_GW_ADDR, &ifcfg.u.modeStatic.gatewayAddr);
                nn::socket::InetAton(STATIC_SUBNET_MASK, &ifcfg.u.modeStatic.subnetMask);
                ifcfg.u.modeStatic.broadcastAddr.S_addr =
                    (ifcfg.u.modeStatic.addr.S_addr & ifcfg.u.modeStatic.subnetMask.S_addr) |
                    ~ifcfg.u.modeStatic.subnetMask.S_addr;
                nn::socket::InetAton(STATIC_DNS_1, &ifcfg.dnsAddrs[0]);
                nn::socket::InetAton(STATIC_DNS_2, &ifcfg.dnsAddrs[1]);
#else
                // dhcp
                ifcfg.mode = nn::bsdsocket::cfg::IfIpAddrMode_Dhcp;
                ifcfg.mtu = 1500;
#endif

                result = nn::bsdsocket::cfg::SetIfUp(const_cast<char*>(interfaceName), &ifcfg);
                if (result.IsFailure())
                {
                    NN_LOG("             failed to configure interface %s - %d:%d\n",
                        interfaceName,
                        result.GetModule(),
                        result.GetDescription());
                }
            }
            else
            {
                nn::bsdsocket::cfg::SetIfDown(const_cast<char*>(interfaceName));
            }
        }

        static unsigned long inet_addr(const char* cp) NN_NOEXCEPT
        {
            nn::socket::InAddr      val;

            if (nn::socket::InetAton(cp, &val))
            {
                return val.S_addr;
            }
            else
            {
                NN_LOG("             IP Addr err\n");
                return nn::socket::InAddr_None;
            }
        }

        std::string GetIpAddress() NN_NOEXCEPT
        {
            int fd_ip;
            nn::socket::IfReq ifr;
            std::string strIpAddress = "";

            /* Create Socket */
            fd_ip = nn::socket::Socket(nn::socket::Family::Af_Inet, nn::socket::Type::Sock_Dgram, nn::socket::Protocol::IpProto_Udp);

            /* IP v4 */
            ifr.ifr_addr.sa_family = nn::socket::Family::Af_Inet;

            /* network adapter */
            strncpy(ifr.ifr_name, WLAN_INTERFACE_NAME, nn::socket::IfNamSiz - 1);

            nn::socket::Ioctl(fd_ip, static_cast<nn::socket::IoctlCommand>(nn::socket::IoctlCommandPrivate::SiocGIfAddr), &ifr, sizeof(nn::socket::IfReq));
            fd_ip = nn::socket::Close(fd_ip);

            strIpAddress = nn::socket::InetNtoa(((nn::socket::SockAddrIn *)&ifr.ifr_addr)->sin_addr);

            return strIpAddress;
        }

        void Connect() NN_NOEXCEPT
        {
            nn::socket::SockAddrIn addr;
            nn::socket::SockAddrIn client;
            int ret;
            nn::socket::SockLenT len;
            nn::socket::InAddr ipAddress = { 0 };
            std::string strIpAddress;

            ConfigureInterface(WLAN_INTERFACE_NAME, STATIC_IP_ADDR, true);
            addr.sin_family = nn::socket::Family::Af_Inet;
            addr.sin_port = nn::socket::InetHtons(PortNo);
            addr.sin_addr.S_addr = nn::socket::InetHtonl(nn::socket::InAddr_Any);

            if (nn::nifm::GetCurrentPrimaryIpAddress(&ipAddress).IsSuccess())
            {
                strIpAddress = nn::socket::InetNtoa(ipAddress);
            }
            else
            {
                strIpAddress = "No IpAddress";
            }
            NN_LOG("             Server IpAddress : %s (Port : %d)\n", strIpAddress.c_str(), PortNo);

            /* ソケットを作成します */
            sockOpen = nn::socket::Socket(nn::socket::Family::Af_Inet, nn::socket::Type::Sock_Stream, nn::socket::Protocol::IpProto_Tcp);
            if (0 > sockOpen)
            {
                NN_LOG("             Socket error : %d\n", sockOpen);
                return;
            }

            // サーバーソケットに名前をつける
            ret = nn::socket::Bind(sockOpen, reinterpret_cast<nn::socket::SockAddr *>(&addr), sizeof(addr));
            if (0 > ret)
            {
                NN_LOG("             Bind error : %d\n", nn::socket::GetLastError());
                return;
            }

            // ソケットを受信待機モードにし，保留接続キューのサイズを確保
            ret = nn::socket::Listen(sockOpen, echo_server_backlog);
            if (0 > ret)
            {
                NN_LOG("             Listen error : %d\n", nn::socket::GetLastError());
                return;
            }
            len = sizeof(client);

            // サーバーソケットを接続要求待ち状態
            sockTcp = nn::socket::Accept(sockOpen, reinterpret_cast<nn::socket::SockAddr *>(&client), &len);
            if (0 > sockTcp)
            {
                NN_LOG("             Server Socket error : %d\n", nn::socket::GetLastError());
                return;
            }
        }

        void Disconnect() NN_NOEXCEPT
        {
            // ソケットを解放.
            if (sockTcp > 0)
            {
                int nResult = nn::socket::Shutdown(sockTcp, nn::socket::ShutdownMethod::Shut_RdWr);
                NN_LOG("             sockTcp Shutdown Result(%d) : %d\n", nResult, nn::socket::GetLastError());
                sockTcp = nn::socket::Close(sockTcp);
            }

            // ソケットを解放.
            if (sockOpen > 0)
            {
                int nResult = nn::socket::Shutdown(sockOpen, nn::socket::ShutdownMethod::Shut_RdWr);
                NN_LOG("             sockOpen Shutdown Result : %d\n", nResult, nn::socket::GetLastError());
                sockOpen = nn::socket::Close(sockOpen);
            }

            if (0 > sockTcp)
            {
                NN_LOG("             Close error : %d\n", nn::socket::GetLastError());
            }
            if (0 > sockOpen)
            {
                NN_LOG("             Close error : %d\n", nn::socket::GetLastError());
            }
        }

        void ClientConnect() NN_NOEXCEPT
        {
            nn::socket::SockAddrIn addr;
            nn::socket::InAddr ipAddress = { 0 };
            std::string strIpAddress;

            ConfigureInterface(WLAN_INTERFACE_NAME, strTcpIpAddress.c_str(), true);
            addr.sin_family = nn::socket::Family::Af_Inet;
            addr.sin_port = nn::socket::InetHtons(PortNo);
            addr.sin_addr.S_addr = nn::socket::InetHtonl(nn::socket::InAddr_Any);

            if (nn::nifm::GetCurrentPrimaryIpAddress(&ipAddress).IsSuccess())
            {
                strIpAddress = nn::socket::InetNtoa(ipAddress);
            }
            else
            {
                strIpAddress = "No IpAddress";
            }
            NN_LOG("             Client IpAddress : %s\n", strIpAddress.c_str());

            addr.sin_family = nn::socket::Family::Pf_Inet;

            // 接続先IP
            nn::socket::InetAton(STATIC_IP_ADDR, &addr.sin_addr);
            addr.sin_port = nn::socket::InetHtons(PortNo);

            /* ソケットを作成します */
            sockTcp = nn::socket::Socket(nn::socket::Family::Pf_Inet, nn::socket::Type::Sock_Stream, nn::socket::Protocol::IpProto_Tcp);
            while (isTcpClientLoad == true)
            {
                /* ソケットを作成します */
                sockTcp = nn::socket::Socket(nn::socket::Family::Pf_Inet, nn::socket::Type::Sock_Stream, nn::socket::Protocol::IpProto_Tcp);
                if (0 > sockTcp)
                {
                    NN_LOG("             Socket error : %d\n", sockTcp);
                    continue;
                }

                int result = nn::socket::Connect(sockTcp, (nn::socket::SockAddr *)&addr, sizeof(addr));
                if (result != 0)
                {
                    NN_LOG("             Socket Connect error : %d\n", nn::socket::GetLastError());
                    sockTcp = nn::socket::Close(sockTcp);
                }
                else
                {
                    isTcpClientConnect = true;
                    NN_LOG("             Socket Client Connect Server IpAddress : %s\n", STATIC_IP_ADDR);
                    break;
                }
            }
            NN_LOG("             Socket Client Connect End\n\n");
        }

        void ClientDisconnect() NN_NOEXCEPT
        {
            // ソケットを解放.
            if (0 != sockTcp)
            {
                nn::socket::Shutdown(sockTcp, nn::socket::ShutdownMethod::Shut_RdWr);
                sockTcp = nn::socket::Close(sockTcp);
            }
            if (0 != sockTcp)
            {
                NN_LOG("             Close error : %d\n", nn::socket::GetLastError());
            }
        }

        void Send() NN_NOEXCEPT
        {
            int len = nn::socket::Send(sockTcp, (const char*)sendBuf.get(), bufSize, nn::socket::MsgFlag::Msg_None);
            if (0 > len)
            {
                NN_LOG("             SendTo error : %d\n", nn::socket::GetLastError());
            }
            NN_LOG("             Send OK\n");
        }

        void ClientSend() NN_NOEXCEPT
        {
            int64_t totalSend = 0;
            logTick = nn::os::GetSystemTick();

            while (isTcpClientLoad == true)
            {
                int len = nn::socket::Send(sockTcp, (const char*)sendBuf.get(), bufSize, nn::socket::MsgFlag::Msg_None);

                if (len != bufSize)
                {
                    NN_LOG("             SendTo error : %d\n", nn::socket::GetLastError());
                    break;
                }

                totalSend += len;

                if ((nn::os::GetSystemTick() - logTick).ToTimeSpan().GetMilliSeconds() >= logOutTime)
                {
                    NN_LOG("             Tcp Client Send : %lld byte\n", totalSend);
                    logTick = nn::os::GetSystemTick();
                }
            }

            NN_LOG("             Tcp Client Send Total : %lld byte\n", totalSend);
        }

        void Receive() NN_NOEXCEPT
        {
            // 受信処理.
            int64_t totalRecv = 0;
            int status;
            while (NN_STATIC_CONDITION(1))
            {
                status = nn::socket::Recv(sockTcp, recvBuf.get(), bufSize, nn::socket::MsgFlag::Msg_None);
                if (0 < status)
                {
                    totalRecv += status;
                }
                else if (0 == status)
                {
                    NN_LOG("             Disconnect\n");
                    break;
                }
                else
                {
                    NN_LOG("             Recv failed (error %d)\n", nn::socket::GetLastError());
                    break;
                }

                if ((nn::os::GetSystemTick() - logTick).ToTimeSpan().GetMilliSeconds() >= logOutTime)
                {
                    NN_LOG("             Tcp Server Recv : %lld byte\n", totalRecv);
                    logTick = nn::os::GetSystemTick();
                }
            }
            NN_LOG("             Tcp Server Recv Total : %lld byte\n", totalRecv);
        }

        void StartTcpServer()
        {
            if (isTcpServerLoad == true)
            {
                return;
            }

            static NN_ALIGNAS(4096) uint8_t tcpServerStack[threadStackSize];

            isTcpServerLoad = true;

            // 受信フレーム解析スレッド生成
            WLANTEST_ASSERT_RESULT_SUCCESS(nn::os::CreateThread(&severTcpId, tcpServerThread, this, tcpServerStack, threadStackSize, nn::os::DefaultThreadPriority));
            nn::os::StartThread(&severTcpId);
            nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(Time3s));
        }

        void StopTcpServer()
        {
            if (isTcpServerLoad != true)
            {
                return;
            }

            isTcpServerLoad = false;
            TCPConnecter::Disconnect();
            NN_LOG("             Tcp Server Thread Release Start\n");
            nn::os::WaitThread(&severTcpId);
            nn::os::DestroyThread(&severTcpId);
            NN_LOG("             Tcp Server Thread Release End\n");
        }

        void StartTcpClient(const char* pClientIpAddress)
        {
            if (isTcpClientLoad == true)
            {
                return;
            }

            static NN_ALIGNAS(4096) uint8_t tcpClientStack[threadStackSize];

            isTcpClientLoad = true;
            isTcpClientConnect = false;

            strTcpIpAddress = pClientIpAddress;

            // TCP CLIENT通信スレッド起動
            WLANTEST_ASSERT_RESULT_SUCCESS(nn::os::CreateThread(&clientTcpId, tcpClientThread, this, tcpClientStack, threadStackSize, nn::os::DefaultThreadPriority));
            nn::os::StartThread(&clientTcpId);

            while (isTcpClientConnect != true)
            {
                nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(1000));
            }
        }

        void StopTcpClient()
        {
            if (isTcpClientLoad != true)
            {
                return;
            }

            isTcpClientLoad = false;
            this->ClientDisconnect();

            NN_LOG("             Tcp Client Thread Release Start\n");
            nn::os::WaitThread(&clientTcpId);
            nn::os::DestroyThread(&clientTcpId);
            NN_LOG("             Tcp Client Thread Release End\n");
        }

        static void tcpServerThread(void* arg)
        {
            NN_ABORT_UNLESS_NOT_NULL(arg);
            TCPConnecter* pThis = static_cast<TCPConnecter*>(arg);

            pThis->Connect();
            pThis->Send();
            pThis->Receive();
        }

        static void tcpClientThread(void* arg)
        {
            NN_ABORT_UNLESS_NOT_NULL(arg);
            TCPConnecter* pThis = static_cast<TCPConnecter*>(arg);

            pThis->ClientConnect();
            pThis->ClientSend();
        }

    };

    class LocalMasterFuncTest : public ::testing::Test, public TCPConnecter
    {
    protected:

        LocalApiClass localMaster;

        std::unique_ptr<uint8_t[]> getFrameBuffer;

        nn::wlan::MacAddress macAddress;
        nn::wlan::MacAddress connectMacArray[ClientMaxConnect];

        uint32_t rxId;
        uint32_t actionRxId;
        bool isCreBss;

        nn::os::ThreadType actionframeThread;
        nn::os::EventType actionframeEvent;

        nn::os::SystemEventType connectionEvent;

        nn::os::Tick sysTick;
        nn::os::Tick logOutTime;

        LocalConnectsInfo connectsInfo;

        int32_t sockOpen;

        nn::os::ThreadType scanThread;
        nn::wlan::ScanParameters unsyncScanParam;
        uint64_t intervalScanTime;
        bool isStartScan;
        const bool isEnabled = true;

        nn::wlan::MacAddress testPutMac;
        int16_t testChannel;

        WlanTest::LocalAutoEvent scanWaitEvent;
        WlanTest::LocalAutoEvent actionWaitEvent;
    protected:

        LocalMasterFuncTest() NN_NOEXCEPT
        {
            rxId = static_cast<uint32_t>(-1);
            actionRxId = static_cast<uint32_t>(-1);
            isCreBss = false;

            isStartScan = false;
        }

        virtual void SetUp() NN_NOEXCEPT NN_OVERRIDE
        {
            WLANTEST_ASSERT_RESULT_SUCCESS(nn::nifm::Initialize());
            nn::settings::fwdbg::SetSettingsItemValue("nifm", "is_communication_control_enabled_for_test", &isEnabled, sizeof(isEnabled));
            nn::nifm::SetWirelessCommunicationEnabledForTest(false);
            nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(Time1s));

            WLANTEST_STATE_SET(WlanTestState_Local);
            LogoutWlanVertion();

            WLANTEST_ASSERT_RESULT_SUCCESS(nn::socket::Initialize(reinterpret_cast<void*>(SocketMemoryPoolBuffer),
                nn::socket::DefaultSocketMemoryPoolSize,
                nn::socket::MinSocketAllocatorSize,
                kLocalConcurrencyLimit));

            WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::InitializeLocalManager());

            localMaster.InitializeBtm();

            // デフォルトのジョイコン設定は０台
            WLANTEST_ASSERT_TRUE(localMaster.SetBtMode(LocalApiBtNodeNone));
#ifdef WLAN_TEST_STATE_ASSER_STOP
            localMaster.StateThreadStart(Time100msec, TraceState);
#else
            NN_UNUSED(Time100msec);
#endif
            WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::OpenMasterMode());
            WLANTEST_EXPECT_RESULT_SUCCESS(nn::wlan::Local::GetConnectionEvent(&connectionEvent));
        }

        virtual void TearDown() NN_NOEXCEPT NN_OVERRIDE
        {
            localMaster.LocalRelease();

            if (connectionEvent._state != nn::os::SystemEventType::State::State_NotInitialized)
            {
                nn::os::DestroySystemEvent(&connectionEvent);
            }

            if (rxId != static_cast<uint32_t>(-1))
            {
                WLANTEST_EXPECT_RESULT_SUCCESS(nn::wlan::Local::DeleteRxEntry(rxId));
            }

            if (actionRxId != static_cast<uint32_t>(-1))
            {
                WLANTEST_EXPECT_RESULT_SUCCESS(nn::wlan::Local::DeleteRxEntryForActionFrame(actionRxId));
            }

            if (isCreBss == true)
            {
                WLANTEST_EXPECT_RESULT_SUCCESS(nn::wlan::Local::DestroyBss());
                isCreBss = false;
            }

            WLANTEST_EXPECT_RESULT_SUCCESS(nn::wlan::Local::CloseMasterMode());

#ifdef WLAN_TEST_STATE_ASSER_STOP
            localMaster.StateThreadStop();
#endif

            WLANTEST_EXPECT_RESULT_SUCCESS(nn::wlan::FinalizeLocalManager());

            WLANTEST_EXPECT_RESULT_SUCCESS(nn::socket::Finalize());
        }

        nn::Result SetupBss(const nn::wlan::MasterBssParameters & bssParamer) NN_NOEXCEPT
        {
            nn::Result result;

            nn::wlan::Local::GetMacAddress(&macAddress);
            result = nn::wlan::Local::CreateBss(bssParamer);
            if (result.IsSuccess() != true)
            {
                return result;
            }

            isCreBss = true;

            return nn::ResultSuccess();
        }

        nn::Result SetupBssAndEntry(const nn::wlan::MasterBssParameters & bssParamer, const nn::wlan::ReceivedDataMatchInfo matchInfo[], uint32_t matchCount) NN_NOEXCEPT
        {
            nn::Result result;

            result = SetupBss(bssParamer);
            if (result.IsSuccess() != true)
            {
                return result;
            }

            result = nn::wlan::Local::CreateRxEntry(&rxId, EtherTypes, sizeof(EtherTypes) / sizeof(EtherTypes), 30);
            if (result.IsSuccess() != true)
            {
                return result;
            }

            for (int32_t i = 0; i < matchCount; i++)
            {
                result = nn::wlan::Local::AddMatchingDataToRxEntry(rxId, matchInfo[i]);
                if (result.IsSuccess() != true)
                {
                    return result;
                }
            }

            return nn::ResultSuccess();
        }

        nn::Result SetupBssAndActionEntry(const nn::wlan::MasterBssParameters & bssParamer) NN_NOEXCEPT
        {
            nn::Result result;

            result = SetupBss(bssParamer);
            if (result.IsSuccess() != true)
            {
                return result;
            }

            result = nn::wlan::Local::CreateRxEntryForActionFrame(&actionRxId, EtherTypes, sizeof(EtherTypes) / sizeof(EtherTypes), 10);
            if (result.IsSuccess() != true)
            {
                return result;
            }

            return nn::ResultSuccess();
        }

        void AllScan() NN_NOEXCEPT
        {
            std::unique_ptr<uint8_t[]> pTestBuffer(new uint8_t[TestScanMaxSize]);
            NN_ABORT_UNLESS_NOT_NULL(pTestBuffer.get());
            nn::wlan::ScanIeMatchInfo info;

            const nn::wlan::ScanParameters StartScanTestParams = {
                nn::wlan::ScanType_Active,{ 1 }, 0, 40, -1, nullptr, 0, nn::wlan::MacAddress::CreateBroadcastMacAddress()
            };

            WLANTEST_EXPECT_RESULT_SUCCESS(nn::wlan::Local::StartScan(pTestBuffer.get(), TestScanMaxSize, StartScanTestParams, &info));
        }

        int GetRandom(int min, int max) NN_NOEXCEPT
        {
            static time_t check;
            int rand_ret;
            if (check != time(nullptr))
            {
                check = time(nullptr);
                srand(static_cast<unsigned int>(time(nullptr)));
            }
            rand_ret = min + static_cast<int>(rand() * (max - min + 1.0) / (1.0 + RAND_MAX));
            return rand_ret;
        }

        bool CompareEtherType(uint16_t dscType, uint8_t typeArray[2])
        {
            uint16_t etherType = static_cast<uint16_t>((typeArray[0] << 8 & 0xFF00) | typeArray[1]);
            return (dscType == etherType);
        }

        bool CompareScanSsid(uint8_t pScanBuffer[], size_t bufferSize, nn::wlan::Ssid compareSsid) NN_NOEXCEPT
        {
            nn::wlan::BeaconScanResultReader resultReader(pScanBuffer);
            uint32_t bssCount = resultReader.GetCount();
            NN_UNUSED(bufferSize);

            for (uint32_t i = 0; i < bssCount; i++)
            {
                nn::wlan::BeaconDescriptionReader beacon = resultReader.GetNextDescription();
                if (compareSsid == beacon.GetSsid())
                {
                    return true;
                }
            }

            return false;
        }

        void StartUnsyncScanThread(nn::wlan::ScanParameters TestScanParam, uint64_t intervalTime) NN_NOEXCEPT
        {
            if (isStartScan == true)
            {
                return;
            }

            static NN_ALIGNAS(4096) uint8_t scanStack[threadStackSize];
            unsyncScanParam = TestScanParam;
            intervalScanTime = intervalTime;

            isStartScan = true;
            WLANTEST_ASSERT_RESULT_SUCCESS(nn::os::CreateThread(&scanThread, UnSyncScanThread, this, scanStack, threadStackSize, nn::os::DefaultThreadPriority));
            nn::os::StartThread(&scanThread);
        }

        void StartSyncScanThread(nn::wlan::ScanParameters TestScanParam, uint64_t intervalTime) NN_NOEXCEPT
        {
            if (isStartScan == true)
            {
                return;
            }

            scanWaitEvent.SetEvent(false, nn::os::EventClearMode::EventClearMode_ManualClear);
            actionWaitEvent.SetEvent(true, nn::os::EventClearMode::EventClearMode_ManualClear);

            static NN_ALIGNAS(4096) uint8_t scanStack[threadStackSize];
            unsyncScanParam = TestScanParam;
            intervalScanTime = intervalTime;

            isStartScan = true;
            WLANTEST_ASSERT_RESULT_SUCCESS(nn::os::CreateThread(&scanThread, SyncScanThread, this, scanStack, threadStackSize, nn::os::DefaultThreadPriority));
            nn::os::StartThread(&scanThread);
        }

        void StopUnsyncScanThread() NN_NOEXCEPT
        {
            if (isStartScan != true)
            {
                return;
            }

            isStartScan = false;

            nn::os::WaitThread(&scanThread);
            nn::os::DestroyThread(&scanThread);
        }

        static void SendActionFrameThread(void* arg) NN_NOEXCEPT
        {
            NN_ABORT_UNLESS_NOT_NULL(arg);
            LocalMasterFuncTest* pThis = static_cast<LocalMasterFuncTest*>(arg);

            size_t putSize = ActionFrameBufferSize;
            std::unique_ptr<uint8_t[]> putBuffer(new uint8_t[putSize]);
            nn::wlan::MacAddress fromMacAddress;

            WLANTEST_EXPECT_RESULT_SUCCESS(nn::wlan::Local::GetMacAddress(&fromMacAddress));

            for (int i = 0; i < 1000; i++)
            {
                WLANTEST_EXPECT_TRUE(LocalApiClass::LocalMakeActionFrame(putBuffer.get(), putSize, pThis->testPutMac, fromMacAddress, EtherTypes[0], TestMatchInfo[0], nn::wlan::ActionFrameType_Local, i));
                WLANTEST_EXPECT_RESULT_SUCCESS(nn::wlan::Local::PutActionFrameOneShot(pThis->testPutMac, putBuffer.get(), putSize, pThis->testChannel, 30));
                nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(Time300msec));
            }
        }

        bool SetActionFrameWithBeaconTestWait(uint32_t stopConnectCnt) NN_NOEXCEPT
        {
            size_t putSize = 1360;
            std::unique_ptr<uint8_t[]> pTestBuffer(new uint8_t[putSize]);
            nn::wlan::MacAddress myMac;
            nn::wlan::MacAddress putMac = nn::wlan::MacAddress::CreateBroadcastMacAddress();

            if (pTestBuffer == nullptr)
            {
                return false;
            }

            if (localMaster.LocalMakeActionFrame(pTestBuffer.get(), putSize, putMac, localMaster.m_macAddress,
                EtherTypes[0], TestMatchInfo[0], nn::wlan::ActionFrameType_Beacon) != true)
            {
                return false;
            }

            WLANTEST_EXPECT_RESULT_SUCCESS(nn::wlan::Local::SetActionFrameWithBeacon(pTestBuffer.get(), putSize));

            sysTick = nn::os::GetSystemTick();
            while ((nn::os::GetSystemTick() - sysTick).ToTimeSpan().GetMilliSeconds() <= Time3s)
            {
                nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(Time100msec));
                // CLIENTが切断状態の場合は常にTick更新。試験CLIENTの終了時に接続待機させループ終了タイムまでTickは更新しない
                if (localMaster.GetStateClientCount(nn::wlan::ConnectionState_Connected) < stopConnectCnt)
                {
                    sysTick = nn::os::GetSystemTick();
                }
            }

            return true;
        }

        static void TraceState(nn::wlan::WlanState oldState, nn::wlan::WlanState newState) NN_NOEXCEPT
        {
            NN_LOG("\n             **** WLAN STATE CHANGE EVENT *****\n");
            NN_LOG("             Previous State : %d\n", oldState);
            NN_LOG("             Current State  : %d\n", newState);
            NN_LOG("             **** WLAN STATE CHANGE EVENT *****\n\n");
        }

        static void TraceState(LocalApiClass* plocal) NN_NOEXCEPT
        {
            NN_ABORT_UNLESS_NOT_NULL(plocal);

            nn::wlan::WlanState state;
            nn::Result result;

#ifdef WLAN_TEST_STATE_ASSER_STOP
            result = plocal->GetState(&state);
#else
            NN_UNUSED(plocal);
            result = nn::wlan::Local::GetState(&state);
#endif
            if (result.IsSuccess() == true)
            {
                NN_LOG("\n             **** WLAN STATE *****\n");
                NN_LOG("             Current State  : %d\n", state);
                NN_LOG("             **** WLAN STATE *****\n\n");
            }
            else
            {
                NN_LOG("             GetState Result failed : 0x%08x\n", result.GetInnerValueForDebug());
            }
        }

        static void UnSyncScanThread(void* arg) NN_NOEXCEPT
        {
            NN_ABORT_UNLESS_NOT_NULL(arg);
            LocalMasterFuncTest* pThis = static_cast<LocalMasterFuncTest*>(arg);

            std::unique_ptr<uint8_t[]> pTestBuffer(new uint8_t[TestScanMaxSize]);
            NN_ABORT_UNLESS_NOT_NULL(pTestBuffer.get());
            nn::wlan::ScanIeMatchInfo info;

            NN_LOG("             Master UnSyncScanThread -> StartScan Start\n");
            while (pThis->isStartScan == true)
            {
                nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(pThis->intervalScanTime));
                WLANTEST_EXPECT_RESULT_SUCCESS(nn::wlan::Local::StartScan(pTestBuffer.get(), TestScanMaxSize, pThis->unsyncScanParam, &info));
            }
            NN_LOG("             Master UnSyncScanThread -> StartScan End\n\n");
        }

        static void SyncScanThread(void* arg) NN_NOEXCEPT
        {
            NN_ABORT_UNLESS_NOT_NULL(arg);
            LocalMasterFuncTest* pThis = static_cast<LocalMasterFuncTest*>(arg);

            std::unique_ptr<uint8_t[]> pTestBuffer(new uint8_t[TestScanMaxSize]);
            NN_ABORT_UNLESS_NOT_NULL(pTestBuffer.get());
            nn::wlan::ScanIeMatchInfo info;

            while (pThis->isStartScan == true)
            {
                pThis->scanWaitEvent.WaitSygnal(Time10s);
                NN_LOG("             Master SyncScanThread -> StartScan Start\n");
                WLANTEST_EXPECT_RESULT_SUCCESS(nn::wlan::Local::StartScan(pTestBuffer.get(), TestScanMaxSize, pThis->unsyncScanParam, &info));
                pThis->scanWaitEvent.ClearSygnal();
                pThis->actionWaitEvent.SetSygnal();
                NN_LOG("             Master SyncScanThread -> StartScan End\n\n");
            }
        }
    };

    class LocalClientFuncTest : public ::testing::Test, public TCPConnecter
    {
    protected:
        nn::os::ThreadType actionframeThread;
        nn::os::EventType actionframeEvent;

        LocalApiClass localClient;

        nn::os::ThreadType scanThread;
        nn::wlan::ScanParameters unsyncScanParam;
        uint64_t intervalScanTime;
        bool isStartScan;

        nn::os::SystemEventType connectionEvent;
        nn::wlan::ConnectionStatus connectionStatus;

        nn::wlan::DisconnectClient disClient;

        nn::os::Tick sysTick;
        nn::os::Tick logOutTime;
        LocalConnectsInfo connectsInfo;

        const bool isEnabled = true;

        nn::wlan::MacAddress testPutMac;
        int16_t testChannel;

        WlanTest::LocalAutoEvent scanWaitEvent;
        WlanTest::LocalAutoEvent actionWaitEvent;

#ifdef WLAN_TEST_LINKLEVEL_ERROR
        nn::os::Tick connectLogApiStart;
        nn::os::Tick linkApiStart;
        nn::os::Tick connectStsApiStart;
        bool isConnectLogCheck;
        bool isLinkApiCheck;
        bool isConnectStsCheck;

        nn::os::ThreadType testCheckThread;
        bool isCheck;
#endif

    protected:
        LocalClientFuncTest() NN_NOEXCEPT
        {
            isStartScan     = false;
        }

        virtual void SetUp() NN_NOEXCEPT NN_OVERRIDE
        {
            WLANTEST_ASSERT_RESULT_SUCCESS(nn::nifm::Initialize());
            nn::settings::fwdbg::SetSettingsItemValue("nifm", "is_communication_control_enabled_for_test", &isEnabled, sizeof(isEnabled));
            nn::nifm::SetWirelessCommunicationEnabledForTest(false);
            nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(Time1s));

            WLANTEST_STATE_SET(WlanTestState_Local);
            LogoutWlanVertion();

            WLANTEST_ASSERT_RESULT_SUCCESS(nn::socket::Initialize(reinterpret_cast<void*>(SocketMemoryPoolBuffer),
                nn::socket::DefaultSocketMemoryPoolSize,
                nn::socket::MinSocketAllocatorSize,
                kLocalConcurrencyLimit));

            WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::InitializeLocalManager());

            localClient.InitializeBtm();

            // デフォルトのジョイコン設定は０台
            WLANTEST_ASSERT_TRUE(localClient.SetBtMode(LocalApiBtNodeNone));
#ifdef WLAN_TEST_STATE_ASSER_STOP
            localClient.StateThreadStart(Time100msec, TraceState);
#endif

            WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::OpenClientMode());

            WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::GetConnectionEvent(&connectionEvent));
        }

        virtual void TearDown() NN_NOEXCEPT NN_OVERRIDE
        {
            localClient.LocalClientDisconnect(Time60s);
            localClient.LocalRelease();

            WLANTEST_EXPECT_RESULT_SUCCESS(nn::wlan::Local::CloseClientMode());

#ifdef WLAN_TEST_STATE_ASSER_STOP
            localClient.StateThreadStop();
#endif

            WLANTEST_EXPECT_RESULT_SUCCESS(nn::wlan::FinalizeLocalManager());
            WLANTEST_EXPECT_RESULT_SUCCESS(nn::socket::Finalize());
        }

        void AllScan() NN_NOEXCEPT
        {
            std::unique_ptr<uint8_t[]> pTestBuffer(new uint8_t[TestScanMaxSize]);
            NN_ABORT_UNLESS_NOT_NULL(pTestBuffer.get());
            nn::wlan::ScanIeMatchInfo info;

            const nn::wlan::ScanParameters StartScanTestParams = {
                nn::wlan::ScanType_Active, { 1 }, 0, 40, -1, nullptr, 0, nn::wlan::MacAddress::CreateBroadcastMacAddress()
            };

            WLANTEST_EXPECT_RESULT_SUCCESS(nn::wlan::Local::StartScan(pTestBuffer.get(), TestScanMaxSize, StartScanTestParams, &info));
        }

        int GetRandom(int min, int max) NN_NOEXCEPT
        {
            static time_t check;
            int rand_ret;
            if (check != time(nullptr))
            {
                check = time(nullptr);
                srand(static_cast<unsigned int>(time(nullptr)));
            }
            rand_ret = min + static_cast<int>(rand() * (max - min + 1.0) / (1.0 + RAND_MAX));
            NN_LOG("             rand : %d\n", rand_ret);
            return rand_ret;
        }

        bool CompareScanSsid(uint8_t pScanBuffer[], size_t bufferSize, nn::wlan::Ssid compareSsid) NN_NOEXCEPT
        {
            nn::wlan::BeaconScanResultReader resultReader(pScanBuffer);
            uint32_t bssCount = resultReader.GetCount();
            NN_UNUSED(bufferSize);

            for (uint32_t i = 0; i < bssCount; i++)
            {
                nn::wlan::BeaconDescriptionReader beacon = resultReader.GetNextDescription();
                if (compareSsid == beacon.GetSsid())
                {
                    return true;
                }
            }

            return false;
        }

        bool ScanGetMacAddress(nn::wlan::MacAddress& resultMacAddress, uint8_t pScanBuffer[], size_t bufferSize, nn::wlan::Ssid compareSsid) NN_NOEXCEPT
        {
            nn::wlan::BeaconScanResultReader resultReader(pScanBuffer);
            uint32_t bssCount = resultReader.GetCount();
            NN_UNUSED(bufferSize);

            for (uint32_t i = 0; i < bssCount; i++)
            {
                nn::wlan::BeaconDescriptionReader beacon = resultReader.GetNextDescription();
                if (compareSsid == beacon.GetSsid())
                {
                    resultMacAddress = beacon.GetBssid();
                    return true;
                }
            }
            return false;
        }

        static void ScanActionFrameThread(void* arg) NN_NOEXCEPT
        {
            NN_ABORT_UNLESS_NOT_NULL(arg);
            TestScanConnectParam* pInfo = static_cast<TestScanConnectParam*>(arg);
            char macStrArray[nn::wlan::MacAddress::MacStringSize];
            nn::wlan::MacAddress outMacAddress;
            uint32_t* pPos = &(pInfo->writePos = 0);
            TestScanActionFrame* pBufInfo = &pInfo->actInfo[*pPos];
            int64_t recvInterval = pInfo->recvInterval;
            int64_t recvWait;
            nn::os::Tick recvTick;
            size_t outSize = 0;
            nn::Result result;

            NN_LOG("             ScanActionFrameThread -> ActionRxID : %d START\n", pInfo->rxId);
            while (pInfo->isStop != true)
            {
                recvTick = nn::os::GetSystemTick();
                result = nn::wlan::Local::GetActionFrame(&outMacAddress, pBufInfo->buffer,
                    pBufInfo->bufferSize, &outSize, pInfo->rxId);
                if (result.IsSuccess() == true)
                {
                    ++*pPos %= pInfo->maxPool;
                    pBufInfo = &pInfo->actInfo[*pPos];
                    pInfo->recvEvent.SetSygnal();
                    NN_LOG("             ScanActionFrameThread -> GetAction Frame From(%s) %lu Byte\n\n",
                                                                outMacAddress.GetString(macStrArray), outSize);

                    recvWait = (nn::os::GetSystemTick() - recvTick).ToTimeSpan().GetMilliSeconds();
                    if (recvInterval > 0 && recvInterval > recvWait)
                    {
                        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(recvInterval - recvWait));
                    }
                }
                else if (nn::wlan::ResultGetFrameCancelled().Includes(result) != true)
                {
                    NN_LOG("             ScanActionFrameThread -> GetActionFrame Error : 0x%08x\n", result.GetInnerValueForDebug());
                }
            }
            NN_LOG("             ScanActionFrameThread -> END\n\n");
        }

        bool ScanConnectRate(int32_t resultArray[Type_End], LocalClientConnectParameter conParameter, int32_t retryCount) NN_NOEXCEPT
        {
            nn::wlan::ScanIeMatchInfo info;
            std::unique_ptr<uint8_t[]> pTestBuffer(new uint8_t[TestScanMaxSize]);
            bool isScanMatch = false;

            const nn::wlan::ScanParameters Scan2GHz = {
                nn::wlan::ScanType_Passive,{ 1, 6, 11 }, 3, 110, 0, nullptr, 0, nn::wlan::MacAddress::CreateBroadcastMacAddress()
            };
            const int16_t Scan5GHzChannel[] = {
                36, 40, 44, 48
            };

            if (pTestBuffer.get() == nullptr)
            {
                NN_LOG("             ScanConnectRate Heap Memory Allocate Error\n\n");
                return false;
            }

            std::memset(resultArray, 0x00, sizeof(int32_t) * Type_End);
            for (int i = 0; i < retryCount; i++)
            {
                std::memset(pTestBuffer.get(), 0x00, TestScanMaxSize);
                // 2.4GHzスキャン
                nn::wlan::Local::StartScan(pTestBuffer.get(), TestScanMaxSize, Scan2GHz, &info);
                isScanMatch = localClient.GetScanResult(conParameter.ssid, pTestBuffer.get(), TestScanMaxSize);
                if (isScanMatch != true)
                {
                    // 5GHzスキャン（LDN仕様）
                    for (auto nChannel : Scan5GHzChannel)
                    {
                        nn::wlan::ScanParameters Scan5GHz = {
                            nn::wlan::ScanType_Passive,{ nChannel }, 1, 110, 0, nullptr, 0, nn::wlan::MacAddress::CreateBroadcastMacAddress()
                        };

                        nn::wlan::Local::StartScan(pTestBuffer.get(), TestScanMaxSize, Scan2GHz, &info);
                        isScanMatch = localClient.GetScanResult(conParameter.ssid, pTestBuffer.get(), TestScanMaxSize);
                        if (isScanMatch == true)
                        {
                            break;
                        }
                    }
                }

                if (isScanMatch != true)
                {
                    resultArray[Type_ScanApiFail]++;
                    continue;
                }

                resultArray[Type_ScanPass]++;
                if (localClient.LocalClientConnect(conParameter, static_cast<int64_t>(Time10s)) == true)
                {
                    resultArray[Type_ConnectPass]++;
                    localClient.LocalClientDisconnect(Time60s);
                }
                else
                {
                    resultArray[Type_ConnectFail]++;
                }
            }

            return true;
        }

        bool ScanActionFrameStart(TestScanConnectParam& actInfo, uint32_t actionRxId) NN_NOEXCEPT
        {
            return ScanActionFrameStart(actInfo, actionRxId, 0);
        }

        bool ScanActionFrameStart(TestScanConnectParam& actInfo, uint32_t actionRxId, int32_t recvInterval) NN_NOEXCEPT
        {
            std::unique_ptr<TestScanActionFrame[]> pActInfo(new TestScanActionFrame[100]);

            if (pActInfo == nullptr)
            {
                return false;
            }

            actInfo.isStop = false;
            actInfo.maxPool = 100;
            actInfo.actInfo = std::move(pActInfo);
            actInfo.recvEvent.SetEvent(false, nn::os::EventClearMode_ManualClear);
            actInfo.rxId = actionRxId;
            actInfo.recvInterval = recvInterval;

            AllScan();
            if (nn::os::CreateThread(&actionframeThread, ScanActionFrameThread,
                &actInfo, threadStack, threadStackSize, nn::os::DefaultThreadPriority).IsSuccess() != true)
            {
                return false;
            }
            nn::os::StartThread(&actionframeThread);

            // 一旦ActionFrameが残っていないか待機して破棄する
            nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(Time1s));
            actInfo.readPos = actInfo.writePos;

            return true;
        }

        bool ScanActionConnectRate(int32_t resultArray[Type_End], TestScanConnectParam& actInfo, LocalClientConnectParameter conParameter, int32_t retryCount) NN_NOEXCEPT
        {
            bool isScanMatch = false;

            std::memset(resultArray, 0x00, sizeof(int32_t) * Type_End);
            for (int i = 0; i < retryCount; i++)
            {
                // スキャン開始前に一旦受信済みシグナルリセットと読み込み済み設定
                actInfo.recvEvent.ClearSygnal();
                actInfo.readPos = actInfo.writePos;

                isScanMatch = localClient.LocalPassiveScanResult(conParameter.ssid, conParameter.channel, g_ScanTime);
                if (isScanMatch == true)
                {
                    // Scan中のみActionFrame受信待機
                    isScanMatch = actInfo.recvEvent.TrySygnal();
                    // Scan後も110msecのActionFrame受信待機
                    //isScanMatch = actInfo.recvEvent.WaitSygnal(Time100msec + Time10msec);
                    if (isScanMatch != true)
                    {
                        resultArray[Type_ScanActionFrameFail]++;
                        continue;
                    }
                }
                else
                {
                    resultArray[Type_ScanApiFail]++;
                    continue;
                }

                ++actInfo.readPos %= actInfo.maxPool;
                if (localClient.LocalClientConnect(conParameter, static_cast<int64_t>(Time10s)) == true)
                {
                    resultArray[Type_ConnectPass]++;
                    localClient.LocalClientDisconnect(Time60s);
                }
                else
                {
                    resultArray[Type_ConnectFail]++;
                }
            }

            return true;
        }

        void ScanActionFrameStop(TestScanConnectParam& actInfo, LocalClientConnectParameter conParameter) NN_NOEXCEPT
        {
            if (actionframeThread._state > nn::os::ThreadType::State_NotInitialized)
            {
                actInfo.isStop = true;
                if (nn::wlan::Local::CancelGetActionFrame(localClient.m_actionRxId).IsSuccess() == true)
                {
                    if (localClient.LocalClientConnect(conParameter, static_cast<int64_t>(Time5s)) == true)
                    {
                        // MasterのBeacon停止条件の接続待機（10秒間接続状態でMasterのループを終了させる）
                        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(Time10s));
                        localClient.LocalClientDisconnect(Time10s);
                    }

                    nn::os::WaitThread(&actionframeThread);
                    nn::os::DestroyThread(&actionframeThread);
                }
            }
        }

        void StartSyncScanThread(nn::wlan::ScanParameters TestScanParam, uint64_t intervalTime) NN_NOEXCEPT
        {
            if (isStartScan == true)
            {
                return;
            }

            static NN_ALIGNAS(4096) uint8_t scanStack[threadStackSize];
            unsyncScanParam = TestScanParam;
            intervalScanTime = intervalTime;

            isStartScan = true;
            WLANTEST_ASSERT_RESULT_SUCCESS(nn::os::CreateThread(&scanThread, SyncScanThread, this, scanStack, threadStackSize, nn::os::DefaultThreadPriority));
            nn::os::StartThread(&scanThread);
        }

        static void SyncScanThread(void* arg) NN_NOEXCEPT
        {
            NN_ABORT_UNLESS_NOT_NULL(arg);
            LocalClientFuncTest* pThis = static_cast<LocalClientFuncTest*>(arg);

            std::unique_ptr<uint8_t[]> pTestBuffer(new uint8_t[TestScanMaxSize]);
            NN_ABORT_UNLESS_NOT_NULL(pTestBuffer.get());
            nn::wlan::ScanIeMatchInfo info;

            NN_LOG("             Client SyncScanThread -> StartScan Start\n");
            while (pThis->isStartScan == true)
            {
                pThis->scanWaitEvent.WaitSygnal(Time10s);
                WLANTEST_EXPECT_RESULT_SUCCESS(nn::wlan::Local::StartScan(pTestBuffer.get(), TestScanMaxSize, pThis->unsyncScanParam, &info));
                pThis->scanWaitEvent.ClearSygnal();
                pThis->actionWaitEvent.SetSygnal();
            }
            NN_LOG("             Client SyncScanThread -> StartScan End\n\n");
        }

        static void TraceState(nn::wlan::WlanState oldState, nn::wlan::WlanState newState) NN_NOEXCEPT
        {
            NN_LOG("\n             **** WLAN STATE CHANGE EVENT *****\n");
            NN_LOG("             Previous State : %d\n", oldState);
            NN_LOG("             Current State  : %d\n", newState);
            NN_LOG("             **** WLAN STATE CHANGE EVENT *****\n\n");
        }

        static void TraceState(LocalApiClass* plocal) NN_NOEXCEPT
        {
            NN_ABORT_UNLESS_NOT_NULL(plocal);

            nn::wlan::WlanState state;
            nn::Result result;

#ifdef WLAN_TEST_STATE_ASSER_STOP
            result = plocal->GetState(&state);
#else
            NN_UNUSED(plocal);
            result = nn::wlan::Local::GetState(&state);
#endif
            if (result.IsSuccess() == true)
            {
                NN_LOG("\n             **** WLAN STATE *****\n");
                NN_LOG("             Current State  : %d\n", state);
                NN_LOG("             **** WLAN STATE *****\n\n");
            }
            else
            {
                NN_LOG("             GetState Result failed : 0x%08x\n", result.GetInnerValueForDebug());
            }
        }

        void StartUnsyncScanThread(nn::wlan::ScanParameters TestScanParam, uint64_t intervalTime) NN_NOEXCEPT
        {
            if (isStartScan == true)
            {
                return;
            }

            static NN_ALIGNAS(4096) uint8_t scanStack[threadStackSize];
            unsyncScanParam = TestScanParam;
            intervalScanTime = intervalTime;

            isStartScan = true;
            WLANTEST_ASSERT_RESULT_SUCCESS(nn::os::CreateThread(&scanThread, UnSyncScanThread, this, scanStack, threadStackSize, nn::os::DefaultThreadPriority));
            nn::os::StartThread(&scanThread);
        }

        void StopUnsyncScanThread() NN_NOEXCEPT
        {
            if (isStartScan != true)
            {
                return;
            }

            isStartScan = false;

            nn::os::WaitThread(&scanThread);
            nn::os::DestroyThread(&scanThread);
        }

        static void SendActionFrameThread(void* arg) NN_NOEXCEPT
        {
            NN_ABORT_UNLESS_NOT_NULL(arg);
            LocalClientFuncTest* pThis = static_cast<LocalClientFuncTest*>(arg);

            size_t putSize = ActionFrameBufferSize;
            std::unique_ptr<uint8_t[]> putBuffer(new uint8_t[putSize]);
            nn::wlan::MacAddress fromMacAddress;

            WLANTEST_EXPECT_RESULT_SUCCESS(nn::wlan::Local::GetMacAddress(&fromMacAddress));

            for (int i = 0; i < 1000; i++)
            {
                WLANTEST_EXPECT_TRUE(LocalApiClass::LocalMakeActionFrame(putBuffer.get(), putSize, pThis->testPutMac, fromMacAddress, EtherTypes[0], TestMatchInfo[0], nn::wlan::ActionFrameType_Local, i));
                WLANTEST_EXPECT_RESULT_SUCCESS(nn::wlan::Local::PutActionFrameOneShot(pThis->testPutMac, putBuffer.get(), putSize, pThis->testChannel, 30));
                nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(Time300msec));
            }
        }

        static void ReceiveActionFrameThread(void* arg) NN_NOEXCEPT
        {
            NN_ABORT_UNLESS_NOT_NULL(arg);
            LocalClientFuncTest* pThis = static_cast<LocalClientFuncTest*>(arg);
            const uint16_t FrameTypes[] = { nn::wlan::ActionFrameType_Beacon, nn::wlan::ActionFrameType_Local };
            nn::wlan::MacAddress outMacAddress;
            std::unique_ptr<uint8_t[]> getBuffer(new uint8_t[GetActionFrameBufferSize]);
            nn::wlan::ConnectionStatus conStatus;
            nn::Result result;
            size_t outSize;

            WLANTEST_ASSERT_TRUE(getBuffer.get() != nullptr);

            // 3回受信する
            for(int i = 0; i < 3; i ++)
            {
                WLANTEST_ASSERT_TRUE(pThis->localClient.LocalCancelActionFrameStart(Time5s));
                result = nn::wlan::Local::GetActionFrame(&outMacAddress, getBuffer.get(), GetActionFrameBufferSize, &outSize, pThis->localClient.m_actionRxId);
                pThis->localClient.LocalCancelActionFrameStop();
                if (result.IsSuccess() != true)
                {
                    WLANTEST_EXPECT_RESULT_SUCCESS(nn::wlan::Local::GetConnectionStatus(&conStatus));
                    // MASTERからの接続が切られていたら評価せずに終了
                    if (conStatus.state != nn::wlan::ConnectionState_Connected)
                    {
                        NN_LOG("             Master Disconnected\n");
                        break;
                    }
                    WLANTEST_EXPECT_RESULT_SUCCESS(result);
                }

                // 設定したフレームタイプ比較
                bool isMatchFramType = false;
                LocalActionFrameData* pActionFrame = reinterpret_cast<LocalActionFrameData*>(getBuffer.get());
                for (auto frameType : FrameTypes)
                {
                    if (frameType == pActionFrame->frameType)
                    {
                        isMatchFramType = true;
                        break;
                    }
                }

                if (outSize > ActionFrameBufferSize)
                {
                    nn::wlan::MacAddress fromMac(pActionFrame->fromMacAddress);
                    char clientMacArray[nn::wlan::MacAddress::MacStringSize];
                    NN_LOG("             ReceiveActionFrameThread UnMatch ActionFrame Size : %lu\n", outSize);
                    NN_LOG("                                              From MacAddress  : %s\n", fromMac.GetString(clientMacArray));
                    i--;
                    continue;
                }
                WLANTEST_EXPECT_TRUE(isMatchFramType);
            }
        }

        static void UnSyncScanThread(void* arg) NN_NOEXCEPT
        {
            NN_ABORT_UNLESS_NOT_NULL(arg);
            LocalClientFuncTest* pThis = static_cast<LocalClientFuncTest*>(arg);

            std::unique_ptr<uint8_t[]> pTestBuffer(new uint8_t[TestScanMaxSize]);
            NN_ABORT_UNLESS_NOT_NULL(pTestBuffer.get());
            nn::wlan::ScanIeMatchInfo info;

            NN_LOG("             Client UnSyncScanThread -> StartScan Start\n");
            while (pThis->isStartScan == true)
            {
                nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(pThis->intervalScanTime));
                WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::StartScan(pTestBuffer.get(), TestScanMaxSize, pThis->unsyncScanParam, &info));
            }
            NN_LOG("             Client UnSyncScanThread -> StartScan End\n\n");
        }

#ifdef WLAN_TEST_LINKLEVEL_ERROR
        static void TestApiCallTimeThread(void* arg) NN_NOEXCEPT
        {
            NN_ABORT_UNLESS_NOT_NULL(arg);
            LocalClientFuncTest* pThis = static_cast<LocalClientFuncTest*>(arg);
            nn::os::Tick logTick;
            bool isLogOut = false;

            logTick = nn::os::GetSystemTick();
            while (pThis->isCheck)
            {
                if ((nn::os::GetSystemTick() - pThis->connectStsApiStart).ToTimeSpan().GetMilliSeconds() >= Time120s)
                {
                    if (pThis->isConnectStsCheck && (nn::os::GetSystemTick() - logTick).ToTimeSpan().GetMilliSeconds() >= Time60s)
                    {
                        NN_LOG("             GetConnectionStatus Api Call Time Over(%lld)\n", (nn::os::GetSystemTick() - pThis->connectStsApiStart).ToTimeSpan().GetMilliSeconds());
                        isLogOut = true;
                    }
                }

                if ((nn::os::GetSystemTick() - pThis->linkApiStart).ToTimeSpan().GetMilliSeconds() >= Time120s)
                {
                    if (pThis->isLinkApiCheck && (nn::os::GetSystemTick() - logTick).ToTimeSpan().GetMilliSeconds() >= Time60s)
                    {
                        NN_LOG("             GetLinkLevel Api Call Time Over(%lld)\n", (nn::os::GetSystemTick() - pThis->linkApiStart).ToTimeSpan().GetMilliSeconds());
                        isLogOut = true;
                    }
                }

                if ((nn::os::GetSystemTick() - pThis->connectLogApiStart).ToTimeSpan().GetMilliSeconds() >= Time600s * 2)
                {
                    if (pThis->isConnectLogCheck && (nn::os::GetSystemTick() - logTick).ToTimeSpan().GetMilliSeconds() >= Time60s)
                    {
                        NN_LOG("             connection Log Api Call Time Over(%lld)\n", (nn::os::GetSystemTick() - pThis->connectLogApiStart).ToTimeSpan().GetMilliSeconds());
                        isLogOut = true;
                    }
                }

                if (isLogOut)
                {
                    logTick = nn::os::GetSystemTick();
                    isLogOut = false;
                }
                nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(Time1s));
            }

        }
#endif

        void Connect(TestLocalConnectParams ConnectParams) NN_NOEXCEPT
        {
            TraceState(&localClient);
            while (NN_STATIC_CONDITION(1))
            {
                WLANTEST_EXPECT_RESULT_SUCCESS(nn::wlan::Local::Connect(ConnectParams.ssid,
                                                                        ConnectParams.bssid,
                                                                        ConnectParams.channel,
                                                                        ConnectParams.security,
                                                                        ConnectParams.autoKeepAlive,
                                                                        ConnectParams.indication,
                                                                        ConnectParams.beaconLostTimeout));

                NN_LOG("             WiFi Connect Start\n");
                nn::os::WaitSystemEvent(&connectionEvent);
                NN_LOG("             WiFi WaitSystemEvent Signal\n");

                WLANTEST_EXPECT_RESULT_SUCCESS(nn::wlan::Local::GetConnectionStatus(&connectionStatus));
                if( connectionStatus.state == nn::wlan::ConnectionState_Connected )
                {
                    NN_LOG("             WlanTest: Connected\n");
                    break;
                }

                NN_LOG("             WiFi ConnectionStatus(%d)\n", connectionStatus.state);
            }
            TraceState(&localClient);
        }

        void Disconnect() NN_NOEXCEPT
        {
            TraceState(&localClient);

            WLANTEST_EXPECT_RESULT_SUCCESS(nn::wlan::Local::GetConnectionStatus(&connectionStatus));
            NN_LOG("             GetConnectionStatus(%d)\n", connectionStatus.state);
            if (connectionStatus.state == nn::wlan::ConnectionState_Connected)
            {
                NN_LOG("             Disconnect Start\n");
                WLANTEST_EXPECT_RESULT_SUCCESS(nn::wlan::Local::Disconnect(nn::wlan::LocalCommunicationMode_ClientSpectator, nullptr));
                NN_LOG("             Disconnect End\n");
            }

            TraceState(&localClient);
            nn::os::Tick disTimeout = nn::os::GetSystemTick();
            while (NN_STATIC_CONDITION(1))
            {
                WLANTEST_ASSERT_TRUE((nn::os::GetSystemTick() - disTimeout).ToTimeSpan().GetMilliSeconds() < WlanTestDisconnectTimeOut);

                nn::os::TimedWaitSystemEvent(&connectionEvent, nn::TimeSpan::FromMilliSeconds(PutInterval));
                WLANTEST_EXPECT_RESULT_SUCCESS(nn::wlan::Local::GetConnectionStatus(&connectionStatus));
                if (connectionStatus.state != nn::wlan::ConnectionState_Connected)
                {
                    NN_LOG("             DIS CONNECTED!!!\n");
                    break;
                }
            }

            TraceState(&localClient);
        }
    };

    class LocalMasterClientFuncTest : public LocalMasterFuncTest
    {
    protected:
        enum OpenModeType
        {
            OpenMode_Init = 0,
            OpenMode_Master,
            OpenMode_Client,
            OpenMode_Spectator,
        };

        LocalApiClass localClient;
        nn::wlan::ConnectionStatus connectionStatus;
        nn::wlan::ConnectionStatus clientStatus;

        uint32_t clientCount;
        uint32_t myNumber;
        uint32_t nextNumber;
        uint32_t lastCount;
        uint32_t testPos;

        nn::os::Tick sysTick;

    protected:
        LocalMasterClientFuncTest() NN_NOEXCEPT
        :   clientCount (0),
            myNumber    (0),
            testPos     (0)
        {
            nextNumber = 1;

            for (int32_t i = 0; i < (sizeof(TestMasterClient) / sizeof(TestMasterClient[0])); i++)
            {
                TestMasterClient[i].seqNumber = i + 1;
                nn::util::SNPrintf(TestMasterClient[i].ssidArray, sizeof(TestMasterClient[i].ssidArray), "WlanTest_Local_%d", (i + 1));
            }
        }

        virtual void SetUp() NN_NOEXCEPT NN_OVERRIDE
        {
            LocalMasterFuncTest::SetUp();
        }

        virtual void TearDown() NN_NOEXCEPT NN_OVERRIDE
        {
            localClient.LocalRelease();
            LocalMasterFuncTest::TearDown();
        }

        uint32_t GetSeqNumber() NN_NOEXCEPT
        {
            uint32_t seqNum = 0;
            nn::wlan::MacAddress myMacAddress;
            nn::Result result;

            if (ChangeMode(OpenModeType::OpenMode_Master) == 0)
            {
                NN_LOG("             ChangeMode Error\n");
                return seqNum;
            }

            result = nn::wlan::Local::GetMacAddress(&myMacAddress);
            if (result.IsSuccess() != true)
            {
                NN_LOG("             GetMacAddress Error\n");
                return seqNum;
            }

            for (auto info : TestMasterClient)
            {
                nn::wlan::MacAddress compareMacAddress(info.sdevMacAddress);
                if (myMacAddress == compareMacAddress)
                {
                    seqNum = info.seqNumber;
                    break;
                }
            }

            return seqNum;
        }

        uint32_t GetFirstSeqNumber() NN_NOEXCEPT
        {
            testPos = 0;
            nextNumber = TestMasterClient[testPos].seqNumber;

            return nextNumber;
        }

        uint32_t GetNextSeqNumber()
        {
            ++testPos %= lastCount;
            nextNumber = TestMasterClient[testPos].seqNumber;

            return nextNumber;
        }

        bool isMasterSeqNumber() NN_NOEXCEPT
        {
            uint32_t myNum = GetSeqNumber();

            return (myNum == nextNumber);
        }

        nn::wlan::Ssid GetMasterSsid()
        {
            return nn::wlan::Ssid(TestMasterClient[testPos].ssidArray);
        }

        void StartClientConnect()
        {
            WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::OpenClientMode());

            WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::CloseClientMode());
        }

        OpenModeType GetMode() NN_NOEXCEPT
        {
            nn::wlan::WlanState state;
            nn::Result result;

            result = localClient.GetState(&state);
            if (result.IsSuccess() != true)
            {
                return OpenModeType::OpenMode_Init;
            }

            if (nn::wlan::WlanState_LocalMasterIdle <= state && state < nn::wlan::WlanState_LocalClientIdle)
            {
                return OpenModeType::OpenMode_Master;
            }

            if (nn::wlan::WlanState_LocalClientIdle <= state && state < nn::wlan::WlanState_LocalSpectatorIdle)
            {
                return OpenModeType::OpenMode_Client;
            }

            if (nn::wlan::WlanState_LocalSpectatorIdle <= state && state < nn::wlan::WlanState_Exit)
            {
                return OpenModeType::OpenMode_Spectator;
            }

            return OpenModeType::OpenMode_Init;
        }

        bool ChangeMode(const OpenModeType mode) NN_NOEXCEPT
        {
            nn::Result result = nn::ResultSuccess();
            bool isModeOpen = false;

            if (CheckOpenMode(mode) != true)
            {
                isModeOpen = CloseMode();
                if (isModeOpen != true)
                {
                    return isModeOpen;
                }

                switch (mode)
                {
                case OpenModeType::OpenMode_Master:
                    result = nn::wlan::Local::OpenMasterMode();
                    NN_LOG("             Open Mode Master : 0x%08x\n", result.GetInnerValueForDebug());
                    break;
                case OpenModeType::OpenMode_Client:
                    result = nn::wlan::Local::OpenClientMode();
                    NN_LOG("             Open Mode Client : 0x%08x\n", result.GetInnerValueForDebug());
                    break;
                case OpenModeType::OpenMode_Spectator:
                    result = nn::wlan::Local::OpenSpectatorMode();
                    NN_LOG("             Open Mode Spectator : 0x%08x\n", result.GetInnerValueForDebug());
                    break;
                default:
                    return false;
                }

                if (result.IsSuccess() != true)
                {
                    return false;
                }
            }

            return true;
        }

        bool CheckOpenMode(const OpenModeType mode) NN_NOEXCEPT
        {
            if (GetMode() != mode)
            {
                return false;
            }

            return true;
        }

        bool CloseMode() NN_NOEXCEPT
        {
            nn::Result result = nn::ResultSuccess();
            switch (GetMode())
            {
            case OpenModeType::OpenMode_Master:
                result = nn::wlan::Local::CloseMasterMode();
                NN_LOG("             Close Master : 0x%08x\n", result.GetInnerValueForDebug());
                break;
            case OpenModeType::OpenMode_Client:
                result = nn::wlan::Local::CloseClientMode();
                NN_LOG("             Close Client : 0x%08x\n", result.GetInnerValueForDebug());
                break;
            case OpenModeType::OpenMode_Spectator:
                result = nn::wlan::Local::CloseSpectatorMode();
                NN_LOG("             Close Spectator : 0x%08x\n", result.GetInnerValueForDebug());
                break;
            default:
                break;
            }

            if (result.IsSuccess() != true)
            {
                return false;
            }

            return true;
        }

        void Connect(TestLocalConnectParams ConnectParams, uint32_t waitTime) NN_NOEXCEPT
        {
            TraceState(&localMaster);

            nn::wlan::ConnectionStatus connectionStatus;

            sysTick = nn::os::GetSystemTick();

            // 指定時間接続リトライ
            while ((nn::os::GetSystemTick() - sysTick).ToTimeSpan().GetMilliSeconds() <= waitTime)
            {
                WLANTEST_EXPECT_RESULT_SUCCESS(nn::wlan::Local::Connect(ConnectParams.ssid,
                    ConnectParams.bssid,
                    ConnectParams.channel,
                    ConnectParams.security,
                    ConnectParams.autoKeepAlive,
                    ConnectParams.indication,
                    ConnectParams.beaconLostTimeout));

                if (nn::os::TimedWaitSystemEvent(&connectionEvent, nn::TimeSpan::FromMilliSeconds(WlanTestConnectTimeOut)) != true)
                {
                    continue;
                }
                NN_LOG("             WiFi WaitSystemEvent Signal\n");

                WLANTEST_EXPECT_RESULT_SUCCESS(nn::wlan::Local::GetConnectionStatus(&connectionStatus));
                NN_LOG("             WiFi ConnectionStatus(%d)\n", connectionStatus.state);
                if (connectionStatus.state == nn::wlan::ConnectionState_Connected)
                {
                    NN_LOG("             WlanTest: Connected\n");
                    break;
                }

                // 3秒待って再試行
                nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(Time3s));
            }

            TraceState(&localMaster);
        }

        void Disconnect() NN_NOEXCEPT
        {
            TraceState(&localMaster);

            WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::GetConnectionStatus(&connectionStatus));
            NN_LOG("             GetConnectionStatus(%d)\n", connectionStatus.state);
            if (connectionStatus.state == nn::wlan::ConnectionState_Connected)
            {
                NN_LOG("             Disconnect Start\n");
                WLANTEST_EXPECT_RESULT_SUCCESS(nn::wlan::Local::Disconnect(nn::wlan::LocalCommunicationMode_ClientSpectator, nullptr));
                NN_LOG("             Disconnect End\n");
            }

            TraceState(&localMaster);
            while (NN_STATIC_CONDITION(1))
            {
                nn::os::TimedWaitSystemEvent(&connectionEvent, nn::TimeSpan::FromMilliSeconds(PutInterval));
                WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::GetConnectionStatus(&connectionStatus));
                if (connectionStatus.state != nn::wlan::ConnectionState_Connected)
                {
                    NN_LOG("             DIS CONNECTED!!!\n");
                    break;
                }
            }
            TraceState(&localMaster);
        }

        void MasterSetup(int32_t clientMacListCount) NN_NOEXCEPT
        {
            std::unique_ptr<uint8_t[]> pTestBuffer(new uint8_t[LocalApiFramePacketSize]);
            TcpMacAddressPacketFmt* pPacketData = reinterpret_cast<TcpMacAddressPacketFmt*>(pTestBuffer.get());
            int32_t offsetPos;
            WLANTEST_ASSERT_TRUE(pTestBuffer.get() != nullptr);

            const nn::wlan::MasterBssParameters fncTestMasterBss = {
                11, false, 30, true, nn::wlan::RateSetLegacy_11bMask, nn::wlan::RateSetLegacy_11bMask,
                { nn::wlan::SecurityMode_StaticAes, nn::wlan::SecurityMode_StaticAes, 0, LocalMasterAes16Key }, nn::wlan::Ssid("LocalGetTcp"), 100
            };

            WLANTEST_ASSERT_TRUE(ChangeMode(OpenModeType::OpenMode_Master));

            WLANTEST_ASSERT_TRUE(localMaster.SetupLocalMaster(fncTestMasterBss, TestMatchInfo, sizeof(TestMatchInfo) / sizeof(TestMatchInfo[0])));
            WLANTEST_ASSERT_TRUE(localMaster.SetupTcpMaster(tcpMaster, clientMacListCount));

            // MASTER CLIENT接続待機
            WLANTEST_ASSERT_TRUE(localMaster.LocalMasterConnectWait(localMaster.m_clientMacArray, WlanTestConnectTimeOut, clientMacListCount) > 0);
            WLANTEST_ASSERT_TRUE(localMaster.TcpMasterAcceptWait(Time1h));

            offsetPos = 0;
            lastCount = clientMacListCount + 1;
            pPacketData->len = 1;
            std::memcpy(pPacketData->macAddress[offsetPos], localMaster.m_macAddress.GetOuiData(), sizeof(pPacketData->macAddress[offsetPos]));
            std::memcpy(TestMasterClient[offsetPos].sdevMacAddress, pPacketData->macAddress[offsetPos], sizeof(TestMasterClient[offsetPos].sdevMacAddress));
            offsetPos++;

            for (int32_t i = 0; i < clientMacListCount; i++)
            {
                char clientMacArray[nn::wlan::MacAddress::MacStringSize];
                pPacketData->len++;
                std::memcpy(pPacketData->macAddress[offsetPos], localMaster.m_clientMacArray[i].GetOuiData(), sizeof(pPacketData->macAddress[offsetPos]));
                std::memcpy(TestMasterClient[offsetPos].sdevMacAddress, pPacketData->macAddress[offsetPos], sizeof(TestMasterClient[offsetPos].sdevMacAddress));
                NN_LOG("             LocalApi MasterSetup No%d.MacAddress : %s\n", TestMasterClient[offsetPos].seqNumber, nn::wlan::MacAddress(TestMasterClient[offsetPos].sdevMacAddress).GetString(clientMacArray));
                offsetPos++;
            }

            localMaster.TcpMasterSend(pTestBuffer.get(), LocalApiFramePacketSize);

            localMaster.LocalRelease();
        }

        void ClientSetup() NN_NOEXCEPT
        {
            std::unique_ptr<uint8_t[]> pTestBuffer(new uint8_t[LocalApiFramePacketSize]);
            TcpMacAddressPacketFmt* pPacketData = reinterpret_cast<TcpMacAddressPacketFmt*>(pTestBuffer.get());

            IpV4Address tcpClient = {
                "0.0.0.0", "255.255.255.0", "192.168.11.1",
                "8.8.8.8", "8.8.8.9", 0, false
            };

            nn::util::SNPrintf(tcpClient.strIpAddress, sizeof(tcpClient.strIpAddress), "192.168.11.%d", g_IpOct4);
            WLANTEST_ASSERT_TRUE(pTestBuffer.get() != nullptr);

            LocalClientConnectParameter ConnectParam = {
                nn::wlan::Ssid("LocalGetTcp"), nn::wlan::MacAddress::CreateBroadcastMacAddress(), 11,
                { nn::wlan::SecurityMode_StaticAes, nn::wlan::SecurityMode_StaticAes, 0, LocalMasterAes16Key }, true, nn::wlan::BeaconIndication_Enable, 10
            };

            WLANTEST_ASSERT_TRUE(ChangeMode(OpenModeType::OpenMode_Client));

            WLANTEST_ASSERT_TRUE(localClient.SetupLocalClient(ConnectParam.ssid, ConnectParam.security, TestMatchInfo, sizeof(TestMatchInfo) / sizeof(TestMatchInfo[0]), true, false));
            WLANTEST_ASSERT_TRUE(localClient.LocalClientConnect(ConnectParam, static_cast<int64_t>(Time60s)));

            WLANTEST_ASSERT_TRUE(localClient.SetupTcpClient(tcpMaster, tcpClient));
            WLANTEST_ASSERT_TRUE(localClient.TcpClientConnectAndRecv(pTestBuffer.get(), LocalApiFramePacketSize, Time1h));

            lastCount = pPacketData->len;
            for (int32_t i = 0; i < pPacketData->len; i++)
            {
                char clientMacArray[nn::wlan::MacAddress::MacStringSize];
                std::memcpy(TestMasterClient[i].sdevMacAddress, pPacketData->macAddress[i], sizeof(TestMasterClient[i].sdevMacAddress));
                NN_LOG("             LocalApi ClientSetup No%d.MacAddress : %s\n", TestMasterClient[i].seqNumber, nn::wlan::MacAddress(TestMasterClient[i].sdevMacAddress).GetString(clientMacArray));
            }

            WLANTEST_ASSERT_TRUE(localClient.LocalClientDisconnect(Time60s));

            localClient.LocalRelease();
        }

        void MasterFrameTest(bool& isStart,nn::os::Tick& outTime, size_t bufferSize, int64_t agingTime) NN_NOEXCEPT
        {
            WLANTEST_ASSERT_TRUE(bufferSize > 0);

            std::unique_ptr<uint8_t[]> putBuffer(new uint8_t[bufferSize]);
            const int32_t TestConnectCount  = lastCount;
            const int32_t TestClientCount   = TestConnectCount - 1;
            const int32_t TestPutFrameCount = Time30s / Time5s;
            char macArray[nn::wlan::MacAddress::MacStringSize];
            LocalConnectsInfo conectsInfo;
            uint32_t seqNumber = 0;
            bool isSygnalWait = false;
            nn::Result result;

            WLANTEST_ASSERT_TRUE(putBuffer.get() != nullptr);

#ifdef WLAN_TEST_SENDRECVDATA_TRACE
            char fromMac[nn::wlan::MacAddress::MacStringSize];
            char toMac[nn::wlan::MacAddress::MacStringSize];
#endif

            WLANTEST_ASSERT_TRUE(ChangeMode(OpenModeType::OpenMode_Master));

            // Master起動
            nn::wlan::MasterBssParameters testChangeMasterBss = {
                11, false, 30, true, nn::wlan::RateSetLegacy_11gMask, nn::wlan::RateSetLegacy_11gMask,
                { nn::wlan::SecurityMode_Open, nn::wlan::SecurityMode_Open, 0, "" }, GetMasterSsid(), 100
            };

            NN_LOG("             LocalMaster SSID : %s\n", GetMasterSsid().GetSsidData());

            WLANTEST_ASSERT_TRUE(localMaster.SetupLocalMaster(testChangeMasterBss, TestMatchInfo, sizeof(TestMatchInfo) / sizeof(TestMatchInfo[0])));
            NN_LOG("             LocalMaster Load(%s)\n", localMaster.m_macAddress.GetString(macArray));

            // MASTER受信スレッド起動
            WLANTEST_EXPECT_TRUE(localMaster.LocalReceiveFrameStart(BufferSize100k, bufferSize, TestClientCount, TestClientCount, TestPutFrameCount));
            isSygnalWait = localMaster.LocalReceiveWait(Time600s, true);
            if (isStart != true)
            {
                outTime = nn::os::GetSystemTick();
                localMaster.LocalSetElapseTime();
                isStart = true;
            }

            // エージング時間が経過していたら終了条件。以外はローカルクライアント台数が揃ってのシグナルか判定
            if ((nn::os::GetSystemTick() - outTime).ToTimeSpan().GetMilliSeconds() > agingTime)
            {
                isSygnalWait = false;
            }
            else
            {
                WLANTEST_EXPECT_TRUE((isSygnalWait == true));
            }

            seqNumber = 0;
            while (isSygnalWait == true)
            {
                WLANTEST_EXPECT_TRUE(localMaster.LocalMakeFrame(putBuffer.get(), bufferSize, nn::wlan::MacAddress::CreateBroadcastMacAddress(), localMaster.m_macAddress, EtherTypes[0], TestMatchInfo[0], seqNumber));
                result = nn::wlan::Local::PutFrameRaw(putBuffer.get(), bufferSize);

#ifdef WLAN_TEST_SENDRECVDATA_TRACE
                NN_LOG("            LocalFunctionTest L115 Master PutFrameRaw From MacAddress[%s] To MacAddress[%s] : Seq Number : %lu\n",
                    localMaster.m_macAddress.GetString(fromMac), localMaster.m_clientMacArray[i].GetString(toMac), seqNumber);
#endif
                if (nn::os::TimedWaitSystemEvent(&localMaster.m_connectEvent, nn::TimeSpan::FromMilliSeconds(Time5s)) == true)
                {
                    if (localMaster.GetStateClientCount(nn::wlan::ConnectionState_Connected) == 0)
                    {
                        break;
                    }
                    nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(PutInterval));
                }

                // ブロードキャスト送信でエラーとなった場合イベントを見てから判定に変更
                WLANTEST_EXPECT_RESULT_SUCCESS(result);
                seqNumber++;
            }

            localMaster.LocalTraceConnectsInfo(true);

            // MASTER受信スレッド停止
            localMaster.LocalReceiveFrameStop();

            // 通信結果ログ出力
            localMaster.LocalTraceConnectsInfo(true);

            localMaster.LocalGetConnectInfo(&conectsInfo);
            WLANTEST_EXPECT_TRUE(conectsInfo.connectCount >= TestClientCount);

            localMaster.LocalRelease();
        }

        void ClientFrameTest(bool& isStart, nn::os::Tick& outTime, size_t bufferSize, int64_t agingTime) NN_NOEXCEPT
        {
            char macArray[nn::wlan::MacAddress::MacStringSize];
            nn::wlan::MacAddress myMacAddress;
            LocalConnectsInfo conectsInfo;

            WLANTEST_ASSERT_TRUE(ChangeMode(OpenModeType::OpenMode_Client));

            TestLocalConnectParams clientChangeConnectParam = {
                GetMasterSsid(), nn::wlan::MacAddress::CreateBroadcastMacAddress(), 11,
                { nn::wlan::SecurityMode_Open, nn::wlan::SecurityMode_Open, 0, "" }, true, nn::wlan::BeaconIndication_Enable, 10
            };

            NN_LOG("             LocalClient SSID : %s\n", GetMasterSsid().GetSsidData());
            WLANTEST_ASSERT_TRUE(localClient.SetupLocalClient(clientChangeConnectParam.ssid, clientChangeConnectParam.security, TestMatchInfo, sizeof(TestMatchInfo) / sizeof(TestMatchInfo[0]), true, false));
            Connect(clientChangeConnectParam, WlanTestMasterWaitTimeOut);

            WLANTEST_EXPECT_RESULT_SUCCESS(nn::wlan::Local::GetMacAddress(&myMacAddress));
            NN_LOG("             LocalClient Load(%s)\n", myMacAddress.GetString(macArray));
            WLANTEST_EXPECT_RESULT_SUCCESS(nn::wlan::Local::GetConnectionStatus(&connectionStatus));

            // 受信スレッド起動
            NN_LOG("             LocalClient Receive Frame\n");
            WLANTEST_EXPECT_TRUE(localClient.LocalReceiveFrameStart(BufferSize100k, bufferSize, 1, 1, WlanTestGetFrameCount));
            // MASTER接続待機
            NN_LOG("             LocalClient Master Start Frame Wait\n");
            bool isRcvStart = localClient.LocalReceiveWait(Time600s, Time1s, true);

            if (isStart != true)
            {
                outTime = nn::os::GetSystemTick();
                localClient.LocalSetElapseTime();
                isStart = true;
            }

            // エージング時間が経過していたら終了条件。以外はローカルクライアント台数が揃ってのシグナルか判定
            if ((nn::os::GetSystemTick() - outTime).ToTimeSpan().GetMilliSeconds() > agingTime)
            {
                isRcvStart = false;
            }
            else
            {
                WLANTEST_EXPECT_TRUE((isRcvStart == true));
            }

            // ローカルクライアント台数が揃わないまたは、エージング時間が経過していたら終了
            if (isRcvStart != true)
            {
                // 受信スレッド停止
                localClient.LocalReceiveFrameStop();
                Disconnect();
                localClient.LocalRelease();
                return;
            }

            // 送信スレッド起動
            NN_LOG("             LocalClient Send Frame\n");
            WLANTEST_EXPECT_TRUE(localClient.LocalSendFrameStart(bufferSize, Time30s, Time5s, true));

            nn::os::Tick sendTick = nn::os::GetSystemTick();
            while ((nn::os::GetSystemTick() - sendTick).ToTimeSpan().GetMilliSeconds() <= Time30s)
            {
                // MASTERからの切断イベント待機
                if (nn::os::TimedWaitSystemEvent(&connectionEvent, nn::TimeSpan::FromMilliSeconds(PutInterval)) == true)
                {
                    WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::GetConnectionStatus(&clientStatus));
                    if (clientStatus.state != nn::wlan::ConnectionState_Connected)
                    {
                        break;
                    }
                }
            }

            // 送信スレッド停止
            localClient.LocalSendFrameStop();
            // 受信スレッド停止
            localClient.LocalReceiveFrameStop();
            // 通信結果ログ出力
            localClient.LocalTraceConnectsInfo(true);

            localClient.LocalGetConnectInfo(&conectsInfo);
            for (int32_t i = 0; i < conectsInfo.connectCount; i++)
            {
                WLANTEST_EXPECT_TRUE(conectsInfo.facing[i].isConnected);
                WLANTEST_EXPECT_TRUE(conectsInfo.facing[i].getFrameCount >= 1);
            }

            Disconnect();
            localClient.LocalRelease();
        }

        void MaseterAgingFrameTest(bool& isStart, size_t recvSize, size_t sendSize, int64_t sendInterval, int64_t connectTime) NN_NOEXCEPT
        {
            char macArray[nn::wlan::MacAddress::MacStringSize];
            LocalConnectsInfo conectsInfo;
            std::unique_ptr<uint8_t[]> putBuffer(new uint8_t[sendSize]);
            nn::os::Tick loopTick, sendTick;
            const int32_t TestConnectCount  = lastCount;
            const int32_t TestClientCount   = TestConnectCount - 1;
            uint32_t seqNumber              = 0;

            WLANTEST_ASSERT_TRUE(putBuffer.get() != nullptr);

            WLANTEST_ASSERT_TRUE(ChangeMode(OpenModeType::OpenMode_Master));

            // Master起動
            nn::wlan::MasterBssParameters testChangeMasterBss = {
                6, false, 30, true, nn::wlan::RateSetLegacy_11gMask, nn::wlan::RateSetLegacy_11gMask,
                { nn::wlan::SecurityMode_Open, nn::wlan::SecurityMode_Open, 0, "" }, GetMasterSsid(), 100
            };

            NN_LOG("             LocalMaster SSID : %s\n", GetMasterSsid().GetSsidData());

            WLANTEST_ASSERT_TRUE(localMaster.SetupLocalMaster(testChangeMasterBss, TestMatchInfo, sizeof(TestMatchInfo) / sizeof(TestMatchInfo[0])));
            NN_LOG("             LocalMaster Load(%s)\n", localMaster.m_macAddress.GetString(macArray));

            // MASTER受信スレッド起動
            localMaster.LocalReceiveFrameStart(BufferSize100k, recvSize, TestClientCount, TestClientCount, WlanTestAgingGetFrameCount);

            localMaster.LocalReceiveWait(WlanTestMasterWaitTimeOut, true);
            if (isStart != true)
            {
                sysTick = nn::os::GetSystemTick();
                localMaster.LocalSetElapseTime();
                isStart = true;
            }

            seqNumber = 0;
            loopTick = nn::os::GetSystemTick();
            sendTick = loopTick;
            while ((nn::os::GetSystemTick() - loopTick).ToTimeSpan().GetMilliSeconds() <= connectTime)
            {
                if (((nn::os::GetSystemTick() - sendTick).ToTimeSpan().GetMilliSeconds()) > Time300msec)
                {
                    for (int32_t i = 0; i < localMaster.m_connectCount; i++)
                    {
                        WLANTEST_EXPECT_TRUE(localMaster.LocalMakeFrame(putBuffer.get(), sendSize, localMaster.m_clientMacArray[i], localMaster.m_macAddress, EtherTypes[0], TestMatchInfo[0]));
                        WLANTEST_EXPECT_TRUE(localMaster.LocalSeqNumberAddFrame(putBuffer.get(), sendSize, seqNumber));
                        WLANTEST_EXPECT_RESULT_SUCCESS(nn::wlan::Local::PutFrameRaw(putBuffer.get(), sendSize));
                    }
                    sendTick = nn::os::GetSystemTick();
                    seqNumber++;
                }
                nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(1));

                if ((nn::os::GetSystemTick() - logOutTime).ToTimeSpan().GetMilliSeconds() >= Time600s)
                {
                    localMaster.LocalTraceConnectsInfo(true);
                    logOutTime = nn::os::GetSystemTick();
                }
            }

            // MASTER受信スレッド停止
            localMaster.LocalReceiveFrameStop();

            localMaster.LocalTraceConnectsInfo(true);

            localMaster.LocalGetConnectInfo(&conectsInfo);
            WLANTEST_EXPECT_TRUE(conectsInfo.connectCount >= TestClientCount);

            localMaster.LocalRelease();
        }

        void ClientAgingFrameTest(bool& isStart, size_t recvSize, size_t sendSize, int64_t sendInterval, int64_t connectTime) NN_NOEXCEPT
        {
            char macArray[nn::wlan::MacAddress::MacStringSize];
            nn::wlan::MacAddress myMacAddress;
            LocalConnectsInfo conectsInfo;
            const int32_t TestClientCount = lastCount - 1;

            WLANTEST_ASSERT_TRUE(ChangeMode(OpenModeType::OpenMode_Client));

            TestLocalConnectParams ConnectParam = {
                GetMasterSsid(), nn::wlan::MacAddress::CreateBroadcastMacAddress(), 6,
                { nn::wlan::SecurityMode_Open, nn::wlan::SecurityMode_Open, 0, "" }, true, nn::wlan::BeaconIndication_Enable, 10
            };

            NN_LOG("             LocalClient SSID : %s\n", GetMasterSsid().GetSsidData());
            WLANTEST_ASSERT_TRUE(localClient.SetupLocalClient(ConnectParam.ssid, ConnectParam.security, TestMatchInfo, sizeof(TestMatchInfo) / sizeof(TestMatchInfo[0]), true, false));
            Connect(ConnectParam, WlanTestMasterWaitTimeOut);

            WLANTEST_EXPECT_RESULT_SUCCESS(nn::wlan::Local::GetMacAddress(&myMacAddress));
            NN_LOG("             LocalClient Load(%s)\n", myMacAddress.GetString(macArray));
            WLANTEST_EXPECT_RESULT_SUCCESS(nn::wlan::Local::GetConnectionStatus(&connectionStatus));

            // 受信スレッド起動
            WLANTEST_ASSERT_TRUE(localClient.LocalReceiveFrameStart(BufferSize100k, recvSize, 1, 1, WlanTestAgingGetFrameCount));
            // MASTER接続待機
            WLANTEST_ASSERT_TRUE(localClient.LocalReceiveWait(WlanMaxClientWaitMsecTimeOut, Time1s, true));
            // 送信スレッド起動
            WLANTEST_ASSERT_TRUE(localClient.LocalSendFrameStart(sendSize, connectTime + Time10s, Time300msec, true));

            if (isStart != true)
            {
                sysTick = nn::os::GetSystemTick();
                localClient.LocalSetElapseTime();
                isStart = true;
            }

            // MASTERからの接続が切られるまで待機
            while (NN_STATIC_CONDITION(1))
            {
                if (nn::os::TimedWaitSystemEvent(&connectionEvent, nn::TimeSpan::FromMilliSeconds(Time1s)) == true)
                {
                    WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::GetConnectionStatus(&clientStatus));
                    if (clientStatus.state != nn::wlan::ConnectionState_Connected)
                    {
                        break;
                    }
                }

                if ((nn::os::GetSystemTick() - logOutTime).ToTimeSpan().GetMilliSeconds() >= Time600s)
                {
                    localClient.LocalTraceConnectsInfo(true);
                    logOutTime = nn::os::GetSystemTick();
                }
            }

            // 送信スレッド停止
            localClient.LocalSendFrameStop();
            // 受信スレッド停止
            localClient.LocalReceiveFrameStop();
            // 通信結果ログ出力
            localClient.LocalTraceConnectsInfo(true);

            localClient.LocalGetConnectInfo(&conectsInfo);
            WLANTEST_EXPECT_TRUE(conectsInfo.connectCount == TestClientCount);

            Disconnect();
            localClient.LocalRelease();
        }

        int GetRandom(int min, int max) NN_NOEXCEPT
        {
            static time_t check;
            int rand_ret;
            if (check != time(nullptr))
            {
                check = time(nullptr);
                srand(static_cast<unsigned int>(time(nullptr)));
            }
            rand_ret = min + static_cast<int>(rand() * (max - min + 1.0) / (1.0 + RAND_MAX));
            return rand_ret;
        }

    };

    // 1,6,11chに限定したアクティブスキャン(MASTER 3台)
    TEST_F(LocalClientFuncTest, L1)
    {
        const nn::wlan::Ssid compSsids[] = { nn::wlan::Ssid(LocalMasterTestSSID), nn::wlan::Ssid(LocalMasterTest2SSID),nn::wlan::Ssid(LocalMasterTest3SSID) };
        nn::wlan::ScanIeMatchInfo info;

        const nn::wlan::ScanParameters StartScanTestParams = {
            nn::wlan::ScanType_Active,{ 1, 6, 11 }, 3, 40, -1, nullptr, 0, nn::wlan::MacAddress::CreateBroadcastMacAddress()
        };

        std::unique_ptr<uint8_t[]> pTestBuffer(new uint8_t[TestScanMaxSize]);
        WLANTEST_ASSERT_TRUE(pTestBuffer.get() != nullptr);

        WLANTEST_EXPECT_RESULT_SUCCESS(nn::wlan::Local::StartScan(pTestBuffer.get(), TestScanMaxSize, StartScanTestParams, &info));

        for (auto compSsid : compSsids)
        {
            WLANTEST_EXPECT_TRUE(CompareScanSsid(pTestBuffer.get(), TestScanMaxSize, compSsid));
        }
    }

    // 1,6,11chに限定したパッシブキャン(MASTER 3台)
    TEST_F(LocalClientFuncTest, L2)
    {
        const nn::wlan::Ssid compSsids[] = { nn::wlan::Ssid(LocalMasterTestSSID), nn::wlan::Ssid(LocalMasterTest2SSID),nn::wlan::Ssid(LocalMasterTest3SSID) };
        nn::wlan::ScanIeMatchInfo info;

        const nn::wlan::ScanParameters StartScanTestParams = {
             nn::wlan::ScanType_Passive,{ 1, 6, 11 }, 3, 110, -1, nullptr, 0, nn::wlan::MacAddress::CreateBroadcastMacAddress()
        };

        std::unique_ptr<uint8_t[]> pTestBuffer(new uint8_t[TestScanMaxSize]);
        WLANTEST_ASSERT_TRUE(pTestBuffer.get() != nullptr);

        WLANTEST_EXPECT_RESULT_SUCCESS(nn::wlan::Local::StartScan(pTestBuffer.get(), TestScanMaxSize, StartScanTestParams, &info));

        for (auto compSsid : compSsids)
        {
            WLANTEST_EXPECT_TRUE(CompareScanSsid(pTestBuffer.get(), TestScanMaxSize, compSsid));
        }
    }

    // 36,40,44,48chに限定したパッシブスキャン
    TEST_F(LocalClientFuncTest, L3)
    {
        const nn::wlan::Ssid compSsids[] = { nn::wlan::Ssid(LocalMasterTestSSID), nn::wlan::Ssid(LocalMasterTest2SSID), nn::wlan::Ssid(LocalMasterTest3SSID), nn::wlan::Ssid(LocalMasterTest4SSID) };
        nn::wlan::ScanIeMatchInfo info;

        const nn::wlan::ScanParameters StartScanTestParams = {
            nn::wlan::ScanType_Passive,{ 36, 40, 44, 48 }, 4, 110, -1, nullptr, 0, nn::wlan::MacAddress::CreateBroadcastMacAddress()
        };

        std::unique_ptr<uint8_t[]> pTestBuffer(new uint8_t[TestScanMaxSize]);
        WLANTEST_ASSERT_TRUE(pTestBuffer.get() != nullptr);

        WLANTEST_EXPECT_RESULT_SUCCESS(nn::wlan::Local::StartScan(pTestBuffer.get(), TestScanMaxSize, StartScanTestParams, &info));

        for (auto compSsid : compSsids)
        {
            WLANTEST_EXPECT_TRUE(CompareScanSsid(pTestBuffer.get(), TestScanMaxSize, compSsid));
        }
    }

    // ClientがMasterと接続後に、1chのアクティブスキャン
    // Clientは接続中のMasterの情報を知る
    TEST_F(LocalClientFuncTest, L4_L99)
    {
        nn::wlan::ScanIeMatchInfo info;
        char ssidStr[nn::wlan::Ssid::SsidHexStringLengthMax];
        char macStr[nn::wlan::MacAddress::MacStringSize];

        const TestLocalConnectParams ConnectParam = {
            nn::wlan::Ssid(LocalMasterTestSSID), nn::wlan::MacAddress::CreateBroadcastMacAddress(), 1,
                { nn::wlan::SecurityMode_Open, nn::wlan::SecurityMode_Open, 0, "" }, true, nn::wlan::BeaconIndication_Enable, 10
        };

        const nn::wlan::ScanParameters StartScanTestParams = {
            nn::wlan::ScanType_Active,{ 1 }, 1, 40, -1, nullptr, 0, nn::wlan::MacAddress::CreateBroadcastMacAddress()
        };

        std::unique_ptr<uint8_t[]> pTestBuffer(new uint8_t[TestScanMaxSize]);
        WLANTEST_ASSERT_TRUE(pTestBuffer.get() != nullptr);

        Connect(ConnectParam);

        NN_LOG("             +++ Connection Status +++\n");
        NN_LOG("             STATE           : %s\n", WlanConnectionStateStr[connectionStatus.state]);
        NN_LOG("             CAUSE           : %d\n", CauseOfInfo[connectionStatus.cause]);
        NN_LOG("             CHANNEL         : %d\n", connectionStatus.channel);
        NN_LOG("             SSID            : %s\n", connectionStatus.ssid.GetHexString(ssidStr));
        NN_LOG("             BSSID           : %s\n", connectionStatus.bssid.GetString(macStr));
        NN_LOG("             AID             : %d\n", connectionStatus.aid);
        NN_LOG("             REASON CODE     : %d\n", connectionStatus.statusReasonCode);
        NN_LOG("             CAPABILITY INFO : 0x%04X\n", connectionStatus.capabilityInfo);
        NN_LOG("             BEACON INTERVAL : %d\n", connectionStatus.beaconInterval);
        NN_LOG("             +++++++++++++++++++++++++\n");

        WLANTEST_EXPECT_RESULT_SUCCESS(nn::wlan::Local::StartScan(pTestBuffer.get(), TestScanMaxSize, StartScanTestParams, &info));
        WLANTEST_EXPECT_TRUE(CompareScanSsid(pTestBuffer.get(), TestScanMaxSize, ConnectParam.ssid));

        WLANTEST_EXPECT_RESULT_SUCCESS(nn::wlan::Local::Disconnect(nn::wlan::LocalCommunicationMode_ClientSpectator, nullptr));
    }

    // ClientがMasterと接続後に、6chのパッシブスキャン
    TEST_F(LocalClientFuncTest, L5)
    {
        nn::wlan::ScanIeMatchInfo info;

        const TestLocalConnectParams ConnectParam = {
            nn::wlan::Ssid(LocalMasterTestSSID), nn::wlan::MacAddress::CreateBroadcastMacAddress(), 6,
                { nn::wlan::SecurityMode_Open, nn::wlan::SecurityMode_Open, 0, "" }, true, nn::wlan::BeaconIndication_Enable, 10
        };

        const nn::wlan::ScanParameters StartScanTestParams = {
            nn::wlan::ScanType_Passive,{ 6 }, 1, 110, -1, &ssidList[0], 1, nn::wlan::MacAddress::CreateBroadcastMacAddress()
        };

        std::unique_ptr<uint8_t[]> pTestBuffer(new uint8_t[TestScanMaxSize]);
        WLANTEST_ASSERT_TRUE(pTestBuffer.get() != nullptr);

        Connect(ConnectParam);

        WLANTEST_EXPECT_RESULT_SUCCESS(nn::wlan::Local::StartScan(pTestBuffer.get(), TestScanMaxSize, StartScanTestParams, &info));
        WLANTEST_EXPECT_TRUE(CompareScanSsid(pTestBuffer.get(), TestScanMaxSize, ConnectParam.ssid));

        WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::Disconnect(nn::wlan::LocalCommunicationMode_ClientSpectator, nullptr));
    }

    // ClientがMasterと接続後に、1chのパッシブスキャン
    TEST_F(LocalClientFuncTest, L6)
    {
        nn::wlan::ScanIeMatchInfo info;

        const TestLocalConnectParams ConnectParam = {
            nn::wlan::Ssid(LocalMasterTestSSID), nn::wlan::MacAddress::CreateBroadcastMacAddress(), 11,
                { nn::wlan::SecurityMode_Open, nn::wlan::SecurityMode_Open, 0, "" }, true, nn::wlan::BeaconIndication_Enable, 10
        };

        const nn::wlan::ScanParameters StartScanTestParams = {
            nn::wlan::ScanType_Passive,{ 1 }, 1, 110, -1, &ssidList[0], 1, nn::wlan::MacAddress::CreateBroadcastMacAddress()
        };

        std::unique_ptr<uint8_t[]> pTestBuffer(new uint8_t[TestScanMaxSize]);
        WLANTEST_ASSERT_TRUE(pTestBuffer.get() != nullptr);

        Connect(ConnectParam);

        WLANTEST_EXPECT_RESULT_SUCCESS(nn::wlan::Local::StartScan(pTestBuffer.get(), TestScanMaxSize, StartScanTestParams, &info));

        WLANTEST_EXPECT_TRUE(CompareScanSsid(pTestBuffer.get(), TestScanMaxSize, ConnectParam.ssid));

        WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::Disconnect(nn::wlan::LocalCommunicationMode_ClientSpectator, nullptr));
    }

    // ClientがMasterと接続後に、1,6,11chの一括パッシブスキャン
    TEST_F(LocalClientFuncTest, L7)
    {
        nn::wlan::MacAddress macAddress = nn::wlan::MacAddress::CreateBroadcastMacAddress();
        nn::wlan::MacAddress dummyMacAddress;
        nn::wlan::ScanIeMatchInfo info;

        const TestLocalConnectParams ConnectParam = {
            nn::wlan::Ssid(LocalMasterTestSSID), nn::wlan::MacAddress::CreateBroadcastMacAddress(), 1,
            { nn::wlan::SecurityMode_Open, nn::wlan::SecurityMode_Open, 0, "" }, true, nn::wlan::BeaconIndication_Enable, 10
        };

        const nn::wlan::ScanParameters StartScanTestParams = {
            nn::wlan::ScanType_Passive,{1, 6, 11 }, 3, 110, -1, &ssidList[0], 1, nn::wlan::MacAddress::CreateBroadcastMacAddress()
        };

        // ジョイコン1セット（各2台）
        WLANTEST_ASSERT_TRUE(localClient.SetBtMode(LocalApiBtNode2));

        std::unique_ptr<uint8_t[]> pTestBuffer(new uint8_t[TestScanMaxSize]);
        WLANTEST_ASSERT_TRUE(pTestBuffer.get() != nullptr);

        Connect(ConnectParam);

        WLANTEST_EXPECT_RESULT_SUCCESS(nn::wlan::Local::StartScan(pTestBuffer.get(), TestScanMaxSize, StartScanTestParams, &info));

        WLANTEST_EXPECT_TRUE(CompareScanSsid(pTestBuffer.get(), TestScanMaxSize, ConnectParam.ssid));

        WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::Disconnect(nn::wlan::LocalCommunicationMode_ClientSpectator, nullptr));
    }

    // Master起動後に、11chのアクティブスキャン CLIENT不要
    TEST_F(LocalMasterFuncTest, L8)
    {
        nn::wlan::ScanIeMatchInfo info;
        std::unique_ptr<uint8_t[]> pTestBuffer(new uint8_t[TestScanMaxSize]);
        WLANTEST_ASSERT_TRUE(pTestBuffer.get() != nullptr);

        const nn::wlan::MasterBssParameters fncTestMasterBss = {
            11, false, 30, true, nn::wlan::RateSetLegacy_11gMask, nn::wlan::RateSetLegacy_11gMask,
            { nn::wlan::SecurityMode_Open, nn::wlan::SecurityMode_Open, 0, "" }, nn::wlan::Ssid(LocalMasterTestSSID), 100
        };

        const nn::wlan::ScanParameters StartScanTestParams = {
            nn::wlan::ScanType_Active,{ 11 }, 1, 40, -1, nullptr, 0, nn::wlan::MacAddress::CreateBroadcastMacAddress()
        };

        WLANTEST_ASSERT_TRUE(localMaster.SetupLocalMaster(fncTestMasterBss, TestMatchInfo, sizeof(TestMatchInfo) / sizeof(TestMatchInfo[0]), false));

        WLANTEST_EXPECT_RESULT_SUCCESS(nn::wlan::Local::StartScan(pTestBuffer.get(), TestScanMaxSize, StartScanTestParams, &info));

        WLANTEST_EXPECT_FALSE(CompareScanSsid(pTestBuffer.get(), TestScanMaxSize, nn::wlan::Ssid(LocalMasterTestSSID)));
    }

    // MASTER起動後に1ch/6ch/11chのパッシブスキャン CLIENT不要
    TEST_F(LocalMasterFuncTest, L9)
    {
        nn::wlan::ScanIeMatchInfo info;
        std::unique_ptr<uint8_t[]> pTestBuffer(new uint8_t[TestScanMaxSize]);
        WLANTEST_ASSERT_TRUE(pTestBuffer.get() != nullptr);

        const nn::wlan::MasterBssParameters fncTestMasterBss = {
            1, false, 30, true, nn::wlan::RateSetLegacy_11gMask, nn::wlan::RateSetLegacy_11gMask,
            { nn::wlan::SecurityMode_Open, nn::wlan::SecurityMode_Open, 0, "" }, nn::wlan::Ssid(LocalMasterTestSSID), 100
        };

        const nn::wlan::ScanParameters StartScanTestParams = {
            nn::wlan::ScanType_Passive,{ 1, 6, 11 }, 3, 110, -1, nullptr, 0, nn::wlan::MacAddress::CreateBroadcastMacAddress()
        };

        WLANTEST_ASSERT_TRUE(localMaster.SetupLocalMaster(fncTestMasterBss, TestMatchInfo, sizeof(TestMatchInfo) / sizeof(TestMatchInfo[0]), false));

        WLANTEST_EXPECT_RESULT_SUCCESS(nn::wlan::Local::StartScan(pTestBuffer.get(), TestScanMaxSize, StartScanTestParams, &info));

        WLANTEST_EXPECT_FALSE(CompareScanSsid(pTestBuffer.get(), TestScanMaxSize, nn::wlan::Ssid(LocalMasterTestSSID)));
    }

    // MasterにClientが7台接続後に、1,6,11chのスキャン MASTER
    TEST_F(LocalMasterFuncTest, L10_1)
    {
        nn::wlan::MacAddress clientMacArray[ClientMaxConnect];
        const nn::wlan::ScanParameters StartScanTestParams = {
            nn::wlan::ScanType_Active,{ 1, 6, 3 }, 3, 40, -1, nullptr, 0, nn::wlan::MacAddress::CreateBroadcastMacAddress()
        };

        const nn::wlan::MasterBssParameters fncTestMasterBss = {
            6, false, 30, true, nn::wlan::RateSetLegacy_11gMask, nn::wlan::RateSetLegacy_11gMask,
            { nn::wlan::SecurityMode_StaticAes, nn::wlan::SecurityMode_StaticAes, 0, LocalMasterAes16Key }, nn::wlan::Ssid(LocalMasterTestSSID), 100
        };

        WLANTEST_ASSERT_TRUE(localMaster.SetupLocalMaster(fncTestMasterBss, TestMatchInfo, sizeof(TestMatchInfo) / sizeof(TestMatchInfo[0])));

        localMaster.LocalMasterConnectWait(clientMacArray, WlanTestMasterWaitTimeOut, ClientMaxConnect);

        StartUnsyncScanThread(StartScanTestParams, 1000);

        // 60秒間通信状態確認
        sysTick = nn::os::GetSystemTick();
        while ((nn::os::GetSystemTick() - sysTick).ToTimeSpan().GetMilliSeconds() > (Time60s + Time5s))
        {
            if (nn::os::TimedWaitSystemEvent(&localMaster.m_connectEvent, nn::TimeSpan::FromMilliSeconds(PutInterval)) == true)
            {
                localMaster.LocalGetConnectInfo(&connectsInfo);

                WLANTEST_ASSERT_TRUE(connectsInfo.connectCount >= ClientMaxConnect);
            }

        }

        StopUnsyncScanThread();
    }

    // MasterにClientが7台接続後に、1,6,11chのスキャン CLIENT
    TEST_F(LocalClientFuncTest, L10_2)
    {
        nn::wlan::MacAddress clientMacAddress;
        nn::wlan::ConnectionStatus clientStatus;

        const TestLocalConnectParams ConnectParam = {
            nn::wlan::Ssid(LocalMasterTestSSID), nn::wlan::MacAddress::CreateBroadcastMacAddress(), 6,
            { nn::wlan::SecurityMode_StaticAes, nn::wlan::SecurityMode_StaticAes, 0, LocalMasterAes16Key }, true, nn::wlan::BeaconIndication_Enable, 10
        };

        WLANTEST_ASSERT_TRUE(localClient.SetupLocalClient(ConnectParam.ssid, ConnectParam.security, TestMatchInfo, sizeof(TestMatchInfo) / sizeof(TestMatchInfo[0]), true, false));

        localClient.LocalTraceConnectsInfo(true);
        Connect(ConnectParam);
        localClient.LocalTraceConnectsInfo(true);

        // CLIENT７台共に６０秒以上接続状態であること
        sysTick = nn::os::GetSystemTick();
        while (NN_STATIC_CONDITION(1))
        {
            if (nn::os::TimedWaitSystemEvent(&connectionEvent, nn::TimeSpan::FromMilliSeconds(PutInterval)) == true)
            {
                WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::GetConnectionStatus(&clientStatus));
                if (clientStatus.state != nn::wlan::ConnectionState_Connected)
                {
                    WLANTEST_EXPECT_TRUE((nn::os::GetSystemTick() - sysTick).ToTimeSpan().GetMilliSeconds() <= Time60s);
                    break;
                }
            }
        }

        Disconnect();
    }

    // 平文設定のMasterとClient接続
    TEST_F(LocalMasterFuncTest, L11)
    {
        nn::wlan::MacAddress clientMacArray[ClientMaxConnect];

        const nn::wlan::MasterBssParameters fncTestMasterBss = {
            1, false, 30, true, nn::wlan::RateSetLegacy_11gMask, nn::wlan::RateSetLegacy_11gMask,
                { nn::wlan::SecurityMode_Open, nn::wlan::SecurityMode_Open, 0, "" }, nn::wlan::Ssid(LocalMasterTestSSID), 100
        };

        WLANTEST_ASSERT_TRUE(localMaster.SetupLocalMaster(fncTestMasterBss, TestMatchInfo, sizeof(TestMatchInfo) / sizeof(TestMatchInfo[0]), false));
        WLANTEST_ASSERT_TRUE(localMaster.LocalMasterConnectWait(clientMacArray, WlanTestConnectTimeOut, 1) > 0);
    }

    // StaticAES設定のMasterとClient接続
    TEST_F(LocalMasterFuncTest, L12)
    {
        nn::wlan::MacAddress clientMacArray[ClientMaxConnect];

        const nn::wlan::MasterBssParameters fncTestMasterBss = {
            11, false, 30, true, nn::wlan::RateSetLegacy_11gMask, nn::wlan::RateSetLegacy_11gMask,
                { nn::wlan::SecurityMode_StaticAes, nn::wlan::SecurityMode_StaticAes, 0, LocalMasterPasKey }, nn::wlan::Ssid(LocalMasterTestSSID), 100
        };

        WLANTEST_ASSERT_TRUE(localMaster.SetupLocalMaster(fncTestMasterBss, TestMatchInfo, sizeof(TestMatchInfo) / sizeof(TestMatchInfo[0]), false));
        WLANTEST_ASSERT_TRUE(localMaster.LocalMasterConnectWait(clientMacArray, WlanTestConnectTimeOut, 1) > 0);
    }

    // 平文設定のMasterに、StaticAES設定のClientは接続しないこと
    TEST_F(LocalClientFuncTest, L13)
    {
        nn::wlan::ScanIeMatchInfo info;
        nn::wlan::MacAddress putMacAddress;
        std::unique_ptr<uint8_t[]> pTestBuffer(new uint8_t[TestScanMaxSize]);
        WLANTEST_ASSERT_TRUE(pTestBuffer.get() != nullptr);

        const nn::wlan::ScanParameters StartScanTestParams = {
            nn::wlan::ScanType_Active,{ 6 }, 1, 40, -1, &ssidList[0], 1, nn::wlan::MacAddress::CreateBroadcastMacAddress()
        };

        const TestLocalConnectParams ConnectParam = {
            nn::wlan::Ssid(LocalMasterTestSSID), nn::wlan::MacAddress::CreateBroadcastMacAddress(), 6,
            { nn::wlan::SecurityMode_StaticAes, nn::wlan::SecurityMode_StaticAes, 0, LocalMasterAes16Key }, true, nn::wlan::BeaconIndication_Enable, 10
        };

        // MASTERの存在確認（存在しない場合でも未起動状態でPASSしてしまうため）
        WLANTEST_EXPECT_RESULT_SUCCESS(nn::wlan::Local::StartScan(pTestBuffer.get(), TestScanMaxSize, StartScanTestParams, &info));
        WLANTEST_EXPECT_TRUE(CompareScanSsid(pTestBuffer.get(), TestScanMaxSize, ConnectParam.ssid));

        sysTick = nn::os::GetSystemTick();

        // 30秒間再試行
        while ((nn::os::GetSystemTick() - sysTick).ToTimeSpan().GetMilliSeconds() <= Time30s)
        {
            WLANTEST_EXPECT_RESULT_SUCCESS(nn::wlan::Local::Connect(ConnectParam.ssid,
                ConnectParam.bssid,
                ConnectParam.channel,
                ConnectParam.security,
                ConnectParam.autoKeepAlive,
                ConnectParam.indication,
                ConnectParam.beaconLostTimeout));

            WLANTEST_EXPECT_RESULT_SUCCESS(nn::wlan::Local::GetConnectionStatus(&connectionStatus));
            WLANTEST_EXPECT_TRUE(connectionStatus.state != nn::wlan::ConnectionState_Connected);

            // 3秒待って再試行
            nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(Time3s));
        }

        Disconnect();
    }

    // StaticAES設定のMasterに、平文設定のClientは接続しないこと
    TEST_F(LocalClientFuncTest, L14)
    {
        nn::wlan::ScanIeMatchInfo info;
        nn::wlan::MacAddress putMacAddress;
        std::unique_ptr<uint8_t[]> pTestBuffer(new uint8_t[TestScanMaxSize]);
        WLANTEST_ASSERT_TRUE(pTestBuffer.get() != nullptr);

        const nn::wlan::ScanParameters StartScanTestParams = {
            nn::wlan::ScanType_Active,{ 11 }, 1, 40, -1, &ssidList[0], 1, nn::wlan::MacAddress::CreateBroadcastMacAddress()
        };

        const TestLocalConnectParams ConnectParam = {
            nn::wlan::Ssid(LocalMasterTestSSID), nn::wlan::MacAddress::CreateBroadcastMacAddress(), 11,
            { nn::wlan::SecurityMode_Open, nn::wlan::SecurityMode_Open, 0, "" }, true, nn::wlan::BeaconIndication_Enable, 10
        };

        // MASTERの存在確認（存在しない場合でも未起動状態でPASSしてしまうため）
        WLANTEST_EXPECT_RESULT_SUCCESS(nn::wlan::Local::StartScan(pTestBuffer.get(), TestScanMaxSize, StartScanTestParams, &info));
        WLANTEST_EXPECT_TRUE(CompareScanSsid(pTestBuffer.get(), TestScanMaxSize, ConnectParam.ssid));

        sysTick = nn::os::GetSystemTick();

        // 30秒間再試行
        while ((nn::os::GetSystemTick() - sysTick).ToTimeSpan().GetMilliSeconds() <= Time30s)
        {
            WLANTEST_EXPECT_RESULT_SUCCESS(nn::wlan::Local::Connect(ConnectParam.ssid,
                ConnectParam.bssid,
                ConnectParam.channel,
                ConnectParam.security,
                ConnectParam.autoKeepAlive,
                ConnectParam.indication,
                ConnectParam.beaconLostTimeout));

            WLANTEST_EXPECT_RESULT_SUCCESS(nn::wlan::Local::GetConnectionStatus(&connectionStatus));
            WLANTEST_EXPECT_TRUE(connectionStatus.state != nn::wlan::ConnectionState_Connected);

            // 3秒待って再試行
            nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(Time3s));
        }

        Disconnect();
    }

    // MasterとClientの暗号鍵設定が異なっても、StaticAES設定同士の場合、接続はできること
    // MasterとClientの暗号鍵設定が異なる場合、通信できないこと MASTER
    TEST_F(LocalMasterFuncTest, L15_L70_1)
    {
        const nn::wlan::MasterBssParameters fncTestMasterBss = {
            1, false, 30, true, nn::wlan::RateSetLegacy_11gMask, nn::wlan::RateSetLegacy_11gMask,
            { nn::wlan::SecurityMode_StaticAes, nn::wlan::SecurityMode_StaticAes, 0, LocalMasterPasKey }, nn::wlan::Ssid(LocalMasterTestSSID), 100
        };

        WLANTEST_ASSERT_TRUE(localMaster.SetupLocalMaster(fncTestMasterBss, TestMatchInfo, sizeof(TestMatchInfo) / sizeof(TestMatchInfo[0]), false));

        // MASTER受信スレッド起動
        WLANTEST_ASSERT_TRUE(localMaster.LocalReceiveFrameStart(BufferSize100k, WlanTestPacketBufferSize, 1, 1, WlanTestGetFrameCount));

        // CLIENT接続待機
        WLANTEST_EXPECT_TRUE(localMaster.LocalReceiveWait(WlanTestMasterWaitTimeOut));

        // CLIENT切断待機
        WLANTEST_EXPECT_TRUE(localMaster.LocalMasterDisConnectWait(Time30s + Time5s) == 0);

        localMaster.LocalReceiveFrameStop();

        localMaster.LocalGetConnectInfo(&connectsInfo);

        // 通信できていないこと
        WLANTEST_ASSERT_TRUE(connectsInfo.connectCount == 0);
    }

    // MasterとClientの暗号鍵設定が異なっても、StaticAES設定同士の場合、接続はできること
    // MasterとClientの暗号鍵設定が異なる場合、通信できないこと CLIENT
    TEST_F(LocalClientFuncTest, L15_L70_2)
    {
        const TestLocalConnectParams ConnectParam = {
            nn::wlan::Ssid(LocalMasterTestSSID), nn::wlan::MacAddress::CreateBroadcastMacAddress(), 1,
            { nn::wlan::SecurityMode_StaticAes, nn::wlan::SecurityMode_StaticAes, 0, LocalMasterWep128PasKey }, true, nn::wlan::BeaconIndication_Enable, 10
        };

        WLANTEST_ASSERT_TRUE(localClient.SetupLocalClient(ConnectParam.ssid, ConnectParam.security, TestMatchInfo, sizeof(TestMatchInfo) / sizeof(TestMatchInfo[0]), true, false));

        sysTick = nn::os::GetSystemTick();
        // 30秒間再試行
        while ((nn::os::GetSystemTick() - sysTick).ToTimeSpan().GetMilliSeconds() <= Time30s)
        {
            WLANTEST_EXPECT_RESULT_SUCCESS(nn::wlan::Local::Connect(ConnectParam.ssid,
                ConnectParam.bssid,
                ConnectParam.channel,
                ConnectParam.security,
                ConnectParam.autoKeepAlive,
                ConnectParam.indication,
                ConnectParam.beaconLostTimeout));

            WLANTEST_EXPECT_RESULT_SUCCESS(nn::wlan::Local::GetConnectionStatus(&connectionStatus));
            if (connectionStatus.state == nn::wlan::ConnectionState_Connected)
            {
                NN_LOG("             L15 - Connected\n");
                break;
            }

            // 3秒待って再試行
            nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(Time3s));
        }

        WLANTEST_ASSERT_TRUE(connectionStatus.state == nn::wlan::ConnectionState_Connected);

        // 受信スレッド起動
        NN_LOG("             L70 - Recv Start\n");
        WLANTEST_EXPECT_TRUE(localClient.LocalReceiveFrameStart(BufferSize100k, WlanTestPacketBufferSize, 1, 1, WlanTestGetFrameCount));
        // MASTER接続待機
        WLANTEST_EXPECT_FALSE(localClient.LocalReceiveWait(Time5s, true));

        // 送信スレッド起動
        NN_LOG("             L70 - Send Start\n");
        WLANTEST_EXPECT_TRUE(localClient.LocalSendFrameStart(WlanTestPacketBufferSize, Time60s + Time5s, PutInterval, true));

        // 30秒通信試行待機
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(Time5s));

        localClient.LocalSendFrameStop();
        NN_LOG("             L70 - Send End\n");
        localClient.LocalReceiveFrameStop();
        NN_LOG("             L70 - Recv End\n");

        localClient.LocalGetConnectInfo(&connectsInfo);

        // 通信できていないこと
        WLANTEST_EXPECT_TRUE(connectsInfo.connectCount == 0);

        Disconnect();
    }

    // SSID指定で、1文字のSSIDを持つMasterに接続できること
    // 1文字のSSIDを設定できること MASTER
    TEST_F(LocalMasterFuncTest, L16_L46_1)
    {
        nn::wlan::MacAddress clientMacArray[ClientMaxConnect];

        const nn::wlan::MasterBssParameters fncTestMasterBss = {
            1, false, 30, true, nn::wlan::RateSetLegacy_11gMask, nn::wlan::RateSetLegacy_11gMask,
            { nn::wlan::SecurityMode_Open, nn::wlan::SecurityMode_Open, 0, "" }, nn::wlan::Ssid(LocalMasterTestShortSSID), 100
        };

        // ジョイコン2セット（各4台）
        WLANTEST_ASSERT_TRUE(localMaster.SetBtMode(LocalApiBtNode4));

        WLANTEST_ASSERT_TRUE(localMaster.SetupLocalMaster(fncTestMasterBss, TestMatchInfo, sizeof(TestMatchInfo) / sizeof(TestMatchInfo[0]), false));
        WLANTEST_ASSERT_TRUE(localMaster.LocalMasterConnectWait(clientMacArray, WlanTestConnectTimeOut, 1) > 0);
    }

    // SSID指定で、1文字のSSIDを持つMasterに接続できること
    // 1文字のSSIDを設定できること CLIENT
    TEST_F(LocalClientFuncTest, L16_L46_2)
    {
        const TestLocalConnectParams ConnectParam = {
            nn::wlan::Ssid(LocalMasterTestShortSSID), nn::wlan::MacAddress::CreateBroadcastMacAddress(), 1,
            { nn::wlan::SecurityMode_Open, nn::wlan::SecurityMode_Open, 0, "" }, true, nn::wlan::BeaconIndication_Enable, 10
        };

        // ジョイコン2セット（各4台）
        WLANTEST_ASSERT_TRUE(localClient.SetBtMode(LocalApiBtNode4));

        sysTick = nn::os::GetSystemTick();
        // 30秒間再試行
        while ((nn::os::GetSystemTick() - sysTick).ToTimeSpan().GetMilliSeconds() <= Time30s)
        {
            WLANTEST_EXPECT_RESULT_SUCCESS(nn::wlan::Local::Connect(ConnectParam.ssid,
                ConnectParam.bssid,
                ConnectParam.channel,
                ConnectParam.security,
                ConnectParam.autoKeepAlive,
                ConnectParam.indication,
                ConnectParam.beaconLostTimeout));

            WLANTEST_EXPECT_RESULT_SUCCESS(nn::wlan::Local::GetConnectionStatus(&connectionStatus));
            if (connectionStatus.state == nn::wlan::ConnectionState_Connected)
            {
                break;
            }

            // 3秒待って再試行
            nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(Time3s));
        }

        WLANTEST_EXPECT_TRUE(connectionStatus.state == nn::wlan::ConnectionState_Connected);

        Disconnect();
    }

    // SSID指定で、32文字のSSIDを持つMaster接続
    TEST_F(LocalClientFuncTest, L17)
    {
        const TestLocalConnectParams ConnectParam = {
            nn::wlan::Ssid(LocalMasterTestLongSSID), nn::wlan::MacAddress::CreateBroadcastMacAddress(), 1,
            { nn::wlan::SecurityMode_Open, nn::wlan::SecurityMode_Open, 0, "" }, true, nn::wlan::BeaconIndication_Enable, 10
        };

        Connect(ConnectParam);

        Disconnect();
    }

    // BSSID指定で、Master接続 Master
    TEST_F(LocalMasterFuncTest, L18_1)
    {
        nn::wlan::MacAddress clientArray[LocalTcpGetMacAddressMax];
        uint32_t macCnt;

        const nn::wlan::MasterBssParameters fncTestMasterBss = {
            1, false, 30, true, nn::wlan::RateSetLegacy_11gMask, nn::wlan::RateSetLegacy_11gMask,
            { nn::wlan::SecurityMode_Open, nn::wlan::SecurityMode_Open, 0, "" }, nn::wlan::Ssid(LocalMasterTestSSID), 100
        };

        WLANTEST_ASSERT_TRUE(localMaster.GetMasterTcpMacAddress(clientArray, macCnt, tcpMaster, 1));
        WLANTEST_ASSERT_TRUE(localMaster.SetupLocalMaster(fncTestMasterBss, TestMatchInfo, sizeof(TestMatchInfo) / sizeof(TestMatchInfo[0]), true));

        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(Time15s));
    }

    // BSSID指定で、Master接続 Client
    TEST_F(LocalClientFuncTest, L18_2)
    {
        nn::wlan::MacAddress clientArray[LocalTcpGetMacAddressMax];
        uint32_t macCnt;

        IpV4Address tcpClient = {
            "0.0.0.0", "255.255.255.0", "192.168.11.1",
            "8.8.8.8", "8.8.8.9", 0, false
        };
        nn::util::SNPrintf(tcpClient.strIpAddress, sizeof(tcpClient.strIpAddress), "192.168.11.%d", g_IpOct4);

        WLANTEST_ASSERT_TRUE(localClient.GetClientTcpMacAddress(clientArray, macCnt, tcpMaster, tcpClient));

        nn::wlan::MacAddress macAddress = nn::wlan::MacAddress(clientArray[0]);

        const TestLocalConnectParams ConnectParam = {
            nn::wlan::Ssid(), macAddress, 1,
            { nn::wlan::SecurityMode_Open, nn::wlan::SecurityMode_Open, 0, "" }, true, nn::wlan::BeaconIndication_Enable, 10
        };

        Connect(ConnectParam);

        Disconnect();
    }

    // Masterへの接続処理をキャンセル
    TEST_F(LocalClientFuncTest, L19)
    {
        const TestLocalConnectParams ConnectParam = {
            nn::wlan::Ssid(LocalMasterTestSSID), nn::wlan::MacAddress::CreateBroadcastMacAddress(), 1,
            { nn::wlan::SecurityMode_Open, nn::wlan::SecurityMode_Open, 0, "" }, true, nn::wlan::BeaconIndication_Enable, 10
        };

        WLANTEST_ASSERT_TRUE(localClient.SetupLocalClient(ConnectParam.ssid, ConnectParam.security, TestMatchInfo, sizeof(TestMatchInfo) / sizeof(TestMatchInfo[0]), true, false));

        localClient.LocalClientAsyncConnect();
        NN_LOG("             L19 - CancelConnect Start\n");
        WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::CancelConnect());
        NN_LOG("             L19 - CancelConnect End\n");
    }

    // 指定したチャネルにMasterが存在しない場合、Clientは接続しない
    // 存在しないMasterに繰り返し接続試行しても、以降の動作に影響がないこと MASTER
    TEST_F(LocalMasterFuncTest, L20_L125_1)
    {
        nn::wlan::MacAddress clientMacArray[ClientMaxConnect];

        const nn::wlan::MasterBssParameters fncTestMasterBss = {
            6, false, 30, true, nn::wlan::RateSetLegacy_11gMask, nn::wlan::RateSetLegacy_11gMask,
            { nn::wlan::SecurityMode_Open, nn::wlan::SecurityMode_Open, 0, "" }, nn::wlan::Ssid(LocalMasterTestSSID), 100
        };

        WLANTEST_ASSERT_TRUE(localMaster.SetupLocalMaster(fncTestMasterBss, TestMatchInfo, sizeof(TestMatchInfo) / sizeof(TestMatchInfo[0]), false));
        WLANTEST_ASSERT_TRUE(localMaster.LocalMasterConnectWait(clientMacArray, Time600s + Time30s, 1) > 0);
        WLANTEST_ASSERT_TRUE(localMaster.LocalMasterDisConnectWait(Time30s) == 0);
    }

    // 指定したチャネルにMasterが存在しない場合、Clientは接続しない
    // 存在しないMasterに繰り返し接続試行しても、以降の動作に影響がないこと CLIENT
    TEST_F(LocalClientFuncTest, L20_L125_2)
    {
        const TestLocalConnectParams NotConnectParam = {
            nn::wlan::Ssid(LocalMasterTestLongSSID), nn::wlan::MacAddress::CreateBroadcastMacAddress(), 6,
            { nn::wlan::SecurityMode_Open, nn::wlan::SecurityMode_Open, 0, "" }, true, nn::wlan::BeaconIndication_Enable, 10
        };

        const TestLocalConnectParams ConnectParam = {
            nn::wlan::Ssid(LocalMasterTestSSID), nn::wlan::MacAddress::CreateBroadcastMacAddress(), 6,
            { nn::wlan::SecurityMode_Open, nn::wlan::SecurityMode_Open, 0, "" }, true, nn::wlan::BeaconIndication_Enable, 10
        };

        sysTick = nn::os::GetSystemTick();
        // 10分間再試行
        while ((nn::os::GetSystemTick() - sysTick).ToTimeSpan().GetMilliSeconds() <= Time600s)
        {
            WLANTEST_EXPECT_RESULT_SUCCESS(nn::wlan::Local::Connect(NotConnectParam.ssid,
                NotConnectParam.bssid,
                NotConnectParam.channel,
                NotConnectParam.security,
                NotConnectParam.autoKeepAlive,
                NotConnectParam.indication,
                NotConnectParam.beaconLostTimeout));

            WLANTEST_EXPECT_RESULT_SUCCESS(nn::wlan::Local::GetConnectionStatus(&connectionStatus));
            if (connectionStatus.state == nn::wlan::ConnectionState_Connected)
            {
                break;
            }

            // 3秒待って再試行
            nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(Time3s));
        }

        WLANTEST_EXPECT_TRUE(connectionStatus.state != nn::wlan::ConnectionState_Connected);

        Connect(ConnectParam);
        Disconnect();
    }

    // MasterとClient7台接続
    // Masterは、接続中の7台の端末情報を得る
    TEST_F(LocalMasterFuncTest, L25_L101)
    {
        nn::wlan::MacAddress clientMacArray[ClientMaxConnect];
        char macArray[nn::wlan::MacAddress::MacStringSize];
        uint32_t clientCount = 0;
        nn::os::Tick setupTick;

        const nn::wlan::MasterBssParameters fncTestMasterBss = {
            6, false, 30, true, nn::wlan::RateSetLegacy_11gMask, nn::wlan::RateSetLegacy_11gMask,
            { nn::wlan::SecurityMode_StaticAes, nn::wlan::SecurityMode_StaticAes, 0, LocalMasterPasKey }, nn::wlan::Ssid(LocalMasterTestSSID), 100
        };

        // ジョイコン1セット（各2台）
        WLANTEST_ASSERT_TRUE(localMaster.SetBtMode(LocalApiBtNode2));

        WLANTEST_ASSERT_TRUE(localMaster.SetupLocalMaster(fncTestMasterBss, TestMatchInfo, sizeof(TestMatchInfo) / sizeof(TestMatchInfo[0]), false));

        clientCount = localMaster.LocalMasterConnectWait(clientMacArray, WlanTestConnectTimeOut, ClientMaxConnect);
        WLANTEST_EXPECT_TRUE(clientCount == ClientMaxConnect);

        NN_LOG("\n             *** CLIENT STATUS ***\n");
        for (int8_t i = nn::wlan::ConnectableClientsCountMax; i >= 0; i--)
        {
            NN_LOG("             GetClientStatusBitMap : 0x%08x\n", localMaster.m_clientStatusBitMap);

            if (localMaster.m_clientStatus[i].state == nn::wlan::ConnectionState_Connected)
            {
                NN_LOG("               MAC ADDRESS     : %s\n", localMaster.m_clientStatus[i].clientMacAddress.GetString(macArray));
                NN_LOG("               STATE           : %d\n", localMaster.m_clientStatus[i].state);
                NN_LOG("               CAUSE           : %d\n", localMaster.m_clientStatus[i].cause);
                NN_LOG("               RESASON CODE    : %d\n", localMaster.m_clientStatus[i].statusReasonCode);
                NN_LOG("               CAPABILITY INFO : %d\n", localMaster.m_clientStatus[i].capabilityInfo);
                NN_LOG("               RSSI            : %d\n\n", localMaster.m_clientStatus[i].rssi);
            }
        }
        NN_LOG("             *** CLIENT STATUS ***\n");
    }

    // 6台接続して通信中のMasterに7台目のClient接続試行 MASTER
    TEST_F(LocalMasterFuncTest, L26_1)
    {
        const nn::wlan::MasterBssParameters fncTestMasterBss = {
            11, false, 30, true, nn::wlan::RateSetLegacy_11gMask, nn::wlan::RateSetLegacy_11gMask,
            { nn::wlan::SecurityMode_Open, nn::wlan::SecurityMode_Open, 0, "" }, nn::wlan::Ssid(LocalMasterTestSSID), 100
        };

        // ジョイコン1セット（各2台）
        WLANTEST_ASSERT_TRUE(localMaster.SetBtMode(LocalApiBtNode2));

        WLANTEST_ASSERT_TRUE(localMaster.SetupLocalMaster(fncTestMasterBss, TestMatchInfo, sizeof(TestMatchInfo) / sizeof(TestMatchInfo[0])));

        // MASTER受信スレッド起動
        WLANTEST_ASSERT_TRUE(localMaster.LocalReceiveFrameStart(BufferSize100k, WlanTestPacketBufferSize, ClientDynamicConnet, ClientMaxConnect, WlanTestGetFrameCount));

        // CLIENT接続待機
        WLANTEST_ASSERT_TRUE(localMaster.LocalReceiveWait(WlanTestMasterWaitTimeOut));

        // MASTER送信スレッド起動
        WLANTEST_ASSERT_TRUE(localMaster.LocalSendFrameOneShotWaitStart(WlanTestPacketBufferSize, Time60s, Interval_100m, true));

        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(Time60s + Time5s));

        // MASTER送信スレッド停止
        localMaster.LocalSendFrameStop();
        // MASTER受信スレッド停止
        localMaster.LocalReceiveFrameStop();

        localMaster.LocalTraceConnectsInfo(true);

        localMaster.LocalGetConnectInfo(&connectsInfo);
        WLANTEST_EXPECT_TRUE(connectsInfo.connectCount >= ClientMaxConnect);

        for (int32_t i = 0; i < connectsInfo.connectCount; i++)
        {
            WLANTEST_EXPECT_TRUE(connectsInfo.facing[i].isConnected);
            WLANTEST_EXPECT_TRUE(connectsInfo.facing[i].getFrameCount > 0);
        }
    }

    // 6台接続して通信中のMasterに7台目のClient接続試行 CLIENT
    TEST_F(LocalClientFuncTest, L26_2)
    {
        nn::wlan::ConnectionStatus ConnectionStatus;

        const TestLocalConnectParams ConnectParam = {
            nn::wlan::Ssid(LocalMasterTestSSID), nn::wlan::MacAddress::CreateBroadcastMacAddress(), 11,
            { nn::wlan::SecurityMode_Open, nn::wlan::SecurityMode_Open, 0, "" }, true, nn::wlan::BeaconIndication_Enable, 10
        };

        // ジョイコン1セット（各2台）
        WLANTEST_ASSERT_TRUE(localClient.SetBtMode(LocalApiBtNode2));

        WLANTEST_ASSERT_TRUE(localClient.SetupLocalClient(ConnectParam.ssid, ConnectParam.security, TestMatchInfo, sizeof(TestMatchInfo) / sizeof(TestMatchInfo[0]), true, false));

        localClient.LocalTraceConnectsInfo(true);
        Connect(ConnectParam);
        localClient.LocalTraceConnectsInfo(true);

        // 受信スレッド起動
        WLANTEST_ASSERT_TRUE(localClient.LocalReceiveFrameStart(BufferSize100k, WlanTestPacketBufferSize, 1, 1, WlanTestGetFrameCount));
        // MASTER接続待機
        WLANTEST_ASSERT_TRUE(localClient.LocalReceiveWait(WlanMaxClientWaitMsecTimeOut, Time15s, true));
        // 送信スレッド起動
        WLANTEST_ASSERT_TRUE(localClient.LocalSendFrameOneShotWaitStart(WlanTestPacketBufferSize, Time60s, Interval_100m, true));

        while (NN_STATIC_CONDITION(1))
        {
            // MASTERからの接続が切られるまで送信し続ける
            if (nn::os::TimedWaitSystemEvent(&connectionEvent, nn::TimeSpan::FromMilliSeconds(PutInterval)) == true)
            {
                WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::GetConnectionStatus(&ConnectionStatus));
                if (ConnectionStatus.state != nn::wlan::ConnectionState_Connected)
                {
                    break;
                }
            }

            if ((nn::os::GetSystemTick() - logOutTime).ToTimeSpan().GetMilliSeconds() >= Time60s)
            {
                localClient.LocalTraceConnectsInfo(true);
                logOutTime = nn::os::GetSystemTick();
            }
        }

        // 送信スレッド停止
        localClient.LocalSendFrameStop();
        // 受信スレッド停止
        localClient.LocalReceiveFrameStop();
        // 通信結果ログ出力
        localClient.LocalTraceConnectsInfo(true);

        Disconnect();
    }

    // Masterに接続中の任意の1台が入れ替わる MASTER
    TEST_F(LocalMasterFuncTest, L27_1)
    {
        const nn::wlan::MasterBssParameters fncTestMasterBss = {
            6, true, 30, true, nn::wlan::RateSetLegacy_11gMask, nn::wlan::RateSetLegacy_11gMask,
            { nn::wlan::SecurityMode_Open, nn::wlan::SecurityMode_Open, 0, "" }, nn::wlan::Ssid(LocalMasterTestSSID), 100
        };

        // ジョイコン1セット（各2台）
        WLANTEST_ASSERT_TRUE(localMaster.SetBtMode(LocalApiBtNode2));

        localMaster.SetupLocalMaster(fncTestMasterBss, TestMatchInfo, sizeof(TestMatchInfo) / sizeof(TestMatchInfo[0]));

        // MASTER受信スレッド起動
        WLANTEST_ASSERT_TRUE(localMaster.LocalReceiveFrameStart(BufferSize100k, WlanTestPacketBufferSize, 2, 2, WlanTestGetFrameCount));

        // MASTER接続待機
        WLANTEST_ASSERT_TRUE(localMaster.LocalReceiveWait(WlanTestMasterWaitTimeOut, true));

        // 60秒間通信
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(Time60s + Time5s));

        // MASTER受信スレッド停止
        localMaster.LocalReceiveFrameStop();
    }

    // Masterに接続中の任意の1台が入れ替わる CLIENT 1台目
    TEST_F(LocalClientFuncTest, L27_2)
    {
        nn::os::Tick changeTime;
        bool isConnected = false;

        const TestLocalConnectParams ConnectParam = {
            nn::wlan::Ssid(LocalMasterTestSSID), nn::wlan::MacAddress::CreateBroadcastMacAddress(), 6,
            { nn::wlan::SecurityMode_Open, nn::wlan::SecurityMode_Open, 0, "" }, true, nn::wlan::BeaconIndication_Enable, 10
        };

        // ジョイコン1セット（各2台）
        WLANTEST_ASSERT_TRUE(localClient.SetBtMode(LocalApiBtNode2));

        WLANTEST_ASSERT_TRUE(localClient.SetupLocalClient(ConnectParam.ssid, ConnectParam.security, TestMatchInfo, sizeof(TestMatchInfo) / sizeof(TestMatchInfo[0]), true, false));

        localClient.LocalTraceConnectsInfo(true);
        Connect(ConnectParam);
        isConnected = true;
        localClient.LocalTraceConnectsInfo(true);

        WLANTEST_ASSERT_TRUE(localClient.LocalReceiveFrameStart(BufferSize100k, WlanTestPacketBufferSize, 1, 1, WlanTestGetFrameCount));
        WLANTEST_ASSERT_TRUE(localClient.LocalReceiveWait(WlanMaxClientWaitMsecTimeOut, Time1s, true));

        // CLIENT1台目は接続状態からスタート
        sysTick = nn::os::GetSystemTick();
        changeTime = sysTick;

        // 60秒間通信
        while ((nn::os::GetSystemTick() - sysTick).ToTimeSpan().GetMilliSeconds() <= Time60s)
        {
            // 10毎にClientを入れ替え
            if ((nn::os::GetSystemTick() - changeTime).ToTimeSpan().GetMilliSeconds() >= Time10s)
            {
                if (isConnected != true)
                {
                    Connect(ConnectParam);
                    isConnected = true;
                }
                else
                {
                    nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(Time1s));
                    Disconnect();
                    isConnected = false;
                }

                changeTime = nn::os::GetSystemTick();
            }

            nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(Time1s));
        }

        localClient.LocalTraceConnectsInfo(true);

        // 受信スレッド停止
        localClient.LocalReceiveFrameStop();

        Disconnect();
    }

    // Masterに接続中の任意の1台が入れ替わる CLIENT 2台目
    TEST_F(LocalClientFuncTest, L27_3)
    {
        nn::os::Tick changeTime;
        bool isConnected = false;

        const TestLocalConnectParams ConnectParam = {
            nn::wlan::Ssid(LocalMasterTestSSID), nn::wlan::MacAddress::CreateBroadcastMacAddress(), 6,
            { nn::wlan::SecurityMode_Open, nn::wlan::SecurityMode_Open, 0, "" }, true, nn::wlan::BeaconIndication_Enable, 10
        };

        // ジョイコン1セット（各2台）
        WLANTEST_ASSERT_TRUE(localClient.SetBtMode(LocalApiBtNode2));

        WLANTEST_ASSERT_TRUE(localClient.SetupLocalClient(ConnectParam.ssid, ConnectParam.security, TestMatchInfo, sizeof(TestMatchInfo) / sizeof(TestMatchInfo[0]), true, false));

        localClient.LocalTraceConnectsInfo(true);
        Connect(ConnectParam);
        localClient.LocalTraceConnectsInfo(true);

        WLANTEST_ASSERT_TRUE(localClient.LocalReceiveFrameStart(BufferSize100k, WlanTestPacketBufferSize, 1, 1, WlanTestGetFrameCount));
        WLANTEST_ASSERT_TRUE(localClient.LocalReceiveWait(WlanMaxClientWaitMsecTimeOut, Time1s, true));

        // CLIENT2台目は切断状態からスタート
        sysTick = nn::os::GetSystemTick();
        changeTime = sysTick;

        Disconnect();
        isConnected = false;
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(Time10s));

        // 60秒間通信
        while ((nn::os::GetSystemTick() - sysTick).ToTimeSpan().GetMilliSeconds() <= Time60s)
        {
            // 10毎にClientを入れ替え
            if ((nn::os::GetSystemTick() - changeTime).ToTimeSpan().GetMilliSeconds() >= Time10s)
            {
                if (isConnected != true)
                {
                    Connect(ConnectParam);
                    isConnected = true;
                }
                else
                {
                    nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(Time1s));
                    Disconnect();
                    isConnected = false;
                }

                changeTime = nn::os::GetSystemTick();
            }
            nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(Time1s));
        }

        localClient.LocalTraceConnectsInfo(true);

        // 受信スレッド停止
        localClient.LocalReceiveFrameStop();

        Disconnect();
    }

    // Clientが接続に失敗した後も、Client7台の接続ができること MASTER
    TEST_F(LocalMasterFuncTest, L28_1)
    {
        nn::wlan::MacAddress clientMacArray[ClientMaxConnect];

        const nn::wlan::MasterBssParameters fncTestMasterBss = {
            11, true, 30, true, nn::wlan::RateSetLegacy_11gMask, nn::wlan::RateSetLegacy_11gMask,
            { nn::wlan::SecurityMode_Open, nn::wlan::SecurityMode_Open, 0, "" }, nn::wlan::Ssid(LocalMasterTestSSID), 100
        };

        // ジョイコン1セット（各2台）
        WLANTEST_ASSERT_TRUE(localMaster.SetBtMode(LocalApiBtNode2));

        localMaster.SetupLocalMaster(fncTestMasterBss, TestMatchInfo, sizeof(TestMatchInfo) / sizeof(TestMatchInfo[0]));

        // MASTER受信スレッド起動
        WLANTEST_ASSERT_TRUE(localMaster.LocalReceiveFrameStart(BufferSize100k, WlanTestPacketBufferSize, ClientMaxConnect, ClientMaxConnect, WlanTestGetFrameCount));

        // MASTER接続待機
        WLANTEST_ASSERT_TRUE(localMaster.LocalReceiveWait(WlanTestMasterWaitTimeOut, true));

        // CLIENT切断待機
        while (NN_STATIC_CONDITION(1))
        {
            if (nn::os::TimedWaitSystemEvent(&localMaster.m_connectEvent, nn::TimeSpan::FromMilliSeconds(WlanTestMasterWaitTimeOut)) == true)
            {
                if (localMaster.GetClientMacAddress(clientMacArray, nn::wlan::ConnectionState_Connected) == 0)
                {
                    break;
                }
            }
        }

        // MASTER受信スレッド停止
        localMaster.LocalReceiveFrameStop();
    }

    // Clientが接続に失敗した後も、Client7台の接続ができること CLIENT 6台 + 1台(L28_3実行終了後)
    TEST_F(LocalClientFuncTest, L28_2)
    {
        nn::os::Tick changeTime;
        bool isConnected = false;

        const TestLocalConnectParams ConnectParam = {
            nn::wlan::Ssid(LocalMasterTestSSID), nn::wlan::MacAddress::CreateBroadcastMacAddress(), 11,
            { nn::wlan::SecurityMode_Open, nn::wlan::SecurityMode_Open, 0, "" }, true, nn::wlan::BeaconIndication_Enable, 10
        };

        // ジョイコン1セット（各2台）
        WLANTEST_ASSERT_TRUE(localClient.SetBtMode(LocalApiBtNode2));

        // まずCLIENT6台を接続
        WLANTEST_ASSERT_TRUE(localClient.SetupLocalClient(ConnectParam.ssid, ConnectParam.security, TestMatchInfo, sizeof(TestMatchInfo) / sizeof(TestMatchInfo[0]), true, false));

        localClient.LocalTraceConnectsInfo(true);
        Connect(ConnectParam);
        isConnected = true;
        localClient.LocalTraceConnectsInfo(true);

        WLANTEST_ASSERT_TRUE(localClient.LocalReceiveFrameStart(BufferSize100k, WlanTestPacketBufferSize, 1, 1, WlanTestGetFrameCount));
        WLANTEST_ASSERT_TRUE(localClient.LocalReceiveWait(WlanMaxClientWaitMsecTimeOut, Time1s, true));

        // CLIENT1台目は接続状態からスタート
        sysTick = nn::os::GetSystemTick();
        changeTime = sysTick;

        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(Time30s));

        // 受信スレッド停止
        localClient.LocalReceiveFrameStop();

        Disconnect();
    }

    // Clientが接続に失敗した後も、Client7台の接続ができること CLIENT 7台目
    TEST_F(LocalClientFuncTest, L28_3)
    {
        const TestLocalConnectParams ConnectParam = {
            nn::wlan::Ssid(LocalMasterTestSSID), nn::wlan::MacAddress::CreateBroadcastMacAddress(), 11,
            { nn::wlan::SecurityMode_Open, nn::wlan::SecurityMode_Open, 0, "" }, true, nn::wlan::BeaconIndication_Enable, 10
        };

        // ジョイコン1セット（各2台）
        WLANTEST_ASSERT_TRUE(localClient.SetBtMode(LocalApiBtNode2));

        // CLIENT7台目を接続
        WLANTEST_ASSERT_TRUE(localClient.SetupLocalClient(ConnectParam.ssid, ConnectParam.security, TestMatchInfo, sizeof(TestMatchInfo) / sizeof(TestMatchInfo[0]), true, false));

        localClient.LocalClientAsyncConnect();
        NN_LOG("             L28_3 - CancelConnect Start\n");
        WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::CancelConnect());
        NN_LOG("             L28_3 - CancelConnect End\n");

        // ここでCLIENT8台目をL28_2で接続
        NN_LOG("             ここで残り１台をL28_2で接続してください\n");
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(Time60s));

        Disconnect();
    }

    // ステルス設定のMasterにClient7台接続 MASTER
    TEST_F(LocalMasterFuncTest, L30_1)
    {
        nn::wlan::MacAddress clientMacArray[ClientMaxConnect];
        const nn::wlan::MasterBssParameters fncTestMasterBss = {
            6, true, 30, true, nn::wlan::RateSetLegacy_11gMask, nn::wlan::RateSetLegacy_11gMask,
            { nn::wlan::SecurityMode_StaticAes, nn::wlan::SecurityMode_StaticAes, 0, LocalMasterPasKey }, nn::wlan::Ssid(LocalMasterTest2SSID), 100
        };

        localMaster.SetupLocalMaster(fncTestMasterBss, TestMatchInfo, sizeof(TestMatchInfo) / sizeof(TestMatchInfo[0]));
        WLANTEST_ASSERT_TRUE(localMaster.LocalMasterConnectWait(clientMacArray, WlanTestConnectTimeOut, ClientMaxConnect) > 0);
    }

    // ステルス設定のMasterにClient7台接続 CLIENT
    TEST_F(LocalClientFuncTest, L30_2)
    {
        const TestLocalConnectParams ConnectParam = {
            nn::wlan::Ssid(LocalMasterTest2SSID), nn::wlan::MacAddress::CreateBroadcastMacAddress(), 6,
            { nn::wlan::SecurityMode_StaticAes, nn::wlan::SecurityMode_StaticAes, 0, LocalMasterPasKey }, true, nn::wlan::BeaconIndication_Enable, 10
        };

        WLANTEST_ASSERT_TRUE(localClient.SetupLocalClient(ConnectParam.ssid, ConnectParam.security, TestMatchInfo, sizeof(TestMatchInfo) / sizeof(TestMatchInfo[0]), true, false));

        localClient.LocalTraceConnectsInfo(true);
        Connect(ConnectParam);
        localClient.LocalTraceConnectsInfo(true);

        WLANTEST_ASSERT_TRUE(nn::os::TimedWaitSystemEvent(&localClient.m_connectEvent, nn::TimeSpan::FromMilliSeconds(Time120s)));

        Disconnect();
    }

    // 7台接続中のMasterに8台目のClient接続試行 MASTER
    TEST_F(LocalMasterFuncTest, L31_1)
    {
        nn::wlan::MacAddress clientMacArray[ClientMaxConnect];

        const nn::wlan::MasterBssParameters fncTestMasterBss = {
            11, false, 30, true, nn::wlan::RateSetLegacy_11gMask, nn::wlan::RateSetLegacy_11gMask,
            { nn::wlan::SecurityMode_Open, nn::wlan::SecurityMode_Open, 0, "" }, nn::wlan::Ssid(LocalMasterTestSSID), 100
        };

        WLANTEST_ASSERT_TRUE(localMaster.SetupLocalMaster(fncTestMasterBss, TestMatchInfo, sizeof(TestMatchInfo) / sizeof(TestMatchInfo[0])));

        // MASTER受信スレッド起動
        WLANTEST_ASSERT_TRUE(localMaster.LocalReceiveFrameStart(BufferSize100k, WlanTestPacketBufferSize, ClientMaxConnect, ClientMaxConnect, WlanTestGetFrameCount));
        // CLIENT接続待機
        WLANTEST_ASSERT_TRUE(localMaster.LocalReceiveWait(WlanTestMasterWaitTimeOut, true));

        // CLIENT切断待機
        sysTick = nn::os::GetSystemTick();
        while ((nn::os::GetSystemTick() - sysTick).ToTimeSpan().GetMilliSeconds() <= Time120s)
        {
            if (nn::os::TimedWaitSystemEvent(&localMaster.m_connectEvent, nn::TimeSpan::FromMilliSeconds(Time1s)) == true)
            {
                if (localMaster.GetClientMacAddress(clientMacArray, nn::wlan::ConnectionState_Connected) == 0)
                {
                    break;
                }
            }
        }

        // MASTER受信スレッド停止
        localMaster.LocalReceiveFrameStop();
    }

    // 7台接続中のMasterに8台目のClient接続試行 CLIENT
    TEST_F(LocalClientFuncTest, L31_2)
    {
        nn::wlan::ConnectionStatus ConnectionStatus;

        const TestLocalConnectParams ConnectParam = {
            nn::wlan::Ssid(LocalMasterTestSSID), nn::wlan::MacAddress::CreateBroadcastMacAddress(), 11,
            { nn::wlan::SecurityMode_Open, nn::wlan::SecurityMode_Open, 0, "" }, true, nn::wlan::BeaconIndication_Enable, 10
        };

        WLANTEST_ASSERT_TRUE(localClient.SetupLocalClient(ConnectParam.ssid, ConnectParam.security, TestMatchInfo, sizeof(TestMatchInfo) / sizeof(TestMatchInfo[0]), true, false));

        localClient.LocalTraceConnectsInfo(true);
        Connect(ConnectParam);
        localClient.LocalTraceConnectsInfo(true);

        // 受信スレッド起動
        WLANTEST_ASSERT_TRUE(localClient.LocalReceiveFrameStart(BufferSize100k, WlanTestPacketBufferSize, 1, 1, WlanTestGetFrameCount));
        // MASTER接続待機
        WLANTEST_ASSERT_TRUE(localClient.LocalReceiveWait(WlanMaxClientWaitMsecTimeOut, Time1s, true));

        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(Time120s));

        // 受信スレッド停止
        localClient.LocalReceiveFrameStop();

        Disconnect();

    }

    // 7台接続して通信中のMasterに8台目のClient接続試行 MASTER
    TEST_F(LocalMasterFuncTest, L32_1)
    {
        const nn::wlan::MasterBssParameters fncTestMasterBss = {
            1, false, 30, true, nn::wlan::RateSetLegacy_11gMask, nn::wlan::RateSetLegacy_11gMask,
            { nn::wlan::SecurityMode_Open, nn::wlan::SecurityMode_Open, 0, LocalMasterPasKey }, nn::wlan::Ssid(LocalMasterTestSSID), 100
        };

        // ジョイコン1セット（各2台）
        WLANTEST_ASSERT_TRUE(localMaster.SetBtMode(LocalApiBtNode2));

        WLANTEST_ASSERT_TRUE(localMaster.SetupLocalMaster(fncTestMasterBss, TestMatchInfo, sizeof(TestMatchInfo) / sizeof(TestMatchInfo[0])));

        // MASTER受信スレッド起動
        WLANTEST_ASSERT_TRUE(localMaster.LocalReceiveFrameStart(BufferSize100k, WlanTestPacketBufferSize, ClientMaxConnect, ClientMaxConnect, WlanTestGetFrameCount));
        // CLIENT接続待機
        WLANTEST_ASSERT_TRUE(localMaster.LocalReceiveWait(WlanTestMasterWaitTimeOut, true));

        // 送信スレッド起動
        WLANTEST_ASSERT_TRUE(localMaster.LocalSendFrameOneShotWaitStart(WlanTestPacketBufferSize, Time60s + Time5s, Interval_100m, true));

        // 60秒間通信状態確認
        WLANTEST_ASSERT_TRUE(localMaster.LocalMasterDisConnectWait(Time60s) > 0);

        // MASTER送信スレッド停止
        localMaster.LocalSendFrameStop();
        // MASTER受信スレッド停止
        localMaster.LocalReceiveFrameStop();
        localMaster.LocalTraceConnectsInfo(true);
    }

    // 7台接続して通信中のMasterに8台目のClient接続試行 CLIENT
    TEST_F(LocalClientFuncTest, L32_2)
    {
        nn::wlan::ConnectionStatus clientStatus;

        const TestLocalConnectParams ConnectParam = {
            //LocalMasterTestSSID   SecurityMode_StaticAes
            nn::wlan::Ssid(LocalMasterTestSSID), nn::wlan::MacAddress::CreateBroadcastMacAddress(), 1,
            { nn::wlan::SecurityMode_Open, nn::wlan::SecurityMode_Open, 0, LocalMasterPasKey }, true, nn::wlan::BeaconIndication_Enable, 10
        };

        // ジョイコン1セット（各2台）
        WLANTEST_ASSERT_TRUE(localClient.SetBtMode(LocalApiBtNode2));

        WLANTEST_EXPECT_TRUE(localClient.SetupLocalClient(ConnectParam.ssid, ConnectParam.security, TestMatchInfo, sizeof(TestMatchInfo) / sizeof(TestMatchInfo[0]), true, false));

        Connect(ConnectParam);

        // 受信スレッド起動
        WLANTEST_ASSERT_TRUE(localClient.LocalReceiveFrameStart(BufferSize100k, WlanTestPacketBufferSize, 1, 1, WlanTestGetFrameCount));
        // MASTER接続待機
        WLANTEST_ASSERT_TRUE(localClient.LocalReceiveWait(WlanMaxClientWaitMsecTimeOut, Time1s, true));
        // 送信スレッド起動
        WLANTEST_ASSERT_TRUE(localClient.LocalSendFrameOneShotWaitStart(WlanTestPacketBufferSize, Time60s + Time5s, Interval_100m, true));

        while (NN_STATIC_CONDITION(1))
        {
            // MASTERからの接続が切られるまで送信し続ける
            if (nn::os::TimedWaitSystemEvent(&connectionEvent, nn::TimeSpan::FromMilliSeconds(Interval_100m)) == true)
            {
                WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::GetConnectionStatus(&clientStatus));
                if (clientStatus.state != nn::wlan::ConnectionState_Connected)
                {
                    break;
                }
            }
        }

        // 送信スレッド停止
        localClient.LocalSendFrameStop();
        // 受信スレッド停止
        localClient.LocalReceiveFrameStop();
        // 通信結果ログ出力
        localClient.LocalTraceConnectsInfo(true);

        Disconnect();
    }

    // 1chでMasterとして起動
    TEST_F(LocalMasterFuncTest, L34)
    {
        const nn::wlan::MasterBssParameters fncTestMasterBss = {
            1, false, 30, true, nn::wlan::RateSetLegacy_11gMask, nn::wlan::RateSetLegacy_11gMask,
            { nn::wlan::SecurityMode_StaticAes, nn::wlan::SecurityMode_StaticAes, 0, LocalMasterPasKey }, nn::wlan::Ssid(LocalMasterTestSSID), 100
        };

        // ジョイコン1セット（各2台）
        WLANTEST_ASSERT_TRUE(localMaster.SetBtMode(LocalApiBtNode2));

        WLANTEST_ASSERT_RESULT_SUCCESS(SetupBss(fncTestMasterBss));
    }

    // 6chでMasterとして起動
    TEST_F(LocalMasterFuncTest, L35)
    {
        const nn::wlan::MasterBssParameters fncTestMasterBss = {
           6, false, 30, true, nn::wlan::RateSetLegacy_11gMask, nn::wlan::RateSetLegacy_11gMask,
            { nn::wlan::SecurityMode_StaticAes, nn::wlan::SecurityMode_StaticAes, 0, LocalMasterPasKey }, nn::wlan::Ssid(LocalMasterTestSSID), 100
        };

        // ジョイコン1セット（各2台）
        WLANTEST_ASSERT_TRUE(localMaster.SetBtMode(LocalApiBtNode2));

        WLANTEST_ASSERT_RESULT_SUCCESS(SetupBss(fncTestMasterBss));
    }

    // 11chでMasterとして起動
    TEST_F(LocalMasterFuncTest, L36)
    {
        const nn::wlan::MasterBssParameters fncTestMasterBss = {
            11, false, 30, false, nn::wlan::RateSetLegacy_11gMask, nn::wlan::RateSetLegacy_11gMask,
            { nn::wlan::SecurityMode_StaticAes, nn::wlan::SecurityMode_StaticAes, 0, LocalMasterPasKey }, nn::wlan::Ssid(LocalMasterTestSSID), 100
        };

        // ジョイコン1セット（各2台）
        WLANTEST_ASSERT_TRUE(localMaster.SetBtMode(LocalApiBtNode2));

        WLANTEST_ASSERT_RESULT_SUCCESS(SetupBss(fncTestMasterBss));
    }

    // Masterで180秒のキープアライブが機能 MASTER
    TEST_F(LocalMasterFuncTest, L37_1)
    {
        const nn::wlan::MasterBssParameters fncTestMasterBss = {
            6, false, 180, true, nn::wlan::RateSetLegacy_11gMask, nn::wlan::RateSetLegacy_11gMask,
            { nn::wlan::SecurityMode_StaticAes, nn::wlan::SecurityMode_StaticAes, 0, LocalMasterPasKey }, nn::wlan::Ssid(LocalMasterTestSSID), 100
        };

        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(Time15s));

        WLANTEST_ASSERT_RESULT_SUCCESS(SetupBss(fncTestMasterBss));

        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(Time240s));
    }

    // Masterで180秒のキープアライブが機能 CLIENT
    TEST_F(LocalClientFuncTest, L37_2)
    {
        const TestLocalConnectParams ConnectParam = {
            nn::wlan::Ssid(LocalMasterTestSSID),  nn::wlan::MacAddress::CreateBroadcastMacAddress(), 6,
            { nn::wlan::SecurityMode_StaticAes, nn::wlan::SecurityMode_StaticAes, 0, LocalMasterPasKey }, false, nn::wlan::BeaconIndication_Enable, 10
        };

        Connect(ConnectParam);

        NN_LOG("               %s ---> L37 Connected(%lld)\n", __FUNCTION__, (nn::os::GetSystemTick() - sysTick).ToTimeSpan().GetMilliSeconds());
        WLANTEST_ASSERT_TRUE(nn::os::TimedWaitSystemEvent(&connectionEvent, nn::TimeSpan::FromMilliSeconds(WlanTestKeepAliveLongTimeOut)));
        NN_LOG("               %s ---> L37 Disconnected(%lld)\n", __FUNCTION__, (nn::os::GetSystemTick() - sysTick).ToTimeSpan().GetMilliSeconds());

        Disconnect();

    }

    // Masterで10秒のキープアライブが機能 MASTER
    TEST_F(LocalMasterFuncTest, L38_1)
    {
        nn::wlan::MacAddress clientMacArray[ClientMaxConnect];
        const nn::wlan::MasterBssParameters fncTestMasterBss = {
            6, false, 10, true, nn::wlan::RateSetLegacy_11gMask, nn::wlan::RateSetLegacy_11gMask,
            { nn::wlan::SecurityMode_StaticAes, nn::wlan::SecurityMode_StaticAes, 0, LocalMasterPasKey }, nn::wlan::Ssid(LocalMasterTestSSID), 100
        };

        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(Time15s));

        WLANTEST_ASSERT_RESULT_SUCCESS(SetupBss(fncTestMasterBss));

        WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::GetConnectionEvent(&localMaster.m_connectEvent));

        WLANTEST_ASSERT_TRUE(localMaster.LocalMasterConnectWait(clientMacArray, WlanTestConnectTimeOut, 1) > 0);

        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(Time60s));

        nn::os::DestroySystemEvent(&localMaster.m_connectEvent);
    }

    // Masterで10秒のキープアライブが機能 CLIENT
    TEST_F(LocalClientFuncTest, L38_2)
    {
        const TestLocalConnectParams ConnectParam = {
            nn::wlan::Ssid(LocalMasterTestSSID),  nn::wlan::MacAddress::CreateBroadcastMacAddress(), 6,
            { nn::wlan::SecurityMode_StaticAes, nn::wlan::SecurityMode_StaticAes, 0, LocalMasterPasKey }, false, nn::wlan::BeaconIndication_Enable, 10
        };

        Connect(ConnectParam);

        NN_LOG("               %s ---> L38 Connected(%lld)\n", __FUNCTION__, (nn::os::GetSystemTick() - sysTick).ToTimeSpan().GetMilliSeconds());
        WLANTEST_ASSERT_TRUE(nn::os::TimedWaitSystemEvent(&connectionEvent, nn::TimeSpan::FromMilliSeconds(WlanTestKeepAliveLongTimeOut)));
        NN_LOG("               %s ---> L38 Disconnected(%lld)\n", __FUNCTION__, (nn::os::GetSystemTick() - sysTick).ToTimeSpan().GetMilliSeconds());

        Disconnect();

    }

    // Masterでキープアライブが無効化 MASTER
    TEST_F(LocalMasterFuncTest, L39_1)
    {
        nn::wlan::MacAddress clientMacArray[ClientMaxConnect];

        const nn::wlan::MasterBssParameters fncTestMasterBss = {
            6, false, 0, false, nn::wlan::RateSetLegacy_11gMask, nn::wlan::RateSetLegacy_11gMask,
            { nn::wlan::SecurityMode_StaticAes, nn::wlan::SecurityMode_StaticAes, 0, LocalMasterPasKey }, nn::wlan::Ssid(LocalMasterTestSSID), 100
        };

        WLANTEST_ASSERT_TRUE(localMaster.SetupLocalMaster(fncTestMasterBss, TestMatchInfo, sizeof(TestMatchInfo) / sizeof(TestMatchInfo[0]), false));

        WLANTEST_ASSERT_TRUE(localMaster.LocalMasterConnectWait(clientMacArray, WlanTestConnectTimeOut, 1) > 0);

        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(Time60s + Time15s));
    }

    // Masterでキープアライブが無効化 CLIENT
    TEST_F(LocalClientFuncTest, L39_2)
    {
        const TestLocalConnectParams ConnectParam = {
            nn::wlan::Ssid(LocalMasterTestSSID),  nn::wlan::MacAddress::CreateBroadcastMacAddress(), 6,
            { nn::wlan::SecurityMode_StaticAes, nn::wlan::SecurityMode_StaticAes, 0, LocalMasterPasKey }, false, nn::wlan::BeaconIndication_Enable, 10
        };

        Connect(ConnectParam);

        NN_LOG("               %s ---> L39 Connected(%lld)\n", __FUNCTION__, (nn::os::GetSystemTick() - sysTick).ToTimeSpan().GetMilliSeconds());
        WLANTEST_ASSERT_TRUE(nn::os::TimedWaitSystemEvent(&connectionEvent, nn::TimeSpan::FromMilliSeconds(WlanTestKeepAliveLongTimeOut)));
        NN_LOG("               %s ---> L39 Disconnected(%lld)\n", __FUNCTION__, (nn::os::GetSystemTick() - sysTick).ToTimeSpan().GetMilliSeconds());

        Disconnect();
    }

    // Beacon間隔が常に100msec間隔であること (要キャプチャ)
    TEST_F(LocalMasterFuncTest, L43)
    {
        const nn::wlan::MasterBssParameters fncTestMasterBss = {
            11, false, 30, true, nn::wlan::RateSetLegacy_11gMask, nn::wlan::RateSetLegacy_11gMask,
            { nn::wlan::SecurityMode_Open, nn::wlan::SecurityMode_Open, 0, "" }, nn::wlan::Ssid(LocalMasterTestSSID), 100
        };

        // ジョイコン1セット（各2台）
        WLANTEST_ASSERT_TRUE(localMaster.SetBtMode(LocalApiBtNode2));

        WLANTEST_ASSERT_TRUE(localMaster.SetupLocalMaster(fncTestMasterBss, TestMatchInfo, sizeof(TestMatchInfo) / sizeof(TestMatchInfo[0])));
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(Time5s));
        WLANTEST_EXPECT_RESULT_SUCCESS(nn::wlan::Local::DestroyBss());
        //キャプチャ時に判別し易くするため 5 秒スリープ
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(Time5s));
    }

    // Beaconに含める、BasciRatesを任意の設定にできること (要キャプチャ)
    TEST_F(LocalMasterFuncTest, L44)
    {
        nn::wlan::RateSetLegacy RateSetLegacyArray[] = {
            nn::wlan::RateSetLegacy_Default,
            nn::wlan::RateSetLegacy_1_0m,
            nn::wlan::RateSetLegacy_2_0m,
            nn::wlan::RateSetLegacy_5_5m,
            nn::wlan::RateSetLegacy_11_0m,
            nn::wlan::RateSetLegacy_6_0m,
            nn::wlan::RateSetLegacy_9_0m,
            nn::wlan::RateSetLegacy_12_0m,
            nn::wlan::RateSetLegacy_18_0m,
            nn::wlan::RateSetLegacy_24_0m,
            nn::wlan::RateSetLegacy_36_0m,
            nn::wlan::RateSetLegacy_48_0m,
            nn::wlan::RateSetLegacy_54_0m,
            nn::wlan::RateSetLegacy_11bMask,
            nn::wlan::RateSetLegacy_11gMask
        };

        for (auto legacy : RateSetLegacyArray)
        {
            const nn::wlan::MasterBssParameters fncTestMasterBss = {
                11, false, 30, true, nn::wlan::RateSetLegacy_12_0m, legacy,
                { nn::wlan::SecurityMode_Open, nn::wlan::SecurityMode_Open, 0, "" }, nn::wlan::Ssid(LocalMasterTestLongSSID), 10
            };

            WLANTEST_ASSERT_TRUE(localMaster.SetupLocalMaster(fncTestMasterBss, TestMatchInfo, sizeof(TestMatchInfo) / sizeof(TestMatchInfo[0])));
            nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(Time5s));
            WLANTEST_EXPECT_RESULT_SUCCESS(nn::wlan::Local::DestroyBss());
            //キャプチャ時に判別し易くするため 5 秒スリープ
            nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(Time5s));
        }
    }

    // Beaconに含める、SupportedRatesを任意の設定にできること (要キャプチャ)
    TEST_F(LocalMasterFuncTest, L45)
    {
        nn::wlan::RateSetLegacy RateSetLegacyArray[] = {
            nn::wlan::RateSetLegacy_Default,
            nn::wlan::RateSetLegacy_1_0m,
            nn::wlan::RateSetLegacy_2_0m,
            nn::wlan::RateSetLegacy_5_5m,
            nn::wlan::RateSetLegacy_11_0m,
            nn::wlan::RateSetLegacy_6_0m,
            nn::wlan::RateSetLegacy_9_0m,
            nn::wlan::RateSetLegacy_12_0m,
            nn::wlan::RateSetLegacy_18_0m,
            nn::wlan::RateSetLegacy_24_0m,
            nn::wlan::RateSetLegacy_36_0m,
            nn::wlan::RateSetLegacy_48_0m,
            nn::wlan::RateSetLegacy_54_0m,
            nn::wlan::RateSetLegacy_11bMask,
            nn::wlan::RateSetLegacy_11gMask
        };

        for (auto legacy : RateSetLegacyArray)
        {
            const nn::wlan::MasterBssParameters fncTestMasterBss = {
                1, false, 30, true, legacy, nn::wlan::RateSetLegacy_18_0m,
                { nn::wlan::SecurityMode_Open, nn::wlan::SecurityMode_Open, 0, "" }, nn::wlan::Ssid(LocalMasterTestLongSSID), 10
            };

            WLANTEST_ASSERT_TRUE(localMaster.SetupLocalMaster(fncTestMasterBss, TestMatchInfo, sizeof(TestMatchInfo) / sizeof(TestMatchInfo[0])));
            nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(Time5s));
            WLANTEST_EXPECT_RESULT_SUCCESS(nn::wlan::Local::DestroyBss());
            //キャプチャ時に判別し易くするため 5 秒スリープ
            nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(Time5s));
        }
    }

    // 32文字のSSID設定
    TEST_F(LocalMasterFuncTest, L47)
    {
        const nn::wlan::MasterBssParameters fncTestMasterBss = {
            6, false, 30, true, nn::wlan::RateSetLegacy_11gMask, nn::wlan::RateSetLegacy_11gMask,
            { nn::wlan::SecurityMode_StaticAes, nn::wlan::SecurityMode_StaticAes, 0, LocalMasterPasKey }, nn::wlan::Ssid(LocalMasterTestLongSSID), 100
        };

        WLANTEST_ASSERT_RESULT_SUCCESS(SetupBss(fncTestMasterBss));

        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(Time15s));
    }

    // 任意のIEを付加キャプチャ
    // 付加したIEを消去キャプチャ MASTER
    TEST_F(LocalMasterFuncTest, L48_L49_1)
    {
        std::unique_ptr<uint8_t[]> setBuffer(new uint8_t[WlanTestPacketBufferSize]);

        WLANTEST_ASSERT_TRUE(setBuffer.get() != nullptr);
        uint32_t ieId;

        const nn::wlan::MasterBssParameters fncTestMasterBss = {
            11, false, 30, true, nn::wlan::RateSetLegacy_11gMask, nn::wlan::RateSetLegacy_11gMask,
            { nn::wlan::SecurityMode_StaticAes, nn::wlan::SecurityMode_StaticAes, 0, LocalMasterPasKey }, nn::wlan::Ssid(LocalMasterTestSSID), 100
        };

        WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::GetMacAddress(&localMaster.m_macAddress));

        // IE追加／削除処理をCreateBssより先にしないとパケットにでないため呼び出し順序を変更及びCreateBssをIE毎に作り直しに変更
        for (auto testIe : LocalTestIe)
        {
            // IE追加
            nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(Time1s));
            WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::AddIe(&ieId, testIe.frameType, testIe.vendrId, testIe.vendrLength));

            // ActionFrame定期送信Actionフレームバッファ生成
            WLANTEST_ASSERT_TRUE(localMaster.LocalMakeFrame(setBuffer.get(), WlanTestPacketBufferSize,
                nn::wlan::MacAddress::CreateBroadcastMacAddress(), localMaster.m_macAddress, EtherTypes[0], TestMatchInfo[0]));
            setBuffer.get()[0] = testIe.vendrId[0];
            std::memcpy(&setBuffer.get()[1], &testIe.vendrId[1], 3);
            setBuffer.get()[4] = testIe.frameType;

            // ActionFrame定期送信
            nn::wlan::Local::SetActionFrameWithBeacon(setBuffer.get(), 20);
            nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(Time5s));

            WLANTEST_ASSERT_TRUE(localMaster.SetupLocalMaster(fncTestMasterBss, TestMatchInfo, sizeof(TestMatchInfo) / sizeof(TestMatchInfo[0])));

            // MASTE接続待機
            WLANTEST_ASSERT_TRUE(localMaster.LocalMasterConnectWait(WlanTestMasterWaitTimeOut, 1) > 0);

            AllScan();

            WLANTEST_EXPECT_TRUE(localMaster.LocalMasterDisConnectWait(Time5s) > 0);

            localMaster.LocalRelease();

            // IE削除
            nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(Time1s));
            WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::DeleteIe(ieId));
        }
    }

    // 任意のIEを付加キャプチャ
    // 付加したIEを消去キャプチャ CLIENT
    TEST_F(LocalClientFuncTest, L48_L49_2)
    {
        uint32_t ieId;
        const TestLocalConnectParams ConnectParam = {
            nn::wlan::Ssid(LocalMasterTestSSID), nn::wlan::MacAddress::CreateBroadcastMacAddress(), 11,
            { nn::wlan::SecurityMode_StaticAes, nn::wlan::SecurityMode_StaticAes, 0, LocalMasterPasKey }, true, nn::wlan::BeaconIndication_Enable, 10
        };

        static const LocalTestIeInfo clientTestIe[] = {
            { nn::wlan::ManagementFrameType_AssocReq, { 0x00, 0x22, 0xAA, 0x12, 0x34, 0x56, 0x83 }, 7 }
        };

        WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::AddIe(&ieId, nn::wlan::ManagementFrameType_AssocReq, clientTestIe[0].vendrId, clientTestIe[0].vendrLength));
        WLANTEST_ASSERT_TRUE(localClient.SetupLocalClient(ConnectParam.ssid, ConnectParam.security, TestMatchInfo, sizeof(TestMatchInfo) / sizeof(TestMatchInfo[0]), true, false));

        // IE毎にCLIENT接続するよう変更
        for (int32_t i = 0; i < sizeof(LocalTestIe) / sizeof(LocalTestIe[0]); i++)
        {
            Connect(ConnectParam);

            // 切断イベント待機
            NN_LOG("               %s ---> L48 L49 Disconnect Wait Start(%lld)\n", __FUNCTION__, (nn::os::GetSystemTick() - sysTick).ToTimeSpan().GetMilliSeconds());
            nn::os::TimedWaitSystemEvent(&connectionEvent, nn::TimeSpan::FromMilliSeconds(Time60s));
            NN_LOG("               %s ---> L48 L49 Disconnect Wait End(%lld)\n", __FUNCTION__, (nn::os::GetSystemTick() - sysTick).ToTimeSpan().GetMilliSeconds());

            Disconnect();
        }

        WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::DeleteIe(ieId));
    }

    // Masterから任意のClient切断
    TEST_F(LocalMasterFuncTest, L50)
    {
        nn::wlan::MacAddress clientMacArray[ClientMaxConnect];
        nn::wlan::DisconnectClient client;
        uint32_t clientCount = 0;

        const nn::wlan::MasterBssParameters fncTestMasterBss = {
            11, false, 30, true, nn::wlan::RateSetLegacy_11gMask, nn::wlan::RateSetLegacy_11gMask,
            { nn::wlan::SecurityMode_Open, nn::wlan::SecurityMode_Open, 0, "" }, nn::wlan::Ssid(LocalMasterTestSSID), 100
        };

        WLANTEST_ASSERT_TRUE(localMaster.SetupLocalMaster(fncTestMasterBss, TestMatchInfo, sizeof(TestMatchInfo) / sizeof(TestMatchInfo[0]), false));

        clientCount = localMaster.LocalMasterConnectWait(clientMacArray, WlanTestConnectTimeOut, 2);
        WLANTEST_EXPECT_TRUE(clientCount >= 2);

        for (int32_t i = 0; i < clientCount; i++)
        {
            client.clientMacAddress = clientMacArray[i];
            client.reasonCode = nn::wlan::Dot11ReasonCode_DisassocInactivity;
            WLANTEST_EXPECT_RESULT_SUCCESS(nn::wlan::Local::Disconnect(nn::wlan::LocalCommunicationMode_Master, &client));
        }
    }

    // ClientがMasterとの接続を自発的に切断
    TEST_F(LocalClientFuncTest, L51)
    {
        const TestLocalConnectParams ConnectParam = {
            nn::wlan::Ssid(LocalMasterTestSSID),  nn::wlan::MacAddress::CreateBroadcastMacAddress(), 1,
            { nn::wlan::SecurityMode_Open, nn::wlan::SecurityMode_Open, 0, "" }, true, nn::wlan::BeaconIndication_Enable, 10
        };

        Connect(ConnectParam);

        WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::Disconnect(nn::wlan::LocalCommunicationMode_ClientSpectator, nullptr));
    }

    // ClientがMasterとの接続を自発的に切断
    TEST_F(LocalMasterFuncTest, L52)
    {
        nn::wlan::MacAddress clientMacArray[ClientMaxConnect];
        uint32_t clientCount = 0;
        const nn::wlan::MasterBssParameters fncTestMasterBss = {
            11, false, 30, true, nn::wlan::RateSetLegacy_11gMask, nn::wlan::RateSetLegacy_11gMask,
            { nn::wlan::SecurityMode_StaticAes, nn::wlan::SecurityMode_StaticAes, 0, LocalMasterPasKey }, nn::wlan::Ssid(LocalMasterTestSSID), 100
        };

        WLANTEST_ASSERT_TRUE(localMaster.SetupLocalMaster(fncTestMasterBss, TestMatchInfo, sizeof(TestMatchInfo) / sizeof(TestMatchInfo[0])));

        clientCount = localMaster.LocalMasterConnectWait(clientMacArray, WlanTestConnectTimeOut, 1);
        WLANTEST_EXPECT_TRUE(clientCount == 1);

        WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::DestroyBss());
    }

    // Client Beaconロスト時間を設定(Client Beacon設定) MASTER
    TEST_F(LocalMasterFuncTest, L53_1)
    {
        const nn::wlan::MasterBssParameters fncTestMasterBss = {
            11, false, 15, true, nn::wlan::RateSetLegacy_11gMask, nn::wlan::RateSetLegacy_11gMask,
            { nn::wlan::SecurityMode_Open, nn::wlan::SecurityMode_Open, 0, "" }, nn::wlan::Ssid(LocalMasterTestSSID), 100
        };

        WLANTEST_ASSERT_TRUE(localMaster.SetupLocalMaster(fncTestMasterBss, TestMatchInfo, sizeof(TestMatchInfo) / sizeof(TestMatchInfo[0])));

        StartTcpServer();

        localMaster.LocalMasterConnectWait(connectMacArray, WlanTestConnectTimeOut, 1);

        StopTcpServer();

        // MASTERを停止しCLIENT側をパケットロスにするためのABORT停止
        NN_LOG("Master Frame Lost Test Master Pawer Off Please\n");
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(Time1day));
    }

    // Client Beaconロスト時間を設定(Client Beacon設定) CLIENT
    TEST_F(LocalClientFuncTest, L53_2)
    {
        const TestLocalConnectParams ConnectParam = {
            nn::wlan::Ssid(LocalMasterTestSSID),  nn::wlan::MacAddress::CreateBroadcastMacAddress(), 11,
            { nn::wlan::SecurityMode_Open, nn::wlan::SecurityMode_Open, 0, "" }, true, nn::wlan::BeaconIndication_Enable, 10
        };

        Connect(ConnectParam);

        sysTick = nn::os::GetSystemTick();
        NN_LOG("               %s ---> L52 Client Beacon設定 Connected(%lld)\n", __FUNCTION__, (nn::os::GetSystemTick() - sysTick).ToTimeSpan().GetMilliSeconds());
        WLANTEST_ASSERT_TRUE(nn::os::TimedWaitSystemEvent(&connectionEvent, nn::TimeSpan::FromSeconds(WlanTestConnectTimeOut)));
        NN_LOG("               %s ---> L52 Client Beacon設定 Disconnected(%lld)\n", __FUNCTION__, (nn::os::GetSystemTick() - sysTick).ToTimeSpan().GetMilliSeconds());

        Disconnect();
    }

    // Client Beaconロスト時間を設定(Master inactivePeriod設定) MASTER
    TEST_F(LocalMasterFuncTest, L53_3)
    {
        const nn::wlan::MasterBssParameters fncTestMasterBss = {
            1, false, 15, true, nn::wlan::RateSetLegacy_11gMask, nn::wlan::RateSetLegacy_11gMask,
            { nn::wlan::SecurityMode_Open, nn::wlan::SecurityMode_Open, 0, "" }, nn::wlan::Ssid(LocalMasterTest2SSID), 100
        };

        WLANTEST_ASSERT_TRUE(localMaster.SetupLocalMaster(fncTestMasterBss, TestMatchInfo, sizeof(TestMatchInfo) / sizeof(TestMatchInfo[0])));

        StartTcpServer();

        localMaster.LocalMasterConnectWait(connectMacArray, WlanTestConnectTimeOut, 1);

        NN_LOG("               %s ---> L52 Master inactivePeriod設定 Event Wait(%lld)\n", __FUNCTION__, (nn::os::GetSystemTick() - sysTick).ToTimeSpan().GetMilliSeconds());
        WLANTEST_ASSERT_TRUE(nn::os::TimedWaitSystemEvent(&connectionEvent, nn::TimeSpan::FromSeconds(Time120s)));
        NN_LOG("               %s ---> L52 Master inactivePeriod設定 Event Sygnal(%lld)\n", __FUNCTION__, (nn::os::GetSystemTick() - sysTick).ToTimeSpan().GetMilliSeconds());

        StopTcpServer();
    }

    // Client Beaconロスト時間を設定(Master inactivePeriod設定) CLIENT
    TEST_F(LocalClientFuncTest, L53_4)
    {
        const TestLocalConnectParams ConnectParam = {
            nn::wlan::Ssid(LocalMasterTest2SSID),  nn::wlan::MacAddress::CreateBroadcastMacAddress(), 1,
            { nn::wlan::SecurityMode_Open, nn::wlan::SecurityMode_Open, 0, "" }, true, nn::wlan::BeaconIndication_Enable, 15
        };

        Connect(ConnectParam);

        // CLIENTを停止しMASTER側の切断イベント発生させるためのABORT停止
        NN_LOG("Client Connection Lost Test Pawer Off Please\n");
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(Time1day));

        StopTcpServer();
    }

    // 1chの無線負荷が高い場合、6ch または 11chが選択される MASTER 1CH
    TEST_F(LocalMasterFuncTest, L59_1)
    {
        const nn::wlan::MasterBssParameters fncTestMasterBss = {
            1, false, 15, true, nn::wlan::RateSetLegacy_11gMask, nn::wlan::RateSetLegacy_11gMask,
            { nn::wlan::SecurityMode_Open, nn::wlan::SecurityMode_Open, 0, "" }, nn::wlan::Ssid(LocalMasterTestSSID), 100
        };

        WLANTEST_ASSERT_TRUE(localMaster.SetupLocalMaster(fncTestMasterBss, TestMatchInfo, sizeof(TestMatchInfo) / sizeof(TestMatchInfo[0])));

        StartTcpServer();

        // MASTER受信スレッド起動
        WLANTEST_ASSERT_TRUE(localMaster.LocalReceiveFrameStart(BufferSize100k, WlanTestPacketBufferSize, 1, 1, WlanTestGetFrameCount));
        WLANTEST_ASSERT_TRUE(localMaster.LocalReceiveWait(WlanTestMasterWaitTimeOut, true));

        // MASTER送信スレッド起動
        WLANTEST_ASSERT_TRUE(localMaster.LocalSendFrameStart(WlanTestPacketBufferSize, Time60s, 0, true));

        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(Time60s));

        // MASTER送信スレッド停止
        localMaster.LocalSendFrameStop();

        StopTcpServer();

        localMaster.LocalReceiveFrameStop();
    }

    // 1chの無線負荷が高い場合、6ch または 11chが選択される CLIENT 1CH
    TEST_F(LocalClientFuncTest, L59_2)
    {
        const TestLocalConnectParams ConnectParam = {
            nn::wlan::Ssid(LocalMasterTestSSID),  nn::wlan::MacAddress::CreateBroadcastMacAddress(), 1,
            { nn::wlan::SecurityMode_Open, nn::wlan::SecurityMode_Open, 0, "" }, true, nn::wlan::BeaconIndication_Enable, 15
        };

        WLANTEST_ASSERT_TRUE(localClient.SetupLocalClient(ConnectParam.ssid, ConnectParam.security, TestMatchInfo, sizeof(TestMatchInfo) / sizeof(TestMatchInfo[0]), true, false));

        Connect(ConnectParam);

        StartTcpClient(STATIC_CLIENT1_IP_ADDR);

        // 受信スレッド起動
        WLANTEST_ASSERT_TRUE(localClient.LocalReceiveFrameStart(BufferSize100k, WlanTestPacketBufferSize, 1, 1, WlanTestGetFrameCount));
        WLANTEST_ASSERT_TRUE(localClient.LocalReceiveWait(WlanMaxClientWaitMsecTimeOut, Time1s, true));

        // 送信スレッド起動
        WLANTEST_ASSERT_TRUE(localClient.LocalSendFrameStart(WlanTestPacketBufferSize, Time60s, 0, false));

        while (NN_STATIC_CONDITION(1))
        {
            // MASTERからの接続が切られるまで送信し続ける
            if (nn::os::TimedWaitSystemEvent(&connectionEvent, nn::TimeSpan::FromMilliSeconds(Interval_100m)) == true)
            {
                WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::GetConnectionStatus(&connectionStatus));
                if (connectionStatus.state != nn::wlan::ConnectionState_Connected)
                {
                    break;
                }
            }
        }

        // 送信スレッド停止
        localClient.LocalSendFrameStop();

        localClient.LocalReceiveFrameStop();
        StopTcpClient();
        Disconnect();
    }

    // 1chの無線負荷が高い場合、6ch または 11chが選択される MASTER 自動CH
    TEST_F(LocalMasterFuncTest, L59_3)
    {
        std::unique_ptr<uint8_t[]> putBuffer(new uint8_t[WlanTestPacketBufferSize]);
        nn::wlan::MacAddress myMacAddress;
        nn::wlan::ConnectionStatus connectStatus;

        WLANTEST_ASSERT_TRUE(putBuffer.get() != nullptr);

        const nn::wlan::MasterBssParameters fncTestMasterBss = {
            -1, false, 15, true, nn::wlan::RateSetLegacy_11gMask, nn::wlan::RateSetLegacy_11gMask,
            { nn::wlan::SecurityMode_Open, nn::wlan::SecurityMode_Open, 0, "" }, nn::wlan::Ssid(LocalMasterTestSSID), 100
        };

        WLANTEST_ASSERT_TRUE(localMaster.SetupLocalMaster(fncTestMasterBss, TestMatchInfo, sizeof(TestMatchInfo) / sizeof(TestMatchInfo[0])));

        WLANTEST_EXPECT_RESULT_SUCCESS(nn::wlan::Local::GetConnectionStatus(&connectStatus));
        WLANTEST_ASSERT_TRUE(connectStatus.channel == 6 || connectStatus.channel == 11);
    }

    // 受信エントリに登録したEtherTypeを持つパケット受信
    TEST_F(LocalMasterFuncTest, L60)
    {
        std::unique_ptr<uint8_t[]> getBuffer(new uint8_t[BufferSize100k]);
        nn::wlan::MacAddress clientMacArray[ClientMaxConnect];
        LocalFrameHeader* pFrameHeader = nullptr;
        size_t outSize = 0;


        const nn::wlan::MasterBssParameters fncTestMasterBss = {
            6, false, 30, true, nn::wlan::RateSetLegacy_11gMask, nn::wlan::RateSetLegacy_11gMask,
            { nn::wlan::SecurityMode_Open, nn::wlan::SecurityMode_Open, 0, "" }, nn::wlan::Ssid(LocalMasterTestSSID), 100
        };

        WLANTEST_ASSERT_TRUE(getBuffer.get() != nullptr);

        localMaster.SetupLocalMaster(fncTestMasterBss, TestMatchInfo, sizeof(TestMatchInfo) / sizeof(TestMatchInfo[0]), false);

        WLANTEST_ASSERT_TRUE(localMaster.LocalMasterConnectWait(clientMacArray, WlanTestConnectTimeOut, 1) > 0);

        for (int32_t cnt = 0; cnt < 2; cnt++)
        {
            WLANTEST_ASSERT_TRUE(localMaster.LocalCancelFrameStart(WlanTestCancelFrameWaitTime));
            WLANTEST_EXPECT_RESULT_SUCCESS(nn::wlan::Local::GetFrameRaw(getBuffer.get(), BufferSize100k, &outSize, localMaster.m_entRxId));
            localMaster.LocalCancelFrameStop();

            WLANTEST_EXPECT_TRUE(outSize >= WlanTestGetFramMineSize);

            pFrameHeader = reinterpret_cast<LocalFrameHeader*>(getBuffer.get());
            WLANTEST_EXPECT_TRUE(CompareEtherType(EtherTypes[0],pFrameHeader->etherType));
        }

    }

    // 受信エントリにEtherTypeを追加受信 MASTER
    TEST_F(LocalMasterFuncTest, L61_1)
    {
        // 時刻ライブラリの初期化
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::time::Initialize());

        nn::time::PosixTime posixTime;

        std::unique_ptr<uint8_t[]> getBuffer(new uint8_t[DataFrameBufferSize]);
        nn::wlan::MacAddress clientMacArray[ClientMaxConnect];
        LocalFrameHeader* pFrameHeader = nullptr;
        size_t outSize = 0;

        const nn::wlan::MasterBssParameters fncTestMasterBss = {
            11, false, 30, true, nn::wlan::RateSetLegacy_11gMask, nn::wlan::RateSetLegacy_11gMask,
            { nn::wlan::SecurityMode_Open, nn::wlan::SecurityMode_Open, 0, "" }, nn::wlan::Ssid(LocalMasterTestSSID), 100
        };

        const uint16_t testEtherType = 0x8878;

        WLANTEST_ASSERT_TRUE(getBuffer.get() != nullptr);

        localMaster.SetupLocalMaster(fncTestMasterBss, TestMatchInfo, sizeof(TestMatchInfo) / sizeof(TestMatchInfo[0]), false);

        WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::AddEthertypeToRxEntry(localMaster.m_entRxId, testEtherType));

        WLANTEST_ASSERT_TRUE(localMaster.LocalMasterConnectWait(clientMacArray, WlanTestConnectTimeOut, 1) > 0);

        //時刻の取得
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::time::StandardUserSystemClock::GetCurrentTime(&posixTime));

        // PosixTime を CalendarTime へ変換します。
        nn::time::CalendarTime calendarTime;
        nn::time::CalendarAdditionalInfo calendarAdditionalInfo;
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::time::ToCalendarTime(&calendarTime, &calendarAdditionalInfo, posixTime));

        NN_LOG("Master Connect Waited Time: %04d/%02d/%02d %02d:%02d:%02d\n",
            calendarTime.year, calendarTime.month, calendarTime.day, calendarTime.hour, calendarTime.minute, (60 + calendarTime.second - 6) % 60);

        for (int32_t cnt = 0; cnt < 2; cnt++)
        {
            WLANTEST_EXPECT_TRUE(localMaster.LocalCancelFrameStart(WlanTestCancelFrameWaitTime));
            WLANTEST_EXPECT_RESULT_SUCCESS(nn::wlan::Local::GetFrameRaw(getBuffer.get(), DataFrameBufferSize, &outSize, localMaster.m_entRxId));
            localMaster.LocalCancelFrameStop();

            WLANTEST_EXPECT_TRUE(outSize >= WlanTestGetFramMineSize);

            pFrameHeader = reinterpret_cast<LocalFrameHeader*>(getBuffer.get());
            WLANTEST_EXPECT_TRUE(CompareEtherType(testEtherType, pFrameHeader->etherType));
        }

        // 時刻ライブラリの終了
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::time::Finalize());
    }

    // 受信エントリにEtherTypeを追加受信 CLIENT
    TEST_F(LocalClientFuncTest, L61_2)
    {
        // 時刻ライブラリの初期化
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::time::Initialize());

        nn::time::PosixTime posixTime;

        std::unique_ptr<uint8_t[]> putBuffer(new uint8_t[DataFrameBufferSize]);
        uint32_t seqNumber = 0;

        WLANTEST_ASSERT_TRUE(putBuffer.get() != nullptr);

        const TestLocalConnectParams ConnectParam = {
            nn::wlan::Ssid(LocalMasterTestSSID), nn::wlan::MacAddress::CreateBroadcastMacAddress(), 11,
            { nn::wlan::SecurityMode_Open, nn::wlan::SecurityMode_Open, 0, "" }, true, nn::wlan::BeaconIndication_Enable, 10
        };

        const uint16_t testEtherType = 0x8878;

        WLANTEST_ASSERT_TRUE(localClient.SetupLocalClient(ConnectParam.ssid, ConnectParam.security, TestMatchInfo, sizeof(TestMatchInfo) / sizeof(TestMatchInfo[0]), true, true));

        Connect(ConnectParam);

        //時刻の取得
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::time::StandardUserSystemClock::GetCurrentTime(&posixTime));

        // PosixTime を CalendarTime へ変換します。
        nn::time::CalendarTime calendarTime;
        nn::time::CalendarAdditionalInfo calendarAdditionalInfo;
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::time::ToCalendarTime(&calendarTime, &calendarAdditionalInfo, posixTime));

        NN_LOG("Client Connect Time: %04d/%02d/%02d %02d:%02d:%02d\n",
            calendarTime.year, calendarTime.month, calendarTime.day, calendarTime.hour, calendarTime.minute, (calendarTime.second + 17) % 60);

        WLANTEST_EXPECT_RESULT_SUCCESS(nn::wlan::Local::GetConnectionStatus(&connectionStatus));

        for (int32_t cnt = 0; cnt < 2; cnt++)
        {
            WLANTEST_EXPECT_TRUE(localClient.LocalMakeFrame(putBuffer.get(), DataFrameBufferSize, connectionStatus.bssid, localClient.m_macAddress, testEtherType, TestMatchInfo[0]));
            WLANTEST_EXPECT_TRUE(localClient.LocalSeqNumberAddFrame(putBuffer.get(), DataFrameBufferSize, seqNumber++));
            WLANTEST_EXPECT_RESULT_SUCCESS(nn::wlan::Local::PutFrameRaw(putBuffer.get(), DataFrameBufferSize));
        }

        // 送信遅延待機
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(1));

        Disconnect();

        // 時刻ライブラリの終了
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::time::Finalize());
    }

    // 受信エントリに登録したEtherTypeを持つパケットは受信しない MASTER
    TEST_F(LocalMasterFuncTest, L62_1)
    {
        std::unique_ptr<uint8_t[]> getBuffer(new uint8_t[DataFrameBufferSize]);
        nn::wlan::MacAddress clientMacArray[ClientMaxConnect];
        LocalFrameHeader* pFrameHeader = nullptr;
        size_t outSize = 0;

        const nn::wlan::MasterBssParameters fncTestMasterBss = {
            6, false, 30, true, nn::wlan::RateSetLegacy_11gMask, nn::wlan::RateSetLegacy_11gMask,
            { nn::wlan::SecurityMode_Open, nn::wlan::SecurityMode_Open, 0, "" }, nn::wlan::Ssid(LocalMasterTestSSID), 100
        };

        const uint16_t testEtherType = 0x8800;

        WLANTEST_ASSERT_TRUE(getBuffer.get() != nullptr);

        WLANTEST_ASSERT_TRUE(localMaster.SetupLocalMaster(fncTestMasterBss, TestMatchInfo, sizeof(TestMatchInfo) / sizeof(TestMatchInfo[0]), false));

        WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::AddEthertypeToRxEntry(localMaster.m_entRxId, testEtherType));

        WLANTEST_ASSERT_TRUE(localMaster.LocalMasterConnectWait(clientMacArray, WlanTestConnectTimeOut, 1) > 0);

        for (int32_t cnt = 0; cnt < 2; cnt++)
        {
            WLANTEST_EXPECT_TRUE(localMaster.LocalCancelFrameStart(WlanTestCancelFrameWaitTime));
            WLANTEST_EXPECT_RESULT_SUCCESS(nn::wlan::Local::GetFrameRaw(getBuffer.get(), DataFrameBufferSize, &outSize, localMaster.m_entRxId));
            localMaster.LocalCancelFrameStop();

            WLANTEST_EXPECT_TRUE(outSize >= WlanTestGetFramMineSize);

            pFrameHeader = reinterpret_cast<LocalFrameHeader*>(getBuffer.get());
            WLANTEST_EXPECT_FALSE(CompareEtherType(testEtherType, pFrameHeader->etherType));
        }
    }

    // 受信エントリに登録したEtherTypeを持つパケットは受信しない CLIENT
    TEST_F(LocalClientFuncTest, L62_2)
    {
        std::unique_ptr<uint8_t[]> putBuffer(new uint8_t[DataFrameBufferSize]);
        uint32_t seqNumber = 0;

        WLANTEST_ASSERT_TRUE(putBuffer.get() != nullptr);

        const TestLocalConnectParams ConnectParam = {
            nn::wlan::Ssid(LocalMasterTestSSID), nn::wlan::MacAddress::CreateBroadcastMacAddress(), 6,
            { nn::wlan::SecurityMode_Open, nn::wlan::SecurityMode_Open, 0, "" }, true, nn::wlan::BeaconIndication_Enable, 10
        };

        WLANTEST_ASSERT_TRUE(localClient.SetupLocalClient(ConnectParam.ssid, ConnectParam.security, TestMatchInfo, sizeof(TestMatchInfo) / sizeof(TestMatchInfo[0]), true, true));

        Connect(ConnectParam);

        WLANTEST_EXPECT_RESULT_SUCCESS(nn::wlan::Local::GetConnectionStatus(&connectionStatus));

        for (int32_t cnt = 0; cnt < 2; cnt++)
        {
            WLANTEST_EXPECT_TRUE(localClient.LocalMakeFrame(putBuffer.get(), DataFrameBufferSize, connectionStatus.bssid, localClient.m_macAddress, EtherTypes[0], TestMatchInfo[0]));
            WLANTEST_EXPECT_TRUE(localClient.LocalSeqNumberAddFrame(putBuffer.get(), DataFrameBufferSize, seqNumber++));
            WLANTEST_EXPECT_RESULT_SUCCESS(nn::wlan::Local::PutFrameRaw(putBuffer.get(), DataFrameBufferSize));
        }

        // 送信遅延待機
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(1));

        Disconnect();
    }

    // 受信エントリへのEtherTypeの登録削除を複数回行える
    // 受信エントリにEtherTypeを最大2個登録
    TEST_F(LocalMasterFuncTest, L63_L65)
    {
        uint32_t rxId;

        const nn::wlan::MasterBssParameters fncTestMasterBss = {
            6, false, 30, true, nn::wlan::RateSetLegacy_11gMask, nn::wlan::RateSetLegacy_11gMask,
            { nn::wlan::SecurityMode_Open, nn::wlan::SecurityMode_Open, 0, "" }, nn::wlan::Ssid(LocalMasterTestSSID), 100
        };

        const uint16_t testEtherTypes[] = { 0x8900, 0x99FF };

        WLANTEST_ASSERT_RESULT_SUCCESS(SetupBss(fncTestMasterBss));

        WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::CreateRxEntry(&rxId, &testEtherTypes[0], sizeof(testEtherTypes[0]) / sizeof(testEtherTypes[0]), 30));
        for (int i = 0; i < 10; i++)
        {
            WLANTEST_EXPECT_RESULT_SUCCESS(nn::wlan::Local::AddEthertypeToRxEntry(rxId, testEtherTypes[1]));
            WLANTEST_EXPECT_RESULT_SUCCESS(nn::wlan::Local::DeleteEthertypeFromRxEntry(&rxId, testEtherTypes[1]));
        }
    }

    // 削除した受信エントリのEtherTypeは削除処理以降受信しない MASTER
    TEST_F(LocalMasterFuncTest, L64_1)
    {
        std::unique_ptr<uint8_t[]> getBuffer(new uint8_t[BufferSize100k]);
        nn::wlan::MacAddress clientMacArray[ClientMaxConnect];
        LocalFrameHeader* pFrameHeader = nullptr;
        uint32_t outRxId;
        size_t outSize = 0;

        const nn::wlan::MasterBssParameters fncTestMasterBss = {
            6, false, 30, true, nn::wlan::RateSetLegacy_11gMask, nn::wlan::RateSetLegacy_11gMask,
            { nn::wlan::SecurityMode_Open, nn::wlan::SecurityMode_Open, 0, "" }, nn::wlan::Ssid(LocalMasterTestSSID), 100
        };

        bool isMatchEtherType = false;

        WLANTEST_ASSERT_TRUE(getBuffer.get() != nullptr);

        localMaster.SetupLocalMaster(fncTestMasterBss, TestMatchInfo, sizeof(TestMatchInfo) / sizeof(TestMatchInfo[0]), false);

        WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::AddEthertypeToRxEntry(localMaster.m_entRxId, TestEtherTypes[1]));

        WLANTEST_ASSERT_TRUE(localMaster.LocalMasterConnectWait(clientMacArray, WlanTestConnectTimeOut, 1) > 0);
        for (int32_t cnt = 0; cnt < 2; cnt++)
        {
            WLANTEST_ASSERT_TRUE(localMaster.LocalCancelFrameStart(WlanTestCancelFrameWaitTime));
            WLANTEST_EXPECT_RESULT_SUCCESS(nn::wlan::Local::GetFrameRaw(getBuffer.get(), BufferSize100k, &outSize, localMaster.m_entRxId));
            localMaster.LocalCancelFrameStop();

            WLANTEST_EXPECT_TRUE(outSize >= WlanTestGetFramMineSize);

            pFrameHeader = reinterpret_cast<LocalFrameHeader*>(getBuffer.get());

            isMatchEtherType = false;
            for (auto etherType : TestEtherTypes)
            {
                if (CompareEtherType(etherType, pFrameHeader->etherType) == true)
                {
                    isMatchEtherType = true;
                    break;
                }
            }

            WLANTEST_EXPECT_TRUE(isMatchEtherType);
        }

        outRxId = localMaster.m_entRxId;
        WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::DeleteEthertypeFromRxEntry(&outRxId, TestEtherTypes[1]));

        for (int32_t cnt = 0; cnt < 2; cnt++)
        {
            WLANTEST_ASSERT_TRUE(localMaster.LocalCancelFrameStart(WlanTestCancelFrameWaitTime));
            WLANTEST_EXPECT_RESULT_SUCCESS(nn::wlan::Local::GetFrameRaw(getBuffer.get(), BufferSize100k, &outSize, localMaster.m_entRxId));
            localMaster.LocalCancelFrameStop();

            WLANTEST_EXPECT_TRUE(outSize >= WlanTestGetFramMineSize);

            pFrameHeader = reinterpret_cast<LocalFrameHeader*>(getBuffer.get());
            WLANTEST_EXPECT_TRUE(CompareEtherType(EtherTypes[0], pFrameHeader->etherType));
        }
    }

    // 削除した受信エントリのEtherTypeは削除処理以降受信しない CLIENT
    TEST_F(LocalClientFuncTest, L64_2)
    {
        const int32_t etherCnt = sizeof(EtherTypes) / sizeof(EtherTypes[0]);
        std::unique_ptr<uint8_t[]> putBuffer(new uint8_t[WlanTestPacketBufferSize]);
        nn::wlan::ConnectionStatus clientStatus;
        uint32_t seqNumber = 0;

        WLANTEST_ASSERT_TRUE(putBuffer.get() != nullptr);

        const TestLocalConnectParams ConnectParam = {
            nn::wlan::Ssid(LocalMasterTestSSID), nn::wlan::MacAddress::CreateBroadcastMacAddress(), 6,
            { nn::wlan::SecurityMode_Open, nn::wlan::SecurityMode_Open, 0, "" }, true, nn::wlan::BeaconIndication_Enable, 10
        };

        WLANTEST_ASSERT_TRUE(localClient.SetupLocalClient(ConnectParam.ssid, ConnectParam.security, TestMatchInfo, sizeof(TestMatchInfo) / sizeof(TestMatchInfo[0]), true, true));

        Connect(ConnectParam);

        WLANTEST_EXPECT_RESULT_SUCCESS(nn::wlan::Local::GetConnectionStatus(&connectionStatus));

        for (int32_t cnt = 0; cnt < 2; cnt++)
        {
            WLANTEST_EXPECT_TRUE(localClient.LocalMakeFrame(putBuffer.get(), WlanTestPacketBufferSize, connectionStatus.bssid, localClient.m_macAddress, EtherTypes[cnt], TestMatchInfo[0]));
            WLANTEST_EXPECT_TRUE(localClient.LocalSeqNumberAddFrame(putBuffer.get(), WlanTestPacketBufferSize, seqNumber));
            WLANTEST_EXPECT_RESULT_SUCCESS(nn::wlan::Local::PutFrameRaw(putBuffer.get(), WlanTestPacketBufferSize));
        }

        for (int32_t cnt = 0; cnt < 6; cnt++)
        {
            WLANTEST_EXPECT_TRUE(localClient.LocalMakeFrame(putBuffer.get(), WlanTestPacketBufferSize, connectionStatus.bssid, localClient.m_macAddress, EtherTypes[(cnt % etherCnt)], TestMatchInfo[0]));
            WLANTEST_EXPECT_TRUE(localClient.LocalSeqNumberAddFrame(putBuffer.get(), WlanTestPacketBufferSize, seqNumber));
            WLANTEST_EXPECT_RESULT_SUCCESS(nn::wlan::Local::PutFrameRaw(putBuffer.get(), WlanTestPacketBufferSize));
        }

        // 切断待機
        while (NN_STATIC_CONDITION(1))
        {
            // MASTERからの接続が切られるまで待機
            if (nn::os::TimedWaitSystemEvent(&connectionEvent, nn::TimeSpan::FromMilliSeconds(Time300msec)) == true)
            {
                WLANTEST_EXPECT_RESULT_SUCCESS(nn::wlan::Local::GetConnectionStatus(&clientStatus));
                if (clientStatus.state != nn::wlan::ConnectionState_Connected)
                {
                    break;
                }
            }
        }

        Disconnect();
    }

    // MasterとClient7台が、平文でブロードキャスト通信 MASTER
    TEST_F(LocalMasterFuncTest, L66_1)
    {
        std::unique_ptr<uint8_t[]> putBuffer(new uint8_t[WlanTestPacketBufferSize]);
        uint32_t seqNumber = 0;

        WLANTEST_ASSERT_TRUE(putBuffer.get() != nullptr);

        const nn::wlan::MasterBssParameters fncTestMasterBss = {
            6, false, 30, true, nn::wlan::RateSetLegacy_11gMask, nn::wlan::RateSetLegacy_11gMask,
            { nn::wlan::SecurityMode_Open, nn::wlan::SecurityMode_Open, 0, "" }, nn::wlan::Ssid(LocalMasterTestSSID), 100
        };

        WLANTEST_ASSERT_TRUE(localMaster.SetupLocalMaster(fncTestMasterBss, TestMatchInfo, sizeof(TestMatchInfo) / sizeof(TestMatchInfo[0])));

        // MASTER受信スレッド起動
        WLANTEST_ASSERT_TRUE(localMaster.LocalReceiveFrameStart(BufferSize100k, WlanTestPacketBufferSize, ClientMaxConnect, ClientMaxConnect, WlanTestGetFrameCount));
        // CLIENT接続待機
        WLANTEST_ASSERT_TRUE(localMaster.LocalReceiveWait(WlanTestMasterWaitTimeOut, true));

        sysTick = nn::os::GetSystemTick();
        // 60秒間通信
        while ((nn::os::GetSystemTick() - sysTick).ToTimeSpan().GetMilliSeconds() <= Time60s)
        {
            localMaster.LocalGetConnectInfo(&connectsInfo);
            WLANTEST_EXPECT_TRUE(localMaster.LocalMakeFrame(putBuffer.get(), WlanTestPacketBufferSize, nn::wlan::MacAddress::CreateBroadcastMacAddress(), localMaster.m_macAddress, EtherTypes[0], TestMatchInfo[0]));
            WLANTEST_EXPECT_TRUE(localMaster.LocalSeqNumberAddFrame(putBuffer.get(), WlanTestPacketBufferSize, seqNumber));
            WLANTEST_EXPECT_RESULT_SUCCESS(nn::wlan::Local::PutFrameRaw(putBuffer.get(), WlanTestPacketBufferSize));
            seqNumber++;
            nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(Time300msec));
        }

        // MASTER受信スレッド停止
        localMaster.LocalReceiveFrameStop();

        localMaster.LocalGetConnectInfo(&connectsInfo);
        WLANTEST_EXPECT_TRUE(connectsInfo.connectCount == ClientMaxConnect);
        localMaster.LocalTraceConnectsInfo(true);
    }

    // MasterとClient7台が、平文でブロードキャスト通信 CLIENT
    TEST_F(LocalClientFuncTest, L66_2)
    {
        nn::wlan::ConnectionStatus clientStatus;

        const TestLocalConnectParams ConnectParam = {
            nn::wlan::Ssid(LocalMasterTestSSID), nn::wlan::MacAddress::CreateBroadcastMacAddress(), 6,
            { nn::wlan::SecurityMode_Open, nn::wlan::SecurityMode_Open, 0, "" }, true, nn::wlan::BeaconIndication_Enable, 10
        };

        WLANTEST_ASSERT_TRUE(localClient.SetupLocalClient(ConnectParam.ssid, ConnectParam.security, TestMatchInfo, sizeof(TestMatchInfo) / sizeof(TestMatchInfo[0]), true, false));

        Connect(ConnectParam);

        // 受信スレッド起動
        WLANTEST_ASSERT_TRUE(localClient.LocalReceiveFrameStart(BufferSize100k, WlanTestPacketBufferSize, 1, 1, WlanTestGetFrameCount));
        // MASTER接続待機
        WLANTEST_ASSERT_TRUE(localClient.LocalReceiveWait(WlanMaxClientWaitMsecTimeOut, Time1s, true));
        // 送信スレッド起動
        WLANTEST_ASSERT_TRUE(localClient.LocalBroadcastSendFrameStart(WlanTestPacketBufferSize, Time60s + Time5s, Time300msec));

        while (NN_STATIC_CONDITION(1))
        {
            // MASTERからの接続が切られるまで送信し続ける
            if (nn::os::TimedWaitSystemEvent(&connectionEvent, nn::TimeSpan::FromMilliSeconds(Time300msec)) == true)
            {
                WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::GetConnectionStatus(&clientStatus));
                if (clientStatus.state != nn::wlan::ConnectionState_Connected)
                {
                    break;
                }
            }
        }

        // 送信スレッド停止
        localClient.LocalSendFrameStop();
        // 受信スレッド停止
        localClient.LocalReceiveFrameStop();
        // 通信結果ログ出力
        localClient.LocalTraceConnectsInfo(true);

        Disconnect();

    }

    // MasterとClient7台が、平文でユニキャスト通信 MASTER
    TEST_F(LocalMasterFuncTest, L67_1)
    {
        const nn::wlan::MasterBssParameters fncTestMasterBss = {
            6, false, 30, true, nn::wlan::RateSetLegacy_11gMask, nn::wlan::RateSetLegacy_11gMask,
            { nn::wlan::SecurityMode_Open, nn::wlan::SecurityMode_Open, 0, "" }, nn::wlan::Ssid(LocalMasterTestSSID), 100
        };

        WLANTEST_ASSERT_TRUE(localMaster.SetupLocalMaster(fncTestMasterBss, TestMatchInfo, sizeof(TestMatchInfo) / sizeof(TestMatchInfo[0])));

        // MASTER受信スレッド起動
        WLANTEST_ASSERT_TRUE(localMaster.LocalReceiveFrameStart(BufferSize100k, WlanTestPacketBufferSize, ClientMaxConnect, ClientMaxConnect, WlanTestGetFrameCount));

        // CLIENT接続待機
        WLANTEST_ASSERT_TRUE(localMaster.LocalReceiveWait(WlanTestMasterWaitTimeOut));

        // MASTER送信スレッド起動
        WLANTEST_ASSERT_TRUE(localMaster.LocalSendFrameOneShotWaitStart(WlanTestPacketBufferSize, Time60s, Interval_100m, true));

        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(Time60s));

        // MASTER送信スレッド停止
        localMaster.LocalSendFrameStop();

        // MASTER受信スレッド停止
        localMaster.LocalReceiveFrameStop();

        // 通信結果ログ出力
        localMaster.LocalTraceConnectsInfo(true);

        localMaster.LocalGetConnectInfo(&connectsInfo);
        WLANTEST_EXPECT_TRUE(connectsInfo.connectCount == ClientMaxConnect);

    }

    // MasterとClient7台が、平文でユニキャスト通信 CLIENT
    TEST_F(LocalClientFuncTest, L67_2)
    {
        const TestLocalConnectParams ConnectParam = {
            nn::wlan::Ssid(LocalMasterTestSSID), nn::wlan::MacAddress::CreateBroadcastMacAddress(), 6,
            { nn::wlan::SecurityMode_Open, nn::wlan::SecurityMode_Open, 0, "" }, true, nn::wlan::BeaconIndication_Enable, 10
        };

        WLANTEST_ASSERT_TRUE(localClient.SetupLocalClient(ConnectParam.ssid, ConnectParam.security, TestMatchInfo, sizeof(TestMatchInfo) / sizeof(TestMatchInfo[0]), true, false));

        Connect(ConnectParam);

        // 受信スレッド起動
        WLANTEST_ASSERT_TRUE(localClient.LocalReceiveFrameStart(BufferSize100k, WlanTestPacketBufferSize, 1, 1, WlanTestGetFrameCount));
        // MASTER接続待機
        WLANTEST_ASSERT_TRUE(localClient.LocalReceiveWait(WlanMaxClientWaitMsecTimeOut, Time1s, true));
        // 送信スレッド起動
        WLANTEST_ASSERT_TRUE(localClient.LocalSendFrameOneShotWaitStart(WlanTestPacketBufferSize, Time60s + Time5s, Interval_100m, true));

        while (NN_STATIC_CONDITION(1))
        {
            if (nn::os::TimedWaitSystemEvent(&connectionEvent, nn::TimeSpan::FromMilliSeconds(PutInterval)) == true)
            {
                // MASTERからの接続が切られるまで送信し続ける
                WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::GetConnectionStatus(&connectionStatus));
                if (connectionStatus.state != nn::wlan::ConnectionState_Connected)
                {
                    break;
                }
            }
        }

        // 送信スレッド停止
        localClient.LocalSendFrameStop();
        // 受信スレッド停止
        localClient.LocalReceiveFrameStop();
        // 通信結果ログ出力
        localClient.LocalTraceConnectsInfo(true);

        Disconnect();
    }

    // MasterとClient7台が、同じ16文字の暗号鍵を持つStaticAESでブロードキャスト通信 MASTER
    TEST_F(LocalMasterFuncTest, L68_1)
    {
        std::unique_ptr<uint8_t[]> putBuffer(new uint8_t[WlanTestPacketBufferSize]);
        nn::wlan::MacAddress myMacAddress;
        uint32_t seqNumber = 0;

        WLANTEST_ASSERT_TRUE(putBuffer.get() != nullptr);

        const nn::wlan::MasterBssParameters fncTestMasterBss = {
            6, false, 30, true, nn::wlan::RateSetLegacy_11gMask, nn::wlan::RateSetLegacy_11gMask,
            { nn::wlan::SecurityMode_StaticAes, nn::wlan::SecurityMode_StaticAes, 0, LocalMasterAes16Key }, nn::wlan::Ssid(LocalMasterTestSSID), 100
        };

        WLANTEST_ASSERT_TRUE(localMaster.SetupLocalMaster(fncTestMasterBss, TestMatchInfo, sizeof(TestMatchInfo) / sizeof(TestMatchInfo[0])));

        WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::GetMacAddress(&myMacAddress));

        // MASTER受信スレッド起動
        WLANTEST_ASSERT_TRUE(localMaster.LocalReceiveFrameStart(BufferSize100k, WlanTestPacketBufferSize, ClientMaxConnect, ClientMaxConnect, WlanTestGetFrameCount));
        // CLIENT接続待機
        WLANTEST_ASSERT_TRUE(localMaster.LocalReceiveWait(WlanTestMasterWaitTimeOut));

        sysTick = nn::os::GetSystemTick();
        // 60秒間通信
        while ((nn::os::GetSystemTick() - sysTick).ToTimeSpan().GetMilliSeconds() <= Time60s)
        {
            localMaster.LocalGetConnectInfo(&connectsInfo);
            WLANTEST_EXPECT_TRUE(localMaster.LocalMakeFrame(putBuffer.get(), WlanTestPacketBufferSize, nn::wlan::MacAddress::CreateBroadcastMacAddress(), myMacAddress, EtherTypes[0], TestMatchInfo[0]));
            WLANTEST_EXPECT_TRUE(localMaster.LocalSeqNumberAddFrame(putBuffer.get(), WlanTestPacketBufferSize, seqNumber));
            WLANTEST_EXPECT_RESULT_SUCCESS(nn::wlan::Local::PutFrameRaw(putBuffer.get(), WlanTestPacketBufferSize));
            seqNumber++;
            nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(Time300msec));
        }

        // MASTER受信スレッド停止
        localMaster.LocalReceiveFrameStop();

        localMaster.LocalGetConnectInfo(&connectsInfo);
        WLANTEST_EXPECT_TRUE(connectsInfo.connectCount == ClientMaxConnect);

        localMaster.LocalTraceConnectsInfo(true);
    }

    // MasterとClient7台が、同じ16文字の暗号鍵を持つStaticAESでブロードキャスト通信 CLIENT
    TEST_F(LocalClientFuncTest, L68_2)
    {
        nn::wlan::ConnectionStatus ConnectionStatus;

        const TestLocalConnectParams ConnectParam = {
            nn::wlan::Ssid(LocalMasterTestSSID), nn::wlan::MacAddress::CreateBroadcastMacAddress(), 6,
            { nn::wlan::SecurityMode_StaticAes, nn::wlan::SecurityMode_StaticAes, 0, LocalMasterAes16Key }, true, nn::wlan::BeaconIndication_Enable, 10
        };

        WLANTEST_ASSERT_TRUE(localClient.SetupLocalClient(ConnectParam.ssid, ConnectParam.security, TestMatchInfo, sizeof(TestMatchInfo) / sizeof(TestMatchInfo[0]), true, false));

        Connect(ConnectParam);

        // 受信スレッド起動
        WLANTEST_ASSERT_TRUE(localClient.LocalReceiveFrameStart(BufferSize100k, WlanTestPacketBufferSize, 1, 1, WlanTestGetFrameCount));
        WLANTEST_ASSERT_TRUE(localClient.LocalReceiveWait(WlanMaxClientWaitMsecTimeOut, Time1s, true));
        sysTick = nn::os::GetSystemTick();

        // 送信スレッド起動
        WLANTEST_ASSERT_TRUE(localClient.LocalBroadcastSendFrameStart(WlanTestPacketBufferSize, Time60s + Time5s, Time300msec));

        while (NN_STATIC_CONDITION(1))
        {
            if (nn::os::TimedWaitSystemEvent(&connectionEvent, nn::TimeSpan::FromMilliSeconds(Time300msec)) == true)
            {
                WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::GetConnectionStatus(&ConnectionStatus));
                if (ConnectionStatus.state != nn::wlan::ConnectionState_Connected)
                {
                    NN_LOG("               %s ---> Master Disconnected(%lld)\n", __FUNCTION__, (nn::os::GetSystemTick() - sysTick).ToTimeSpan().GetMilliSeconds());
                    break;
                }
            }
        }

        // 送信スレッド停止
        localClient.LocalSendFrameStop();
        // 受信スレッド停止
        localClient.LocalReceiveFrameStop();

        localClient.LocalTraceConnectsInfo(true);

        Disconnect();
    }

    // MasterとClient7台が、同じ16文字の暗号鍵を持つStaticAESでユニキャスト通信 MASTER
    // MasterとClient7台で、16msec間隔の連続3パケットを60秒間のブロードキャスト通信 MASTER
    TEST_F(LocalMasterFuncTest, L69_1)
    {
        nn::wlan::MacAddress broadMac = nn::wlan::MacAddress::CreateBroadcastMacAddress();
        std::unique_ptr<uint8_t[]> putBroadBuffer(new uint8_t[WlanTestBroadCastPacketBufferSize]);
        std::unique_ptr<uint8_t[]> putBuffer(new uint8_t[WlanTestRatePacketBufferSize]);
        uint32_t seqNumber = 0;
        uint32_t outRxId;
        nn::Result result;
        uint32_t broadSendCnt;

        // 通信評価定義
        const uint32_t broadPacketNormal    = (Time60s / Interval_16m * 3);
        const uint32_t broadPacketMin       = broadPacketNormal - (broadPacketNormal * 0.1);
        const uint32_t broadPacketMax       = broadPacketNormal + (broadPacketNormal * 0.1);
        const uint64_t unicastByteNormal    = 30 * 1400 * (Time60s / 1000);
        const uint64_t unicastByteMin       = unicastByteNormal - (unicastByteNormal * 0.1);
        const uint64_t unicastByteMax       = unicastByteNormal + (unicastByteNormal * 0.1);

        WLANTEST_ASSERT_TRUE(putBroadBuffer.get() != nullptr);
        WLANTEST_ASSERT_TRUE(putBuffer.get() != nullptr);

        const nn::wlan::MasterBssParameters fncTestMasterBss = {
            6, false, 30, true, nn::wlan::RateSetLegacy_11gMask, nn::wlan::RateSetLegacy_11gMask,
            { nn::wlan::SecurityMode_StaticAes, nn::wlan::SecurityMode_StaticAes, 0, LocalMasterAes16Key }, nn::wlan::Ssid(LocalMasterTestSSID), 100
        };

        WLANTEST_ASSERT_TRUE(localMaster.SetupLocalMaster(fncTestMasterBss, TestMatchInfo, sizeof(TestMatchInfo) / sizeof(TestMatchInfo[0])));

        WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::AddEthertypeToRxEntry(localMaster.m_entRxId, TestEtherTypes[1]));

        // MASTER受信スレッド起動
        WLANTEST_ASSERT_TRUE(localMaster.LocalReceiveFrameStart(BufferSize200k, WlanTestRecvBufferSize, ClientMaxConnect, ClientMaxConnect, WlanTestGetFrameCount));
        // CLIENT接続待機
        WLANTEST_ASSERT_TRUE(localMaster.LocalReceiveWait(WlanTestMasterWaitTimeOut, true));

        // 送信スレッド起動
        WLANTEST_ASSERT_TRUE(localMaster.LocalSendFrameOneShotWaitStart(WlanTestRatePacketBufferSize, Time60s, Interval_33m, true));

        seqNumber = 0;
        broadSendCnt = 0;
        // 全CLIENTが切断されるまでループに変更
        while (localMaster.LocalMasterDisConnectWait(Interval_16m) > 0)
        {
            // 3パケット通信
            for (int i = 0; i < 3; i++)
            {
                WLANTEST_EXPECT_TRUE(localMaster.LocalMakeFrame(putBroadBuffer.get(), WlanTestBroadCastPacketBufferSize, broadMac, localMaster.m_macAddress, TestEtherTypes[1], TestMatchInfo[0], seqNumber));
                result = nn::wlan::Local::PutFrameRaw(putBroadBuffer.get(), WlanTestBroadCastPacketBufferSize);
                if (result.IsSuccess() != true)
                {
                    if (localMaster.GetStateClientCount(nn::wlan::ConnectionState_Connected) == 0)
                    {
                        break;
                    }
                    NN_LOG("             L69_1 Broad Send(Count:%d) Error(0x%08x:module:%d destription:%d)\n", broadSendCnt,
                        result.GetInnerValueForDebug(), result.GetModule(), result.GetDescription());
                }
                WLANTEST_EXPECT_RESULT_SUCCESS(result);
                seqNumber++;
                broadSendCnt++;
            }
        }

        // MASTER送信スレッド停止
        localMaster.LocalSendFrameStop();
        // MASTER受信スレッド停止
        localMaster.LocalReceiveFrameStop();

        localMaster.LocalTraceConnectsInfo(true);
        localMaster.LocalGetConnectInfo(&connectsInfo);

        // ブロードキャストパケット数範囲評価（1.max(最小, pkt)、2.min(1.結果, 最大）⇒ 範囲内ならpktが戻り値。以外は最大／最小が戻り値）
        NN_LOG("             L69_1 Broad Send Total Packet:%lu(Range:%lu - %lu)\n", broadSendCnt, broadPacketMin, broadPacketMax);
        WLANTEST_EXPECT_TRUE(std::min(std::max(broadPacketMin, broadSendCnt), broadPacketMax) == broadSendCnt);
        // ユニキャスト送信サイズ範囲評価（1.max(最小, 送信サイズ)、2.min(1.結果, 最大）⇒ 範囲内なら送信サイズが戻り値。以外は最大／最小が戻り値）
        NN_LOG("             L69_1 Frame Send Total Size:%lld(Range:%lld - %lld)\n", connectsInfo.totalSendSize, unicastByteMin, unicastByteMax);
        WLANTEST_EXPECT_TRUE(std::min(std::max(unicastByteMin, connectsInfo.totalSendSize), unicastByteMax) == connectsInfo.totalSendSize);

        WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::DeleteEthertypeFromRxEntry(&outRxId, TestEtherTypes[1]));
    }

    // MasterとClient7台が、同じ16文字の暗号鍵を持つStaticAESでユニキャスト通信 CLIENT
    // MasterとClient7台で、16msec間隔の連続3パケットを60秒間のブロードキャスト通信 CLIENT
    TEST_F(LocalClientFuncTest, L69_2)
    {
        nn::wlan::ConnectionStatus clientStatus;
        nn::wlan::MacAddress broadMac = nn::wlan::MacAddress::CreateBroadcastMacAddress();
        std::unique_ptr<uint8_t[]> putBroadBuffer(new uint8_t[WlanTestBroadCastPacketBufferSize]);
        uint32_t seqNumber = 0;
        uint32_t outRxId;
        nn::Result result;
        uint32_t broadSendCnt;

        // 通信評価定義
        const uint32_t broadPacketNormal = (Time60s / Interval_16m * 3);
        const uint32_t broadPacketMin = broadPacketNormal - (broadPacketNormal * 0.1);
        const uint32_t broadPacketMax = broadPacketNormal + (broadPacketNormal * 0.1);
        const uint64_t unicastByteNormal = 30 * 1400 * (Time60s / 1000);
        const uint64_t unicastByteMin = unicastByteNormal - (unicastByteNormal * 0.1);
        const uint64_t unicastByteMax = unicastByteNormal + (unicastByteNormal * 0.1);

        WLANTEST_ASSERT_TRUE(putBroadBuffer.get() != nullptr);

        const TestLocalConnectParams ConnectParam = {
            nn::wlan::Ssid(LocalMasterTestSSID), nn::wlan::MacAddress::CreateBroadcastMacAddress(), 6,
            { nn::wlan::SecurityMode_StaticAes, nn::wlan::SecurityMode_StaticAes, 0, LocalMasterAes16Key }, true, nn::wlan::BeaconIndication_Enable, 10
        };

        WLANTEST_ASSERT_TRUE(localClient.SetupLocalClient(ConnectParam.ssid, ConnectParam.security, TestMatchInfo, sizeof(TestMatchInfo) / sizeof(TestMatchInfo[0]), true, false));
        WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::AddEthertypeToRxEntry(localClient.m_entRxId, TestEtherTypes[1]));

        Connect(ConnectParam);
        WLANTEST_EXPECT_RESULT_SUCCESS(nn::wlan::Local::GetMacAddress(&localClient.m_macAddress));

        // 受信スレッド起動
        WLANTEST_ASSERT_TRUE(localClient.LocalReceiveFrameStart(BufferSize200k, WlanTestRecvBufferSize, 1, 1, WlanTestAgingGetFrameCount));
        // MASTER接続待機
        WLANTEST_ASSERT_TRUE(localClient.LocalReceiveWait(WlanMaxClientWaitMsecTimeOut, Time1s, true));

        // 送信スレッド起動
        WLANTEST_ASSERT_TRUE(localClient.LocalSendFrameOneShotWaitStart(WlanTestRatePacketBufferSize, Time60s + Time5s, Interval_33m, true));

        seqNumber = 0;
        broadSendCnt = 0;
        sysTick = nn::os::GetSystemTick();
        // 60秒間通信
        while ((nn::os::GetSystemTick() - sysTick).ToTimeSpan().GetMilliSeconds() <= Time60s)
        {
            // 一定周期送信。通信が切断されたらNG
            if (nn::os::TimedWaitSystemEvent(&connectionEvent, nn::TimeSpan::FromMilliSeconds(Interval_16m)) == true)
            {
                WLANTEST_EXPECT_RESULT_SUCCESS(nn::wlan::Local::GetConnectionStatus(&clientStatus));
                WLANTEST_EXPECT_TRUE(clientStatus.state == nn::wlan::ConnectionState_Connected);
                if (clientStatus.state != nn::wlan::ConnectionState_Connected)
                {
                    localClient.LocalTraceConnectsInfo(true);
                    break;
                }
            }

            // 3パケット通信
            for (int i = 0; i < 3; i++)
            {
                WLANTEST_EXPECT_TRUE(localClient.LocalMakeFrame(putBroadBuffer.get(), WlanTestBroadCastPacketBufferSize, broadMac, localClient.m_macAddress, TestEtherTypes[1], TestMatchInfo[0], seqNumber));
                result = nn::wlan::Local::PutFrameRaw(putBroadBuffer.get(), WlanTestBroadCastPacketBufferSize);
                if (result.IsSuccess() != true)
                {
                    // Masterから切断されていたら送信終了。ステータス取得してループから抜ける
                    WLANTEST_EXPECT_RESULT_SUCCESS(nn::wlan::Local::GetConnectionStatus(&clientStatus));
                    WLANTEST_EXPECT_TRUE(clientStatus.state == nn::wlan::ConnectionState_Connected);
                    if (clientStatus.state != nn::wlan::ConnectionState_Connected)
                    {
                        break;
                    }

                    NN_LOG("             L69_2 Broad Send(Count:%d) Error(0x%08x:module:%d destription:%d)\n", broadSendCnt,
                                                    result.GetInnerValueForDebug(), result.GetModule(), result.GetDescription());
                }
                WLANTEST_EXPECT_RESULT_SUCCESS(result);
                seqNumber++;
                broadSendCnt++;
            }
        }
        NN_LOG("             L69_2 Broad Send Total Count:%d\n", broadSendCnt);

        // 送信スレッド停止
        localClient.LocalSendFrameStop();
        // 受信スレッド停止
        localClient.LocalReceiveFrameStop();
        // 通信結果ログ出力
        localClient.LocalTraceConnectsInfo(true);
        localClient.LocalGetConnectInfo(&connectsInfo);
        Disconnect();

        // ブロードキャストパケット数範囲評価（1.max(最小, pkt)、2.min(1.結果, 最大）⇒ 範囲内ならpktが戻り値。以外は最大／最小が戻り値）
        NN_LOG("             L69_2 Broad Send Total Packet:%lu(Range:%lu - %lu)\n", broadSendCnt, broadPacketMin, broadPacketMax);
        WLANTEST_EXPECT_TRUE(std::min(std::max(broadPacketMin, broadSendCnt), broadPacketMax) == broadSendCnt);
        // ユニキャスト送信サイズ範囲評価（1.max(最小, 送信サイズ)、2.min(1.結果, 最大）⇒ 範囲内なら送信サイズが戻り値。以外は最大／最小が戻り値）
        NN_LOG("             L69_2 Frame Send Total Size:%lld(Range:%lld - %lld)\n", connectsInfo.totalSendSize, unicastByteMin, unicastByteMax);
        WLANTEST_EXPECT_TRUE(std::min(std::max(unicastByteMin, connectsInfo.totalSendSize), unicastByteMax) == connectsInfo.totalSendSize);

        WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::DeleteEthertypeFromRxEntry(&outRxId, TestEtherTypes[1]));
    }

    // Masterでの既接続スキャン中にActionFrameの送信
    TEST_F(LocalMasterFuncTest, L75)
    {
        nn::wlan::ScanIeMatchInfo info;
        const nn::wlan::ScanParameters StartScanTestParams = {
            nn::wlan::ScanType_Active,{ 1 }, 1, 40, -1, &ssidList[0], 1, nn::wlan::MacAddress::CreateBroadcastMacAddress()
        };

        const nn::wlan::MasterBssParameters fncTestMasterBss = {
            6, false, 30, true, nn::wlan::RateSetLegacy_11gMask, nn::wlan::RateSetLegacy_11gMask,
            { nn::wlan::SecurityMode_Open, nn::wlan::SecurityMode_Open, 0, "" }, nn::wlan::Ssid(LocalMasterTestSSID), 100
        };

        std::unique_ptr<uint8_t[]> pTestBuffer(new uint8_t[TestScanMaxSize]);
        WLANTEST_ASSERT_TRUE(pTestBuffer.get() != nullptr);

        WLANTEST_ASSERT_TRUE(localMaster.SetupLocalMaster(fncTestMasterBss, TestMatchInfo, sizeof(TestMatchInfo) / sizeof(TestMatchInfo[0]), true));
        WLANTEST_ASSERT_TRUE(localMaster.LocalMasterConnectWait(localMaster.m_clientMacArray, WlanTestConnectTimeOut, 1) > 0);

        testPutMac  = nn::wlan::MacAddress(localMaster.m_clientMacArray[0]);
        testChannel = fncTestMasterBss.channel;
        WLANTEST_ASSERT_RESULT_SUCCESS(nn::os::CreateThread(&actionframeThread, SendActionFrameThread, this, threadStack, threadStackSize, nn::os::DefaultThreadPriority));

        nn::os::StartThread(&actionframeThread);

        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(Time1s));
        WLANTEST_EXPECT_RESULT_SUCCESS(nn::wlan::Local::StartScan(pTestBuffer.get(), TestScanMaxSize, StartScanTestParams, &info));

        nn::os::WaitThread(&actionframeThread);
        nn::os::DestroyThread(&actionframeThread);
    }

    // Master起動中に周期的なActionFrameの送信 Master
    TEST_F(LocalMasterFuncTest, L78_1)
    {
        nn::wlan::MacAddress clientArray[LocalTcpGetMacAddressMax];
        size_t putSize = ActionFrameBufferSize;
        std::unique_ptr<uint8_t[]> putBuffer(new uint8_t[putSize]);
        nn::wlan::MacAddress putMacAddress;
        nn::wlan::MacAddress fromMacAddress;
        uint32_t macCnt;

        const nn::wlan::MasterBssParameters fncTestMasterBss = {
            6, false, 30, true, nn::wlan::RateSetLegacy_11gMask, nn::wlan::RateSetLegacy_11gMask,
            { nn::wlan::SecurityMode_Open, nn::wlan::SecurityMode_Open, 0, "" }, nn::wlan::Ssid(LocalMasterTestSSID), 100
        };

        WLANTEST_ASSERT_TRUE(putBuffer.get() != nullptr);

        WLANTEST_ASSERT_TRUE(localMaster.GetMasterTcpMacAddress(clientArray, macCnt, tcpMaster, 1));
        WLANTEST_ASSERT_TRUE(localMaster.SetupLocalMaster(fncTestMasterBss, TestMatchInfo, sizeof(TestMatchInfo) / sizeof(TestMatchInfo[0]), true));

        putMacAddress = nn::wlan::MacAddress(clientArray[1]);

        WLANTEST_EXPECT_RESULT_SUCCESS(nn::wlan::Local::GetMacAddress(&fromMacAddress));

        // 1秒周期で10回送信する
        for (int32_t cnt = 0; cnt < 10; cnt++)
        {
            WLANTEST_EXPECT_TRUE(LocalApiClass::LocalMakeActionFrame(putBuffer.get(), putSize, putMacAddress, fromMacAddress, EtherTypes[0], TestMatchInfo[0], nn::wlan::ActionFrameType_Local, cnt));
            WLANTEST_EXPECT_RESULT_SUCCESS(nn::wlan::Local::PutActionFrameOneShot(putMacAddress, static_cast<uint8_t*>(putBuffer.get()), putSize, fncTestMasterBss.channel, 30));
            nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(Time1s));
        }
    }

    // Master起動中に周期的なActionFrameの送信 Client
    TEST_F(LocalClientFuncTest, L78_2)
    {
        const TestLocalConnectParams ConnectParam = {
            nn::wlan::Ssid(LocalMasterTestSSID), nn::wlan::MacAddress::CreateBroadcastMacAddress(), 6,
            { nn::wlan::SecurityMode_Open, nn::wlan::SecurityMode_Open, 0, "" }, true, nn::wlan::BeaconIndication_Enable, 10
        };

        nn::wlan::MacAddress clientArray[LocalTcpGetMacAddressMax];
        uint32_t macCnt;

        IpV4Address tcpClient = {
            "0.0.0.0", "255.255.255.0", "192.168.11.1",
            "8.8.8.8", "8.8.8.9", 0, false
        };
        nn::util::SNPrintf(tcpClient.strIpAddress, sizeof(tcpClient.strIpAddress), "192.168.11.%d", g_IpOct4);

        WLANTEST_ASSERT_TRUE(localClient.GetClientTcpMacAddress(clientArray, macCnt, tcpMaster, tcpClient));
        WLANTEST_ASSERT_TRUE(localClient.SetupLocalClient(ConnectParam.ssid, ConnectParam.security, TestMatchInfo, sizeof(TestMatchInfo) / sizeof(TestMatchInfo[0]), true, true));

        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(Time15s));
    }

    // 受信エントリに登録したActionFrameTypeを持つパケットを受信 MASTER
    TEST_F(LocalMasterFuncTest, L80_1)
    {
        size_t putSize = ActionFrameBufferSize;
        std::unique_ptr<uint8_t[]> putBuffer(new uint8_t[putSize]);
        const uint16_t FrameTypes[] = { nn::wlan::ActionFrameType_Beacon, nn::wlan::ActionFrameType_Local };
        const int32_t typeCnt = sizeof(FrameTypes) / sizeof(uint16_t);
        nn::wlan::MacAddress putMacAddress;
        nn::wlan::MacAddress fromMacAddress;

        const nn::wlan::MasterBssParameters fncTestMasterBss = {
            11, false, 30, true, nn::wlan::RateSetLegacy_11gMask, nn::wlan::RateSetLegacy_11gMask,
            { nn::wlan::SecurityMode_Open, nn::wlan::SecurityMode_Open, 0, "" }, nn::wlan::Ssid(LocalMasterTestSSID), 100
        };

        WLANTEST_ASSERT_TRUE(putBuffer.get() != nullptr);

        WLANTEST_ASSERT_TRUE(localMaster.SetupLocalMaster(fncTestMasterBss, TestMatchInfo, sizeof(TestMatchInfo) / sizeof(TestMatchInfo[0]), false));

        WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::CreateRxEntryForActionFrame(&localMaster.m_actionRxId, FrameTypes, typeCnt, 10));
        WLANTEST_ASSERT_TRUE(localMaster.LocalMasterConnectWait(WlanTestConnectTimeOut, 1) > 0);

        putMacAddress = nn::wlan::MacAddress::CreateBroadcastMacAddress();
        fromMacAddress = localMaster.m_macAddress;

        // 300msec周期で1回送信する
        for (int32_t cnt = 0; cnt < 50; cnt++)
        {
            nn::wlan::ActionFrameType actFrameType = static_cast<nn::wlan::ActionFrameType>(FrameTypes[cnt % typeCnt]);
            WLANTEST_EXPECT_TRUE(LocalApiClass::LocalMakeActionFrame(putBuffer.get(), putSize, putMacAddress, fromMacAddress, EtherTypes[0], TestMatchInfo[0], actFrameType, cnt));
            WLANTEST_EXPECT_RESULT_SUCCESS(nn::wlan::Local::PutActionFrameOneShot(putMacAddress, static_cast<uint8_t*>(putBuffer.get()), putSize, fncTestMasterBss.channel, 30));
            if (localMaster.LocalMasterDisConnectWait(Time300msec) == 0)
            {
                break;
            }
        }

        WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::DeleteRxEntryForActionFrame(localMaster.m_actionRxId));
    }

    // 受信エントリに登録したActionFrameTypeを持つパケットを受信 CLIENT
    TEST_F(LocalClientFuncTest, L80_2)
    {
        uint32_t actRxId;
        const uint16_t FrameTypes[] = { nn::wlan::ActionFrameType_Beacon, nn::wlan::ActionFrameType_Local };
        nn::wlan::MacAddress outMacAddress;
        std::unique_ptr<uint8_t[]> getBuffer(new uint8_t[TestScanMaxSize]);
        size_t outSize;
        int32_t frameTypeCnt = 0;

        nn::wlan::MacAddress macAddress = nn::wlan::MacAddress::CreateBroadcastMacAddress();
        const TestLocalConnectParams ConnectParam = {
            nn::wlan::Ssid(LocalMasterTestSSID), nn::wlan::MacAddress::CreateBroadcastMacAddress(), 11,
            { nn::wlan::SecurityMode_Open, nn::wlan::SecurityMode_Open, 0, "" }, true, nn::wlan::BeaconIndication_Enable, 10
        };

        AllScan();

        WLANTEST_ASSERT_TRUE(localClient.SetupLocalClient(ConnectParam.ssid, ConnectParam.security, TestMatchInfo, sizeof(TestMatchInfo) / sizeof(TestMatchInfo[0]), true, false));

        WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::CreateRxEntryForActionFrame(&actRxId, FrameTypes, sizeof(FrameTypes) / sizeof(uint16_t), 10));
        Connect(ConnectParam);

        while (frameTypeCnt < (sizeof(FrameTypes) / sizeof(uint16_t)))
        {
            WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::GetActionFrame(&outMacAddress, getBuffer.get(), TestScanMaxSize, &outSize, actRxId));
            WLANTEST_EXPECT_TRUE(outSize > WlanTestGetFramMineSize);
            LocalActionFrameData *pActionFrame = reinterpret_cast<LocalActionFrameData*>(getBuffer.get());

            if (pActionFrame->frameType == FrameTypes[frameTypeCnt])
            {
                frameTypeCnt++;
            }
        }
        WLANTEST_EXPECT_TRUE(frameTypeCnt == (sizeof(FrameTypes) / sizeof(uint16_t)));

        WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::DeleteRxEntryForActionFrame(actRxId));
        WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::Disconnect(nn::wlan::LocalCommunicationMode_ClientSpectator, nullptr));
    }


    // 受信エントリにActionFrameTypeを追加し、受信 MASTER
    TEST_F(LocalMasterFuncTest, L81_1)
    {
        size_t putSize = ActionFrameBufferSize;
        std::unique_ptr<uint8_t[]> putBuffer(new uint8_t[putSize]);
        const uint16_t FrameTypes[] = { nn::wlan::ActionFrameType_Beacon, nn::wlan::ActionFrameType_Local };
        const int32_t typeCnt = sizeof(FrameTypes) / sizeof(uint16_t);
        nn::wlan::MacAddress putMacAddress;
        nn::wlan::MacAddress fromMacAddress;

        const nn::wlan::MasterBssParameters fncTestMasterBss = {
            11, false, 30, true, nn::wlan::RateSetLegacy_11gMask, nn::wlan::RateSetLegacy_11gMask,
            { nn::wlan::SecurityMode_Open, nn::wlan::SecurityMode_Open, 0, "" }, nn::wlan::Ssid(LocalMasterTestSSID), 100
        };

        WLANTEST_ASSERT_TRUE(putBuffer.get() != nullptr);

        WLANTEST_ASSERT_TRUE(localMaster.SetupLocalMaster(fncTestMasterBss, TestMatchInfo, sizeof(TestMatchInfo) / sizeof(TestMatchInfo[0]), false));

        WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::CreateRxEntryForActionFrame(&localMaster.m_actionRxId, FrameTypes, typeCnt, 10));
        WLANTEST_ASSERT_TRUE(localMaster.LocalMasterConnectWait(WlanTestConnectTimeOut, 1) > 0);

        putMacAddress = nn::wlan::MacAddress::CreateBroadcastMacAddress();
        fromMacAddress = localMaster.m_macAddress;

        // 300msec周期で1回送信する
        for (int32_t cnt = 0; cnt < 50; cnt++)
        {
            nn::wlan::ActionFrameType actFrameType = static_cast<nn::wlan::ActionFrameType>(FrameTypes[cnt % typeCnt]);
            WLANTEST_EXPECT_TRUE(LocalApiClass::LocalMakeActionFrame(putBuffer.get(), putSize, putMacAddress, fromMacAddress, EtherTypes[0], TestMatchInfo[0], actFrameType, cnt));
            WLANTEST_EXPECT_RESULT_SUCCESS(nn::wlan::Local::PutActionFrameOneShot(putMacAddress, static_cast<uint8_t*>(putBuffer.get()), putSize, fncTestMasterBss.channel, 30));
            if (localMaster.LocalMasterDisConnectWait(Time300msec) == 0)
            {
                break;
            }
        }

        WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::DeleteRxEntryForActionFrame(localMaster.m_actionRxId));
    }

    // 受信エントリにActionFrameTypeを追加し、受信
    TEST_F(LocalClientFuncTest, L81_2)
    {
        uint32_t actRxId;
        const uint16_t FrameTypes[] = { nn::wlan::ActionFrameType_Beacon, nn::wlan::ActionFrameType_Local };
        nn::wlan::MacAddress outMacAddress;
        std::unique_ptr<uint8_t[]> getBuffer(new uint8_t[TestScanMaxSize]);
        size_t outSize;
        int32_t frameTypeCnt = 0;

        nn::wlan::MacAddress macAddress = nn::wlan::MacAddress::CreateBroadcastMacAddress();
        const TestLocalConnectParams ConnectParam = {
            nn::wlan::Ssid(LocalMasterTestSSID), nn::wlan::MacAddress::CreateBroadcastMacAddress(), 11,
            { nn::wlan::SecurityMode_Open, nn::wlan::SecurityMode_Open, 0, "" }, true, nn::wlan::BeaconIndication_Enable, 10
        };

        AllScan();

        WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::CreateRxEntryForActionFrame(&actRxId, &FrameTypes[1], 1, 10));
        WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::AddSubtypeToRxEntryForActionFrame(actRxId, nn::wlan::ActionFrameType_Beacon));
        Connect(ConnectParam);
        WLANTEST_EXPECT_RESULT_SUCCESS(nn::wlan::Local::GetConnectionStatus(&connectionStatus));

        while (frameTypeCnt < (sizeof(FrameTypes) / sizeof(uint16_t)))
        {
            WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::GetActionFrame(&outMacAddress, getBuffer.get(), TestScanMaxSize, &outSize, actRxId));
            WLANTEST_EXPECT_TRUE(outSize > WlanTestGetFramMineSize);
            LocalActionFrameData *pActionFrame = reinterpret_cast<LocalActionFrameData*>(getBuffer.get());

            if (pActionFrame->frameType == FrameTypes[frameTypeCnt])
            {
                frameTypeCnt++;
            }
        }
        WLANTEST_EXPECT_TRUE(frameTypeCnt == (sizeof(FrameTypes) / sizeof(uint16_t)));


        WLANTEST_EXPECT_TRUE(connectionStatus.bssid == outMacAddress);

        WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::DeleteRxEntryForActionFrame(actRxId));
        WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::Disconnect(nn::wlan::LocalCommunicationMode_ClientSpectator, nullptr));
    }

    // 受信エントリへのActionFrameTypeの登録削除を複数回行える
    TEST_F(LocalClientFuncTest, L83)
    {
        uint32_t actRxId;
        const uint16_t FrameTypes1[] = { nn::wlan::ActionFrameType_Beacon };
        const uint16_t FrameTypes2[] = { nn::wlan::ActionFrameType_Beacon };

        nn::wlan::MacAddress outMacAddress;
        std::unique_ptr<uint8_t[]> getBuffer(new uint8_t[TestScanMaxSize]);

        AllScan();

        WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::CreateRxEntryForActionFrame(&actRxId, FrameTypes1, sizeof(FrameTypes1) / sizeof(uint16_t), 10));
        WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::DeleteRxEntryForActionFrame(actRxId));

        WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::CreateRxEntryForActionFrame(&actRxId, FrameTypes2, sizeof(FrameTypes2) / sizeof(uint16_t), 10));
        WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::DeleteRxEntryForActionFrame(actRxId));

        WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::CreateRxEntryForActionFrame(&actRxId, FrameTypes1, sizeof(FrameTypes1) / sizeof(uint16_t), 10));
        WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::DeleteRxEntryForActionFrame(actRxId));
    }

    // 削除した受信エントリのActionFrameTypeは削除処理以降受信しない MASTER
    TEST_F(LocalMasterFuncTest, L84_1)
    {
        size_t putSize = ActionFrameBufferSize;
        std::unique_ptr<uint8_t[]> putBuffer(new uint8_t[putSize]);
        nn::wlan::MacAddress clientMacArray[ClientMaxConnect];
        WLANTEST_ASSERT_TRUE(putBuffer.get() != nullptr);
        nn::wlan::ConnectionStatus clientStatus;
        uint32_t seqNumber = 0;
        nn::Result result;

        const nn::wlan::MasterBssParameters fncTestMasterBss = {
            11, false, 30, true, nn::wlan::RateSetLegacy_11gMask, nn::wlan::RateSetLegacy_11gMask,
            { nn::wlan::SecurityMode_Open, nn::wlan::SecurityMode_Open, 0, "" }, nn::wlan::Ssid(LocalMasterTestSSID), 100
        };

        WLANTEST_ASSERT_TRUE(localMaster.SetupLocalMaster(fncTestMasterBss, TestMatchInfo, sizeof(TestMatchInfo) / sizeof(TestMatchInfo[0]), true));
        WLANTEST_ASSERT_TRUE(localMaster.LocalMasterConnectWait(clientMacArray, WlanTestConnectTimeOut, 1) > 0);

        // 120秒間ActionFrame送信
        sysTick = nn::os::GetSystemTick();
        NN_LOG("             MASETER - PutActionFrameOneShot Start\n");
        while ((nn::os::GetSystemTick() - sysTick).ToTimeSpan().GetMilliSeconds() <= Time120s)
        {
            // MASTERからの接続が切られるまで送信し続ける
            if (nn::os::TimedWaitSystemEvent(&connectionEvent, nn::TimeSpan::FromMilliSeconds(Time1s)) == true)
            {
                if (localMaster.GetClientMacAddress(clientMacArray, nn::wlan::ConnectionState_Connected) == 0)
                {
                    break;
                }
            }

            // local frame type
            WLANTEST_EXPECT_TRUE(localMaster.LocalMakeActionFrame(putBuffer.get(), putSize, clientMacArray[0],
                localMaster.m_macAddress, EtherTypes[0], TestMatchInfo[0], nn::wlan::ActionFrameType_Local));
            WLANTEST_EXPECT_TRUE(localMaster.LocalSeqNumberAddFrame(putBuffer.get(), putSize, seqNumber++));
            result = nn::wlan::Local::PutActionFrameOneShot(clientMacArray[0], putBuffer.get(), putSize, fncTestMasterBss.channel, 30);
            NN_LOG("             MASETER - PutActionFrameOneShot Result : 0x%08x - Time : %lldmsec\n", result.GetInnerValueForDebug(),
                (nn::os::GetSystemTick() - sysTick).ToTimeSpan().GetMilliSeconds());
        }
        NN_LOG("             MASETER - PutActionFrameOneShot End\n");
    }

    // 削除した受信エントリのActionFrameTypeは削除処理以降受信しない CLIENT
    TEST_F(LocalClientFuncTest, L84_2)
    {
        nn::wlan::MacAddress macAddress = nn::wlan::MacAddress::CreateBroadcastMacAddress();
        const TestLocalConnectParams ConnectParam = {
            nn::wlan::Ssid(LocalMasterTestSSID), nn::wlan::MacAddress::CreateBroadcastMacAddress(), 11,
            { nn::wlan::SecurityMode_Open, nn::wlan::SecurityMode_Open, 0, "" }, true, nn::wlan::BeaconIndication_Enable, 10
        };

        nn::wlan::MacAddress outMacAddress;
        std::unique_ptr<uint8_t[]> getBuffer(new uint8_t[TestScanMaxSize]);
        LocalActionFrameData* pActionFrame = nullptr;
        size_t outSize;
        uint32_t outActionRxId;
        nn::Result result;

        WLANTEST_ASSERT_TRUE(getBuffer.get() != nullptr);

        WLANTEST_ASSERT_TRUE(localClient.SetupLocalClient(ConnectParam.ssid, ConnectParam.security, TestMatchInfo, sizeof(TestMatchInfo) / sizeof(TestMatchInfo[0]), false, true));
        WLANTEST_EXPECT_RESULT_SUCCESS(nn::wlan::Local::AddSubtypeToRxEntryForActionFrame(localClient.m_actionRxId, nn::wlan::ActionFrameType_Local));

        pActionFrame = reinterpret_cast<LocalActionFrameData*>(getBuffer.get());

        AllScan();

        Connect(ConnectParam);

        // 設定したフレームタイプ比較
        bool isMatchFramType = false;
        WLANTEST_EXPECT_RESULT_SUCCESS(nn::wlan::Local::GetConnectionStatus(&connectionStatus));
        while (connectionStatus.state == nn::wlan::ConnectionState_Connected)
        {
            WLANTEST_ASSERT_TRUE(localClient.LocalCancelActionFrameStart(Time15s));
            WLANTEST_EXPECT_RESULT_SUCCESS(nn::wlan::Local::GetActionFrame(&outMacAddress, getBuffer.get(), TestScanMaxSize, &outSize, localClient.m_actionRxId));
            localClient.LocalCancelActionFrameStop();

            if (nn::wlan::ActionFrameType_Local == pActionFrame->frameType)
            {
                isMatchFramType = true;
                break;
            }

            WLANTEST_EXPECT_RESULT_SUCCESS(nn::wlan::Local::GetConnectionStatus(&connectionStatus));
        }
        WLANTEST_ASSERT_TRUE(isMatchFramType);

        outActionRxId = localClient.m_actionRxId;
        WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::DeleteSubtypeFromRxEntryForActionFrame(&outActionRxId, nn::wlan::ActionFrameType_Local));

        NN_LOG("             CLIENT - GetActionFrame Wait Start\n");
        WLANTEST_ASSERT_TRUE(localClient.LocalCancelActionFrameStart(Time15s));
        result = nn::wlan::Local::GetActionFrame(&outMacAddress, getBuffer.get(), TestScanMaxSize, &outSize, localClient.m_actionRxId);
        WLANTEST_EXPECT_TRUE(nn::wlan::ResultGetFrameCancelled::Includes(result));
        NN_LOG("             CLIENT - GetActionFrame Wait End\n");
        localClient.LocalCancelActionFrameStop();

        Disconnect();
    }

    // 受信エントリにActionFrameTypeを2個登録確認（MAX2個ではない）
    TEST_F(LocalClientFuncTest, L85)
    {
        uint32_t actRxId, outActionRxId;
        const uint16_t FrameTypes[] = { nn::wlan::ActionFrameType_Beacon };
        nn::wlan::ActionFrameType AddFrameTypes[] = { nn::wlan::ActionFrameType_Local, nn::wlan::ActionFrameType_Reserved1 };

        nn::wlan::MacAddress macAddress = nn::wlan::MacAddress::CreateBroadcastMacAddress();
        const TestLocalConnectParams ConnectParam = {
            nn::wlan::Ssid(LocalMasterTestSSID), nn::wlan::MacAddress::CreateBroadcastMacAddress(), 1,
            { nn::wlan::SecurityMode_Open, nn::wlan::SecurityMode_Open, 0, "" }, true, nn::wlan::BeaconIndication_Enable, 10
        };

        AllScan();

        WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::CreateRxEntryForActionFrame(&actRxId, FrameTypes, sizeof(FrameTypes) / sizeof(uint16_t), 10));

        for (auto frameType : AddFrameTypes)
        {
            WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::AddSubtypeToRxEntryForActionFrame(actRxId, frameType));
        }

        for (auto frameType : AddFrameTypes)
        {
            WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::DeleteSubtypeFromRxEntryForActionFrame(&outActionRxId, frameType));
        }

        WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::DeleteRxEntryForActionFrame(actRxId));
    }

    // 無線がONになっている状態(IDLE)でActionFrameの受信 MASTER
    TEST_F(LocalMasterFuncTest, L86_1)
    {
        size_t putSize = ActionFrameBufferSize;
        std::unique_ptr<uint8_t[]> putBuffer(new uint8_t[putSize]);
        nn::wlan::MacAddress clientMacArray[ClientMaxConnect];
        WLANTEST_ASSERT_TRUE(putBuffer.get() != nullptr);
        uint32_t seqNumber = 0;

        const nn::wlan::MasterBssParameters fncTestMasterBss = {
            11, false, 30, true, nn::wlan::RateSetLegacy_11gMask, nn::wlan::RateSetLegacy_11gMask,
            { nn::wlan::SecurityMode_Open, nn::wlan::SecurityMode_Open, 0, "" }, nn::wlan::Ssid(LocalMasterTestSSID), 100
        };

        WLANTEST_ASSERT_TRUE(localMaster.SetupLocalMaster(fncTestMasterBss, TestMatchInfo, sizeof(TestMatchInfo) / sizeof(TestMatchInfo[0]), true));
        WLANTEST_ASSERT_TRUE(localMaster.LocalMasterConnectWait(clientMacArray, WlanTestConnectTimeOut, 1) > 0);

        // 15秒間ActionFrame送信
        sysTick = nn::os::GetSystemTick();
        NN_LOG("             MASETER - PutActionFrameOneShot Start\n");
        while ((nn::os::GetSystemTick() - sysTick).ToTimeSpan().GetMilliSeconds() <= Time120s)
        {
            // MASTERからの接続が切られるまで送信し続ける
            if (nn::os::TimedWaitSystemEvent(&connectionEvent, nn::TimeSpan::FromMilliSeconds(Time100msec)) == true)
            {
                if (localMaster.GetClientMacAddress(clientMacArray, nn::wlan::ConnectionState_Connected) == 0)
                {
                    NN_LOG("             MASETER - CLIENT DISCONNECTED\n");
                    break;
                }
            }

            // beacon frame type
            WLANTEST_EXPECT_TRUE(localMaster.LocalMakeActionFrame(putBuffer.get(), putSize, clientMacArray[0],
                localMaster.m_macAddress, EtherTypes[0], TestMatchInfo[0], nn::wlan::ActionFrameType_Beacon));
            WLANTEST_EXPECT_TRUE(localMaster.LocalSeqNumberAddFrame(putBuffer.get(), putSize, seqNumber));
            nn::wlan::Local::PutActionFrameOneShot(clientMacArray[0], putBuffer.get(), putSize, fncTestMasterBss.channel, 30);

            // local frame type
            WLANTEST_EXPECT_TRUE(localMaster.LocalMakeActionFrame(putBuffer.get(), putSize, clientMacArray[0],
                localMaster.m_macAddress, EtherTypes[0], TestMatchInfo[0], nn::wlan::ActionFrameType_Local));
            WLANTEST_EXPECT_TRUE(localMaster.LocalSeqNumberAddFrame(putBuffer.get(), putSize, seqNumber));
            nn::wlan::Local::PutActionFrameOneShot(clientMacArray[0], putBuffer.get(), putSize, fncTestMasterBss.channel, 30);
            seqNumber++;
        }
        NN_LOG("             MASETER - PutActionFrameOneShot End(%llu)\n", (nn::os::GetSystemTick() - sysTick).ToTimeSpan().GetMilliSeconds());
    }

    // 無線がONになっている状態(IDLE)でActionFrameの受信 無線がONになっている状態(IDLE)でActionFrameの受信
    TEST_F(LocalClientFuncTest, L86_2)
    {
        nn::wlan::MacAddress outMacAddress;
        std::unique_ptr<uint8_t[]> getBuffer(new uint8_t[ActionFrameBufferSize]);
        size_t outSize;

        WLANTEST_ASSERT_TRUE(getBuffer.get() != nullptr);

        nn::wlan::MacAddress macAddress = nn::wlan::MacAddress::CreateBroadcastMacAddress();
        const TestLocalConnectParams ConnectParam = {
            nn::wlan::Ssid(LocalMasterTestSSID), nn::wlan::MacAddress::CreateBroadcastMacAddress(), 11,
            { nn::wlan::SecurityMode_Open, nn::wlan::SecurityMode_Open, 0, "" }, true, nn::wlan::BeaconIndication_Enable, 10
        };

        AllScan();

        WLANTEST_ASSERT_TRUE(localClient.SetupLocalClient(ConnectParam.ssid, ConnectParam.security, TestMatchInfo, sizeof(TestMatchInfo) / sizeof(TestMatchInfo[0]), true, true));

        Connect(ConnectParam);

        // 3回受信する
        for (int i = 0; i < 3; i++)
        {
            WLANTEST_ASSERT_TRUE(localClient.LocalCancelActionFrameStart(Time5s));
            WLANTEST_EXPECT_RESULT_SUCCESS(nn::wlan::Local::GetActionFrame(&outMacAddress, getBuffer.get(), ActionFrameBufferSize, &outSize, localClient.m_actionRxId));
            localClient.LocalCancelActionFrameStop();
        }

        Disconnect();
    }

    // スキャン処理中にActionFrameの受信 MASTER
    TEST_F(LocalMasterFuncTest, L87_1)
    {
        size_t putSize = ActionFrameBufferSize;
        std::unique_ptr<uint8_t[]> putBuffer(new uint8_t[putSize]);
        nn::wlan::MacAddress clientMacArray[ClientMaxConnect];
        WLANTEST_ASSERT_TRUE(putBuffer.get() != nullptr);
        uint32_t seqNumber = 0;

        const nn::wlan::MasterBssParameters fncTestMasterBss = {
            11, false, 30, true, nn::wlan::RateSetLegacy_11gMask, nn::wlan::RateSetLegacy_11gMask,
            { nn::wlan::SecurityMode_Open, nn::wlan::SecurityMode_Open, 0, "" }, nn::wlan::Ssid(LocalMasterTestSSID), 100
        };

        WLANTEST_ASSERT_TRUE(localMaster.SetupLocalMaster(fncTestMasterBss, TestMatchInfo, sizeof(TestMatchInfo) / sizeof(TestMatchInfo[0]), true));
        WLANTEST_ASSERT_TRUE(localMaster.LocalMasterConnectWait(clientMacArray, WlanTestConnectTimeOut, 1) > 0);

        // 15秒間ActionFrame送信
        sysTick = nn::os::GetSystemTick();
        NN_LOG("             MASETER - PutActionFrameOneShot Start\n");
        while ((nn::os::GetSystemTick() - sysTick).ToTimeSpan().GetMilliSeconds() <= Time15s)
        {
            // beacon frame type
            WLANTEST_EXPECT_TRUE(localMaster.LocalMakeActionFrame(putBuffer.get(), putSize, clientMacArray[0],
                localMaster.m_macAddress, EtherTypes[0], TestMatchInfo[0], nn::wlan::ActionFrameType_Beacon));
            WLANTEST_EXPECT_TRUE(localMaster.LocalSeqNumberAddFrame(putBuffer.get(), putSize, seqNumber++));
            nn::wlan::Local::PutActionFrameOneShot(clientMacArray[0], putBuffer.get(), putSize, fncTestMasterBss.channel, 30);
        }
        NN_LOG("             MASETER - PutActionFrameOneShot End\n");
    }

    // スキャン処理中にActionFrameの受信 CLIENT
    TEST_F(LocalClientFuncTest, L87_2)
    {
        nn::wlan::ScanIeMatchInfo info;

        const nn::wlan::ScanParameters StartScanTestParams = {
            nn::wlan::ScanType_Passive,{ 1, 6, 11 }, 3, 110, -1, &ssidList[0], 1, nn::wlan::MacAddress::CreateBroadcastMacAddress()
        };

        nn::wlan::MacAddress macAddress = nn::wlan::MacAddress::CreateBroadcastMacAddress();
        const TestLocalConnectParams ConnectParam = {
            nn::wlan::Ssid(LocalMasterTestSSID), nn::wlan::MacAddress::CreateBroadcastMacAddress(), 11,
            { nn::wlan::SecurityMode_Open, nn::wlan::SecurityMode_Open, 0, "" }, true, nn::wlan::BeaconIndication_Enable, 10
        };
        std::unique_ptr<uint8_t[]> pTestBuffer(new uint8_t[TestScanMaxSize]);
        WLANTEST_ASSERT_TRUE(pTestBuffer.get() != nullptr);

        AllScan();

        WLANTEST_ASSERT_TRUE(localClient.SetupLocalClient(ConnectParam.ssid, ConnectParam.security, TestMatchInfo, sizeof(TestMatchInfo) / sizeof(TestMatchInfo[0]), false, true));

        Connect(ConnectParam);

        WLANTEST_ASSERT_RESULT_SUCCESS(nn::os::CreateThread(&actionframeThread, ReceiveActionFrameThread, this, threadStack, threadStackSize, nn::os::DefaultThreadPriority));

        nn::os::StartThread(&actionframeThread);

        WLANTEST_EXPECT_RESULT_SUCCESS(nn::wlan::Local::StartScan(pTestBuffer.get(), TestScanMaxSize, StartScanTestParams, &info));

        nn::os::WaitThread(&actionframeThread);
        nn::os::DestroyThread(&actionframeThread);

        Disconnect();
    }

    // 接続処理中にActionFrameの受信 MASTER
    TEST_F(LocalMasterFuncTest, L88_1)
    {
        size_t putSize = ActionFrameBufferSize;
        std::unique_ptr<uint8_t[]> putBuffer(new uint8_t[putSize]);
        const uint16_t FrameTypes[] = { nn::wlan::ActionFrameType_Beacon, nn::wlan::ActionFrameType_Local };
        const int32_t typeCnt = sizeof(FrameTypes) / sizeof(uint16_t);
        nn::wlan::MacAddress putMacAddress;
        nn::wlan::MacAddress fromMacAddress;

        const nn::wlan::MasterBssParameters fncTestMasterBss = {
            11, false, 30, true, nn::wlan::RateSetLegacy_11gMask, nn::wlan::RateSetLegacy_11gMask,
            { nn::wlan::SecurityMode_Open, nn::wlan::SecurityMode_Open, 0, "" }, nn::wlan::Ssid(LocalMasterTestSSID), 100
        };

        WLANTEST_ASSERT_TRUE(putBuffer.get() != nullptr);

        WLANTEST_ASSERT_TRUE(localMaster.SetupLocalMaster(fncTestMasterBss, TestMatchInfo, sizeof(TestMatchInfo) / sizeof(TestMatchInfo[0]), false));

        WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::CreateRxEntryForActionFrame(&localMaster.m_actionRxId, FrameTypes, typeCnt, 10));
        WLANTEST_ASSERT_TRUE(localMaster.LocalMasterConnectWait(WlanTestConnectTimeOut, 1) > 0);

        putMacAddress = nn::wlan::MacAddress::CreateBroadcastMacAddress();
        fromMacAddress = localMaster.m_macAddress;

        // 300msec周期で1回送信する
        for (int32_t cnt = 0; cnt < 50; cnt++)
        {
            nn::wlan::ActionFrameType actFrameType = static_cast<nn::wlan::ActionFrameType>(FrameTypes[cnt % typeCnt]);
            WLANTEST_EXPECT_TRUE(LocalApiClass::LocalMakeActionFrame(putBuffer.get(), putSize, putMacAddress, fromMacAddress, EtherTypes[0], TestMatchInfo[0], actFrameType, cnt));
            WLANTEST_EXPECT_RESULT_SUCCESS(nn::wlan::Local::PutActionFrameOneShot(putMacAddress, static_cast<uint8_t*>(putBuffer.get()), putSize, fncTestMasterBss.channel, 30));
            if (localMaster.LocalMasterDisConnectWait(Time300msec) == 0)
            {
                break;
            }
        }

        WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::DeleteRxEntryForActionFrame(localMaster.m_actionRxId));
    }

    // 接続処理中にActionFrameの受信 CLIENT
    TEST_F(LocalClientFuncTest, L88_2)
    {
        nn::wlan::MacAddress macAddress = nn::wlan::MacAddress::CreateBroadcastMacAddress();
        const TestLocalConnectParams ConnectParam = {
            nn::wlan::Ssid(LocalMasterTestSSID), nn::wlan::MacAddress::CreateBroadcastMacAddress(), 11,
            { nn::wlan::SecurityMode_Open, nn::wlan::SecurityMode_Open, 0, "" }, true, nn::wlan::BeaconIndication_Enable, 10
        };
        std::unique_ptr<uint8_t[]> pTestBuffer(new uint8_t[TestScanMaxSize]);
        WLANTEST_ASSERT_TRUE(pTestBuffer.get() != nullptr);

        AllScan();

        WLANTEST_ASSERT_TRUE(localClient.SetupLocalClient(ConnectParam.ssid, ConnectParam.security, TestMatchInfo, sizeof(TestMatchInfo) / sizeof(TestMatchInfo[0]), false, true));

        nn::os::InitializeEvent(&actionframeEvent, false, nn::os::EventClearMode::EventClearMode_ManualClear);
        WLANTEST_ASSERT_RESULT_SUCCESS(nn::os::CreateThread(&actionframeThread, ReceiveActionFrameThread, this, threadStack, threadStackSize, nn::os::DefaultThreadPriority));
        nn::os::StartThread(&actionframeThread);

        Connect(ConnectParam);
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(5000));

        WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::Disconnect(nn::wlan::LocalCommunicationMode_ClientSpectator, nullptr));
        nn::os::WaitThread(&actionframeThread);
        nn::os::DestroyThread(&actionframeThread);
        nn::os::FinalizeEvent(&actionframeEvent);
    }

    // 接続後にActionFrameの受信 MASTER
    TEST_F(LocalMasterFuncTest, L89_1)
    {
        size_t putSize = ActionFrameBufferSize;
        std::unique_ptr<uint8_t[]> putBuffer(new uint8_t[putSize]);
        const uint16_t FrameTypes[] = { nn::wlan::ActionFrameType_Beacon, nn::wlan::ActionFrameType_Local };
        const int32_t typeCnt = sizeof(FrameTypes) / sizeof(uint16_t);
        nn::wlan::MacAddress putMacAddress;
        nn::wlan::MacAddress fromMacAddress;

        const nn::wlan::MasterBssParameters fncTestMasterBss = {
            1, false, 30, true, nn::wlan::RateSetLegacy_11gMask, nn::wlan::RateSetLegacy_11gMask,
            { nn::wlan::SecurityMode_Open, nn::wlan::SecurityMode_Open, 0, "" }, nn::wlan::Ssid(LocalMasterTestSSID), 100
        };

        WLANTEST_ASSERT_TRUE(putBuffer.get() != nullptr);

        WLANTEST_ASSERT_TRUE(localMaster.SetupLocalMaster(fncTestMasterBss, TestMatchInfo, sizeof(TestMatchInfo) / sizeof(TestMatchInfo[0]), false));

        WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::CreateRxEntryForActionFrame(&localMaster.m_actionRxId, FrameTypes, typeCnt, 10));
        WLANTEST_ASSERT_TRUE(localMaster.LocalMasterConnectWait(WlanTestConnectTimeOut, 1) > 0);

        putMacAddress = nn::wlan::MacAddress::CreateBroadcastMacAddress();
        fromMacAddress = localMaster.m_macAddress;

        // 300msec周期で1回送信する
        for (int32_t cnt = 0; cnt < 50; cnt++)
        {
            nn::wlan::ActionFrameType actFrameType = static_cast<nn::wlan::ActionFrameType>(FrameTypes[cnt % typeCnt]);
            WLANTEST_EXPECT_TRUE(LocalApiClass::LocalMakeActionFrame(putBuffer.get(), putSize, putMacAddress, fromMacAddress, EtherTypes[0], TestMatchInfo[0], actFrameType, cnt));
            WLANTEST_EXPECT_RESULT_SUCCESS(nn::wlan::Local::PutActionFrameOneShot(putMacAddress, static_cast<uint8_t*>(putBuffer.get()), putSize, fncTestMasterBss.channel, 30));
            if (localMaster.LocalMasterDisConnectWait(Time300msec) == 0)
            {
                break;
            }
        }

        WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::DeleteRxEntryForActionFrame(localMaster.m_actionRxId));
    }

    // 接続後にActionFrameの受信 CLIENT
    TEST_F(LocalClientFuncTest, L89_2)
    {
        nn::wlan::MacAddress macAddress = nn::wlan::MacAddress::CreateBroadcastMacAddress();
        const TestLocalConnectParams ConnectParam = {
            nn::wlan::Ssid(LocalMasterTestSSID), nn::wlan::MacAddress::CreateBroadcastMacAddress(), 1,
            { nn::wlan::SecurityMode_Open, nn::wlan::SecurityMode_Open, 0, "" }, true, nn::wlan::BeaconIndication_Enable, 10
        };
        // ジョイコン1セット（各2台）
        WLANTEST_ASSERT_TRUE(localClient.SetBtMode(LocalApiBtNode2));

        WLANTEST_ASSERT_TRUE(localClient.SetupLocalClient(ConnectParam.ssid, ConnectParam.security, TestMatchInfo, sizeof(TestMatchInfo) / sizeof(TestMatchInfo[0]), false, true));

        nn::os::InitializeEvent(&actionframeEvent, false, nn::os::EventClearMode::EventClearMode_ManualClear);
        WLANTEST_ASSERT_RESULT_SUCCESS(nn::os::CreateThread(&actionframeThread, ReceiveActionFrameThread, this, threadStack, threadStackSize, nn::os::DefaultThreadPriority));

        Connect(ConnectParam);
        nn::os::StartThread(&actionframeThread);

        nn::os::WaitThread(&actionframeThread);
        WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::Disconnect(nn::wlan::LocalCommunicationMode_ClientSpectator, nullptr));

        nn::os::DestroyThread(&actionframeThread);
        nn::os::FinalizeEvent(&actionframeEvent);
    }

    // Master動作の既接続スキャン中に複数個のActionFrameが受信 MASTER
    TEST_F(LocalMasterFuncTest, L90_1)
    {
        size_t gutSize = ActionFrameBufferSize;
        nn::wlan::MacAddress clientMacArray[ClientMaxConnect];
        std::unique_ptr<uint8_t[]> getBuffer(new uint8_t[gutSize]);
        nn::wlan::MacAddress outMacAddress;
        size_t outSize;

        WLANTEST_ASSERT_TRUE(getBuffer.get() != nullptr);

        const nn::wlan::MasterBssParameters fncTestMasterBss = {
            1, false, 30, true, nn::wlan::RateSetLegacy_11gMask, nn::wlan::RateSetLegacy_11gMask,
            { nn::wlan::SecurityMode_StaticAes, nn::wlan::SecurityMode_StaticAes, 0, LocalMasterAes16Key }, nn::wlan::Ssid(LocalMasterTestSSID), 100
        };

        WLANTEST_ASSERT_TRUE(localMaster.SetupLocalMaster(fncTestMasterBss, TestMatchInfo, sizeof(TestMatchInfo) / sizeof(TestMatchInfo[0]), true));
        WLANTEST_EXPECT_RESULT_SUCCESS(nn::wlan::Local::AddSubtypeToRxEntryForActionFrame(localMaster.m_actionRxId, nn::wlan::ActionFrameType_Local));

        const nn::wlan::ScanParameters StartScanTestParams = {
            nn::wlan::ScanType_Active,{ 1, 6, 11 }, 3, 40, -1, &ssidList[0], 1, nn::wlan::MacAddress::CreateBroadcastMacAddress()
        };

        StartUnsyncScanThread(StartScanTestParams, 100);

        // MASTER CLIENT接続待機
        WLANTEST_ASSERT_TRUE(localMaster.LocalMasterConnectWait(clientMacArray, WlanTestConnectTimeOut, 1) > 0);

        sysTick = nn::os::GetSystemTick();
        // 60秒間通信
        while ((nn::os::GetSystemTick() - sysTick).ToTimeSpan().GetMilliSeconds() <= Time60s)
        {
            WLANTEST_ASSERT_TRUE(localMaster.LocalCancelActionFrameStart(Time15s));
            WLANTEST_EXPECT_RESULT_SUCCESS(nn::wlan::Local::GetActionFrame(&outMacAddress, getBuffer.get(), gutSize, &outSize, localMaster.m_actionRxId));
            localMaster.LocalCancelActionFrameStop();
        }

        // MASTER受信スレッド停止
        localMaster.LocalReceiveFrameStop();

        StopUnsyncScanThread();

    }

    // Master動作の既接続スキャン中に複数個のActionFrameが受信 CLIENT
    TEST_F(LocalClientFuncTest, L90_2)
    {
        size_t putSize = ActionFrameBufferSize;
        std::unique_ptr<uint8_t[]> putBuffer(new uint8_t[putSize]);
        nn::wlan::MacAddress clientMacAddress;
        nn::wlan::ConnectionStatus clientStatus;
        nn::os::Tick connectTime;
        uint32_t seqNumber = 0;

        WLANTEST_ASSERT_TRUE(putBuffer.get() != nullptr);

        const TestLocalConnectParams ConnectParam = {
            nn::wlan::Ssid(LocalMasterTestSSID), nn::wlan::MacAddress::CreateBroadcastMacAddress(), 1,
            { nn::wlan::SecurityMode_StaticAes, nn::wlan::SecurityMode_StaticAes, 0, LocalMasterAes16Key }, true, nn::wlan::BeaconIndication_Enable, 10
        };

        WLANTEST_ASSERT_TRUE(localClient.SetupLocalClient(ConnectParam.ssid, ConnectParam.security, TestMatchInfo, sizeof(TestMatchInfo) / sizeof(TestMatchInfo[0]), true, true));

        Connect(ConnectParam);

        WLANTEST_EXPECT_RESULT_SUCCESS(nn::wlan::Local::GetMacAddress(&clientMacAddress));
        WLANTEST_EXPECT_RESULT_SUCCESS(nn::wlan::Local::GetConnectionStatus(&connectionStatus));

        sysTick = nn::os::GetSystemTick();
        while (NN_STATIC_CONDITION(1))
        {
            WLANTEST_EXPECT_RESULT_SUCCESS(nn::wlan::Local::GetConnectionStatus(&clientStatus));

            // MASTERからの接続が切られるまで送信し続ける
            if (clientStatus.state != nn::wlan::ConnectionState_Connected)
            {
                break;
            }

            // beacon frame
            WLANTEST_EXPECT_TRUE(localClient.LocalMakeActionFrame(putBuffer.get(), putSize, connectionStatus.bssid, clientMacAddress, EtherTypes[0], TestMatchInfo[0], nn::wlan::ActionFrameType_Beacon));
            localClient.LocalSeqNumberAddFrame(putBuffer.get(), putSize, seqNumber++);
            WLANTEST_EXPECT_RESULT_SUCCESS(nn::wlan::Local::PutActionFrameOneShot(connectionStatus.bssid, putBuffer.get(), putSize, ConnectParam.channel, 30));
            nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(Time300msec));

            // local frame
            WLANTEST_EXPECT_TRUE(localClient.LocalMakeActionFrame(putBuffer.get(), putSize, connectionStatus.bssid, clientMacAddress, EtherTypes[0], TestMatchInfo[0], nn::wlan::ActionFrameType_Local));
            localClient.LocalSeqNumberAddFrame(putBuffer.get(), putSize, seqNumber++);
            WLANTEST_EXPECT_RESULT_SUCCESS(nn::wlan::Local::PutActionFrameOneShot(connectionStatus.bssid, putBuffer.get(), putSize, ConnectParam.channel, 30));
            nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(Time300msec));
        }

        Disconnect();
    }

    // Client動作の既接続スキャン中に複数個のActionFrameが受信 MASTER
    TEST_F(LocalMasterFuncTest, L91_1)
    {
        size_t putSize = ActionFrameBufferSize;
        std::unique_ptr<uint8_t[]> putBuffer(new uint8_t[putSize]);
        nn::wlan::MacAddress clientMacArray[ClientMaxConnect];
        nn::wlan::MacAddress clientMacAddress;
        uint32_t seqNumber = 0;

        WLANTEST_ASSERT_TRUE(putBuffer.get() != nullptr);

        const nn::wlan::MasterBssParameters fncTestMasterBss = {
            6, false, 30, true, nn::wlan::RateSetLegacy_11gMask, nn::wlan::RateSetLegacy_11gMask,
            { nn::wlan::SecurityMode_StaticAes, nn::wlan::SecurityMode_StaticAes, 0, LocalMasterAes16Key }, nn::wlan::Ssid(LocalMasterTestSSID), 100
        };

        WLANTEST_ASSERT_TRUE(localMaster.SetupLocalMaster(fncTestMasterBss, TestMatchInfo, sizeof(TestMatchInfo) / sizeof(TestMatchInfo[0]), true));
        WLANTEST_EXPECT_RESULT_SUCCESS(nn::wlan::Local::AddSubtypeToRxEntryForActionFrame(localMaster.m_actionRxId, nn::wlan::ActionFrameType_Local));

        // MASTER CLIENT接続待機
        WLANTEST_ASSERT_TRUE(localMaster.LocalMasterConnectWait(clientMacArray, WlanTestConnectTimeOut, 1) > 0);

        sysTick = nn::os::GetSystemTick();
        // 60秒間通信
        while ((nn::os::GetSystemTick() - sysTick).ToTimeSpan().GetMilliSeconds() <= Time60s)
        {
            WLANTEST_EXPECT_TRUE(localMaster.LocalMakeActionFrame(putBuffer.get(), putSize, clientMacArray[0],
                localMaster.m_macAddress, EtherTypes[0], TestMatchInfo[0], nn::wlan::ActionFrameType_Beacon));
            WLANTEST_EXPECT_TRUE(localMaster.LocalSeqNumberAddFrame(putBuffer.get(), putSize, seqNumber));
            WLANTEST_EXPECT_RESULT_SUCCESS(nn::wlan::Local::PutActionFrameOneShot(clientMacArray[0], putBuffer.get(), putSize, fncTestMasterBss.channel, 30));
            nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(Time300msec));

            WLANTEST_EXPECT_TRUE(localMaster.LocalMakeActionFrame(putBuffer.get(), putSize, clientMacArray[0],
                localMaster.m_macAddress, EtherTypes[0], TestMatchInfo[0], nn::wlan::ActionFrameType_Local));
            WLANTEST_EXPECT_TRUE(localMaster.LocalSeqNumberAddFrame(putBuffer.get(), putSize, seqNumber));
            WLANTEST_EXPECT_RESULT_SUCCESS(nn::wlan::Local::PutActionFrameOneShot(clientMacArray[0], putBuffer.get(), putSize, fncTestMasterBss.channel, 30));
            nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(Time300msec));
            seqNumber++;
        }

        // MASTER受信スレッド停止
        localMaster.LocalReceiveFrameStop();

        localMaster.LocalGetConnectInfo(&connectsInfo, GetLocalFrameType::FrameType_ActionFrame);
        for (int32_t i = 0; i < connectsInfo.connectCount; i++)
        {
            WLANTEST_EXPECT_TRUE(connectsInfo.facing[i].isConnected);
            WLANTEST_EXPECT_TRUE(connectsInfo.facing[i].getFrameCount > 0);
            WLANTEST_EXPECT_TRUE(connectsInfo.facing[i].lostFrameCount == 0);
        }

    }

    // Client動作の既接続スキャン中に複数個のActionFrameが受信 CLIENT
    TEST_F(LocalClientFuncTest, L91_2)
    {
        size_t gutSize = ActionFrameBufferSize * 2;
        std::unique_ptr<uint8_t[]> getBuffer(new uint8_t[gutSize]);
        nn::wlan::MacAddress clientMacAddress;
        nn::wlan::ConnectionStatus clientStatus;
        nn::wlan::MacAddress outMacAddress;
        size_t outSize;
        nn::Result result;

        WLANTEST_ASSERT_TRUE(getBuffer.get() != nullptr);

        const TestLocalConnectParams ConnectParam = {
            nn::wlan::Ssid(LocalMasterTestSSID), nn::wlan::MacAddress::CreateBroadcastMacAddress(), 6,
            { nn::wlan::SecurityMode_StaticAes, nn::wlan::SecurityMode_StaticAes, 0, LocalMasterAes16Key }, true, nn::wlan::BeaconIndication_Enable, 10
        };

        WLANTEST_ASSERT_TRUE(localClient.SetupLocalClient(ConnectParam.ssid, ConnectParam.security, TestMatchInfo, sizeof(TestMatchInfo) / sizeof(TestMatchInfo[0]), true, true));

        const nn::wlan::ScanParameters StartScanTestParams = {
            nn::wlan::ScanType_Active,{ 1, 6, 11 }, 3, 40, -1, &ssidList[0], 1, nn::wlan::MacAddress::CreateBroadcastMacAddress()
        };

        StartUnsyncScanThread(StartScanTestParams, 1000);

        Connect(ConnectParam);

        WLANTEST_EXPECT_RESULT_SUCCESS(nn::wlan::Local::GetMacAddress(&clientMacAddress));
        WLANTEST_EXPECT_RESULT_SUCCESS(nn::wlan::Local::GetConnectionStatus(&connectionStatus));

        sysTick = nn::os::GetSystemTick();
        while (NN_STATIC_CONDITION(1))
        {
            WLANTEST_EXPECT_RESULT_SUCCESS(nn::wlan::Local::GetConnectionStatus(&clientStatus));

            // MASTERからの接続が切られるまで送信し続ける
            if (clientStatus.state != nn::wlan::ConnectionState_Connected)
            {
                break;
            }

            WLANTEST_ASSERT_TRUE(localClient.LocalCancelActionFrameStart(Time15s));
            result = nn::wlan::Local::GetActionFrame(&outMacAddress, getBuffer.get(), gutSize, &outSize, localClient.m_actionRxId);
            if (result.IsSuccess() != true)
            {
                WLANTEST_EXPECT_RESULT_SUCCESS(nn::wlan::Local::GetConnectionStatus(&clientStatus));
                // MASTERからの接続が切られていたら評価せずに終了
                if (clientStatus.state != nn::wlan::ConnectionState_Connected)
                {
                    NN_LOG("             Master Disconnected\n");
                    break;
                }
                WLANTEST_EXPECT_RESULT_SUCCESS(result);
            }
            localClient.LocalCancelActionFrameStop();
        }

        // CLIENT受信スレッド停止
        localClient.LocalReceiveFrameStop();

        Disconnect();

        StopUnsyncScanThread();
    }

    // 切断後にActionFrameの受信 MASTER
    TEST_F(LocalMasterFuncTest, L92_1)
    {
        std::unique_ptr<uint8_t[]> putBuffer(new uint8_t[ActionFrameBufferSize]);
        nn::wlan::MacAddress clientMacArray[ClientMaxConnect];
        nn::wlan::MacAddress clientMacAddress;
        uint32_t seqNumber = 0;

        WLANTEST_ASSERT_TRUE(putBuffer.get() != nullptr);

        const nn::wlan::MasterBssParameters fncTestMasterBss = {
            1, false, 30, true, nn::wlan::RateSetLegacy_11gMask, nn::wlan::RateSetLegacy_11gMask,
            { nn::wlan::SecurityMode_Open, nn::wlan::SecurityMode_Open, 0, "" }, nn::wlan::Ssid(LocalMasterTestSSID), 100
        };

        WLANTEST_ASSERT_TRUE(localMaster.SetupLocalMaster(fncTestMasterBss, TestMatchInfo, sizeof(TestMatchInfo) / sizeof(TestMatchInfo[0]), true));
        WLANTEST_EXPECT_RESULT_SUCCESS(nn::wlan::Local::AddSubtypeToRxEntryForActionFrame(localMaster.m_actionRxId, nn::wlan::ActionFrameType_Local));

        // MASTER CLIENT接続待機
        WLANTEST_ASSERT_TRUE(localMaster.LocalMasterConnectWait(clientMacArray, WlanTestConnectTimeOut, 1) > 0);
        // MASTER CLIENT切断待機
        WLANTEST_ASSERT_TRUE(localMaster.LocalMasterDisConnectWait(WlanTestConnectTimeOut) == 0);

        // 切断後のActionFrame送信
        for (int32_t i = 0; i < 20; i++)
        {
            WLANTEST_EXPECT_TRUE(localMaster.LocalMakeActionFrame(putBuffer.get(), ActionFrameBufferSize, clientMacArray[0],
                localMaster.m_macAddress, EtherTypes[0], TestMatchInfo[0], nn::wlan::ActionFrameType_Beacon));
            WLANTEST_EXPECT_TRUE(localMaster.LocalSeqNumberAddFrame(putBuffer.get(), ActionFrameBufferSize, seqNumber));
            WLANTEST_EXPECT_RESULT_SUCCESS(nn::wlan::Local::PutActionFrameOneShot(clientMacArray[0], putBuffer.get(), ActionFrameBufferSize, fncTestMasterBss.channel, 30));
            nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(Time300msec));
        }

        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(Time10s));
    }

    // 切断後にActionFrameの受信 CLIENT
    TEST_F(LocalClientFuncTest, L92_2)
    {
        nn::wlan::MacAddress macAddress = nn::wlan::MacAddress::CreateBroadcastMacAddress();
        const TestLocalConnectParams ConnectParam = {
            nn::wlan::Ssid(LocalMasterTestSSID), nn::wlan::MacAddress::CreateBroadcastMacAddress(), 1,
            { nn::wlan::SecurityMode_Open, nn::wlan::SecurityMode_Open, 0, "" }, true, nn::wlan::BeaconIndication_Enable, 10
        };
        std::unique_ptr<uint8_t[]> pTestBuffer(new uint8_t[TestScanMaxSize]);
        WLANTEST_ASSERT_TRUE(pTestBuffer.get() != nullptr);

        WLANTEST_ASSERT_TRUE(localClient.SetupLocalClient(ConnectParam.ssid, ConnectParam.security, TestMatchInfo, sizeof(TestMatchInfo) / sizeof(TestMatchInfo[0]), false, true));

        nn::os::InitializeEvent(&actionframeEvent, false, nn::os::EventClearMode::EventClearMode_ManualClear);
        WLANTEST_ASSERT_RESULT_SUCCESS(nn::os::CreateThread(&actionframeThread, ReceiveActionFrameThread, this, threadStack, threadStackSize, nn::os::DefaultThreadPriority));

        Connect(ConnectParam);
        WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::Disconnect(nn::wlan::LocalCommunicationMode_ClientSpectator, nullptr));

        nn::os::StartThread(&actionframeThread);
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(5000));

        nn::os::WaitThread(&actionframeThread);
        nn::os::DestroyThread(&actionframeThread);
        nn::os::FinalizeEvent(&actionframeEvent);
    }

    // Clientが通信負荷がかかった状態で、自発的にMasterと切断した場合、即座に切断 MASTER
    TEST_F(LocalMasterFuncTest, L94_1)
    {
        nn::wlan::DisconnectClient client;
        const nn::wlan::MasterBssParameters fncTestMasterBss = {
            11, false, 30, true, nn::wlan::RateSetLegacy_11bMask, nn::wlan::RateSetLegacy_11bMask,
            { nn::wlan::SecurityMode_Open, nn::wlan::SecurityMode_Open, 0, "" }, nn::wlan::Ssid(LocalMasterTest2SSID), 100
        };

        // ジョイコン2セット（各4台）
        WLANTEST_ASSERT_TRUE(localMaster.SetBtMode(LocalApiBtNode4));

        WLANTEST_ASSERT_TRUE(localMaster.SetupLocalMaster(fncTestMasterBss, TestMatchInfo, sizeof(TestMatchInfo) / sizeof(TestMatchInfo[0])));

        // TCP SERVERスレッド起動
        StartTcpServer();

        // MASTER受信スレッド起動
        WLANTEST_ASSERT_TRUE(localMaster.LocalReceiveFrameStart(BufferSize100k, WlanTestPacketBufferSize, 1, 1, WlanTestGetFrameCount));
        // CLIENT接続待機
        WLANTEST_ASSERT_TRUE(localMaster.LocalReceiveWait(WlanTestMasterWaitTimeOut));

        // MASTER送信スレッド起動
        WLANTEST_ASSERT_TRUE(localMaster.LocalSendFrameOneShotWaitStart(WlanTestPacketBufferSize, Time120s, Interval_33m, true));

        sysTick = nn::os::GetSystemTick();

        // 60秒待機
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(Time60s));

        // MASTER送信スレッド停止
        localMaster.LocalSendFrameStop();
        // MASTER受信スレッド停止
        localMaster.LocalReceiveFrameStop();

        client.clientMacAddress = nn::wlan::MacAddress::CreateBroadcastMacAddress();
        client.reasonCode = nn::wlan::Dot11ReasonCode_DisassocInactivity;
        NN_LOG("             L94 - Master Disconnect Start(%lld msec)\n", (nn::os::GetSystemTick() - sysTick).ToTimeSpan().GetMilliSeconds());
        WLANTEST_EXPECT_RESULT_SUCCESS(nn::wlan::Local::Disconnect(nn::wlan::LocalCommunicationMode_Master, &client));
        NN_LOG("             L94 - Master Disconnect End(%lld msec)\n", (nn::os::GetSystemTick() - sysTick).ToTimeSpan().GetMilliSeconds());

        StopTcpServer();
    }

    // Clientが通信負荷がかかった状態で、自発的にMasterと切断した場合、即座に切断 CLIENT
    TEST_F(LocalClientFuncTest, L94_2)
    {
        const TestLocalConnectParams ConnectParam = {
            nn::wlan::Ssid(LocalMasterTest2SSID), nn::wlan::MacAddress::CreateBroadcastMacAddress(), 11,
            { nn::wlan::SecurityMode_Open, nn::wlan::SecurityMode_Open, 0, "" }, true, nn::wlan::BeaconIndication_Enable, 10
        };

        // ジョイコン2セット（各4台）
        WLANTEST_ASSERT_TRUE(localClient.SetBtMode(LocalApiBtNode4));

        WLANTEST_ASSERT_TRUE(localClient.SetupLocalClient(ConnectParam.ssid, ConnectParam.security, TestMatchInfo, sizeof(TestMatchInfo) / sizeof(TestMatchInfo[0]), true, false));

        sysTick = nn::os::GetSystemTick();
        Connect(ConnectParam);
        NN_LOG("             L94 - Cliet Connected(%lld msec)\n", (nn::os::GetSystemTick() - sysTick).ToTimeSpan().GetMilliSeconds());

        WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::GetConnectionStatus(&connectionStatus));

        // TCP CLIENTスレッド起動
        StartTcpClient(STATIC_CLIENT1_IP_ADDR);

        // 送信スレッド起動
        WLANTEST_ASSERT_TRUE(localClient.LocalSendFrameOneShotWaitStart(WlanTestPacketBufferSize, Time120s, Interval_33m, false));

        sysTick = nn::os::GetSystemTick();
        // 通信ループ
        while (NN_STATIC_CONDITION(1))
        {
            // MASTERからの切断待機
            if (nn::os::TimedWaitSystemEvent(&connectionEvent, nn::TimeSpan::FromMilliSeconds(Interval_100m)) == true)
            {
                WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::GetConnectionStatus(&connectionStatus));
                if (connectionStatus.state != nn::wlan::ConnectionState_Connected)
                {
                    break;
                }
            }
        }

        // TCP CLIENTスレッド停止
        StopTcpClient();

        // 送信スレッド停止
        localClient.LocalSendFrameStop();

        // 切断イベント待機
        NN_LOG("             L94 - Cliet Disconnect Event Start(%lld msec)\n", (nn::os::GetSystemTick() - sysTick).ToTimeSpan().GetMilliSeconds());
        Disconnect();
        NN_LOG("             L94 - Cliet Disconnect Event End(%lld msec)\n", (nn::os::GetSystemTick() - sysTick).ToTimeSpan().GetMilliSeconds());

    }

    // Master接続後、過多な負荷通信を行い、通信中にMasterから切断された際、即座に切断 MASTER
    // Master接続後、過多な負荷通信を行い、通信中にMasterから切断された後、Masterと再接続
    TEST_F(LocalMasterFuncTest, L95_L96_1)
    {
        nn::wlan::DisconnectClient client;
        const nn::wlan::MasterBssParameters fncTestMasterBss = {
            11, false, 30, true, nn::wlan::RateSetLegacy_11bMask, nn::wlan::RateSetLegacy_11bMask,
            { nn::wlan::SecurityMode_Open, nn::wlan::SecurityMode_Open, 0, "" }, nn::wlan::Ssid(LocalMasterTestSSID), 100
        };

        WLANTEST_ASSERT_TRUE(localMaster.SetupLocalMaster(fncTestMasterBss, TestMatchInfo, sizeof(TestMatchInfo) / sizeof(TestMatchInfo[0])));

        // TCP SERVERスレッド起動
        StartTcpServer();

        // MASTER受信スレッド起動
        WLANTEST_ASSERT_TRUE(localMaster.LocalReceiveFrameStart(BufferSize100k, WlanTestPacketBufferSize, 1, 1, WlanTestGetFrameCount));
        // CLIENT接続待機
        WLANTEST_ASSERT_TRUE(localMaster.LocalReceiveWait(WlanTestMasterWaitTimeOut));

        // MASTER送信スレッド起動
        WLANTEST_ASSERT_TRUE(localMaster.LocalSendFrameOneShotWaitStart(WlanTestPacketBufferSize, Time120s, Interval_33m, true));

        sysTick = nn::os::GetSystemTick();

        // 待機
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(Time60s));

        // MASTER送信スレッド停止
        localMaster.LocalSendFrameStop();

        // MASTER受信スレッド停止
        localMaster.LocalReceiveFrameStop();

        StopTcpServer();

        client.clientMacAddress = nn::wlan::MacAddress::CreateBroadcastMacAddress();
        client.reasonCode = nn::wlan::Dot11ReasonCode_DisassocInactivity;
        NN_LOG("             L95 - Master Disconnect Event Start(%lld msec)\n", (nn::os::GetSystemTick() - sysTick).ToTimeSpan().GetMilliSeconds());
        WLANTEST_EXPECT_RESULT_SUCCESS(nn::wlan::Local::Disconnect(nn::wlan::LocalCommunicationMode_Master, &client));
        NN_LOG("             L95 - Master Disconnect Event End(%lld msec)\n", (nn::os::GetSystemTick() - sysTick).ToTimeSpan().GetMilliSeconds());

        // MASTER再接続
        WLANTEST_ASSERT_TRUE(localMaster.LocalMasterConnectWait(connectMacArray, WlanTestConnectTimeOut, 1) > 0);
    }

    // Master接続後、過多な負荷通信を行い、通信中にMasterから切断された際、即座に切断 CLIENT
    // Master接続後、過多な負荷通信を行い、通信中にMasterから切断された後、Masterと再接続
    TEST_F(LocalClientFuncTest, L95_L96_2)
    {
        nn::wlan::MacAddress clientMacAddress;
        nn::wlan::ConnectionStatus clientStatus;
        std::unique_ptr<uint8_t[]> putBuffer(new uint8_t[WlanTestPacketBufferSize]);
        WLANTEST_ASSERT_TRUE(putBuffer.get() != nullptr);

        const TestLocalConnectParams ConnectParam = {
            nn::wlan::Ssid(LocalMasterTestSSID), nn::wlan::MacAddress::CreateBroadcastMacAddress(), 11,
            { nn::wlan::SecurityMode_Open, nn::wlan::SecurityMode_Open, 0, "" }, true, nn::wlan::BeaconIndication_Enable, 10
        };

        WLANTEST_ASSERT_TRUE(localClient.SetupLocalClient(ConnectParam.ssid, ConnectParam.security, TestMatchInfo, sizeof(TestMatchInfo) / sizeof(TestMatchInfo[0]), true, false));

        Connect(ConnectParam);
        NN_LOG("               %s ---> Cliet Connected Frame Start(%lld msec)\n", __FUNCTION__, (nn::os::GetSystemTick() - sysTick).ToTimeSpan().GetMilliSeconds());

        WLANTEST_EXPECT_RESULT_SUCCESS(nn::wlan::Local::GetMacAddress(&clientMacAddress));
        WLANTEST_EXPECT_RESULT_SUCCESS(nn::wlan::Local::GetConnectionStatus(&connectionStatus));
        WLANTEST_ASSERT_TRUE(localClient.LocalReceiveFrameStart(BufferSize100k, WlanTestPacketBufferSize, 1, 1, WlanTestAgingGetFrameCount));
        WLANTEST_ASSERT_TRUE(localClient.LocalReceiveWait(WlanMaxClientWaitMsecTimeOut, Time1s, true));

        // 送信スレッド起動
        WLANTEST_ASSERT_TRUE(localClient.LocalSendFrameOneShotWaitStart(WlanTestPacketBufferSize, Time120s, Interval_33m, false));

        sysTick = nn::os::GetSystemTick();
        // 通信ループ
        while (NN_STATIC_CONDITION(1))
        {
            // MASTERからの切断待機
            if (nn::os::TimedWaitSystemEvent(&connectionEvent, nn::TimeSpan::FromMilliSeconds(Interval_100m)) == true)
            {
                WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::GetConnectionStatus(&connectionStatus));
                if (connectionStatus.state != nn::wlan::ConnectionState_Connected)
                {
                    break;
                }
            }
        }
        NN_LOG("               %s ---> Cliet Connected Frame End(%lld msec)\n", __FUNCTION__, (nn::os::GetSystemTick() - sysTick).ToTimeSpan().GetMilliSeconds());

        // TCP CLIENTスレッド停止
        StopTcpClient();

        // 送信スレッド停止
        localClient.LocalSendFrameStop();
        // 受信スレッド停止
        localClient.LocalReceiveFrameStop();
        Disconnect();

        // 再接続
        sysTick = nn::os::GetSystemTick();
        Connect(ConnectParam);
        NN_LOG("               %s ---> Cliet Connected Time(%lld msec)\n", __FUNCTION__, (nn::os::GetSystemTick() - sysTick).ToTimeSpan().GetMilliSeconds());
        sysTick = nn::os::GetSystemTick();

        Disconnect();
    }

    // Master接続後、過多な負荷通信を行い、通信中にMasterをロストした際、設定した時間経過後に切断通知 MASTER
    TEST_F(LocalMasterFuncTest, L97_1)
    {
        int64_t stopTime = 0;

        const nn::wlan::MasterBssParameters fncTestMasterBss = {
            11, false, 30, true, nn::wlan::RateSetLegacy_11bMask, nn::wlan::RateSetLegacy_11bMask,
            { nn::wlan::SecurityMode_Open, nn::wlan::SecurityMode_Open, 0, "" }, nn::wlan::Ssid(LocalMasterTest2SSID), 100
        };

        WLANTEST_ASSERT_TRUE(localMaster.SetupLocalMaster(fncTestMasterBss, TestMatchInfo, sizeof(TestMatchInfo) / sizeof(TestMatchInfo[0])));

        StartTcpServer();

        // MASTER受信スレッド起動
        WLANTEST_ASSERT_TRUE(localMaster.LocalReceiveFrameStart(BufferSize100k, WlanTestPacketBufferSize, 1, 1, WlanTestGetFrameCount));

        // CLIENT接続待機
        WLANTEST_ASSERT_TRUE(localMaster.LocalReceiveWait(WlanTestMasterWaitTimeOut));

        sysTick = nn::os::GetSystemTick();
        // 30秒～60秒のランダムな時間で停止
        stopTime = GetRandom(30, 60) * 1000;
        NN_LOG("Master Random Send Stop Time:%lld msec\n", stopTime);

        // MASTER送信スレッド起動
        WLANTEST_ASSERT_TRUE(localMaster.LocalSendFrameOneShotWaitStart(WlanTestPacketBufferSize, stopTime, Interval_33m, true));

        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(stopTime));
        NN_LOG("Master KeepAlive Lost Test Client Pawer Off please\n");
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(Time1day));

        WLANTEST_ASSERT_TRUE(localMaster.LocalMasterDisConnectWait(stopTime) == 0);

        // MASTER送信スレッド停止
        localMaster.LocalSendFrameStop();

        // MASTER受信スレッド停止
        localMaster.LocalReceiveFrameStop();

        StopTcpServer();
    }

    // Master接続後、過多な負荷通信を行い、通信中にMasterをロストした際、設定した時間経過後に切断通知 CLIENT  (TCP CLIENT1台のみ)
    TEST_F(LocalClientFuncTest, L97_2)
    {
        nn::wlan::ConnectionStatus clientStatus;
        const TestLocalConnectParams ConnectParam = {
            nn::wlan::Ssid(LocalMasterTest2SSID), nn::wlan::MacAddress::CreateBroadcastMacAddress(), 11,
            { nn::wlan::SecurityMode_Open, nn::wlan::SecurityMode_Open, 0, "" }, true, nn::wlan::BeaconIndication_Enable, 10
        };

        WLANTEST_ASSERT_TRUE(localClient.SetupLocalClient(ConnectParam.ssid, ConnectParam.security, TestMatchInfo, sizeof(TestMatchInfo) / sizeof(TestMatchInfo[0]), true, false));

        Connect(ConnectParam);

        // 受信スレッド起動
        WLANTEST_ASSERT_TRUE(localClient.LocalReceiveFrameStart(BufferSize200k, WlanTestRecvBufferSize, 1, 1, WlanTestAgingGetFrameCount));
        WLANTEST_ASSERT_TRUE(localClient.LocalReceiveWait(WlanMaxClientWaitMsecTimeOut, Time1s, true));

        sysTick = nn::os::GetSystemTick();
        NN_LOG("             L97 - Cliet Connected(%lld msec)\n", (nn::os::GetSystemTick() - sysTick).ToTimeSpan().GetMilliSeconds());

        // TCP CLIENTスレッド起動
        StartTcpClient(STATIC_CLIENT1_IP_ADDR);

        NN_LOG("             L97 - Cliet Connect Start(%lld msec)\n", (nn::os::GetSystemTick() - sysTick).ToTimeSpan().GetMilliSeconds());

        // 送信スレッド起動
        WLANTEST_ASSERT_TRUE(localClient.LocalSendFrameOneShotWaitStart(WlanTestPacketBufferSize, Time60s, Interval_33m, false));

        // 切断イベント待機
        sysTick = nn::os::GetSystemTick();
        NN_LOG("             L97 - Cliet Disconnect Event Start(%lld msec)\n", (nn::os::GetSystemTick() - sysTick).ToTimeSpan().GetMilliSeconds());
        WLANTEST_EXPECT_TRUE(nn::os::TimedWaitSystemEvent(&connectionEvent, nn::TimeSpan::FromSeconds(WlanTestConnectTimeOut)));
        NN_LOG("             L97 - Cliet Disconnect Event End(%lld msec)\n", (nn::os::GetSystemTick() - sysTick).ToTimeSpan().GetMilliSeconds());

        // 送信スレッド停止
        localClient.LocalSendFrameStop();
        // 受信スレッド停止
        localClient.LocalReceiveFrameStop();

        // TCP CLIENTスレッド停止
        StopTcpClient();

        Disconnect();
    }

    // Masterに通信負荷がかかった状態で、Destroy処理ができる MASTER
    TEST_F(LocalMasterFuncTest, L98_1)
    {
        const nn::wlan::MasterBssParameters fncTestMasterBss = {
            11, false, 30, true, nn::wlan::RateSetLegacy_11bMask, nn::wlan::RateSetLegacy_11bMask,
            { nn::wlan::SecurityMode_Open, nn::wlan::SecurityMode_Open, 0, "" }, nn::wlan::Ssid(LocalMasterTest2SSID), 100
        };

        WLANTEST_ASSERT_TRUE(localMaster.SetupLocalMaster(fncTestMasterBss, TestMatchInfo, sizeof(TestMatchInfo) / sizeof(TestMatchInfo[0])));

        // MASTER受信スレッド起動
        WLANTEST_ASSERT_TRUE(localMaster.LocalReceiveFrameStart(BufferSize100k, WlanTestPacketBufferSize, 1, 1, WlanTestGetFrameCount));

        // CLIENT接続待機
        WLANTEST_ASSERT_TRUE(localMaster.LocalReceiveWait(WlanTestMasterWaitTimeOut));

        // TCP SERVERスレッド起動
        StartTcpServer();

        // MASTER送信スレッド起動
        WLANTEST_ASSERT_TRUE(localMaster.LocalSendFrameOneShotWaitStart(WlanTestPacketBufferSize, Time60s, Interval_33m, true));

        // 60秒で停止
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(Time60s));

        WLANTEST_EXPECT_RESULT_SUCCESS(nn::wlan::Local::DestroyBss());

        // MASTER受信スレッド停止
        localMaster.LocalReceiveFrameStop();

        StopTcpServer();
    }

    // Masterに通信負荷がかかった状態で、Destroy処理ができる MASTER CLIENT
    TEST_F(LocalClientFuncTest, L98_2)
    {
        nn::wlan::ConnectionStatus clientStatus;
        const TestLocalConnectParams ConnectParam = {
            nn::wlan::Ssid(LocalMasterTest2SSID), nn::wlan::MacAddress::CreateBroadcastMacAddress(), 11,
            { nn::wlan::SecurityMode_Open, nn::wlan::SecurityMode_Open, 0, "" }, true, nn::wlan::BeaconIndication_Enable, 10
        };

        WLANTEST_ASSERT_TRUE(localClient.SetupLocalClient(ConnectParam.ssid, ConnectParam.security, TestMatchInfo, sizeof(TestMatchInfo) / sizeof(TestMatchInfo[0]), true, false));

        Connect(ConnectParam);
        sysTick = nn::os::GetSystemTick();
        NN_LOG("             L98 - Client Connected(%lld msec)\n", (nn::os::GetSystemTick() - sysTick).ToTimeSpan().GetMilliSeconds());

        // TCP CLIENTスレッド起動
        StartTcpClient(STATIC_CLIENT1_IP_ADDR);

        // 送信スレッド起動
        WLANTEST_ASSERT_TRUE(localClient.LocalSendFrameOneShotWaitStart(WlanTestPacketBufferSize, Time60s, Interval_33m, false));

        sysTick = nn::os::GetSystemTick();
        NN_LOG("             L98 - Client PutFrame Start(%lld msec)\n", (nn::os::GetSystemTick() - sysTick).ToTimeSpan().GetMilliSeconds());

        while (NN_STATIC_CONDITION(1))
        {
            // MASTERからの接続が切られるまで送信し続ける
            if (nn::os::TimedWaitSystemEvent(&connectionEvent, nn::TimeSpan::FromMilliSeconds(Interval_100m)) == true)
            {
                WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::GetConnectionStatus(&connectionStatus));
                if (connectionStatus.state != nn::wlan::ConnectionState_Connected)
                {
                    break;
                }
            }
        }

        NN_LOG("             L98 - Client PutFrame End(%lld msec)\n", (nn::os::GetSystemTick() - sysTick).ToTimeSpan().GetMilliSeconds());
        StopTcpClient();

        Disconnect();
    }

    // 1,6,11chを1時間連続してスキャンし続ける
    TEST_F(LocalClientFuncTest, L102)
    {
        nn::os::Tick spanTick;
        const nn::wlan::ScanParameters StartScanTestParams = {
             nn::wlan::ScanType_Passive,{ 1, 6, 11 }, 3, 110, -1, nullptr, 0, nn::wlan::MacAddress::CreateBroadcastMacAddress()
        };
        const TestLocalConnectParams ConnectParam = {
            nn::wlan::Ssid(LocalMasterTestSSID), nn::wlan::MacAddress::CreateBroadcastMacAddress(), 1,
                { nn::wlan::SecurityMode_Open, nn::wlan::SecurityMode_Open, 0, "" }, true, nn::wlan::BeaconIndication_Enable, 10
        };

        // ジョイコン1セット（各2台）
        WLANTEST_ASSERT_TRUE(localClient.SetBtMode(LocalApiBtNode2));

        std::unique_ptr<uint8_t[]> pTestBuffer(new uint8_t[TestScanMaxSize]);
        WLANTEST_ASSERT_TRUE(pTestBuffer.get() != nullptr);
        nn::wlan::ScanIeMatchInfo info;
        sysTick = nn::os::GetSystemTick();
        uint32_t scanFailCnt = 0;
        uint32_t scanCnt = 0;
        nn::Result result;
        bool isResult = false;

        while (NN_STATIC_CONDITION(1))
        {
            // 1時間連続でScanを行う
            if( (nn::os::GetSystemTick() - sysTick).ToTimeSpan().GetMilliSeconds() <= Time1h )
            {
                scanCnt++;

                NN_LOG("             StartScan Start\n");
                spanTick = nn::os::GetSystemTick();
                result = nn::wlan::Local::StartScan(pTestBuffer.get(), TestScanMaxSize, StartScanTestParams, &info);
                NN_LOG("             StartScan End(%lld msec)\n", (nn::os::GetSystemTick() - spanTick).ToTimeSpan().GetMilliSeconds());
                WLANTEST_EXPECT_RESULT_SUCCESS(result);
                if (result.IsSuccess() == true)
                {
                    isResult = CompareScanSsid(pTestBuffer.get(), TestScanMaxSize, ConnectParam.ssid);
                    if (isResult != true)
                    {
                        scanFailCnt++;
                    }
                }
                else
                {
                    scanFailCnt++;
                }
            }
            else
            {
                break;
            }
        }

        NN_LOG("             StartScan Fail Count(%lu/%lu)\n", scanFailCnt, scanCnt);
        NN_LOG("             StartScan Total Count(%lu)\n", scanCnt);

        // 仮にReleaseJ5でPassとした成功率で判定
        double passRate = 100 - ((static_cast<double>(scanFailCnt) / static_cast<double>(scanCnt)) * 100);
        WLANTEST_EXPECT_TRUE(passRate >= 99);
    }

    // 同一Masterに1時間連続で接続と切断を繰り返し
    TEST_F(LocalClientFuncTest, L103)
    {
        const TestLocalConnectParams ConnectParam = {
            nn::wlan::Ssid(LocalMasterTestSSID), nn::wlan::MacAddress::CreateBroadcastMacAddress(), 1,
                { nn::wlan::SecurityMode_Open, nn::wlan::SecurityMode_Open, 0, "" }, true, nn::wlan::BeaconIndication_Enable, 10
        };

        // ジョイコン1セット（各2台）
        WLANTEST_ASSERT_TRUE(localClient.SetBtMode(LocalApiBtNode2));

        sysTick = nn::os::GetSystemTick();
        while ((nn::os::GetSystemTick() - sysTick).ToTimeSpan().GetMilliSeconds() <= Time1h)
        {
            // 1時間連続で接続/切断を行う
            Connect(ConnectParam);
            Disconnect();
        }
    }

    // 同一Masterが7台のClientに対して連続して8時間接続と切断を繰り返し MASTER
    TEST_F(LocalMasterFuncTest, L104_1)
    {
        nn::wlan::MacAddress clientMacArray[ClientMaxConnect];

        const nn::wlan::MasterBssParameters fncTestMasterBss = {
            6, false, 30, true, nn::wlan::RateSetLegacy_11gMask, nn::wlan::RateSetLegacy_11gMask,
            { nn::wlan::SecurityMode_StaticAes, nn::wlan::SecurityMode_StaticAes, 0, LocalMasterAes16Key }, nn::wlan::Ssid(LocalMasterTestSSID), 100
        };

        // ジョイコン1セット（各2台）
        WLANTEST_ASSERT_TRUE(localMaster.SetBtMode(LocalApiBtNode2));

        WLANTEST_ASSERT_TRUE(localMaster.SetupLocalMaster(fncTestMasterBss, TestMatchInfo, sizeof(TestMatchInfo) / sizeof(TestMatchInfo[0])));

        // MASTER CLIENT接続待機
        WLANTEST_ASSERT_TRUE(localMaster.LocalMasterConnectWait(clientMacArray, WlanTestConnectTimeOut, 1) > 0);

        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(Time8h + Time180s));
    }

    // 同一Masterが7台のClientに対して連続して8時間接続と切断を繰り返し CLIENT
    TEST_F(LocalClientFuncTest, L104_2)
    {
        nn::os::Tick connectTime;
        bool isConnect = false;
        int disConnectTime = 0;

        LocalClientConnectParameter ConnectParam = {
            nn::wlan::Ssid(LocalMasterTestSSID), nn::wlan::MacAddress::CreateBroadcastMacAddress(), 6,
            { nn::wlan::SecurityMode_StaticAes, nn::wlan::SecurityMode_StaticAes, 0, LocalMasterAes16Key }, true, nn::wlan::BeaconIndication_Enable, 10
        };

        // ジョイコン1セット（各2台）
        WLANTEST_ASSERT_TRUE(localClient.SetBtMode(LocalApiBtNode2));

        WLANTEST_ASSERT_TRUE(localClient.SetupLocalClient(ConnectParam.ssid, ConnectParam.security, TestMatchInfo, sizeof(TestMatchInfo) / sizeof(TestMatchInfo[0]), true, false));

        sysTick = nn::os::GetSystemTick();
        while ((nn::os::GetSystemTick() - sysTick).ToTimeSpan().GetMilliSeconds() <= Time8h)
        {
            if (isConnect != true)
            {
                disConnectTime = GetRandom(5, 10);
                localClient.LocalClientConnect(ConnectParam, Time30s);
                isConnect = true;
                connectTime = nn::os::GetSystemTick();
            }
            else
            {
                if ((nn::os::GetSystemTick() - connectTime).ToTimeSpan().GetSeconds() <= disConnectTime)
                {
                    Disconnect();
                    isConnect = false;
                }
            }
            nn::os::SleepThread(nn::TimeSpan::FromSeconds(1));
        }

        StopUnsyncScanThread();
        // 1秒待機
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(Time1s));

        Disconnect();
    }

    // Masterと接続後、自発的な通信が無い状態で30分間、Masterとのリンクがきれないこと MASTER
    TEST_F(LocalMasterFuncTest, L105_1)
    {
        nn::wlan::MacAddress clientMacArray[ClientMaxConnect];
        nn::wlan::MacAddress outMacAddress;
        std::unique_ptr<uint8_t[]> getBuffer(new uint8_t[TestScanMaxSize]);

        WLANTEST_ASSERT_TRUE(getBuffer.get() != nullptr);

        const nn::wlan::MasterBssParameters fncTestMasterBss = {
            11, false, 180, true, nn::wlan::RateSetLegacy_11gMask, nn::wlan::RateSetLegacy_11gMask,
            { nn::wlan::SecurityMode_Open, nn::wlan::SecurityMode_Open, 0, "" }, nn::wlan::Ssid(LocalMasterTestSSID), 100
        };

        // ジョイコン1セット（各2台）
        WLANTEST_ASSERT_TRUE(localMaster.SetBtMode(LocalApiBtNode2));

        WLANTEST_ASSERT_TRUE(localMaster.SetupLocalMaster(fncTestMasterBss, TestMatchInfo, sizeof(TestMatchInfo) / sizeof(TestMatchInfo[0]), true));

        // MASTER CLIENT接続待機
        WLANTEST_ASSERT_TRUE(localMaster.LocalMasterConnectWait(clientMacArray, WlanTestConnectTimeOut, ClientMaxConnect) > 0);

        // MASTER CLIENT切断待機
        sysTick = nn::os::GetSystemTick();
        NN_LOG("             L105 - Master Client Disconnect Wait Start : %lu msec\n", (nn::os::GetSystemTick() - sysTick).ToTimeSpan().GetMilliSeconds());
        WLANTEST_ASSERT_TRUE(localMaster.LocalMasterDisConnectWait(clientMacArray, Time1h) == 0);
        NN_LOG("             L105 - Master Client Disconnect Wait End : %lu msec\n", (nn::os::GetSystemTick() - sysTick).ToTimeSpan().GetMilliSeconds());
    }

    // Masterと接続後、自発的な通信が無い状態で30分間、Masterとのリンクがきれないこと CLIENT
    TEST_F(LocalClientFuncTest, L105_2)
    {
        nn::wlan::ConnectionStatus clientStatus;
        nn::wlan::LinkLevel level;
        const TestLocalConnectParams ConnectParam = {
            nn::wlan::Ssid(LocalMasterTestSSID), nn::wlan::MacAddress::CreateBroadcastMacAddress(), 11,
            { nn::wlan::SecurityMode_Open, nn::wlan::SecurityMode_Open, 0, "" }, true, nn::wlan::BeaconIndication_Enable, 10
        };

        // ジョイコン1セット（各2台）
        WLANTEST_ASSERT_TRUE(localClient.SetBtMode(LocalApiBtNode2));

        WLANTEST_ASSERT_TRUE(localClient.SetupLocalClient(ConnectParam.ssid, ConnectParam.security, TestMatchInfo, sizeof(TestMatchInfo) / sizeof(TestMatchInfo[0]), true, false));
        Connect(ConnectParam);

        sysTick = nn::os::GetSystemTick();
        while ((nn::os::GetSystemTick() - sysTick).ToTimeSpan().GetMilliSeconds() <= Time180s)
        {
            // 通信が切断されたらNG判定
            if (nn::os::TimedWaitSystemEvent(&connectionEvent, nn::TimeSpan::FromMilliSeconds(Time100msec)) == true)
            {
                WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::GetConnectionStatus(&clientStatus));
                WLANTEST_EXPECT_TRUE(clientStatus.state == nn::wlan::ConnectionState_Connected);
                if (clientStatus.state != nn::wlan::ConnectionState_Connected)
                {
                    localClient.LocalTraceConnectsInfo(true);
                    break;
                }
            }
            WLANTEST_EXPECT_RESULT_SUCCESS(nn::wlan::Local::GetLinkLevel(&level));
            WLANTEST_EXPECT_TRUE(nn::wlan::LinkLevel_0 != level);
        }

        Disconnect();
    }

    // MasterとClient1台接続で、1時間の連続通信を行い、途切れず100Mbps以上のスループットで通信できること CLIENT
    TEST_F(LocalMasterFuncTest, L106_1)
    {
        uint connectCount = 0;
        bool isEventSygnal = false;

        // 試験時間短縮フラグを確認
        int time = Time1h;
        if (g_IsShort) time = Time1800s;

        const nn::wlan::MasterBssParameters fncTestMasterBss = {
            6, false, 30, true, nn::wlan::RateSetLegacy_11gMask, nn::wlan::RateSetLegacy_11gMask,
            { nn::wlan::SecurityMode_StaticAes, nn::wlan::SecurityMode_StaticAes, 0, LocalMasterAes16Key }, nn::wlan::Ssid(LocalMasterTest2SSID), 10
        };

        WLANTEST_ASSERT_TRUE(localMaster.SetupLocalMaster(fncTestMasterBss, TestMatchInfo, sizeof(TestMatchInfo) / sizeof(TestMatchInfo[0])));

        // MASTER受信スレッド起動
        WLANTEST_ASSERT_TRUE(localMaster.LocalReceiveFrameStart(BufferSize200k, WlanTestRecvBufferSize, 1, 1, WlanTestAgingGetFrameCount));

        // CLIENT接続待機
        WLANTEST_ASSERT_TRUE(localMaster.LocalReceiveWait(WlanTestMasterWaitTimeOut));

        // 送信スレッド起動
        WLANTEST_ASSERT_TRUE(localMaster.LocalSendFrameStart(WlanTestRatePacketBufferSize, time, 0, true));

        sysTick = nn::os::GetSystemTick();
        logOutTime = sysTick;

        localMaster.LocalGetConnectInfo(&connectsInfo);
        while ((nn::os::GetSystemTick() - sysTick).ToTimeSpan().GetMilliSeconds() <= time)
        {
            isEventSygnal = nn::os::TimedWaitSystemEvent(&connectionEvent, nn::TimeSpan::FromMilliSeconds(Time5s));
            if (isEventSygnal == true)
            {
                connectCount = localMaster.GetClientMacAddress(localMaster.m_clientMacArray, nn::wlan::ConnectionState_Connected);
                WLANTEST_EXPECT_TRUE(connectCount != 0);
                if (connectCount == 0)
                {
                    break;
                }
                continue;
            }

            if ((nn::os::GetSystemTick() - logOutTime).ToTimeSpan().GetMilliSeconds() >= Time60s)
            {
                logOutTime = nn::os::GetSystemTick();
                localMaster.LocalGetConnectInfo(&connectsInfo);
                localMaster.LocalTraceConnectsInfo(true);
            }
        }

        // 送信スレッド停止
        localMaster.LocalSendFrameStop();

        // MASTER受信スレッド停止
        localMaster.LocalReceiveFrameStop();

        localMaster.LocalTraceConnectsInfo(true);
    }

    // MasterとClient1台接続で、1時間の連続通信を行い、途切れず100Mbps以上のスループットで通信できること CLIENT
    TEST_F(LocalClientFuncTest, L106_2)
    {
        nn::wlan::MacAddress clientMacAddress;
        nn::wlan::ConnectionStatus clientStatus;

        // 試験時間短縮フラグを確認
        int time = Time1h;
        if (g_IsShort) time = Time1800s;

        const TestLocalConnectParams ConnectParam = {
            nn::wlan::Ssid(LocalMasterTest2SSID), nn::wlan::MacAddress::CreateBroadcastMacAddress(), 6,
            { nn::wlan::SecurityMode_StaticAes, nn::wlan::SecurityMode_StaticAes, 0, LocalMasterAes16Key }, true, nn::wlan::BeaconIndication_Enable, 10
        };

        WLANTEST_ASSERT_TRUE(localClient.SetupLocalClient(ConnectParam.ssid, ConnectParam.security, TestMatchInfo, sizeof(TestMatchInfo) / sizeof(TestMatchInfo[0]), true, false));

        Connect(ConnectParam);

        WLANTEST_EXPECT_RESULT_SUCCESS(nn::wlan::Local::GetMacAddress(&clientMacAddress));
        WLANTEST_EXPECT_RESULT_SUCCESS(nn::wlan::Local::GetConnectionStatus(&connectionStatus));

        // 受信スレッド起動
        WLANTEST_ASSERT_TRUE(localClient.LocalReceiveFrameStart(BufferSize200k, WlanTestRecvBufferSize, 1, 1, WlanTestAgingGetFrameCount));
        WLANTEST_ASSERT_TRUE(localClient.LocalReceiveWait(WlanMaxClientWaitMsecTimeOut, Time1s, true));

        // 送信スレッド起動
        WLANTEST_ASSERT_TRUE(localClient.LocalSendFrameStart(WlanTestRatePacketBufferSize, time, 0, false));
        logOutTime = nn::os::GetSystemTick();
        sysTick = logOutTime;

        while (NN_STATIC_CONDITION(1))
        {
            // MASTERからの接続が切られるまで送信し続ける
            if (nn::os::TimedWaitSystemEvent(&connectionEvent, nn::TimeSpan::FromMilliSeconds(Time5s)) == true)
            {
                WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::GetConnectionStatus(&clientStatus));
                if (clientStatus.state != nn::wlan::ConnectionState_Connected)
                {
                    break;
                }
            }

            if ((nn::os::GetSystemTick() - logOutTime).ToTimeSpan().GetMilliSeconds() >= Time60s)
            {
                localClient.LocalTraceConnectsInfo(true);
                logOutTime = nn::os::GetSystemTick();
                localClient.LocalGetConnectInfo(&connectsInfo);
            }
        }

        // 送信スレッド停止
        localClient.LocalSendFrameStop();
        // 受信スレッド停止
        localClient.LocalReceiveFrameStop();

        localClient.LocalTraceConnectsInfo(true);

        Disconnect();
    }

    // MasterとClient7台接続で、16msec毎に512Bytesのブロードキャスト通信を12時間の連続通信を行い、PER10.0%以下でリンクダウンせず通信 MASTER
    TEST_F(LocalMasterFuncTest, L107_1)
    {
        int64_t agingTime = Time12h;

        // 通信評価定義
        const uint64_t broadcastByteNormal   = 60 * WlanTestPacketBufferSize * (agingTime / 1000);
        const uint64_t broadcastByteMin      = broadcastByteNormal - (broadcastByteNormal * 0.1);
        const uint64_t broadcastByteMax      = broadcastByteNormal + (broadcastByteNormal * 0.1);

        const nn::wlan::MasterBssParameters fncTestMasterBss = {
            11, false, 30, true, nn::wlan::RateSetLegacy_11gMask, nn::wlan::RateSetLegacy_11gMask,
            { nn::wlan::SecurityMode_StaticAes, nn::wlan::SecurityMode_StaticAes, 0, LocalMasterAes16Key }, nn::wlan::Ssid(LocalMasterTest2SSID), 10
        };

        WLANTEST_ASSERT_TRUE(localMaster.SetupLocalMaster(fncTestMasterBss, TestMatchInfo, sizeof(TestMatchInfo) / sizeof(TestMatchInfo[0])));

        // MASTER受信スレッド起動
        WLANTEST_ASSERT_TRUE(localMaster.LocalReceiveFrameStart(BufferSize100k, WlanTestPacketBufferSize, ClientMaxConnect, ClientMaxConnect, WlanTestAgingGetFrameCount));
        // CLIENT接続待機
        WLANTEST_ASSERT_TRUE(localMaster.LocalReceiveWait(WlanTestMasterWaitTimeOut));
        // 送信スレッド起動
        WLANTEST_ASSERT_TRUE(localMaster.LocalBroadcastSendFrameStart(WlanTestPacketBufferSize, agingTime, LocalApiBroadFrameInterval));

        logOutTime = nn::os::GetSystemTick();
        // 全CLIENTが切断されるまでループに変更
        while (localMaster.LocalMasterDisConnectWait(Time1s) > 0)
        {
            if ((nn::os::GetSystemTick() - logOutTime).ToTimeSpan().GetMilliSeconds() >= Time180s)
            {
                localMaster.LocalTraceConnectsInfo(true);
                logOutTime = nn::os::GetSystemTick();
            }
        }

        // MASTER送信スレッド停止
        localMaster.LocalSendFrameStop();
        // MASTER受信スレッド停止
        localMaster.LocalReceiveFrameStop();

        localMaster.LocalGetConnectInfo(&connectsInfo);
        WLANTEST_EXPECT_TRUE(connectsInfo.connectCount == ClientMaxConnect);

        for (int32_t i = 0; i < connectsInfo.connectCount; i++)
        {
            WLANTEST_EXPECT_TRUE(connectsInfo.facing[i].isConnected);
            WLANTEST_EXPECT_TRUE(connectsInfo.facing[i].perRate <= 10.00);
        }

        // ブロードキャスト送信サイズ範囲評価（1.max(最小, 送信サイズ)、2.min(1.結果, 最大）⇒ 範囲内なら送信サイズが戻り値。以外は最大／最小が戻り値）
        NN_LOG("             L107_1 Broad Send Total Size:%lld(Range:%lld - %lld)\n", connectsInfo.totalSendSize, broadcastByteMin, broadcastByteMax);
        WLANTEST_EXPECT_TRUE(std::min(std::max(broadcastByteMin, connectsInfo.totalSendSize), broadcastByteMax) == connectsInfo.totalSendSize);
    }

    // MasterとClient7台接続で、16msec毎に512Bytesのブロードキャスト通信を12時間の連続通信を行い、PER10.0%以下でリンクダウンせず通信 CLIENT
    TEST_F(LocalClientFuncTest, L107_2)
    {
        int64_t agingTime = Time12h;
        nn::wlan::MacAddress clientMacAddress;
        nn::wlan::ConnectionStatus clientStatus;
        nn::wlan::LinkLevel level;
        nn::Result result;

        // 通信評価定義
        const uint64_t broadcastByteNormal = 60 * WlanTestPacketBufferSize * (agingTime / 1000);
        const uint64_t broadcastByteMin = broadcastByteNormal - (broadcastByteNormal * 0.1);
        const uint64_t broadcastByteMax = broadcastByteNormal + (broadcastByteNormal * 0.1);

        const TestLocalConnectParams ConnectParam = {
            nn::wlan::Ssid(LocalMasterTest2SSID), nn::wlan::MacAddress::CreateBroadcastMacAddress(), 11,
            { nn::wlan::SecurityMode_StaticAes, nn::wlan::SecurityMode_StaticAes, 0, LocalMasterAes16Key }, true, nn::wlan::BeaconIndication_Enable, 10
        };

        WLANTEST_ASSERT_TRUE(localClient.SetupLocalClient(ConnectParam.ssid, ConnectParam.security, TestMatchInfo, sizeof(TestMatchInfo) / sizeof(TestMatchInfo[0]), true, false));

        Connect(ConnectParam);

        WLANTEST_EXPECT_RESULT_SUCCESS(nn::wlan::Local::GetMacAddress(&clientMacAddress));
        WLANTEST_EXPECT_RESULT_SUCCESS(nn::wlan::Local::GetConnectionStatus(&connectionStatus));

        // 受信スレッド起動
        WLANTEST_ASSERT_TRUE(localClient.LocalReceiveFrameStart(BufferSize100k, WlanTestPacketBufferSize, 1, 1, WlanTestAgingGetFrameCount));
        WLANTEST_ASSERT_TRUE(localClient.LocalReceiveWait(WlanMaxClientWaitMsecTimeOut, Time1s, true));
        // 送信スレッド起動
        WLANTEST_ASSERT_TRUE(localClient.LocalBroadcastSendFrameStart(WlanTestPacketBufferSize, agingTime, LocalApiBroadFrameInterval));

        sysTick = nn::os::GetSystemTick();
        logOutTime = nn::os::GetSystemTick();
        while ((nn::os::GetSystemTick() - sysTick).ToTimeSpan().GetMilliSeconds() <= agingTime)
        {
            // 一定周期送信。通信が切断されたらNG
            if (nn::os::TimedWaitSystemEvent(&connectionEvent, nn::TimeSpan::FromMilliSeconds(LocalApiBroadFrameInterval)) == true)
            {
                WLANTEST_EXPECT_RESULT_SUCCESS(nn::wlan::Local::GetConnectionStatus(&clientStatus));
                WLANTEST_EXPECT_TRUE(clientStatus.state == nn::wlan::ConnectionState_Connected);
                if (clientStatus.state != nn::wlan::ConnectionState_Connected)
                {
                    localClient.LocalTraceConnectsInfo(true);
                    break;
                }
            }

            if ((nn::os::GetSystemTick() - logOutTime).ToTimeSpan().GetMilliSeconds() >= Time180s)
            {
                localClient.LocalTraceConnectsInfo(true);
                logOutTime = nn::os::GetSystemTick();
            }

            result = nn::wlan::Local::GetLinkLevel(&level);
            if (result.IsSuccess() != true)
            {
                WLANTEST_EXPECT_RESULT_SUCCESS(nn::wlan::Local::GetConnectionStatus(&clientStatus));
                if (clientStatus.state == nn::wlan::ConnectionState_Connected)
                {
                    WLANTEST_EXPECT_TRUE(nn::wlan::LinkLevel_0 != level);
                }
            }
        }

        // 送信スレッド停止
        localClient.LocalSendFrameStop();
        // 受信スレッド停止
        localClient.LocalReceiveFrameStop();

        localClient.LocalGetConnectInfo(&connectsInfo);
        WLANTEST_EXPECT_TRUE(connectsInfo.connectCount == ClientMaxConnect);

        for (int32_t i = 0; i < connectsInfo.connectCount; i++)
        {
            WLANTEST_EXPECT_TRUE(connectsInfo.facing[i].isConnected);
            WLANTEST_EXPECT_TRUE(connectsInfo.facing[i].perRate <= 10.00);
        }

        Disconnect();

        // ブロードキャスト送信サイズ範囲評価（1.max(最小, 送信サイズ)、2.min(1.結果, 最大）⇒ 範囲内なら送信サイズが戻り値。以外は最大／最小が戻り値）
        NN_LOG("             L107_2 Broad Send Total Size:%lld(Range:%lld - %lld)\n", connectsInfo.totalSendSize, broadcastByteMin, broadcastByteMax);
        WLANTEST_EXPECT_TRUE(std::min(std::max(broadcastByteMin, connectsInfo.totalSendSize), broadcastByteMax) == connectsInfo.totalSendSize);
    }

    // ClientがMasterと接続後、16msec毎に512Bytesの通信を行い、1秒間隔で1,6,11chの既接続スキャンを12時間連続通信 MASTER
    TEST_F(LocalMasterFuncTest, L108_1)
    {
        int64_t agingTime = Time12h;
        nn::wlan::MacAddress myMacAddress;

        // 通信評価定義
        const int64_t sendInterval = static_cast<int64_t>(Time1s / 60);
        const uint64_t frameByteNormal = 60 * WlanTestPacketBufferSize * (agingTime / 1000);
        const uint64_t frameByteMin = frameByteNormal - (frameByteNormal * 0.1);
        const uint64_t frameByteMax = frameByteNormal + (frameByteNormal * 0.1);

        const nn::wlan::MasterBssParameters fncTestMasterBss = {
            11, false, 30, true, nn::wlan::RateSetLegacy_11gMask, nn::wlan::RateSetLegacy_11gMask,
            { nn::wlan::SecurityMode_StaticAes, nn::wlan::SecurityMode_StaticAes, 0, LocalMasterAes16Key }, nn::wlan::Ssid(LocalMasterTestSSID), 100
        };

        // ジョイコン1セット（各2台）
        WLANTEST_ASSERT_TRUE(localMaster.SetBtMode(LocalApiBtNode2));

        WLANTEST_ASSERT_TRUE(localMaster.SetupLocalMaster(fncTestMasterBss, TestMatchInfo, sizeof(TestMatchInfo) / sizeof(TestMatchInfo[0])));

        // MASTER受信スレッド起動
        WLANTEST_ASSERT_TRUE(localMaster.LocalReceiveFrameStart(BufferSize100k, WlanTestPacketBufferSize, 1, 1, WlanTestAgingGetFrameCount));
        // CLIENT接続待機
        WLANTEST_ASSERT_TRUE(localMaster.LocalReceiveWait(WlanTestMasterWaitTimeOut));
        // MASTER送信スレッド起動
        WLANTEST_ASSERT_TRUE(localMaster.LocalSendFrameOneShotWaitStart(WlanTestPacketBufferSize, agingTime, sendInterval, true));

        logOutTime = nn::os::GetSystemTick();
        // 全CLIENTが切断されるまでループに変更
        while (localMaster.LocalMasterDisConnectWait(Time1s) > 0)
        {
            if ((nn::os::GetSystemTick() - logOutTime).ToTimeSpan().GetMilliSeconds() >= Time180s)
            {
                localMaster.LocalTraceConnectsInfo(true);
                logOutTime = nn::os::GetSystemTick();
            }
        }

        // MASTER送信スレッド停止
        localMaster.LocalSendFrameStop();
        // MASTER受信スレッド停止
        localMaster.LocalReceiveFrameStop();

        localMaster.LocalGetConnectInfo(&connectsInfo);

        // フレーム送信サイズ範囲評価（1.max(最小, 送信サイズ)、2.min(1.結果, 最大）⇒ 範囲内なら送信サイズが戻り値。以外は最大／最小が戻り値）
        NN_LOG("             L108_1 Frame Send Total Size:%lld(Range:%lld - %lld)\n", connectsInfo.totalSendSize, frameByteMin, frameByteMax);
        WLANTEST_EXPECT_TRUE(std::min(std::max(frameByteMin, connectsInfo.totalSendSize), frameByteMax) == connectsInfo.totalSendSize);
    }

    // ClientがMasterと接続後、16msec毎に512Bytesの通信を行い、1秒間隔で1,6,11chの既接続スキャンを12時間連続通信 CLIENT
    TEST_F(LocalClientFuncTest, L108_2)
    {
        int64_t agingTime = Time12h;
        nn::wlan::MacAddress clientMacAddress;
        nn::wlan::ConnectionStatus clientStatus;

        // 通信評価定義
        const uint64_t frameByteNormal = 60 * WlanTestPacketBufferSize * (agingTime / 1000);
        const uint64_t frameByteMin = frameByteNormal - (frameByteNormal * 0.1);
        const uint64_t frameByteMax = frameByteNormal + (frameByteNormal * 0.1);

        const nn::wlan::ScanParameters StartScanTestParams = {
            nn::wlan::ScanType_Active,{ 1, 6, 11 }, 3, 40, -1, nullptr, 0, nn::wlan::MacAddress::CreateBroadcastMacAddress()
        };
        const TestLocalConnectParams ConnectParam = {
            nn::wlan::Ssid(LocalMasterTestSSID), nn::wlan::MacAddress::CreateBroadcastMacAddress(), 11,
            { nn::wlan::SecurityMode_StaticAes, nn::wlan::SecurityMode_StaticAes, 0, LocalMasterAes16Key }, true, nn::wlan::BeaconIndication_Enable, 10
        };

        // ジョイコン1セット（各2台）
        WLANTEST_ASSERT_TRUE(localClient.SetBtMode(LocalApiBtNode2));

        WLANTEST_ASSERT_TRUE(localClient.SetupLocalClient(ConnectParam.ssid, ConnectParam.security, TestMatchInfo, sizeof(TestMatchInfo) / sizeof(TestMatchInfo[0]), true, false));

        localClient.LocalTraceConnectsInfo(true);
        Connect(ConnectParam);
        localClient.LocalTraceConnectsInfo(true);

        WLANTEST_EXPECT_RESULT_SUCCESS(nn::wlan::Local::GetMacAddress(&clientMacAddress));
        WLANTEST_EXPECT_RESULT_SUCCESS(nn::wlan::Local::GetConnectionStatus(&connectionStatus));

        // 受信スレッド起動
        WLANTEST_ASSERT_TRUE(localClient.LocalReceiveFrameStart(BufferSize100k, WlanTestPacketBufferSize, 1, 1, WlanTestAgingGetFrameCount));
        WLANTEST_ASSERT_TRUE(localClient.LocalReceiveWait(WlanMaxClientWaitMsecTimeOut, Time1s, true));

        StartUnsyncScanThread(StartScanTestParams, 1000);

        // 送信スレッド起動
        WLANTEST_ASSERT_TRUE(localClient.LocalSendFrameOneShotWaitStart(WlanTestPacketBufferSize, agingTime, Interval_16m, true));

        sysTick = nn::os::GetSystemTick();
        logOutTime = sysTick;
        while ((nn::os::GetSystemTick() - sysTick).ToTimeSpan().GetMilliSeconds() <= agingTime)
        {
            // 一定周期送信。通信が切断されたらNG
            if (nn::os::TimedWaitSystemEvent(&connectionEvent, nn::TimeSpan::FromMilliSeconds(Interval_16m)) == true)
            {
                WLANTEST_EXPECT_RESULT_SUCCESS(nn::wlan::Local::GetConnectionStatus(&clientStatus));
                WLANTEST_EXPECT_TRUE(clientStatus.state == nn::wlan::ConnectionState_Connected);
                if (clientStatus.state != nn::wlan::ConnectionState_Connected)
                {
                    localClient.LocalTraceConnectsInfo(true);
                    break;
                }
            }

            if ((nn::os::GetSystemTick() - logOutTime).ToTimeSpan().GetMilliSeconds() >= Time180s)
            {
                localClient.LocalTraceConnectsInfo(true);
                logOutTime = nn::os::GetSystemTick();
            }
        }

        // 送信スレッド停止
        localClient.LocalSendFrameStop();
        // 受信スレッド停止
        localClient.LocalReceiveFrameStop();
        StopUnsyncScanThread();
        localClient.LocalGetConnectInfo(&connectsInfo);

        Disconnect();

        // フレーム送信サイズ範囲評価（1.max(最小, 送信サイズ)、2.min(1.結果, 最大）⇒ 範囲内なら送信サイズが戻り値。以外は最大／最小が戻り値）
        NN_LOG("             L108_2 Frame Send Total Size:%lld(Range:%lld - %lld)\n", connectsInfo.totalSendSize, frameByteMin, frameByteMax);
        WLANTEST_EXPECT_TRUE(std::min(std::max(frameByteMin, connectsInfo.totalSendSize), frameByteMax) == connectsInfo.totalSendSize);
    }

    // 8台の端末が、Masterを順次変えながら4時間通信 Master
    TEST_F(LocalMasterClientFuncTest, L115_1)
    {
        const int64_t agingTime = Time4h;
        nn::os::Tick elapseTime;
        bool isStart = false;

        MasterSetup(ClientMaxConnect);

        myNumber = GetSeqNumber();
        WLANTEST_ASSERT_TRUE(myNumber != 0);

        elapseTime = nn::os::GetSystemTick();
        GetFirstSeqNumber();
        while ((nn::os::GetSystemTick() - elapseTime).ToTimeSpan().GetMilliSeconds() <= agingTime)
        {
            if (isMasterSeqNumber() == true)
            {
                MasterFrameTest(isStart, elapseTime, WlanTestPacketBufferSize, agingTime);
            }
            else
            {
                ClientFrameTest(isStart, elapseTime, WlanTestPacketBufferSize, agingTime);
            }
            GetNextSeqNumber();
        }

        WLANTEST_ASSERT_TRUE(ChangeMode(OpenModeType::OpenMode_Master));
    }

    // 8台の端末が、Masterを順次変えながら4時間通信 Client
    TEST_F(LocalMasterClientFuncTest, L115_2)
    {
        const int64_t agingTime = Time4h;
        nn::os::Tick elapseTime;
        bool isStart = false;

        ClientSetup();

        myNumber = GetSeqNumber();
        WLANTEST_ASSERT_TRUE(myNumber != 0);

        elapseTime = nn::os::GetSystemTick();
        GetFirstSeqNumber();
        while ((nn::os::GetSystemTick() - elapseTime).ToTimeSpan().GetMilliSeconds() <= agingTime)
        {
            if (isMasterSeqNumber() == true)
            {
                MasterFrameTest(isStart, elapseTime, WlanTestPacketBufferSize, agingTime);
            }
            else
            {
                ClientFrameTest(isStart, elapseTime, WlanTestPacketBufferSize, agingTime);
            }
            GetNextSeqNumber();
        }

        WLANTEST_ASSERT_TRUE(ChangeMode(OpenModeType::OpenMode_Master));
    }

    // Masterが7台のClientと接続後、16msec毎に512Bytesの通信を行い、1秒間隔で1,6,11chの既接続スキャンを8時間連続通信 MASTER
    // Masterが7台のClientと接続後、16msec毎に512Bytesの通信を行い、1秒間隔で1,6,11chの既接続スキャンを8時間連続通信とClientのリンク維持 MASTER
    TEST_F(LocalMasterFuncTest, L109_L110_1)
    {
        int64_t agingTime = Time8h;
        // 通信評価定義
        const uint64_t frameByteNormal = 60 * WlanTestPacketBufferSize * (agingTime / 1000);
        const uint64_t frameByteMin = frameByteNormal - (frameByteNormal * 0.1);
        const uint64_t frameByteMax = frameByteNormal + (frameByteNormal * 0.1);

        const nn::wlan::MasterBssParameters fncTestMasterBss = {
            11, false, 30, true, nn::wlan::RateSetLegacy_11gMask, nn::wlan::RateSetLegacy_11gMask,
            { nn::wlan::SecurityMode_StaticAes, nn::wlan::SecurityMode_StaticAes, 0, LocalMasterAes16Key }, nn::wlan::Ssid(LocalMasterTest3SSID), 100
        };

        WLANTEST_ASSERT_TRUE(localMaster.SetupLocalMaster(fncTestMasterBss, TestMatchInfo, sizeof(TestMatchInfo) / sizeof(TestMatchInfo[0])));
        // MASTER受信スレッド起動
        WLANTEST_ASSERT_TRUE(localMaster.LocalReceiveFrameStart(BufferSize100k, WlanTestPacketBufferSize, ClientMaxConnect, ClientMaxConnect, WlanTestAgingGetFrameCount));
        // CLIENT接続待機
        WLANTEST_ASSERT_TRUE(localMaster.LocalReceiveWait(WlanTestMasterWaitTimeOut));
        // MASTER送信スレッド起動
        WLANTEST_ASSERT_TRUE(localMaster.LocalSendFrameOneShotWaitStart(WlanTestPacketBufferSize, agingTime, Interval_16m, false));

        logOutTime = nn::os::GetSystemTick();
        // 全CLIENTが切断されるまでループに変更
        while (localMaster.LocalMasterDisConnectWait(Time1s) > 0)
        {
            if ((nn::os::GetSystemTick() - logOutTime).ToTimeSpan().GetMilliSeconds() >= Time180s)
            {
                localMaster.LocalTraceConnectsInfo(true);
                logOutTime = nn::os::GetSystemTick();
            }
        }

        // MASTER送信スレッド停止
        localMaster.LocalSendFrameStop();
        // MASTER受信スレッド停止
        localMaster.LocalReceiveFrameStop();

        localMaster.LocalGetConnectInfo(&connectsInfo);
        WLANTEST_EXPECT_TRUE(connectsInfo.connectCount >= ClientMaxConnect);

        localMaster.LocalGetConnectInfo(&connectsInfo);

        // フレーム送信サイズ範囲評価（1.max(最小, 送信サイズ)、2.min(1.結果, 最大）⇒ 範囲内なら送信サイズが戻り値。以外は最大／最小が戻り値）
        NN_LOG("             L109_L110_1 Frame Send Total Size:%lld(Range:%lld - %lld)\n", connectsInfo.totalSendSize, frameByteMin, frameByteMax);
        WLANTEST_EXPECT_TRUE(std::min(std::max(frameByteMin, connectsInfo.totalSendSize), frameByteMax) == connectsInfo.totalSendSize);
    }

    // Masterが7台のClientと接続後、16msec毎に512Bytesの通信を行い、1秒間隔で1,6,11chの既接続スキャンを8時間連続通信 CLIENT
    // Masterが7台のClientと接続後、16msec毎に512Bytesの通信を行い、1秒間隔で1,6,11chの既接続スキャンを8時間連続通信とClientのリンク維持 CLIENT
    TEST_F(LocalClientFuncTest, L109_L110_2)
    {
        int64_t agingTime = Time8h;
        nn::wlan::ConnectionStatus clientStatus;
        nn::wlan::LinkLevel level;
        nn::Result result;

        // 通信評価定義
        const uint64_t frameByteNormal = 60 * WlanTestPacketBufferSize * (agingTime / 1000);
        const uint64_t frameByteMin = frameByteNormal - (frameByteNormal * 0.1);
        const uint64_t frameByteMax = frameByteNormal + (frameByteNormal * 0.1);

        const nn::wlan::ScanParameters StartScanTestParams = {
            nn::wlan::ScanType_Active,{ 1, 6, 11 }, 3, 40, -1, nullptr, 0, nn::wlan::MacAddress::CreateBroadcastMacAddress()
        };
        const TestLocalConnectParams ConnectParam = {
            nn::wlan::Ssid(LocalMasterTest3SSID), nn::wlan::MacAddress::CreateBroadcastMacAddress(), 11,
            { nn::wlan::SecurityMode_StaticAes, nn::wlan::SecurityMode_StaticAes, 0, LocalMasterAes16Key }, true, nn::wlan::BeaconIndication_Enable, 10
        };

#ifdef WLAN_TEST_LINKLEVEL_ERROR
        static NN_ALIGNAS(4096) uint8_t testApiCallStack[threadStackSize];
        connectLogApiStart = connectStsApiStart = linkApiStart = nn::os::GetSystemTick();
        isConnectStsCheck = isConnectLogCheck = isLinkApiCheck = false;

        isCheck = true;
        WLANTEST_ASSERT_RESULT_SUCCESS(nn::os::CreateThread(&testCheckThread, TestApiCallTimeThread, this, testApiCallStack, threadStackSize, nn::os::DefaultThreadPriority));
        nn::os::StartThread(&testCheckThread);
#endif

        WLANTEST_ASSERT_TRUE(localClient.SetupLocalClient(ConnectParam.ssid, ConnectParam.security, TestMatchInfo, sizeof(TestMatchInfo) / sizeof(TestMatchInfo[0]), true, false));

        Connect(ConnectParam);
        WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::GetConnectionStatus(&connectionStatus));

        // 受信スレッド起動
        WLANTEST_ASSERT_TRUE(localClient.LocalReceiveFrameStart(BufferSize100k, WlanTestPacketBufferSize, 1, 1, WlanTestAgingGetFrameCount));
        WLANTEST_ASSERT_TRUE(localClient.LocalReceiveWait(WlanMaxClientWaitMsecTimeOut, Time1s, true));
        // 送信スレッド起動
        WLANTEST_ASSERT_TRUE(localClient.LocalSendFrameOneShotWaitStart(WlanTestPacketBufferSize, agingTime, Interval_16m, true));
        StartUnsyncScanThread(StartScanTestParams, 1000);

        sysTick = nn::os::GetSystemTick();
        logOutTime = sysTick;
        while ((nn::os::GetSystemTick() - sysTick).ToTimeSpan().GetMilliSeconds() <= agingTime)
        {
            // 一定周期送信。通信が切断されたらNG
            if (nn::os::TimedWaitSystemEvent(&connectionEvent, nn::TimeSpan::FromMilliSeconds(Interval_16m)) == true)
            {
                WLANTEST_EXPECT_RESULT_SUCCESS(nn::wlan::Local::GetConnectionStatus(&clientStatus));
                WLANTEST_EXPECT_TRUE(clientStatus.state == nn::wlan::ConnectionState_Connected);
                if (clientStatus.state != nn::wlan::ConnectionState_Connected)
                {
                    localClient.LocalTraceConnectsInfo(true);
                    break;
                }
            }

            if ((nn::os::GetSystemTick() - logOutTime).ToTimeSpan().GetMilliSeconds() >= Time600s)
            {
#ifdef WLAN_TEST_LINKLEVEL_ERROR
                isConnectStsCheck = isLinkApiCheck = false;
                isConnectLogCheck = true;
                connectLogApiStart = nn::os::GetSystemTick();
#endif
                localClient.LocalTraceConnectsInfo(true);
                logOutTime = nn::os::GetSystemTick();
            }

#ifdef WLAN_TEST_LINKLEVEL_ERROR
            isConnectLogCheck = isConnectStsCheck = false;
            isLinkApiCheck = true;
            linkApiStart = nn::os::GetSystemTick();
#endif
#ifdef WLAN_TEST_LINKLEVEL_WAIT
            result = nn::wlan::Local::GetLinkLevel(&level);
            if (result.IsSuccess() != true)
            {

#ifdef WLAN_TEST_LINKLEVEL_ERROR
                isConnectLogCheck = isLinkApiCheck = false;
                isConnectStsCheck = true;
                connectStsApiStart = nn::os::GetSystemTick();
#endif
                WLANTEST_EXPECT_RESULT_SUCCESS(nn::wlan::Local::GetConnectionStatus(&clientStatus));
                if (clientStatus.state == nn::wlan::ConnectionState_Connected)
                {
                    WLANTEST_EXPECT_TRUE(nn::wlan::LinkLevel_0 != level);
                }
            }
#endif
        }
#ifdef WLAN_TEST_LINKLEVEL_ERROR
        isCheck = false;
        nn::os::WaitThread(&testCheckThread);
        nn::os::DestroyThread(&testCheckThread);
#endif
        localClient.LocalTraceConnectsInfo(true);

        localClient.LocalGetConnectInfo(&connectsInfo);
        for (int32_t i = 0; i < connectsInfo.connectCount; i++)
        {
            WLANTEST_EXPECT_TRUE(connectsInfo.facing[i].isConnected);
            WLANTEST_EXPECT_TRUE(connectsInfo.facing[i].perRate <= 10.00);
        }

        // 送信スレッド停止
        localClient.LocalSendFrameStop();
        // 受信スレッド停止
        localClient.LocalReceiveFrameStop();

        StopUnsyncScanThread();
        // 1秒待機
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(Time1s));

        Disconnect();

        // フレーム送信サイズ範囲評価（1.max(最小, 送信サイズ)、2.min(1.結果, 最大）⇒ 範囲内なら送信サイズが戻り値。以外は最大／最小が戻り値）
        NN_LOG("             L109_L110_2 Frame Send Total Size:%lld(Range:%lld - %lld)\n", connectsInfo.totalSendSize, frameByteMin, frameByteMax);
        WLANTEST_EXPECT_TRUE(std::min(std::max(frameByteMin, connectsInfo.totalSendSize), frameByteMax) == connectsInfo.totalSendSize);
    }

    // 6ch, 11chの無線負荷が高い場合、1chが選択される MASTER 6CH
    TEST_F(LocalMasterFuncTest, L114_1)
    {
        // ジョイコン2セット（各4台）
        WLANTEST_ASSERT_TRUE(localMaster.SetBtMode(LocalApiBtNode4));

        const nn::wlan::MasterBssParameters fncTestMasterBss = {
            6, false, 15, true, nn::wlan::RateSetLegacy_11gMask, nn::wlan::RateSetLegacy_11gMask,
            { nn::wlan::SecurityMode_Open, nn::wlan::SecurityMode_Open, 0, "" }, nn::wlan::Ssid(LocalMasterTest2SSID), 100
        };

        WLANTEST_ASSERT_TRUE(localMaster.SetupLocalMaster(fncTestMasterBss, TestMatchInfo, sizeof(TestMatchInfo) / sizeof(TestMatchInfo[0])));

        StartTcpServer();

        // MASTER受信スレッド起動
        WLANTEST_ASSERT_TRUE(localMaster.LocalReceiveFrameStart(BufferSize100k, WlanTestPacketBufferSize, 1, 1, WlanTestAgingGetFrameCount));
        WLANTEST_ASSERT_TRUE(localMaster.LocalReceiveWait(WlanTestMasterWaitTimeOut, true));

        // MASTER送信スレッド起動
        WLANTEST_ASSERT_TRUE(localMaster.LocalSendFrameStart(WlanTestPacketBufferSize, Time60s, 0, false));

        // MASTER切断待機
        WLANTEST_ASSERT_TRUE(localMaster.LocalMasterDisConnectWait(Time600s) == 0);

        // MASTER送信スレッド停止
        localMaster.LocalSendFrameStop();

        StopTcpServer();

        localMaster.LocalReceiveFrameStop();
    }

    // 6ch, 11chの無線負荷が高い場合、1chが選択される MASTER 11CH
    TEST_F(LocalMasterFuncTest, L114_2)
    {
        // ジョイコン2セット（各4台）
        WLANTEST_ASSERT_TRUE(localMaster.SetBtMode(LocalApiBtNode4));

        const nn::wlan::MasterBssParameters fncTestMasterBss = {
            11, false, 15, true, nn::wlan::RateSetLegacy_11gMask, nn::wlan::RateSetLegacy_11gMask,
            { nn::wlan::SecurityMode_Open, nn::wlan::SecurityMode_Open, 0, "" }, nn::wlan::Ssid(LocalMasterTest3SSID), 100
        };

        WLANTEST_ASSERT_TRUE(localMaster.SetupLocalMaster(fncTestMasterBss, TestMatchInfo, sizeof(TestMatchInfo) / sizeof(TestMatchInfo[0])));

        StartTcpServer();

        // MASTER受信スレッド起動
        WLANTEST_ASSERT_TRUE(localMaster.LocalReceiveFrameStart(BufferSize100k, WlanTestPacketBufferSize, 1, 1, WlanTestAgingGetFrameCount));
        WLANTEST_ASSERT_TRUE(localMaster.LocalReceiveWait(WlanTestMasterWaitTimeOut, true));

        // MASTER送信スレッド起動
        WLANTEST_ASSERT_TRUE(localMaster.LocalSendFrameStart(WlanTestPacketBufferSize, Time60s, 0, false));

        // MASTER切断待機
        WLANTEST_ASSERT_TRUE(localMaster.LocalMasterDisConnectWait(Time600s) == 0);

        // MASTER送信スレッド停止
        localMaster.LocalSendFrameStop();

        StopTcpServer();

        localMaster.LocalReceiveFrameStop();
    }

    // 6ch, 11chの無線負荷が高い場合、1chが選択される CLIENT 6CH
    TEST_F(LocalClientFuncTest, L114_3)
    {
        // ジョイコン2セット（各4台）
        WLANTEST_ASSERT_TRUE(localClient.SetBtMode(LocalApiBtNode4));

        const TestLocalConnectParams ConnectParam = {
            nn::wlan::Ssid(LocalMasterTest2SSID),  nn::wlan::MacAddress::CreateBroadcastMacAddress(), 6,
            { nn::wlan::SecurityMode_Open, nn::wlan::SecurityMode_Open, 0, "" }, true, nn::wlan::BeaconIndication_Enable, 15
        };

        WLANTEST_ASSERT_TRUE(localClient.SetupLocalClient(ConnectParam.ssid, ConnectParam.security, TestMatchInfo, sizeof(TestMatchInfo) / sizeof(TestMatchInfo[0]), true, false));

        Connect(ConnectParam);

        StartTcpClient(STATIC_CLIENT1_IP_ADDR);

        // 受信スレッド起動
        WLANTEST_ASSERT_TRUE(localClient.LocalReceiveFrameStart(BufferSize100k, WlanTestPacketBufferSize, 1, 1, WlanTestAgingGetFrameCount));
        WLANTEST_ASSERT_TRUE(localClient.LocalReceiveWait(WlanMaxClientWaitMsecTimeOut, Time1s, true));

        // 送信スレッド起動
        WLANTEST_ASSERT_TRUE(localClient.LocalSendFrameStart(WlanTestPacketBufferSize, Time60s, 0, false));

        sysTick = nn::os::GetSystemTick();
        while ((nn::os::GetSystemTick() - sysTick).ToTimeSpan().GetMilliSeconds() <= Time60s)
        {
            // MASTERからの接続が切られるまで送信し続ける
            if (nn::os::TimedWaitSystemEvent(&connectionEvent, nn::TimeSpan::FromMilliSeconds(Interval_100m)) == true)
            {
                WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::GetConnectionStatus(&connectionStatus));
                if (connectionStatus.state != nn::wlan::ConnectionState_Connected)
                {
                    break;
                }
            }
        }

        // 送信スレッド停止
        localClient.LocalSendFrameStop();
        localClient.LocalReceiveFrameStop();
        StopTcpClient();

        sysTick = nn::os::GetSystemTick();
        NN_LOG("             L114 Client1 Disconnect Start : %lld\n", (nn::os::GetSystemTick() - sysTick).ToTimeSpan().GetMilliSeconds());
        Disconnect();
        NN_LOG("             L114 Client1 Disconnect End : %lld\n", (nn::os::GetSystemTick() - sysTick).ToTimeSpan().GetMilliSeconds());
    }

    // 6ch, 11chの無線負荷が高い場合、1chが選択される CLIENT 11CH
    TEST_F(LocalClientFuncTest, L114_4)
    {
        // ジョイコン2セット（各4台）
        WLANTEST_ASSERT_TRUE(localClient.SetBtMode(LocalApiBtNode4));

        const TestLocalConnectParams ConnectParam = {
            nn::wlan::Ssid(LocalMasterTest3SSID),  nn::wlan::MacAddress::CreateBroadcastMacAddress(), 11,
            { nn::wlan::SecurityMode_Open, nn::wlan::SecurityMode_Open, 0, "" }, true, nn::wlan::BeaconIndication_Enable, 15
        };

        WLANTEST_ASSERT_TRUE(localClient.SetupLocalClient(ConnectParam.ssid, ConnectParam.security, TestMatchInfo, sizeof(TestMatchInfo) / sizeof(TestMatchInfo[0]), true, false));

        Connect(ConnectParam);

        StartTcpClient(STATIC_CLIENT2_IP_ADDR);

        // 受信スレッド起動
        WLANTEST_ASSERT_TRUE(localClient.LocalReceiveFrameStart(BufferSize100k, WlanTestPacketBufferSize, 1, 1, WlanTestAgingGetFrameCount));
        WLANTEST_ASSERT_TRUE(localClient.LocalReceiveWait(WlanMaxClientWaitMsecTimeOut, Time1s, true));

        // 送信スレッド起動
        WLANTEST_ASSERT_TRUE(localClient.LocalSendFrameStart(WlanTestPacketBufferSize, Time60s, 0, false));

        sysTick = nn::os::GetSystemTick();
        while ((nn::os::GetSystemTick() - sysTick).ToTimeSpan().GetMilliSeconds() <= Time60s)
        {
            // MASTERからの接続が切られるまで送信し続ける
            if (nn::os::TimedWaitSystemEvent(&connectionEvent, nn::TimeSpan::FromMilliSeconds(Interval_100m)) == true)
            {
                WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::GetConnectionStatus(&connectionStatus));
                if (connectionStatus.state != nn::wlan::ConnectionState_Connected)
                {
                    break;
                }
            }
        }

        // 送信スレッド停止
        localClient.LocalSendFrameStop();

        localClient.LocalReceiveFrameStop();
        StopTcpClient();

        sysTick = nn::os::GetSystemTick();
        NN_LOG("             L114 Client2 Disconnect Start : %lld\n", (nn::os::GetSystemTick() - sysTick).ToTimeSpan().GetMilliSeconds());
        Disconnect();
        NN_LOG("             L114 Client2 Disconnect End : %lld\n", (nn::os::GetSystemTick() - sysTick).ToTimeSpan().GetMilliSeconds());
    }

    // 6ch, 11chの無線負荷が高い場合、1chが選択される MASTER 自動
    TEST_F(LocalMasterFuncTest, L114_5)
    {
        // ジョイコン2セット（各4台）
        WLANTEST_ASSERT_TRUE(localMaster.SetBtMode(LocalApiBtNode4));

        nn::wlan::ConnectionStatus connectStatus;
        const nn::wlan::MasterBssParameters fncTestMasterBss = {
            -1, false, 15, true, nn::wlan::RateSetLegacy_11gMask, nn::wlan::RateSetLegacy_11gMask,
            { nn::wlan::SecurityMode_Open, nn::wlan::SecurityMode_Open, 0, "" }, nn::wlan::Ssid(LocalMasterTestSSID), 100
        };

        WLANTEST_ASSERT_TRUE(localMaster.SetupLocalMaster(fncTestMasterBss, TestMatchInfo, sizeof(TestMatchInfo) / sizeof(TestMatchInfo[0])));

        WLANTEST_EXPECT_RESULT_SUCCESS(nn::wlan::Local::GetConnectionStatus(&connectStatus));
        WLANTEST_ASSERT_TRUE(connectStatus.channel == 1);

    }

    TEST_F(LocalMasterFuncTest, G3_L93)
    {
        WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::CloseMasterMode());
        nn::wlan::InitializeInfraManager();
        LocalOpenModeType mode;
        uint32_t masterCnt  = 0;
        uint32_t clientCnt  = 0;
        uint32_t infraCnt   = 0;

        // ジョイコン1セット（各2台）
        WLANTEST_ASSERT_TRUE(localMaster.SetBtMode(LocalApiBtNode2));

        sysTick = nn::os::GetSystemTick();
        // 60秒間通信
        while ((nn::os::GetSystemTick() - sysTick).ToTimeSpan().GetMilliSeconds() <= Time1day)
        {
            mode = static_cast<LocalOpenModeType>(
                GetRandom(LocalOpenModeType::OpenMode_Init, LocalOpenModeType::OpenMode_Client));

            switch (mode)
            {
            case LocalOpenModeType::OpenMode_Master:
                masterCnt++;
                WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::OpenMasterMode());
                WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::CloseMasterMode());
                break;
            case LocalOpenModeType::OpenMode_Client:
                clientCnt++;
                WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::OpenClientMode());
                WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::CloseClientMode());
                break;
            default:
                infraCnt++;
                WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Infra::OpenMode());
                WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Infra::CloseMode());
                break;
            }

            if ((nn::os::GetSystemTick() - logTick).ToTimeSpan().GetMilliSeconds() > Time30s ||
                (nn::os::GetSystemTick() - sysTick).ToTimeSpan().GetMilliSeconds() > Time1day)
            {
                int64_t elapsedTime = (nn::os::GetSystemTick() - sysTick).ToTimeSpan().GetMilliSeconds();
                logTick = nn::os::GetSystemTick();

                uint32_t msecTime = elapsedTime % 1000;
                uint32_t secTime = (elapsedTime / 1000) % 60;
                uint32_t minuteTime = (elapsedTime / 1000) % (60 * 60) / 60;
                uint32_t hourTime = (elapsedTime / 1000) / (60 * 60);


                NN_LOG("\n             *** G3 L93 Mode Open Close Test ***\n");
                NN_LOG("               Elapsed Time             : %lu:%02lu:%02lu.%03lu\n", hourTime, minuteTime, secTime, msecTime);
                NN_LOG("               Master Open/Close Count  : %d\n", masterCnt);
                NN_LOG("               Client Open/Close Count  : %d\n", clientCnt);
                NN_LOG("               Infra Open/Close Count   : %d\n", infraCnt);
                NN_LOG("             *** G3 L93 Mode Open Close Test ***\n");
            }
        }

        localMaster.LocalRelease();

        nn::wlan::FinalizeInfraManager();
        WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::OpenMasterMode());
    }


    // Masterでキープアライブの応答が無い場合、端末が切断扱い MASTER
    TEST_F(LocalMasterFuncTest, L116_1)
    {
        const nn::wlan::MasterBssParameters fncTestMasterBss = {
            6, false, 10, true, nn::wlan::RateSetLegacy_11gMask, nn::wlan::RateSetLegacy_11gMask,
            { nn::wlan::SecurityMode_StaticAes, nn::wlan::SecurityMode_StaticAes, 0, LocalMasterAes16Key }, nn::wlan::Ssid(LocalMasterTestSSID), 100
        };

        WLANTEST_ASSERT_TRUE(localMaster.SetupLocalMaster(fncTestMasterBss, TestMatchInfo, sizeof(TestMatchInfo) / sizeof(TestMatchInfo[0])));

        sysTick = nn::os::GetSystemTick();
        NN_LOG("               %s ---> L116 Connect Wait Start(%lld)\n", __FUNCTION__, (nn::os::GetSystemTick() - sysTick).ToTimeSpan().GetMilliSeconds());
        nn::os::TimedWaitSystemEvent(&connectionEvent, nn::TimeSpan::FromSeconds(WlanTestConnectTimeOut));
        NN_LOG("               %s ---> L116 Connect Wait End(%lld)\n", __FUNCTION__, (nn::os::GetSystemTick() - sysTick).ToTimeSpan().GetMilliSeconds());

        NN_LOG("               %s ---> L116 Disconnect Wait Start(%lld)\n", __FUNCTION__, (nn::os::GetSystemTick() - sysTick).ToTimeSpan().GetMilliSeconds());
        nn::os::TimedWaitSystemEvent(&connectionEvent, nn::TimeSpan::FromSeconds(WlanTestConnectTimeOut));
        NN_LOG("               %s ---> L116 Disconnect Wait End(%lld)\n", __FUNCTION__, (nn::os::GetSystemTick() - sysTick).ToTimeSpan().GetMilliSeconds());

    }

    // Masterでキープアライブの応答が無い場合、端末が切断扱い CLIENT
    TEST_F(LocalClientFuncTest, L116_2)
    {
        nn::os::Tick connectTime;

        const TestLocalConnectParams ConnectParam = {
            nn::wlan::Ssid(LocalMasterTestSSID), nn::wlan::MacAddress::CreateBroadcastMacAddress(), 6,
            { nn::wlan::SecurityMode_StaticAes, nn::wlan::SecurityMode_StaticAes, 0, LocalMasterAes16Key }, true, nn::wlan::BeaconIndication_Enable, 10
        };

        WLANTEST_ASSERT_TRUE(localClient.SetupLocalClient(ConnectParam.ssid, ConnectParam.security, TestMatchInfo, sizeof(TestMatchInfo) / sizeof(TestMatchInfo[0]), true, false));
        Connect(ConnectParam);

        NN_LOG("Master KeepAlive Lost Test Client Pawer Off please\n");
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(Time1day));
    }

    // Masterは5秒間、Clientから無線フレームを受信しないと、キープアライブ動作（要キャプチャ） MASTER
    TEST_F(LocalMasterFuncTest, L118_1)
    {
        const nn::wlan::MasterBssParameters fncTestMasterBss = {
            11, false, 5, true, nn::wlan::RateSetLegacy_11gMask, nn::wlan::RateSetLegacy_11gMask,
            { nn::wlan::SecurityMode_StaticAes, nn::wlan::SecurityMode_StaticAes, 0, LocalMasterAes16Key }, nn::wlan::Ssid(LocalMasterTestSSID), 100
        };

        WLANTEST_ASSERT_TRUE(localMaster.SetupLocalMaster(fncTestMasterBss, TestMatchInfo, sizeof(TestMatchInfo) / sizeof(TestMatchInfo[0])));

        // MASTER受信スレッド起動
        WLANTEST_ASSERT_TRUE(localMaster.LocalReceiveFrameStart(BufferSize100k, WlanTestPacketBufferSize, 1, 1, WlanTestGetFrameCount));
        WLANTEST_ASSERT_TRUE(localMaster.LocalReceiveWait(WlanTestMasterWaitTimeOut, true));

        NN_LOG("               %s ---> L118 Disconnect Wait Start(%lld)\n", __FUNCTION__, (nn::os::GetSystemTick() - sysTick).ToTimeSpan().GetMilliSeconds());
        nn::os::TimedWaitSystemEvent(&connectionEvent, nn::TimeSpan::FromSeconds(WlanTestConnectTimeOut));
        NN_LOG("               %s ---> L118 Disconnect Wait End(%lld)\n", __FUNCTION__, (nn::os::GetSystemTick() - sysTick).ToTimeSpan().GetMilliSeconds());
    }

    // Masterは5秒間、Clientから無線フレームを受信しないと、キープアライブ動作（要キャプチャ） CLIENT
    TEST_F(LocalClientFuncTest, L118_2)
    {
        nn::os::Tick connectTime;

        const TestLocalConnectParams ConnectParam = {
            nn::wlan::Ssid(LocalMasterTestSSID), nn::wlan::MacAddress::CreateBroadcastMacAddress(), 11,
            { nn::wlan::SecurityMode_StaticAes, nn::wlan::SecurityMode_StaticAes, 0, LocalMasterAes16Key }, true, nn::wlan::BeaconIndication_Enable, 10
        };

        WLANTEST_ASSERT_TRUE(localClient.SetupLocalClient(ConnectParam.ssid, ConnectParam.security, TestMatchInfo, sizeof(TestMatchInfo) / sizeof(TestMatchInfo[0]), true, false));
        Connect(ConnectParam);

        // 受信スレッド起動
        WLANTEST_ASSERT_TRUE(localClient.LocalReceiveFrameStart(BufferSize100k, WlanTestPacketBufferSize, 1, 1, WlanTestGetFrameCount));
        WLANTEST_ASSERT_TRUE(localClient.LocalReceiveWait(WlanMaxClientWaitMsecTimeOut, Time1s, true));

        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(Time15s));
        localClient.LocalReceiveFrameStop();

        Disconnect();
    }

    // Masterは、ClientからDataFrameを受信するとキープアライブタイマを初期化すること（要キャプチャ） MASTER
    TEST_F(LocalMasterFuncTest, L120_1)
    {
        nn::wlan::MacAddress clientMacArray[ClientMaxConnect];
        nn::wlan::MacAddress outMacAddress;
        std::unique_ptr<uint8_t[]> getBuffer(new uint8_t[TestScanMaxSize]);
        size_t outSize;

        WLANTEST_ASSERT_TRUE(getBuffer.get() != nullptr);

        const nn::wlan::MasterBssParameters fncTestMasterBss = {
            1, false, 5, true, nn::wlan::RateSetLegacy_11gMask, nn::wlan::RateSetLegacy_11gMask,
            { nn::wlan::SecurityMode_Open, nn::wlan::SecurityMode_Open, 0, "" }, nn::wlan::Ssid(LocalMasterTestSSID), 100
        };

        WLANTEST_ASSERT_TRUE(localMaster.SetupLocalMaster(fncTestMasterBss, TestMatchInfo, sizeof(TestMatchInfo) / sizeof(TestMatchInfo[0])));

        // MASTER CLIENT接続待機
        WLANTEST_ASSERT_TRUE(localMaster.LocalMasterConnectWait(clientMacArray, WlanTestConnectTimeOut, 1) > 0);

        WLANTEST_ASSERT_TRUE(localMaster.LocalCancelFrameStart(Time5s));
        WLANTEST_EXPECT_RESULT_SUCCESS(nn::wlan::Local::GetFrameRaw(getBuffer.get(), TestScanMaxSize, &outSize, localMaster.m_entRxId));
        localMaster.LocalCancelFrameStop();

        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(Time15s));
    }

    // MMasterは、ClientからDataFrameを受信するとキープアライブタイマを初期化すること（要キャプチャ） CLIENT
    TEST_F(LocalClientFuncTest, L120_2)
    {
        const TestLocalConnectParams ConnectParam = {
            nn::wlan::Ssid(LocalMasterTestSSID), nn::wlan::MacAddress::CreateBroadcastMacAddress(), 1,
            { nn::wlan::SecurityMode_Open, nn::wlan::SecurityMode_Open, 0, "" }, true, nn::wlan::BeaconIndication_Enable, 10
        };
        size_t putSize = ActionFrameBufferSize;
        std::unique_ptr<uint8_t[]> putBuffer(new uint8_t[putSize]);
        nn::wlan::MacAddress fromMacAddress;

        WLANTEST_ASSERT_TRUE(putBuffer.get() != nullptr);

        WLANTEST_ASSERT_TRUE(localClient.SetupLocalClient(ConnectParam.ssid, ConnectParam.security, TestMatchInfo, sizeof(TestMatchInfo) / sizeof(TestMatchInfo[0]), true, false));
        Connect(ConnectParam);

        WLANTEST_EXPECT_RESULT_SUCCESS(nn::wlan::Local::GetConnectionStatus(&connectionStatus));
        WLANTEST_EXPECT_RESULT_SUCCESS(nn::wlan::Local::GetMacAddress(&fromMacAddress));
        WLANTEST_EXPECT_TRUE(LocalApiClass::LocalMakeFrame(putBuffer.get(), putSize, connectionStatus.bssid, fromMacAddress, EtherTypes[0], TestMatchInfo[0]));

        WLANTEST_EXPECT_RESULT_SUCCESS(nn::wlan::Local::PutFrameRaw(putBuffer.get(), putSize));

        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(Time60s));
        Disconnect();
    }

    // Masterは、複数Clientを個別判別してキープアライブを行う（要キャプチャ） MASTER
    TEST_F(LocalMasterFuncTest, L121_1)
    {
        nn::wlan::MacAddress clientMacArray[ClientMaxConnect];
        const nn::wlan::MasterBssParameters fncTestMasterBss = {
            11, false, 5, true, nn::wlan::RateSetLegacy_11gMask, nn::wlan::RateSetLegacy_11gMask,
            { nn::wlan::SecurityMode_StaticAes, nn::wlan::SecurityMode_StaticAes, 0, LocalMasterAes16Key }, nn::wlan::Ssid(LocalMasterTestSSID), 100
        };

        WLANTEST_ASSERT_TRUE(localMaster.SetupLocalMaster(fncTestMasterBss, TestMatchInfo, sizeof(TestMatchInfo) / sizeof(TestMatchInfo[0])));

        // MASTER CLIENT接続待機
        WLANTEST_ASSERT_TRUE(localMaster.LocalMasterConnectWait(clientMacArray, WlanTestConnectTimeOut, ClientMaxConnect) > 0);

        // MASTER CLIENT切断待機
        sysTick = nn::os::GetSystemTick();
        NN_LOG("             L121 - Master Client Disconnect Wait Start : %lu msec\n", (nn::os::GetSystemTick() - sysTick).ToTimeSpan().GetMilliSeconds());
        WLANTEST_ASSERT_TRUE(localMaster.LocalMasterDisConnectWait(clientMacArray, Time1h) == 0);
        NN_LOG("             L121 - Master Client Disconnect Wait End : %lu msec\n", (nn::os::GetSystemTick() - sysTick).ToTimeSpan().GetMilliSeconds());

    }

    // Masterは、複数Clientを個別判別してキープアライブを行う（要キャプチャ） CLIENT 7台
    TEST_F(LocalClientFuncTest, L121_2)
    {
        nn::os::Tick connectTime;

        const TestLocalConnectParams ConnectParam = {
            nn::wlan::Ssid(LocalMasterTestSSID), nn::wlan::MacAddress::CreateBroadcastMacAddress(), 11,
            { nn::wlan::SecurityMode_StaticAes, nn::wlan::SecurityMode_StaticAes, 0, LocalMasterAes16Key }, true, nn::wlan::BeaconIndication_Enable, 10
        };

        WLANTEST_ASSERT_TRUE(localClient.SetupLocalClient(ConnectParam.ssid, ConnectParam.security, TestMatchInfo, sizeof(TestMatchInfo) / sizeof(TestMatchInfo[0]), true, false));
        Connect(ConnectParam);

        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(Time60s));

        Disconnect();
    }

    // LDPCが無効となっていること（要キャプチャ） MASTER
    TEST_F(LocalMasterFuncTest, L122_1)
    {
        const nn::wlan::MasterBssParameters fncTestMasterBss = {
            1, false, 5, true, nn::wlan::RateSetLegacy_11gMask, nn::wlan::RateSetLegacy_11gMask,
            { nn::wlan::SecurityMode_StaticAes, nn::wlan::SecurityMode_StaticAes, 0, LocalMasterAes16Key }, nn::wlan::Ssid(LocalMasterTestSSID), 10
        };

        WLANTEST_ASSERT_TRUE(localMaster.SetupLocalMaster(fncTestMasterBss, TestMatchInfo, sizeof(TestMatchInfo) / sizeof(TestMatchInfo[0])));

        // MASTER CLIENT接続待機
        WLANTEST_ASSERT_TRUE(localMaster.LocalMasterConnectWait(localMaster.m_clientMacArray, WlanTestConnectTimeOut, 1) > 0);

        AllScan();

        nn::os::TimedWaitSystemEvent(&connectionEvent, nn::TimeSpan::FromSeconds(WlanTestConnectTimeOut));
    }

    // LDPCが無効となっていること（要キャプチャ） CLIENT
    TEST_F(LocalClientFuncTest, L122_2)
    {
        const TestLocalConnectParams ConnectParam = {
            nn::wlan::Ssid(LocalMasterTestSSID), nn::wlan::MacAddress::CreateBroadcastMacAddress(), 1,
            { nn::wlan::SecurityMode_StaticAes, nn::wlan::SecurityMode_StaticAes, 0, LocalMasterAes16Key }, true, nn::wlan::BeaconIndication_Enable, 10
        };

        WLANTEST_ASSERT_TRUE(localClient.SetupLocalClient(ConnectParam.ssid, ConnectParam.security, TestMatchInfo, sizeof(TestMatchInfo) / sizeof(TestMatchInfo[0]), true, false));
        Connect(ConnectParam);

        AllScan();

        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(Time30s));

        Disconnect();
    }

    // GetConnectionStatusの値が即時に反映されること
    // Masterは接続/切断されたClientの情報が即時に反映されること MASTER
    TEST_F(LocalMasterFuncTest, L123_L124_1)
    {
        nn::wlan::MacAddress clientMacArray[ClientMaxConnect];
        char macStrArray[nn::wlan::MacAddress::MacStringSize];

        const nn::wlan::MasterBssParameters fncTestMasterBss = {
            1, false, 5, true, nn::wlan::RateSetLegacy_11gMask, nn::wlan::RateSetLegacy_11gMask,
            { nn::wlan::SecurityMode_StaticAes, nn::wlan::SecurityMode_StaticAes, 0, LocalMasterAes16Key }, nn::wlan::Ssid(LocalMasterTest4SSID), 100
        };

        // ジョイコン1セット（各2台）
        WLANTEST_ASSERT_TRUE(localMaster.SetBtMode(LocalApiBtNode2));

        WLANTEST_ASSERT_TRUE(localMaster.SetupLocalMaster(fncTestMasterBss, TestMatchInfo, sizeof(TestMatchInfo) / sizeof(TestMatchInfo[0])));

        // MASTER CLIENT接続待機
        WLANTEST_ASSERT_TRUE(localMaster.LocalMasterConnectWait(clientMacArray, WlanTestConnectTimeOut, 1) > 0);

        sysTick = nn::os::GetSystemTick();
        while ((nn::os::GetSystemTick() - sysTick).ToTimeSpan().GetMilliSeconds() <= Time1800s)
        {
            WLANTEST_EXPECT_TRUE(nn::os::TimedWaitSystemEvent(&connectionEvent, nn::TimeSpan::FromSeconds(WlanTestConnectTimeOut)));
            NN_LOG("\n             *** MASTER CLIENT STATUS ***\n");
            NN_LOG("             L123 L124 - MASTER EVENT TIME : %lld\n", (nn::os::GetSystemTick() - sysTick).ToTimeSpan().GetMilliSeconds());

            WLANTEST_EXPECT_RESULT_SUCCESS(nn::wlan::Local::GetClientStatus(localMaster.m_clientStatus, &localMaster.m_clientStatusBitMap));
            for (int8_t i = nn::wlan::ConnectableClientsCountMax; i >= 0; i--)
            {
                if ((localMaster.m_clientStatusBitMap >> i) & 0x00000001)
                {
                    NN_LOG("               GetClientStatusBitMap : 0x%08x\n", localMaster.m_clientStatusBitMap);
                    NN_LOG("               MAC ADDRESS     : %s\n", localMaster.m_clientStatus[i].clientMacAddress.GetString(macStrArray));
                    NN_LOG("               STATE           : %s\n", WlanConnectionStateStr[localMaster.m_clientStatus[i].state]);
                    NN_LOG("               CAUSE           : %d\n", localMaster.m_clientStatus[i].cause);
                    NN_LOG("               RESASON CODE    : %d\n", localMaster.m_clientStatus[i].statusReasonCode);
                    NN_LOG("               CAPABILITY INFO : %d\n", localMaster.m_clientStatus[i].capabilityInfo);
                    NN_LOG("               RSSI            : %d\n\n", localMaster.m_clientStatus[i].rssi);
                }
            }
            NN_LOG("             *** MASTER CLIENT STATUS ***\n");
        }

    }

    // GetConnectionStatusの値が即時に反映されること
    // Masterは接続/切断されたClientの情報が即時に反映されること CLIENT
    TEST_F(LocalClientFuncTest, L123_L124_2)
    {
        char macStr[nn::wlan::MacAddress::MacStringSize];
        char ssidStr[nn::wlan::Ssid::SsidHexStringLengthMax];
        nn::os::Tick connectTime;
        bool isConnected = true;

        const TestLocalConnectParams ConnectParam = {
            nn::wlan::Ssid(LocalMasterTest4SSID), nn::wlan::MacAddress::CreateBroadcastMacAddress(), 1,
            { nn::wlan::SecurityMode_StaticAes, nn::wlan::SecurityMode_StaticAes, 0, LocalMasterAes16Key }, true, nn::wlan::BeaconIndication_Enable, 10
        };

        // ジョイコン1セット（各2台）
        WLANTEST_ASSERT_TRUE(localClient.SetBtMode(LocalApiBtNode2));

        WLANTEST_ASSERT_TRUE(localClient.SetupLocalClient(ConnectParam.ssid, ConnectParam.security, TestMatchInfo, sizeof(TestMatchInfo) / sizeof(TestMatchInfo[0]), true, false));
        Connect(ConnectParam);

        sysTick = nn::os::GetSystemTick();
        while ((nn::os::GetSystemTick() - sysTick).ToTimeSpan().GetMilliSeconds() <= Time1800s)
        {
            int sleepTime = GetRandom(3000, 30000);
            nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(sleepTime));
            if (isConnected == true)
            {
                Disconnect();
                isConnected = false;
            }
            else
            {
                Connect(ConnectParam);
                isConnected = true;
            }

            WLANTEST_EXPECT_RESULT_SUCCESS(nn::wlan::Local::GetConnectionStatus(&connectionStatus));
            NN_LOG("\n             *** CLIENT CONNECTION STATUS ***\n");
            NN_LOG("             L123 L124 - CLIENT EVENT TIME : %lld\n", (nn::os::GetSystemTick() - sysTick).ToTimeSpan().GetMilliSeconds());
            NN_LOG("             STATE           : %s\n", WlanConnectionStateStr[connectionStatus.state]);
            NN_LOG("             CAUSE           : %s\n", CauseOfInfo[connectionStatus.cause]);
            NN_LOG("             CHANNEL         : %d\n", connectionStatus.channel);
            NN_LOG("             SSID            : %s\n", connectionStatus.ssid.GetHexString(ssidStr));
            NN_LOG("             BSSID           : %s\n", connectionStatus.bssid.GetString(macStr));
            NN_LOG("             AID             : %d\n", connectionStatus.aid);
            NN_LOG("             REASON CODE     : %d\n", connectionStatus.statusReasonCode);
            NN_LOG("             CAPABILITY INFO : 0x%04X\n", connectionStatus.capabilityInfo);
            NN_LOG("             BEACON INTERVAL : %d\n", connectionStatus.beaconInterval);
            NN_LOG("             *** CLIENT CONNECTION STATUS ***\n");
        }

        Disconnect();
    }

    // 8台の端末が、4時間ごとにMasterになる端末を変えながら、22msecに1024Byteの通信 Master
    TEST_F(LocalMasterClientFuncTest, LocalLongTermAging_1)
    {
        bool isStart = false;

        MasterSetup(ClientMaxConnect);

        sysTick = nn::os::GetSystemTick();
        GetFirstSeqNumber();

        while (NN_STATIC_CONDITION(1))
        {
            if (isMasterSeqNumber() == true)
            {
                MaseterAgingFrameTest(isStart, AgingFrameBufferSize, AgingFrameBufferSize, PutInterval, Time4h);
            }
            else
            {
                ClientAgingFrameTest(isStart, AgingFrameBufferSize, AgingFrameBufferSize, PutInterval, Time4h);
            }

            if ((nn::os::GetSystemTick() - sysTick).ToTimeSpan().GetMilliSeconds() >= Time1day)
            {
                break;
            }
            GetNextSeqNumber();
        }

        WLANTEST_ASSERT_TRUE(ChangeMode(OpenModeType::OpenMode_Master));
    }

    // 8台の端末が、4時間ごとにMasterになる端末を変えながら、22msecに1024Byteの通信 Client
    TEST_F(LocalMasterClientFuncTest, LocalLongTermAging_2)
    {
        bool isStart = false;

        ClientSetup();

        sysTick = nn::os::GetSystemTick();
        GetFirstSeqNumber();

        while (NN_STATIC_CONDITION(1))
        {
            if (isMasterSeqNumber() == true)
            {
                MaseterAgingFrameTest(isStart, AgingFrameBufferSize, AgingFrameBufferSize, PutInterval, Time4h);
            }
            else
            {
                ClientAgingFrameTest(isStart, AgingFrameBufferSize, AgingFrameBufferSize, PutInterval, Time4h);
            }

            if ((nn::os::GetSystemTick() - sysTick).ToTimeSpan().GetMilliSeconds() >= Time1day)
            {
                break;
            }
            GetNextSeqNumber();
        }

        WLANTEST_ASSERT_TRUE(ChangeMode(OpenModeType::OpenMode_Master));
    }

    // Masterが同チャネルのClient7台から16msec毎のActionFrame受信 MASTER
    TEST_F(LocalMasterFuncTest, L126_1)
    {
        const nn::wlan::MasterBssParameters fncTestMasterBss = {
            6, false, 30, true, nn::wlan::RateSetLegacy_11gMask, nn::wlan::RateSetLegacy_11gMask,
            { nn::wlan::SecurityMode_Open, nn::wlan::SecurityMode_Open, 0, "" }, nn::wlan::Ssid(LocalMasterTest2SSID), 100
        };

        // ジョイコン1セット（各2台）
        WLANTEST_ASSERT_TRUE(localMaster.SetBtMode(LocalApiBtNode2));

        WLANTEST_ASSERT_TRUE(localMaster.SetupLocalMaster(fncTestMasterBss, TestMatchInfo, sizeof(TestMatchInfo) / sizeof(TestMatchInfo[0]), true));

        // MASTER受信スレッド起動
        WLANTEST_ASSERT_TRUE(localMaster.LocalReceiveFrameStart(BufferSize200k, DataFrameBufferSize, ClientMaxConnect, ClientMaxConnect, WlanTestAgingGetFrameCount));
        // CLIENT接続待機
        WLANTEST_ASSERT_TRUE(localMaster.LocalReceiveWait(WlanTestMasterWaitTimeOut));

        sysTick = nn::os::GetSystemTick();
        logOutTime = sysTick;

        // 全CLIENTが切断されるまで待機
        WLANTEST_ASSERT_TRUE(localMaster.LocalMasterDisConnectWait(Time120s) == 0)

        // MASTER受信スレッド停止
        localMaster.LocalReceiveFrameStop();

        localMaster.LocalGetConnectInfo(&connectsInfo);
        WLANTEST_EXPECT_TRUE(connectsInfo.connectCount >= ClientMaxConnect);

        for (int32_t i = 0; i < connectsInfo.connectCount; i++)
        {
            WLANTEST_EXPECT_TRUE(connectsInfo.facing[i].isConnected);
            WLANTEST_EXPECT_TRUE(connectsInfo.facing[i].perRate <= 10.00);
        }

        localMaster.LocalTraceConnectsInfo(true);
    }

    // Masterが同チャネルのClient7台から16msec毎のActionFrame受信 CLIENT
    TEST_F(LocalClientFuncTest, L126_2)
    {
        size_t gutSize = DataFrameBufferSize;
        std::unique_ptr<uint8_t[]> getBuffer(new uint8_t[gutSize]);
        nn::wlan::ConnectionStatus clientStatus;
        size_t outSize;
        nn::Result result;

        // 通信評価定義
        const uint64_t actionFrameByteNormal = 60 * WlanTestPacketBufferSize * (Time60s / 1000);
        const uint64_t actionFrameByteMin = actionFrameByteNormal - (actionFrameByteNormal * 0.1);
        const uint64_t actionFrameByteMax = actionFrameByteNormal + (actionFrameByteNormal * 0.1);

        const TestLocalConnectParams ConnectParam = {
            nn::wlan::Ssid(LocalMasterTest2SSID), nn::wlan::MacAddress::CreateBroadcastMacAddress(), 6,
            { nn::wlan::SecurityMode_Open, nn::wlan::SecurityMode_Open, 0, "" }, true, nn::wlan::BeaconIndication_Enable, 10
        };

        // ジョイコン1セット（各2台）
        WLANTEST_ASSERT_TRUE(localClient.SetBtMode(LocalApiBtNode2));

        WLANTEST_ASSERT_TRUE(localClient.SetupLocalClient(ConnectParam.ssid, ConnectParam.security, TestMatchInfo, sizeof(TestMatchInfo) / sizeof(TestMatchInfo[0]), true, true));

        Connect(ConnectParam);

        WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::GetConnectionStatus(&connectionStatus));

        nn::wlan::Local::GetFrameRaw(getBuffer.get(), gutSize, &outSize, localClient.m_entRxId);

        // 送信スレッド起動
        WLANTEST_ASSERT_TRUE(localClient.LocalSendActionFrameStart(WlanTestPacketBufferSize, Time120s, Interval_16m, nn::wlan::ActionFrameType_Beacon));

        sysTick = nn::os::GetSystemTick();
        // 60秒間通信
        while ((nn::os::GetSystemTick() - sysTick).ToTimeSpan().GetMilliSeconds() <= Time60s)
        {
            // MASTERからの接続がされたらNG
            if (nn::os::TimedWaitSystemEvent(&connectionEvent, nn::TimeSpan::FromMilliSeconds(Interval_16m)) == true)
            {
                WLANTEST_EXPECT_RESULT_SUCCESS(nn::wlan::Local::GetConnectionStatus(&clientStatus));
                WLANTEST_EXPECT_TRUE(clientStatus.state == nn::wlan::ConnectionState_Connected);
                if (clientStatus.state != nn::wlan::ConnectionState_Connected)
                {
                    break;
                }
            }
        }

        // 送信スレッド停止
        localClient.LocalSendFrameStop();
        localClient.LocalGetConnectInfo(&connectsInfo);
        Disconnect();

        // ActionFrame送信サイズ範囲評価（1.max(最小, 送信サイズ)、2.min(1.結果, 最大）⇒ 範囲内なら送信サイズが戻り値。以外は最大／最小が戻り値）
        NN_LOG("             L126_2 ActionFrame Send Total Byte:%lld(Range:%lld - %lld)\n", connectsInfo.totalSendSize, actionFrameByteMin, actionFrameByteMax);
        WLANTEST_EXPECT_TRUE(std::min(std::max(actionFrameByteMin, connectsInfo.totalSendSize), actionFrameByteMax) == connectsInfo.totalSendSize);
    }

    // MasterとClient1台のStaticAES暗号接続で、1時間の連続通信を行い、途切れず60Mbps(前回の結果からの変動幅10%以内)以上のスループットで通信できること MASTER
    TEST_F(LocalMasterFuncTest, L127_1)
    {
        uint connectCount = 0;
        bool isEventSygnal = false;

        const nn::wlan::MasterBssParameters fncTestMasterBss = {
            6, false, 30, true, nn::wlan::RateSetLegacy_11gMask, nn::wlan::RateSetLegacy_11gMask,
            { nn::wlan::SecurityMode_StaticAes, nn::wlan::SecurityMode_StaticAes, 0, LocalMasterAes16Key }, nn::wlan::Ssid(LocalMasterTest2SSID), 10
        };

        // ジョイコン2セット（各4台）
        WLANTEST_ASSERT_TRUE(localMaster.SetBtMode(LocalApiBtNode4));

        WLANTEST_ASSERT_TRUE(localMaster.SetupLocalMaster(fncTestMasterBss, TestMatchInfo, sizeof(TestMatchInfo) / sizeof(TestMatchInfo[0])));

        // MASTER受信スレッド起動
        WLANTEST_ASSERT_TRUE(localMaster.LocalReceiveFrameStart(BufferSize200k, WlanTestRecvBufferSize, 1, 1, WlanTestAgingGetFrameCount));

        // CLIENT接続待機
        WLANTEST_ASSERT_TRUE(localMaster.LocalReceiveWait(WlanTestMasterWaitTimeOut));

        // 送信スレッド起動
        WLANTEST_ASSERT_TRUE(localMaster.LocalSendFrameStart(WlanTestRatePacketBufferSize, Time1h, 0, true));

        sysTick = nn::os::GetSystemTick();
        logOutTime = sysTick;

        localMaster.LocalGetConnectInfo(&connectsInfo);
        while ((nn::os::GetSystemTick() - sysTick).ToTimeSpan().GetMilliSeconds() <= Time1h)
        {
            isEventSygnal = nn::os::TimedWaitSystemEvent(&connectionEvent, nn::TimeSpan::FromMilliSeconds(Time5s));
            if (isEventSygnal == true)
            {
                connectCount = localMaster.GetClientMacAddress(localMaster.m_clientMacArray, nn::wlan::ConnectionState_Connected);
                WLANTEST_EXPECT_TRUE(connectCount != 0);
                if (connectCount == 0)
                {
                    break;
                }
                continue;
            }

            if ((nn::os::GetSystemTick() - logOutTime).ToTimeSpan().GetMilliSeconds() >= Time60s)
            {
                logOutTime = nn::os::GetSystemTick();
                localMaster.LocalGetConnectInfo(&connectsInfo);
                localMaster.LocalTraceConnectsInfo(true);
            }
        }

        // 送信スレッド停止
        localMaster.LocalSendFrameStop();

        // MASTER受信スレッド停止
        localMaster.LocalReceiveFrameStop();

        localMaster.LocalTraceConnectsInfo(true);
    }

    // MasterとClient1台のStaticAES暗号接続で、1時間の連続通信を行い、途切れず60Mbps(前回の結果からの変動幅10%以内)以上のスループットで通信できること CLIENT
    TEST_F(LocalClientFuncTest, L127_2)
    {
        nn::wlan::MacAddress clientMacAddress;
        nn::wlan::ConnectionStatus clientStatus;

        const TestLocalConnectParams ConnectParam = {
            nn::wlan::Ssid(LocalMasterTest2SSID), nn::wlan::MacAddress::CreateBroadcastMacAddress(), 6,
            { nn::wlan::SecurityMode_StaticAes, nn::wlan::SecurityMode_StaticAes, 0, LocalMasterAes16Key }, true, nn::wlan::BeaconIndication_Enable, 10
        };

        // ジョイコン2セット（各4台）
        WLANTEST_ASSERT_TRUE(localClient.SetBtMode(LocalApiBtNode4));

        WLANTEST_ASSERT_TRUE(localClient.SetupLocalClient(ConnectParam.ssid, ConnectParam.security, TestMatchInfo, sizeof(TestMatchInfo) / sizeof(TestMatchInfo[0]), true, false));

        Connect(ConnectParam);

        WLANTEST_EXPECT_RESULT_SUCCESS(nn::wlan::Local::GetMacAddress(&clientMacAddress));
        WLANTEST_EXPECT_RESULT_SUCCESS(nn::wlan::Local::GetConnectionStatus(&connectionStatus));

        // 受信スレッド起動
        WLANTEST_ASSERT_TRUE(localClient.LocalReceiveFrameStart(BufferSize200k, WlanTestRecvBufferSize, 1, 1, WlanTestAgingGetFrameCount));
        WLANTEST_ASSERT_TRUE(localClient.LocalReceiveWait(WlanMaxClientWaitMsecTimeOut, Time1s, true));

        // 送信スレッド起動
        WLANTEST_ASSERT_TRUE(localClient.LocalSendFrameStart(WlanTestRatePacketBufferSize, Time1h, 0, false));
        logOutTime = nn::os::GetSystemTick();
        sysTick = logOutTime;

        while (NN_STATIC_CONDITION(1))
        {
            // MASTERからの接続が切られるまで送信し続ける
            if (nn::os::TimedWaitSystemEvent(&connectionEvent, nn::TimeSpan::FromMilliSeconds(Time5s)) == true)
            {
                WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::GetConnectionStatus(&clientStatus));
                if (clientStatus.state != nn::wlan::ConnectionState_Connected)
                {
                    break;
                }
            }

            if ((nn::os::GetSystemTick() - logOutTime).ToTimeSpan().GetMilliSeconds() >= Time60s)
            {
                localClient.LocalTraceConnectsInfo(true);
                logOutTime = nn::os::GetSystemTick();
                localClient.LocalGetConnectInfo(&connectsInfo);
            }
        }

        // 送信スレッド停止
        localClient.LocalSendFrameStop();
        // 受信スレッド停止
        localClient.LocalReceiveFrameStop();

        localClient.LocalTraceConnectsInfo(true);

        Disconnect();
    }

    // MasterとClient7台接続で、16msec毎に512Bytesのブロードキャスト通信を6時間の連続通信を行い、PER10.0%以下でリンクダウンせず通信(BT各２台ペアリング) MASTER
    TEST_F(LocalMasterFuncTest, L128_1)
    {
        int64_t agingTime = Time6h;

        // 通信評価定義
        const uint64_t broadcastByteNormal = 60 * WlanTestPacketBufferSize * (agingTime / 1000);
        const uint64_t broadcastByteMin = broadcastByteNormal - (broadcastByteNormal * 0.1);
        const uint64_t broadcastByteMax = broadcastByteNormal + (broadcastByteNormal * 0.1);

        const nn::wlan::MasterBssParameters fncTestMasterBss = {
            11, false, 30, true, nn::wlan::RateSetLegacy_11gMask, nn::wlan::RateSetLegacy_11gMask,
            { nn::wlan::SecurityMode_StaticAes, nn::wlan::SecurityMode_StaticAes, 0, LocalMasterAes16Key }, nn::wlan::Ssid(LocalMasterTest2SSID), 10
        };

        // ジョイコン1セット（各2台）
        WLANTEST_ASSERT_TRUE(localMaster.SetBtMode(LocalApiBtNode2));

        WLANTEST_ASSERT_TRUE(localMaster.SetupLocalMaster(fncTestMasterBss, TestMatchInfo, sizeof(TestMatchInfo) / sizeof(TestMatchInfo[0])));

        // MASTER受信スレッド起動
        WLANTEST_ASSERT_TRUE(localMaster.LocalReceiveFrameStart(BufferSize100k, WlanTestPacketBufferSize, ClientMaxConnect, ClientMaxConnect, WlanTestAgingGetFrameCount));
        // CLIENT接続待機
        WLANTEST_ASSERT_TRUE(localMaster.LocalReceiveWait(WlanTestMasterWaitTimeOut));
        // MASTER送信スレッド起動
        WLANTEST_ASSERT_TRUE(localMaster.LocalBroadcastSendFrameStart(WlanTestPacketBufferSize, agingTime, LocalApiBroadFrameInterval));

        sysTick = nn::os::GetSystemTick();
        logOutTime = sysTick;
        // 全CLIENTが切断されるまでループに変更
        while (localMaster.LocalMasterDisConnectWait(Time1s) > 0)
        {
            if ((nn::os::GetSystemTick() - logOutTime).ToTimeSpan().GetMilliSeconds() >= Time180s)
            {
                localMaster.LocalTraceConnectsInfo(true);
                logOutTime = nn::os::GetSystemTick();
            }
        }

        // MASTER送信スレッド停止
        localMaster.LocalSendFrameStop();
        // MASTER受信スレッド停止
        localMaster.LocalReceiveFrameStop();

        localMaster.LocalGetConnectInfo(&connectsInfo);
        WLANTEST_EXPECT_TRUE(connectsInfo.connectCount == ClientMaxConnect);

        for (int32_t i = 0; i < connectsInfo.connectCount; i++)
        {
            WLANTEST_EXPECT_TRUE(connectsInfo.facing[i].isConnected);
            WLANTEST_EXPECT_TRUE(connectsInfo.facing[i].perRate <= 10.00);
        }

        // ブロードキャスト送信サイズ範囲評価（1.max(最小, 送信サイズ)、2.min(1.結果, 最大）⇒ 範囲内なら送信サイズが戻り値。以外は最大／最小が戻り値）
        NN_LOG("             L128_1 Broad Send Total Size:%lld(Range:%lld - %lld)\n", connectsInfo.totalSendSize, broadcastByteMin, broadcastByteMax);
        WLANTEST_EXPECT_TRUE(std::min(std::max(broadcastByteMin, connectsInfo.totalSendSize), broadcastByteMax) == connectsInfo.totalSendSize);
    }

    // MasterとClient7台接続で、16msec毎に512Bytesのブロードキャスト通信を6時間の連続通信を行い、PER10.0%以下でリンクダウンせず通信(BT各２台ペアリング) CLIENT
    TEST_F(LocalClientFuncTest, L128_2)
    {
        int64_t agingTime = Time6h;

        nn::wlan::MacAddress clientMacAddress;
        nn::wlan::ConnectionStatus clientStatus;
        nn::wlan::LinkLevel level;
        nn::Result result;

        // 通信評価定義
        const uint64_t broadcastByteNormal = 60 * WlanTestPacketBufferSize * (agingTime / 1000);
        const uint64_t broadcastByteMin = broadcastByteNormal - (broadcastByteNormal * 0.1);
        const uint64_t broadcastByteMax = broadcastByteNormal + (broadcastByteNormal * 0.1);

        const TestLocalConnectParams ConnectParam = {
            nn::wlan::Ssid(LocalMasterTest2SSID), nn::wlan::MacAddress::CreateBroadcastMacAddress(), 11,
            { nn::wlan::SecurityMode_StaticAes, nn::wlan::SecurityMode_StaticAes, 0, LocalMasterAes16Key }, true, nn::wlan::BeaconIndication_Enable, 10
        };

        // ジョイコン1セット（各2台）
        WLANTEST_ASSERT_TRUE(localClient.SetBtMode(LocalApiBtNode2));

        WLANTEST_ASSERT_TRUE(localClient.SetupLocalClient(ConnectParam.ssid, ConnectParam.security, TestMatchInfo, sizeof(TestMatchInfo) / sizeof(TestMatchInfo[0]), true, false));

        Connect(ConnectParam);

        WLANTEST_EXPECT_RESULT_SUCCESS(nn::wlan::Local::GetMacAddress(&clientMacAddress));
        WLANTEST_EXPECT_RESULT_SUCCESS(nn::wlan::Local::GetConnectionStatus(&connectionStatus));

        // 受信スレッド起動
        WLANTEST_ASSERT_TRUE(localClient.LocalReceiveFrameStart(BufferSize100k, WlanTestPacketBufferSize, 1, 1, WlanTestAgingGetFrameCount));
        WLANTEST_ASSERT_TRUE(localClient.LocalReceiveWait(WlanMaxClientWaitMsecTimeOut, Time1s, true));

        // 送信スレッド起動
        WLANTEST_ASSERT_TRUE(localClient.LocalBroadcastSendFrameStart(WlanTestPacketBufferSize, agingTime, LocalApiBroadFrameInterval));

        sysTick = nn::os::GetSystemTick();
        logOutTime = sysTick;
        while ((nn::os::GetSystemTick() - sysTick).ToTimeSpan().GetMilliSeconds() <= agingTime)
        {
            // 一定周期送信。通信が切断されたらNG
            if (nn::os::TimedWaitSystemEvent(&connectionEvent, nn::TimeSpan::FromMilliSeconds(Interval_16m)) == true)
            {
                WLANTEST_EXPECT_RESULT_SUCCESS(nn::wlan::Local::GetConnectionStatus(&clientStatus));
                WLANTEST_EXPECT_TRUE(clientStatus.state == nn::wlan::ConnectionState_Connected);
                if (clientStatus.state != nn::wlan::ConnectionState_Connected)
                {
                    localClient.LocalTraceConnectsInfo(true);
                    break;
                }
            }

            if ((nn::os::GetSystemTick() - logOutTime).ToTimeSpan().GetMilliSeconds() >= Time180s)
            {
                localClient.LocalTraceConnectsInfo(true);
                logOutTime = nn::os::GetSystemTick();
            }

            result = nn::wlan::Local::GetLinkLevel(&level);
            if (result.IsSuccess() != true)
            {
                WLANTEST_EXPECT_RESULT_SUCCESS(nn::wlan::Local::GetConnectionStatus(&clientStatus));
                if (clientStatus.state == nn::wlan::ConnectionState_Connected)
                {
                    WLANTEST_EXPECT_TRUE(nn::wlan::LinkLevel_0 != level);
                }
            }
        }

        // 送信スレッド停止
        localClient.LocalSendFrameStop();
        // 受信スレッド停止
        localClient.LocalReceiveFrameStop();

        localClient.LocalGetConnectInfo(&connectsInfo);
        WLANTEST_EXPECT_TRUE(connectsInfo.connectCount == ClientMaxConnect);

        for (int32_t i = 0; i < connectsInfo.connectCount; i++)
        {
            WLANTEST_EXPECT_TRUE(connectsInfo.facing[i].isConnected);
            WLANTEST_EXPECT_TRUE(connectsInfo.facing[i].perRate <= 10.00);
        }

        Disconnect();

        // ブロードキャスト送信サイズ範囲評価（1.max(最小, 送信サイズ)、2.min(1.結果, 最大）⇒ 範囲内なら送信サイズが戻り値。以外は最大／最小が戻り値）
        NN_LOG("             L128_2 Broad Send Total Size:%lld(Range:%lld - %lld)\n", connectsInfo.totalSendSize, broadcastByteMin, broadcastByteMax);
        WLANTEST_EXPECT_TRUE(std::min(std::max(broadcastByteMin, connectsInfo.totalSendSize), broadcastByteMax) == connectsInfo.totalSendSize);
    }

    // Masterでの既接続スキャン中にSetActionFrameWithBeaconの送信
    TEST_F(LocalMasterFuncTest, L130)
    {
        size_t putSize = ActionFrameBufferSize;
        std::unique_ptr<uint8_t[]> putBuffer(new uint8_t[putSize]);
        nn::wlan::MacAddress fromMacAddress;

        nn::wlan::ScanIeMatchInfo info;
        const uint32_t scanTestCount = 100;
        const nn::wlan::ScanParameters StartScanTestParams = {
            nn::wlan::ScanType_Active,{ 1 }, 1, 40, -1, &ssidList[0], 1, nn::wlan::MacAddress::CreateBroadcastMacAddress()
        };

        const nn::wlan::MasterBssParameters fncTestMasterBss = {
            6, false, 30, true, nn::wlan::RateSetLegacy_11gMask, nn::wlan::RateSetLegacy_11gMask,
            { nn::wlan::SecurityMode_Open, nn::wlan::SecurityMode_Open, 0, "" }, nn::wlan::Ssid(LocalMasterTestSSID), 100
        };

        std::unique_ptr<uint8_t[]> pTestBuffer(new uint8_t[TestScanMaxSize]);
        WLANTEST_ASSERT_TRUE(pTestBuffer.get() != nullptr);

        WLANTEST_ASSERT_TRUE(localMaster.SetupLocalMaster(fncTestMasterBss, TestMatchInfo, sizeof(TestMatchInfo) / sizeof(TestMatchInfo[0]), true));
        WLANTEST_ASSERT_TRUE(localMaster.LocalMasterConnectWait(localMaster.m_clientMacArray, WlanTestConnectTimeOut, 1) > 0);

        testPutMac  = nn::wlan::MacAddress(localMaster.m_clientMacArray[0]);
        testChannel = fncTestMasterBss.channel;

        WLANTEST_EXPECT_RESULT_SUCCESS(nn::wlan::Local::GetMacAddress(&fromMacAddress));

        WLANTEST_EXPECT_TRUE(LocalApiClass::LocalMakeActionFrame(putBuffer.get(), putSize, testPutMac, fromMacAddress, EtherTypes[0], TestMatchInfo[0], nn::wlan::ActionFrameType_Beacon));
        WLANTEST_EXPECT_RESULT_SUCCESS(nn::wlan::Local::SetActionFrameWithBeacon(putBuffer.get(), putSize));

        for (int32_t i = 0; i < scanTestCount; i++)
        {
            NN_LOG("             L75 - StartScan START %d / %d\n", i + 1, scanTestCount);
            WLANTEST_EXPECT_RESULT_SUCCESS(nn::wlan::Local::StartScan(pTestBuffer.get(), TestScanMaxSize, StartScanTestParams, &info));
            NN_LOG("             L75 - StartScan END   %d / %d\n", i + 1, scanTestCount);
        }
    }

    // Master起動中に周期的なSetActionFrameWithBeaconの送信 Master
    TEST_F(LocalMasterFuncTest, L131_1)
    {
        nn::wlan::MacAddress clientArray[LocalTcpGetMacAddressMax];
        size_t putSize = ActionFrameBufferSize;
        std::unique_ptr<uint8_t[]> putBuffer(new uint8_t[putSize]);
        nn::wlan::MacAddress putMacAddress;
        nn::wlan::MacAddress fromMacAddress;
        uint32_t macCnt;

        const nn::wlan::MasterBssParameters fncTestMasterBss = {
            6, false, 30, true, nn::wlan::RateSetLegacy_11gMask, nn::wlan::RateSetLegacy_11gMask,
            { nn::wlan::SecurityMode_Open, nn::wlan::SecurityMode_Open, 0, "" }, nn::wlan::Ssid(LocalMasterTestSSID), 100
        };

        WLANTEST_ASSERT_TRUE(putBuffer.get() != nullptr);

        WLANTEST_ASSERT_TRUE(localMaster.GetMasterTcpMacAddress(clientArray, macCnt, tcpMaster, 1));
        WLANTEST_ASSERT_TRUE(localMaster.SetupLocalMaster(fncTestMasterBss, TestMatchInfo, sizeof(TestMatchInfo) / sizeof(TestMatchInfo[0]), true));

        putMacAddress = nn::wlan::MacAddress(clientArray[1]);

        WLANTEST_EXPECT_RESULT_SUCCESS(nn::wlan::Local::GetMacAddress(&fromMacAddress));

        // 1秒周期で10回送信する
        for (int32_t cnt = 0; cnt < 10; cnt++)
        {
            WLANTEST_EXPECT_TRUE(LocalApiClass::LocalMakeActionFrame(putBuffer.get(), putSize, putMacAddress, fromMacAddress, EtherTypes[0], TestMatchInfo[0], nn::wlan::ActionFrameType_Beacon, cnt));
            WLANTEST_EXPECT_RESULT_SUCCESS(nn::wlan::Local::SetActionFrameWithBeacon(putBuffer.get(), putSize));
            nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(Time1s));
        }
    }

    // Master起動中に周期的なSetActionFrameWithBeaconの送信 Client
    TEST_F(LocalClientFuncTest, L131_2)
    {
        const TestLocalConnectParams ConnectParam = {
            nn::wlan::Ssid(LocalMasterTestSSID), nn::wlan::MacAddress::CreateBroadcastMacAddress(), 6,
            { nn::wlan::SecurityMode_Open, nn::wlan::SecurityMode_Open, 0, "" }, true, nn::wlan::BeaconIndication_Enable, 10
        };

        nn::wlan::MacAddress clientArray[LocalTcpGetMacAddressMax];
        uint32_t macCnt;

        IpV4Address tcpClient = {
            "0.0.0.0", "255.255.255.0", "192.168.11.1",
            "8.8.8.8", "8.8.8.9", 0, false
        };
        nn::util::SNPrintf(tcpClient.strIpAddress, sizeof(tcpClient.strIpAddress), "192.168.11.%d", g_IpOct4);

        WLANTEST_ASSERT_TRUE(localClient.GetClientTcpMacAddress(clientArray, macCnt, tcpMaster, tcpClient));
        WLANTEST_ASSERT_TRUE(localClient.SetupLocalClient(ConnectParam.ssid, ConnectParam.security, TestMatchInfo, sizeof(TestMatchInfo) / sizeof(TestMatchInfo[0]), true, true));

        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(Time15s));
    }

    // 受信エントリに登録したActionFrameTypeを持つパケットを受信
    TEST_F(LocalClientFuncTest, L132)
    {
        const uint16_t FrameTypes[] = { nn::wlan::ActionFrameType_Beacon };
        std::unique_ptr<uint8_t[]> getBuffer(new uint8_t[TestScanMaxSize]);
        nn::wlan::MacAddress outMacAddress;
        size_t outSize;
        uint32_t actRxId;

        nn::wlan::MacAddress macAddress = nn::wlan::MacAddress::CreateBroadcastMacAddress();
        const TestLocalConnectParams ConnectParam = {
            nn::wlan::Ssid(LocalMasterTestSSID), nn::wlan::MacAddress::CreateBroadcastMacAddress(), 11,
            { nn::wlan::SecurityMode_Open, nn::wlan::SecurityMode_Open, 0, "" }, true, nn::wlan::BeaconIndication_Enable, 10
        };

        AllScan();

        WLANTEST_ASSERT_TRUE(localClient.SetupLocalClient(ConnectParam.ssid, ConnectParam.security, TestMatchInfo, sizeof(TestMatchInfo) / sizeof(TestMatchInfo[0]), true, false));

        WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::CreateRxEntryForActionFrame(&actRxId, FrameTypes, sizeof(FrameTypes) / sizeof(uint16_t), 10));
        Connect(ConnectParam);
        WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::GetActionFrame(&outMacAddress, getBuffer.get(), TestScanMaxSize, &outSize, actRxId));
        WLANTEST_EXPECT_TRUE(outSize > WlanTestGetFramMineSize);

        LocalActionFrameData *pActionFrame = reinterpret_cast<LocalActionFrameData*>(getBuffer.get());
        WLANTEST_EXPECT_TRUE(pActionFrame->frameType == FrameTypes[0]);

        WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::DeleteRxEntryForActionFrame(actRxId));
        WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::Disconnect(nn::wlan::LocalCommunicationMode_ClientSpectator, nullptr));
    }

    // 受信エントリにActionFrameTypeを追加し、受信
    TEST_F(LocalClientFuncTest, L133)
    {
        uint32_t actRxId;
        const uint16_t FrameTypes[] = { nn::wlan::ActionFrameType_Beacon };
        nn::wlan::MacAddress outMacAddress;
        std::unique_ptr<uint8_t[]> getBuffer(new uint8_t[TestScanMaxSize]);
        size_t outSize;
        uint32_t outActionRxId;

        nn::wlan::MacAddress macAddress = nn::wlan::MacAddress::CreateBroadcastMacAddress();
        const TestLocalConnectParams ConnectParam = {
            nn::wlan::Ssid(LocalMasterTestSSID), nn::wlan::MacAddress::CreateBroadcastMacAddress(), 11,
            { nn::wlan::SecurityMode_Open, nn::wlan::SecurityMode_Open, 0, "" }, true, nn::wlan::BeaconIndication_Enable, 10
        };

        AllScan();

        WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::CreateRxEntryForActionFrame(&actRxId, FrameTypes, 1, 10));
        // 一度ActionFrameType_Beaconを削除して登録
        WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::DeleteSubtypeFromRxEntryForActionFrame(&outActionRxId, nn::wlan::ActionFrameType_Beacon));
        WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::AddSubtypeToRxEntryForActionFrame(actRxId, nn::wlan::ActionFrameType_Beacon));
        Connect(ConnectParam);
        WLANTEST_EXPECT_RESULT_SUCCESS(nn::wlan::Local::GetConnectionStatus(&connectionStatus));
        WLANTEST_EXPECT_RESULT_SUCCESS(nn::wlan::Local::GetActionFrame(&outMacAddress, getBuffer.get(), TestScanMaxSize, &outSize, actRxId));
        WLANTEST_EXPECT_TRUE(outSize > WlanTestGetFramMineSize);

        LocalActionFrameData *pActionFrame = reinterpret_cast<LocalActionFrameData*>(getBuffer.get());
        WLANTEST_EXPECT_TRUE(pActionFrame->frameType == FrameTypes[0]);
        WLANTEST_EXPECT_TRUE(connectionStatus.bssid == outMacAddress);

        WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::DeleteRxEntryForActionFrame(actRxId));
        WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::Disconnect(nn::wlan::LocalCommunicationMode_ClientSpectator, nullptr));
    }

    // 削除した受信エントリのActionFrameTypeは削除処理以降受信しない MASTER
    TEST_F(LocalMasterFuncTest, L134_1)
    {
        size_t putSize = ActionFrameBufferSize;
        std::unique_ptr<uint8_t[]> putBuffer(new uint8_t[putSize]);
        nn::wlan::MacAddress clientMacArray[ClientMaxConnect];
        WLANTEST_ASSERT_TRUE(putBuffer.get() != nullptr);
        nn::wlan::ConnectionStatus clientStatus;
        nn::Result result;

        const nn::wlan::MasterBssParameters fncTestMasterBss = {
            11, false, 30, true, nn::wlan::RateSetLegacy_11gMask, nn::wlan::RateSetLegacy_11gMask,
            { nn::wlan::SecurityMode_Open, nn::wlan::SecurityMode_Open, 0, "" }, nn::wlan::Ssid(LocalMasterTestSSID), 100
        };

        WLANTEST_ASSERT_TRUE(localMaster.SetupLocalMaster(fncTestMasterBss, TestMatchInfo, sizeof(TestMatchInfo) / sizeof(TestMatchInfo[0]), true));
        WLANTEST_ASSERT_TRUE(localMaster.LocalMasterConnectWait(clientMacArray, WlanTestConnectTimeOut, 1) > 0);

        // 120秒間ActionFrame送信
        sysTick = nn::os::GetSystemTick();
        NN_LOG("             MASETER - SetActionFrameWithBeacon Start\n");

        // beacon frame type
        WLANTEST_EXPECT_TRUE(localMaster.LocalMakeActionFrame(putBuffer.get(), putSize, clientMacArray[0],
            localMaster.m_macAddress, EtherTypes[0], TestMatchInfo[0], nn::wlan::ActionFrameType_Beacon));
        result = nn::wlan::Local::SetActionFrameWithBeacon(putBuffer.get(), putSize);
        NN_LOG("             MASETER - SetActionFrameWithBeacon Result : 0x%08x - Time : %lldmsec\n", result.GetInnerValueForDebug(),
            (nn::os::GetSystemTick() - sysTick).ToTimeSpan().GetMilliSeconds());
        while ((nn::os::GetSystemTick() - sysTick).ToTimeSpan().GetMilliSeconds() <= Time120s)
        {
            // CLIENTからの接続が切られるまで送信し続ける
            if (nn::os::TimedWaitSystemEvent(&connectionEvent, nn::TimeSpan::FromMilliSeconds(Time1s)) == true)
            {
                if (localMaster.GetClientMacAddress(clientMacArray, nn::wlan::ConnectionState_Connected) == 0)
                {
                    break;
                }
            }
        }
        NN_LOG("             MASETER - SetActionFrameWithBeacon End\n");
    }

    // 削除した受信エントリのActionFrameTypeは削除処理以降受信しない CLIENT
    TEST_F(LocalClientFuncTest, L134_2)
    {
        nn::wlan::MacAddress macAddress = nn::wlan::MacAddress::CreateBroadcastMacAddress();
        const TestLocalConnectParams ConnectParam = {
            nn::wlan::Ssid(LocalMasterTestSSID), nn::wlan::MacAddress::CreateBroadcastMacAddress(), 11,
            { nn::wlan::SecurityMode_Open, nn::wlan::SecurityMode_Open, 0, "" }, true, nn::wlan::BeaconIndication_Enable, 10
        };

        nn::wlan::MacAddress outMacAddress;
        std::unique_ptr<uint8_t[]> getBuffer(new uint8_t[TestScanMaxSize]);
        LocalActionFrameData* pActionFrame = nullptr;
        size_t outSize;
        uint32_t outActionRxId;
        nn::Result result;

        WLANTEST_ASSERT_TRUE(getBuffer.get() != nullptr);

        WLANTEST_ASSERT_TRUE(localClient.SetupLocalClient(ConnectParam.ssid, ConnectParam.security, TestMatchInfo, sizeof(TestMatchInfo) / sizeof(TestMatchInfo[0]), false, true));

        pActionFrame = reinterpret_cast<LocalActionFrameData*>(getBuffer.get());

        AllScan();

        Connect(ConnectParam);

        // 設定したフレームタイプ比較
        bool isMatchFramType = false;
        WLANTEST_EXPECT_RESULT_SUCCESS(nn::wlan::Local::GetConnectionStatus(&connectionStatus));
        while (connectionStatus.state == nn::wlan::ConnectionState_Connected)
        {
            WLANTEST_ASSERT_TRUE(localClient.LocalCancelActionFrameStart(Time15s));
            WLANTEST_EXPECT_RESULT_SUCCESS(nn::wlan::Local::GetActionFrame(&outMacAddress, getBuffer.get(), TestScanMaxSize, &outSize, localClient.m_actionRxId));
            localClient.LocalCancelActionFrameStop();

            if (nn::wlan::ActionFrameType_Beacon == pActionFrame->frameType)
            {
                isMatchFramType = true;
                break;
            }

            WLANTEST_EXPECT_RESULT_SUCCESS(nn::wlan::Local::GetConnectionStatus(&connectionStatus));
        }
        WLANTEST_ASSERT_TRUE(isMatchFramType);

        outActionRxId = localClient.m_actionRxId;
        WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::DeleteSubtypeFromRxEntryForActionFrame(&outActionRxId, nn::wlan::ActionFrameType_Beacon));

        NN_LOG("             CLIENT - GetActionFrame Wait Start\n");
        WLANTEST_ASSERT_TRUE(localClient.LocalCancelActionFrameStart(Time15s));
        result = nn::wlan::Local::GetActionFrame(&outMacAddress, getBuffer.get(), TestScanMaxSize, &outSize, localClient.m_actionRxId);
        WLANTEST_EXPECT_TRUE(nn::wlan::ResultGetFrameCancelled::Includes(result));
        NN_LOG("             CLIENT - GetActionFrame Wait End\n");
        localClient.LocalCancelActionFrameStop();

        Disconnect();
    }

    // 無線がONになっている状態(IDLE)でActionFrameの受信 MASTER
    TEST_F(LocalMasterFuncTest, L135_1)
    {
        size_t putSize = ActionFrameBufferSize;
        std::unique_ptr<uint8_t[]> putBuffer(new uint8_t[putSize]);
        nn::wlan::MacAddress clientMacArray[ClientMaxConnect];
        WLANTEST_ASSERT_TRUE(putBuffer.get() != nullptr);

        const nn::wlan::MasterBssParameters fncTestMasterBss = {
            11, false, 30, true, nn::wlan::RateSetLegacy_11gMask, nn::wlan::RateSetLegacy_11gMask,
            { nn::wlan::SecurityMode_Open, nn::wlan::SecurityMode_Open, 0, "" }, nn::wlan::Ssid(LocalMasterTestSSID), 100
        };

        WLANTEST_ASSERT_TRUE(localMaster.SetupLocalMaster(fncTestMasterBss, TestMatchInfo, sizeof(TestMatchInfo) / sizeof(TestMatchInfo[0]), true));
        WLANTEST_ASSERT_TRUE(localMaster.LocalMasterConnectWait(clientMacArray, WlanTestConnectTimeOut, 1) > 0);

        // 15秒間ActionFrame送信
        sysTick = nn::os::GetSystemTick();
        NN_LOG("             MASETER - SetActionFrameWithBeacon Start\n");
        // beacon frame type
        WLANTEST_EXPECT_TRUE(localMaster.LocalMakeActionFrame(putBuffer.get(), putSize, clientMacArray[0],
            localMaster.m_macAddress, EtherTypes[0], TestMatchInfo[0], nn::wlan::ActionFrameType_Beacon));
        WLANTEST_EXPECT_RESULT_SUCCESS(nn::wlan::Local::SetActionFrameWithBeacon(putBuffer.get(), putSize));
        while ((nn::os::GetSystemTick() - sysTick).ToTimeSpan().GetMilliSeconds() <= Time120s)
        {
            // MASTERからの接続が切られるまで送信し続ける
            if (nn::os::TimedWaitSystemEvent(&connectionEvent, nn::TimeSpan::FromMilliSeconds(Time100msec)) == true)
            {
                if (localMaster.GetClientMacAddress(clientMacArray, nn::wlan::ConnectionState_Connected) == 0)
                {
                    NN_LOG("             MASETER - CLIENT DISCONNECTED\n");
                    break;
                }
            }
        }
        NN_LOG("             MASETER - SetActionFrameWithBeacon End(%llu)\n", (nn::os::GetSystemTick() - sysTick).ToTimeSpan().GetMilliSeconds());
    }

    // 無線がONになっている状態(IDLE)でActionFrameの受信 無線がONになっている状態(IDLE)でActionFrameの受信
    TEST_F(LocalClientFuncTest, L135_2)
    {
        nn::wlan::MacAddress outMacAddress;
        std::unique_ptr<uint8_t[]> getBuffer(new uint8_t[ActionFrameBufferSize]);
        size_t outSize;

        WLANTEST_ASSERT_TRUE(getBuffer.get() != nullptr);

        nn::wlan::MacAddress macAddress = nn::wlan::MacAddress::CreateBroadcastMacAddress();
        const TestLocalConnectParams ConnectParam = {
            nn::wlan::Ssid(LocalMasterTestSSID), nn::wlan::MacAddress::CreateBroadcastMacAddress(), 11,
            { nn::wlan::SecurityMode_Open, nn::wlan::SecurityMode_Open, 0, "" }, true, nn::wlan::BeaconIndication_Enable, 10
        };

        AllScan();

        WLANTEST_ASSERT_TRUE(localClient.SetupLocalClient(ConnectParam.ssid, ConnectParam.security, TestMatchInfo, sizeof(TestMatchInfo) / sizeof(TestMatchInfo[0]), true, true));

        Connect(ConnectParam);

        // 3回受信する
        for (int i = 0; i < 3; i++)
        {
            WLANTEST_ASSERT_TRUE(localClient.LocalCancelActionFrameStart(Time5s));
            WLANTEST_EXPECT_RESULT_SUCCESS(nn::wlan::Local::GetActionFrame(&outMacAddress, getBuffer.get(), ActionFrameBufferSize, &outSize, localClient.m_actionRxId));
            localClient.LocalCancelActionFrameStop();
        }

        Disconnect();
    }

    // スキャン処理中にActionFrameの受信 MASTER
    TEST_F(LocalMasterFuncTest, L136_1)
    {
        size_t putSize = ActionFrameBufferSize;
        std::unique_ptr<uint8_t[]> putBuffer(new uint8_t[putSize]);
        nn::wlan::MacAddress clientMacArray[ClientMaxConnect];
        WLANTEST_ASSERT_TRUE(putBuffer.get() != nullptr);

        const nn::wlan::MasterBssParameters fncTestMasterBss = {
            11, false, 30, true, nn::wlan::RateSetLegacy_11gMask, nn::wlan::RateSetLegacy_11gMask,
            { nn::wlan::SecurityMode_Open, nn::wlan::SecurityMode_Open, 0, "" }, nn::wlan::Ssid(LocalMasterTestSSID), 100
        };

        WLANTEST_ASSERT_TRUE(localMaster.SetupLocalMaster(fncTestMasterBss, TestMatchInfo, sizeof(TestMatchInfo) / sizeof(TestMatchInfo[0]), true));
        WLANTEST_ASSERT_TRUE(localMaster.LocalMasterConnectWait(clientMacArray, WlanTestConnectTimeOut, 1) > 0);

        // 15秒間ActionFrame送信
        sysTick = nn::os::GetSystemTick();
        NN_LOG("             MASETER - SetActionFrameWithBeacon Start\n");
        // beacon frame type
        WLANTEST_EXPECT_TRUE(localMaster.LocalMakeActionFrame(putBuffer.get(), putSize, clientMacArray[0],
            localMaster.m_macAddress, EtherTypes[0], TestMatchInfo[0], nn::wlan::ActionFrameType_Beacon));
        WLANTEST_EXPECT_RESULT_SUCCESS(nn::wlan::Local::SetActionFrameWithBeacon(putBuffer.get(), putSize));
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(Time15s));
        NN_LOG("             MASETER - SetActionFrameWithBeacon End\n");
    }

    // スキャン処理中にActionFrameの受信 CLIENT
    TEST_F(LocalClientFuncTest, L136_2)
    {
        nn::wlan::ScanIeMatchInfo info;

        const nn::wlan::ScanParameters StartScanTestParams = {
            nn::wlan::ScanType_Passive,{ 1, 6, 11 }, 3, 110, -1, &ssidList[0], 1, nn::wlan::MacAddress::CreateBroadcastMacAddress()
        };

        nn::wlan::MacAddress macAddress = nn::wlan::MacAddress::CreateBroadcastMacAddress();
        const TestLocalConnectParams ConnectParam = {
            nn::wlan::Ssid(LocalMasterTestSSID), nn::wlan::MacAddress::CreateBroadcastMacAddress(), 11,
            { nn::wlan::SecurityMode_Open, nn::wlan::SecurityMode_Open, 0, "" }, true, nn::wlan::BeaconIndication_Enable, 10
        };
        std::unique_ptr<uint8_t[]> pTestBuffer(new uint8_t[TestScanMaxSize]);
        WLANTEST_ASSERT_TRUE(pTestBuffer.get() != nullptr);

        AllScan();

        WLANTEST_ASSERT_TRUE(localClient.SetupLocalClient(ConnectParam.ssid, ConnectParam.security, TestMatchInfo, sizeof(TestMatchInfo) / sizeof(TestMatchInfo[0]), false, true));

        Connect(ConnectParam);

        WLANTEST_ASSERT_RESULT_SUCCESS(nn::os::CreateThread(&actionframeThread, ReceiveActionFrameThread, this, threadStack, threadStackSize, nn::os::DefaultThreadPriority));

        nn::os::StartThread(&actionframeThread);

        WLANTEST_EXPECT_RESULT_SUCCESS(nn::wlan::Local::StartScan(pTestBuffer.get(), TestScanMaxSize, StartScanTestParams, &info));

        nn::os::WaitThread(&actionframeThread);
        nn::os::DestroyThread(&actionframeThread);

        Disconnect();
    }

    // 接続処理中にActionFrameの受信
    TEST_F(LocalClientFuncTest, L137)
    {
        nn::wlan::MacAddress macAddress = nn::wlan::MacAddress::CreateBroadcastMacAddress();
        const TestLocalConnectParams ConnectParam = {
            nn::wlan::Ssid(LocalMasterTestSSID), nn::wlan::MacAddress::CreateBroadcastMacAddress(), 11,
            { nn::wlan::SecurityMode_Open, nn::wlan::SecurityMode_Open, 0, "" }, true, nn::wlan::BeaconIndication_Enable, 10
        };
        std::unique_ptr<uint8_t[]> pTestBuffer(new uint8_t[TestScanMaxSize]);
        WLANTEST_ASSERT_TRUE(pTestBuffer.get() != nullptr);

        AllScan();

        WLANTEST_ASSERT_TRUE(localClient.SetupLocalClient(ConnectParam.ssid, ConnectParam.security, TestMatchInfo, sizeof(TestMatchInfo) / sizeof(TestMatchInfo[0]), false, true));

        nn::os::InitializeEvent(&actionframeEvent, false, nn::os::EventClearMode::EventClearMode_ManualClear);
        WLANTEST_ASSERT_RESULT_SUCCESS(nn::os::CreateThread(&actionframeThread, ReceiveActionFrameThread, this, threadStack, threadStackSize, nn::os::DefaultThreadPriority));
        nn::os::StartThread(&actionframeThread);

        Connect(ConnectParam);
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(5000));

        WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::Disconnect(nn::wlan::LocalCommunicationMode_ClientSpectator, nullptr));
        nn::os::WaitThread(&actionframeThread);
        nn::os::DestroyThread(&actionframeThread);
        nn::os::FinalizeEvent(&actionframeEvent);
    }

    // 接続後にActionFrameの受信
    TEST_F(LocalClientFuncTest, L138)
    {
        nn::wlan::MacAddress macAddress = nn::wlan::MacAddress::CreateBroadcastMacAddress();
        const TestLocalConnectParams ConnectParam = {
            nn::wlan::Ssid(LocalMasterTestSSID), nn::wlan::MacAddress::CreateBroadcastMacAddress(), 1,
            { nn::wlan::SecurityMode_Open, nn::wlan::SecurityMode_Open, 0, "" }, true, nn::wlan::BeaconIndication_Enable, 10
        };
        // ジョイコン1セット（各2台）
        WLANTEST_ASSERT_TRUE(localClient.SetBtMode(LocalApiBtNode2));

        WLANTEST_ASSERT_TRUE(localClient.SetupLocalClient(ConnectParam.ssid, ConnectParam.security, TestMatchInfo, sizeof(TestMatchInfo) / sizeof(TestMatchInfo[0]), false, true));

        nn::os::InitializeEvent(&actionframeEvent, false, nn::os::EventClearMode::EventClearMode_ManualClear);
        WLANTEST_ASSERT_RESULT_SUCCESS(nn::os::CreateThread(&actionframeThread, ReceiveActionFrameThread, this, threadStack, threadStackSize, nn::os::DefaultThreadPriority));

        Connect(ConnectParam);
        nn::os::StartThread(&actionframeThread);

        nn::os::WaitThread(&actionframeThread);
        WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::Disconnect(nn::wlan::LocalCommunicationMode_ClientSpectator, nullptr));

        nn::os::DestroyThread(&actionframeThread);
        nn::os::FinalizeEvent(&actionframeEvent);
    }


    // Master動作の既接続スキャン中に複数個のActionFrameが受信 MASTER
    TEST_F(LocalMasterFuncTest, L139_1)
    {
        std::unique_ptr<uint8_t[]> putBuffer(new uint8_t[ActionFrameBufferSize]);
        std::unique_ptr<uint8_t[]> getBuffer(new uint8_t[ActionFrameBufferSize]);

        nn::wlan::MacAddress clientMacArray[ClientMaxConnect];
        char macStrArray[nn::wlan::MacAddress::MacStringSize];
        size_t putSize = ActionFrameBufferSize;
        nn::wlan::MacAddress outMacAddress;
        size_t outSize;

        WLANTEST_ASSERT_TRUE(putBuffer.get() != nullptr);
        WLANTEST_ASSERT_TRUE(getBuffer.get() != nullptr);

        const nn::wlan::MasterBssParameters fncTestMasterBss = {
            1, false, 30, true, nn::wlan::RateSetLegacy_11gMask, nn::wlan::RateSetLegacy_11gMask,
            { nn::wlan::SecurityMode_Open, nn::wlan::SecurityMode_Open, 0, "" }, nn::wlan::Ssid(LocalMasterTestSSID), 100
        };

        const nn::wlan::ScanParameters StartScanTestParams = {
            nn::wlan::ScanType_Active,{ 1, 6, 11 }, 3, 40, -1, &ssidList[0], 1, nn::wlan::MacAddress::CreateBroadcastMacAddress()
        };

        WLANTEST_ASSERT_TRUE(localMaster.SetupLocalMaster(fncTestMasterBss, TestMatchInfo, sizeof(TestMatchInfo) / sizeof(TestMatchInfo[0]), true));

        WLANTEST_ASSERT_TRUE(localMaster.LocalMasterConnectWait(clientMacArray, WlanTestConnectTimeOut, 1) > 0);
        WLANTEST_ASSERT_TRUE(localMaster.LocalMasterDisConnectWait(WlanTestConnectTimeOut) == 0);

        StartSyncScanThread(StartScanTestParams, 100);

        // beacon frame
        WLANTEST_EXPECT_TRUE(localMaster.LocalMakeActionFrame(putBuffer.get(), putSize, nn::wlan::MacAddress::CreateBroadcastMacAddress(),
            localMaster.m_macAddress, EtherTypes[0], TestMatchInfo[0], nn::wlan::ActionFrameType_Beacon));
        WLANTEST_EXPECT_RESULT_SUCCESS(nn::wlan::Local::SetActionFrameWithBeacon(putBuffer.get(), putSize));

        // 30秒間通信
        sysTick = nn::os::GetSystemTick();
        while ((nn::os::GetSystemTick() - sysTick).ToTimeSpan().GetMilliSeconds() <= Time30s)
        {
            actionWaitEvent.WaitSygnal(Time10s);
            WLANTEST_EXPECT_TRUE(localMaster.LocalCancelActionFrameStart(Time5s));
            WLANTEST_EXPECT_RESULT_SUCCESS(nn::wlan::Local::GetActionFrame(&outMacAddress, getBuffer.get(), TestScanMaxSize, &outSize, localMaster.m_actionRxId));
            NN_LOG("             L90_1 -> GetAction Frame From(%s) %lu Byte\n\n", outMacAddress.GetString(macStrArray), outSize);
            localMaster.LocalCancelActionFrameStop();
            actionWaitEvent.ClearSygnal();
            scanWaitEvent.SetSygnal();
        }

        // 3台全て終了するため5秒待機（定期送信ActionFrameのため待機だけで送信される）
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(Time5s));

        StopUnsyncScanThread();
    }

    // Master動作の既接続スキャン中に複数個のActionFrameが受信 MASTER2台目
    TEST_F(LocalMasterFuncTest, L139_2)
    {
        std::unique_ptr<uint8_t[]> putBuffer(new uint8_t[ActionFrameBufferSize]);
        std::unique_ptr<uint8_t[]> getBuffer(new uint8_t[ActionFrameBufferSize]);

        nn::wlan::MacAddress clientMacArray[ClientMaxConnect];
        char macStrArray[nn::wlan::MacAddress::MacStringSize];
        size_t putSize = ActionFrameBufferSize;
        nn::wlan::MacAddress outMacAddress;
        size_t outSize;

        WLANTEST_ASSERT_TRUE(putBuffer.get() != nullptr);
        WLANTEST_ASSERT_TRUE(getBuffer.get() != nullptr);

        const nn::wlan::MasterBssParameters fncTestMasterBss = {
            1, false, 30, true, nn::wlan::RateSetLegacy_11gMask, nn::wlan::RateSetLegacy_11gMask,
            { nn::wlan::SecurityMode_Open, nn::wlan::SecurityMode_Open, 0, "" }, nn::wlan::Ssid(LocalMasterTest2SSID), 100
        };

        const nn::wlan::ScanParameters StartScanTestParams = {
            nn::wlan::ScanType_Active,{ 1, 6, 11 }, 3, 40, -1, &ssidList[0], 1, nn::wlan::MacAddress::CreateBroadcastMacAddress()
        };

        WLANTEST_ASSERT_TRUE(localMaster.SetupLocalMaster(fncTestMasterBss, TestMatchInfo, sizeof(TestMatchInfo) / sizeof(TestMatchInfo[0]), true));

        WLANTEST_ASSERT_TRUE(localMaster.LocalMasterConnectWait(clientMacArray, WlanTestConnectTimeOut, 1) > 0);

        StartSyncScanThread(StartScanTestParams, 100);

        // beacon frame
        WLANTEST_EXPECT_TRUE(localMaster.LocalMakeActionFrame(putBuffer.get(), putSize, nn::wlan::MacAddress::CreateBroadcastMacAddress(),
            localMaster.m_macAddress, EtherTypes[0], TestMatchInfo[0], nn::wlan::ActionFrameType_Beacon));
        WLANTEST_EXPECT_RESULT_SUCCESS(nn::wlan::Local::SetActionFrameWithBeacon(putBuffer.get(), putSize));

        // 30秒間通信
        sysTick = nn::os::GetSystemTick();
        while ((nn::os::GetSystemTick() - sysTick).ToTimeSpan().GetMilliSeconds() <= Time30s)
        {
            actionWaitEvent.WaitSygnal(Time10s);
            WLANTEST_EXPECT_TRUE(localMaster.LocalCancelActionFrameStart(Time5s));
            WLANTEST_EXPECT_RESULT_SUCCESS(nn::wlan::Local::GetActionFrame(&outMacAddress, getBuffer.get(), TestScanMaxSize, &outSize, localMaster.m_actionRxId));
            NN_LOG("             L90_2 -> GetAction Frame From(%s) %lu Byte\n\n", outMacAddress.GetString(macStrArray), outSize);
            localMaster.LocalCancelActionFrameStop();
            actionWaitEvent.ClearSygnal();
            scanWaitEvent.SetSygnal();
        }

        // 3台全て終了するため5秒待機（定期送信ActionFrameのため待機だけで送信される）
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(Time5s));

        StopUnsyncScanThread();
    }

    // Master動作の既接続スキャン中に複数個のActionFrameが受信 CLIENT
    TEST_F(LocalClientFuncTest, L139_3)
    {
        std::unique_ptr<uint8_t[]> getBuffer(new uint8_t[ActionFrameBufferSize]);
        nn::wlan::MacAddress outMacAddress;
        char macStrArray[nn::wlan::MacAddress::MacStringSize];
        size_t outSize;

        WLANTEST_ASSERT_TRUE(getBuffer.get() != nullptr);

        const TestLocalConnectParams ConnectParam[] = {
            { nn::wlan::Ssid(LocalMasterTestSSID), nn::wlan::MacAddress::CreateBroadcastMacAddress(), 1,
            { nn::wlan::SecurityMode_Open, nn::wlan::SecurityMode_Open, 0, "" }, true, nn::wlan::BeaconIndication_Enable, 10 },

            { nn::wlan::Ssid(LocalMasterTest2SSID), nn::wlan::MacAddress::CreateBroadcastMacAddress(), 1,
            { nn::wlan::SecurityMode_Open, nn::wlan::SecurityMode_Open, 0, "" }, true, nn::wlan::BeaconIndication_Enable, 10 },
        };

        WLANTEST_ASSERT_TRUE(localClient.SetupLocalClient(ConnectParam[0].ssid, ConnectParam[0].security, TestMatchInfo, sizeof(TestMatchInfo) / sizeof(TestMatchInfo[0]), true, true));
        Connect(ConnectParam[0]);
        Disconnect();
        Connect(ConnectParam[1]);

        while (NN_STATIC_CONDITION(1))
        {
            // MASTERから切断されるまでActionFrame受信
            if (nn::os::TimedWaitSystemEvent(&connectionEvent, nn::TimeSpan::FromMilliSeconds(Time300msec)) == true)
            {
                WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::GetConnectionStatus(&connectionStatus));
                if (connectionStatus.state != nn::wlan::ConnectionState_Connected)
                {
                    break;
                }
            }
            WLANTEST_EXPECT_TRUE(localClient.LocalCancelActionFrameStart(Time5s));
            WLANTEST_EXPECT_RESULT_SUCCESS(nn::wlan::Local::GetActionFrame(&outMacAddress, getBuffer.get(), TestScanMaxSize, &outSize, localClient.m_actionRxId));
            NN_LOG("             L90_3 -> GetAction Frame From(%s) %lu Byte\n\n", outMacAddress.GetString(macStrArray), outSize);
            localClient.LocalCancelActionFrameStop();
        }

        StopUnsyncScanThread();
        Disconnect();
    }

    // Client動作の既接続スキャン中に複数個のActionFrameが受信 MASTER
    TEST_F(LocalMasterFuncTest, L140_1)
    {
        size_t putSize = ActionFrameBufferSize;
        std::unique_ptr<uint8_t[]> putBuffer(new uint8_t[putSize]);
        nn::wlan::MacAddress clientMacArray[ClientMaxConnect];
        nn::wlan::MacAddress clientMacAddress;

        WLANTEST_ASSERT_TRUE(putBuffer.get() != nullptr);

        const nn::wlan::MasterBssParameters fncTestMasterBss = {
            6, false, 30, true, nn::wlan::RateSetLegacy_11gMask, nn::wlan::RateSetLegacy_11gMask,
            { nn::wlan::SecurityMode_StaticAes, nn::wlan::SecurityMode_StaticAes, 0, LocalMasterAes16Key }, nn::wlan::Ssid(LocalMasterTestSSID), 100
        };

        WLANTEST_ASSERT_TRUE(localMaster.SetupLocalMaster(fncTestMasterBss, TestMatchInfo, sizeof(TestMatchInfo) / sizeof(TestMatchInfo[0]), true));
        WLANTEST_EXPECT_RESULT_SUCCESS(nn::wlan::Local::AddSubtypeToRxEntryForActionFrame(localMaster.m_actionRxId, nn::wlan::ActionFrameType_Local));

        // MASTER CLIENT接続待機
        WLANTEST_ASSERT_TRUE(localMaster.LocalMasterConnectWait(clientMacArray, WlanTestConnectTimeOut, 1) > 0);

        sysTick = nn::os::GetSystemTick();
        // 60秒間通信
        WLANTEST_EXPECT_TRUE(localMaster.LocalMakeActionFrame(putBuffer.get(), putSize, clientMacArray[0],
            localMaster.m_macAddress, EtherTypes[0], TestMatchInfo[0], nn::wlan::ActionFrameType_Beacon));
        WLANTEST_EXPECT_RESULT_SUCCESS(nn::wlan::Local::SetActionFrameWithBeacon(putBuffer.get(), putSize));
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(Time60s));

        // MASTER受信スレッド停止
        localMaster.LocalReceiveFrameStop();

        localMaster.LocalGetConnectInfo(&connectsInfo, GetLocalFrameType::FrameType_ActionFrame);
        for (int32_t i = 0; i < connectsInfo.connectCount; i++)
        {
            WLANTEST_EXPECT_TRUE(connectsInfo.facing[i].isConnected);
            WLANTEST_EXPECT_TRUE(connectsInfo.facing[i].getFrameCount > 0);
            WLANTEST_EXPECT_TRUE(connectsInfo.facing[i].lostFrameCount == 0);
        }

    }

    // Client動作の既接続スキャン中に複数個のActionFrameが受信 CLIENT
    TEST_F(LocalClientFuncTest, L140_2)
    {
        size_t gutSize = ActionFrameBufferSize * 2;
        std::unique_ptr<uint8_t[]> getBuffer(new uint8_t[gutSize]);
        nn::wlan::MacAddress clientMacAddress;
        nn::wlan::ConnectionStatus clientStatus;
        nn::wlan::MacAddress outMacAddress;
        size_t outSize;
        nn::Result result;

        WLANTEST_ASSERT_TRUE(getBuffer.get() != nullptr);

        const TestLocalConnectParams ConnectParam = {
            nn::wlan::Ssid(LocalMasterTestSSID), nn::wlan::MacAddress::CreateBroadcastMacAddress(), 6,
            { nn::wlan::SecurityMode_StaticAes, nn::wlan::SecurityMode_StaticAes, 0, LocalMasterAes16Key }, true, nn::wlan::BeaconIndication_Enable, 10
        };

        WLANTEST_ASSERT_TRUE(localClient.SetupLocalClient(ConnectParam.ssid, ConnectParam.security, TestMatchInfo, sizeof(TestMatchInfo) / sizeof(TestMatchInfo[0]), true, true));

        const nn::wlan::ScanParameters StartScanTestParams = {
            nn::wlan::ScanType_Active,{ 1, 6, 11 }, 3, 40, -1, &ssidList[0], 1, nn::wlan::MacAddress::CreateBroadcastMacAddress()
        };

        Connect(ConnectParam);

        StartUnsyncScanThread(StartScanTestParams, 100);

        WLANTEST_EXPECT_RESULT_SUCCESS(nn::wlan::Local::GetMacAddress(&clientMacAddress));
        WLANTEST_EXPECT_RESULT_SUCCESS(nn::wlan::Local::GetConnectionStatus(&connectionStatus));

        sysTick = nn::os::GetSystemTick();
        while (NN_STATIC_CONDITION(1))
        {
            if (nn::os::TimedWaitSystemEvent(&localClient.m_connectEvent, nn::TimeSpan::FromMilliSeconds(Interval_16m)) == true)
            {
                WLANTEST_EXPECT_RESULT_SUCCESS(nn::wlan::Local::GetConnectionStatus(&clientStatus));
                // MASTERからの接続が切られるまで受信
                if (clientStatus.state != nn::wlan::ConnectionState_Connected)
                {
                    break;
                }
            }

            WLANTEST_ASSERT_TRUE(localClient.LocalCancelActionFrameStart(Time5s));
            result = nn::wlan::Local::GetActionFrame(&outMacAddress, getBuffer.get(), gutSize, &outSize, localClient.m_actionRxId);
            if (result.IsSuccess() != true)
            {
                WLANTEST_EXPECT_RESULT_SUCCESS(nn::wlan::Local::GetConnectionStatus(&clientStatus));
                // MASTERからの接続が切られていたら評価せずに終了
                if (clientStatus.state != nn::wlan::ConnectionState_Connected)
                {
                    NN_LOG("             Master Disconnected\n");
                    break;
                }
                // キャンセル応答の場合はエラーとしない
                if (false == nn::wlan::ResultGetFrameCancelled::Includes(result))
                {
                    WLANTEST_EXPECT_RESULT_SUCCESS(result);
                }
            }
            localClient.LocalCancelActionFrameStop();
        }

        // CLIENT受信スレッド停止
        localClient.LocalReceiveFrameStop();

        StopUnsyncScanThread();
        // 1秒待機
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(Time1s));

        Disconnect();
    }

    // 切断後にActionFrameの受信 MASTER
    TEST_F(LocalMasterFuncTest, L141_1)
    {
        std::unique_ptr<uint8_t[]> putBuffer(new uint8_t[ActionFrameBufferSize]);
        nn::wlan::MacAddress clientMacArray[ClientMaxConnect];
        nn::wlan::MacAddress clientMacAddress;

        WLANTEST_ASSERT_TRUE(putBuffer.get() != nullptr);

        const nn::wlan::MasterBssParameters fncTestMasterBss = {
            1, false, 30, true, nn::wlan::RateSetLegacy_11gMask, nn::wlan::RateSetLegacy_11gMask,
            { nn::wlan::SecurityMode_Open, nn::wlan::SecurityMode_Open, 0, "" }, nn::wlan::Ssid(LocalMasterTestSSID), 100
        };

        WLANTEST_ASSERT_TRUE(localMaster.SetupLocalMaster(fncTestMasterBss, TestMatchInfo, sizeof(TestMatchInfo) / sizeof(TestMatchInfo[0]), true));
        WLANTEST_EXPECT_RESULT_SUCCESS(nn::wlan::Local::AddSubtypeToRxEntryForActionFrame(localMaster.m_actionRxId, nn::wlan::ActionFrameType_Local));

        // MASTER CLIENT接続待機
        WLANTEST_ASSERT_TRUE(localMaster.LocalMasterConnectWait(clientMacArray, WlanTestConnectTimeOut, 1) > 0);
        // MASTER CLIENT切断待機
        WLANTEST_ASSERT_TRUE(localMaster.LocalMasterDisConnectWait(WlanTestConnectTimeOut) == 0);

        // 切断後のActionFrame送信
        WLANTEST_EXPECT_TRUE(localMaster.LocalMakeActionFrame(putBuffer.get(), ActionFrameBufferSize, clientMacArray[0],
            localMaster.m_macAddress, EtherTypes[0], TestMatchInfo[0], nn::wlan::ActionFrameType_Beacon));
        WLANTEST_EXPECT_RESULT_SUCCESS(nn::wlan::Local::SetActionFrameWithBeacon(putBuffer.get(), ActionFrameBufferSize));

        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(Time10s));
    }

    // 切断後にActionFrameの受信 CLIENT
    TEST_F(LocalClientFuncTest, L141_2)
    {
        nn::wlan::MacAddress macAddress = nn::wlan::MacAddress::CreateBroadcastMacAddress();
        const TestLocalConnectParams ConnectParam = {
            nn::wlan::Ssid(LocalMasterTestSSID), nn::wlan::MacAddress::CreateBroadcastMacAddress(), 1,
            { nn::wlan::SecurityMode_Open, nn::wlan::SecurityMode_Open, 0, "" }, true, nn::wlan::BeaconIndication_Enable, 10
        };
        std::unique_ptr<uint8_t[]> pTestBuffer(new uint8_t[TestScanMaxSize]);
        WLANTEST_ASSERT_TRUE(pTestBuffer.get() != nullptr);

        WLANTEST_ASSERT_TRUE(localClient.SetupLocalClient(ConnectParam.ssid, ConnectParam.security, TestMatchInfo, sizeof(TestMatchInfo) / sizeof(TestMatchInfo[0]), false, true));

        nn::os::InitializeEvent(&actionframeEvent, false, nn::os::EventClearMode::EventClearMode_ManualClear);
        WLANTEST_ASSERT_RESULT_SUCCESS(nn::os::CreateThread(&actionframeThread, ReceiveActionFrameThread, this, threadStack, threadStackSize, nn::os::DefaultThreadPriority));

        Connect(ConnectParam);
        WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::Disconnect(nn::wlan::LocalCommunicationMode_ClientSpectator, nullptr));

        nn::os::StartThread(&actionframeThread);
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(5000));

        nn::os::WaitThread(&actionframeThread);
        nn::os::DestroyThread(&actionframeThread);
        nn::os::FinalizeEvent(&actionframeEvent);
    }

    TEST_F(LocalMasterFuncTest, Sdev4_2GHz_Master)
    {
        const nn::wlan::MasterBssParameters fncTestMasterBss = {
            11, false, 30, true, nn::wlan::RateSetLegacy_11gMask, nn::wlan::RateSetLegacy_11gMask,
            { nn::wlan::SecurityMode_StaticAes, nn::wlan::SecurityMode_StaticAes, 0, LocalMasterAes16Key }, nn::wlan::Ssid("D"), 10
        };

        // ジョイコン2セット（各4台）
        WLANTEST_ASSERT_TRUE(localMaster.SetBtMode(LocalApiBtNode4));

        WLANTEST_ASSERT_TRUE(localMaster.SetupLocalMaster(fncTestMasterBss, TestMatchInfo, sizeof(TestMatchInfo) / sizeof(TestMatchInfo[0])));
        // MASTER CLIENT接続待機
        WLANTEST_EXPECT_TRUE(SetActionFrameWithBeaconTestWait(Client2SetBt));
        localMaster.LocalMasterDisConnectWait(Time60s);
    }

    TEST_F(LocalClientFuncTest, Sdev4_2GHz_Client)
    {
        nn::wlan::ConnectionStatus clientStatus;
        LocalClientConnectParameter ConnectParam = {
            nn::wlan::Ssid("D"), nn::wlan::MacAddress::CreateBroadcastMacAddress(), 11,
            { nn::wlan::SecurityMode_StaticAes, nn::wlan::SecurityMode_StaticAes, 0, LocalMasterAes16Key }, true, nn::wlan::BeaconIndication_Enable, 10
        };

        // ジョイコン2セット（各4台）
        WLANTEST_ASSERT_TRUE(localClient.SetBtMode(LocalApiBtNode4));
        WLANTEST_ASSERT_TRUE(localClient.SetupLocalClient(ConnectParam.ssid, ConnectParam.security, TestMatchInfo, sizeof(TestMatchInfo) / sizeof(TestMatchInfo[0]), true, true));
        WLANTEST_ASSERT_TRUE(localClient.LocalClientConnect(ConnectParam, static_cast<int64_t>(Time60s)));

        // MASTERからの接続が切られるまで待機
        while (NN_STATIC_CONDITION(1))
        {
            if (nn::os::TimedWaitSystemEvent(&connectionEvent, nn::TimeSpan::FromMilliSeconds(Time1s)) == true)
            {
                WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::GetConnectionStatus(&clientStatus));
                if (clientStatus.state != nn::wlan::ConnectionState_Connected)
                {
                    break;
                }
            }
        }
        localClient.LocalClientDisconnect(Time60s);
    }

    TEST_F(LocalClientFuncTest, Sdev4_2GHzScanConnect_Client)
    {
        TestScanConnectParam actInfo;
        int32_t resultArray[TestScanResultType::Type_End] = {0, 0, 0, 0};

        LocalClientConnectParameter ConnectParam = {
            nn::wlan::Ssid("D"), nn::wlan::MacAddress::CreateBroadcastMacAddress(), 11,
            { nn::wlan::SecurityMode_StaticAes, nn::wlan::SecurityMode_StaticAes, 0, LocalMasterAes16Key }, true, nn::wlan::BeaconIndication_Enable, 10
        };
        std::unique_ptr<uint8_t[]> pTestBuffer(new uint8_t[TestScanMaxSize]);
        WLANTEST_ASSERT_TRUE(pTestBuffer.get() != nullptr);

        // ジョイコン2セット（各4台）
        WLANTEST_ASSERT_TRUE(localClient.SetBtMode(LocalApiBtNode4));
        WLANTEST_ASSERT_TRUE(localClient.SetupLocalClient(ConnectParam.ssid, ConnectParam.security, TestMatchInfo, sizeof(TestMatchInfo) / sizeof(TestMatchInfo[0]), true, true));

        if (g_SetScanTime != true)
        {
            WLANTEST_EXPECT_TRUE(ScanActionFrameStart(actInfo, localClient.m_actionRxId));
        }
        else
        {
            WLANTEST_EXPECT_TRUE(ScanActionFrameStart(actInfo, localClient.m_actionRxId, g_ScanTime));
        }
        WLANTEST_EXPECT_TRUE(ScanActionConnectRate(resultArray, actInfo, ConnectParam, 500));
        ScanActionFrameStop(actInfo, ConnectParam);

        // Scam110msec合格基準判定
        if (g_ScanTime == WlanTestLdnScanTime)
        {
            // PASS 400以上
            WLANTEST_EXPECT_TRUE(resultArray[Type_ConnectPass] >= 400);
            // SCAN API NG 50以下
            WLANTEST_EXPECT_TRUE(resultArray[Type_ScanApiFail] <= 50);
            // SCAN ACTIONFRAME NG 50以下
            WLANTEST_EXPECT_TRUE(resultArray[Type_ScanActionFrameFail] <= 50);
            // CONNECT NG 4以下
            WLANTEST_EXPECT_TRUE(resultArray[Type_ConnectFail] <= 4);
        }

        NN_LOG("\n             *** SCAN RESULT INFO ***\n");
        NN_LOG("                 SDEV COUNT          : 4\n");
        NN_LOG("                 CHANNEL             : %d\n\n", ConnectParam.channel);
        NN_LOG("                 PASS                : %d\n", resultArray[Type_ConnectPass]);
        NN_LOG("                 SCAN API NG         : %d\n", resultArray[Type_ScanApiFail]);
        NN_LOG("                 SCAN ACTIONFRAME NG : %d\n", resultArray[Type_ScanActionFrameFail]);
        NN_LOG("                 CONNECT NG          : %d\n", resultArray[Type_ConnectFail]);
        NN_LOG("             *** SCAN RESULT INFO ***\n\n");
    }

    TEST_F(LocalMasterFuncTest, Sdev4_5GHz_Master)
    {
        const nn::wlan::MasterBssParameters fncTestMasterBss = {
            44, false, 30, true, nn::wlan::RateSetLegacy_11gMask, nn::wlan::RateSetLegacy_11gMask,
            { nn::wlan::SecurityMode_StaticAes, nn::wlan::SecurityMode_StaticAes, 0, LocalMasterAes16Key }, nn::wlan::Ssid("E"), 10
        };

        // ジョイコン2セット（各4台）
        WLANTEST_ASSERT_TRUE(localMaster.SetBtMode(LocalApiBtNode4));
        WLANTEST_ASSERT_TRUE(localMaster.SetupLocalMaster(fncTestMasterBss, TestMatchInfo, sizeof(TestMatchInfo) / sizeof(TestMatchInfo[0])));

        // MASTER CLIENT接続待機
        WLANTEST_EXPECT_TRUE(SetActionFrameWithBeaconTestWait(Client2SetBt));
        localMaster.LocalMasterDisConnectWait(Time60s);
    }

    TEST_F(LocalClientFuncTest, Sdev4_5GHz_Client)
    {
        nn::wlan::ConnectionStatus clientStatus;
        LocalClientConnectParameter ConnectParam = {
            nn::wlan::Ssid("E"), nn::wlan::MacAddress::CreateBroadcastMacAddress(), 44,
            { nn::wlan::SecurityMode_StaticAes, nn::wlan::SecurityMode_StaticAes, 0, LocalMasterAes16Key }, true, nn::wlan::BeaconIndication_Enable, 10
        };

        // ジョイコン2セット（各4台）
        WLANTEST_ASSERT_TRUE(localClient.SetBtMode(LocalApiBtNode4));
        WLANTEST_ASSERT_TRUE(localClient.SetupLocalClient(ConnectParam.ssid, ConnectParam.security, TestMatchInfo, sizeof(TestMatchInfo) / sizeof(TestMatchInfo[0]), true, true));
        WLANTEST_ASSERT_TRUE(localClient.LocalClientConnect(ConnectParam, static_cast<int64_t>(Time60s)));

        // MASTERからの接続が切られるまで待機
        while (NN_STATIC_CONDITION(1))
        {
            if (nn::os::TimedWaitSystemEvent(&connectionEvent, nn::TimeSpan::FromMilliSeconds(Time1s)) == true)
            {
                WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::GetConnectionStatus(&clientStatus));
                if (clientStatus.state != nn::wlan::ConnectionState_Connected)
                {
                    break;
                }
            }
        }
        localClient.LocalClientDisconnect(Time60s);
    }

    TEST_F(LocalClientFuncTest, Sdev4_5GHzScanConnect_Client)
    {
        TestScanConnectParam actInfo;
        int32_t resultArray[TestScanResultType::Type_End] = { 0, 0, 0, 0 };

        LocalClientConnectParameter ConnectParam = {
            nn::wlan::Ssid("E"), nn::wlan::MacAddress::CreateBroadcastMacAddress(), 44,
            { nn::wlan::SecurityMode_StaticAes, nn::wlan::SecurityMode_StaticAes, 0, LocalMasterAes16Key }, true, nn::wlan::BeaconIndication_Enable, 10
        };

        // ジョイコン2セット（各4台）
        WLANTEST_ASSERT_TRUE(localClient.SetBtMode(LocalApiBtNode4));
        WLANTEST_ASSERT_TRUE(localClient.SetupLocalClient(ConnectParam.ssid, ConnectParam.security, TestMatchInfo, sizeof(TestMatchInfo) / sizeof(TestMatchInfo[0]), true, true));

        if (g_SetScanTime != true)
        {
            WLANTEST_EXPECT_TRUE(ScanActionFrameStart(actInfo, localClient.m_actionRxId));
        }
        else
        {
            WLANTEST_EXPECT_TRUE(ScanActionFrameStart(actInfo, localClient.m_actionRxId, g_ScanTime));
        }
        WLANTEST_EXPECT_TRUE(ScanActionConnectRate(resultArray, actInfo, ConnectParam, 500));
        ScanActionFrameStop(actInfo, ConnectParam);

        // Scam110msec合格基準判定
        if (g_ScanTime == WlanTestLdnScanTime)
        {
            // PASS 400以上
            WLANTEST_EXPECT_TRUE(resultArray[Type_ConnectPass] >= 400);
            // SCAN API NG 50以下
            WLANTEST_EXPECT_TRUE(resultArray[Type_ScanApiFail] <= 50);
            // SCAN ACTIONFRAME NG 50以下
            WLANTEST_EXPECT_TRUE(resultArray[Type_ScanActionFrameFail] <= 50);
            // CONNECT NG 4以下
            WLANTEST_EXPECT_TRUE(resultArray[Type_ConnectFail] <= 4);
        }

        NN_LOG("\n             *** SCAN RESULT INFO ***\n");
        NN_LOG("                 SDEV COUNT          : 4\n");
        NN_LOG("                 CHANNEL             : %d\n\n", ConnectParam.channel);
        NN_LOG("                 PASS                : %d\n", resultArray[Type_ConnectPass]);
        NN_LOG("                 SCAN API NG         : %d\n", resultArray[Type_ScanApiFail]);
        NN_LOG("                 SCAN ACTIONFRAME NG : %d\n", resultArray[Type_ScanActionFrameFail]);
        NN_LOG("                 CONNECT NG          : %d\n", resultArray[Type_ConnectFail]);
        NN_LOG("             *** SCAN RESULT INFO ***\n\n");
    }

    TEST_F(LocalMasterFuncTest, Sdev8_2GHz_Master)
    {
        const nn::wlan::MasterBssParameters fncTestMasterBss = {
            11, false, 30, true, nn::wlan::RateSetLegacy_11gMask, nn::wlan::RateSetLegacy_11gMask,
            { nn::wlan::SecurityMode_StaticAes, nn::wlan::SecurityMode_StaticAes, 0, LocalMasterAes16Key }, nn::wlan::Ssid("F"), 10
        };

        // ジョイコン1セット（各2台）
        WLANTEST_ASSERT_TRUE(localMaster.SetBtMode(LocalApiBtNode2));
        WLANTEST_ASSERT_TRUE(localMaster.SetupLocalMaster(fncTestMasterBss, TestMatchInfo, sizeof(TestMatchInfo) / sizeof(TestMatchInfo[0])));

        // MASTER CLIENT接続待機
        WLANTEST_EXPECT_TRUE(SetActionFrameWithBeaconTestWait(Client1SetBt));
        localMaster.LocalMasterDisConnectWait(Time60s);
    }

    TEST_F(LocalClientFuncTest, Sdev8_2GHz_Client)
    {
        nn::wlan::ConnectionStatus clientStatus;
        LocalClientConnectParameter ConnectParam = {
            nn::wlan::Ssid("F"), nn::wlan::MacAddress::CreateBroadcastMacAddress(), 11,
            { nn::wlan::SecurityMode_StaticAes, nn::wlan::SecurityMode_StaticAes, 0, LocalMasterAes16Key }, true, nn::wlan::BeaconIndication_Enable, 10
        };

        // ジョイコン1セット（各2台）
        WLANTEST_ASSERT_TRUE(localClient.SetBtMode(LocalApiBtNode2));
        WLANTEST_ASSERT_TRUE(localClient.SetupLocalClient(ConnectParam.ssid, ConnectParam.security, TestMatchInfo, sizeof(TestMatchInfo) / sizeof(TestMatchInfo[0]), true, true));
        WLANTEST_ASSERT_TRUE(localClient.LocalClientConnect(ConnectParam, static_cast<int64_t>(Time60s)));

        // MASTERからの接続が切られるまで待機
        while (NN_STATIC_CONDITION(1))
        {
            if (nn::os::TimedWaitSystemEvent(&connectionEvent, nn::TimeSpan::FromMilliSeconds(Time1s)) == true)
            {
                WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::GetConnectionStatus(&clientStatus));
                if (clientStatus.state != nn::wlan::ConnectionState_Connected)
                {
                    break;
                }
            }
        }
        localClient.LocalClientDisconnect(Time60s);
    }

    TEST_F(LocalClientFuncTest, Sdev8_2GHzScanConnect_Client)
    {
        TestScanConnectParam actInfo;
        int32_t resultArray[TestScanResultType::Type_End] = { 0, 0, 0, 0 };

        LocalClientConnectParameter ConnectParam = {
            nn::wlan::Ssid("F"), nn::wlan::MacAddress::CreateBroadcastMacAddress(), 11,
            { nn::wlan::SecurityMode_StaticAes, nn::wlan::SecurityMode_StaticAes, 0, LocalMasterAes16Key }, true, nn::wlan::BeaconIndication_Enable, 10
        };
        std::unique_ptr<uint8_t[]> pTestBuffer(new uint8_t[TestScanMaxSize]);
        WLANTEST_ASSERT_TRUE(pTestBuffer.get() != nullptr);

        // ジョイコン1セット（各2台）
        WLANTEST_ASSERT_TRUE(localClient.SetBtMode(LocalApiBtNode2));
        WLANTEST_ASSERT_TRUE(localClient.SetupLocalClient(ConnectParam.ssid, ConnectParam.security, TestMatchInfo, sizeof(TestMatchInfo) / sizeof(TestMatchInfo[0]), true, true));

        if (g_SetScanTime != true)
        {
            WLANTEST_EXPECT_TRUE(ScanActionFrameStart(actInfo, localClient.m_actionRxId));
        }
        else
        {
            WLANTEST_EXPECT_TRUE(ScanActionFrameStart(actInfo, localClient.m_actionRxId, g_ScanTime));
        }
        WLANTEST_EXPECT_TRUE(ScanActionConnectRate(resultArray, actInfo, ConnectParam, 500));
        ScanActionFrameStop(actInfo, ConnectParam);

        // Scam110msec合格基準判定
        if (g_ScanTime == WlanTestLdnScanTime)
        {
            // PASS 400以上
            WLANTEST_EXPECT_TRUE(resultArray[Type_ConnectPass] >= 400);
            // SCAN API NG 50以下
            WLANTEST_EXPECT_TRUE(resultArray[Type_ScanApiFail] <= 50);
            // SCAN ACTIONFRAME NG 50以下
            WLANTEST_EXPECT_TRUE(resultArray[Type_ScanActionFrameFail] <= 50);
            // CONNECT NG 4以下
            WLANTEST_EXPECT_TRUE(resultArray[Type_ConnectFail] <= 4);
        }

        NN_LOG("\n             *** SCAN RESULT INFO ***\n");
        NN_LOG("                 SDEV COUNT          : 8\n");
        NN_LOG("                 CHANNEL             : %d\n\n", ConnectParam.channel);
        NN_LOG("                 PASS                : %d\n", resultArray[Type_ConnectPass]);
        NN_LOG("                 SCAN API NG         : %d\n", resultArray[Type_ScanApiFail]);
        NN_LOG("                 SCAN ACTIONFRAME NG : %d\n", resultArray[Type_ScanActionFrameFail]);
        NN_LOG("                 CONNECT NG          : %d\n", resultArray[Type_ConnectFail]);
        NN_LOG("             *** SCAN RESULT INFO ***\n\n");
    }

    TEST_F(LocalMasterFuncTest, Sdev8_5GHz_Master)
    {
        const nn::wlan::MasterBssParameters fncTestMasterBss = {
            44, false, 30, true, nn::wlan::RateSetLegacy_11gMask, nn::wlan::RateSetLegacy_11gMask,
            { nn::wlan::SecurityMode_StaticAes, nn::wlan::SecurityMode_StaticAes, 0, LocalMasterAes16Key }, nn::wlan::Ssid("G"), 10
        };

        // ジョイコン1セット（各2台）
        WLANTEST_ASSERT_TRUE(localMaster.SetBtMode(LocalApiBtNode2));

        WLANTEST_ASSERT_TRUE(localMaster.SetupLocalMaster(fncTestMasterBss, TestMatchInfo, sizeof(TestMatchInfo) / sizeof(TestMatchInfo[0])));

        // MASTER CLIENT接続待機
        WLANTEST_EXPECT_TRUE(SetActionFrameWithBeaconTestWait(Client1SetBt));
        localMaster.LocalMasterDisConnectWait(Time60s);
    }

    TEST_F(LocalClientFuncTest, Sdev8_5GHz_Client)
    {
        nn::wlan::ConnectionStatus clientStatus;
        LocalClientConnectParameter ConnectParam = {
            nn::wlan::Ssid("G"), nn::wlan::MacAddress::CreateBroadcastMacAddress(), 44,
            { nn::wlan::SecurityMode_StaticAes, nn::wlan::SecurityMode_StaticAes, 0, LocalMasterAes16Key }, true, nn::wlan::BeaconIndication_Enable, 10
        };

        // ジョイコン1セット（各2台）
        WLANTEST_ASSERT_TRUE(localClient.SetBtMode(LocalApiBtNode2));
        WLANTEST_ASSERT_TRUE(localClient.SetupLocalClient(ConnectParam.ssid, ConnectParam.security, TestMatchInfo, sizeof(TestMatchInfo) / sizeof(TestMatchInfo[0]), true, true));
        WLANTEST_ASSERT_TRUE(localClient.LocalClientConnect(ConnectParam, static_cast<int64_t>(Time60s)));

        // MASTERからの接続が切られるまで待機
        while (NN_STATIC_CONDITION(1))
        {
            if (nn::os::TimedWaitSystemEvent(&connectionEvent, nn::TimeSpan::FromMilliSeconds(Time1s)) == true)
            {
                WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::GetConnectionStatus(&clientStatus));
                if (clientStatus.state != nn::wlan::ConnectionState_Connected)
                {
                    break;
                }
            }
        }
        localClient.LocalClientDisconnect(Time60s);
    }

    TEST_F(LocalClientFuncTest, Sdev8_5GHzScanConnect_Client)
    {
        TestScanConnectParam actInfo;
        int32_t resultArray[TestScanResultType::Type_End] = { 0, 0, 0, 0 };

        LocalClientConnectParameter ConnectParam = {
            nn::wlan::Ssid("G"), nn::wlan::MacAddress::CreateBroadcastMacAddress(), 44,
            { nn::wlan::SecurityMode_StaticAes, nn::wlan::SecurityMode_StaticAes, 0, LocalMasterAes16Key }, true, nn::wlan::BeaconIndication_Enable, 10
        };
        std::unique_ptr<uint8_t[]> pTestBuffer(new uint8_t[TestScanMaxSize]);
        WLANTEST_ASSERT_TRUE(pTestBuffer.get() != nullptr);

        // ジョイコン1セット（各2台）
        WLANTEST_ASSERT_TRUE(localClient.SetBtMode(LocalApiBtNode2));
        WLANTEST_ASSERT_TRUE(localClient.SetupLocalClient(ConnectParam.ssid, ConnectParam.security, TestMatchInfo, sizeof(TestMatchInfo) / sizeof(TestMatchInfo[0]), true, true));

        if (g_SetScanTime != true)
        {
            WLANTEST_EXPECT_TRUE(ScanActionFrameStart(actInfo, localClient.m_actionRxId));
        }
        else
        {
            WLANTEST_EXPECT_TRUE(ScanActionFrameStart(actInfo, localClient.m_actionRxId, g_ScanTime));
        }
        WLANTEST_EXPECT_TRUE(ScanActionConnectRate(resultArray, actInfo, ConnectParam, 500));
        ScanActionFrameStop(actInfo, ConnectParam);

        // Scam110msec合格基準判定
        if (g_ScanTime == WlanTestLdnScanTime)
        {
            // PASS 400以上
            WLANTEST_EXPECT_TRUE(resultArray[Type_ConnectPass] >= 400);
            // SCAN API NG 50以下
            WLANTEST_EXPECT_TRUE(resultArray[Type_ScanApiFail] <= 50);
            // SCAN ACTIONFRAME NG 50以下
            WLANTEST_EXPECT_TRUE(resultArray[Type_ScanActionFrameFail] <= 50);
            // CONNECT NG 4以下
            WLANTEST_EXPECT_TRUE(resultArray[Type_ConnectFail] <= 4);
        }

        NN_LOG("\n             *** SCAN RESULT INFO ***\n");
        NN_LOG("                 SDEV COUNT          : 8\n");
        NN_LOG("                 CHANNEL             : %d\n\n", ConnectParam.channel);
        NN_LOG("                 PASS                : %d\n", resultArray[Type_ConnectPass]);
        NN_LOG("                 SCAN API NG         : %d\n", resultArray[Type_ScanApiFail]);
        NN_LOG("                 SCAN ACTIONFRAME NG : %d\n", resultArray[Type_ScanActionFrameFail]);
        NN_LOG("                 CONNECT NG          : %d\n", resultArray[Type_ConnectFail]);
        NN_LOG("             *** SCAN RESULT INFO ***\n\n");
    }

    // 実行引数でCLIENT台数指定
    // PLR　1.1：30packet/sec, 1400byte, 2.4GHz(1ch), Unicast(Unicast To All) １時間通信 MASTER
    TEST_F(LocalMasterFuncTest, PLR_M_1_1)
    {
        char ssidStr[nn::wlan::Ssid::SsidHexStringLengthMax];
        nn::util::SNPrintf(ssidStr, nn::wlan::Ssid::SsidHexStringLengthMax, "PLR_%d_1_1", g_ClientNum + 1);
        NN_LOG("             [%s] START\n", ssidStr);

        const int64_t agingTime         = Time1h;
        // 通信評価定義
        const uint32_t sendTotalNormal  = 30 * WlanTestRatePacketBufferSize * (agingTime / 1000);
        const uint32_t sendSizeMin      = sendTotalNormal - (sendTotalNormal * 0.1);
        const uint32_t sendSizeMax      = sendTotalNormal + (sendTotalNormal * 0.1);

        nn::wlan::MasterBssParameters fncTestMasterBss = {
                11, false, 30, true, nn::wlan::RateSetLegacy_11gMask, nn::wlan::RateSetLegacy_11gMask,
                { nn::wlan::SecurityMode_StaticAes, nn::wlan::SecurityMode_StaticAes, 0, LocalMasterAes16Key }, nn::wlan::Ssid(ssidStr), 100
        };

        if(ClientMaxConnect == g_ClientNum)
        {
            // ジョイコン1セット（各2台）
            WLANTEST_ASSERT_TRUE(localMaster.SetBtMode(LocalApiBtNode2));
        }
        else
        {
            // ジョイコン2セット（各4台）
            WLANTEST_ASSERT_TRUE(localMaster.SetBtMode(LocalApiBtNode4));
        }
        WLANTEST_ASSERT_TRUE(localMaster.SetupLocalMaster(fncTestMasterBss, TestMatchInfo, sizeof(TestMatchInfo) / sizeof(TestMatchInfo[0])));

        // MASTER受信スレッド起動
        WLANTEST_ASSERT_TRUE(localMaster.LocalReceiveFrameStart(BufferSize200k, WlanTestRecvBufferSize, g_ClientNum, g_ClientNum, WlanTestAgingGetFrameCount));
        // CLIENT接続待機
        WLANTEST_ASSERT_TRUE(localMaster.LocalReceiveWait(WlanTestMasterWaitTimeOut, true));

        // 送信スレッド起動
        WLANTEST_ASSERT_TRUE(localMaster.LocalSendFrameOneShotWaitStart(WlanTestRatePacketBufferSize, agingTime + Time5s, Interval_33m, true));

        // １時間通信(経過後も接続状態である評価)
        WLANTEST_EXPECT_TRUE(localMaster.LocalMasterDisConnectWait(agingTime) > 0);

        // MASTER送信スレッド停止
        localMaster.LocalSendFrameStop();

        // MASTER受信スレッド停止
        localMaster.LocalReceiveFrameStop();

        localMaster.LocalTraceConnetsRttInfo(true);

        localMaster.LocalGetConnectInfo(&connectsInfo);
        WLANTEST_EXPECT_TRUE(connectsInfo.connectCount >= g_ClientNum);

        for (int32_t i = 0; i < connectsInfo.connectCount; i++)
        {
            WLANTEST_EXPECT_TRUE(connectsInfo.facing[i].isConnected);
            WLANTEST_EXPECT_TRUE(connectsInfo.facing[i].perRate <= TestUnicastPlrPerRate);
            WLANTEST_EXPECT_TRUE(connectsInfo.facing[i].seqNumberGap <= TestPlrPaketLostCount);
        }

        NN_LOG("             PLR_%d_1_1 Master Frame Send Total Size:%lld(Range:%lld - %lld)\n",
            g_ClientNum + 1, connectsInfo.totalSendSize, sendSizeMin, sendSizeMax);

        NN_LOG("             [%s] END\n", ssidStr);
    }

    // 実行引数でCLIENT台数指定
    // PLR　1.1：30packet/sec, 1400byte, 2.4GHz(11ch), Unicast(Unicast To All) １時間通信 CLIENT
    TEST_F(LocalClientFuncTest, PLR_C_1_1)
    {
        nn::wlan::ConnectionStatus clientStatus;
        char ssidStr[nn::wlan::Ssid::SsidHexStringLengthMax];
        nn::util::SNPrintf(ssidStr, nn::wlan::Ssid::SsidHexStringLengthMax, "PLR_%d_1_1", g_ClientNum + 1);
        NN_LOG("             [%s] START\n", ssidStr);

        const int64_t agingTime         = Time1h;
        // 通信評価定義
        const uint32_t sendTotalNormal  = 30 * WlanTestRatePacketBufferSize * (agingTime / 1000);
        const uint32_t sendSizeMin      = sendTotalNormal - (sendTotalNormal * 0.1);
        const uint32_t sendSizeMax      = sendTotalNormal + (sendTotalNormal * 0.1);

        const TestLocalConnectParams ConnectParam = {
                nn::wlan::Ssid(ssidStr), nn::wlan::MacAddress::CreateBroadcastMacAddress(), 11,
                { nn::wlan::SecurityMode_StaticAes, nn::wlan::SecurityMode_StaticAes, 0, LocalMasterAes16Key }, true, nn::wlan::BeaconIndication_Enable, 10
        };

        if(ClientMaxConnect == g_ClientNum)
        {
            // ジョイコン1セット（各2台）
            WLANTEST_ASSERT_TRUE(localClient.SetBtMode(LocalApiBtNode2));
        }
        else
        {
            // ジョイコン2セット（各4台）
            WLANTEST_ASSERT_TRUE(localClient.SetBtMode(LocalApiBtNode4));
        }
        WLANTEST_ASSERT_TRUE(localClient.SetupLocalClient(ConnectParam.ssid, ConnectParam.security, TestMatchInfo, sizeof(TestMatchInfo) / sizeof(TestMatchInfo[0]), true, false));

        Connect(ConnectParam);
        WLANTEST_EXPECT_RESULT_SUCCESS(nn::wlan::Local::GetMacAddress(&localClient.m_macAddress));

        // 受信スレッド起動
        WLANTEST_ASSERT_TRUE(localClient.LocalReceiveFrameStart(BufferSize200k, WlanTestRecvBufferSize, 1, 1, WlanTestAgingGetFrameCount));
        // MASTER接続待機
        WLANTEST_ASSERT_TRUE(localClient.LocalReceiveWait(WlanMaxClientWaitMsecTimeOut, Time1s, true));

        // 送信スレッド起動
        WLANTEST_ASSERT_TRUE(localClient.LocalSendFrameOneShotWaitStart(WlanTestRatePacketBufferSize, agingTime + Time5s, Interval_33m, true));

        while (NN_STATIC_CONDITION(1))
        {
            // MASTERからの接続が切られるまで送信し続ける
            if (nn::os::TimedWaitSystemEvent(&connectionEvent, nn::TimeSpan::FromMilliSeconds(Time1s)) == true)
            {
                WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::GetConnectionStatus(&clientStatus));
                if (clientStatus.state != nn::wlan::ConnectionState_Connected)
                {
                    break;
                }
            }
        }
        // 送信スレッド停止
        localClient.LocalSendFrameStop();
        // 受信スレッド停止
        localClient.LocalReceiveFrameStop();
        // 通信結果ログ出力
        localClient.LocalTraceConnetsRttInfo(true);

        localClient.LocalGetConnectInfo(&connectsInfo);
        WLANTEST_EXPECT_TRUE(connectsInfo.connectCount == g_ClientNum);

        for (int32_t i = 0; i < connectsInfo.connectCount; i++)
        {
            WLANTEST_EXPECT_TRUE(connectsInfo.facing[i].isConnected);
            WLANTEST_EXPECT_TRUE(connectsInfo.facing[i].perRate <= TestUnicastPlrPerRate);
            WLANTEST_EXPECT_TRUE(connectsInfo.facing[i].seqNumberGap <= TestPlrPaketLostCount);
        }

        Disconnect();

        NN_LOG("             PLR_%d_1_1 Client Frame Send Total Size:%lld(Range:%lld - %lld)\n",
            g_ClientNum + 1, connectsInfo.totalSendSize, sendSizeMin, sendSizeMax);

        NN_LOG("             [%s] END\n", ssidStr);
    }

    // 実行引数でCLIENT台数指定
    // PLR　1.2: 30packet/sec, 1400byte, 5GHz(36ch), Unicast(Unicast To All) １時間通信 MASTER
    TEST_F(LocalMasterFuncTest, PLR_M_1_2)
    {
        char ssidStr[nn::wlan::Ssid::SsidHexStringLengthMax];
        nn::util::SNPrintf(ssidStr, nn::wlan::Ssid::SsidHexStringLengthMax, "PLR_%d_1_2", g_ClientNum + 1);
        NN_LOG("             [%s] START\n", ssidStr);

        const int64_t agingTime         = Time1h;
        // 通信評価定義
        const uint32_t sendTotalNormal  = 30 * WlanTestRatePacketBufferSize * (agingTime / 1000);
        const uint32_t sendSizeMin      = sendTotalNormal - (sendTotalNormal * 0.1);
        const uint32_t sendSizeMax      = sendTotalNormal + (sendTotalNormal * 0.1);

        const nn::wlan::MasterBssParameters fncTestMasterBss = {
                36, false, 30, true, nn::wlan::RateSetLegacy_11gMask, nn::wlan::RateSetLegacy_11gMask,
                { nn::wlan::SecurityMode_StaticAes, nn::wlan::SecurityMode_StaticAes, 0, LocalMasterAes16Key }, nn::wlan::Ssid(ssidStr), 100
        };

        if(ClientMaxConnect == g_ClientNum)
        {
            // ジョイコン1セット（各2台）
            WLANTEST_ASSERT_TRUE(localMaster.SetBtMode(LocalApiBtNode2));
        }
        else
        {
            // ジョイコン2セット（各4台）
            WLANTEST_ASSERT_TRUE(localMaster.SetBtMode(LocalApiBtNode4));
        }
        WLANTEST_ASSERT_TRUE(localMaster.SetupLocalMaster(fncTestMasterBss, TestMatchInfo, sizeof(TestMatchInfo) / sizeof(TestMatchInfo[0])));

        // MASTER受信スレッド起動
        WLANTEST_ASSERT_TRUE(localMaster.LocalReceiveFrameStart(BufferSize200k, WlanTestRecvBufferSize, g_ClientNum, g_ClientNum, WlanTestAgingGetFrameCount));
        // CLIENT接続待機
        WLANTEST_ASSERT_TRUE(localMaster.LocalReceiveWait(WlanTestMasterWaitTimeOut, true));

        // 送信スレッド起動
        WLANTEST_ASSERT_TRUE(localMaster.LocalSendFrameOneShotWaitStart(WlanTestRatePacketBufferSize, agingTime + Time5s, Interval_33m, true));

        // １時間通信(経過後も接続状態である評価)
        WLANTEST_EXPECT_TRUE(localMaster.LocalMasterDisConnectWait(agingTime) > 0);

        // MASTER送信スレッド停止
        localMaster.LocalSendFrameStop();

        // MASTER受信スレッド停止
        localMaster.LocalReceiveFrameStop();

        localMaster.LocalTraceConnetsRttInfo(true);

        localMaster.LocalGetConnectInfo(&connectsInfo);
        WLANTEST_EXPECT_TRUE(connectsInfo.connectCount >= g_ClientNum);

        for (int32_t i = 0; i < connectsInfo.connectCount; i++)
        {
            WLANTEST_EXPECT_TRUE(connectsInfo.facing[i].isConnected);
            WLANTEST_EXPECT_TRUE(connectsInfo.facing[i].perRate <= TestUnicastPlrPerRate);
            WLANTEST_EXPECT_TRUE(connectsInfo.facing[i].seqNumberGap <= TestPlrPaketLostCount);
        }

        NN_LOG("             PLR_%d_1_2 Master Frame Send Total Size:%lld(Range:%lld - %lld)\n",
            g_ClientNum + 1, connectsInfo.totalSendSize, sendSizeMin, sendSizeMax);

        NN_LOG("             [%s] END\n", ssidStr);
    }

    // 実行引数でCLIENT台数指定
    // PLR　1.2: 30packet/sec, 1400byte, 5GHz(36ch), Unicast(Unicast To All) １時間通信 CLIENT
    TEST_F(LocalClientFuncTest, PLR_C_1_2)
    {
        nn::wlan::ConnectionStatus clientStatus;
        char ssidStr[nn::wlan::Ssid::SsidHexStringLengthMax];
        nn::util::SNPrintf(ssidStr, nn::wlan::Ssid::SsidHexStringLengthMax, "PLR_%d_1_2", g_ClientNum + 1);
        NN_LOG("             [%s] START\n", ssidStr);

        const int64_t agingTime         = Time1h;
        // 通信評価定義
        const uint32_t sendTotalNormal  = 30 * WlanTestRatePacketBufferSize * (agingTime / 1000);
        const uint32_t sendSizeMin      = sendTotalNormal - (sendTotalNormal * 0.1);
        const uint32_t sendSizeMax      = sendTotalNormal + (sendTotalNormal * 0.1);

        const TestLocalConnectParams ConnectParam = {
                nn::wlan::Ssid(ssidStr), nn::wlan::MacAddress::CreateBroadcastMacAddress(), 36,
                { nn::wlan::SecurityMode_StaticAes, nn::wlan::SecurityMode_StaticAes, 0, LocalMasterAes16Key }, true, nn::wlan::BeaconIndication_Enable, 10
        };

        if(ClientMaxConnect == g_ClientNum)
        {
            // ジョイコン1セット（各2台）
            WLANTEST_ASSERT_TRUE(localClient.SetBtMode(LocalApiBtNode2));
        }
        else
        {
            // ジョイコン2セット（各4台）
            WLANTEST_ASSERT_TRUE(localClient.SetBtMode(LocalApiBtNode4));
        }
        WLANTEST_ASSERT_TRUE(localClient.SetupLocalClient(ConnectParam.ssid, ConnectParam.security, TestMatchInfo, sizeof(TestMatchInfo) / sizeof(TestMatchInfo[0]), true, false));

        Connect(ConnectParam);
        WLANTEST_EXPECT_RESULT_SUCCESS(nn::wlan::Local::GetMacAddress(&localClient.m_macAddress));

        // 受信スレッド起動
        WLANTEST_ASSERT_TRUE(localClient.LocalReceiveFrameStart(BufferSize200k, WlanTestRecvBufferSize, 1, 1, WlanTestAgingGetFrameCount));
        // MASTER接続待機
        WLANTEST_ASSERT_TRUE(localClient.LocalReceiveWait(WlanMaxClientWaitMsecTimeOut, Time1s, true));

        // 送信スレッド起動
        WLANTEST_ASSERT_TRUE(localClient.LocalSendFrameOneShotWaitStart(WlanTestRatePacketBufferSize, agingTime + Time5s, Interval_33m, true));

        while (NN_STATIC_CONDITION(1))
        {
            // MASTERからの接続が切られるまで送信し続ける
            if (nn::os::TimedWaitSystemEvent(&connectionEvent, nn::TimeSpan::FromMilliSeconds(Time1s)) == true)
            {
                WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::GetConnectionStatus(&clientStatus));
                if (clientStatus.state != nn::wlan::ConnectionState_Connected)
                {
                    break;
                }
            }
        }
        // 送信スレッド停止
        localClient.LocalSendFrameStop();
        // 受信スレッド停止
        localClient.LocalReceiveFrameStop();
        // 通信結果ログ出力
        localClient.LocalTraceConnetsRttInfo(true);

        localClient.LocalGetConnectInfo(&connectsInfo);
        WLANTEST_EXPECT_TRUE(connectsInfo.connectCount == g_ClientNum);

        for (int32_t i = 0; i < connectsInfo.connectCount; i++)
        {
            WLANTEST_EXPECT_TRUE(connectsInfo.facing[i].isConnected);
            WLANTEST_EXPECT_TRUE(connectsInfo.facing[i].perRate <= TestUnicastPlrPerRate);
            WLANTEST_EXPECT_TRUE(connectsInfo.facing[i].seqNumberGap <= TestPlrPaketLostCount);
        }

        Disconnect();

        NN_LOG("             PLR_%d_1_2 Client Frame Send Total Size:%lld(Range:%lld - %lld)\n",
            g_ClientNum + 1, connectsInfo.totalSendSize, sendSizeMin, sendSizeMax);

        NN_LOG("             [%s] END\n", ssidStr);
    }

    // 実行引数でCLIENT台数指定
    // PLR　1.3: 120packet/sec, 600byte, 2.4GHz(11ch), Broadcast 1時間通信 MASTER
    TEST_F(LocalMasterFuncTest, PLR_M_1_3)
    {
        const int32_t secPakets         = 120;
        const int64_t sendInterval      = static_cast<int64_t>(Time1s / secPakets);

        const int64_t agingTime         = Time1h;
        // 通信評価定義
        const uint32_t sendTotalNormal  = secPakets * WlanTestBroadCastPacketBufferSize * (agingTime / 1000);
        const uint32_t sendSizeMin      = sendTotalNormal - (sendTotalNormal * 0.1);
        const uint32_t sendSizeMax      = sendTotalNormal + (sendTotalNormal * 0.1);

        char ssidStr[nn::wlan::Ssid::SsidHexStringLengthMax];
        nn::util::SNPrintf(ssidStr, nn::wlan::Ssid::SsidHexStringLengthMax, "PLR_%d_1_3", g_ClientNum + 1);
        NN_LOG("             [%s] START\n", ssidStr);

        const nn::wlan::MasterBssParameters fncTestMasterBss = {
                11, false, 30, true, nn::wlan::RateSetLegacy_11gMask, nn::wlan::RateSetLegacy_11gMask,
                { nn::wlan::SecurityMode_StaticAes, nn::wlan::SecurityMode_StaticAes, 0, LocalMasterAes16Key }, nn::wlan::Ssid(ssidStr), 100
        };

        if(ClientMaxConnect == g_ClientNum)
        {
            // ジョイコン1セット（各2台）
            WLANTEST_ASSERT_TRUE(localMaster.SetBtMode(LocalApiBtNode2));
        }
        else
        {
            // ジョイコン2セット（各4台）
            WLANTEST_ASSERT_TRUE(localMaster.SetBtMode(LocalApiBtNode4));
        }
        WLANTEST_ASSERT_TRUE(localMaster.SetupLocalMaster(fncTestMasterBss, TestMatchInfo, sizeof(TestMatchInfo) / sizeof(TestMatchInfo[0])));

        // MASTER受信スレッド起動
        WLANTEST_ASSERT_TRUE(localMaster.LocalReceiveFrameStart(BufferSize200k, WlanTestRecvBufferSize, g_ClientNum, g_ClientNum, WlanTestAgingGetFrameCount));
        // CLIENT接続待機
        WLANTEST_ASSERT_TRUE(localMaster.LocalReceiveWait(WlanTestMasterWaitTimeOut, true));
        // MASTER送信スレッド起動
        WLANTEST_ASSERT_TRUE(localMaster.LocalBroadcastSendFrameStart(WlanTestBroadCastPacketBufferSize, agingTime + Time5s, sendInterval));

        // １時間通信(経過後も接続状態である評価)
        WLANTEST_EXPECT_TRUE(localMaster.LocalMasterDisConnectWait(agingTime) > 0);

        // MASTER送信スレッド停止
        localMaster.LocalSendFrameStop();

        // MASTER受信スレッド停止
        localMaster.LocalReceiveFrameStop();

        localMaster.LocalTraceConnetsRttInfo(true);

        localMaster.LocalGetConnectInfo(&connectsInfo);
        WLANTEST_EXPECT_TRUE(connectsInfo.connectCount >= g_ClientNum);

        for (int32_t i = 0; i < connectsInfo.connectCount; i++)
        {
            WLANTEST_EXPECT_TRUE(connectsInfo.facing[i].isConnected);
            WLANTEST_EXPECT_TRUE(connectsInfo.facing[i].perRate <= TestBroadcastPlrPerRate);
            WLANTEST_EXPECT_TRUE(connectsInfo.facing[i].seqNumberGap <= TestPlrPaketLostCount);
        }

        NN_LOG("             PLR_%d_1_3 Master Frame Send Total Size:%lld(Range:%lld - %lld)\n",
            g_ClientNum + 1, connectsInfo.totalSendSize, sendSizeMin, sendSizeMax);

        NN_LOG("             [%s] END\n", ssidStr);
    }

    // 実行引数でCLIENT台数指定
    // PLR　1.3: 120packet/sec, 600byte, 2.4GHz(11ch), Broadcast 1時間通信 CLIENT
    TEST_F(LocalClientFuncTest, PLR_C_1_3)
    {
        const int32_t secPakets         = 120;
        const int64_t sendInterval      = static_cast<int64_t>(Time1s / secPakets);
        nn::wlan::ConnectionStatus clientStatus;

        const int64_t agingTime         = Time1h;
        // 通信評価定義
        const uint32_t sendTotalNormal  = secPakets * WlanTestBroadCastPacketBufferSize * (agingTime / 1000);
        const uint32_t sendSizeMin      = sendTotalNormal - (sendTotalNormal * 0.1);
        const uint32_t sendSizeMax      = sendTotalNormal + (sendTotalNormal * 0.1);

        char ssidStr[nn::wlan::Ssid::SsidHexStringLengthMax];
        nn::util::SNPrintf(ssidStr, nn::wlan::Ssid::SsidHexStringLengthMax, "PLR_%d_1_3", g_ClientNum + 1);
        NN_LOG("             [%s] START\n", ssidStr);

        const TestLocalConnectParams ConnectParam = {
                nn::wlan::Ssid(ssidStr), nn::wlan::MacAddress::CreateBroadcastMacAddress(), 11,
                { nn::wlan::SecurityMode_StaticAes, nn::wlan::SecurityMode_StaticAes, 0, LocalMasterAes16Key }, true, nn::wlan::BeaconIndication_Enable, 10
        };

        if(ClientMaxConnect == g_ClientNum)
        {
            // ジョイコン1セット（各2台）
            WLANTEST_ASSERT_TRUE(localClient.SetBtMode(LocalApiBtNode2));
        }
        else
        {
            // ジョイコン2セット（各4台）
            WLANTEST_ASSERT_TRUE(localClient.SetBtMode(LocalApiBtNode4));
        }
        WLANTEST_ASSERT_TRUE(localClient.SetupLocalClient(ConnectParam.ssid, ConnectParam.security, TestMatchInfo, sizeof(TestMatchInfo) / sizeof(TestMatchInfo[0]), true, false));

        Connect(ConnectParam);
        WLANTEST_EXPECT_RESULT_SUCCESS(nn::wlan::Local::GetMacAddress(&localClient.m_macAddress));

        // 受信スレッド起動
        WLANTEST_ASSERT_TRUE(localClient.LocalReceiveFrameStart(BufferSize200k, WlanTestRecvBufferSize, 1, 1, WlanTestAgingGetFrameCount));
        // MASTER接続待機
        WLANTEST_ASSERT_TRUE(localClient.LocalReceiveWait(WlanMaxClientWaitMsecTimeOut, Time1s, true));
        // 送信スレッド起動
        WLANTEST_ASSERT_TRUE(localClient.LocalBroadcastSendFrameStart(WlanTestBroadCastPacketBufferSize, agingTime + Time5s, sendInterval));

        while (NN_STATIC_CONDITION(1))
        {
            // MASTERからの接続が切られるまで送信し続ける
            if (nn::os::TimedWaitSystemEvent(&connectionEvent, nn::TimeSpan::FromMilliSeconds(Time1s)) == true)
            {
                WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::GetConnectionStatus(&clientStatus));
                if (clientStatus.state != nn::wlan::ConnectionState_Connected)
                {
                    break;
                }
            }
        }

        // 送信スレッド停止
        localClient.LocalSendFrameStop();
        // 受信スレッド停止
        localClient.LocalReceiveFrameStop();
        // 通信結果ログ出力
        localClient.LocalTraceConnetsRttInfo(true);

        localClient.LocalGetConnectInfo(&connectsInfo);
        WLANTEST_EXPECT_TRUE(connectsInfo.connectCount == g_ClientNum);

        for (int32_t i = 0; i < connectsInfo.connectCount; i++)
        {
            WLANTEST_EXPECT_TRUE(connectsInfo.facing[i].isConnected);
            WLANTEST_EXPECT_TRUE(connectsInfo.facing[i].perRate <= TestBroadcastPlrPerRate);
            WLANTEST_EXPECT_TRUE(connectsInfo.facing[i].seqNumberGap <= TestPlrPaketLostCount);
        }

        Disconnect();

        NN_LOG("             PLR_%d_1_3 Client Frame Send Total Size:%lld(Range:%lld - %lld)\n",
            g_ClientNum + 1, connectsInfo.totalSendSize, sendSizeMin, sendSizeMax);

        NN_LOG("             [%s] END\n", ssidStr);
    }

    // 実行引数でCLIENT台数指定
    // PLR　1.4: 120packet/sec, 600byte, 5GHz(36ch), Broadcast MASTER
    TEST_F(LocalMasterFuncTest, PLR_M_1_4)
    {
        const int32_t secPakets         = 120;
        const int64_t sendInterval      = static_cast<int64_t>(Time1s / secPakets);

        const int64_t agingTime         = Time1h;
        // 通信評価定義
        const uint32_t sendTotalNormal  = secPakets * WlanTestBroadCastPacketBufferSize * (agingTime / 1000);
        const uint32_t sendSizeMin      = sendTotalNormal - (sendTotalNormal * 0.1);
        const uint32_t sendSizeMax      = sendTotalNormal + (sendTotalNormal * 0.1);

        char ssidStr[nn::wlan::Ssid::SsidHexStringLengthMax];
        nn::util::SNPrintf(ssidStr, nn::wlan::Ssid::SsidHexStringLengthMax, "PLR_%d_1_4", g_ClientNum + 1);
        NN_LOG("             [%s] START\n", ssidStr);

        const nn::wlan::MasterBssParameters fncTestMasterBss = {
                36, false, 30, true, nn::wlan::RateSetLegacy_11gMask, nn::wlan::RateSetLegacy_11gMask,
                { nn::wlan::SecurityMode_StaticAes, nn::wlan::SecurityMode_StaticAes, 0, LocalMasterAes16Key }, nn::wlan::Ssid(ssidStr), 100
        };

        if(ClientMaxConnect == g_ClientNum)
        {
            // ジョイコン1セット（各2台）
            WLANTEST_ASSERT_TRUE(localMaster.SetBtMode(LocalApiBtNode2));
        }
        else
        {
            // ジョイコン2セット（各4台）
            WLANTEST_ASSERT_TRUE(localMaster.SetBtMode(LocalApiBtNode4));
        }
        WLANTEST_ASSERT_TRUE(localMaster.SetupLocalMaster(fncTestMasterBss, TestMatchInfo, sizeof(TestMatchInfo) / sizeof(TestMatchInfo[0])));

        // MASTER受信スレッド起動
        WLANTEST_ASSERT_TRUE(localMaster.LocalReceiveFrameStart(BufferSize200k, WlanTestRecvBufferSize, g_ClientNum, g_ClientNum, WlanTestAgingGetFrameCount));
        // CLIENT接続待機
        WLANTEST_ASSERT_TRUE(localMaster.LocalReceiveWait(WlanTestMasterWaitTimeOut, true));
        // MASTER送信スレッド起動
        WLANTEST_ASSERT_TRUE(localMaster.LocalBroadcastSendFrameStart(WlanTestBroadCastPacketBufferSize, agingTime + Time5s, sendInterval));

        // １時間通信(経過後も接続状態である評価)
        WLANTEST_EXPECT_TRUE(localMaster.LocalMasterDisConnectWait(agingTime) > 0);

        // MASTER送信スレッド停止
        localMaster.LocalSendFrameStop();

        // MASTER受信スレッド停止
        localMaster.LocalReceiveFrameStop();

        localMaster.LocalTraceConnetsRttInfo(true);

        localMaster.LocalGetConnectInfo(&connectsInfo);
        WLANTEST_EXPECT_TRUE(connectsInfo.connectCount >= g_ClientNum);

        for (int32_t i = 0; i < connectsInfo.connectCount; i++)
        {
            WLANTEST_EXPECT_TRUE(connectsInfo.facing[i].isConnected);
            WLANTEST_EXPECT_TRUE(connectsInfo.facing[i].perRate <= TestBroadcastPlrPerRate);
            WLANTEST_EXPECT_TRUE(connectsInfo.facing[i].seqNumberGap <= TestPlrPaketLostCount);
        }

        NN_LOG("             PLR_%d_1_4 Master Frame Send Total Size:%lld(Range:%lld - %lld)\n",
            g_ClientNum + 1, connectsInfo.totalSendSize, sendSizeMin, sendSizeMax);

        NN_LOG("             [%s] END\n", ssidStr);
    }

    // 実行引数でCLIENT台数指定
    // PLR　1.4: 120packet/sec, 600byte, 5GHz(36ch), Broadcast CLIENT
    TEST_F(LocalClientFuncTest, PLR_C_1_4)
    {
        const int32_t secPakets         = 120;
        const int64_t sendInterval      = static_cast<int64_t>(Time1s / secPakets);
        nn::wlan::ConnectionStatus clientStatus;

        const int64_t agingTime         = Time1h;
        // 通信評価定義
        const uint32_t sendTotalNormal  = secPakets * WlanTestBroadCastPacketBufferSize * (agingTime / 1000);
        const uint32_t sendSizeMin      = sendTotalNormal - (sendTotalNormal * 0.1);
        const uint32_t sendSizeMax      = sendTotalNormal + (sendTotalNormal * 0.1);

        char ssidStr[nn::wlan::Ssid::SsidHexStringLengthMax];
        nn::util::SNPrintf(ssidStr, nn::wlan::Ssid::SsidHexStringLengthMax, "PLR_%d_1_4", g_ClientNum + 1);
        NN_LOG("             [%s] START\n", ssidStr);

        const TestLocalConnectParams ConnectParam = {
                nn::wlan::Ssid(ssidStr), nn::wlan::MacAddress::CreateBroadcastMacAddress(), 36,
                { nn::wlan::SecurityMode_StaticAes, nn::wlan::SecurityMode_StaticAes, 0, LocalMasterAes16Key }, true, nn::wlan::BeaconIndication_Enable, 10
        };

        if(ClientMaxConnect == g_ClientNum)
        {
            // ジョイコン1セット（各2台）
            WLANTEST_ASSERT_TRUE(localClient.SetBtMode(LocalApiBtNode2));
        }
        else
        {
            // ジョイコン2セット（各4台）
            WLANTEST_ASSERT_TRUE(localClient.SetBtMode(LocalApiBtNode4));
        }
        WLANTEST_ASSERT_TRUE(localClient.SetupLocalClient(ConnectParam.ssid, ConnectParam.security, TestMatchInfo, sizeof(TestMatchInfo) / sizeof(TestMatchInfo[0]), true, false));

        Connect(ConnectParam);
        WLANTEST_EXPECT_RESULT_SUCCESS(nn::wlan::Local::GetMacAddress(&localClient.m_macAddress));

        // 受信スレッド起動
        WLANTEST_ASSERT_TRUE(localClient.LocalReceiveFrameStart(BufferSize200k, WlanTestRecvBufferSize, 1, 1, WlanTestAgingGetFrameCount));
        // MASTER接続待機
        WLANTEST_ASSERT_TRUE(localClient.LocalReceiveWait(WlanMaxClientWaitMsecTimeOut, Time1s, true));
        // 送信スレッド起動
        WLANTEST_ASSERT_TRUE(localClient.LocalBroadcastSendFrameStart(WlanTestBroadCastPacketBufferSize, agingTime + Time5s, sendInterval));

        while (NN_STATIC_CONDITION(1))
        {
            // MASTERからの接続が切られるまで送信し続ける
            if (nn::os::TimedWaitSystemEvent(&connectionEvent, nn::TimeSpan::FromMilliSeconds(Time1s)) == true)
            {
                WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::GetConnectionStatus(&clientStatus));
                if (clientStatus.state != nn::wlan::ConnectionState_Connected)
                {
                    break;
                }
            }
        }

        // 送信スレッド停止
        localClient.LocalSendFrameStop();
        // 受信スレッド停止
        localClient.LocalReceiveFrameStop();
        // 通信結果ログ出力
        localClient.LocalTraceConnetsRttInfo(true);

        localClient.LocalGetConnectInfo(&connectsInfo);
        WLANTEST_EXPECT_TRUE(connectsInfo.connectCount == g_ClientNum);

        for (int32_t i = 0; i < connectsInfo.connectCount; i++)
        {
            WLANTEST_EXPECT_TRUE(connectsInfo.facing[i].isConnected);
            WLANTEST_EXPECT_TRUE(connectsInfo.facing[i].perRate <= TestBroadcastPlrPerRate);
            WLANTEST_EXPECT_TRUE(connectsInfo.facing[i].seqNumberGap <= TestPlrPaketLostCount);
        }

        Disconnect();

        NN_LOG("             PLR_%d_1_4 Client Frame Send Total Size:%lld(Range:%lld - %lld)\n",
            g_ClientNum + 1, connectsInfo.totalSendSize, sendSizeMin, sendSizeMax);

        NN_LOG("             [%s] END\n", ssidStr);
    }

    // 実行引数でCLIENT台数指定
    // RTT　1.1: 2.4GHz(11ch) MASTER
    TEST_F(LocalMasterFuncTest, RTT_M_1_1)
    {
        const int64_t agingTime                 = Time600s;
        const int32_t secBroadDummyPakets       = 90;
        const int32_t secBroadPingPakets        = 1;
        const int64_t sendBroadDummyInterval    = static_cast<int64_t>(Time1s / secBroadDummyPakets);
        const int64_t sendBroadPingInterval     = static_cast<int64_t>(Time1s / secBroadPingPakets);

        // 通信評価定義
        const uint32_t sendTotalNormal          = secBroadDummyPakets * WlanTestRTTPacketBufferSize * (agingTime / 1000);
        const uint32_t sendSizeMin              = sendTotalNormal - (sendTotalNormal * 0.1);
        const uint32_t sendSizeMax              = sendTotalNormal + (sendTotalNormal * 0.1);

        nn::wlan::MacAddress broadMac           = nn::wlan::MacAddress::CreateBroadcastMacAddress();
        std::unique_ptr<uint8_t[]> putBroadBuffer(new uint8_t[WlanTestRTTPacketBufferSize]);
        uint32_t seqNumberPing;
        uint32_t outRxId;
        nn::Result result;

        WLANTEST_ASSERT_TRUE(putBroadBuffer.get() != nullptr);

        char ssidStr[nn::wlan::Ssid::SsidHexStringLengthMax];
        nn::util::SNPrintf(ssidStr, nn::wlan::Ssid::SsidHexStringLengthMax, "RTT_%d_1_1", g_ClientNum + 1);
        NN_LOG("             [%s] START\n", ssidStr);

        const nn::wlan::MasterBssParameters fncTestMasterBss = {
                11, false, 30, true, nn::wlan::RateSetLegacy_11gMask, nn::wlan::RateSetLegacy_11gMask,
                { nn::wlan::SecurityMode_StaticAes, nn::wlan::SecurityMode_StaticAes, 0, LocalMasterAes16Key }, nn::wlan::Ssid(ssidStr), 100
        };

        if(ClientMaxConnect == g_ClientNum)
        {
            // ジョイコン1セット（各2台）
            WLANTEST_ASSERT_TRUE(localMaster.SetBtMode(LocalApiBtNode2));
        }
        else
        {
            // ジョイコン2セット（各4台）
            WLANTEST_ASSERT_TRUE(localMaster.SetBtMode(LocalApiBtNode4));
        }
        WLANTEST_ASSERT_TRUE(localMaster.SetupLocalMaster(fncTestMasterBss, TestMatchInfo, sizeof(TestMatchInfo) / sizeof(TestMatchInfo[0])));

        WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::AddEthertypeToRxEntry(localMaster.m_entRxId, TestEtherTypes[1]));

        // MASTER受信スレッド起動
        WLANTEST_ASSERT_TRUE(localMaster.LocalReceiveFrameStart(BufferSize200k, WlanTestRecvBufferSize, g_ClientNum, g_ClientNum, WlanTestAgingGetFrameCount));
        // CLIENT接続待機
        WLANTEST_ASSERT_TRUE(localMaster.LocalReceiveWait(WlanTestMasterWaitTimeOut, true));

        // MASTER送信スレッド起動
        WLANTEST_ASSERT_TRUE(localMaster.LocalBroadcastSendFrameStart(WlanTestRTTPacketBufferSize, agingTime, sendBroadDummyInterval));

        seqNumberPing = 0;
        sysTick = nn::os::GetSystemTick();
        // 600秒間通信
        while ((nn::os::GetSystemTick() - sysTick).ToTimeSpan().GetMilliSeconds() <= agingTime)
        {
            WLANTEST_EXPECT_TRUE(localMaster.LocalMakeFrame(putBroadBuffer.get(), WlanTestRTTPacketBufferSize, broadMac, localMaster.m_macAddress, TestEtherTypes[1], TestMatchInfo[0], seqNumberPing, true));

            result = nn::wlan::Local::PutFrameRaw(putBroadBuffer.get(), WlanTestRTTPacketBufferSize);
            WLANTEST_EXPECT_RESULT_SUCCESS(result);
            if (result.IsSuccess() != true)
            {
                NN_LOG("             nn::wlan::Local::PutFrameRaw Broad Send(Count:%d) Error(%d:%d) 0x%08x\n", seqNumberPing,
                        result.GetModule(), result.GetDescription(), result.GetInnerValueForDebug());
            }
            else
            {
                seqNumberPing++;
                nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(sendBroadPingInterval));
            }
        }

        // MASTER送信スレッド停止
        localMaster.LocalSendFrameStop();

        // MASTER受信スレッド停止
        localMaster.LocalReceiveFrameStop();

        localMaster.LocalTraceConnetsRttInfo(true);

        WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::DeleteEthertypeFromRxEntry(&outRxId, TestEtherTypes[1]));

        localMaster.LocalGetConnectInfo(&connectsInfo);
        WLANTEST_EXPECT_TRUE(connectsInfo.connectCount >= g_ClientNum);

        for (int32_t i = 0; i < connectsInfo.connectCount; i++)
        {
            WLANTEST_EXPECT_TRUE(connectsInfo.facing[i].isConnected);
            WLANTEST_EXPECT_TRUE(connectsInfo.facing[i].rttAverage <= TestRttPlrRate);
        }

        NN_LOG("             RTT_%d_1_1 Master Frame Send Total Size:%lld(Range:%lld - %lld)\n",
            g_ClientNum + 1, connectsInfo.totalSendSize, sendSizeMin, sendSizeMax);

        NN_LOG("             [%s] END\n", ssidStr);
    }

    // 実行引数でCLIENT台数指定
    // RTT　1.1: 2.4GHz(11ch) CLIENT
    TEST_F(LocalClientFuncTest, RTT_C_1_1)
    {
        const int64_t agingTime                 = Time600s;
        const int32_t secBroadDummyPakets       = 90;
        const int32_t secBroadPingPakets        = 1;
        const int64_t sendBroadDummyInterval    = static_cast<int64_t>(Time1s / secBroadDummyPakets);
        const int64_t sendBroadPingInterval     = static_cast<int64_t>(Time1s / secBroadPingPakets);

        // 通信評価定義
        const uint32_t sendTotalNormal          = secBroadDummyPakets * WlanTestRTTPacketBufferSize * (agingTime / 1000);
        const uint32_t sendSizeMin              = sendTotalNormal - (sendTotalNormal * 0.1);
        const uint32_t sendSizeMax              = sendTotalNormal + (sendTotalNormal * 0.1);

        nn::wlan::MacAddress broadMac           = nn::wlan::MacAddress::CreateBroadcastMacAddress();
        std::unique_ptr<uint8_t[]> putBroadBuffer(new uint8_t[WlanTestRTTPacketBufferSize]);
        nn::wlan::ConnectionStatus clientStatus;
        uint32_t seqNumberPing;
        uint32_t outRxId;
        nn::Result result;

        WLANTEST_ASSERT_TRUE(putBroadBuffer.get() != nullptr);

        char ssidStr[nn::wlan::Ssid::SsidHexStringLengthMax];
        nn::util::SNPrintf(ssidStr, nn::wlan::Ssid::SsidHexStringLengthMax, "RTT_%d_1_1", g_ClientNum + 1);
        NN_LOG("             [%s] START\n", ssidStr);

        const TestLocalConnectParams ConnectParam = {
                nn::wlan::Ssid(ssidStr), nn::wlan::MacAddress::CreateBroadcastMacAddress(), 11,
                { nn::wlan::SecurityMode_StaticAes, nn::wlan::SecurityMode_StaticAes, 0, LocalMasterAes16Key }, true, nn::wlan::BeaconIndication_Enable, 10
        };

        if(ClientMaxConnect == g_ClientNum)
        {
            // ジョイコン1セット（各2台）
            WLANTEST_ASSERT_TRUE(localClient.SetBtMode(LocalApiBtNode2));
        }
        else
        {
            // ジョイコン2セット（各4台）
            WLANTEST_ASSERT_TRUE(localClient.SetBtMode(LocalApiBtNode4));
        }
        WLANTEST_ASSERT_TRUE(localClient.SetupLocalClient(ConnectParam.ssid, ConnectParam.security, TestMatchInfo, sizeof(TestMatchInfo) / sizeof(TestMatchInfo[0]), true, false));
        WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::AddEthertypeToRxEntry(localClient.m_entRxId, TestEtherTypes[1]));

        Connect(ConnectParam);
        WLANTEST_EXPECT_RESULT_SUCCESS(nn::wlan::Local::GetMacAddress(&localClient.m_macAddress));

        // 受信スレッド起動
        WLANTEST_ASSERT_TRUE(localClient.LocalReceiveFrameStart(BufferSize200k, WlanTestRecvBufferSize, 1, 1, WlanTestAgingGetFrameCount));
        // MASTER接続待機
        WLANTEST_ASSERT_TRUE(localClient.LocalReceiveWait(WlanMaxClientWaitMsecTimeOut, Time1s, true));

        // 送信スレッド起動
        WLANTEST_ASSERT_TRUE(localClient.LocalBroadcastSendFrameStart(WlanTestRTTPacketBufferSize, agingTime, sendBroadDummyInterval));

        seqNumberPing = 0;
        while (NN_STATIC_CONDITION(1))
        {
            // MASTERからの接続が切られるまで送信し続ける
            if (nn::os::TimedWaitSystemEvent(&connectionEvent, nn::TimeSpan::FromMilliSeconds(sendBroadPingInterval)) == true)
            {
                WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::GetConnectionStatus(&clientStatus));
                if (clientStatus.state != nn::wlan::ConnectionState_Connected)
                {
                    break;
                }
            }

            WLANTEST_EXPECT_TRUE(localClient.LocalMakeFrame(putBroadBuffer.get(), WlanTestRTTPacketBufferSize, broadMac, localClient.m_macAddress, TestEtherTypes[1], TestMatchInfo[0], seqNumberPing, true));

            result = nn::wlan::Local::PutFrameRaw(putBroadBuffer.get(), WlanTestRTTPacketBufferSize);
            WLANTEST_EXPECT_RESULT_SUCCESS(result);
            if (result.IsSuccess() != true)
            {
                NN_LOG("             nn::wlan::Local::PutFrameRaw Broad Send(Count:%d) Error(%d:%d) 0x%08x\n", seqNumberPing,
                        result.GetModule(), result.GetDescription(), result.GetInnerValueForDebug());
            }
            else
            {
                seqNumberPing++;
            }
        }

        // 送信スレッド停止
        localClient.LocalSendFrameStop();
        // 受信スレッド停止
        localClient.LocalReceiveFrameStop();
        // 通信結果ログ出力
        localClient.LocalTraceConnetsRttInfo(true);

        localClient.LocalGetConnectInfo(&connectsInfo);
        WLANTEST_EXPECT_TRUE(connectsInfo.connectCount == g_ClientNum);

        for (int32_t i = 0; i < connectsInfo.connectCount; i++)
        {
            WLANTEST_EXPECT_TRUE(connectsInfo.facing[i].isConnected);
            WLANTEST_EXPECT_TRUE(connectsInfo.facing[i].rttAverage <= TestRttPlrRate);
        }

        Disconnect();

        WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::DeleteEthertypeFromRxEntry(&outRxId, TestEtherTypes[1]));

        NN_LOG("             RTT_%d_1_1 Client Frame Send Total Size:%lld(Range:%lld - %lld)\n",
            g_ClientNum + 1, connectsInfo.totalSendSize, sendSizeMin, sendSizeMax);

        NN_LOG("             [%s] END\n", ssidStr);
    }

    // 実行引数でCLIENT台数指定
    // RTT　1.2: 5GHz(36ch) MASTER
    TEST_F(LocalMasterFuncTest, RTT_M_1_2)
    {
        const int64_t agingTime                 = Time600s;
        const int32_t secBroadDummyPakets       = 90;
        const int32_t secBroadPingPakets        = 1;
        const int64_t sendBroadDummyInterval    = static_cast<int64_t>(Time1s / secBroadDummyPakets);
        const int64_t sendBroadPingInterval     = static_cast<int64_t>(Time1s / secBroadPingPakets);

        // 通信評価定義
        const uint32_t sendTotalNormal          = secBroadDummyPakets * WlanTestRTTPacketBufferSize * (agingTime / 1000);
        const uint32_t sendSizeMin              = sendTotalNormal - (sendTotalNormal * 0.1);
        const uint32_t sendSizeMax              = sendTotalNormal + (sendTotalNormal * 0.1);

        nn::wlan::MacAddress broadMac           = nn::wlan::MacAddress::CreateBroadcastMacAddress();
        std::unique_ptr<uint8_t[]> putBroadBuffer(new uint8_t[WlanTestRTTPacketBufferSize]);
        uint32_t seqNumberPing;
        uint32_t outRxId;
        nn::Result result;

        WLANTEST_ASSERT_TRUE(putBroadBuffer.get() != nullptr);

        char ssidStr[nn::wlan::Ssid::SsidHexStringLengthMax];
        nn::util::SNPrintf(ssidStr, nn::wlan::Ssid::SsidHexStringLengthMax, "RTT_%d_1_2", g_ClientNum + 1);
        NN_LOG("             [%s] START\n", ssidStr);

        const nn::wlan::MasterBssParameters fncTestMasterBss = {
                36, false, 30, true, nn::wlan::RateSetLegacy_11gMask, nn::wlan::RateSetLegacy_11gMask,
                { nn::wlan::SecurityMode_StaticAes, nn::wlan::SecurityMode_StaticAes, 0, LocalMasterAes16Key }, nn::wlan::Ssid(ssidStr), 100
        };

        if(ClientMaxConnect == g_ClientNum)
        {
            // ジョイコン1セット（各2台）
            WLANTEST_ASSERT_TRUE(localMaster.SetBtMode(LocalApiBtNode2));
        }
        else
        {
            // ジョイコン2セット（各4台）
            WLANTEST_ASSERT_TRUE(localMaster.SetBtMode(LocalApiBtNode4));
        }
        WLANTEST_ASSERT_TRUE(localMaster.SetupLocalMaster(fncTestMasterBss, TestMatchInfo, sizeof(TestMatchInfo) / sizeof(TestMatchInfo[0])));

        WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::AddEthertypeToRxEntry(localMaster.m_entRxId, TestEtherTypes[1]));

        // MASTER受信スレッド起動
        WLANTEST_ASSERT_TRUE(localMaster.LocalReceiveFrameStart(BufferSize200k, WlanTestRecvBufferSize, g_ClientNum, g_ClientNum, WlanTestAgingGetFrameCount));
        // CLIENT接続待機
        WLANTEST_ASSERT_TRUE(localMaster.LocalReceiveWait(WlanTestMasterWaitTimeOut, true));

        // MASTER送信スレッド起動
        WLANTEST_ASSERT_TRUE(localMaster.LocalBroadcastSendFrameStart(WlanTestRTTPacketBufferSize, agingTime, sendBroadDummyInterval));

        seqNumberPing = 0;
        sysTick = nn::os::GetSystemTick();
        // 600秒間通信
        while ((nn::os::GetSystemTick() - sysTick).ToTimeSpan().GetMilliSeconds() <= agingTime)
        {
            WLANTEST_EXPECT_TRUE(localMaster.LocalMakeFrame(putBroadBuffer.get(), WlanTestRTTPacketBufferSize, broadMac, localMaster.m_macAddress, TestEtherTypes[1], TestMatchInfo[0], seqNumberPing, true));

            result = nn::wlan::Local::PutFrameRaw(putBroadBuffer.get(), WlanTestRTTPacketBufferSize);
            WLANTEST_EXPECT_RESULT_SUCCESS(result);
            if (result.IsSuccess() != true)
            {
                NN_LOG("             nn::wlan::Local::PutFrameRaw Broad Send(Count:%d) Error(%d:%d) 0x%08x\n", seqNumberPing,
                        result.GetModule(), result.GetDescription(), result.GetInnerValueForDebug());
            }
            else
            {
                seqNumberPing++;
                nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(sendBroadPingInterval));
            }
        }

        // MASTER送信スレッド停止
        localMaster.LocalSendFrameStop();

        // MASTER受信スレッド停止
        localMaster.LocalReceiveFrameStop();

        localMaster.LocalTraceConnetsRttInfo(true);

        WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::DeleteEthertypeFromRxEntry(&outRxId, TestEtherTypes[1]));

        localMaster.LocalGetConnectInfo(&connectsInfo);
        WLANTEST_EXPECT_TRUE(connectsInfo.connectCount >= g_ClientNum);

        for (int32_t i = 0; i < connectsInfo.connectCount; i++)
        {
            WLANTEST_EXPECT_TRUE(connectsInfo.facing[i].isConnected);
            WLANTEST_EXPECT_TRUE(connectsInfo.facing[i].rttAverage <= TestRttPlrRate);
        }

        NN_LOG("             RTT_%d_1_2 Master Frame Send Total Size:%lld(Range:%lld - %lld)\n",
            g_ClientNum + 1, connectsInfo.totalSendSize, sendSizeMin, sendSizeMax);

        NN_LOG("             [%s] END\n", ssidStr);
    }

    // 実行引数でCLIENT台数指定
    // RTT　1.2: 5GHz(36ch) CLIENT
    TEST_F(LocalClientFuncTest, RTT_C_1_2)
    {
        const int64_t agingTime                 = Time600s;
        const int32_t secBroadDummyPakets       = 90;
        const int32_t secBroadPingPakets        = 1;
        const int64_t sendBroadDummyInterval    = static_cast<int64_t>(Time1s / secBroadDummyPakets);
        const int64_t sendBroadPingInterval     = static_cast<int64_t>(Time1s / secBroadPingPakets);

        // 通信評価定義
        const uint32_t sendTotalNormal          = secBroadDummyPakets * WlanTestRTTPacketBufferSize * (agingTime / 1000);
        const uint32_t sendSizeMin              = sendTotalNormal - (sendTotalNormal * 0.1);
        const uint32_t sendSizeMax              = sendTotalNormal + (sendTotalNormal * 0.1);

        nn::wlan::MacAddress broadMac = nn::wlan::MacAddress::CreateBroadcastMacAddress();
        std::unique_ptr<uint8_t[]> putBroadBuffer(new uint8_t[WlanTestRTTPacketBufferSize]);
        nn::wlan::ConnectionStatus clientStatus;
        uint32_t seqNumberPing;
        uint32_t outRxId;
        nn::Result result;

        WLANTEST_ASSERT_TRUE(putBroadBuffer.get() != nullptr);

        char ssidStr[nn::wlan::Ssid::SsidHexStringLengthMax];
        nn::util::SNPrintf(ssidStr, nn::wlan::Ssid::SsidHexStringLengthMax, "RTT_%d_1_2", g_ClientNum + 1);
        NN_LOG("             [%s] START\n", ssidStr);

        const TestLocalConnectParams ConnectParam = {
                nn::wlan::Ssid(ssidStr), nn::wlan::MacAddress::CreateBroadcastMacAddress(), 36,
                { nn::wlan::SecurityMode_StaticAes, nn::wlan::SecurityMode_StaticAes, 0, LocalMasterAes16Key }, true, nn::wlan::BeaconIndication_Enable, 10
        };

        if(ClientMaxConnect == g_ClientNum)
        {
            // ジョイコン1セット（各2台）
            WLANTEST_ASSERT_TRUE(localClient.SetBtMode(LocalApiBtNode2));
        }
        else
        {
            // ジョイコン2セット（各4台）
            WLANTEST_ASSERT_TRUE(localClient.SetBtMode(LocalApiBtNode4));
        }
        WLANTEST_ASSERT_TRUE(localClient.SetupLocalClient(ConnectParam.ssid, ConnectParam.security, TestMatchInfo, sizeof(TestMatchInfo) / sizeof(TestMatchInfo[0]), true, false));
        WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::AddEthertypeToRxEntry(localClient.m_entRxId, TestEtherTypes[1]));

        Connect(ConnectParam);
        WLANTEST_EXPECT_RESULT_SUCCESS(nn::wlan::Local::GetMacAddress(&localClient.m_macAddress));

        // 受信スレッド起動
        WLANTEST_ASSERT_TRUE(localClient.LocalReceiveFrameStart(BufferSize200k, WlanTestRecvBufferSize, 1, 1, WlanTestAgingGetFrameCount));
        // MASTER接続待機
        WLANTEST_ASSERT_TRUE(localClient.LocalReceiveWait(WlanMaxClientWaitMsecTimeOut, Time1s, true));

        // 送信スレッド起動
        WLANTEST_ASSERT_TRUE(localClient.LocalBroadcastSendFrameStart(WlanTestRTTPacketBufferSize, agingTime, sendBroadDummyInterval));

        seqNumberPing = 0;
        while (NN_STATIC_CONDITION(1))
        {
            // MASTERからの接続が切られるまで送信し続ける
            if (nn::os::TimedWaitSystemEvent(&connectionEvent, nn::TimeSpan::FromMilliSeconds(sendBroadPingInterval)) == true)
            {
                WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::GetConnectionStatus(&clientStatus));
                if (clientStatus.state != nn::wlan::ConnectionState_Connected)
                {
                    break;
                }
            }

            WLANTEST_EXPECT_TRUE(localClient.LocalMakeFrame(putBroadBuffer.get(), WlanTestRTTPacketBufferSize, broadMac, localClient.m_macAddress, TestEtherTypes[1], TestMatchInfo[0], seqNumberPing, true));

            result = nn::wlan::Local::PutFrameRaw(putBroadBuffer.get(), WlanTestRTTPacketBufferSize);
            WLANTEST_EXPECT_RESULT_SUCCESS(result);
            if (result.IsSuccess() != true)
            {
                NN_LOG("             nn::wlan::Local::PutFrameRaw Broad Send(Count:%d) Error(%d:%d) 0x%08x\n", seqNumberPing,
                        result.GetModule(), result.GetDescription(), result.GetInnerValueForDebug());
            }
            else
            {
                seqNumberPing++;
            }
        }

        // 送信スレッド停止
        localClient.LocalSendFrameStop();
        // 受信スレッド停止
        localClient.LocalReceiveFrameStop();
        // 通信結果ログ出力
        localClient.LocalTraceConnetsRttInfo(true);

        localClient.LocalGetConnectInfo(&connectsInfo);
        WLANTEST_EXPECT_TRUE(connectsInfo.connectCount == g_ClientNum);

        for (int32_t i = 0; i < connectsInfo.connectCount; i++)
        {
            WLANTEST_EXPECT_TRUE(connectsInfo.facing[i].isConnected);
            WLANTEST_EXPECT_TRUE(connectsInfo.facing[i].rttAverage <= TestRttPlrRate);
        }

        Disconnect();

        WLANTEST_ASSERT_RESULT_SUCCESS(nn::wlan::Local::DeleteEthertypeFromRxEntry(&outRxId, TestEtherTypes[1]));

        NN_LOG("             RTT_%d_1_2 Client Frame Send Total Size:%lld(Range:%lld - %lld)\n",
            g_ClientNum + 1, connectsInfo.totalSendSize, sendSizeMin, sendSizeMax);

        NN_LOG("             [%s] END\n", ssidStr);
    }

    void GetParam(int argc, char** argv)
    {
        for (int i = 0; i < argc; i++)
        {
            char* search = strstr(argv[i], IpAddressParam);
            if (search != nullptr)
            {
                int offset = nn::util::Strnlen(IpAddressParam, sizeof(IpAddressParam));
                g_IpOct4 = strtol(&search[offset], 0, 10);
                continue;
            }

            search = strstr(argv[i], ClientNum);
            if (search != nullptr)
            {
                int offset = nn::util::Strnlen(ClientNum, sizeof(ClientNum));
                g_ClientNum = strtol(&search[offset], 0, 10);
                continue;
            }

            search = strstr(argv[i], ScanTime);
            if (search != nullptr)
            {
                int offset = nn::util::Strnlen(ScanTime, sizeof(ScanTime));
                g_ScanTime = strtol(&search[offset], 0, 10);
                g_SetScanTime = true;
                NN_LOG("             Set Parameter ScanTime : %d ms\n", g_ScanTime);
                continue;
            }

            search = strstr(argv[i], JoyconPairing);
            if (search != nullptr)
            {
                g_SetJoyconPairing = true;
                NN_LOG("             Set Joycon Pairing Parameter\n");
                continue;
            }
        }
    }

extern "C" void nnMain()
{
    int     argc = nn::os::GetHostArgc();
    char**  argv = nn::os::GetHostArgv();
    NN_LOG("argc=%d\n", argc);
    for (int i = 0; i < argc; i++)
    {
        if (!strcmp(argv[i], "--short"))
        {
            g_IsShort = true;
        }
        NN_LOG("argv[%d]=%s\n", i, argv[i]);
    }

    GetParam(argc, argv);
    ::testing::InitGoogleTest(&argc, argv);

    int result = RUN_ALL_TESTS();
    nnt::Exit(result);
}

} // namespace WlanTest
