﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <memory>
#include <algorithm>
#include <nnt.h>
#include <nnt/result/testResult_Assert.h>

#include "../../Common/testWlan_localApiClass.h"
#include "../../Common/testWlan_UnitTest.h"
#include "../../Common/testWlan_UnitTestCommon.h"

#include <nn/nifm/nifm_ApiForNetworkMiddleware.h>
#include <nn/init.h>
#include <nn/settings/fwdbg/settings_SettingsSetterApi.h>

namespace {
    const char SsidDefault[]                = "WlanTest";

    const char SsidParam[]                  = "-ssid=";
    const char OpenSecurtyMode[]            = "-open";
    const char AesSecurtyMode[]             = "-aes";
    const char Wep64SecurtyMode[]           = "-w64";
    const char Wep128SecurtyMode[]          = "-w128";
    const char Stealth[]                    = "-stealth";
    const char SendReceiveParam[]           = "-sr=";
    const char FrameTypeParam[]             = "-af";
    const char ChannelParam[]               = "-ch=";
    const char ConnectOnryParam[]           = "-c";
    const char BeaconParam[]                = "-bcon=";
    const char BtNodeParam[]                = "-bt=";
    const char TimeoutParam[]               = "-t=";

    const uint32_t SendReceiveDefaultCount  = 10;
    const size_t WlanTestBufferSize         = 512;
    const uint32_t CancelFrameTimeout       = 5;

    const size_t StackSize                  = 64 * 1024;

    const bool IsEnabled = true;

    // localmasterの設定パラメータ
    const nn::wlan::MasterBssParameters TestMasterBss = {
        -1, false, 30, false, nn::wlan::RateSetLegacy_11gMask, nn::wlan::RateSetLegacy_11gMask,
        { nn::wlan::SecurityMode_Open, nn::wlan::SecurityMode_Open, 0, "" }, nn::wlan::Ssid(), 100
    };

    enum ThreadEvent
    {
        EventType_CancelFrame = 0,
        EventType_Stop,
        EventType_End
    };

    enum ThreadStackType
    {
        FrameRecvStack = 0,
        ActionRecvStack,
        CancelStack,
        TimeoutStack,
        StackEnd
    };
}

// Master通信対向アプリケーション
namespace WlanTest {

    class WlanTestMsterFacingMachine : public LocalApiClass
    {
    public:
        void GetParam(int argc, char** argv)
        {
            char argSsid[Wlantest_ssidlengthmax + 1];

            // defalut setting
            ssid.Set(SsidDefault);
            sendReceiveCount = SendReceiveDefaultCount;
            bss = TestMasterBss;
            bss.ssid = ssid;
            btNode = LocalApiBtNodeNone;
            isActionFrame = false;
            isConnectOnly = false;
            timeout = 0;
            memset(argSsid, 0x00, sizeof(argSsid));

            for (int i = 0; i < argc - 1; i++)
            {
                char* search = strstr(argv[i], SsidParam);
                size_t size = 0;

                // SSID(Master of SSID)
                search = strstr(argv[i], SsidParam);
                if (search != nullptr)
                {
                    size = Wlantest_ssidlengthmax;
                    if (size > strlen(argv[i]) - strlen(SsidParam))
                    {
                        size = strlen(argv[i]) - strlen(SsidParam);
                    }
                    memcpy(argSsid, &search[strlen(SsidParam)], size);
                    bss.ssid.Set(argSsid);
                    ssid.Set(argSsid);
                    continue;
                }

                // long value Parameters Setting
                if (SetLongParam(argv[i]) == true)
                {
                    continue;
                }

                // Security Prameters Setting
                if (SetSecurityParam(argv[i]) == true)
                {
                    continue;
                }

                // frame type
                search = strstr(argv[i], FrameTypeParam);
                if (search != nullptr)
                {
                    isActionFrame = true;
                }

                // Client Connect Only
                search = strstr(argv[i], ConnectOnryParam);
                if (search != nullptr)
                {
                    isConnectOnly = true;
                }
            }
        }

        bool SetLongParam(char* pLongParam)
        {
            char* search;

            // Channel
            search = strstr(pLongParam, ChannelParam);
            if (search != nullptr)
            {
                bss.channel = strtol(&search[strlen(ChannelParam)], 0, 10);
                return true;
            }
            // Beacon Time
            search = strstr(pLongParam, BeaconParam);
            if (search != nullptr)
            {
                bss.beaconInterval = strtol(&search[strlen(BeaconParam)], 0, 10);
                return true;
            }
            // Timeout
            search = strstr(pLongParam, TimeoutParam);
            if (search != nullptr)
            {
                timeout = strtol(&search[strlen(TimeoutParam)], 0, 10);
                return true;
            }

            // PutGetFrame Count
            search = strstr(pLongParam, SendReceiveParam);
            if (search != nullptr)
            {
                sendReceiveCount = strtol(&search[strlen(SendReceiveParam)], 0, 10);
                return true;
            }
            // Btm Setting
            search = strstr(pLongParam, BtNodeParam);
            if (search != nullptr)
            {
                btNode = strtol(&search[strlen(BtNodeParam)], 0, 10);
                return true;
            }
            return false;
        }

        bool SetSecurityParam(char* pSecurityParam)
        {
            char* search;

            // Security Open
            search = strstr(pSecurityParam, OpenSecurtyMode);
            if (search != nullptr)
            {
                bss.security.privacyMode = nn::wlan::SecurityMode_Open;
                bss.security.groupPrivacyMode = nn::wlan::SecurityMode_Open;
                return true;
            }
            // Security AES
            search = strstr(pSecurityParam, AesSecurtyMode);
            if (search != nullptr)
            {
                bss.security.privacyMode = nn::wlan::SecurityMode_StaticAes;
                bss.security.groupPrivacyMode = nn::wlan::SecurityMode_StaticAes;
                bss.security.keyIdx = 0;
                std::memcpy(bss.security.key, LocalMasterPasKey, sizeof(LocalMasterPasKey));
                return true;
            }
            // Security Wep64
            search = strstr(pSecurityParam, Wep64SecurtyMode);
            if (search != nullptr)
            {
                bss.security.privacyMode = nn::wlan::SecurityMode_Wep64Open;
                bss.security.groupPrivacyMode = nn::wlan::SecurityMode_Wep64Open;
                bss.security.keyIdx = 0;
                std::memcpy(bss.security.key, LocalMasterWep64PasKey, sizeof(LocalMasterWep64PasKey));
                return true;
            }
            // Security Wep128
            search = strstr(pSecurityParam, Wep128SecurtyMode);
            if (search != nullptr)
            {
                bss.security.privacyMode = nn::wlan::SecurityMode_Wep128Open;
                bss.security.groupPrivacyMode = nn::wlan::SecurityMode_Wep128Open;
                bss.security.keyIdx = 0;
                std::memcpy(bss.security.key, LocalMasterWep128PasKey, sizeof(LocalMasterWep128PasKey));
                return true;
            }
            // Stealth
            search = strstr(pSecurityParam, Stealth);
            if (search != nullptr)
            {
                bss.hiddenSsid = true;
                return true;
            }
            return false;
        }

        bool Setup(const nn::wlan::ReceivedDataMatchInfo matchInfo[], uint32_t matchCount)
        {
            matchData = matchInfo[0];
            bool isResult = false;

            for (auto etherType : TestEtherTypes)
            {
                if (AddEtherType(etherType) != true)
                {
                    NN_LOG("             %s ---> Add EtherType Error\n", __FUNCTION__);
                    return false;
                }
            }

            // ジョイコン台数が0台未満又は、8台を超える場合は無効
            if (btNode < LocalApiBtNodeNone || btNode > LocalApiBtNode8)
            {
                btNode      = LocalApiBtNodeNone;
            }

            // ジョイコン設定
            isResult = SetBtMode(btNode);
            NN_LOG("             %s ---> Bt Count(%d) Result : %s\n", __FUNCTION__, btNode, isResult == true ? "Pass" : "Failler");

            return SetupLocalMaster(bss, matchInfo, matchCount, isActionFrame);
        }

        static void TimeoutThread(void* arg)
        {
            WlanTestMsterFacingMachine* pThis = static_cast<WlanTestMsterFacingMachine*>(arg);
            nn::os::TimerEventType event;
            nn::os::InitializeTimerEvent(&event, nn::os::EventClearMode_AutoClear);
            nn::os::StartOneShotTimerEvent(&event, nn::TimeSpan::FromSeconds(pThis->timeout));
            nn::os::WaitTimerEvent(&event);
            NN_LOG("             %s ---> TIMEOUT\n", __FUNCTION__);

            if(pThis->isActionFrame == true)
            {
                nn::wlan::Local::CancelGetActionFrame(pThis->m_actionRxId);
            }

            pThis->isTimeout = true;
        }

        static void CancelFrameThread(void* arg)
        {
            bool isLoop = true;
            WlanTestMsterFacingMachine* pThis = static_cast<WlanTestMsterFacingMachine*>(arg);

            while (isLoop == true)
            {
                pThis->cancelEvent.WaitSygnal();
                pThis->cancelEvent.SetSygnal();

                for (int i = 0; i < CancelFrameTimeout; i++)
                {
                    if (pThis->threadEvent != ThreadEvent::EventType_CancelFrame)
                    {
                        break;
                    }
                    nn::os::SleepThread(nn::TimeSpan::FromSeconds(1));
                }
                switch (pThis->threadEvent)
                {
                case ThreadEvent::EventType_CancelFrame:
                    nn::wlan::Local::CancelGetFrame(pThis->m_entRxId);
                    break;
                case ThreadEvent::EventType_End:
                    nn::wlan::Local::CancelGetFrame(pThis->m_entRxId);
                    isLoop = false;
                    break;
                case ThreadEvent::EventType_Stop:
                default:
                    break;
                }
            }
        }

        static void FrameReceiveThread(void* arg)
        {
            WlanTestMsterFacingMachine* pThis = static_cast<WlanTestMsterFacingMachine*>(arg);
            std::unique_ptr<uint8_t[]> getBuffer(new uint8_t[WlanTestBufferSize]);
            bool isLoop = true;
            size_t outSize;
            nn::Result result;

            while (isLoop == true)
            {
                result = nn::wlan::Local::GetFrameRaw(getBuffer.get(), WlanTestBufferSize, &outSize, pThis->m_entRxId);
                if (result.IsSuccess() != true || pThis->threadEvent == ThreadEvent::EventType_End)
                {
                    isLoop = false;
                    if (result.IsSuccess() != true)
                    {
                        NN_LOG("    Local Master ---> Client GetFrame Fail(0x%08x).\n", result.GetInnerValueForDebug());
                    }
                }
                else
                {
                    NN_LOG("    Local Master ---> Client GetFrame Size %lu Byte\n", outSize);
                }

            }
        }

        static void ActionFrameReceiveThread(void* arg)
        {
            WlanTestMsterFacingMachine* pThis = static_cast<WlanTestMsterFacingMachine*>(arg);
            std::unique_ptr<uint8_t[]> getBuffer(new uint8_t[WlanTestBufferSize]);
            nn::wlan::MacAddress outMac;
            bool isLoop = true;
            size_t outSize;
            nn::Result result;

            NN_LOG("    Local Master ---> Start ActionFrameReceiveThread\n");
            while (isLoop == true)
            {
                result = nn::wlan::Local::GetActionFrame(&outMac, getBuffer.get(), WlanTestBufferSize, &outSize, pThis->m_actionRxId);
                if (result.IsSuccess() != true || pThis->threadEvent == ThreadEvent::EventType_End)
                {
                    isLoop = false;
                    if (result.IsSuccess() != true)
                    {
                        NN_LOG("    Local Master ---> Client GetActionFrame Fail(0x%08x).\n", result.GetInnerValueForDebug());
                    }
                }
                else
                {
                    NN_LOG("    Local Master ---> Client GetActionFrame Size %lu Byte\n", outSize);
                }
            }
            NN_LOG("    Local Master ---> End ActionFrameReceiveThread\n");
        }

        void TraceMasterStatus(nn::wlan::ConnectionStatus& connectionStatus)
        {
            char strMacAddress[nn::wlan::MacAddress::MacStringSize + 1];
            char strSsid[nn::wlan::Ssid::SsidHexStringLengthMax];
            connectionStatus.bssid.GetString(strMacAddress);

            NN_LOG("\n                 ***** MASTER BSS INFO *****\n");
            NN_LOG("                   SSID               : %s\n", bss.ssid.GetHexString(strSsid));
            NN_LOG("                   CHANNEL            : %d\n", bss.channel);
            NN_LOG("                   STEALTH            : %d\n", bss.hiddenSsid);
            NN_LOG("                   INACTIVE PERIOD    : %lu\n", bss.inactivePeriod);
            NN_LOG("                   SUPPORTED RATES    : 0x%08x\n", bss.supportedRates);
            NN_LOG("                   BASIC RATES        : %lu\n", bss.basicRates);
            NN_LOG("                   SECURITY GROUP     : %s\n", SecurityTypeString[bss.security.groupPrivacyMode]);
            NN_LOG("                   SECURITY PRIVACY   : %s\n", SecurityTypeString[bss.security.privacyMode]);
            NN_LOG("                   SECURITY KEYIDX    : %d\n", bss.security.keyIdx);
            NN_LOG("                   BEACON INTERVAL    : %d\n", bss.beaconInterval);
            NN_LOG("                 ***** MASTER BSS INFO *****\n");

            NN_LOG("\n                 ***** MASTER STATUS INFO *****\n");
            NN_LOG("                   SSID               : %s\n", connectionStatus.ssid.GetSsidData());
            NN_LOG("                   MAC ADDRESS        : %s\n", strMacAddress);
            NN_LOG("                   CHANNEL            : %d\n", connectionStatus.channel);
            NN_LOG("                   CAPABILITY INFO    : %d\n", connectionStatus.capabilityInfo);
            NN_LOG("                   STATE              : %d\n", connectionStatus.state);
            NN_LOG("                 ***** MASTER STATUS INFO *****\n");
        }

        void ConnectOnlyLoop()
        {
            uint32_t connectClientCnt = 0;
            while (NN_STATIC_CONDITION(1))
            {
                if (nn::os::TimedWaitSystemEvent(&m_connectEvent, nn::TimeSpan::FromMilliSeconds(20)) == true)
                {
                    connectClientCnt = LocalMasterConnectWait(clientMacAddress, 3, 1);
                    NN_LOG("    Local Master ---> Client Event Sygnal Connect(%d)\n", connectClientCnt);
                    if (connectClientCnt == 0)
                    {
                        break;
                    }
                }
            }
        }

        void SendFrame(uint8_t buffer[], size_t size, uint32_t connectClientCnt)
        {
            nn::Result result;

            result = nn::wlan::Local::GetMacAddress(&macAddress);
            if (result.IsSuccess() != true)
            {
                NN_LOG("    Local Master ---> GetMacAddress Fail(0x%08x)\n", result.GetInnerValueForDebug());
            }

            for (int c = 0; c < connectClientCnt; c++)
            {
                for (auto etherType : TestEtherTypes)
                {
                    if (LocalMakeFrame(buffer, size, clientMacAddress[c], macAddress, etherType, matchData) != true)
                    {
                        continue;
                    }
                    if ((result = nn::wlan::Local::PutFrameRaw(buffer, size)).IsSuccess() == true)
                    {
                        NN_LOG("    Local Master ---> Client PutFrameRaw Size %lu Byte\n", size);
                    }
                    else
                    {
                        NN_LOG("    Local Master ---> Client PutFrameRaw Error Module:%d Description%d\n", result.GetModule(), result.GetDescription());
                    }
                    nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(100));
                }
            }
        }

        nn::Result CreateTestThreads(uint8_t threadStacks[StackEnd][StackSize])
        {
            nn::Result result;

            cancelEvent.SetEvent(false, nn::os::EventClearMode_ManualClear);

            // Timeout が設定されている場合
            if (timeout != 0)
            {
                nn::os::InitializeTimerEvent(&timeoutEvent, nn::os::EventClearMode_AutoClear);
                result = nn::os::CreateThread(&timeoutThread, TimeoutThread, this, threadStacks[TimeoutStack], StackSize, nn::os::DefaultThreadPriority - 5);
                if (result.IsSuccess() != true)
                {
                    return result;
                }
                nn::os::StartThread(&timeoutThread);
            }

            result = nn::os::CreateThread(&cancelThread, CancelFrameThread, this, threadStacks[CancelStack], StackSize, nn::os::DefaultThreadPriority);
            if (result.IsSuccess() != true)
            {
                return result;
            }
            nn::os::StartThread(&cancelThread);

            if (isConnectOnly != true)
            {
                result = nn::os::CreateThread(&receiveThread, FrameReceiveThread, this, threadStacks[FrameRecvStack], StackSize, nn::os::DefaultThreadPriority);
                if (result.IsSuccess() != true)
                {
                    return result;
                }
                nn::os::StartThread(&receiveThread);
                if (isActionFrame == true)
                {
                    result = nn::os::CreateThread(&getActionFrameThread, ActionFrameReceiveThread, this, threadStacks[ActionRecvStack], StackSize, nn::os::DefaultThreadPriority);
                    if (result.IsSuccess() != true)
                    {
                        return result;
                    }
                    nn::os::StartThread(&getActionFrameThread);
                }
            }
            return result;
        }

        void DestroyTestThread()
        {
            threadEvent = ThreadEvent::EventType_End;
            cancelEvent.SetSygnal();
            nn::os::WaitThread(&cancelThread);
            nn::os::DestroyThread(&cancelThread);
            NN_LOG("    Local Master ---> cancel Thread Stop\n", __FUNCTION__);

            if (isConnectOnly != true)
            {
                nn::os::WaitThread(&receiveThread);
                nn::os::DestroyThread(&receiveThread);
                NN_LOG("    Local Master ---> receive Thread Stop\n", __FUNCTION__);
                if (isActionFrame == true)
                {
                    nn::os::WaitThread(&getActionFrameThread);
                    nn::os::DestroyThread(&getActionFrameThread);
                }
            }

            // Timeout が設定されている場合
            if (timeout != 0)
            {
                nn::os::SignalTimerEvent(&timeoutEvent);
                nn::os::WaitThread(&timeoutThread);
                nn::os::DestroyThread(&timeoutThread);
                nn::os::FinalizeTimerEvent(&timeoutEvent);
            }
        }

        void PutGetFrame()
        {
            std::unique_ptr<uint8_t[]> putBuffer(new uint8_t[WlanTestBufferSize]);
            nn::os::Tick sysTick;
            int64_t waitTime = 10000;
            uint32_t connectClientCnt = 0;
            int sendPos = 0;
            bool isLoop = true;
            static NN_ALIGNAS(4096) uint8_t threadStacks[StackEnd][StackSize];

            nn::wlan::Local::GetConnectionStatus(&connectionStatus);
            NN_LOG("    Local Master ---> %s Start\n", __FUNCTION__);
            TraceMasterStatus(connectionStatus);

            if (CreateTestThreads(threadStacks).IsSuccess() != true)
            {
                DestroyTestThread();
                return;
            }

            if (isActionFrame == true)
            {
                nn::Result result;

                isLoop = LocalMakeActionFrame(putBuffer.get(), WlanTestBufferSize,
                    nn::wlan::MacAddress::CreateBroadcastMacAddress(), m_macAddress,
                    TestEtherTypes[0], matchData, nn::wlan::ActionFrameType_Beacon);

                result = nn::wlan::Local::SetActionFrameWithBeacon(putBuffer.get(), WlanTestBufferSize);
                if (result.IsSuccess() == true)
                {
                    NN_LOG("    Local Master ---> SetActionFrameWithBeacon Beacon Size %lu Byte\n", WlanTestBufferSize);
                }
                else
                {
                    NN_LOG("    Local Master ---> SetActionFrameWithBeacon Error Module:%d Description%d\n", result.GetModule(), result.GetDescription());
                }
            }

            while (isLoop == true)
            {
                if (sendPos >= sendReceiveCount || isTimeout == true)
                {
                    isLoop = false;
                    continue;
                }

                // Timeout が設定されている場合
                if(timeout != 0)
                {
                    waitTime = timeout;
                }
                NN_LOG("    Local Master ---> Client Event Sygnal Wait(%d sec)\n", waitTime);
                connectClientCnt = LocalMasterConnectWait(clientMacAddress, waitTime, 1);

                NN_LOG("    Local Master ---> Client Event Sygnal Connect(%d msec)\n", connectClientCnt);
                if (connectClientCnt == 0 && timeout != 0)
                {
                    break;
                }
                else if (connectClientCnt == 0)
                {
                    continue;
                }

                sysTick = nn::os::GetSystemTick();
                NN_LOG("    Local Master ---> Connected(%lld)\n", __FUNCTION__, (nn::os::GetSystemTick() - sysTick).ToTimeSpan().GetMilliSeconds());
                if (isConnectOnly == true)
                {
                    ConnectOnlyLoop();
                    continue;
                }

                for (int i = sendPos; i < sendReceiveCount; i++)
                {
                    NN_LOG("    Local Master ---> Start(%d / %d)\n", i + 1, sendReceiveCount);
                    if (nn::os::TimedWaitSystemEvent(&m_connectEvent, nn::TimeSpan::FromMilliSeconds(20)) == true)
                    {
                        int waitTime = (timeout != 0 ? timeout : 10);
                        connectClientCnt = LocalMasterConnectWait(clientMacAddress, waitTime, 1);
                        NN_LOG("    Local Master ---> Client Event Sygnal Connect(%d)\n", connectClientCnt);
                    }
                    if (connectClientCnt == 0 || isTimeout == true)
                    {
                        NN_LOG("    Local Master ---> Client Disconnected(%lld)\n", (nn::os::GetSystemTick() - sysTick).ToTimeSpan().GetMilliSeconds());
                        break;
                    }
                    sendPos++;
                    SendFrame(putBuffer.get(), WlanTestBufferSize, connectClientCnt);
                    NN_LOG("    Local Master ---> End\n");
                }
                nn::wlan::DisconnectClient dis;
                dis.clientMacAddress = nn::wlan::MacAddress::CreateBroadcastMacAddress();
                dis.reasonCode = nn::wlan::Dot11ReasonCode_UnspecifiedFailure;
                nn::wlan::Local::Disconnect(nn::wlan::LocalCommunicationMode_Master, &dis);
            }
            DestroyTestThread();
            NN_LOG("    Local Master ---> %s End\n", __FUNCTION__);
        }

    protected:
        nn::wlan::Ssid ssid;
        nn::wlan::MacAddress macAddress;
        nn::wlan::MacAddress putMacAddress;
        uint32_t sendReceiveCount;
        int32_t btNode;
        nn::wlan::Security security;
        nn::wlan::MasterBssParameters bss;
        nn::os::SystemEventType waitEvent;

        nn::wlan::MacAddress clientMacAddress[nn::wlan::ConnectableClientsCountMax];
        nn::wlan::ClientStatus status[nn::wlan::ConnectableClientsCountMax];
        nn::Bit32 clientStatusBitMap;

        nn::wlan::ConnectionStatus connectionStatus;

        ThreadEvent threadEvent;
        LocalAutoEvent cancelEvent;
        bool isActionFrame;
        bool isConnectOnly;
        int  timeout;
        bool isTimeout = false;

        nn::wlan::ReceivedDataMatchInfo matchData;

        nn::os::TimerEventType timeoutEvent;

        nn::os::ThreadType cancelThread;
        nn::os::ThreadType receiveThread;
        nn::os::ThreadType getActionFrameThread;
        nn::os::ThreadType timeoutThread;
    };

    void FachingMachineProc(WlanTestMsterFacingMachine* pLocalMaster)
    {
        if (pLocalMaster == nullptr)
        {
            NN_LOG("    Local Master ---> pLocalMaster null pointer Fail\n");
            return;
        }

        if (pLocalMaster->Setup(TestMatchInfo, 1) != true)
        {
            NN_LOG("    Local Master ---> Setup Fail\n");
            return;
        }

        pLocalMaster->PutGetFrame();

    }

    TEST(MaserFacingMachine, No1)
    {
        WlanTestMsterFacingMachine localMaster;
        nn::Result result;

        result = nn::nifm::Initialize();
        if (result.IsSuccess() != true)
        {
            NN_LOG("    Local Master ---> nifm Initialize Fail(0x%08x)\n", result.GetInnerValueForDebug());
            return;
        }

        nn::settings::fwdbg::SetSettingsItemValue("nifm", "is_communication_control_enabled_for_test", &IsEnabled, sizeof(IsEnabled));
        nn::nifm::SetWirelessCommunicationEnabledForTest(false);

        result = nn::wlan::InitializeLocalManager();
        if (result.IsSuccess() != true)
        {
            NN_LOG("    Local Master ---> InitializeLocalManager Fail(0x%08x)\n", result.GetInnerValueForDebug());
            return;
        }

        localMaster.InitializeBtm();

        result = nn::wlan::Local::OpenMasterMode();
        if (result.IsSuccess() != true)
        {
            NN_LOG("    Local Master ---> OpenMasterMode Fail(0x%08x)\n", result.GetInnerValueForDebug());
            return;
        }

        localMaster.GetParam(nn::os::GetHostArgc(), nn::os::GetHostArgv());

        FachingMachineProc(&localMaster);

        localMaster.LocalRelease();

        nn::wlan::Local::CloseMasterMode();
        result = nn::wlan::FinalizeLocalManager();
        if (result.IsSuccess() != true)
        {
            NN_LOG("    Local Master ---> FinalizeLocalManager Fail(0x%08x)\n", result.GetInnerValueForDebug());
        }
    }

    extern "C" void nnMain()
    {
        int argc = nn::os::GetHostArgc();
        char **argv = nn::os::GetHostArgv();
        NN_LOG("argc=%d\n", argc);
        for (int i = 0; i < argc; i++)
        {
            NN_LOG("argv[%d]=%s\n", argc, argv[i]);
        }
        NN_LOG("Running nnMain() from testMain_Main.cpp\n");
        ::testing::InitGoogleTest(&argc, argv);
        int exitCode = RUN_ALL_TESTS();
        nnt::Exit(exitCode);
    }

} // namespace WlanTest
