﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <memory>
#include <algorithm>
#include <nnt.h>
#include <nnt/result/testResult_Assert.h>

#include "../../Common/testWlan_localApiClass.h"
#include "../../Common/testWlan_UnitTest.h"
#include "../../Common/testWlan_UnitTestCommon.h"

#include <nn/nifm/nifm_ApiForNetworkMiddleware.h>
#include <nn/init.h>
#include <nn/settings/fwdbg/settings_SettingsSetterApi.h>

namespace {
    const char SsidDefault[]                = "WlanTest";

    const char SsidParam[]                  = "-ssid=";
    const char ChannelParam[]               = "-ch=";
    const char OpenSecurtyMode[]            = "-open";
    const char AesSecurtyMode[]             = "-aes";
    const char Wep64SecurtyMode[]           = "-w64";
    const char Wep128SecurtyMode[]          = "-w128";
    const char SendReceiveParam[]           = "-sr=";
    const char ConnectWaitTimeParam[]       = "-conwait=";
    const char FrameTypeParam[]             = "-af";
    const char BtNodeParam[]                = "-bt=";
    const char TimeoutParam[]               = "-t=";
    const uint32_t ConnectWaitTime          = 180;
    const uint32_t SendReceiveDefaultCount  = 10;
    const size_t WlanTestBufferSize         = 1512;

    const size_t StackSize                  = 64 * 1024;

    const nn::wlan::Security DefaultSecurity = {
        nn::wlan::SecurityMode_Open, nn::wlan::SecurityMode_Open, 0, ""
    };

    const uint16_t TestEtherType = 0x88B7;

    const bool IsEnabled = true;

    enum ThreadEvent
    {
        EventType_CancelFrame = 0,
        EventType_Stop,
        EventType_End
    };

    enum ThreadStackType
    {
        FrameRecvStack = 0,
        ActionRecvStack,
        CancelStack,
        TimeoutStack,
        StackEnd
    };
}

// Client通信対向アプリケーション
namespace WlanTest {

    class WlanTestClientFacingMachine : public LocalApiClass
    {
    public:
        void GetParam(int argc, char** argv)
        {
            char argSsid[Wlantest_ssidlengthmax + 1];

            memset(argSsid, 0x00, sizeof(argSsid));

            // defalut setting
            ssid.Set(SsidDefault);
            channel = -1;
            security = DefaultSecurity;
            waitTime = ConnectWaitTime;
            sendReceiveCount = SendReceiveDefaultCount;
            btNode = LocalApiBtNodeNone;
            isActionFrame = false;
            timeout = 0;

            for (int i = 0; i < argc - 1; i++)
            {
                char* search = strstr(argv[i], SsidParam);
                size_t size = 0;

                // SSID(Master of SSID)
                search = strstr(argv[i], SsidParam);
                if (search != nullptr)
                {
                    size = Wlantest_ssidlengthmax;
                    if (size > strlen(argv[i]) - strlen(SsidParam))
                    {
                        size = strlen(argv[i]) - strlen(SsidParam);
                    }
                    memcpy(argSsid, &search[strlen(SsidParam)], size);
                    ssid.Set(argSsid);

                    continue;
                }

                // long value Parameters Setting
                if (SetLongParam(argv[i]) == true)
                {
                    continue;
                }

                // Security Prameters Setting
                if (SetSecurityParam(argv[i]) == true)
                {
                    continue;
                }

                // frame type
                search = strstr(argv[i], FrameTypeParam);
                if (search != nullptr)
                {
                    isActionFrame = true;
                    continue;
                }

            }

        }

        bool SetLongParam(char* pLongParam)
        {
            char* search;

            // Channel
            search = strstr(pLongParam, ChannelParam);
            if (search != nullptr)
            {
                channel = strtol(&search[strlen(ChannelParam)], 0, 10);
                return true;
            }

            // Timeout
            search = strstr(pLongParam, TimeoutParam);
            if (search != nullptr)
            {
                timeout = strtol(&search[strlen(TimeoutParam)], 0, 10);
                return true;
            }

            // ConnectWaitTime
            search = strstr(pLongParam, ConnectWaitTimeParam);
            if (search != nullptr)
            {
                waitTime = strtol(&search[strlen(ConnectWaitTimeParam)], 0, 10);
                return true;
            }

            // PutGetFrame Count
            search = strstr(pLongParam, SendReceiveParam);
            if (search != nullptr)
            {
                sendReceiveCount = strtol(&search[strlen(SendReceiveParam)], 0, 10);
                return true;
            }

            // Btm Setting
            search = strstr(pLongParam, BtNodeParam);
            if (search != nullptr)
            {
                btNode = strtol(&search[strlen(BtNodeParam)], 0, 10);
                return true;
            }
            return false;
        }

        bool SetSecurityParam(char* pSecurityParam)
        {
            char* search;

            search = strstr(pSecurityParam, OpenSecurtyMode);
            if (search != nullptr)
            {
                security.privacyMode = nn::wlan::SecurityMode_Open;
                security.groupPrivacyMode = nn::wlan::SecurityMode_Open;
                return true;
            }

            search = strstr(pSecurityParam, AesSecurtyMode);
            if (search != nullptr)
            {
                security.privacyMode = nn::wlan::SecurityMode_StaticAes;
                security.groupPrivacyMode = nn::wlan::SecurityMode_StaticAes;

                security.keyIdx = 0;
                std::memcpy(security.key, LocalMasterPasKey, sizeof(LocalMasterPasKey));
                return true;
            }

            search = strstr(pSecurityParam, Wep64SecurtyMode);
            if (search != nullptr)
            {
                security.privacyMode = nn::wlan::SecurityMode_Wep64Open;
                security.groupPrivacyMode = nn::wlan::SecurityMode_Wep64Open;
                security.keyIdx = 0;
                std::memcpy(security.key, LocalMasterWep64PasKey, sizeof(LocalMasterWep64PasKey));
                return true;
            }

            search = strstr(pSecurityParam, Wep128SecurtyMode);
            if (search != nullptr)
            {
                security.privacyMode = nn::wlan::SecurityMode_Wep128Open;
                security.groupPrivacyMode = nn::wlan::SecurityMode_Wep128Open;
                security.keyIdx = 0;
                std::memcpy(security.key, LocalMasterWep128PasKey, sizeof(LocalMasterWep128PasKey));
                return true;
            }
            return false;
        }

        bool Setup(const nn::wlan::ReceivedDataMatchInfo matchInfo[], uint32_t matchCount)
        {
            matchData = matchInfo[0];
            bool isResult = false;

            for (auto etherType : TestEtherTypes)
            {
                if (AddEtherType(etherType) != true)
                {
                    NN_LOG("             %s ---> Add EtherType Error\n", __FUNCTION__);
                    return false;
                }
            }

            NN_LOG("\n                 ***** CLIENT SETUP INFO *****\n");
            NN_LOG("                   SSID               : %s\n", ssid.GetSsidData());
            NN_LOG("                   GROUP PRIVACY MODE : %d\n", security.groupPrivacyMode);
            NN_LOG("                   CAPABILITY INFO    : %d\n", security.privacyMode);
            NN_LOG("                   KEY                : %d\n", security.keyIdx);
            NN_LOG("                   PW                 : %s\n", security.key);
            NN_LOG("                 ***** CLIENT SETUP INFO *****\n");

            // ジョイコン台数が0台未満又は、8台を超える場合は無効
            if (btNode < LocalApiBtNodeNone || btNode > LocalApiBtNode8)
            {
                btNode = LocalApiBtNodeNone;
            }

            // ジョイコン設定
            isResult = SetBtMode(btNode);
            NN_LOG("             %s ---> Bt Count(%d) Result : %s\n", __FUNCTION__, btNode, isResult == true ? "Pass" : "Failler");

            if (isActionFrame == true)
            {
                return SetupLocalClient(ssid, security, matchInfo, matchCount, true, true);
            }
            else
            {
                return SetupLocalClient(ssid, security, matchInfo, matchCount, true, false);
            }
        }

        bool ConnectWait()
        {
            macAddress = nn::wlan::MacAddress::CreateBroadcastMacAddress();
            nn::Result conResult,statusResult;
            nn::os::Tick conTick;
            uint32_t retryCnt = 0;

            nn::wlan::Local::GetConnectionEvent(&waitEvent);
            conTick = nn::os::GetSystemTick();
            for (int32_t i = 0; i < waitTime / 10; i++)
            {
                conResult = nn::wlan::Local::Connect(ssid, macAddress, channel, security, false, nn::wlan::BeaconIndication_Enable, 10);
                if (conResult.IsSuccess() != true)
                {
                    NN_LOG("    Local Client Connect Error(%d:%d) 0x%08x\n", conResult.GetModule(), conResult.GetDescription(), conResult.GetInnerValueForDebug());
                    break;
                }
                NN_LOG("             Client Wait Time (%lld msec)\n", (nn::os::GetSystemTick() - conTick).ToTimeSpan().GetMilliSeconds());

                nn::os::WaitSystemEvent(&waitEvent);

                statusResult = nn::wlan::Local::GetConnectionStatus(&connectStatus);
                if (statusResult.IsSuccess() == true && connectStatus.state == nn::wlan::ConnectionState_Connected)
                {
                    NN_LOG("             Client Connect Time (%lld msec) Retry Count (%lu)\n",
                        (nn::os::GetSystemTick() - conTick).ToTimeSpan().GetMilliSeconds(), retryCnt);
                    LocalTraceCurrentInfo();
                    return true;
                }

                if (isTimeout == true)
                {
                    break;
                }

                retryCnt++;
            }
            LocalTraceCurrentInfo();

            return false;
        }

        static void CancelFrameThread(void* arg)
        {
            bool isLoop = true;
            WlanTestClientFacingMachine* pThis = static_cast<WlanTestClientFacingMachine*>(arg);

            while (isLoop == true)
            {
                pThis->cancelEvent.WaitSygnal();
                pThis->cancelEvent.ClearSygnal();

                switch (pThis->threadEvent)
                {
                case ThreadEvent::EventType_CancelFrame:
                case ThreadEvent::EventType_End:
                    nn::wlan::Local::CancelGetFrame(pThis->m_entRxId);

                    if (pThis->isActionFrame == true)
                    {
                        nn::wlan::Local::CancelGetActionFrame(pThis->m_actionRxId);
                    }

                    if (pThis->threadEvent == ThreadEvent::EventType_End)
                    {
                        isLoop = false;
                    }
                    break;
                case ThreadEvent::EventType_Stop:
                default:
                    break;
                }
            }
        }

        static void ActionFrameReceiveThread(void* arg)
        {
            WlanTestClientFacingMachine* pThis = static_cast<WlanTestClientFacingMachine*>(arg);
            std::unique_ptr<uint8_t[]> getBuffer(new uint8_t[WlanTestBufferSize]);
            nn::wlan::MacAddress outMac;
            bool isLoop = true;
            size_t outSize;
            nn::Result result;

            while (isLoop == true)
            {
                result = nn::wlan::Local::GetActionFrame(&outMac, getBuffer.get(), WlanTestBufferSize, &outSize, pThis->m_actionRxId);
                if (result.IsSuccess() != true || pThis->threadEvent == ThreadEvent::EventType_End)
                {
                    isLoop = false;
                    if (result.IsSuccess() != true)
                    {
                        NN_LOG("    Local Client ---> Client GetActionFrame Fail(0x%08x).\n", result.GetInnerValueForDebug());
                    }
                }
                else
                {
                    NN_LOG("    Local Client ---> Client GetActionFrame Size %lu Byte\n", outSize);
                }
            }
        }

        static void TimeoutThread(void* arg)
        {
            WlanTestClientFacingMachine* pThis = static_cast<WlanTestClientFacingMachine*>(arg);

            nn::os::StartOneShotTimerEvent(&pThis->timeoutEvent, nn::TimeSpan::FromSeconds(pThis->timeout));
            nn::os::WaitTimerEvent(&pThis->timeoutEvent);
            NN_LOG("             %s ---> TIMEOUT\n", __FUNCTION__);

            pThis->threadEvent = ThreadEvent::EventType_End;
            pThis->cancelEvent.SetSygnal();

            pThis->isTimeout = true;
        }

        nn::Result CreateTestThreads(uint8_t threadStacks[StackEnd][StackSize])
        {
            nn::Result result;

            cancelEvent.SetEvent(false, nn::os::EventClearMode_ManualClear);
            result = nn::os::CreateThread(&cancelThread, CancelFrameThread, this, threadStacks[CancelStack], StackSize, nn::os::DefaultThreadPriority);
            if (result.IsSuccess() != true)
            {
                return result;
            }
            nn::os::StartThread(&cancelThread);

            if (isActionFrame == true)
            {
                result = nn::os::CreateThread(&getActionFrameThread, ActionFrameReceiveThread, this, threadStacks[FrameRecvStack], StackSize, nn::os::DefaultThreadPriority);
                if (result.IsSuccess() != true)
                {
                    return result;
                }
                nn::os::StartThread(&getActionFrameThread);
            }

            // Timeout が設定されている場合
            if (timeout != 0)
            {
                nn::os::InitializeTimerEvent(&timeoutEvent, nn::os::EventClearMode_AutoClear);

                result = nn::os::CreateThread(&timeoutThread, TimeoutThread, this, threadStacks[TimeoutStack], StackSize, nn::os::DefaultThreadPriority - 5);
                if (result.IsSuccess() != true)
                {
                    return result;
                }
                nn::os::StartThread(&timeoutThread);
            }
            return result;
        }

        void DestroyTestThread()
        {
            isTimeout = true;
            threadEvent = ThreadEvent::EventType_End;
            cancelEvent.SetSygnal();
            if (cancelThread._state > nn::os::ThreadType::State_NotInitialized)
            {
                nn::os::WaitThread(&cancelThread);
                nn::os::DestroyThread(&cancelThread);
            }

            if (isActionFrame == true &&
                getActionFrameThread._state > nn::os::ThreadType::State_NotInitialized)
            {
                nn::os::WaitThread(&getActionFrameThread);
                nn::os::DestroyThread(&getActionFrameThread);
            }

            // Timeout が設定されている場合
            if (timeout != 0)
            {
                if (timeoutThread._state > nn::os::ThreadType::State_NotInitialized)
                {
                    nn::os::SignalTimerEvent(&timeoutEvent);
                    nn::os::WaitThread(&timeoutThread);
                    nn::os::DestroyThread(&timeoutThread);
                    nn::os::FinalizeTimerEvent(&timeoutEvent);
                }
            }
        }

        void PutGetFrame()
        {
            std::unique_ptr<uint8_t[]> putBuffer(new uint8_t[WlanTestBufferSize]);
            static NN_ALIGNAS(4096) uint8_t threadStacks[StackEnd][StackSize];
            bool isLoop = true;
            int sendPos = 0;
            nn::Result result;

            if (CreateTestThreads(threadStacks).IsSuccess() != true)
            {
                DestroyTestThread();
                return;
            }

            while (isLoop == true)
            {
                NN_LOG("    Local Client ---> Start(%d / %d)\n", sendPos + 1, sendReceiveCount);
                if (sendPos >= sendReceiveCount || isTimeout == true)
                {
                    isLoop = false;
                    continue;
                }

                if (ConnectWait() != true)
                {
                    NN_LOG("    Local Client ---> Connect Error End\n");
                    if(isTimeout == false)
                    {
                        continue;
                    }
                    else
                    {
                        break;
                    }
                }

                nn::wlan::Local::GetMacAddress(&macAddress);

                for (int i = sendPos; i < sendReceiveCount; i++)
                {
                    if (isTimeout == true)
                    {
                        break;
                    }

                    sendPos++;
                    result = nn::wlan::Local::GetConnectionStatus(&connectStatus);
                    if (result.IsSuccess() == true && connectStatus.state != nn::wlan::ConnectionState_Connected)
                    {
                        NN_LOG("    Local Client ---> Master Disconnected\n");
                        break;
                    }

                    if (LocalMakeFrame(putBuffer.get(), WlanTestBufferSize, nn::wlan::MacAddress::CreateBroadcastMacAddress(), macAddress, TestEtherType, matchData) != true)
                    {
                        NN_LOG("    Local Client ---> LocalMakeFrame Error\n");
                        break;
                    }

                    if ((result = nn::wlan::Local::PutFrameRaw(putBuffer.get(), WlanTestBufferSize)).IsSuccess() != true)
                    {
                        NN_LOG("    Local Client ---> PutFrameRaw Error Module:%d Description%d\n", result.GetModule(), result.GetDescription());
                        break;
                    }
                    nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(100));

                    NN_LOG("    Local Client ---> PutFrameRaw Send %dByte\n", WlanTestBufferSize);
                }

                nn::wlan::Local::Disconnect(nn::wlan::LocalCommunicationMode_ClientSpectator, nullptr);
                NN_LOG("    Local Client ---> End\n");
            }

            DestroyTestThread();
        }

    protected:
        nn::wlan::Ssid ssid;
        int16_t channel;
        nn::wlan::MacAddress macAddress;
        nn::wlan::MacAddress putMacAddress;
        uint32_t sendReceiveCount;
        uint64_t waitTime;
        int32_t btNode;
        nn::wlan::Security security;
        nn::os::SystemEventType waitEvent;

        nn::wlan::ClientStatus status[nn::wlan::ConnectableClientsCountMax];
        nn::Bit32 clientStatusBitMap;

        ThreadEvent threadEvent;
        LocalAutoEvent cancelEvent;
        bool isActionFrame;

        bool isTimeout = false;
        int  timeout;

        nn::wlan::ConnectionStatus connectStatus;
        nn::wlan::ReceivedDataMatchInfo matchData;

        nn::os::TimerEventType timeoutEvent;

        nn::os::ThreadType cancelThread;
        nn::os::ThreadType getActionFrameThread;
        nn::os::ThreadType timeoutThread;
    };

    void FachingMachineProc(WlanTestClientFacingMachine* pLocalClient)
    {
        if (pLocalClient == nullptr)
        {
            NN_LOG("    Local Client ---> pLocalClient null pointer Fail\n");
            return;
        }

        if (pLocalClient->Setup(TestMatchInfo, 1) != true)
        {
            NN_LOG("    Local Client ---> Setup Fail\n");
            return;
        }

        pLocalClient->PutGetFrame();
    }

TEST(MaserFacingMachine, No1)
{
    WlanTestClientFacingMachine localClient;
    nn::Result result;

    result = nn::nifm::Initialize();
    if (result.IsSuccess() != true)
    {
        NN_LOG("    Local Client ---> nifm Initialize Fail Module:%d Description%d\n", result.GetModule(), result.GetDescription());
        return;
    }

    nn::settings::fwdbg::SetSettingsItemValue("nifm", "is_communication_control_enabled_for_test", &IsEnabled, sizeof(IsEnabled));
    nn::nifm::SetWirelessCommunicationEnabledForTest(false);

    result = nn::wlan::InitializeLocalManager();
    if (result.IsSuccess() != true)
    {
        NN_LOG("    Local Client ---> InitializeLocalManager Module:%d Description%d\n", result.GetModule(), result.GetDescription());
        return;
    }

    localClient.InitializeBtm();

    result = nn::wlan::Local::OpenClientMode();
    if (result.IsSuccess() != true)
    {
        NN_LOG("    Local Client ---> OpenClientMode Fail Module:%d Description%d\n", result.GetModule(), result.GetDescription());
        return;
    }

    localClient.GetParam(nn::os::GetHostArgc(), nn::os::GetHostArgv());

    FachingMachineProc(&localClient);

    localClient.LocalRelease();

    nn::wlan::Local::CloseClientMode();
    nn::wlan::FinalizeLocalManager();
}

extern "C" void nnMain()
{
    int argc = nn::os::GetHostArgc();
    char **argv = nn::os::GetHostArgv();
    NN_LOG("argc=%d\n", argc);
    for (int i = 0; i < argc; i++)
    {
        NN_LOG("argv[%d]=%s\n", argc, argv[i]);
    }
    NN_LOG("Running nnMain() from testMain_Main.cpp\n");
    ::testing::InitGoogleTest(&argc, argv);
    int exitCode = RUN_ALL_TESTS();
    nnt::Exit(exitCode);
}

} // namespace WlanTest
