﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <string>
#include <cstdlib>
#include "../common.h"

namespace {

// Timer
char g_TimeStr[sizeof("00:00:00")];
int64_t g_TickFreq;
nn::os::Tick g_StartTick;

const int TargetApCount = 4;
struct TargetApInfo {
    nn::wlan::Ssid ssid;
    nn::wlan::Security security;
    int16_t channel;
};
TargetApInfo g_ApInfoList[TargetApCount];
int g_ConnectSuccessCount[TargetApCount];

nn::os::SystemEventType g_ConnectionEvent;
}

void InitializeTargetApInfo()
{
    /* AP0 */
    g_ApInfoList[3].ssid.Set("WLAN_TEST_WEP128_OPEN");
    g_ApInfoList[3].security.privacyMode = nn::wlan::SecurityMode_Wep128Open;
    g_ApInfoList[3].security.groupPrivacyMode = nn::wlan::SecurityMode_Wep128Open;
    g_ApInfoList[3].security.keyIdx = 0;
    std::strcpy(reinterpret_cast<char*>(g_ApInfoList[3].security.key), "30313233343536373839616263");  // ASCII string "0123456789abc"
    g_ApInfoList[3].channel = -1;

    /* AP1 */
    g_ApInfoList[1].ssid.Set("WLAN_TEST_WPA_TKIP");
    g_ApInfoList[1].security.privacyMode = nn::wlan::SecurityMode_WpaTkip;
    g_ApInfoList[1].security.groupPrivacyMode = nn::wlan::SecurityMode_WpaTkip;
    g_ApInfoList[1].security.keyIdx = 0;
    std::strcpy(reinterpret_cast<char*>(g_ApInfoList[1].security.key), "012345678");
    g_ApInfoList[1].channel = 6;

    /* AP2 */
    g_ApInfoList[2].ssid.Set("WLAN_TEST_OPEN_HIDDEN");
    g_ApInfoList[2].security.privacyMode = nn::wlan::SecurityMode_Open;
    g_ApInfoList[2].security.groupPrivacyMode = nn::wlan::SecurityMode_Open;
    g_ApInfoList[2].security.keyIdx = 0;
    g_ApInfoList[2].channel = 40;

    /* AP3 */
    g_ApInfoList[0].ssid.Set("WLAN_TEST_WPA2_AES");
    g_ApInfoList[0].security.privacyMode = nn::wlan::SecurityMode_Wpa2Aes;
    g_ApInfoList[0].security.groupPrivacyMode = nn::wlan::SecurityMode_Wpa2Aes;
    g_ApInfoList[0].security.keyIdx = 0;
    std::strcpy(reinterpret_cast<char*>(g_ApInfoList[0].security.key), "012345678");
    g_ApInfoList[0].channel = -1;

    for( int i = 0; i < TargetApCount; i++ )
    {
        g_ConnectSuccessCount[i] = 0;
    }

    nn::wlan::Infra::GetConnectionEvent(&g_ConnectionEvent);
}

nn::Result InfraConnect(int pattern)
{
    nn::Result result;
    nn::wlan::ConnectionStatus connectionStatus;

    WlanTest::ElapsedTime(g_TimeStr, g_StartTick, g_TickFreq);
    NN_LOG("[%s]Try to connect to AP[%d]\n", g_TimeStr, pattern);

    // Starts to connect
    result = nn::wlan::Infra::Connect(
            g_ApInfoList[pattern].ssid,
            nn::wlan::MacAddress::CreateBroadcastMacAddress(),
            g_ApInfoList[pattern].channel,
            g_ApInfoList[pattern].security,
            false);

    nn::os::WaitSystemEvent(&g_ConnectionEvent);

    // Get connection status
    nn::wlan::Infra::GetConnectionStatus(&connectionStatus);
    WlanTest::PrintConnectionStatus(&connectionStatus);
    if( connectionStatus.state == nn::wlan::ConnectionState_Connected )
    {
        g_ConnectSuccessCount[pattern]++;
        WlanTest::ElapsedTime(g_TimeStr, g_StartTick, g_TickFreq);
        NN_LOG("[%s]Connection success\n", g_TimeStr);

        nn::os::SleepThread(nn::TimeSpan::FromSeconds(3));

        // Disconnect
        nn::wlan::Infra::Disconnect();
        nn::os::WaitSystemEvent(&g_ConnectionEvent);
        nn::wlan::Infra::GetConnectionStatus(&connectionStatus);
        if( connectionStatus.state == nn::wlan::ConnectionState_Connected )
        {
            NN_ABORT("disconnect failed\n");
        }
    }
    else
    {
        WlanTest::ElapsedTime(g_TimeStr, g_StartTick, g_TickFreq);
        NN_LOG("[%s]Connection failed\n", g_TimeStr);
    }

    return result;
}

void InitializeWlanInfra()
{
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::wlan::InitializeInfraManager());
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::wlan::Infra::OpenMode());
}

void FinalizeWlanInfra()
{
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::wlan::Infra::CloseMode());
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::wlan::FinalizeInfraManager());
}

extern "C" void nnMain()
{
    NN_LOG("\n\n Start InfraConnectionRepeat Test\n\n");

    WlanTest::SystemInitialize();
    InitializeWlanInfra();
    InitializeTargetApInfo();

    g_TickFreq = nn::os::GetSystemTickFrequency();
    g_StartTick = nn::os::GetSystemTick();

    int AttemptCount = 0;
    while( 1 )
    {
        AttemptCount++;
        for( int i = 0; i < TargetApCount; i++ )
        {
            InfraConnect(i);
            nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(16));
        }
        NN_LOG("\n\n_______________________\n");
        for( int i = 0; i < TargetApCount; i++ )
        {
            NN_LOG("[%s] %d/%d\n", g_ApInfoList[i].ssid.GetSsidData(), g_ConnectSuccessCount[i], AttemptCount);
        }
        NN_LOG("\n\n");
    }

    FinalizeWlanInfra();
    WlanTest::SystemFinalize();

    NN_LOG("\n\n End InfraScanRepeat Test \n\n");
} //NOLINT(impl/function_size)

