﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nnt/nntest.h>

#include <nn/util/util_Arithmetic.h>
#include <nn/util/util_Constant.h>
#include <nn/util/util_MatrixApi.h>
#include <nn/util/util_QuaternionApi.h>
#include <nn/util/util_VectorApi.h>

#include "testUtil_Vector.h"

#if defined(NN_BUILD_CONFIG_COMPILER_SUPPORTS_VC)
// do{}while(0) 文を使うと、C4127（定数条件式に対する警告）が発生するため、これを無効化
#pragma warning( push )
#pragma warning( disable:4127 )
#endif

namespace
{
    const float FloatError = 0.000001f;
}

//
// クオータニオン
//
class MathQuaternionTest : public ::testing::Test
{
};

TEST_F(MathQuaternionTest, Identity)
{
    nn::util::Vector4fType quaternion1;
    nn::util::QuaternionIdentity(&quaternion1);

    NNT_UTIL_VECTOR4_EXPECT_EQ(quaternion1, 0.f, 0.f, 0.f, 1.f);
}

TEST_F(MathQuaternionTest, Add)
{
    nn::util::Vector4fType quaternion1;
    nn::util::VectorSet(&quaternion1, -3.f, 1.f, -2.f, 4.f);

    nn::util::Vector4fType quaternion2;
    nn::util::VectorSet(&quaternion2, -6.f, 2.f, 4.f, -8.f);

    nn::util::QuaternionAdd(&quaternion1, quaternion1, quaternion2);

    NNT_UTIL_VECTOR4_EXPECT_EQ(quaternion1, -9.f, 3.f, 2.f, -4.f);
}

TEST_F(MathQuaternionTest, Subtract)
{
    nn::util::Vector4fType quaternion1;
    nn::util::VectorSet(&quaternion1, -3.f, 1.f, -2.f, 4.f);

    nn::util::Vector4fType quaternion2;
    nn::util::VectorSet(&quaternion2, -6.f, 2.f, 4.f, -8.f);

    nn::util::QuaternionSubtract(&quaternion1, quaternion1, quaternion2);

    NNT_UTIL_VECTOR4_EXPECT_EQ(quaternion1, 3.f, -1.f, -6.f, 12.f);
}

TEST_F(MathQuaternionTest, Multiply)
{
    {
        nn::util::Vector4fType vector1;
        nn::util::VectorSet(&vector1, -3.f, 1.f, -2.f, 4.f);

        nn::util::QuaternionMultiply(&vector1, vector1, -1.f / 2.f);
        NNT_UTIL_VECTOR4_EXPECT_EQ(vector1, 1.5f, -0.5f, 1.f, -2.f);
    }

    {
        nn::util::Vector4fType quaternion1;
        nn::util::VectorSet(&quaternion1, -3.f, 1.f, -2.f, 4.f);

        nn::util::Vector4fType quaternion2;
        nn::util::VectorSet(&quaternion2, -6.f, 2.f, 4.f, -8.f);

        nn::util::QuaternionMultiply(&quaternion1, quaternion1, quaternion2);
        NNT_UTIL_VECTOR4_EXPECT_EQ(quaternion1, 8.f, 24.f, 32.f, -44.f);
    }
}

TEST_F(MathQuaternionTest, Divide)
{
    nn::util::Vector4fType quaternion1;
    nn::util::VectorSet(&quaternion1, -3.f, 1.f, -2.f, 4.f);

    nn::util::Vector4fType quaternion2;
    nn::util::VectorSet(&quaternion2, -6.f, 2.f, 4.f, -8.f);

    nn::util::QuaternionDivide(&quaternion1, quaternion1, quaternion2);

    NNT_UTIL_VECTOR4_EXPECT_NEARLY_EQ(quaternion1, 0.4666666984558106f, 0.0666666850447655f, 0.0000000000000000f, -0.1666666716337204f, FloatError);
}

TEST_F(MathQuaternionTest, Dot)
{
    nn::util::Vector4fType vector1;
    nn::util::VectorSet(&vector1, -3.f, 1.f, -2.f, 4.f);
    nn::util::Vector4fType vector2;
    nn::util::VectorSet(&vector2, -6.f, 2.f, 4.f, -8.f);

    EXPECT_EQ(-20.f, nn::util::QuaternionDot(vector1, vector2));
}

TEST_F(MathQuaternionTest, Normalize)
{
    {
        nn::util::Vector4fType vector1;
        nn::util::VectorSet(&vector1, -3.f, 1.f, -2.f, 4.f);

        float result = nn::util::QuaternionNormalize(&vector1, vector1);

        NNT_UTIL_VECTOR4_EXPECT_NEARLY_EQ(vector1, -0.54772255750f, 0.182574185835055f, -0.36514837167011f, 0.73029674334022f, FloatError);
        EXPECT_EQ(30.f, result);
    }

    {
        nn::util::Vector4fType vector1;
        nn::util::VectorSet(&vector1, 0.0001f, -0.0002f, 0.0003f, -0.0004f);

        float result = nn::util::QuaternionNormalize(&vector1, vector1);

        NNT_UTIL_VECTOR4_EXPECT_EQ(vector1, 0.f, 0.f, 0.f, 0.f);
        EXPECT_EQ(0.f, result);
    }
}

TEST_F(MathQuaternionTest, Inverse)
{
    {
        nn::util::Vector4fType quaternion1;
        nn::util::VectorSet(&quaternion1, -3.f, 1.f, -2.f, 4.f);

        nn::util::QuaternionInverse(&quaternion1, quaternion1);

        NNT_UTIL_VECTOR4_EXPECT_NEARLY_EQ(quaternion1, 0.1000000089406967f, -0.0333333350718021f, 0.0666666701436043f, 0.1333333402872086f, FloatError);
    }

    {
        nn::util::Vector4fType quaternion1;
        nn::util::VectorSet(&quaternion1, 0.f, 0.f, 0.f, 0.f);

        nn::util::Vector4fType quaternion2;
        nn::util::VectorSet(&quaternion2, -3.f, 1.f, -2.f, 4.f);

        nn::util::QuaternionInverse(&quaternion2, quaternion1);

        NNT_UTIL_VECTOR4_EXPECT_EQ(quaternion2, 0.f, 0.f, 0.f, 0.f);
    }
}

TEST_F(MathQuaternionTest, Exp)
{
    {
        nn::util::Vector4fType quaternion1;
        nn::util::VectorSet(&quaternion1, -3.f, 1.f, -2.f, 0.f);

        nn::util::QuaternionExp(&quaternion1, quaternion1);

        NNT_UTIL_VECTOR4_EXPECT_NEARLY_EQ(quaternion1, 0.4527637660503388f, -0.1509212553501129f, 0.3018425107002259f, -0.8252992033958435f, FloatError);
    }

    {
        nn::util::Vector4fType quaternion1;
        nn::util::VectorSet(&quaternion1, -0.00003f, 0.00001f, -0.00002f, 0.f);

        nn::util::QuaternionExp(&quaternion1, quaternion1);

        NNT_UTIL_VECTOR4_EXPECT_EQ(quaternion1, -0.00003f, 0.00001f, -0.00002f, 1.f);
    }
}

#if defined(NN_BUILD_CONFIG_COMPILER_SUPPORTS_VC)
// C4723（0 除算に対する警告）が発生するため、これを無効化
#pragma warning( push )
#pragma warning( disable:4723 )
#endif

TEST_F(MathQuaternionTest, LogN)
{
    {
        nn::util::Vector4fType quaternion1;
        nn::util::VectorSet(&quaternion1, -3.f, 1.f, -2.f, 0.f);

        nn::util::QuaternionLogN(&quaternion1, quaternion1);

        NNT_UTIL_VECTOR4_EXPECT_NEARLY_EQ(quaternion1, -1.2594389915466308f, 0.4198130071163178f, -0.8396260142326355f, 0.0000000000000000f, FloatError);
    }

    {
        nn::util::Vector4fType quaternion1;
        nn::util::VectorSet(&quaternion1, 0.f, 0.f, 0.f, 0.f);

        nn::util::QuaternionLogN(&quaternion1, quaternion1);

        NNT_UTIL_VECTOR4_EXPECT_EQ(quaternion1, 0.f, 0.f, 0.f, 0.f);
    }
}

#if defined(NN_BUILD_CONFIG_COMPILER_SUPPORTS_VC)
#pragma warning( pop )
#endif

TEST_F(MathQuaternionTest, Lerp)
{
    nn::util::Vector4fType quaternion1;
    nn::util::VectorSet(&quaternion1, -3.f, 1.f, -2.f, 4.f);
    nn::util::Vector4fType quaternion2;
    nn::util::VectorSet(&quaternion2, 6.f, -2.f, 4.f, -8.f);

    nn::util::Vector4fType quaternion3;

    nn::util::QuaternionLerp(&quaternion3, quaternion1, quaternion2, 0.f);
    NNT_UTIL_VECTOR4_EXPECT_EQ(quaternion3, -3.f, 1.f, -2.f, 4.f);

    nn::util::QuaternionLerp(&quaternion3, quaternion1, quaternion2, 1.f);
    NNT_UTIL_VECTOR4_EXPECT_EQ(quaternion3, 6.f, -2.f, 4.f, -8.f);

    nn::util::QuaternionLerp(&quaternion3, quaternion1, quaternion2, 0.5f);
    NNT_UTIL_VECTOR4_EXPECT_EQ(quaternion3, 1.5f, -0.5f, 1.f, -2.f);
}

TEST_F(MathQuaternionTest, Slerp)
{
    {
        nn::util::Vector4fType quaternion1;
        nn::util::VectorSet(&quaternion1, -3.f, 1.f, -2.f, 4.f);
        nn::util::Vector4fType quaternion2;
        nn::util::VectorSet(&quaternion2, 6.f, -2.f, 4.f, -8.f);

        nn::util::Vector4fType quaternion3;

        nn::util::QuaternionSlerp(&quaternion3, quaternion1, quaternion2, 0.f);
        NNT_UTIL_VECTOR4_EXPECT_EQ(quaternion3, -3.f, 1.f, -2.f, 4.f);

        nn::util::QuaternionSlerp(&quaternion3, quaternion1, quaternion2, 1.f);
        NNT_UTIL_VECTOR4_EXPECT_EQ(quaternion3, -6.f, 2.f, -4.f, 8.f);

        nn::util::QuaternionSlerp(&quaternion3, quaternion1, quaternion2, 0.5f);
        NNT_UTIL_VECTOR4_EXPECT_EQ(quaternion3, -4.5f, 1.5f, -3.f, 6.f);
    }

    {
        nn::util::Vector4fType quaternion1;
        nn::util::VectorSet(&quaternion1, -0.03f, 0.01f, -0.02f, 0.04f);
        nn::util::Vector4fType quaternion2;
        nn::util::VectorSet(&quaternion2, 0.06f, -0.02f, 0.04f, -0.08f);

        nn::util::Vector4fType quaternion3;

        nn::util::QuaternionSlerp(&quaternion3, quaternion1, quaternion2, 0.f);
        NNT_UTIL_VECTOR4_EXPECT_NEARLY_EQ(quaternion3, -0.0299999993294477f, 0.0099999997764826f, -0.0199999995529652f, 0.0399999991059303f, FloatError);

        nn::util::QuaternionSlerp(&quaternion3, quaternion1, quaternion2, 1.f);
        NNT_UTIL_VECTOR4_EXPECT_NEARLY_EQ(quaternion3, -0.0599999986588955f, 0.0199999995529652f, -0.0399999991059303f, 0.0799999982118607f, FloatError);

        nn::util::QuaternionSlerp(&quaternion3, quaternion1, quaternion2, 0.5f);
        NNT_UTIL_VECTOR4_EXPECT_NEARLY_EQ(quaternion3, -0.0634495541453362f, 0.0211498513817787f, -0.0422997027635574f, 0.0845994055271149f, FloatError);
    }
}

TEST_F(MathQuaternionTest, Squad)
{
    {
        nn::util::Vector4fType p, a, b, q;
        nn::util::VectorSet(&p, -3.f, 1.f, -2.f, 4.f);
        nn::util::VectorSet(&a, 6.f, -2.f, 4.f, -8.f);
        nn::util::VectorSet(&b, -12.f, 4.f, -8.f, 16.f);
        nn::util::VectorSet(&q, 24.f, -8.f, 16.f, -32.f);

        nn::util::Vector4fType quaternion1;

        nn::util::QuaternionSquad(&quaternion1, p, a, b, q, 0.f);
        NNT_UTIL_VECTOR4_EXPECT_EQ(quaternion1, -3.f, 1.f, -2.f, 4.f);

        nn::util::QuaternionSquad(&quaternion1, p, a, b, q, 1.f);
        NNT_UTIL_VECTOR4_EXPECT_EQ(quaternion1, -24.f, 8.f, -16.f, 32.f);

        nn::util::QuaternionSquad(&quaternion1, p, a, b, q, 0.5f);
        NNT_UTIL_VECTOR4_EXPECT_EQ(quaternion1, -11.25f, 3.75f, -7.5f, 15.f);
    }

    {
        nn::util::Vector3fType axis;
        nn::util::VectorSet(&axis, 1.f, 0.5f, 0.3f);
        nn::util::VectorNormalize(&axis, axis);

        nn::util::Vector4fType p, a, b, q;
        nn::util::QuaternionRotateAxis(&p, axis, 0.1f);
        nn::util::QuaternionRotateAxis(&a, axis, 0.2f);
        nn::util::QuaternionRotateAxis(&b, axis, 0.3f);
        nn::util::QuaternionRotateAxis(&q, axis, 0.5f);

        nn::util::Vector4fType quaternion1;

        nn::util::QuaternionSquad(&quaternion1, p, a, b, q, 0.f);
        NNT_UTIL_VECTOR4_EXPECT_NEARLY_EQ(quaternion1, 0.0431754f, 0.0215877f, 0.0129526f, 0.99875f, FloatError);

        nn::util::QuaternionSquad(&quaternion1, p, a, b, q, 1.f);
        NNT_UTIL_VECTOR4_EXPECT_NEARLY_EQ(quaternion1, 0.213724f, 0.106862f, 0.0641173f, 0.968912f, FloatError);

        nn::util::QuaternionSquad(&quaternion1, p, a, b, q, 0.5f);
        NNT_UTIL_VECTOR4_EXPECT_NEARLY_EQ(quaternion1, 0.118408f, 0.059204f, 0.0355224f, 0.990562f, FloatError);
    }
}

TEST_F(MathQuaternionTest, MakeClosest)
{
    {
        nn::util::Vector4fType quaternion1;
        nn::util::VectorSet(&quaternion1, -3.f, 1.f, -2.f, 4.f);

        nn::util::Vector4fType quaternion2;
        nn::util::VectorSet(&quaternion2, 6.f, -2.f, 4.f, -8.f);

        nn::util::QuaternionMakeClosest(&quaternion1, quaternion1, quaternion2);

        NNT_UTIL_VECTOR4_EXPECT_EQ(quaternion1, 3.f, -1.f, 2.f, -4.f);
    }

    {
        nn::util::Vector4fType quaternion1;
        nn::util::VectorSet(&quaternion1, -3.f, 1.f, -2.f, 4.f);

        nn::util::Vector4fType quaternion2;
        nn::util::VectorSet(&quaternion2, -6.f, 2.f, -4.f, 8.f);

        nn::util::QuaternionMakeClosest(&quaternion1, quaternion1, quaternion2);

        NNT_UTIL_VECTOR4_EXPECT_EQ(quaternion1, -3.f, 1.f, -2.f, 4.f);
    }
}

TEST_F(MathQuaternionTest, RotateAxisRadian)
{
    nn::util::Vector4fType quaternion1;
    nn::util::Vector3fType axis;
    nn::util::VectorSet(&axis, -3.f, 1.f, -2.f);

    nn::util::QuaternionRotateAxis(&quaternion1, axis, 60.f / 180.f * nn::util::FloatPi);

    NNT_UTIL_VECTOR4_EXPECT_NEARLY_EQ(quaternion1, -0.4008918404579163f, 0.1336306184530258f, -0.2672612369060517f, 0.8660254478454590f, FloatError);
}

TEST_F(MathQuaternionTest, MakeVectorRotation)
{
    {
        nn::util::Vector3fType from;
        nn::util::VectorSet(&from, -3.f, 1.f, -2.f);

        nn::util::Vector3fType to;
        nn::util::VectorSet(&to, -6.f, 2.f, 4.f);

        nn::util::Vector4fType quaternion1;

        nn::util::QuaternionMakeVectorRotation(&quaternion1, from, to);

        NNT_UTIL_VECTOR4_EXPECT_NEARLY_EQ(quaternion1, 1.5689290761947632f, 4.7067871093750000f, 0.0000000000000000f, 2.5495097637176514f, FloatError);
    }

    {
        nn::util::Vector3fType from;
        nn::util::VectorSet(&from, -3.f, 1.f, -2.f);

        nn::util::Vector3fType to;
        nn::util::VectorSet(&to, 6.f, -2.f, 4.f);

        nn::util::Vector4fType quaternion1;

        nn::util::QuaternionMakeVectorRotation(&quaternion1, from, to);

        NNT_UTIL_VECTOR4_EXPECT_EQ(quaternion1, 1.f, 0.f, 0.f, 0.f);
    }
}

TEST_F(MathQuaternionTest, FromMatrix)
{
    {
        // 以下のロール、ピッチ、ヨーによる回転行列
        // roll = 40.f / 180.f * PI;
        // pitch = 60.f / 180.f * PI;
        // yaw = 30.f / 180.f * PI;
        nn::util::MatrixRowMajor4x3fType matrix1;
        nn::util::MatrixSet(&matrix1,
            0.9417491555213928f, 0.3213937878608704f, 0.0990685001015663f,
            -0.2249634265899658f, 0.3830222487449646f, 0.8959270715713501f,
            0.2499999701976776f, -0.8660253286361694f, 0.4330127239227295f,
            -10.f, 11.f, -12.f);

        nn::util::Vector4fType quaternion1;
        nn::util::QuaternionFromMatrix(&quaternion1, matrix1);

        NNT_UTIL_VECTOR4_EXPECT_NEARLY_EQ(quaternion1, 0.5304983258247376f, 0.0454432815313339f, 0.1645002365112305f, 0.8303288221359253f, FloatError);
    }

    {
        nn::util::MatrixColumnMajor4x3fType matrix1;
        nn::util::MatrixSet(&matrix1,
            0.9417491555213928f, 0.3213937878608704f, 0.0990685001015663f,
            -0.2249634265899658f, 0.3830222487449646f, 0.8959270715713501f,
            0.2499999701976776f, -0.8660253286361694f, 0.4330127239227295f,
            -10.f, 11.f, -12.f);

        nn::util::Vector4fType quaternion1;
        nn::util::QuaternionFromMatrix(&quaternion1, matrix1);

        NNT_UTIL_VECTOR4_EXPECT_NEARLY_EQ(quaternion1, 0.5304983258247376f, 0.0454432815313339f, 0.1645002365112305f, 0.8303288221359253f, FloatError);
    }
}

#if defined(NN_BUILD_CONFIG_COMPILER_SUPPORTS_VC)
#pragma warning( pop )
#endif
