﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nn/util/util_Optional.h>
#include <nnt/nntest.h>

#include <string>

NN_PRAGMA_PUSH_WARNINGS
NN_DISABLE_WARNING_SHADOW

using nn::util::optional;
using nn::util::nullopt;
using nn::util::in_place;
using nn::util::make_optional;

static_assert(std::is_trivially_destructible<optional<int>>::value, "");

TEST(Optional, Nullopt)
{
    optional<int> o1;
    ASSERT_TRUE(!o1);

    optional<int> o2 = nullopt;
    ASSERT_TRUE(!o2);

    optional<int> o3 = o2;
    ASSERT_TRUE(!o3);

    ASSERT_TRUE(o3 == nullopt);
    ASSERT_TRUE(o3 == optional<int>());

    ASSERT_TRUE(o1 == o2);
    ASSERT_TRUE(o2 == o3);
    ASSERT_TRUE(o3 == o1);
}

struct Value
{
    int a;
    std::string b;
    double c;
    Value() {}
    Value(int a, std::string b, double c) : a(a), b(b), c(c) {}

    friend bool operator==(const Value& lhs, const Value& rhs)
    {
        return true
            && lhs.a == rhs.a
            && lhs.b == rhs.b
            && lhs.c == rhs.c;
    }
    friend bool operator!=(const Value& lhs, const Value& rhs)
    {
        return !(lhs == rhs);
    }
};

TEST(Optional, Value)
{
    Value v1(1, "1", 1.0);
    Value v2(2, "2", 2.0);

    {
        optional<Value> o(v1);

        ASSERT_TRUE(o);
        ASSERT_TRUE(o->a == v1.a);
        ASSERT_TRUE(o->b == v1.b);
        ASSERT_TRUE(o->c == v1.c);
        ASSERT_TRUE(o == v1);
        ASSERT_TRUE(*o == v1);
        ASSERT_TRUE(o.value() == v1);
        ASSERT_TRUE(o != v2);
        ASSERT_TRUE(o != nullopt);
    }

    {
        optional<Value> o;
        o = v1;

        ASSERT_TRUE(o);
        ASSERT_TRUE(o->a == v1.a);
        ASSERT_TRUE(o->b == v1.b);
        ASSERT_TRUE(o->c == v1.c);
        ASSERT_TRUE(o == v1);
        ASSERT_TRUE(*o == v1);
        ASSERT_TRUE(o.value() == v1);
        ASSERT_TRUE(o != v2);
        ASSERT_TRUE(o != nullopt);
    }

    {
        optional<Value> o1 = v1;
        optional<Value> o(o1);

        ASSERT_TRUE(o);
        ASSERT_TRUE(o->a == v1.a);
        ASSERT_TRUE(o->b == v1.b);
        ASSERT_TRUE(o->c == v1.c);
        ASSERT_TRUE(o == v1);
        ASSERT_TRUE(*o == v1);
        ASSERT_TRUE(o.value() == v1);
        ASSERT_TRUE(o != v2);
        ASSERT_TRUE(o != nullopt);
    }

    {
        optional<Value> o1 = v1;
        optional<Value> o(std::move(o1));

        ASSERT_TRUE(o);
        ASSERT_TRUE(o->a == v1.a);
        ASSERT_TRUE(o->b == v1.b);
        ASSERT_TRUE(o->c == v1.c);
        ASSERT_TRUE(o == v1);
        ASSERT_TRUE(*o == v1);
        ASSERT_TRUE(o.value() == v1);
        ASSERT_TRUE(o != v2);
        ASSERT_TRUE(o != nullopt);
    }

    {
        optional<Value> o = make_optional(v1);

        ASSERT_TRUE(o);
        ASSERT_TRUE(o->a == v1.a);
        ASSERT_TRUE(o->b == v1.b);
        ASSERT_TRUE(o->c == v1.c);
        ASSERT_TRUE(o == v1);
        ASSERT_TRUE(*o == v1);
        ASSERT_TRUE(o.value() == v1);
        ASSERT_TRUE(o != v2);
        ASSERT_TRUE(o != nullopt);
    }

    {
        optional<Value> o(in_place, v1.a, v1.b, v1.c);

        ASSERT_TRUE(o);
        ASSERT_TRUE(o->a == v1.a);
        ASSERT_TRUE(o->b == v1.b);
        ASSERT_TRUE(o->c == v1.c);
        ASSERT_TRUE(o == v1);
        ASSERT_TRUE(*o == v1);
        ASSERT_TRUE(o.value() == v1);
        ASSERT_TRUE(o != v2);
        ASSERT_TRUE(o != nullopt);
    }

    {
        optional<Value> o;
        optional<Value> o1 = v1;
        o = o1;

        ASSERT_TRUE(o);
        ASSERT_TRUE(o->a == v1.a);
        ASSERT_TRUE(o->b == v1.b);
        ASSERT_TRUE(o->c == v1.c);
        ASSERT_TRUE(o == v1);
        ASSERT_TRUE(*o == v1);
        ASSERT_TRUE(o.value() == v1);
        ASSERT_TRUE(o != v2);
        ASSERT_TRUE(o != nullopt);
    }

    {
        optional<Value> o;
        o.emplace(v1.a, v1.b, v1.c);

        ASSERT_TRUE(o);
        ASSERT_TRUE(o->a == v1.a);
        ASSERT_TRUE(o->b == v1.b);
        ASSERT_TRUE(o->c == v1.c);
        ASSERT_TRUE(o == v1);
        ASSERT_TRUE(*o == v1);
        ASSERT_TRUE(o.value() == v1);
        ASSERT_TRUE(o != v2);
        ASSERT_TRUE(o != nullopt);
    }

    {
        optional<Value> o = v1;
        o.emplace();

        ASSERT_TRUE(o);

        o = nullopt;

        ASSERT_TRUE(!o);
    }

    {
        optional<Value> o = v1;
        ASSERT_TRUE(o.value_or(v2) == v1);
        o = nullopt;
        ASSERT_TRUE(o.value_or(v2) == v2);
    }

    {
        optional<int> o1;
        optional<int> o2(o1);
        optional<int> o3 = o1;
        o2 = o3;
        ASSERT_TRUE(!o1);
        ASSERT_TRUE(!o2);
        ASSERT_TRUE(!o3);

        optional<int> o4 = 4;
        o3 = o4;
        ASSERT_TRUE(o3 == 4);
        ASSERT_TRUE(o4 == 4);

        o3 = 3;
        ASSERT_TRUE(o3 == 3);
    }
} // NOLINT(readability/fn_size)

int g_ConstructCount = 0;
int g_DestructCount = 0;

struct Counter
{
    int x;
    explicit Counter(int x = 0)
        : x(x)
    {
        ++g_ConstructCount;
    }
    Counter(const Counter& other)
    {
        this->x = other.x;
        ++g_ConstructCount;
    }
    ~Counter()
    {
        ++g_DestructCount;
    }
    friend bool operator==(const Counter& lhs, const Counter& rhs)
    {
        return lhs.x == rhs.x;
    }
    void swap(Counter& other)
    {
        using std::swap;
        swap(this->x, other.x);
    }
};

void swap(Counter& x, Counter& y)
{
    x.swap(y);
}

struct CountChecker
{
    int constructCount;
    int destructCount;

    CountChecker()
        : constructCount(g_ConstructCount), destructCount(g_DestructCount)
    {
    }

    void Assert(int constructCount, int destructCount)
    {
        ASSERT_TRUE(g_ConstructCount - this->constructCount == constructCount);
        ASSERT_TRUE(g_DestructCount - this->destructCount == destructCount);
    }

    void AssertInvariant()
    {
        ASSERT_TRUE(g_ConstructCount - constructCount == g_DestructCount - destructCount);
    }

    ~CountChecker()
    {
        AssertInvariant();
    }
};

TEST(Optional, ConstructDestruct)
{
    {
        CountChecker checker;
        {
            optional<Counter> counter;
            checker.Assert(0, 0);
        }
        checker.Assert(0, 0);
    }
    {
        CountChecker checker;
        {
            optional<Counter> counter(in_place);
            checker.Assert(1, 0);
        }
        checker.Assert(1, 1);
    }
    {
        Counter c1(10);
        Counter c2(20);
        CountChecker checker;
        {
            optional<Counter> counter;
            counter = c1;
            checker.Assert(1, 0);
            counter = c2;
            checker.Assert(1, 0);
            counter = nullopt;
            checker.Assert(1, 1);
            counter = c2;
            checker.Assert(2, 1);
        }
        checker.Assert(2, 2);
    }
    {
        CountChecker checker;
        {
            optional<Counter> counter;
            counter.emplace();
            checker.Assert(1, 0);
            counter.emplace(1);
            checker.Assert(2, 1);
        }
        checker.Assert(2, 2);
    }
}

TEST(Optional, Swap)
{
    using std::swap;
    Counter c1(10);
    Counter c2(20);

    {
        CountChecker checker;
        {
            optional<Counter> o1(c1);
            optional<Counter> o2(c2);
            checker.Assert(2, 0);

            swap(o1, o2);

            ASSERT_TRUE(o1 == c2);
            ASSERT_TRUE(o2 == c1);
            checker.Assert(2, 0);
        }
    }

    {
        CountChecker checker;
        {
            optional<Counter> o1;
            optional<Counter> o2(c2);
            checker.Assert(1, 0);

            swap(o1, o2);

            ASSERT_TRUE(o1 == c2);
            ASSERT_TRUE(o2 == nullopt);
            checker.Assert(2, 1);
        }
    }

    {
        CountChecker checker;
        {
            optional<Counter> o1(c1);
            optional<Counter> o2;
            checker.Assert(1, 0);

            swap(o1, o2);

            ASSERT_TRUE(o1 == nullopt);
            ASSERT_TRUE(o2 == c1);
            checker.Assert(2, 1);
        }
    }

    {
        CountChecker checker;
        {
            optional<Counter> o1;
            optional<Counter> o2;
            checker.Assert(0, 0);

            swap(o1, o2);

            ASSERT_TRUE(o1 == nullopt);
            ASSERT_TRUE(o2 == nullopt);
            checker.Assert(0, 0);
        }
    }
}

struct Movable
{
    int data;
    bool valid;
    Movable() : valid(false) {}
    explicit Movable(int data) : data(data), valid(true) {}
    Movable(const Movable& other) : data(other.data), valid(other.valid) {}
    Movable(Movable&& other)
    {
        if (other.valid)
        {
            this->data = other.data;
            this->valid = true;
            other.valid = false;
        }
        else
        {
            this->valid = false;
        }
    }
    Movable& operator=(const Movable& other)
    {
        this->data = other.data;
        this->valid = other.valid;
        return *this;
    }
    Movable& operator=(Movable&& other)
    {
        if (other.valid)
        {
            this->data = other.data;
            this->valid = true;
            other.valid = false;
        }
        else
        {
            this->valid = false;
        }
        return *this;
    }
    void Set(int data)
    {
        this->data = data;
        this->valid = true;
    }
    void Invalidate()
    {
        this->valid = false;
    }
};

TEST(Optional, Move)
{
    {
        optional<int> o1 = 1;
        optional<int> o2;

        o2 = std::move(o1);
        ASSERT_TRUE(static_cast<bool>(o2) == static_cast<bool>(o1));
        ASSERT_TRUE(o2);
    }
    {
        optional<int> o1 = 1;
        optional<int> o2(std::move(o1));

        ASSERT_TRUE(static_cast<bool>(o2) == static_cast<bool>(o1));
        ASSERT_TRUE(o2);
    }

    {
        Movable m1(1);
        ASSERT_TRUE(m1.valid);

        optional<Movable> o1(std::move(m1));

        ASSERT_TRUE(!m1.valid);
        ASSERT_TRUE(o1);
        ASSERT_TRUE(o1->valid);
    }
    {
        Movable m1(1);
        ASSERT_TRUE(m1.valid);

        optional<Movable> o1;
        o1 = std::move(m1);

        ASSERT_TRUE(!m1.valid);
        ASSERT_TRUE(o1);
        ASSERT_TRUE(o1->valid);
    }
    {
        Movable m0;
        Movable m1(1);
        ASSERT_TRUE(!m0.valid);
        ASSERT_TRUE(m1.valid);

        optional<Movable> o1(m0);
        o1 = std::move(m1);

        ASSERT_TRUE(!m1.valid);
        ASSERT_TRUE(o1);
        ASSERT_TRUE(o1->valid);
    }
    {
        Movable m1(1);
        ASSERT_TRUE(m1.valid);

        optional<Movable> o1;
        o1.emplace(std::move(m1));

        ASSERT_TRUE(!m1.valid);
        ASSERT_TRUE(o1);
        ASSERT_TRUE(o1->valid);
    }
    {
        Movable m0;
        Movable m1(1);
        ASSERT_TRUE(!m0.valid);
        ASSERT_TRUE(m1.valid);

        optional<Movable> o1 = m0;
        o1.emplace(std::move(m1));

        ASSERT_TRUE(!m1.valid);
        ASSERT_TRUE(o1);
        ASSERT_TRUE(o1->valid);
    }
}

TEST(Optional, Comparison)
{
    optional<int> o0 = nullopt;
    optional<int> o1 = 1;
    optional<int> o2 = 2;

    ASSERT_TRUE(!(o0 == o1));
    ASSERT_TRUE(!(o1 == o2));
    ASSERT_TRUE(!(o2 == o0));
    ASSERT_TRUE(!(o1 == o0));
    ASSERT_TRUE(!(o2 == o1));
    ASSERT_TRUE(!(o0 == o2));
    ASSERT_TRUE( (o0 == nullopt));
    ASSERT_TRUE( (nullopt == o0));
    ASSERT_TRUE(!(o1 == nullopt));
    ASSERT_TRUE(!(nullopt == o1));
    ASSERT_TRUE(!(o0 == 1));
    ASSERT_TRUE(!( 1 == o0));
    ASSERT_TRUE( (o1 ==  1));
    ASSERT_TRUE( ( 1 == o1));
    ASSERT_TRUE(!(o2 ==  1));
    ASSERT_TRUE(!( 1 == o2));

    ASSERT_TRUE( (o0 != o1));
    ASSERT_TRUE( (o1 != o2));
    ASSERT_TRUE( (o2 != o0));
    ASSERT_TRUE( (o1 != o0));
    ASSERT_TRUE( (o2 != o1));
    ASSERT_TRUE( (o0 != o2));
    ASSERT_TRUE(!(o0 != nullopt));
    ASSERT_TRUE(!(nullopt != o0));
    ASSERT_TRUE( (o1 != nullopt));
    ASSERT_TRUE( (nullopt != o1));
    ASSERT_TRUE( (o0 != 1));
    ASSERT_TRUE( ( 1 != o0));
    ASSERT_TRUE(!(o1 !=  1));
    ASSERT_TRUE(!( 1 != o1));
    ASSERT_TRUE( (o2 !=  1));
    ASSERT_TRUE( ( 1 != o2));

    ASSERT_TRUE( (o0 < o1));
    ASSERT_TRUE( (o1 < o2));
    ASSERT_TRUE(!(o2 < o0));
    ASSERT_TRUE(!(o1 < o0));
    ASSERT_TRUE(!(o2 < o1));
    ASSERT_TRUE( (o0 < o2));
    ASSERT_TRUE(!(o0 < nullopt));
    ASSERT_TRUE(!(nullopt < o0));
    ASSERT_TRUE(!(o1 < nullopt));
    ASSERT_TRUE( (nullopt < o1));
    ASSERT_TRUE( (o0 < 1));
    ASSERT_TRUE(!( 1 < o0));
    ASSERT_TRUE(!(o1 <  1));
    ASSERT_TRUE(!( 1 < o1));
    ASSERT_TRUE(!(o2 <  1));
    ASSERT_TRUE( ( 1 < o2));

    ASSERT_TRUE( (o0 <= o1));
    ASSERT_TRUE( (o1 <= o2));
    ASSERT_TRUE(!(o2 <= o0));
    ASSERT_TRUE(!(o1 <= o0));
    ASSERT_TRUE(!(o2 <= o1));
    ASSERT_TRUE( (o0 <= o2));
    ASSERT_TRUE( (o0 <= nullopt));
    ASSERT_TRUE( (nullopt <= o0));
    ASSERT_TRUE(!(o1 <= nullopt));
    ASSERT_TRUE( (nullopt <= o1));
    ASSERT_TRUE( (o0 <= 1));
    ASSERT_TRUE(!( 1 <= o0));
    ASSERT_TRUE( (o1 <=  1));
    ASSERT_TRUE( ( 1 <= o1));
    ASSERT_TRUE(!(o2 <=  1));
    ASSERT_TRUE( ( 1 <= o2));

    ASSERT_TRUE(!(o0 > o1));
    ASSERT_TRUE(!(o1 > o2));
    ASSERT_TRUE( (o2 > o0));
    ASSERT_TRUE( (o1 > o0));
    ASSERT_TRUE( (o2 > o1));
    ASSERT_TRUE(!(o0 > o2));
    ASSERT_TRUE(!(o0 > nullopt));
    ASSERT_TRUE(!(nullopt > o0));
    ASSERT_TRUE( (o1 > nullopt));
    ASSERT_TRUE(!(nullopt > o1));
    ASSERT_TRUE(!(o0 > 1));
    ASSERT_TRUE( ( 1 > o0));
    ASSERT_TRUE(!(o1 >  1));
    ASSERT_TRUE(!( 1 > o1));
    ASSERT_TRUE( (o2 >  1));
    ASSERT_TRUE(!( 1 > o2));

    ASSERT_TRUE(!(o0 >= o1));
    ASSERT_TRUE(!(o1 >= o2));
    ASSERT_TRUE( (o2 >= o0));
    ASSERT_TRUE( (o1 >= o0));
    ASSERT_TRUE( (o2 >= o1));
    ASSERT_TRUE(!(o0 >= o2));
    ASSERT_TRUE( (o0 >= nullopt));
    ASSERT_TRUE( (nullopt >= o0));
    ASSERT_TRUE( (o1 >= nullopt));
    ASSERT_TRUE(!(nullopt >= o1));
    ASSERT_TRUE(!(o0 >= 1));
    ASSERT_TRUE( ( 1 >= o0));
    ASSERT_TRUE( (o1 >=  1));
    ASSERT_TRUE( ( 1 >= o1));
    ASSERT_TRUE( (o2 >=  1));
    ASSERT_TRUE(!( 1 >= o2));
}

NN_PRAGMA_POP_WARNINGS
