﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nnt/nntest.h>

#include <nn/util/util_IntUtil.h>

#include <nn/nn_Common.h>
#include <limits>

namespace {

TEST(testUtil_IntUtil, IsIntValueRepresentable_0)
{
    {
        auto value = static_cast<int8_t>(0);
        EXPECT_TRUE(nn::util::IsIntValueRepresentable<int8_t>(value));
        EXPECT_TRUE(nn::util::IsIntValueRepresentable<int32_t>(value));
        EXPECT_TRUE(nn::util::IsIntValueRepresentable<int64_t>(value));
        EXPECT_TRUE(nn::util::IsIntValueRepresentable<uint8_t>(value));
        EXPECT_TRUE(nn::util::IsIntValueRepresentable<uint32_t>(value));
        EXPECT_TRUE(nn::util::IsIntValueRepresentable<uint64_t>(value));
    }
    {
        auto value = static_cast<uint8_t>(0);
        EXPECT_TRUE(nn::util::IsIntValueRepresentable<int8_t>(value));
        EXPECT_TRUE(nn::util::IsIntValueRepresentable<int32_t>(value));
        EXPECT_TRUE(nn::util::IsIntValueRepresentable<int64_t>(value));
        EXPECT_TRUE(nn::util::IsIntValueRepresentable<uint8_t>(value));
        EXPECT_TRUE(nn::util::IsIntValueRepresentable<uint32_t>(value));
        EXPECT_TRUE(nn::util::IsIntValueRepresentable<uint64_t>(value));
    }
    {
        auto value = static_cast<int64_t>(0);
        EXPECT_TRUE(nn::util::IsIntValueRepresentable<int8_t>(value));
        EXPECT_TRUE(nn::util::IsIntValueRepresentable<int32_t>(value));
        EXPECT_TRUE(nn::util::IsIntValueRepresentable<int64_t>(value));
        EXPECT_TRUE(nn::util::IsIntValueRepresentable<uint8_t>(value));
        EXPECT_TRUE(nn::util::IsIntValueRepresentable<uint32_t>(value));
        EXPECT_TRUE(nn::util::IsIntValueRepresentable<uint64_t>(value));
    }
    {
        auto value = static_cast<uint64_t>(0);
        EXPECT_TRUE(nn::util::IsIntValueRepresentable<int8_t>(value));
        EXPECT_TRUE(nn::util::IsIntValueRepresentable<int32_t>(value));
        EXPECT_TRUE(nn::util::IsIntValueRepresentable<int64_t>(value));
        EXPECT_TRUE(nn::util::IsIntValueRepresentable<uint8_t>(value));
        EXPECT_TRUE(nn::util::IsIntValueRepresentable<uint32_t>(value));
        EXPECT_TRUE(nn::util::IsIntValueRepresentable<uint64_t>(value));
    }
}

TEST(testUtil_IntUtil, IsIntValueRepresentable_SignedMin)
{
    {
        auto value = std::numeric_limits<int8_t>::min();
        EXPECT_TRUE(nn::util::IsIntValueRepresentable<int8_t>(value));
        EXPECT_TRUE(nn::util::IsIntValueRepresentable<int32_t>(value));
        EXPECT_TRUE(nn::util::IsIntValueRepresentable<int64_t>(value));
        EXPECT_TRUE(!nn::util::IsIntValueRepresentable<uint8_t>(value));
        EXPECT_TRUE(!nn::util::IsIntValueRepresentable<uint32_t>(value));
        EXPECT_TRUE(!nn::util::IsIntValueRepresentable<uint64_t>(value));
    }
    {
        auto value = std::numeric_limits<int32_t>::min();
        EXPECT_TRUE(!nn::util::IsIntValueRepresentable<int8_t>(value));
        EXPECT_TRUE(nn::util::IsIntValueRepresentable<int32_t>(value));
        EXPECT_TRUE(nn::util::IsIntValueRepresentable<int64_t>(value));
        EXPECT_TRUE(!nn::util::IsIntValueRepresentable<uint8_t>(value));
        EXPECT_TRUE(!nn::util::IsIntValueRepresentable<uint32_t>(value));
        EXPECT_TRUE(!nn::util::IsIntValueRepresentable<uint64_t>(value));
    }
    {
        auto value = std::numeric_limits<int64_t>::min();
        EXPECT_TRUE(!nn::util::IsIntValueRepresentable<int8_t>(value));
        EXPECT_TRUE(!nn::util::IsIntValueRepresentable<int32_t>(value));
        EXPECT_TRUE(nn::util::IsIntValueRepresentable<int64_t>(value));
        EXPECT_TRUE(!nn::util::IsIntValueRepresentable<uint8_t>(value));
        EXPECT_TRUE(!nn::util::IsIntValueRepresentable<uint32_t>(value));
        EXPECT_TRUE(!nn::util::IsIntValueRepresentable<uint64_t>(value));
    }
}

TEST(testUtil_IntUtil, IsIntValueRepresentable_SignedMax)
{
    {
        auto value = std::numeric_limits<int8_t>::max();
        EXPECT_TRUE(nn::util::IsIntValueRepresentable<int8_t>(value));
        EXPECT_TRUE(nn::util::IsIntValueRepresentable<int32_t>(value));
        EXPECT_TRUE(nn::util::IsIntValueRepresentable<int64_t>(value));
        EXPECT_TRUE(nn::util::IsIntValueRepresentable<uint8_t>(value));
        EXPECT_TRUE(nn::util::IsIntValueRepresentable<uint32_t>(value));
        EXPECT_TRUE(nn::util::IsIntValueRepresentable<uint64_t>(value));
    }
    {
        auto value = std::numeric_limits<int32_t>::max();
        EXPECT_TRUE(!nn::util::IsIntValueRepresentable<int8_t>(value));
        EXPECT_TRUE(nn::util::IsIntValueRepresentable<int32_t>(value));
        EXPECT_TRUE(nn::util::IsIntValueRepresentable<int64_t>(value));
        EXPECT_TRUE(!nn::util::IsIntValueRepresentable<uint8_t>(value));
        EXPECT_TRUE(nn::util::IsIntValueRepresentable<uint32_t>(value));
        EXPECT_TRUE(nn::util::IsIntValueRepresentable<uint64_t>(value));
    }
    {
        auto value = std::numeric_limits<int64_t>::max();
        EXPECT_TRUE(!nn::util::IsIntValueRepresentable<int8_t>(value));
        EXPECT_TRUE(!nn::util::IsIntValueRepresentable<int32_t>(value));
        EXPECT_TRUE(nn::util::IsIntValueRepresentable<int64_t>(value));
        EXPECT_TRUE(!nn::util::IsIntValueRepresentable<uint8_t>(value));
        EXPECT_TRUE(!nn::util::IsIntValueRepresentable<uint32_t>(value));
        EXPECT_TRUE(nn::util::IsIntValueRepresentable<uint64_t>(value));
    }
}

TEST(testUtil_IntUtil, IsIntValueRepresentable_UnsignedMax)
{
    {
        auto value = std::numeric_limits<uint8_t>::max();
        EXPECT_TRUE(!nn::util::IsIntValueRepresentable<int8_t>(value));
        EXPECT_TRUE(nn::util::IsIntValueRepresentable<int32_t>(value));
        EXPECT_TRUE(nn::util::IsIntValueRepresentable<int64_t>(value));
        EXPECT_TRUE(nn::util::IsIntValueRepresentable<uint8_t>(value));
        EXPECT_TRUE(nn::util::IsIntValueRepresentable<uint32_t>(value));
        EXPECT_TRUE(nn::util::IsIntValueRepresentable<uint64_t>(value));
    }
    {
        auto value = std::numeric_limits<uint32_t>::max();
        EXPECT_TRUE(!nn::util::IsIntValueRepresentable<int8_t>(value));
        EXPECT_TRUE(!nn::util::IsIntValueRepresentable<int32_t>(value));
        EXPECT_TRUE(nn::util::IsIntValueRepresentable<int64_t>(value));
        EXPECT_TRUE(!nn::util::IsIntValueRepresentable<uint8_t>(value));
        EXPECT_TRUE(nn::util::IsIntValueRepresentable<uint32_t>(value));
        EXPECT_TRUE(nn::util::IsIntValueRepresentable<uint64_t>(value));
    }
    {
        auto value = std::numeric_limits<uint64_t>::max();
        EXPECT_TRUE(!nn::util::IsIntValueRepresentable<int8_t>(value));
        EXPECT_TRUE(!nn::util::IsIntValueRepresentable<int32_t>(value));
        EXPECT_TRUE(!nn::util::IsIntValueRepresentable<int64_t>(value));
        EXPECT_TRUE(!nn::util::IsIntValueRepresentable<uint8_t>(value));
        EXPECT_TRUE(!nn::util::IsIntValueRepresentable<uint32_t>(value));
        EXPECT_TRUE(nn::util::IsIntValueRepresentable<uint64_t>(value));
    }
}

TEST(testUtil_IntUtil, IsIntValueRepresentable_SignedMinHalfExp)
{
    {
        auto value = static_cast<std::int32_t>(std::numeric_limits<int16_t>::min());
        EXPECT_TRUE(!nn::util::IsIntValueRepresentable<int8_t>(value));
        EXPECT_TRUE(nn::util::IsIntValueRepresentable<int16_t>(value));
        EXPECT_TRUE(nn::util::IsIntValueRepresentable<int32_t>(value));
        EXPECT_TRUE(nn::util::IsIntValueRepresentable<int64_t>(value));
        EXPECT_TRUE(!nn::util::IsIntValueRepresentable<uint8_t>(value));
        EXPECT_TRUE(!nn::util::IsIntValueRepresentable<uint16_t>(value));
        EXPECT_TRUE(!nn::util::IsIntValueRepresentable<uint32_t>(value));
        EXPECT_TRUE(!nn::util::IsIntValueRepresentable<uint64_t>(value));
    }
    {
        auto value = static_cast<std::int64_t>(std::numeric_limits<int32_t>::min());
        EXPECT_TRUE(!nn::util::IsIntValueRepresentable<int8_t>(value));
        EXPECT_TRUE(nn::util::IsIntValueRepresentable<int32_t>(value));
        EXPECT_TRUE(nn::util::IsIntValueRepresentable<int64_t>(value));
        EXPECT_TRUE(!nn::util::IsIntValueRepresentable<uint8_t>(value));
        EXPECT_TRUE(!nn::util::IsIntValueRepresentable<uint32_t>(value));
        EXPECT_TRUE(!nn::util::IsIntValueRepresentable<uint64_t>(value));
    }
}

TEST(testUtil_IntUtil, IsIntValueRepresentable_SignedMaxHalfExp)
{
    {
        auto value = static_cast<std::int32_t>(std::numeric_limits<int16_t>::max());
        EXPECT_TRUE(!nn::util::IsIntValueRepresentable<int8_t>(value));
        EXPECT_TRUE(nn::util::IsIntValueRepresentable<int16_t>(value));
        EXPECT_TRUE(nn::util::IsIntValueRepresentable<int32_t>(value));
        EXPECT_TRUE(nn::util::IsIntValueRepresentable<int64_t>(value));
        EXPECT_TRUE(!nn::util::IsIntValueRepresentable<uint8_t>(value));
        EXPECT_TRUE(nn::util::IsIntValueRepresentable<uint16_t>(value));
        EXPECT_TRUE(nn::util::IsIntValueRepresentable<uint32_t>(value));
        EXPECT_TRUE(nn::util::IsIntValueRepresentable<uint64_t>(value));
    }
    {
        auto value = static_cast<std::int64_t>(std::numeric_limits<int32_t>::max());
        EXPECT_TRUE(!nn::util::IsIntValueRepresentable<int8_t>(value));
        EXPECT_TRUE(nn::util::IsIntValueRepresentable<int32_t>(value));
        EXPECT_TRUE(nn::util::IsIntValueRepresentable<int64_t>(value));
        EXPECT_TRUE(!nn::util::IsIntValueRepresentable<uint8_t>(value));
        EXPECT_TRUE(nn::util::IsIntValueRepresentable<uint32_t>(value));
        EXPECT_TRUE(nn::util::IsIntValueRepresentable<uint64_t>(value));
    }
}

TEST(testUtil_IntUtil, IsIntValueRepresentable_UnsignedMaxHalfExp)
{
    {
        auto value = static_cast<std::uint32_t>(std::numeric_limits<uint16_t>::max());
        EXPECT_TRUE(!nn::util::IsIntValueRepresentable<int8_t>(value));
        EXPECT_TRUE(!nn::util::IsIntValueRepresentable<int16_t>(value));
        EXPECT_TRUE(nn::util::IsIntValueRepresentable<int32_t>(value));
        EXPECT_TRUE(nn::util::IsIntValueRepresentable<int64_t>(value));
        EXPECT_TRUE(!nn::util::IsIntValueRepresentable<uint8_t>(value));
        EXPECT_TRUE(nn::util::IsIntValueRepresentable<uint16_t>(value));
        EXPECT_TRUE(nn::util::IsIntValueRepresentable<uint32_t>(value));
        EXPECT_TRUE(nn::util::IsIntValueRepresentable<uint64_t>(value));
    }
    {
        auto value = static_cast<std::uint64_t>(std::numeric_limits<uint32_t>::max());
        EXPECT_TRUE(!nn::util::IsIntValueRepresentable<int8_t>(value));
        EXPECT_TRUE(!nn::util::IsIntValueRepresentable<int32_t>(value));
        EXPECT_TRUE(nn::util::IsIntValueRepresentable<int64_t>(value));
        EXPECT_TRUE(!nn::util::IsIntValueRepresentable<uint8_t>(value));
        EXPECT_TRUE(nn::util::IsIntValueRepresentable<uint32_t>(value));
        EXPECT_TRUE(nn::util::IsIntValueRepresentable<uint64_t>(value));
    }
}

TEST(testUtil_IntUtil, IsIntValueRepresentable_SignedMinHalfExpMinus1)
{
    {
        auto value = static_cast<std::int32_t>(std::numeric_limits<int16_t>::min()) - 1;
        EXPECT_TRUE(!nn::util::IsIntValueRepresentable<int8_t>(value));
        EXPECT_TRUE(!nn::util::IsIntValueRepresentable<int16_t>(value));
        EXPECT_TRUE(nn::util::IsIntValueRepresentable<int32_t>(value));
        EXPECT_TRUE(nn::util::IsIntValueRepresentable<int64_t>(value));
        EXPECT_TRUE(!nn::util::IsIntValueRepresentable<uint8_t>(value));
        EXPECT_TRUE(!nn::util::IsIntValueRepresentable<uint16_t>(value));
        EXPECT_TRUE(!nn::util::IsIntValueRepresentable<uint32_t>(value));
        EXPECT_TRUE(!nn::util::IsIntValueRepresentable<uint64_t>(value));
    }
    {
        auto value = static_cast<std::int64_t>(std::numeric_limits<int32_t>::min()) - 1;
        EXPECT_TRUE(!nn::util::IsIntValueRepresentable<int8_t>(value));
        EXPECT_TRUE(!nn::util::IsIntValueRepresentable<int32_t>(value));
        EXPECT_TRUE(nn::util::IsIntValueRepresentable<int64_t>(value));
        EXPECT_TRUE(!nn::util::IsIntValueRepresentable<uint8_t>(value));
        EXPECT_TRUE(!nn::util::IsIntValueRepresentable<uint32_t>(value));
        EXPECT_TRUE(!nn::util::IsIntValueRepresentable<uint64_t>(value));
    }
}

TEST(testUtil_IntUtil, IsIntValueRepresentable_SignedMaxHalfExpPlus1)
{
    {
        auto value = static_cast<std::int32_t>(std::numeric_limits<int16_t>::max()) + 1;
        EXPECT_TRUE(!nn::util::IsIntValueRepresentable<int8_t>(value));
        EXPECT_TRUE(!nn::util::IsIntValueRepresentable<int16_t>(value));
        EXPECT_TRUE(nn::util::IsIntValueRepresentable<int32_t>(value));
        EXPECT_TRUE(nn::util::IsIntValueRepresentable<int64_t>(value));
        EXPECT_TRUE(!nn::util::IsIntValueRepresentable<uint8_t>(value));
        EXPECT_TRUE(nn::util::IsIntValueRepresentable<uint16_t>(value));
        EXPECT_TRUE(nn::util::IsIntValueRepresentable<uint32_t>(value));
        EXPECT_TRUE(nn::util::IsIntValueRepresentable<uint64_t>(value));
    }
    {
        auto value = static_cast<std::int64_t>(std::numeric_limits<int32_t>::max()) + 1;
        EXPECT_TRUE(!nn::util::IsIntValueRepresentable<int8_t>(value));
        EXPECT_TRUE(!nn::util::IsIntValueRepresentable<int32_t>(value));
        EXPECT_TRUE(nn::util::IsIntValueRepresentable<int64_t>(value));
        EXPECT_TRUE(!nn::util::IsIntValueRepresentable<uint8_t>(value));
        EXPECT_TRUE(nn::util::IsIntValueRepresentable<uint32_t>(value));
        EXPECT_TRUE(nn::util::IsIntValueRepresentable<uint64_t>(value));
    }
}

template <typename Int>
void TestTryAddWithoutOverflow()
{
    const Int Zero = static_cast<Int>(0);
    const Int PlusOne = static_cast<Int>(1);

    Int r;

    // 0 + 0
    EXPECT_TRUE(nn::util::CanAddWithoutOverflow(Zero, Zero));
    EXPECT_TRUE( nn::util::TryAddWithoutOverflow(&r, Zero, Zero));
    EXPECT_EQ(r, Zero);

    // max + 0
    EXPECT_TRUE(nn::util::CanAddWithoutOverflow(std::numeric_limits<Int>::max(), Zero));
    EXPECT_TRUE( nn::util::TryAddWithoutOverflow(&r, std::numeric_limits<Int>::max(), Zero));
    EXPECT_EQ(r, std::numeric_limits<Int>::max());

    // 0 + max
    EXPECT_TRUE(nn::util::CanAddWithoutOverflow(Zero, std::numeric_limits<Int>::max()));
    EXPECT_TRUE( nn::util::TryAddWithoutOverflow(&r, Zero, std::numeric_limits<Int>::max()));
    EXPECT_EQ(r, std::numeric_limits<Int>::max());

    // max + 1
    EXPECT_FALSE(nn::util::CanAddWithoutOverflow(std::numeric_limits<Int>::max(), PlusOne));
    EXPECT_FALSE(nn::util::TryAddWithoutOverflow(&r, std::numeric_limits<Int>::max(), PlusOne));

    // 1 + max
    EXPECT_FALSE(nn::util::CanAddWithoutOverflow(PlusOne, std::numeric_limits<Int>::max()));
    EXPECT_FALSE(nn::util::TryAddWithoutOverflow(&r, PlusOne, std::numeric_limits<Int>::max()));

    // min + 1
    EXPECT_TRUE(nn::util::CanAddWithoutOverflow(std::numeric_limits<Int>::min(), PlusOne));
    EXPECT_TRUE(nn::util::TryAddWithoutOverflow(&r, std::numeric_limits<Int>::min(), PlusOne));
    EXPECT_EQ(r, std::numeric_limits<Int>::min() + PlusOne);

    // 1 + min
    EXPECT_TRUE(nn::util::CanAddWithoutOverflow(PlusOne, std::numeric_limits<Int>::min()));
    EXPECT_TRUE(nn::util::TryAddWithoutOverflow(&r, PlusOne, std::numeric_limits<Int>::min()));
    EXPECT_EQ(r, std::numeric_limits<Int>::min() + PlusOne);

    // min + 0
    EXPECT_TRUE(nn::util::CanAddWithoutOverflow(std::numeric_limits<Int>::min(), Zero));
    EXPECT_TRUE(nn::util::TryAddWithoutOverflow(&r, std::numeric_limits<Int>::min(), Zero));
    EXPECT_EQ(r, std::numeric_limits<Int>::min());

    // 0 + min
    EXPECT_TRUE(nn::util::CanAddWithoutOverflow(Zero, std::numeric_limits<Int>::min()));
    EXPECT_TRUE(nn::util::TryAddWithoutOverflow(&r, Zero, std::numeric_limits<Int>::min()));
    EXPECT_EQ(r, std::numeric_limits<Int>::min());

    if(NN_STATIC_CONDITION(std::is_signed<Int>::value))
    {
        const Int MinusOne = static_cast<Int>(-1);

        // max + (-1)
        EXPECT_TRUE(nn::util::CanAddWithoutOverflow(std::numeric_limits<Int>::max(), MinusOne));
        EXPECT_TRUE( nn::util::TryAddWithoutOverflow(&r, std::numeric_limits<Int>::max(), MinusOne));
        EXPECT_EQ(r, std::numeric_limits<Int>::max() - PlusOne);

        // (-1) + max
        EXPECT_TRUE(nn::util::CanAddWithoutOverflow(MinusOne, std::numeric_limits<Int>::max()));
        EXPECT_TRUE( nn::util::TryAddWithoutOverflow(&r, MinusOne, std::numeric_limits<Int>::max()));
        EXPECT_EQ(r, std::numeric_limits<Int>::max() - PlusOne);

        // min + (-1)
        EXPECT_FALSE(nn::util::CanAddWithoutOverflow(std::numeric_limits<Int>::min(), MinusOne));
        EXPECT_FALSE(nn::util::TryAddWithoutOverflow(&r, std::numeric_limits<Int>::min(), MinusOne));

        // (-1) + min
        EXPECT_FALSE(nn::util::CanAddWithoutOverflow(MinusOne, std::numeric_limits<Int>::min()));
        EXPECT_FALSE(nn::util::TryAddWithoutOverflow(&r, MinusOne, std::numeric_limits<Int>::min()));
    }
} // NOLINT(impl/function_size)

template <typename Int>
void TestTrySubtractWithoutOverflow()
{
    const Int Zero = static_cast<Int>(0);
    const Int PlusOne = static_cast<Int>(1);
    const Int PlusTwo = static_cast<Int>(2);

    Int r;

    // 0 - 0
    EXPECT_TRUE(nn::util::CanSubtractWithoutOverflow(Zero, Zero));
    EXPECT_TRUE(nn::util::TrySubtractWithoutOverflow(&r, Zero,  Zero));
    EXPECT_EQ(r, Zero);

    // 1 - 0
    EXPECT_TRUE(nn::util::CanSubtractWithoutOverflow(PlusOne, Zero));
    EXPECT_TRUE(nn::util::TrySubtractWithoutOverflow(&r, PlusOne,  Zero));
    EXPECT_EQ(r, PlusOne);

    // 0 - 1
    // (signedは成功, unsignedは失敗)
    if(NN_STATIC_CONDITION(std::is_signed<Int>::value))
    {
        EXPECT_TRUE(nn::util::CanSubtractWithoutOverflow(Zero, PlusOne));
        EXPECT_TRUE(nn::util::TrySubtractWithoutOverflow(&r, Zero, PlusOne));
        EXPECT_EQ(r, static_cast<Int>(-1));
    }
    else
    {
        EXPECT_FALSE(nn::util::CanSubtractWithoutOverflow(Zero, PlusOne));
        EXPECT_FALSE(nn::util::TrySubtractWithoutOverflow(&r, Zero, PlusOne));
    }

    // max - 0
    EXPECT_TRUE(nn::util::CanSubtractWithoutOverflow(std::numeric_limits<Int>::max(), Zero));
    EXPECT_TRUE(nn::util::TrySubtractWithoutOverflow(&r, std::numeric_limits<Int>::max(), Zero));
    EXPECT_EQ(r, std::numeric_limits<Int>::max());

    // max - 1
    EXPECT_TRUE(nn::util::CanSubtractWithoutOverflow(std::numeric_limits<Int>::max(), PlusOne));
    EXPECT_TRUE(nn::util::TrySubtractWithoutOverflow(&r, std::numeric_limits<Int>::max(), PlusOne));
    EXPECT_EQ(r, std::numeric_limits<Int>::max() - PlusOne);

    // min - 0
    EXPECT_TRUE(nn::util::CanSubtractWithoutOverflow(std::numeric_limits<Int>::min(),  Zero));
    EXPECT_TRUE(nn::util::TrySubtractWithoutOverflow(&r, std::numeric_limits<Int>::min(),  Zero));
    EXPECT_EQ(r, std::numeric_limits<Int>::min());

    // min - 1
    EXPECT_FALSE(nn::util::CanSubtractWithoutOverflow(std::numeric_limits<Int>::min(),  PlusOne));
    EXPECT_FALSE(nn::util::TrySubtractWithoutOverflow(&r, std::numeric_limits<Int>::min(), PlusOne));

    // 0 - max
    // (signedは成功, unsignedは失敗)
    if(NN_STATIC_CONDITION(std::is_signed<Int>::value))
    {
        EXPECT_TRUE(nn::util::CanSubtractWithoutOverflow(Zero, std::numeric_limits<Int>::max()));
        EXPECT_TRUE(nn::util::TrySubtractWithoutOverflow(&r, Zero, std::numeric_limits<Int>::max()));
        EXPECT_EQ(r, std::numeric_limits<Int>::min() + PlusOne);
    }
    else
    {
        EXPECT_FALSE(nn::util::CanSubtractWithoutOverflow(Zero, std::numeric_limits<Int>::max()));
        EXPECT_FALSE(nn::util::TrySubtractWithoutOverflow(&r, Zero, std::numeric_limits<Int>::max()));
    }

    // 1 - max
    // (signedは成功, unsignedは失敗)
    if(NN_STATIC_CONDITION(std::is_signed<Int>::value))
    {
        EXPECT_TRUE(nn::util::CanSubtractWithoutOverflow(PlusOne, std::numeric_limits<Int>::max()));
        EXPECT_TRUE(nn::util::TrySubtractWithoutOverflow(&r, PlusOne, std::numeric_limits<Int>::max()));
        EXPECT_EQ(r, std::numeric_limits<Int>::min() + PlusTwo);
    }
    else
    {
        EXPECT_FALSE(nn::util::CanSubtractWithoutOverflow(PlusOne, std::numeric_limits<Int>::max()));
        EXPECT_FALSE(nn::util::TrySubtractWithoutOverflow(&r, PlusOne, std::numeric_limits<Int>::max()));
    }

    // 0 - min
    // (signedは失敗, unsignedは成功)
    if(NN_STATIC_CONDITION(std::is_signed<Int>::value))
    {
        EXPECT_FALSE(nn::util::CanSubtractWithoutOverflow(Zero, std::numeric_limits<Int>::min()));
        EXPECT_FALSE(nn::util::TrySubtractWithoutOverflow(&r, Zero, std::numeric_limits<Int>::min()));
    }
    else
    {
        EXPECT_TRUE(nn::util::CanSubtractWithoutOverflow(Zero, std::numeric_limits<Int>::min()));
        EXPECT_TRUE(nn::util::TrySubtractWithoutOverflow(&r, Zero, std::numeric_limits<Int>::min()));
        EXPECT_EQ(r, Zero);
    }

    if(NN_STATIC_CONDITION(std::is_signed<Int>::value))
    {
        const Int MinusOne = static_cast<Int>(-1);
        const Int MinusTwo = static_cast<Int>(-2);

        // min - (-1)
        EXPECT_TRUE(nn::util::CanSubtractWithoutOverflow(std::numeric_limits<Int>::min(), MinusOne));
        EXPECT_TRUE(nn::util::TrySubtractWithoutOverflow(&r, std::numeric_limits<Int>::min(), MinusOne));
        EXPECT_EQ(r, std::numeric_limits<Int>::min() + PlusOne);

        // max - (-1)
        EXPECT_FALSE(nn::util::CanSubtractWithoutOverflow(std::numeric_limits<Int>::max(), MinusOne));
        EXPECT_FALSE(nn::util::TrySubtractWithoutOverflow(&r, std::numeric_limits<Int>::max(), MinusOne));

        // -1 - min
        EXPECT_TRUE(nn::util::CanSubtractWithoutOverflow(MinusOne, std::numeric_limits<Int>::min()));
        EXPECT_TRUE(nn::util::TrySubtractWithoutOverflow(&r, MinusOne, std::numeric_limits<Int>::min()));
        EXPECT_EQ(r, std::numeric_limits<Int>::max());

        // -1 - max
        EXPECT_TRUE(nn::util::CanSubtractWithoutOverflow(MinusOne, std::numeric_limits<Int>::max()));
        EXPECT_TRUE(nn::util::TrySubtractWithoutOverflow(&r, MinusOne, std::numeric_limits<Int>::max()));
        EXPECT_EQ(r, std::numeric_limits<Int>::min());

        // -2 - max
        EXPECT_FALSE(nn::util::CanSubtractWithoutOverflow(MinusTwo, std::numeric_limits<Int>::max()));
        EXPECT_FALSE(nn::util::TrySubtractWithoutOverflow(&r, MinusTwo, std::numeric_limits<Int>::max()));
    }
} // NOLINT(impl/function_size)

TEST(testUtil_IntUtil, TryAddWithoutOverflow)
{
    TestTryAddWithoutOverflow<int>();

    TestTryAddWithoutOverflow<int8_t>();
    TestTryAddWithoutOverflow<int16_t>();
    TestTryAddWithoutOverflow<int32_t>();
    TestTryAddWithoutOverflow<int64_t>();

    TestTryAddWithoutOverflow<uint8_t>();
    TestTryAddWithoutOverflow<uint16_t>();
    TestTryAddWithoutOverflow<uint32_t>();
    TestTryAddWithoutOverflow<uint64_t>();
}

TEST(testUtil_IntUtil, TrySubtractWithoutOverflow)
{
    TestTrySubtractWithoutOverflow<int>();

    TestTrySubtractWithoutOverflow<int8_t>();
    TestTrySubtractWithoutOverflow<int16_t>();
    TestTrySubtractWithoutOverflow<int32_t>();
    TestTrySubtractWithoutOverflow<int64_t>();

    TestTrySubtractWithoutOverflow<uint8_t>();
    TestTrySubtractWithoutOverflow<uint16_t>();
    TestTrySubtractWithoutOverflow<uint32_t>();
    TestTrySubtractWithoutOverflow<uint64_t>();
}

}
