﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstdlib>
#include <nnt/nntest.h>
#include <nn/util/util_BitArray.h>

namespace {

TEST( BitArrayTest, BitArray )
{
    int bits[] = { 11, 22, 33, 44 };
    const int numBits = 4;
    const int length = 128;

    const size_t bufSize = nn::util::BitArray::CalculateWorkMemorySize( length );
    void* buffer = malloc( bufSize );
    nn::util::BitArray arr( buffer, bufSize, length );
    const nn::util::BitArray& carr = arr;

    arr.set();
    EXPECT_TRUE( carr.all() );
    EXPECT_TRUE( carr.any() );
    EXPECT_FALSE( carr.none() );
    EXPECT_TRUE( carr.test( 55 ) );
    EXPECT_EQ( 128, carr.count() );

    arr.flip( 55 );
    EXPECT_FALSE( carr.test( 55 ) );
    EXPECT_EQ( 127, carr.count() );

    arr.flip( 55 );
    EXPECT_TRUE( carr.test( 55 ) );

    arr.reset( 55 );
    EXPECT_FALSE( carr.test( 55 ) );

    arr.set( 55 );
    EXPECT_TRUE( carr.test( 55 ) );

    arr.reset();
    EXPECT_FALSE( carr.all() );
    EXPECT_FALSE( carr.any() );
    EXPECT_TRUE( carr.none() );
    EXPECT_EQ( 0, carr.count() );

    arr.flip( 55 );
    EXPECT_FALSE( carr.all() );
    EXPECT_TRUE( carr.any() );
    EXPECT_FALSE( carr.none() );
    EXPECT_EQ( 1, carr.count() );
    EXPECT_TRUE( carr.test( 55 ) );

    for ( int idxBits = 0; idxBits < numBits; ++idxBits )
    {
        arr.set( bits[ idxBits ] );
    }

    for ( int idxBits = 0; idxBits < numBits; ++idxBits )
    {
        EXPECT_TRUE( carr.test( bits[ idxBits ] ) );
    }

    free( buffer );
}

TEST( BitArrayTest, Empty )
{
    // 空の BitArray をチェック
    nn::util::BitArray arr, arr2;
    arr.flip();
    arr.reset();
    arr.set();

    EXPECT_TRUE( arr == arr2 );
    EXPECT_FALSE( arr != arr2 );

    EXPECT_FALSE( arr.all() );
    EXPECT_FALSE( arr.any() );
    EXPECT_TRUE( arr.none() );
    EXPECT_EQ( 0, arr.count() );
}

// 0 から length 分のビットが 1 であるときに pos 分シフトする
void TestShift( nn::util::BitArray& arr, int pos, int length )
{
    const int count = 128;
    const size_t bufSize = nn::util::BitArray::CalculateWorkMemorySize( count );

    void* rightBuffer = malloc( bufSize );
    void* leftBuffer = malloc( bufSize );

    nn::util::BitArray rightDst( rightBuffer, bufSize, count );
    nn::util::BitArray leftDst( leftBuffer, bufSize, count );

    nn::util::BitArray::RightShift( &rightDst, arr, pos );
    for (int i = 0; i < count ; ++i)
    {
        if ( pos <= i && i < pos + length  )
        {
            EXPECT_TRUE( rightDst.test( i ) );
        }
        else
        {
            EXPECT_FALSE( rightDst.test( i ) );
        }
    }
    EXPECT_EQ( length, rightDst.count() );

    nn::util::BitArray::LeftShift( &leftDst, rightDst, pos );
    for (int i = 0, bitCount = arr.size(); i < bitCount ; ++i)
    {
        if ( i < length )
        {
            EXPECT_TRUE( arr.test( i ) );
        }
        else
        {
            EXPECT_FALSE( arr.test( i ) );
        }
    }
    EXPECT_EQ( length, arr.count() );

    free( rightBuffer );
    free( leftBuffer );
}

TEST( BitArrayTest, Shift)
{
    const int length = 128;
    const size_t bufSize = nn::util::BitArray::CalculateWorkMemorySize( length );
    void* buffer = malloc( bufSize );
    nn::util::BitArray arr( buffer, bufSize, length );

    arr.set();
    EXPECT_EQ( 128, arr.count() );

    nn::util::BitArray::RightShift( &arr, arr, 32 );
    for (int i = 0; i < length; ++i)
    {
        if ( i < 32 )
        {
            EXPECT_FALSE( arr.test( i ) );
        }
        else
        {
            EXPECT_TRUE( arr.test( i ) );
        }
    }
    EXPECT_EQ( 96, arr.count() );

    nn::util::BitArray::LeftShift( &arr, arr, 64 );
    for (int i = 0; i < length; ++i)
    {
        if ( i < 64)
        {
            EXPECT_TRUE( arr.test( i ) );
        }
        else
        {
            EXPECT_FALSE( arr.test( i ) );
        }
    }
    EXPECT_EQ( 64, arr.count() );

    nn::util::BitArray::RightShift( &arr, arr, 96 );
    for (int i = 0; i < length; ++i)
    {
        if ( 96 <= i )
        {
            EXPECT_TRUE( arr.test( i ) );
        }
        else
        {
            EXPECT_FALSE( arr.test( i ) );
        }
    }
    EXPECT_EQ( 32, arr.count() );

    nn::util::BitArray::LeftShift( &arr, arr, 112 );
    for (int i = 0; i < length; ++i)
    {
        if ( i < 16 )
        {
            EXPECT_TRUE( arr.test( i ) );
        }
        else
        {
            EXPECT_FALSE( arr.test( i ) );
        }
    }
    EXPECT_EQ( 16, arr.count() );

    for (int i = 0; i < length - 16; i++)
    {
        TestShift( arr, i, 16 );
    }

    free( buffer );
}

TEST( BitArrayTest, Calculate)
{
    const int length = 128;
    const size_t bufSize = nn::util::BitArray::CalculateWorkMemorySize( length );

    void* buffer = malloc( bufSize );
    void* rightBuffer = malloc( bufSize );
    void* leftBuffer = malloc( bufSize );

    nn::util::BitArray result( buffer, bufSize, length );
    nn::util::BitArray lhs( rightBuffer, bufSize, length  );
    nn::util::BitArray rhs( leftBuffer, bufSize, length  );

    lhs.reset();
    for ( int i = 0 ; i < length; i += 2 )
    {
        lhs.set( i );
    }

    rhs.reset();

    nn::util::BitArray::And( &result, lhs, rhs );
    EXPECT_TRUE( result.none() );

    nn::util::BitArray::Or( &result, lhs, rhs );
    EXPECT_TRUE( lhs == result );

    for ( int i = 1; i < length; i += 2 )
    {
        rhs.set( i );
    }
    nn::util::BitArray::Xor( &result, lhs, rhs );
    EXPECT_TRUE( result.all() );

    nn::util::BitArray::RightShift( &result, lhs, 1 );
    EXPECT_TRUE( rhs == result );

    nn::util::BitArray::LeftShift( &result, rhs, 1 );
    EXPECT_TRUE( lhs == result );

    nn::util::BitArray::Not( &result, lhs );
    EXPECT_TRUE( rhs == result );

    free( buffer );
    free( rightBuffer );
    free( leftBuffer );
}

TEST( BitArrayTest, Iterator )
{
    const int length = 256;
    const size_t bufSize = nn::util::BitArray::CalculateWorkMemorySize( length );
    void* buffer = malloc( bufSize );
    nn::util::BitArray bitArray( buffer, bufSize, length );

    bitArray.set();
    EXPECT_TRUE( bitArray.begin() == bitArray.end() - length );
    EXPECT_TRUE( bitArray.begin() + length == bitArray.end() );

    int count = 0;
    for ( nn::util::BitArray::iterator iter = bitArray.begin(), end = bitArray.end(); iter != end; ++iter, ++count )
    {
        EXPECT_EQ( count, std::distance( bitArray.begin(), iter ) );
        if ( count + 1 < length )
        {
            EXPECT_EQ( bitArray[ count + 1 ], iter[ 1 ] );
        }
        EXPECT_TRUE( iter < iter + 1 );
        (*iter).flip();
    }

    EXPECT_TRUE( bitArray.none() );

    for ( nn::util::BitArray::const_iterator iter = bitArray.cbegin(), end = bitArray.cend(); iter != end; ++iter)
    {
        EXPECT_FALSE( *iter );
    }

    for ( nn::util::BitArray::reverse_iterator iter = bitArray.rbegin(), end = bitArray.rend(); iter != end; ++iter)
    {
        (*iter) = true;
    }

    EXPECT_TRUE( bitArray.all() );

    free( buffer );
}

} // anonymous namespace
