﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <vector>
#include <nn/usb/usb_Pm.h>
#include <nn/usb/usb_Host.h>
#include <nn/usb/usb_Device.h>
#include <nn/usb/pd/usb_PdCradle.h>
#include <glv_notification.h>
#include <glv.h>

namespace nnt {
namespace usb {

enum Event
{
    Event_UpdatePowerState,
    Event_UpdateDataRole,
    Event_UpdateCradleState,
    Event_UpdateDevice,
    Event_UpdateDetail,
    Event_UpdateDriveStrength,
    Event_ShowDetail,
    Event_HideDetail,
    Event_ShowMenu,
    Event_HideMenu,

    Event_NumOfEvents
};

struct Endpoint
{
    nn::usb::UsbEndpointDescriptor  epDesc;
};

struct Interface
{
    nn::usb::UsbInterfaceDescriptor ifDesc;
    std::vector<Endpoint>           endpoints;
};

struct Device
{
    char                            name[nn::usb::HsLimitMaxDebugNameSize];
    nn::usb::UsbDeviceDescriptor    deviceDesc;
    nn::usb::UsbConfigDescriptor    configDesc;

    std::vector<Interface>          interfaces;
};

typedef std::map<nn::usb::DeviceUid, Device> DeviceMap;

class UsbModel : public glv::Notifier
{
public:
    UsbModel() NN_NOEXCEPT;
    ~UsbModel() NN_NOEXCEPT;

    void MonitorStart() NN_NOEXCEPT;
    void MonitorStop() NN_NOEXCEPT;

    void attach(Callback cb, usb::Event event, void *rcvr) NN_NOEXCEPT
    {
        Notifier::attach(cb, static_cast<glv::Update::t>(event), rcvr);
    }

    void detach(Callback cb, usb::Event event, void * rcvr) NN_NOEXCEPT
    {
        Notifier::detach(cb, static_cast<glv::Update::t>(event), rcvr);
    }

    void notify(usb::Event event, void *data) NN_NOEXCEPT
    {
        Notifier::notify(static_cast<glv::Update::t>(event), data);
    }

    bool IsLogEnabled(nn::usb::LogModule module);
    bool IsPollingCradle();

    void ToggleMenu();
    void ToggleMenu(bool isOn);
    void ToggleLogEnabled(nn::usb::LogModule module, bool enabled);
    void ToggleCradlePolling(bool enabled);
    void ToggleDummyDevice(bool enabled);

    void UpdateDetail(glv::View& detailView) NN_NOEXCEPT;

    void SetTestMode(nn::usb::TestMode mode);
    void SetDriveStrengthOffset(int offset);

private:
    // USB Port Manager
    nn::usb::PmClient                    m_Pm;
    nn::usb::UsbPowerState               m_PowerState;
    nn::usb::UsbDataRole                 m_DataRole;

    // USB Host
    nn::usb::Host                        m_Host;

    // USB Device
    nn::usb::DsClient                    m_Device;
    nn::usb::DsInterface                 m_Interface;
    nn::usb::DsEndpoint                  m_Endpoint;

    // Cradle
    nn::usb::pd::CradleSession           m_Cradle;
    nn::usb::pd::VdmPdcHFwVersion        m_CradlePdcHFwVersion;

    // Monitor Thread
    nn::os::ThreadType                   m_Thread;
    NN_OS_ALIGNAS_THREAD_STACK uint8_t   m_ThreadStack[1024 * 4];
    bool                                 m_Done;
    bool                                 m_PollCradle;

    int32_t                              m_IfCount;
    nn::usb::InterfaceQueryOutput        m_QueryOutputs[nn::usb::HsLimitMaxInterfacesCount];
    nn::usb::InterfaceQueryOutput        m_Buffer[nn::usb::HsLimitMaxInterfacesCount];

    DeviceMap                            m_DeviceMap;

    bool                                 m_IsMenuOn;

    glv::View                           *m_pDetailView;
    bool                                 m_ShowDetail;

    nn::usb::TestMode                    m_TestMode;
    int                                  m_DriveStrengthOffset;
    bool                                 m_IsOffsetSet;

private:
    static void ThreadEntry(void *arg) NN_NOEXCEPT
    {
        reinterpret_cast<UsbModel*>(arg)->Monitor();
    }

    void Monitor() NN_NOEXCEPT;

    void UpdatePowerState() NN_NOEXCEPT;
    void UpdateDataRole() NN_NOEXCEPT;
    void UpdateCradleState() NN_NOEXCEPT;
    void UpdateInterfaceList() NN_NOEXCEPT;
};

extern UsbModel g_Model;

} // ~usb
} // ~nnt
