﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nn/nn_Assert.h>
#include <nn/nn_SdkLog.h>
#include <nn/nn_SdkAssert.h>
#include <nn/init.h>
#include <nn/os.h>
#include <nn/os/os_UserExceptionHandler.h>
#include <nnt/nntest.h>

#include <nn/usb/pd/usb_PdCradle.h>
#include <nnt/usbPdUtil/testUsbPd_util.h>

//#define ENABLE_WAIT_EVENT

namespace {
    const nn::TimeSpan NoticeWaitTimeoutSpan = nn::TimeSpan::FromMilliSeconds( 1000 );
}

bool CheckCradleResult( nn::Result result, nn::usb::pd::Session* pSession )
{
    if ( result <= nn::usb::pd::ResultInactive() )
    {
        NN_SDK_LOG("ResultInactive\n");
        return false;
    }
    else if ( result <= nn::usb::pd::ResultInvalidDevice() )
    {
        NN_SDK_LOG("ResultInvalidDevice\n");
        return false;
    }
    else if ( result <= nn::usb::pd::ResultTimeout() )
    {
        NN_SDK_LOG("ResultTimeout\n");
        nn::usb::pd::Status status;
        nn::usb::pd::Notice notice;
        // クレードル挿抜未発生の確認のため Status -> Notice の順に読む必要あり
        nn::usb::pd::GetStatus( &status, pSession );
        nn::usb::pd::GetNotice( &notice, pSession );
        bool hasCradleBeenActive = nnt::usb::pd::IsCradleFamily( status.GetDeviceType() ) &&
                                   status.IsActive() &&
                                   !notice.IsActiveNotice();
        // クレードル接続継続中はタイムアウトしないはず
        if ( hasCradleBeenActive )
        {
            NN_ABORT_UNLESS_RESULT_SUCCESS(result);
        }
        return false;
    }
    else if ( result.IsFailure() )
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS(result);
        return false;
    }

    return true;
}

TEST( UsbPdResetCradleUsbHub, ResetCradleUsbHub )
{
    NN_SDK_LOG("[testUsbPd_ResetCradleUsbHub] Main\n");

    nn::Result result;

    nn::usb::pd::Initialize();
    nn::usb::pd::InitializeCradle();
    nn::usb::pd::Session session;
    nn::usb::pd::OpenSession( &session );
    nn::usb::pd::CradleSession cradleSession;
    nn::usb::pd::OpenCradleSession( &cradleSession );
    nn::os::SystemEvent noticeEvent(nn::os::EventClearMode_AutoClear, true);
    nn::usb::pd::BindNoticeEvent( noticeEvent.GetBase(), &session );
    nn::usb::pd::Vdo vdo;
    nn::usb::pd::Notice notice;
    // 通知フラグ空読みによるクリア
    nn::usb::pd::GetNotice( &notice, &session );
    NN_SDK_LOG("[usb::pd] Notice = %04x\n", notice.storage);

    // クレードル USB HUB のリセット処理
    NN_SDK_LOG("[usb::pd:c] Resetting cradle USB HUB...\n");
    result = nn::usb::pd::ResetCradleUsbHub( &cradleSession );
    if ( CheckCradleResult( result, &session ) )
    {
        NN_SDK_LOG("[usb::pd:c] Cradle USB HUB was reset.\n");
    }

    // クレードルリセット後の VDM 通信
    for ( int i = 0; i <= nn::usb::pd::CradleVdmCommand_McuFwVersion; i++ )
    {
        nn::usb::pd::Status status;
        nn::util::BitPack32* s = &status.m_Data;
        NN_UNUSED(s);
        nn::usb::pd::GetStatus( &status, &session );
        NN_SDK_LOG("[usb::pd] Status = %08x, %08x, %08x, %08x, %08x\n", s[0], s[1], s[2], s[3], s[4]);
        result = nn::usb::pd::GetCradleVdo( &vdo, &cradleSession, nn::usb::pd::CradleVdmCommand(i) );
        NN_SDK_LOG("[usb::pd:c] VDO[%d] = %08x\n", i, vdo);
        CheckCradleResult( result, &session );
#ifdef ENABLE_WAIT_EVENT
        noticeEvent.Wait();
#else
        noticeEvent.TimedWait( NoticeWaitTimeoutSpan );
#endif
        noticeEvent.Clear();
        nn::usb::pd::GetNotice( &notice, &session );
        NN_SDK_LOG("[usb::pd] Notice = %04x\n", notice.storage);
        nn::usb::pd::GetNotice( &notice, &session );
        NN_SDK_LOG("[usb::pd] Notice = %04x\n", notice.storage);
    }
    nn::usb::pd::UnbindNoticeEvent( &session );
    nn::usb::pd::CloseCradleSession( &cradleSession );
    nn::usb::pd::CloseSession( &session );
    nn::usb::pd::FinalizeCradle();
    nn::usb::pd::Finalize();
}

