﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <mutex>
#include <cmath>
#include <nn/nn_Macro.h>
#include <nn/nn_Assert.h>
#include <nn/os/os_Mutex.h>
#include "SpeedChangeableVibrationPlayer.h"

namespace {

//!< BasicLockable な MutexType を表す構造体です。
struct LockableMutexType final
{
    ::nn::os::MutexType _mutex;

    void lock() NN_NOEXCEPT
    {
        ::nn::os::LockMutex(&_mutex);
    }

    void unlock() NN_NOEXCEPT
    {
        ::nn::os::UnlockMutex(&_mutex);
    }
};

//!< SpeedChangeableVibrationPlayer 内の排他処理に用いるミューテックス
LockableMutexType s_Mutex = { NN_OS_MUTEX_INITIALIZER(false) };

}

namespace nns {
namespace iodevices {

SpeedChangeableVibrationPlayer::SpeedChangeableVibrationPlayer() NN_NOEXCEPT :
    m_PlaySpeed(1.0f),
    m_Count(0.0f) {}

SpeedChangeableVibrationPlayer::~SpeedChangeableVibrationPlayer() NN_NOEXCEPT {}

void SpeedChangeableVibrationPlayer::SetPlaySpeed(float speed) NN_NOEXCEPT
{
    std::lock_guard<decltype(s_Mutex)> lock(s_Mutex);

    NN_ASSERT_GREATER_EQUAL(speed, 0.0f);
    m_PlaySpeed = speed;
}

float SpeedChangeableVibrationPlayer::GetPlaySpeed() const NN_NOEXCEPT
{
    std::lock_guard<decltype(s_Mutex)> lock(s_Mutex);
    return m_PlaySpeed;
}

void SpeedChangeableVibrationPlayer::OnNextSampleRequired(
    nn::hid::VibrationValue* pValue,
    nn::hid::VibrationNodeConnection::List* pInputConnections) NN_NOEXCEPT
{
    std::lock_guard<decltype(s_Mutex)> lock(s_Mutex);

    NN_ASSERT_GREATER_EQUAL(m_PlaySpeed, 0.0f);
    NN_ASSERT_GREATER_EQUAL(m_Count, 0.0f);
    NN_ASSERT_LESS(m_Count, 1.0f);

    // 接続元の振動値は利用しません
    NN_UNUSED(pInputConnections);

    for (m_Count += m_PlaySpeed; m_Count >= 1.0f; m_Count -= 1.0f)
    {
        VibrationPlayer::OnNextSampleRequired(pValue, pInputConnections);
    }
}

} // namespace iodevices
} // namespace nns
