﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @examplesource{IoDevices.cpp,PageSampleIoDevices}
 *
 * @brief
 * Various Io devices sample. Audio, hid, hid vibration, hdmi, usb eth
 * Arguments description in order:
 *      audio in buffer length in miliseconds
 *      number of network connections (1-32)
 *      URL of file to download through USB ethernet
 * Sample arguments: 5 24 http://172.31.253.38:81/mb100.bin
 *
 * For downloading mb100.bin this part of the NetworkPerfTest yml file needs to be ran:
 *
 * # Generate 100MB Test File for downloads
 *   - Units:
 *     - Path          : ${NINTENDO_SDK_ROOT}/Tests/Outputs/${WIN_PLATFORM}/Tests/GenTestFile/${BUILD}/GenTestFile.exe
 *       Platform      : Win32
 *       Option        : --FileName=${NINTENDO_SDK_ROOT}/Tests/Net/Resources/TestFiles/mb100.bin --FileSizeMb=100
 *     - null
 * # Launch NGINX Http Server
 * - Units:
 *   - Path          : ${NINTENDO_SDK_ROOT}/Tests/Net/Resources/Start-DetachedProcess.ps1
 *     Option        : ${NINTENDO_SDK_ROOT}/Tests/Outputs/${WIN_PLATFORM}/Tests/SystemCmd/${BUILD}/SystemCmd.exe --CommandType=Process --PidsFile=NginxHttp.txt '--Command=${NINTENDO_SDK_ROOT}\Externals\Binaries\nginx-1.8.1\nginx.exe -c ${NINTENDO_SDK_ROOT}\Tests\Net\Resources\NginxConfig\nginx-perf-http_AutoGen.conf'
 *   - null
 */

#include <cstdlib>
#include <cmath>
#include <mutex>
#include <new>

#include <nns/nns_Log.h>
#include <nn/nn_Log.h>
#include <nn/nn_Assert.h>
#include <nn/nn_Common.h>
#include <nn/nn_Abort.h>
#include <nn/os.h>
#include <nn/mem.h>
#include <nn/fs.h>
#include <nn/nn_TimeSpan.h>

#include <nn/audio.h>
#include <nns/audio/audio_HidUtilities.h>
#include <nns/audio/audio_WavFormat.h>
#include <nn/audio/audio_AudioInApi.private.h>

#include <nn/hid.h>
#include <nn/hid/hid_KeyboardKey.h>
#include <nn/hid/hid_Npad.h>
#include <nn/hid/hid_Vibration.h>

#include <nn/settings/settings_DebugPad.h>
#include <nv/nv_MemoryManagement.h>
#include <nn/socket.h>
#include <curl/curl.h>
#include <nn/nifm.h>
#include <nn/nifm/nifm_ApiIpAddress.h>

#include "HttpsHelper.h"
#include "NpadController.h"
#include "GraphicsSystem.h"
#include "VibrationValueDrawer.h"
#include "SpeedChangeableVibrationPlayer.h"
#include "Microphone.h"
#include "MicData.h"

namespace nns {
namespace iodevices {

// Audio helper objects
struct AudioOutArgs
{
    nn::os::SystemEvent* bufferEvent;
    nn::audio::AudioOut* audioOut;
    bool isRunning;
};

struct AudioInArgs
{
    nn::mem::StandardAllocator* allocator;
    int bufferLengthInMilliseconds;
    bool isRunning;
};

enum EventType
{
    EventType_Invalid,
    EventType_Connect,
    EventType_Disconnect,
    EventType_Buffer,
};

class EventData
{
public:
    EventData()
        : m_Event(nullptr)
        , m_Index(-1)
        , m_Type(0)
        , m_InUse(false)
    {}

    void Acquire(int audioIndex, int eventType, nn::os::SystemEventType* event)
    {
        NN_ASSERT(!m_InUse);
        m_Index = audioIndex;
        m_Type = eventType;
        m_Event = event;
        m_InUse = true;
    }

    void Release()
    {
        NN_ASSERT(m_InUse);
        m_InUse = false;
        m_Index = -1;
        m_Event = nullptr;
        m_Type = EventType_Invalid;
    }

    int GetIndex()
    {
        return m_Index;
    }

    int GetType()
    {
        return m_Type;
    }

    bool IsInUse()
    {
        return m_InUse;
    }

    void ClearEvent()
    {
        if (m_Event)
        {
            nn::os::ClearSystemEvent(m_Event);
        }
    }
private:
    nn::os::SystemEventType* m_Event;
    int m_Index;
    int m_Type;
    bool m_InUse;
};

// Audio in mic helper object
struct MultiWaitInfo
{
public:
    void Link(nn::os::SystemEventType* bufferEvent, int micIndex)
    {
        int bufferEventIndex = (2 * micIndex) + 1;
        eventData[bufferEventIndex].Acquire(micIndex, EventType_Buffer, bufferEvent);
        nn::os::InitializeMultiWaitHolder(&bufferReadyHolder[micIndex], bufferEvent);
        nn::os::SetMultiWaitHolderUserData(&bufferReadyHolder[micIndex], static_cast<uintptr_t>(bufferEventIndex));
        nn::os::LinkMultiWaitHolder(multiWait, &bufferReadyHolder[micIndex]);
    }

    void Unlink(int micIndex)
    {
        int bufferEventIndex = (2 * micIndex) + 1;
        eventData[bufferEventIndex].Release();
        nn::os::UnlinkMultiWaitHolder(&bufferReadyHolder[micIndex]);
        nn::os::FinalizeMultiWaitHolder(&bufferReadyHolder[micIndex]);
    }

    EventData* eventData;
    nn::os::MultiWaitType* multiWait;
    nn::os::MultiWaitHolderType* bufferReadyHolder;
};

// Network helpber objects
typedef struct EasyData
{
    CURL* easyHandle;
    char pErrorBuffer[CURL_ERROR_SIZE];
} EasyData;

const int   kMaxUrlLength = 512;

typedef struct Options
{
    uint32_t    numConnections;
    uint32_t    timesToDownload;
    int         sndBufSize;
    int         rcvBufSize;
    int         disableTcpNagle;
    int         uploadSize;
    char        url[kMaxUrlLength + 1];
} Options;

// choose rendering engine sample rate
//const int RenderRate = 48000;
const int RenderRate = 32000;
const int RenderCount = (RenderRate / 200);

// choose number of files to play
const int BgmCount = 1;
const int SeCount = 4;

// - and add / remove them to / from the files lists
const char* g_BgmFileNames[BgmCount] =
{
    "asset:/IoDevices/SampleBgm0-2ch.wav",
};

const char* g_SeFileNames[SeCount] =
{
    "asset:/AudioCommon/SampleSe0.adpcm",
    "asset:/AudioCommon/SampleSe1.adpcm",
    "asset:/AudioCommon/SampleSe2.adpcm",
    "asset:/AudioCommon/SampleSe3.adpcm",
};

const int   DisplayFrameRate = 60;
const char  ProgramName[] = "IoDevices Sample";

const nn::util::Unorm8x4 TextColor = { { 255, 255, 255, 255 } };
const nn::util::Unorm8x4 ActiveColor = { { 255, 255, 255, 255 } };
const nn::util::Unorm8x4 DeactiveColor = { { 32, 32, 32, 255 } };
const nn::util::Unorm8x4 RedColor = { { 255, 0, 0, 255 } };
const nn::util::Unorm8x4 GreenColor = { { 0, 255, 0, 255 } };
const nn::util::Unorm8x4 BlueColor = { { 0, 0, 255, 255 } };
const nn::util::Unorm8x4 OrangeColor = { { 255, 128, 0, 255 } };

NN_ALIGNAS(4096) char g_WorkBuffer[8 * 1024 * 1024];
NN_AUDIO_ALIGNAS_MEMORY_POOL_ALIGN char g_WaveBufferPoolMemory[14 * 1024 * 1024];

bool g_IsQuitRequired = false;

GraphicsSystem* g_pGraphicsSystem;

nn::mem::StandardAllocator* g_pAppAllocator;
nn::mem::StandardAllocator g_Allocator;
nn::mem::StandardAllocator g_WaveBufferAllocator;

nn::Bit8* g_pAppMemory;

char* g_MountRomCacheBuffer = NULL;

nns::iodevices::SpeedChangeableVibrationPlayer g_PlayerA;
nns::iodevices::SpeedChangeableVibrationPlayer g_PlayerB;

const size_t VibrationFileSizeMax = 4048;
uint8_t g_FileDataA[VibrationFileSizeMax];
uint8_t g_FileDataB[VibrationFileSizeMax];
size_t g_FileSizeA = 0;
size_t g_FileSizeB = 0;

const int MaxAudioIns = 2;
char g_HeapBuffer[128 * 1024 * 16];
MicData g_MicData[MaxAudioIns];
int g_AudioInBufferLengthInMilliseconds = 5;
const size_t ThreadStackSize = 8192 * 4;
NN_OS_ALIGNAS_THREAD_STACK char g_AudioOutThreadStack[ThreadStackSize];
nn::os::ThreadType    g_AudioOutThread;
NN_OS_ALIGNAS_THREAD_STACK char g_AudioInThreadStack[ThreadStackSize];
nn::os::ThreadType    g_AudioInThread;

NN_OS_ALIGNAS_THREAD_STACK char     g_NetworkDownloadStack[ThreadStackSize];

uint32_t                    g_NumConnections = 24;
int                         g_NumofDisconnects = 0;
int                         g_ProgressFunctionCount = 0;
int                         g_LastProgressFunctionCount = 0;
float                       g_CurrentNetworkDownloadThroughput = 0.0f;
float                       g_CurrentDownloadedSize = 0.0f;
float                       g_TotalDownloadSize = 0.0f;
float                       g_MaxProgressDownload = 0.0f;
nn::os::Tick                g_TickPrev = nn::os::Tick(), g_TickNow = nn::os::Tick();
bool                        g_IsDownloading = false;
char*                       g_CurrentIpAddress;
char                        g_NetworkDownloadUrl[kMaxUrlLength + 1] = "http://172.31.253.38:81/mb100.bin";
NN_ALIGNAS(4096) uint8_t    g_SocketMemoryPoolBuffer[nn::socket::DefaultSocketMemoryPoolSize] = { 0 };
HttpsHelperForCtxImport     g_HttpsHelper = HttpsHelperForCtxImport();

const float kGigabyte = 1024.0f * 1024.0f * 1024.0f;
const float kMegabyte = 1024.0f * 1024.0f;
const float kKilobyte = 1024.0f;

size_t CurlSetSockOpts(void *pData, curl_socket_t curlfd, curlsocktype purpose)
{
    NN_UNUSED(purpose);

    int     rval = 0;
    nn::socket::Errno err = nn::socket::Errno::ESuccess;
    Options *pOptions = static_cast<Options *>(pData);

    if (nullptr != pOptions)
    {
        if (-1 != pOptions->sndBufSize)
        {
            rval = nn::socket::SetSockOpt(curlfd, nn::socket::Level::Sol_Socket, nn::socket::Option::So_SndBuf, &pOptions->sndBufSize, sizeof(pOptions->sndBufSize));
            if (rval != 0)
            {
                err = nn::socket::GetLastError();
                NN_LOG(" * nn::socket::SetSockOpt on So_SndBuf failed with error: %d\n\n", err);

                return CURL_SOCKOPT_ERROR;
            }
            else
            {
                NN_LOG(" * nn::socket::SetSockOpt So_SndBuf set to: %d\n", pOptions->sndBufSize);
            }
        }

        if (-1 != pOptions->rcvBufSize)
        {
            rval = nn::socket::SetSockOpt(curlfd, nn::socket::Level::Sol_Socket, nn::socket::Option::So_RcvBuf, &pOptions->rcvBufSize, sizeof(pOptions->rcvBufSize));
            if (rval != 0)
            {
                err = nn::socket::GetLastError();
                NN_LOG(" * nn::socket::SetSockOpt on So_RcvBuf failed with error: %d\n\n", err);

                return CURL_SOCKOPT_ERROR;
            }
            else
            {
                NN_LOG(" * nn::socket::SetSockOpt So_RcvBuf set to: %d\n", pOptions->rcvBufSize);
            }
        }

        rval = nn::socket::SetSockOpt(curlfd, nn::socket::Level::Sol_Tcp, nn::socket::Option::Tcp_NoDelay, &pOptions->disableTcpNagle, sizeof(pOptions->disableTcpNagle));
        if (rval != 0)
        {
            err = nn::socket::GetLastError();
            NN_LOG(" * nn::socket::SetSockOpt on Tcp_NoDelay failed with error: %d\n\n", err);

            return CURL_SOCKOPT_ERROR;
        }
    }

    return CURL_SOCKOPT_OK;
}

size_t CurlWriteFunction(char *data, size_t blobsize, size_t blobcount, void *userdata)
{
    NN_UNUSED(data);
    NN_UNUSED(userdata);

    size_t count = blobsize*blobcount;
    return count;
}

bool GetFileSize(const Options &options, uint32_t* pSize)
{
    bool    bRet = false;
    CURL*   curl = curl_easy_init();

    if (nullptr != curl)
    {
        char pErrorBuffer[CURL_ERROR_SIZE] = { 0 };

#ifdef USE_PROXY
        PROXY_SETTINGS(curl)
#endif

            NN_LOG("url: %s\n", options.url);
        curl_easy_setopt(curl, CURLOPT_URL, options.url);
        curl_easy_setopt(curl, CURLOPT_FOLLOWLOCATION, 1);
        curl_easy_setopt(curl, CURLOPT_VERBOSE, 1);
        curl_easy_setopt(curl, CURLOPT_ERRORBUFFER, pErrorBuffer);
        curl_easy_setopt(curl, CURLOPT_SOCKOPTFUNCTION, CurlSetSockOpts);
        curl_easy_setopt(curl, CURLOPT_SOCKOPTDATA, &options);

        if (HttpsHelperBase::IsUrlHttps(options.url) == true)
        {
            /* Import the SSL context into libcurl. Because there's only one SSL context
            maintained in httpsHelper, same SSL context will be shared between CURL
            handles.
            */
            if (g_HttpsHelper.ImportSslContext(curl) < 0)
            {
                NN_LOG("[ERROR] Failed to import the SSL context.\n");
                goto cleanup;
            }
        }

        // This is the KEY option that tells the server not to send to body(the actual content).
        curl_easy_setopt(curl, CURLOPT_NOBODY, 1);

        CURLcode rval = curl_easy_perform(curl);
        if (rval == CURLE_OK)
        {
            double dSize = 0.0;
            if (curl_easy_getinfo(curl, CURLINFO_CONTENT_LENGTH_DOWNLOAD, &dSize) == CURLE_OK)
            {
                if (dSize < 0)
                {
                    NN_LOG(" * Unknown content length!\n");
                }
                else
                {
                    *pSize = static_cast<uint32_t>(dSize);
                    NN_LOG("\n*******Got File size! %u\n\n", *pSize);
                    bRet = true;
                }
            }
            else
            {
                NN_LOG("Failed to get content length\n");
            }
        }
        else
        {
            NN_LOG("ERROR: Error fetching URL: %s. Error: %d\n", options.url, rval);
            NN_LOG("Details: %s\n", pErrorBuffer);
            if (HttpsHelperBase::IsUrlHttps(options.url) == true)
            {
                g_HttpsHelper.PrintErrorMessage(curl, rval);
            }
        }
    }
    else
    {
        NN_LOG(" * Failed to create curl handle!\n");
    }

cleanup:

    if (true == bRet)
    {
        curl_easy_cleanup(curl);
    }

    return bRet;
}

int ProgressFunction(void* pUserData, double totalDownloaded, double nowDownloaded, double totalUploaded, double nowUploaded)
{
    NN_UNUSED(totalUploaded);
    NN_UNUSED(nowUploaded);
    float fSeconds = 0.0f;
    nn::os::Tick currentTick = nn::os::Tick();

    currentTick = nn::os::GetSystemTick();
    fSeconds = nn::os::ConvertToTimeSpan(currentTick - g_TickNow).GetNanoSeconds() / 1000000000.0f;
    g_CurrentNetworkDownloadThroughput = g_NumConnections * nowDownloaded * 8.0f / fSeconds / kMegabyte;

    g_ProgressFunctionCount++;
    g_TotalDownloadSize     = totalDownloaded;
    g_CurrentDownloadedSize = nowDownloaded;

    bool* pIsDownloading = static_cast<bool*>(pUserData);

    if (!(*pIsDownloading))
    {
        return 1;
    }

    g_TickPrev = nn::os::GetSystemTick();
    return 0;
}

float DownloadFile(uint32_t index, const Options &options, uint32_t fileSize)
{
    float           fThroughput = 0.0f;
    nn::os::Tick    tickPrev = nn::os::Tick(),
                    tickNow = nn::os::Tick();
    uint32_t        downloadSize = 0;
    uint32_t        chunk = 0;
    uint32_t        remainder = 0;
    uint32_t        previous = 0;
    float           fSeconds = 0.0f;

    uint32_t        curlEasyHandleCount = 0;
    CURLM*          curlMultiHandle = NULL;

    CURLMcode       status = CURLM_OK;
    int             stillRunning = 0;

    chunk = fileSize / options.numConnections;
    remainder = fileSize % options.numConnections;

    EasyData *curlEasyHandles = static_cast<EasyData *>(std::calloc(options.numConnections, sizeof(EasyData)));

    if (nullptr == curlEasyHandles)
    {
        NN_LOG("[ERROR] Failed to allocate %d curl easy handles.\n", options.numConnections);
        return 0.0f;
    }

    for (curlEasyHandleCount = 0;
        curlEasyHandleCount < options.numConnections;
        curlEasyHandleCount++)
    {
        CURL *curlEasyHandle = curl_easy_init();
        if (nullptr == curlEasyHandle)
        {
            NN_LOG("ERROR: Could not create curl easy handle for url:%s", options.url);
            goto cleanup;
        }

        curlEasyHandles[curlEasyHandleCount].easyHandle = curlEasyHandle;
        memset(curlEasyHandles[curlEasyHandleCount].pErrorBuffer, 0, CURL_ERROR_SIZE);

        curl_easy_setopt(curlEasyHandle, CURLOPT_NOPROGRESS, 0);
        curl_easy_setopt(curlEasyHandle, CURLOPT_PROGRESSDATA, &g_IsDownloading);
        curl_easy_setopt(curlEasyHandle, CURLOPT_PROGRESSFUNCTION, ProgressFunction);

        curl_easy_setopt(curlEasyHandle, CURLOPT_URL, options.url);
        curl_easy_setopt(curlEasyHandle, CURLOPT_VERBOSE, 1);
        curl_easy_setopt(curlEasyHandle, CURLOPT_FOLLOWLOCATION, 1);
        curl_easy_setopt(curlEasyHandle, CURLOPT_ERRORBUFFER, curlEasyHandles[curlEasyHandleCount].pErrorBuffer);
        curl_easy_setopt(curlEasyHandle, CURLOPT_WRITEFUNCTION, CurlWriteFunction);
        if (HttpsHelperBase::IsUrlHttps(options.url) == true)
        {
            /* Import the SSL context into libcurl. Because there's only one SSL context
            maintained in g_HttpsHelper, same SSL context will be shared between CURL
            handles.
            */
            if (g_HttpsHelper.ImportSslContext(curlEasyHandle) < 0)
            {
                NN_LOG("[ERROR] Failed to import the SSL context.\n");
                goto cleanup;
            }
        }

        curl_easy_setopt(curlEasyHandle, CURLOPT_SOCKOPTFUNCTION, CurlSetSockOpts);
        curl_easy_setopt(curlEasyHandle, CURLOPT_SOCKOPTDATA, &options);

        if (options.numConnections > 1)
        {
            char        pBuffer[64 + 1] = { '\0' };
            uint32_t    increment = 0;

            if (remainder > 0)
            {
                increment = chunk + 1;
                --remainder;
            }
            else
            {
                increment = chunk;
            }

            // snprintf() isn't supported in Visual Studio 2013 so we can "safely" use sprintf() since we
            // know the maximum length string that would ever be put into pBuffer using 32-bit values is 22
            sprintf(pBuffer, "%u-%u", previous, previous + increment - 1);
            curl_easy_setopt(curlEasyHandle, CURLOPT_RANGE, pBuffer);

            previous += increment;

            NN_LOG("Chunk: %u  %s\n", chunk, pBuffer);
        }

#ifdef USE_PROXY
        PROXY_SETTINGS(curlEasyHandle)
#endif

    }

    // init a multi stack
    curlMultiHandle = curl_multi_init();
    if (nullptr == curlMultiHandle)
    {
        NN_LOG("ERROR: Could not create libcurl multi handle.");
        goto cleanup;
    }

    // add the individual transfers
    for (uint32_t i = 0; i < curlEasyHandleCount; i++)
    {
        curl_multi_add_handle(curlMultiHandle, curlEasyHandles[i].easyHandle);
    }

    g_CurrentDownloadedSize = 0;
    tickPrev = g_TickNow = g_TickPrev = nn::os::GetSystemTick();
    NN_LOG("Downloading...\n");
    do
    {
        status = curl_multi_perform(curlMultiHandle, &stillRunning);
    } while (status == CURLM_CALL_MULTI_PERFORM && !g_IsQuitRequired);

    NN_LOG("Status: %u\n", status);

    while (stillRunning && !g_IsQuitRequired)
    {
        nn::socket::TimeVal timeout = { 0, 0 };

        nn::socket::FdSet  fdread = nn::socket::FdSet();
        nn::socket::FdSet  fdwrite = nn::socket::FdSet();
        nn::socket::FdSet  fdexcep = nn::socket::FdSet();
        int     maxfd = -1;
        long    curl_timeo = -1;

        nn::socket::FdSetZero(&fdread);
        nn::socket::FdSetZero(&fdwrite);
        nn::socket::FdSetZero(&fdexcep);

        // set a suitable timeout to play around with
        timeout.tv_sec = 1;
        timeout.tv_usec = 0;

        curl_multi_timeout(curlMultiHandle, &curl_timeo);
        if (curl_timeo >= 0)
        {
            timeout.tv_sec = curl_timeo / 1000;
            if (timeout.tv_sec > 1)
            {
                timeout.tv_sec = 1;
            }
            else
            {
                timeout.tv_usec = (curl_timeo % 1000) * 1000;
            }
        }

        //Get file descriptors from the transfers
        curl_multi_fdset(curlMultiHandle, &fdread, &fdwrite, &fdexcep, &maxfd);

        int rc = nn::socket::Select(maxfd + 1, &fdread, &fdwrite, &fdexcep, &timeout);

        switch (rc)
        {
        case nn::socket::SocketError:
            // select error
            stillRunning = 0;
            NN_LOG("ERROR: select() returned error %d\n", nn::socket::GetLastError());
            break;
        case 0:
        default:
            // timeout or readable/writable sockets
            do
            {
                status = curl_multi_perform(curlMultiHandle, &stillRunning);
            } while (status == CURLM_CALL_MULTI_PERFORM);
            break;
        }
    }

    tickNow = nn::os::GetSystemTick();

    for (uint32_t iEasyHandle = 0; iEasyHandle < options.numConnections; ++iEasyHandle)
    {
        long    responseCode = 0;
        double  val = 0.0f;

        CURLcode ret = curl_easy_getinfo(curlEasyHandles[iEasyHandle].easyHandle, CURLINFO_RESPONSE_CODE, &responseCode);
        if (ret != CURLE_OK)
        {
            NN_LOG("curl_easy_getinfo failed: %d\n", ret);
        }
        else
        {
            NN_LOG("CURLINFO_RESPONSE_CODE: %d\n", responseCode);
        }

        ret = curl_easy_getinfo(curlEasyHandles[iEasyHandle].easyHandle, CURLINFO_HTTP_CONNECTCODE, &responseCode);
        if (ret != CURLE_OK)
        {
            NN_LOG("curl_easy_getinfo failed: %d\n", ret);
        }
        else
        {
            NN_LOG("CURLINFO_HTTP_CONNECTCODE: %d\n", responseCode);
        }

        ret = curl_easy_getinfo(curlEasyHandles[iEasyHandle].easyHandle, CURLINFO_SIZE_DOWNLOAD, &val);
        if (ret != CURLE_OK)
        {
            NN_LOG("curl_easy_getinfo failed: %d\n", ret);
        }
        else
        {
            uint32_t numBytes = static_cast<uint32_t>(val);
            downloadSize += numBytes;
            NN_LOG("CURLINFO_SIZE_DOWNLOAD: %u\n", numBytes);
        }

        /* check for total download time */
        ret = curl_easy_getinfo(curlEasyHandles[iEasyHandle].easyHandle, CURLINFO_TOTAL_TIME, &val);
        if ((CURLE_OK == ret) && (val > 0))
        {
            NN_LOG("CURLINFO_TOTAL_TIME: %0.3f sec.\n", val);
        }

        /* check for average download speed */
        ret = curl_easy_getinfo(curlEasyHandles[iEasyHandle].easyHandle, CURLINFO_SPEED_DOWNLOAD, &val);
        if ((CURLE_OK == ret) && (val > 0))
        {
            NN_LOG("CURLINFO_SPEED_DOWNLOAD: %0.3f kbyte/sec.\n", val / 1024);
        }

        /* check for name resolution time */
        ret = curl_easy_getinfo(curlEasyHandles[iEasyHandle].easyHandle, CURLINFO_NAMELOOKUP_TIME, &val);
        if ((CURLE_OK == ret) && (val > 0))
        {
            NN_LOG("CURLINFO_NAMELOOKUP_TIME: %0.3f sec.\n", val);
        }

        /* check for connect time */
        ret = curl_easy_getinfo(curlEasyHandles[iEasyHandle].easyHandle, CURLINFO_CONNECT_TIME, &val);
        if ((CURLE_OK == ret) && (val > 0))
        {
            NN_LOG("CURLINFO_CONNECT_TIME: %0.3f sec.\n", val);
        }

        NN_LOG("Error Details %d: %s\n", iEasyHandle, curlEasyHandles[iEasyHandle].pErrorBuffer);
    }

    fSeconds = nn::os::ConvertToTimeSpan(tickNow - tickPrev).GetMilliSeconds() / 1000.0f;

    NN_LOG("\n ** Finished download %u with Status: %d\n", index, status);

    NN_LOG(" ** Duration: %.3f sec\n", fSeconds);

    if (downloadSize / kGigabyte >= 1.0f)
    {
        NN_LOG(" ** Downloaded: %.3f GB\n", downloadSize / kGigabyte);
    }
    else if (downloadSize / kMegabyte >= 1.0f)
    {
        NN_LOG(" ** Downloaded: %.3f MB\n", downloadSize / kMegabyte);
    }
    else if (downloadSize / kKilobyte >= 1.0f)
    {
        NN_LOG(" ** Downloaded: %.3f KB\n", downloadSize / kKilobyte);
    }
    else
    {
        NN_LOG(" ** Downloaded: %u Bytes\n", downloadSize);
    }

    /* bytes/ms -> Mb/s */
    fThroughput = downloadSize * 8.0f / fSeconds / kMegabyte;
    NN_LOG(" ** Speed: %.3f Mbits/sec\n\n", fThroughput);

cleanup:

    for (uint32_t i = 0; i < curlEasyHandleCount; i++)
    {
        if (curlMultiHandle)
        {
            curl_multi_remove_handle(curlMultiHandle, curlEasyHandles[i].easyHandle);
        }

        curl_easy_cleanup(curlEasyHandles[i].easyHandle);
    }

    if (curlMultiHandle)
    {
        curl_multi_cleanup(curlMultiHandle);
    }

    free(curlEasyHandles);

    return fThroughput;
} // NOLINT(impl/function_size)


void AudioOutThread(void* arg)
{
    AudioOutArgs* args = reinterpret_cast<AudioOutArgs*>(arg);
    while (args->isRunning && !g_IsQuitRequired)
    {
        // 録音が完了したバッファを取得します。
        args->bufferEvent->Wait();
        nn::audio::AudioOutBuffer* pAudioOutBuffer = nullptr;

        pAudioOutBuffer = nn::audio::GetReleasedAudioOutBuffer(args->audioOut);

        int bufferCount = 0;
        while (pAudioOutBuffer)
        {
            ++bufferCount;
            // データをコピーし、再度登録します。
            void* pOutBuffer = nn::audio::GetAudioOutBufferDataPointer(pAudioOutBuffer);
            size_t outSize = nn::audio::GetAudioOutBufferDataSize(pAudioOutBuffer);

            memset(pOutBuffer, 0, outSize);
            for (int i = 0; i < MaxAudioIns; ++i)
            {
                if (g_MicData[i].inUse)
                {
                    g_MicData[i].GetFrameAdd(pOutBuffer, outSize);
                }
            }

            nn::audio::AppendAudioOutBuffer(args->audioOut, pAudioOutBuffer);
            pAudioOutBuffer = nn::audio::GetReleasedAudioOutBuffer(args->audioOut);
        }
        if (bufferCount != 1)
        {
            //NN_LOG("out bufferCount: %d\n", bufferCount);
        }
    }
}

bool IsMicrophoneUsed(Microphone* mics, int numMics, const char* name)
{
    for (int i = 0; i < numMics; ++i)
    {
        if (mics[i].IsOpen() && strcmp(name, mics[i].GetName()) == 0)
        {
            return true;
        }
    }
    return false;
}

bool IsMicrophoneConnected(Microphone* mic, nn::audio::AudioInInfo* infos, int infoCount)
{
    for (int i = 0; i < infoCount; ++i)
    {
        auto& info = infos[i];
        if (mic->IsOpen() && strcmp(info.name, mic->GetName()) == 0)
        {
            return true;
        }
    }
    return false;
}

void HandleConnect(Microphone* mics, int numMics, MultiWaitInfo* multiWaitInfo, nn::audio::AudioInInfo* audioInInfos, int count)
{
    for (int i = 0; i < count; ++i)
    {
        auto& info = audioInInfos[i];
        if (strcmp(info.name, "BuiltInHeadset") == 0)
        {
            continue;
        }
        if (!IsMicrophoneUsed(mics, numMics, info.name))
        {
            for (int j = 0; j < numMics; ++j)
            {
                if (!mics[j].IsOpen())
                {
                    auto open = mics[j].Open(info);
                    if (open)
                    {
                        multiWaitInfo->Link(mics[j].GetBufferEvent(), j);
                        mics[j].Start();
                        g_MicData[j].inUse = true;
                        g_MicData[j].Reset();
                        //TODO: Write this to screen
                        NN_LOG("Starting %s\n", mics[j].GetName());
                    }
                    break;
                }
            }
        }
    }
}

void HandleDisconnect(Microphone* mics, int index, MultiWaitInfo* multiWaitInfo)
{
    auto& mic = mics[index];
    if (mics[index].IsOpen())
    {
        auto info = mic.GetInfo();
        //TODO: Write this to screen
        NN_LOG("Stopping %s\n", info.name);
        g_MicData[index].inUse = false;
        mic.Stop();
        mic.Close();
        multiWaitInfo->Unlink(index);
        ::std::memset(g_MicData[index].rms, 0, sizeof(g_MicData[index].rms));
        g_MicData[index].rmsIndex = 0;
    }
}

void HandleUpdate(Microphone* mics, int index)
{
    auto& mic = mics[index];

    auto callback = [=](void* buffer, size_t bufferSize)
    {
        auto& micData = g_MicData[index];
        micData.PutFrame(buffer, bufferSize);
    };

    mic.Update(callback);
}

void CheckConnections(Microphone* mics, int numMics, MultiWaitInfo* multiWaitInfo)
{
    nn::audio::AudioInInfo audioInInfos[8];
    int count = nn::audio::ListAudioIns(&audioInInfos[0], 8);
    for (int i = 0; i < numMics; ++i)
    {
        if (mics[i].IsStarted())
        {
            bool nameFound = false;
            auto name = mics[i].GetName();
            for (int j = 0; j < count; ++j)
            {
                if (strcmp(name, audioInInfos[j].name) == 0)
                {
                    nameFound = true;
                    break;
                }
            }
            //Disconnect any microphones that haven't received data for a while
            if (!nameFound)
            {
                HandleDisconnect(mics, i, multiWaitInfo);
            }
        }
    }

    //Try to connect to any new microphones
    HandleConnect(mics, numMics, multiWaitInfo, audioInInfos, count);
}

void AudioInThread(void* arg)
{
    AudioInArgs* args = reinterpret_cast<AudioInArgs*>(arg);

    nn::os::SystemEvent connectEvent;
    Microphone mics[MaxAudioIns];
    nn::os::MultiWaitType           multiWait;
    nn::os::MultiWaitHolderType     bufferReadyHolder[MaxAudioIns];
    nn::os::MultiWaitHolderType     connectHolder;

    EventData eventData[MaxAudioIns * 2 + 1];

    MultiWaitInfo multiWaitInfo;
    multiWaitInfo.eventData = eventData;
    multiWaitInfo.multiWait = &multiWait;
    multiWaitInfo.bufferReadyHolder = bufferReadyHolder;

    nn::os::InitializeMultiWait(&multiWait);

    nn::audio::AcquireAudioDeviceNotificationForInput(&connectEvent);

    nn::os::InitializeMultiWaitHolder(&connectHolder, connectEvent.GetBase());
    eventData[0].Acquire(0, EventType_Connect, connectEvent.GetBase());
    nn::os::SetMultiWaitHolderUserData(&connectHolder, static_cast<uintptr_t>(0));
    nn::os::LinkMultiWaitHolder(&multiWait, &connectHolder);

    for (int i = 0; i < MaxAudioIns; ++i)
    {
        mics[i].Initialize(args->bufferLengthInMilliseconds, args->allocator);
    }

    nn::audio::AudioInInfo audioInInfos[8];
    int count = nn::audio::ListAudioIns(&audioInInfos[0], 8);
    HandleConnect(mics, MaxAudioIns, &multiWaitInfo, audioInInfos, count);

    while (args->isRunning && !g_IsQuitRequired)
    {
        auto holder = nn::os::WaitAny(&multiWait);

        auto dataIndex = nn::os::GetMultiWaitHolderUserData(holder);
        auto data = eventData[dataIndex];
        data.ClearEvent();

        switch (data.GetType())
        {
        case EventType_Connect:
            CheckConnections(mics, MaxAudioIns, &multiWaitInfo);
            break;
        case EventType_Buffer:
            HandleUpdate(mics, data.GetIndex());
            break;
        default:
            NN_UNEXPECTED_DEFAULT;
        }
    }

    for (int i = 0; i < MaxAudioIns; ++i)
    {
        HandleDisconnect(mics, i, &multiWaitInfo);
    }

    nn::os::UnlinkMultiWaitHolder(&connectHolder);
    nn::os::FinalizeMultiWaitHolder(&connectHolder);
    nn::os::FinalizeMultiWait(&multiWait);
    nn::os::DestroySystemEvent(connectEvent.GetBase());
}


void ReadVibrationFile(size_t* pOutSize, void* pOutBuffer, const char* filepath) NN_NOEXCEPT
{
    nn::Result result;
    nn::fs::FileHandle file;
    int64_t filesize;

    result = nn::fs::OpenFile(&file, filepath, nn::fs::OpenMode_Read);
    NN_ASSERT(result.IsSuccess());

    // ファイルサイズを確認する
    result = nn::fs::GetFileSize(&filesize, file);
    *pOutSize = static_cast<size_t>(filesize);
    NN_ASSERT(result.IsSuccess());
    NN_ASSERT(*pOutSize <= VibrationFileSizeMax);

    // データを読み込む
    result = nn::fs::ReadFile(file, 0, pOutBuffer, *pOutSize);
    NN_ASSERT(result.IsSuccess());

    nn::fs::CloseFile(file);

    NN_LOG("Read File(%s) is %s\n", filepath, result.IsSuccess() ? "Success" : "Failed");
}

void LoadVibrationResources() NN_NOEXCEPT
{
    // ファイルから振動データをメモリ上に展開する
    {
        size_t cacheSize = 0;
        nn::Result result = nn::fs::QueryMountRomCacheSize(&cacheSize);
        NN_ASSERT(result.IsSuccess());

        char* mountRomCacheBuffer = new(std::nothrow) char[cacheSize];
        NN_ASSERT_NOT_NULL(mountRomCacheBuffer);

        result = nn::fs::MountRom("rom", mountRomCacheBuffer, cacheSize);
        NN_ABORT_UNLESS_RESULT_SUCCESS(result);

        ReadVibrationFile(&g_FileSizeA, g_FileDataA, "asset:/IoDevices/SampleA.bnvib");
        ReadVibrationFile(&g_FileSizeB, g_FileDataB, "asset:/IoDevices/SampleB.bnvib");
        nn::fs::Unmount("rom");
        delete[] mountRomCacheBuffer;
    }

    // 振動データを VibrationPlayer で読み込む
    g_PlayerA.Load(g_FileDataA, g_FileSizeA);
    g_PlayerA.SetLoop(true);
    g_PlayerA.SetLoopInterval(20);

    g_PlayerB.Load(g_FileDataB, g_FileSizeB);
    g_PlayerB.SetLoop(true);
    g_PlayerB.SetLoopInterval(30);

}

nn::os::TimerEventType  g_TimerEvent;
const size_t            StackSize = 8192;
NN_OS_ALIGNAS_THREAD_STACK char   g_ThreadStack1[StackSize];

void VibrationNodeThreadFunction(void *arg) NN_NOEXCEPT
{
    NN_UNUSED(arg);
    while (!g_IsQuitRequired)
    {
        nn::os::WaitTimerEvent(&g_TimerEvent);
        nn::hid::VibrationNode::Update();
    }
    nn::os::StopTimerEvent(&g_TimerEvent);
}

void InitializeGraphics() NN_NOEXCEPT
{
    // Memory
    g_pAppAllocator = new nn::mem::StandardAllocator();
    const size_t appMemorySize = 128 * 1024 * 1024;
    g_pAppMemory = new nn::Bit8[appMemorySize];
    g_pAppAllocator->Initialize(g_pAppMemory, appMemorySize);

#if defined(NN_BUILD_TARGET_PLATFORM_NX)
    const size_t graphicsMemorySize = 256 * 1024 * 1024;
    void* pGraphicsMemory = nns::gfx::GraphicsFramework::DefaultAllocateFunction(graphicsMemorySize, 1, nullptr);
    nv::SetGraphicsAllocator(nns::gfx::GraphicsFramework::DefaultAllocateFunction, nns::gfx::GraphicsFramework::DefaultFreeFunction, nns::gfx::GraphicsFramework::DefaultReallocateFunction, nullptr);
    nv::SetGraphicsDevtoolsAllocator(nns::gfx::GraphicsFramework::DefaultAllocateFunction, nns::gfx::GraphicsFramework::DefaultFreeFunction, nns::gfx::GraphicsFramework::DefaultReallocateFunction, nullptr);
    nv::InitializeGraphics(pGraphicsMemory, graphicsMemorySize);
#endif
    // Graphics
    g_pGraphicsSystem = new GraphicsSystem();
    g_pGraphicsSystem->SetApplicationHeap(g_pAppAllocator);
    g_pGraphicsSystem->Initialize();
}

void DrawTimeSeriesGraph(NpadControllerBase* controllerBase, int ctrlIdx, bool isLow) NN_NOEXCEPT
{
    int vibrationDeviceCount = controllerBase->GetVibrationDeviceCount();

    float baseX = 30.0f + 600.0f * (ctrlIdx / 4);
    float baseY = 70.0f + 160.0f * (ctrlIdx % 4);

    nn::gfx::util::DebugFontTextWriter* pTextWriter = &g_pGraphicsSystem->GetDebugFont();
    pTextWriter->SetTextColor(controllerBase->IsConnected() ? ActiveColor : DeactiveColor);
    pTextWriter->SetScale(1.0f, 1.0f);
    pTextWriter->SetCursor(baseX, baseY);
    pTextWriter->Print("%s : Time Series amplitude%s value", controllerBase->GetName(), isLow ? "Low" : "High");

    for (int vibIdx = 0; vibIdx < vibrationDeviceCount; vibIdx++)
    {
        const VibrationState& v = controllerBase->GetVibrationState(vibIdx);

        float x = baseX + 20.0f + 280.0f * vibIdx;
        float y = baseY + 25.0f;

        pTextWriter->SetTextColor(controllerBase->IsConnected() ? ActiveColor : DeactiveColor);
        pTextWriter->SetScale(0.8f, 0.8f);
        pTextWriter->SetCursor(x, y);
        switch (v.deviceInfo.position)
        {
        case nn::hid::VibrationDevicePosition_Left:
            pTextWriter->Print("Left");
            break;
        case nn::hid::VibrationDevicePosition_Right:
            pTextWriter->Print("Right");
            break;
        default:
            pTextWriter->Print("Unknown");
            break;
        }

        VibrationValueDrawer vibDrawer(pTextWriter, &g_pGraphicsSystem->GetCommandBuffer(), &g_pGraphicsSystem->GetPrimitiveRenderer());
        const VibrationValueBuffer& buffer = controllerBase->GetVibrationValueBuffer(vibIdx);
        vibDrawer.SetPosition(x, y + 15.0f);
        vibDrawer.SetBrightColor(controllerBase->IsConnected());
        vibDrawer.DrawTimeSeriesGraph(buffer, isLow);
    }
}

void DrawVibrationValue(NpadControllerBase* controllerBase, int ctrlIdx) NN_NOEXCEPT
{
    int vibrationDeviceCount = controllerBase->GetVibrationDeviceCount();

    float baseX = 30.0f + 600.0f * (ctrlIdx / 4);
    float baseY = 70.0f + 150.0f * (ctrlIdx % 4);

    nn::gfx::util::DebugFontTextWriter* pTextWriter = &g_pGraphicsSystem->GetDebugFont();
    pTextWriter->SetTextColor(controllerBase->IsConnected() ? ActiveColor : DeactiveColor);
    pTextWriter->SetScale(1.0f, 1.0f);
    pTextWriter->SetCursor(baseX, baseY);
    pTextWriter->Print("%s", controllerBase->GetName());

    for (int vibIdx = 0; vibIdx < vibrationDeviceCount; vibIdx++)
    {
        const VibrationState& v = controllerBase->GetVibrationState(vibIdx);

        float x = baseX + 20.0f + 280.0f * vibIdx;
        float y = baseY + 25.0f;

        pTextWriter->SetTextColor(controllerBase->IsConnected() ? ActiveColor : DeactiveColor);
        pTextWriter->SetScale(0.8f, 0.8f);
        pTextWriter->SetCursor(x, y);
        switch (v.deviceInfo.position)
        {
        case nn::hid::VibrationDevicePosition_Left:
            pTextWriter->Print("Left");
            break;
        case nn::hid::VibrationDevicePosition_Right:
            pTextWriter->Print("Right");
            break;
        default:
            pTextWriter->Print("Unknown");
            break;
        }

        VibrationValueDrawer vibDrawer(pTextWriter, &g_pGraphicsSystem->GetCommandBuffer(), &g_pGraphicsSystem->GetPrimitiveRenderer());
        vibDrawer.SetPosition(x, y + 15.0f);
        vibDrawer.SetScale(1.0f);
        vibDrawer.SetBrightColor(controllerBase->IsConnected());
        vibDrawer.DrawVibrationValue(v.actualVibrationValue, true);
    }
}

void UpdateGraphics(std::vector<NpadControllerBase*> controllers) NN_NOEXCEPT
{

    nn::gfx::util::DebugFontTextWriter* pTextWriter = &g_pGraphicsSystem->GetDebugFont();
    g_pGraphicsSystem->BeginDraw();
    pTextWriter->Draw(&g_pGraphicsSystem->GetCommandBuffer());

    // Sample Title
    {
        float scale = 2.0f;
        pTextWriter->SetScale(scale, scale);
        pTextWriter->SetCursor(10.0f, 10.0f);
        pTextWriter->SetTextColor(TextColor);
        pTextWriter->Print("%s", ProgramName);
    }

    // Left Column
    {
        float scale = 1.0;
        float baseX = 40.0f;
        float baseY = 100.0f;
        int progressBarWidth = 20;
        int pos = 0;
        float progress = 0.0f;
        static bool switchColor = false;
        const int switchOffset = 10;

        // Control instructions and status
        pTextWriter->SetScale(scale, scale);
        pTextWriter->SetCursor(baseX, baseY);
        pTextWriter->SetTextColor(TextColor);
        pTextWriter->Print("L : VibrationPlayer-A is ");
        pTextWriter->SetTextColor(g_PlayerA.IsPlaying() ? RedColor : TextColor);
        pTextWriter->Print("%s\n", g_PlayerA.IsPlaying() ? "Playing" : "Stopped");
        baseY += 30.0f;

        pTextWriter->SetCursor(baseX, baseY);
        pTextWriter->SetTextColor(TextColor);
        pTextWriter->Print("R : VibrationPlayer-B is ");
        pTextWriter->SetTextColor(g_PlayerB.IsPlaying() ? RedColor : TextColor);
        pTextWriter->Print("%s\n", g_PlayerB.IsPlaying() ? "Playing" : "Stopped");
        baseY += 50.0f;

        pTextWriter->SetCursor(baseX, baseY);
        pTextWriter->SetTextColor(TextColor);
        pTextWriter->Print("A : Increase Frequency\n");
        pTextWriter->Print("Y : Reduce Frequency\n");
        pTextWriter->Print("X : Volume Up\n");
        pTextWriter->Print("B : Volume Down\n");
        baseY += 20.0f * 5;

        pTextWriter->SetCursor(baseX, baseY);
        pTextWriter->SetTextColor(TextColor);
        pTextWriter->Print("Up : Speed Up\n");
        pTextWriter->Print("Down : Speed Down\n");
        pTextWriter->Print("Left : Pan left\n");
        pTextWriter->Print("Right : Pan right\n");
        baseY += 20.0f * 5;

        pTextWriter->SetCursor(baseX, baseY);
        pTextWriter->SetTextColor(TextColor);
        pTextWriter->Print("[A]             StartSound                   (SampleSe0)\n");
        pTextWriter->Print("[B]             StartSound                   (SampleSe1)\n");
        pTextWriter->Print("[X]             StartSound                   (SampleSe2)\n");
        pTextWriter->Print("[Y]             StartSound                   (SampleSe3)\n");
        pTextWriter->Print("[ZLeft/U]       ControlVolume                (SampleBgm0-2ch)\n");
        pTextWriter->Print("[ZRight/V]      ControlVolume                (SampleBgm0-2ch)\n");
        pTextWriter->Print("[Start/Space]   Shut down sample program\n");
        baseY += 20.0f * 8;

        // Network Download Status
        scale = 1.25f;
        pTextWriter->SetScale(scale, scale);
        pTextWriter->SetTextColor(TextColor);
        pTextWriter->SetCursor(baseX, baseY);
        pTextWriter->Print("Background Network Download\n");
        scale = 1.0f;
        pTextWriter->SetScale(scale, scale);
        pTextWriter->Print("Ip Address:\t\t\t%s\n", g_CurrentIpAddress);
        pTextWriter->Print("Number of Connections:\t");
        pTextWriter->SetTextColor(BlueColor);
        pTextWriter->Print("%d\n", g_NumConnections);
        pTextWriter->SetCursorX(baseX);
        pTextWriter->SetTextColor(TextColor);
        pTextWriter->Print("URL:\t\t\t\t");
        pTextWriter->SetTextColor(BlueColor);
        pTextWriter->Print("%s\n", g_NetworkDownloadUrl);
        pTextWriter->SetCursorX(baseX);
        pTextWriter->SetTextColor(TextColor);
        pTextWriter->Print("Download speed:\t\t");
        pTextWriter->SetTextColor(g_IsDownloading ? GreenColor : RedColor);
        pTextWriter->Print("%.2f Mbits/sec\n", g_CurrentNetworkDownloadThroughput);
        pTextWriter->SetCursorX(baseX);
        pTextWriter->SetTextColor(TextColor);
        pTextWriter->Print("Status:\t\t\t\t");
        pTextWriter->SetTextColor(g_IsDownloading ? GreenColor : RedColor);
        pTextWriter->Print("%s\n", g_IsDownloading ? "Downloading" : "Error");
        pTextWriter->SetCursorX(baseX);
        pTextWriter->SetTextColor(TextColor);
        pTextWriter->Print("Downloaded: [");

        if (g_MaxProgressDownload == 1.0 || !g_IsDownloading)
        {
            g_MaxProgressDownload = 0.0f;
            g_CurrentDownloadedSize = 0.0f;
            g_TotalDownloadSize = 0.0f;
        }
        else
        {
            progress = g_CurrentDownloadedSize / g_TotalDownloadSize;
        }

        g_MaxProgressDownload = std::max(g_MaxProgressDownload, progress);
        pos = progressBarWidth * g_MaxProgressDownload;
        if (g_ProgressFunctionCount > g_LastProgressFunctionCount + switchOffset)
        {
            switchColor = !switchColor;
            g_LastProgressFunctionCount = g_ProgressFunctionCount;
        }
        pTextWriter->SetTextColor(switchColor ? OrangeColor : RedColor);
        for (int i = 1; i < progressBarWidth; ++i)
        {
            if (i <= pos)
            {
                pTextWriter->Print("■");
            }
            else
            {
                pTextWriter->Print("□");
            }
        }

        pTextWriter->SetTextColor(TextColor);
        pTextWriter->Print("] %d %\n", (int)(g_MaxProgressDownload * 100.0));
        pTextWriter->SetCursorX(baseX);
        pTextWriter->Print("Disconnected:\t\t\t%d times\n", g_NumofDisconnects);
        pTextWriter->Print("\n");
    }

    // Right Column
    {
        // Microphone UI
        float scale = 1.25;
        float baseX = 30.0f + 600.0f;
        float baseY = 30.0f;

        pTextWriter->SetScale(scale, scale);
        pTextWriter->SetTextColor(TextColor);
        pTextWriter->SetCursor(baseX, baseY);
        pTextWriter->Print("マイク Microphones\n");
        scale = 1.0f;
        pTextWriter->SetScale(scale, scale);
        pTextWriter->Print("\n");
        pTextWriter->Print("Audio In buffer length in milliseconds: ");
        pTextWriter->SetTextColor(BlueColor);
        pTextWriter->Print("%d\n", g_AudioInBufferLengthInMilliseconds);
        pTextWriter->SetCursorX(baseX);
        pTextWriter->SetTextColor(TextColor);
        pTextWriter->Print("\n");
        for (auto i = 0; i < 2; ++i)
        {
            pTextWriter->SetTextColor(TextColor);
            if (g_MicData[i].inUse)
            {
                pTextWriter->SetTextColor(OrangeColor);
            }
            pTextWriter->Print("マイク %d / Mic %d : %s\n", i, i, g_MicData[i].inUse ? "接続済 / Connected" : "未接続 / Disconnected");
            std::string level = "レベル / Level : ";
            float max = 0;
            for (auto j = 0; j < g_MicData[i].RmsCountMax; ++j)
            {
                max = std::max(max, g_MicData[i].rms[j]);
            }
            int count = static_cast<int>((max) * 20);
            for (auto j = 0; j < count; ++j)
            {
                level.append("■");
            }
            pTextWriter->Print("%s\n", level.c_str());
            pTextWriter->Print("\n");
        }

        // Vibration UI
        NpadControllerBase* controller = controllers.front();
        int ctrlIdx = 5;
        DrawVibrationValue(controller, ctrlIdx++);
        DrawTimeSeriesGraph(controller, ctrlIdx++, true);
        DrawTimeSeriesGraph(controller, ctrlIdx++, false);

    }

    g_pGraphicsSystem->EndDraw();
    g_pGraphicsSystem->Synchronize(
    nn::TimeSpan::FromNanoSeconds(1000L * 1000L * 1000L / DisplayFrameRate));
} //NOLINT(impl/function_size)

void FinalizeGraphics() NN_NOEXCEPT
{
    g_pGraphicsSystem->Finalize();
    delete g_pGraphicsSystem;

    g_pAppAllocator->Finalize();
    delete g_pAppAllocator;

    delete[] g_pAppMemory;
}

void* Allocate(size_t size)
{
    return g_Allocator.Allocate(size);
}

void Deallocate(void* p, size_t size)
{
    NN_UNUSED(size);
    g_Allocator.Free(p);
}

std::size_t ReadAdpcmFile(nn::audio::AdpcmHeaderInfo* header, void** adpcmData, const char* filename)
{
    nn::fs::FileHandle handle;
    nn::Result result = nn::fs::OpenFile(&handle, filename, nn::fs::OpenMode_Read);
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);

    int64_t size;
    uint8_t adpcmheader[nn::audio::AdpcmHeaderSize];

    result = nn::fs::GetFileSize(&size, handle);
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);

    *adpcmData = g_WaveBufferAllocator.Allocate(static_cast<std::size_t>(size) - sizeof(adpcmheader), nn::audio::BufferAlignSize);
    NN_ABORT_UNLESS_NOT_NULL(*adpcmData);

    result = nn::fs::ReadFile(handle, 0, adpcmheader, sizeof(adpcmheader));
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);
    result = nn::fs::ReadFile(handle, sizeof(adpcmheader), *adpcmData, static_cast<size_t>(size) - sizeof(adpcmheader));
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);
    nn::fs::CloseFile(handle);

    nn::audio::ParseAdpcmHeader(header, adpcmheader, sizeof(adpcmheader));

    return static_cast<std::size_t>(size) - sizeof(adpcmheader);
}

std::size_t ReadWavFile(nns::audio::WavFormat* format, void** data, const char* filename)
{
    nn::fs::FileHandle handle;
    nn::Result result = nn::fs::OpenFile(&handle, filename, nn::fs::OpenMode_Read);
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);

    int64_t size;

    result = nn::fs::GetFileSize(&size, handle);
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);

    *data = g_WaveBufferAllocator.Allocate(static_cast<std::size_t>(size), nn::audio::BufferAlignSize);
    NN_ABORT_UNLESS_NOT_NULL(*data);

    // DATA チャンクを読む必要がありますが、ここではそれが 1024 バイト以内に見つかると仮定しています
    const std::size_t WavHeaderDataSize = 1024;

    result = nn::fs::ReadFile(handle, 0, *data, WavHeaderDataSize);
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);

    nns::audio::WavResult wavResult = nns::audio::ParseWavFormat(format, *data, WavHeaderDataSize);
    NN_ABORT_UNLESS_EQUAL(wavResult, nns::audio::WavResult_Success);
    NN_ABORT_UNLESS_EQUAL(format->bitsPerSample, 16);  // このサンプルでは 16bit PCM を仮定しています

    result = nn::fs::ReadFile(handle, static_cast<std::size_t>(format->dataOffset), *data, static_cast<std::size_t>(format->dataSize));
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);
    nn::fs::CloseFile(handle);

    return static_cast<std::size_t>(format->dataSize);
}

void InitializeFileSystem()
{
    nn::fs::SetAllocator(Allocate, Deallocate);

    size_t cacheSize = 0;
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::QueryMountRomCacheSize(&cacheSize));
    g_MountRomCacheBuffer = new(std::nothrow) char[cacheSize];
    NN_ABORT_UNLESS_NOT_NULL(g_MountRomCacheBuffer);

    NN_ABORT_UNLESS_RESULT_SUCCESS(
        nn::fs::MountRom("asset", g_MountRomCacheBuffer, cacheSize)
    );
}

void FinalizeFileSystem()
{
    nn::fs::Unmount("asset");

    delete[] g_MountRomCacheBuffer;
    g_MountRomCacheBuffer = NULL;
}

void InitializeHidDevices()
{
    nn::hid::InitializeDebugPad();
    nn::hid::InitializeNpad();
    const nn::hid::NpadIdType npadIds[2] = { nn::hid::NpadId::No1, nn::hid::NpadId::Handheld };
    nn::hid::SetSupportedNpadStyleSet(nn::hid::NpadStyleFullKey::Mask | nn::hid::NpadStyleHandheld::Mask);
    nn::hid::SetSupportedNpadIdType(npadIds, sizeof(npadIds) / sizeof(npadIds[0]));

    //キーボードのキーを DebugPad のボタンに割り当てます。
    nn::settings::DebugPadKeyboardMap map;
    nn::settings::GetDebugPadKeyboardMap(&map);
    map.buttonA     = nn::hid::KeyboardKey::A::Index;
    map.buttonB     = nn::hid::KeyboardKey::B::Index;
    map.buttonX     = nn::hid::KeyboardKey::X::Index;
    map.buttonY     = nn::hid::KeyboardKey::Y::Index;
    map.buttonL     = nn::hid::KeyboardKey::L::Index;
    map.buttonR     = nn::hid::KeyboardKey::R::Index;
    map.buttonZL    = nn::hid::KeyboardKey::U::Index;
    map.buttonZR    = nn::hid::KeyboardKey::V::Index;
    map.buttonLeft  = nn::hid::KeyboardKey::LeftArrow::Index;
    map.buttonRight = nn::hid::KeyboardKey::RightArrow::Index;
    map.buttonUp    = nn::hid::KeyboardKey::UpArrow::Index;
    map.buttonDown  = nn::hid::KeyboardKey::DownArrow::Index;
    map.buttonStart = nn::hid::KeyboardKey::Space::Index;
    nn::settings::SetDebugPadKeyboardMap(map);
}

void BackgroundNetworkDownload(void *arg) NN_NOEXCEPT
{
    NN_UNUSED(arg);

    nn::Result      result;
    in_addr         outIpAddress = { 0 };
    CURLcode        res = CURLE_OK;
    uint32_t        fileSize = 104857600;
    int64_t         serverRTT = -1;
    Options         options = { g_NumConnections,      // numConnections
                                1,      // timesToDownload
                                -1,     // sndBufSize
                                -1,     // rcvBufSize
                                0,      // disableTcpNagle
                                0,      // uploadSize
                                ""      // url
                                };

    memcpy(options.url, g_NetworkDownloadUrl, kMaxUrlLength + 1);

    // initialize system network interface
    nn::nifm::Initialize();
    while(!g_IsQuitRequired)
    {
        nn::nifm::SubmitNetworkRequest();

        /* wait for network interface availability, while providing status */
        while (nn::nifm::IsNetworkRequestOnHold())
        {
            NN_LOG("Waiting for network interface availability...\n");
            nn::os::SleepThread(nn::TimeSpan::FromSeconds(1));
        }

        if (!nn::nifm::IsNetworkAvailable())
        {
            NN_LOG("\n[ERROR] network is not available.\n");
            goto CLEANUP_NIFM;
        }

        result = nn::nifm::GetCurrentPrimaryIpAddress(&outIpAddress);
        if (result.IsSuccess())
        {
            g_CurrentIpAddress = nn::socket::InetNtoa(outIpAddress);
            NN_LOG("IP Address: %s\n", g_CurrentIpAddress);
        }
        else
        {
            NN_LOG("server:  GetCurrentIpAddress failed (error %d)\n", result.GetDescription());
            goto CLEANUP_SOCKET_AND_NIFM;
        }

        /* initialize the socket library, providing application use case specific amount of memory */
        result = nn::socket::Initialize(g_SocketMemoryPoolBuffer,
            nn::socket::DefaultSocketMemoryPoolSize,
            nn::socket::MinSocketAllocatorSize,
            nn::socket::DefaultConcurrencyLimit);

        if (result.IsFailure())
        {
            NN_LOG("\n[ERROR] nn::socket::Initialize() failed. Err Desc: %d\n\n", result.GetDescription());
            goto CLEANUP_NIFM;
        }
        /* initialize using system malloc for libcurl for testing */
        res = curl_global_init(CURL_GLOBAL_DEFAULT);
        if (res != CURLE_OK)
        {
            NN_LOG("\n[ERROR] curl_global_init failed. Err: %d\n\n", res);
            goto CLEANUP_SOCKET_AND_NIFM;
        }
        NN_LOG("Network ready\n");

        /* Setup for HTTPS. The SSL context will be allocated inside g_HttpsHelper.
        The allocated SSL context will be imported into libcurl by g_HttpsHelper.ImportSslContext
        below after CURL handle is created, and when URL contains https.
        The allocated SSL context will be freed in g_HttpsHelper.Finalize after all curl handles
        which imported this context are closed at the bottom.
        */
        if (g_HttpsHelper.Initialize() < 0)
        {
            NN_LOG("[ERROR] Failed to initialize HTTPS helper util.\n");
            goto CLEANUP_CURL_AND_SOCKET_AND_NIFM;
        }

        //if (!GetFileSize(options, &fileSize))
        //{
        //    NN_LOG("Failed to get file size!\n");
        //    goto CLEANUP_CURL_AND_SOCKET_AND_NIFM;
        //}

        g_IsDownloading = true;
        while (NN_STATIC_CONDITION(true))
        {
            g_CurrentNetworkDownloadThroughput = DownloadFile(1, options, fileSize);
            if (g_CurrentNetworkDownloadThroughput == 0.0f)
            {
                g_IsDownloading = false;
                g_NumofDisconnects++;
                break;
            }
        }

        //NN_LOG("times to download: %d\n", options.timesToDownload);
        //fAverageThroughput /= options.timesToDownload;

        //NN_LOG("\n******* Results *******\n");
        //NN_LOG("******* URL: %s\n", options.url);
        //NN_LOG("******* Average Throughput: %.3f Mbits/sec\n", fAverageThroughput);

        //if (fileSize / kGigabyte >= 1.0f)
        //{
        //  NN_LOG("******* File Size: %.3f GB\n", fileSize / kGigabyte);
        //}
        //else if (fileSize / kMegabyte >= 1.0f)
        //{
        //  NN_LOG("******* File Size: %.3f MB\n", fileSize / kMegabyte);
        //}
        //else if (fileSize / kKilobyte >= 1.0f)
        //{
        //  NN_LOG("******* File Size: %.3f KB\n", fileSize / kKilobyte);
        //}
        //else
        //{
        //  NN_LOG("******* File Size: %u Bytes\n", fileSize);
        //}

        //NN_LOG("******* Number of Downloads: %u\n", options.timesToDownload);
        //NN_LOG("******* Number of Connections: %u\n", options.numConnections);

        if (serverRTT >= 0)
        {
            NN_LOG("******* Server RTT: %lld ms\n\n", serverRTT);
        }
        else
        {
            NN_LOG("******* Server RTT: Unknown\n\n");
        }

CLEANUP_CURL_AND_SOCKET_AND_NIFM:
    /* cleanup libcurl library */
    curl_global_cleanup();

CLEANUP_SOCKET_AND_NIFM:
    /* cleanup socket library */
    nn::socket::Finalize();
CLEANUP_NIFM:
    NN_LOG("Network Disconnected.\n");
    nn::nifm::CancelNetworkRequest();
    nn::os::SleepThread(nn::TimeSpan::FromSeconds(1));
    }
}


void mainLoop()
{
    nn::Result          result;
    AudioOutArgs        audioOutArgs;
    AudioInArgs         audioInArgs;

    g_Allocator.Initialize(g_WorkBuffer, sizeof(g_WorkBuffer));
    g_WaveBufferAllocator.Initialize(g_WaveBufferPoolMemory, sizeof(g_WaveBufferPoolMemory));

    InitializeFileSystem();
    InitializeHidDevices();
    InitializeGraphics();

    // Audio in
    nn::mem::StandardAllocator allocator(g_HeapBuffer, sizeof(g_HeapBuffer));
    audioInArgs.allocator = &allocator;
    nn::audio::SetAudioInEnabled(true);
    nn::audio::AudioOutParameter audioOutParameter;
    nn::audio::AudioOut audioOut;

    auto argc = nn::os::GetHostArgc();
    auto argv = nn::os::GetHostArgv();

    // Arguments: audioInBuffernLengthInMilliseconds numConnections networkDownloadUrl
    if (argc >= 2)
    {
        g_AudioInBufferLengthInMilliseconds = atoi(argv[1]);
    }
    if (argc >= 3)
    {
        g_NumConnections = atoi(argv[2]);
    }
    if (g_NumConnections > 32)
    {
        NN_LOG("Invalid numConnection argument of %d. Setting to 1\n");
        g_NumConnections = 1;
    }
    if (argc >= 4)
    {
        memset(g_NetworkDownloadUrl, 0, kMaxUrlLength + 1);
        memcpy(g_NetworkDownloadUrl, argv[3], strlen(argv[3]));
    }

    if (g_AudioInBufferLengthInMilliseconds > 50 || g_AudioInBufferLengthInMilliseconds < 0)
    {
        NN_LOG("Clamping bufferLengthInMilliseconds from %d to 50\n", g_AudioInBufferLengthInMilliseconds);
        g_AudioInBufferLengthInMilliseconds = 50;
    }

    audioInArgs.bufferLengthInMilliseconds = g_AudioInBufferLengthInMilliseconds;
    nn::os::SystemEvent audioInConectEvent;
    nn::os::SystemEvent audioOutBufferEvent;

    nn::audio::InitializeAudioOutParameter(&audioOutParameter);

    NN_ABORT_UNLESS(
        nn::audio::OpenDefaultAudioOut(&audioOut, &audioOutBufferEvent, audioOutParameter).IsSuccess(),
        "Failed to open AudioOut."
    );

    int channelCount = nn::audio::GetAudioOutChannelCount(&audioOut);

    int sampleRate = nn::audio::GetAudioOutSampleRate(&audioOut);
    nn::audio::SampleFormat sampleFormat = nn::audio::GetAudioOutSampleFormat(&audioOut);
    NN_LOG("channelCount: %d, sampleRate: %d, sampleFormat: %d\n", channelCount, sampleRate, sampleFormat);
    // バッファに関するパラメータを準備します。
    NN_LOG("bufferLengthInMilliseconds = %d\n", g_AudioInBufferLengthInMilliseconds);
    int millisecondsPerSecond = 1000;
    int audioFrameRate = millisecondsPerSecond / g_AudioInBufferLengthInMilliseconds;;
    int frameSampleCount = sampleRate / audioFrameRate;
    size_t dataSize = frameSampleCount * channelCount * nn::audio::GetSampleByteSize(sampleFormat);
    size_t bufferSize = nn::util::align_up(dataSize, nn::audio::AudioInBuffer::SizeGranularity);
    const int outBufferCount = 2;

    for (int i = 0; i < MaxAudioIns; ++i)
    {
        g_MicData[i].SetFrameSize(dataSize);
    }

    nn::audio::AudioOutBuffer audioOutBuffer[outBufferCount];
    void* outBuffer[outBufferCount];

    for (int i = 0; i < outBufferCount; ++i)
    {
        outBuffer[i] = allocator.Allocate(bufferSize, nn::audio::AudioOutBuffer::AddressAlignment);
        NN_ASSERT(outBuffer[i]);
        std::memset(outBuffer[i], 0, bufferSize);
        nn::audio::SetAudioOutBufferInfo(&audioOutBuffer[i], outBuffer[i], bufferSize, dataSize);
        nn::audio::AppendAudioOutBuffer(&audioOut, &audioOutBuffer[i]);
    }

    NN_ABORT_UNLESS(
        nn::audio::StartAudioOut(&audioOut).IsSuccess(),
        "Failed to start playback."
    );

    // 1 フレーム待ちます。
    const nn::TimeSpan frameInterval(nn::TimeSpan::FromNanoSeconds(1000 * 1000 * 1000 / audioFrameRate));
    nn::os::SleepThread(frameInterval);

    audioOutArgs.bufferEvent = &audioOutBufferEvent;
    audioOutArgs.audioOut = &audioOut;
    audioOutArgs.isRunning = true;

    audioInArgs.isRunning = true;
    nn::os::CreateThread(&g_AudioInThread, AudioInThread, &audioInArgs, g_AudioInThreadStack,
        ThreadStackSize, nn::os::DefaultThreadPriority);


    nn::os::CreateThread(&g_AudioOutThread, AudioOutThread, &audioOutArgs, g_AudioOutThreadStack,
        ThreadStackSize, nn::os::DefaultThreadPriority);

    nn::os::StartThread(&g_AudioInThread);
    nn::os::StartThread(&g_AudioOutThread);

    // Hid Vibration objects
    std::vector<NpadControllerBase*> controllers;
    nns::iodevices::SpeedChangeableVibrationPlayer* players[2] = { &g_PlayerA, &g_PlayerB };

    // Network download thread
    nn::os::ThreadType  networkDownloadThread;
    result = nn::os::CreateThread(&networkDownloadThread, BackgroundNetworkDownload, NULL, g_NetworkDownloadStack, ThreadStackSize, nn::os::DefaultThreadPriority);
    NN_ASSERT(result.IsSuccess(), "Cannot create thread.");
    nn::os::StartThread(&networkDownloadThread);

    nn::os::ThreadType  thread;
    // タイマーイベントを初期化する
    nn::os::InitializeTimerEvent(&g_TimerEvent, nn::os::EventClearMode_AutoClear);
    // スレッドを生成する
    result = nn::os::CreateThread(&thread, VibrationNodeThreadFunction, NULL, g_ThreadStack1, StackSize, nn::os::DefaultThreadPriority);
    NN_ASSERT(result.IsSuccess(), "Cannot create thread.");
    // スレッドの実行を開始する
    nn::os::StartThread(&thread);
    // タイマーイベントは周期タイマーイベントとして開始する
    const nn::TimeSpan interval = nn::hid::VibrationNode::DefaultVibrationSampleInterval;
    nn::os::StartPeriodicTimerEvent(&g_TimerEvent, interval, interval);


    // レンダラのパラメータを指定します。
    nn::audio::AudioRendererParameter parameter;
    nn::audio::InitializeAudioRendererParameter(&parameter);
    parameter.sampleRate = RenderRate;
    parameter.sampleCount = RenderCount;
    parameter.mixBufferCount = 6 + 2; // FinalMix(6) + SubMix(2)
    parameter.voiceCount = 24;
    parameter.subMixCount = 2;
    parameter.sinkCount = 1;
    parameter.effectCount = 2;
    parameter.performanceFrameCount = 0;

    // ミックスバッファとオーディオバスの関係を定義します。
    channelCount = 2;
    int8_t mainBus[2];
    mainBus[nn::audio::ChannelMapping_FrontLeft] = 4;
    mainBus[nn::audio::ChannelMapping_FrontRight] = 5;
    int8_t auxBusA[2];
    auxBusA[nn::audio::ChannelMapping_FrontLeft] = 0;
    auxBusA[nn::audio::ChannelMapping_FrontRight] = 1;

    // パラメータがシステムでサポートされているかどうかを確認します。
    NN_ABORT_UNLESS(
        nn::audio::IsValidAudioRendererParameter(parameter),
        "Invalid AudioRendererParameter specified."
    );

    // レンダラのワークバッファを準備します。
    size_t workBufferSize = nn::audio::GetAudioRendererWorkBufferSize(parameter);
    void* workBuffer = g_Allocator.Allocate(workBufferSize, nn::os::MemoryPageSize);
    NN_ABORT_UNLESS_NOT_NULL(workBuffer);

    nn::os::SystemEvent systemEvent;

    // レンダラを初期化します。
    nn::audio::AudioRendererHandle handle;
    NN_ABORT_UNLESS(
        nn::audio::OpenAudioRenderer(&handle, &systemEvent, parameter, workBuffer, workBufferSize).IsSuccess(),
        "Failed to open AudioRenderer"
    );

    // AudioRendererConfig を初期化します。
    size_t configBufferSize = nn::audio::GetAudioRendererConfigWorkBufferSize(parameter);
    void* configBuffer = g_Allocator.Allocate(configBufferSize, nn::os::MemoryPageSize);
    NN_ABORT_UNLESS_NOT_NULL(configBuffer);
    nn::audio::AudioRendererConfig config;
    nn::audio::InitializeAudioRendererConfig(&config, parameter, configBuffer, configBufferSize);

    nn::audio::FinalMixType finalMix;
    nn::audio::AcquireFinalMix(&config, &finalMix, 6);

    nn::audio::SubMixType subMix0;
    nn::audio::AcquireSubMix(&config, &subMix0, parameter.sampleRate, 1);
    nn::audio::SubMixType subMix1;
    nn::audio::AcquireSubMix(&config, &subMix1, parameter.sampleRate, 1);

    // レンダラの出力先を用意します。
    nn::audio::DeviceSinkType deviceSink;
    // オーディオ出力デバイスへの入力を設定します。
    // mainBus に指定したミックスバッファのインデックスに応じて出力チャンネルが決定されます。
    // mainBus[nn::audio::ChannelMapping_FrontLeft] が L チャンネルに、
    // mainBus[nn::audio::ChannelMapping_FrontRight] が R チャンネルにそれぞれ出力されます。
    result = nn::audio::AddDeviceSink(&config, &deviceSink, &finalMix, mainBus, channelCount, "MainAudioOut");
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);

    // SubMix(0) を FinalMix へ接続します
    nn::audio::SetSubMixDestination(&config, &subMix0, &finalMix);

    // SubMix(0) の 0 番目のバッファを FinalMix の mainBus[0]、mainBus[1] へ
    ///ボリューム 0.5f でミックスするように設定します
    nn::audio::SetSubMixMixVolume(&subMix0, &finalMix, 0.5f, 0, mainBus[0]);
    nn::audio::SetSubMixMixVolume(&subMix0, &finalMix, 0.5f, 0, mainBus[1]);

    // SubMix(1) を SubMix(0) へ接続します
    nn::audio::SetSubMixDestination(&config, &subMix1, &subMix0);

    // SubMix(1) の 0 番目のバッファを SubMix(0) の 0 番目のバッファへ
    // ボリューム 0.5f でミックスするように設定します
    nn::audio::SetSubMixMixVolume(&subMix1, &subMix0, 0.5f, 0, 0);

    // BufferMixer を用意し、登録します。
    nn::audio::BufferMixerType mixer1;
    result = nn::audio::AddBufferMixer(&config, &mixer1, &finalMix);
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);
    nn::audio::SetBufferMixerInputOutput(&mixer1, auxBusA, mainBus, channelCount);

    // BufferMixer パラメータを設定します。
    nn::audio::SetBufferMixerVolume(&mixer1, 0, 1.0f);
    nn::audio::SetBufferMixerVolume(&mixer1, 1, 1.0f);

    // 設定したパラメータをレンダラに反映させます。
    result = nn::audio::RequestUpdateAudioRenderer(handle, &config);
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);

    // レンダリングを開始します。
    result = nn::audio::StartAudioRenderer(handle);
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);

    // WaveBuffer に追加するサンプルデータを保持するためのメモリプールを準備します。
    nn::audio::MemoryPoolType waveBufferMemoryPool;
    auto ret = AcquireMemoryPool(&config, &waveBufferMemoryPool, g_WaveBufferPoolMemory, sizeof(g_WaveBufferPoolMemory));
    NN_ABORT_UNLESS(ret);
    ret = RequestAttachMemoryPool(&waveBufferMemoryPool);
    NN_ABORT_UNLESS(ret);



    // BGM
    nn::audio::VoiceType voiceBgm[BgmCount];
    nn::audio::WaveBuffer waveBufferBgm;
    void* dataBgm[BgmCount];

    for (int i = 0; i < BgmCount; ++i)
    {
        nns::audio::WavFormat format;
        std::size_t dataSize = ReadWavFile(&format, &dataBgm[i], g_BgmFileNames[i]);

        // 読み込んだ WAV 形式ファイルのチャンネル数を指定してボイスを取得します。
        // マルチチャンネルのデータを読み込んだ場合には、チャンネル数分のボイスを消費することに注意してください。
        nn::audio::AcquireVoiceSlot(&config, &voiceBgm[i], format.sampleRate, format.channelCount, nn::audio::SampleFormat_PcmInt16, nn::audio::VoiceType::PriorityHighest, nullptr, 0);
        nn::audio::SetVoiceDestination(&config, &voiceBgm[i], &finalMix);

        waveBufferBgm.buffer = dataBgm[i];
        waveBufferBgm.size = dataSize;
        waveBufferBgm.startSampleOffset = 0;
        waveBufferBgm.endSampleOffset = static_cast<int32_t>(dataSize / sizeof(int16_t)) / format.channelCount;
        waveBufferBgm.loop = true;
        waveBufferBgm.isEndOfStream = false;
        waveBufferBgm.pContext = nullptr;
        waveBufferBgm.contextSize = 0;

        nn::audio::AppendWaveBuffer(&voiceBgm[i], &waveBufferBgm);
        nn::audio::SetVoicePlayState(&voiceBgm[i], nn::audio::VoiceType::PlayState_Play);
        // ボイスの 0 チャンネルを mainBus[0] へ、1 チャンネルを mainBus[1] へ出力するミックスボリュームを設定します。
        nn::audio::SetVoiceMixVolume(&voiceBgm[i], &finalMix, 0.5f, 0, mainBus[0]);
        nn::audio::SetVoiceMixVolume(&voiceBgm[i], &finalMix, 0.5f, 1, mainBus[1]);
    }

    // SE
    nn::audio::VoiceType voiceSe[SeCount];
    nn::audio::WaveBuffer waveBufferSe[SeCount];
    nn::audio::AdpcmHeaderInfo* header[SeCount];
    void* dataSe[SeCount];

    for (int i = 0; i < SeCount; ++i)
    {
        header[i] = reinterpret_cast<nn::audio::AdpcmHeaderInfo*>(g_WaveBufferAllocator.Allocate(sizeof(nn::audio::AdpcmHeaderInfo), NN_ALIGNOF(nn::audio::AdpcmHeaderInfo)));
        std::size_t dataSeSize = ReadAdpcmFile(header[i], &dataSe[i], g_SeFileNames[i]);
        nn::audio::AcquireVoiceSlot(&config, &voiceSe[i], header[i]->sampleRate, 1, nn::audio::SampleFormat_Adpcm, nn::audio::VoiceType::PriorityHighest, &header[i]->parameter, sizeof(nn::audio::AdpcmParameter));
        nn::audio::SetVoiceDestination(&config, &voiceSe[i], &finalMix);

        waveBufferSe[i].buffer = dataSe[i];
        waveBufferSe[i].size = dataSeSize;
        waveBufferSe[i].startSampleOffset = 0;
        waveBufferSe[i].endSampleOffset = header[i]->sampleCount;
        waveBufferSe[i].loop = false;
        waveBufferSe[i].isEndOfStream = false;
        waveBufferSe[i].pContext = &header[i]->loopContext;
        waveBufferSe[i].contextSize = sizeof(nn::audio::AdpcmContext);

        nn::audio::AppendWaveBuffer(&voiceSe[i], &waveBufferSe[i]);
        nn::audio::SetVoicePlayState(&voiceSe[i], nn::audio::VoiceType::PlayState_Play);
        nn::audio::SetVoiceMixVolume(&voiceSe[i], &finalMix, 0.707f / 2, 0, auxBusA[0]);
        nn::audio::SetVoiceMixVolume(&voiceSe[i], &finalMix, 0.707f / 2, 0, auxBusA[1]);
    }

    // Initialize controller vibration
    controllers.push_back(new NpadFullKeyController(nn::hid::NpadId::No1, "Id:1 FullKeyStyle"));

    for (std::vector<NpadControllerBase*>::iterator it = controllers.begin();
        it != controllers.end();
        ++it)
    {
        (*it)->Initialize();
    }

    LoadVibrationResources();


    for (std::vector<NpadControllerBase*>::iterator it = controllers.begin();
        it != controllers.end();
        ++it)
    {
        (*it)->SetVibrationPlayers(players, NN_ARRAY_SIZE(players));
        (*it)->InitializeVibrationNode();
    }
    g_IsQuitRequired = false;

    // 波形再生が終わるまで待ちつつ、パラメータを更新します。
    for (;;)
    {
        for (std::vector<NpadControllerBase*>::iterator it = controllers.begin();
            it != controllers.end();
            ++it)
        {
            (*it)->Update();

            if ((*it)->IsQuitRequired())
            {
                g_IsQuitRequired = true;
            }
        }

        UpdateGraphics(controllers);

        systemEvent.Wait();

        nn::hid::NpadButtonSet npadButtonCurrent = {};
        nn::hid::NpadButtonSet npadButtonDown = {};
        nn::hid::AnalogStickState analogStickStateL = {};
        nn::hid::AnalogStickState analogStickStateR = {};

        // Npad の入力を取得します。
        if (nn::hid::GetNpadStyleSet(nn::hid::NpadId::No1).Test<nn::hid::NpadStyleFullKey>())
        {
            static nn::hid::NpadFullKeyState npadFullKeyState = {};
            nn::hid::NpadFullKeyState state;
            nn::hid::GetNpadState(&state, nn::hid::NpadId::No1);
            npadButtonCurrent |= state.buttons;
            npadButtonDown |= state.buttons & ~npadFullKeyState.buttons;
            analogStickStateL.x += state.analogStickL.x;
            analogStickStateL.y += state.analogStickL.y;
            analogStickStateR.x += state.analogStickR.x;
            analogStickStateR.y += state.analogStickR.y;
            npadFullKeyState = state;
        }
        if (nn::hid::GetNpadStyleSet(nn::hid::NpadId::Handheld).Test<nn::hid::NpadStyleHandheld>())
        {
            static nn::hid::NpadHandheldState npadHandheldState = {};
            nn::hid::NpadHandheldState state;
            nn::hid::GetNpadState(&state, nn::hid::NpadId::Handheld);
            npadButtonCurrent |= state.buttons;
            npadButtonDown |= state.buttons & ~npadHandheldState.buttons;
            analogStickStateL.x += state.analogStickL.x;
            analogStickStateL.y += state.analogStickL.y;
            analogStickStateR.x += state.analogStickR.x;
            analogStickStateR.y += state.analogStickR.y;
            npadHandheldState = state;
        }

        // DebugPad の入力を取得します。
        {
            static nn::hid::DebugPadState debugPadState = {};
            nn::hid::DebugPadButtonSet debugPadButtonCurrent = {};
            nn::hid::DebugPadButtonSet debugPadButtonDown = {};
            nn::hid::DebugPadState state;
            nn::hid::GetDebugPadState(&state);
            debugPadButtonCurrent |= state.buttons;
            debugPadButtonDown |= state.buttons & ~debugPadState.buttons;
            analogStickStateL.x += state.analogStickL.x;
            analogStickStateL.y += state.analogStickL.y;
            analogStickStateR.x += state.analogStickR.x;
            analogStickStateR.y += state.analogStickR.y;
            debugPadState = state;
            nns::audio::ConvertDebugPadButtonsToNpadButtons(&npadButtonCurrent, debugPadButtonCurrent);
            nns::audio::ConvertDebugPadButtonsToNpadButtons(&npadButtonDown, debugPadButtonDown);
        }

        //SE を再生します（同じ SE を多重再生しません）。
        for (int i = 0; i < SeCount; ++i)
        {
            // SE番号 {0, 1, 2, 3, 4, 5} が、ボタン {A, B, X, Y, L, R} に対応します。
            if(npadButtonDown.Test(i))
            {
                if (nn::audio::GetReleasedWaveBuffer(&voiceSe[i]))
                {
                    nn::audio::AppendWaveBuffer(&voiceSe[i], &waveBufferSe[i]);
                }
            }
        }

        for (int i = 0; i < BgmCount; ++i)
        {
            // 左チャンネル
            float bgmLeftVolume = nn::audio::GetVoiceMixVolume(&voiceBgm[i], &finalMix, 0, mainBus[0]);
            if(npadButtonCurrent.Test< ::nn::hid::NpadButton::ZL >())
            {
                bgmLeftVolume += 0.01f;
            }

            if(npadButtonCurrent.Test< ::nn::hid::NpadButton::ZR >())
            {
                bgmLeftVolume -= 0.01f;
            }

            if (bgmLeftVolume < 1.0f && bgmLeftVolume > 0.0f)
            {
                nn::audio::SetVoiceMixVolume(&voiceBgm[i], &finalMix, bgmLeftVolume, 0, mainBus[0]);
            }

            // 右チャンネル
            float bgmRightVolume = 1.0f - bgmLeftVolume;

            if (bgmRightVolume < 1.0f && bgmRightVolume > 0.0f)
            {
                nn::audio::SetVoiceMixVolume(&voiceBgm[i], &finalMix, bgmRightVolume, 1, mainBus[1]);
            }
        }

        if(npadButtonDown.Test< ::nn::hid::NpadButton::Plus >())
        {
            g_IsQuitRequired = true;
            break;
        }

        result = nn::audio::RequestUpdateAudioRenderer(handle, &config);
        NN_ABORT_UNLESS_RESULT_SUCCESS(result);

    }

    // 管理対象の操作形態に関する削除
    for (std::vector<NpadControllerBase*>::iterator it = controllers.begin();
        it != controllers.end();
        ++it)
    {
        delete *it;
    }

    // スレッドが終了するのを待機する
    nn::os::WaitThread(&thread);
    // スレッドを破棄する
    nn::os::DestroyThread(&thread);
    // タイマーイベントをファイナライズする
    nn::os::FinalizeTimerEvent(&g_TimerEvent);

    // Clean up network download thread
    nn::os::WaitThread(&networkDownloadThread);
    nn::os::DestroyThread(&networkDownloadThread);

    /* this application no longer requires use of network interface */
    nn::nifm::CancelNetworkRequest();

    // Clean up audio threads
    nn::os::WaitThread(&g_AudioOutThread);
    nn::os::DestroyThread(&g_AudioOutThread);

    nn::os::WaitThread(&g_AudioInThread);
    nn::os::DestroyThread(&g_AudioInThread);

    FinalizeGraphics();

    // レンダリングを終了します。
    nn::audio::StopAudioRenderer(handle);
    nn::audio::CloseAudioRenderer(handle);
    nn::os::DestroySystemEvent(systemEvent.GetBase());

    for (int i = 0; i < BgmCount; ++i)
    {
        if (dataBgm[i])
        {
            g_WaveBufferAllocator.Free(dataBgm[i]);
            dataBgm[i] = nullptr;
        }
    }
    for (int i = 0; i < SeCount; ++i)
    {
        if (dataSe[i])
        {
            g_WaveBufferAllocator.Free(dataSe[i]);
            dataSe[i] = nullptr;
            g_WaveBufferAllocator.Free(header[i]);
            header[i] = nullptr;
        }
    }
    if (configBuffer)
    {
        g_Allocator.Free(configBuffer);
        configBuffer = nullptr;
    }
    if (workBuffer)
    {
        g_Allocator.Free(workBuffer);
        workBuffer = nullptr;
    }

    FinalizeFileSystem();
}  // NOLINT(readability/fn_size)


} //namespace iodevices
} //namespace nns

extern "C" void nnMain()
{
    nns::iodevices::mainLoop();
}
