﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstring>
#include <cstdlib>

#include <nn/os.h>
#include <nn/os/os_SystemEvent.h>
#include <nn/nn_Abort.h>
#include <nn/nn_SdkLog.h>
#include <nn/nn_Common.h>
#include <nn/nn_Log.h>
#include <nn/sm/sm_Result.h>

#include <nnt.h>

#include <nn/usb/usb_Host.h>

#include "../Fx3Methods/Fx3.h"

namespace nnt {
namespace usb {
namespace hs {

nn::usb::DeviceFilter g_Filter = {
    .matchFlags = nn::usb::DeviceFilterMatchFlags_Vendor | nn::usb::DeviceFilterMatchFlags_Product,
    .idVendor   = FX3_VID,
    .idProduct  = FX3_PID,
};

nn::usb::Host                 g_Client;
nn::usb::HostInterface        g_Interface;
nn::usb::HostEndpoint         g_EpIn;
nn::usb::HostEndpoint         g_EpOut;

int32_t g_IfCount;
nn::usb::InterfaceQueryOutput g_IfList[nn::usb::HsLimitMaxInterfacesCount];

NN_ALIGNAS(4096) uint8_t g_TxBuffer[nn::usb::HsLimitMaxUrbPerEndpointCount][FX3_MAX_TRANSFER_SIZE];
NN_ALIGNAS(4096) uint8_t g_RxBuffer[nn::usb::HsLimitMaxUrbPerEndpointCount][FX3_MAX_TRANSFER_SIZE];
NN_ALIGNAS(4096) uint8_t g_TxThreadStack[4096 * 4];
NN_ALIGNAS(4096) uint8_t g_RxThreadStack[4096 * 4];

NN_ALIGNAS(4096) uint8_t g_TxReport[4096];
NN_ALIGNAS(4096) uint8_t g_RxReport[4096];

void Fx3MakeGaloisPattern(uint8_t* pData, uint32_t size, uint32_t seed, uint32_t *newSeed)
{
    uint32_t processed;
    uint32_t lfsr = seed;

    for (processed = 0; processed < size; processed += 4)
    {
        uint32_t remaining = size - processed;
        /* taps: 32 31 29 1; characteristic polynomial: x^32 + x^31 + x^29 + x + 1 */
        lfsr = (lfsr >> 1) ^ (uint32_t)((0 - (lfsr & 1u)) & 0xd0000001u);
        /* we cannot always have 32-bit alignment in buffer, so it's best
           to memcpy */
        memcpy(pData + processed, &lfsr, (remaining >= 4) ? 4 : remaining);
    }

    *newSeed = lfsr;
}

bool Fx3CheckGaloisPattern(uint8_t* pData, uint32_t size, uint32_t seed, uint32_t *newSeed)
{
    uint32_t processed;
    uint32_t lfsr = seed;

    for (processed = 0; processed < size; processed += 4)
    {
        uint32_t remaining = size - processed;
        /* taps: 32 31 29 1; characteristic polynomial: x^32 + x^31 + x^29 + x + 1 */
        lfsr = (lfsr >> 1) ^ (uint32_t)((0 - (lfsr & 1u)) & 0xd0000001u);

        /* we cannot always have 32-bit alignment in buffer, so it's best
           to memcmp */
        if (memcmp(pData + processed, &lfsr, (remaining >= 4) ? 4 : remaining) != 0)
        {
            FX3_LOG("FAIL: Data varification error!\n");
            return false;
        }
    }

    *newSeed = lfsr;
    return true;
}

void Fx3InstallFirmware()
{
    nnt::usb::TestFx3Utility      g_Fx3Utility; // Install firmware

    FX3_LOG("Installing Fx3 Firmware...\n");
    Fx3FwErrorCode errorCode = Fx3FwErrorCode_Success;
    NNT_EXPECT_RESULT_SUCCESS(g_Fx3Utility.Initialize(WAIT_SECONDS_FOR_ATTACH));
    errorCode = g_Fx3Utility.UpdateFirmware(UsbTestSuiteFx3Firmware, sizeof(UsbTestSuiteFx3Firmware));
    g_Fx3Utility.Finalize();

    if (errorCode != Fx3FwErrorCode_Success)
    {
        FX3_LOG("Warning: failed to UpdateFirmware to Fx3. Error code: %d\n", errorCode);
    }
    //nn::os::SleepThread(nn::TimeSpan::FromSeconds(1));
}

// Enable FX3 interface Bulk-2, alt setting 1
void Fx3Setup()
{
    nn::os::SystemEventType       ifAvailableEvent;

    NN_USB_DMA_ALIGN Fx3DeviceMode mode = {
        FX3_DEVICE_SUPER_SPEED,
        FX3_CTRL_MPS_512,
        FX3_INTERFACE_BULK_2,
    };

    size_t bytesTransferred;

    Fx3InstallFirmware();

    NNT_ASSERT_RESULT_SUCCESS(g_Client.Initialize());

    NNT_ASSERT_RESULT_SUCCESS(
        g_Client.CreateInterfaceAvailableEvent(&ifAvailableEvent,
                                             nn::os::EventClearMode_ManualClear,
                                             0,
                                             &g_Filter)
    );

    // Attach
    NN_LOG("Looking for FX3...");
    if (nn::os::TimedWaitSystemEvent(&ifAvailableEvent, nn::TimeSpan::FromSeconds(WAIT_SECONDS_FOR_ATTACH)))
    {
        nn::os::ClearSystemEvent(&ifAvailableEvent);
        NN_LOG("Found!\n");
    }
    else
    {
        NN_LOG("Failed! (timeout = %ds)\n", WAIT_SECONDS_FOR_ATTACH);
        goto bail;
    }

    memset(g_IfList, 0, sizeof(g_IfList));
    NNT_ASSERT_RESULT_SUCCESS(
        g_Client.QueryAvailableInterfaces(&g_IfCount, g_IfList, sizeof(g_IfList), &g_Filter)
    );

    for (int i = 0; i < g_IfCount; i++)
    {
        nn::usb::InterfaceProfile& profile = g_IfList[i].ifProfile;

        if (profile.ifDesc.bInterfaceNumber == FX3_INTERFACE_NUMBER_CTRL)
        {
            NNT_ASSERT_RESULT_SUCCESS(
                g_Interface.Initialize(&g_Client, profile.handle)
            );
            break;
        }
    }
    ASSERT_TRUE(g_Interface.IsInitialized());

    // Enable Bulk-2
    NN_LOG("Enable FX3 interface BULK-2 (detach -> retach)\n");
    NNT_ASSERT_RESULT_SUCCESS(
        g_Interface.ControlRequest(
            &bytesTransferred,          // bytes transferred
            (uint8_t*)&mode,            // buffer
            0x40,                       // host to device | vendor | device
            FX3_REQUEST_DEVICE_MODE,    // bRequest
            0,                          // wValue
            0,                          // wIndex
            sizeof(Fx3DeviceMode)       // wLength
        )
    );

    NNT_ASSERT_RESULT_SUCCESS(g_Interface.Finalize());

    // Re-attach
    NN_LOG("Looking for FX3 (re-attach)...");
    if (nn::os::TimedWaitSystemEvent(&ifAvailableEvent, nn::TimeSpan::FromSeconds(WAIT_SECONDS_FOR_ATTACH)))
    {
        nn::os::ClearSystemEvent(&ifAvailableEvent);
        NN_LOG("Found!\n");
    }
    else
    {
        NN_LOG("Failed! (timeout = %ds)\n", WAIT_SECONDS_FOR_ATTACH);
        goto bail;
    }

    memset(g_IfList, 0, sizeof(g_IfList));
    NNT_ASSERT_RESULT_SUCCESS(
        g_Client.QueryAvailableInterfaces(&g_IfCount, g_IfList, sizeof(g_IfList), &g_Filter)
    );

    for (int i = 0; i < g_IfCount; i++)
    {
        nn::usb::InterfaceProfile& profile = g_IfList[i].ifProfile;

        if (profile.ifDesc.bInterfaceNumber == FX3_INTERFACE_NUMBER_BULK_2)
        {
            NNT_ASSERT_RESULT_SUCCESS(
                g_Interface.Initialize(&g_Client, profile.handle)
            );
            break;
        }
    }
    ASSERT_TRUE(g_Interface.IsInitialized());

    NNT_ASSERT_RESULT_SUCCESS(g_Interface.SetInterface(1));

    // Update the profile
    NNT_ASSERT_RESULT_SUCCESS(
        g_Client.QueryAcquiredInterfaces(&g_IfCount, g_IfList, sizeof(g_IfList))
    );

    NNT_ASSERT_RESULT_SUCCESS(g_Interface.Finalize());
    NNT_ASSERT_RESULT_SUCCESS(g_Client.DestroyInterfaceAvailableEvent(&ifAvailableEvent, 0));
    NNT_ASSERT_RESULT_SUCCESS(g_Client.Finalize());
    return;

bail:
    NNT_ASSERT_RESULT_SUCCESS(g_Client.DestroyInterfaceAvailableEvent(&ifAvailableEvent, 0));
    NNT_ASSERT_RESULT_SUCCESS(g_Client.Finalize());
    FAIL();
}


void Fx3TearDown()
{
    nn::os::SystemEventType       ifAvailableEvent;

    size_t bytesTransferred;

    NNT_ASSERT_RESULT_SUCCESS(g_Client.Initialize());

    NNT_ASSERT_RESULT_SUCCESS(
        g_Client.CreateInterfaceAvailableEvent(&ifAvailableEvent,
                                             nn::os::EventClearMode_ManualClear,
                                             0,
                                             &g_Filter)
    );

    // Attach
    NN_LOG("Looking for FX3 (to reset)...");
    if (nn::os::TimedWaitSystemEvent(&ifAvailableEvent, nn::TimeSpan::FromSeconds(WAIT_SECONDS_FOR_ATTACH)))
    {
        nn::os::ClearSystemEvent(&ifAvailableEvent);
        NN_LOG("Found!\n");
    }
    else
    {
        NN_LOG("Failed! (timeout = %ds)\n", WAIT_SECONDS_FOR_ATTACH);
        goto bail;
    }

    memset(g_IfList, 0, sizeof(g_IfList));
    NNT_ASSERT_RESULT_SUCCESS(
        g_Client.QueryAvailableInterfaces(&g_IfCount, g_IfList, sizeof(g_IfList), &g_Filter)
    );

    for (int i = 0; i < g_IfCount; i++)
    {
        nn::usb::InterfaceProfile& profile = g_IfList[i].ifProfile;

        if (profile.ifDesc.bInterfaceNumber == FX3_INTERFACE_NUMBER_CTRL)
        {
            NNT_ASSERT_RESULT_SUCCESS(
                g_Interface.Initialize(&g_Client, profile.handle)
            );
            break;
        }
    }
    ASSERT_TRUE(g_Interface.IsInitialized());

    FX3_LOG("Hard Reset FX3 to run boot loader..\n");

    NNT_ASSERT_RESULT_SUCCESS(
        g_Interface.ControlRequest(
            &bytesTransferred,          // bytes transferred
            NULL,                       // buffer
            0x40,                       // host to device | vendor | device
            FX3_REQUEST_RESET,          // bRequest
            0,                          // wValue
            0,                          // wIndex
            0                           // wLength
        )
    );

    for (;;)
    {
        NN_LOG("Waiting for all Fx3 interfaces to disconnect..\n");

        NNT_ASSERT_RESULT_SUCCESS(
            g_Client.QueryAllInterfaces(&g_IfCount, g_IfList, sizeof(g_IfList), &g_Filter)
            );

        NN_LOG("Found %d Fx3 interfaces\n", g_IfCount);
        if (g_IfCount == 0)
        {
            break;
        }
        nn::os::SleepThread(nn::TimeSpan::FromSeconds(1));
    }

    NNT_ASSERT_RESULT_SUCCESS(g_Interface.Finalize());
    NNT_ASSERT_RESULT_SUCCESS(g_Client.DestroyInterfaceAvailableEvent(&ifAvailableEvent, 0));
    NNT_ASSERT_RESULT_SUCCESS(g_Client.Finalize());
    nn::os::SleepThread(nn::TimeSpan::FromSeconds(1)); // wait for fx3 hard reset
    return;

bail:
    NNT_ASSERT_RESULT_SUCCESS(g_Client.DestroyInterfaceAvailableEvent(&ifAvailableEvent, 0));
    NNT_ASSERT_RESULT_SUCCESS(g_Client.Finalize());
    FAIL();
}

/**
 * Make sure we support HsLimitMaxClientCount clients, no more, no less
 */
TEST(HsBasic, MaxClientCount)
{
    // one for hid, one for eth, one for audio
    const int ClientCount = nn::usb::HsLimitMaxClientCount - 1 - 1 - 1;

    nn::usb::Host client[ClientCount + 1];

    for (int i = 0; i < ClientCount; i++)
    {
        NNT_ASSERT_RESULT_SUCCESS(client[i].Initialize());
    }

    // This one must fail
    NNT_ASSERT_RESULT_FAILURE(nn::sm::ResultMaxSessions,
                              client[ClientCount].Initialize());

    for (int i = 0; i < ClientCount; i++)
    {
        NNT_ASSERT_RESULT_SUCCESS(client[i].Finalize());
    }
}

/**
 * Unit test for DeviceFilter implementation
 */
TEST(HsBasic, DeviceFilter)
{
    nn::usb::InterfaceQueryOutput output;
    int32_t count = 0;

    struct {
        nn::usb::DeviceFilter filter;
        int32_t               count;
    } tests[] = {
        {
            {
                .matchFlags = nn::usb::DeviceFilterMatchFlags_Vendor,
                .idVendor   = FX3_VID,
            },
            1,
        }, {
            {
                .matchFlags = nn::usb::DeviceFilterMatchFlags_Vendor,
                .idVendor   = 0,
            },
            0,
        }, {
            {
                .matchFlags = nn::usb::DeviceFilterMatchFlags_Product,
                .idProduct  = FX3_PID,
            },
            1,
        }, {
            {
                .matchFlags = nn::usb::DeviceFilterMatchFlags_Product,
                .idProduct  = 0,
            },
            0,
        }, {
            {
                .matchFlags  = nn::usb::DeviceFilterMatchFlags_DeviceLo,
                .bcdDeviceLo = 0,
            },
            1,
        }, {
            {
                .matchFlags  = nn::usb::DeviceFilterMatchFlags_DeviceLo,
                .bcdDeviceLo = 1,
            },
            0,
        }, {
            {
                .matchFlags  = nn::usb::DeviceFilterMatchFlags_DeviceHi,
                .bcdDeviceHi = 0,
            },
            1,
        }, {
            {
                .matchFlags   = nn::usb::DeviceFilterMatchFlags_DeviceClass,
                .bDeviceClass = 0,
            },
            1,
        }, {
            {
                .matchFlags   = nn::usb::DeviceFilterMatchFlags_DeviceClass,
                .bDeviceClass = 1,
            },
            0,
        }, {
            {
                .matchFlags      = nn::usb::DeviceFilterMatchFlags_DeviceSubClass,
                .bDeviceSubClass = 0,
            },
            1,
        }, {
            {
                .matchFlags      = nn::usb::DeviceFilterMatchFlags_DeviceSubClass,
                .bDeviceSubClass = 1,
            },
            0,
        }, {
            {
                .matchFlags      = nn::usb::DeviceFilterMatchFlags_DeviceProtocol,
                .bDeviceProtocol = 0,
            },
            1,
        }, {
            {
                .matchFlags      = nn::usb::DeviceFilterMatchFlags_DeviceProtocol,
                .bDeviceProtocol = 1,
            },
            0,
        }, {
            {
                .matchFlags      = nn::usb::DeviceFilterMatchFlags_InterfaceClass,
                .bInterfaceClass = 0xff,
            },
            1,
        }, {
            {
                .matchFlags      = nn::usb::DeviceFilterMatchFlags_InterfaceClass,
                .bInterfaceClass = 0x00,
            },
            0,
        }, {
            {
                .matchFlags         = nn::usb::DeviceFilterMatchFlags_InterfaceSubClass,
                .bInterfaceSubClass = 0,
            },
            1,
        }, {
            {
                .matchFlags         = nn::usb::DeviceFilterMatchFlags_InterfaceSubClass,
                .bInterfaceSubClass = 1,
            },
            0,
        }, {
            {
                .matchFlags         = nn::usb::DeviceFilterMatchFlags_InterfaceProtocol,
                .bInterfaceProtocol = 0,
            },
            1,
        }, {
            {
                .matchFlags         = nn::usb::DeviceFilterMatchFlags_InterfaceProtocol,
                .bInterfaceProtocol = 1,
            },
            0,
        },
    };

    ASSERT_NO_FATAL_FAILURE(Fx3Setup());
    NNT_ASSERT_RESULT_SUCCESS(g_Client.Initialize());

    NN_LOG("MatchFlag\tExpected\tActual\n");
    for (auto& test : tests)
    {
        NNT_ASSERT_RESULT_SUCCESS(
            g_Client.QueryAllInterfaces(&count,
                                        &output,
                                        sizeof(output),
                                        &test.filter
            )
        );
        NN_LOG("%04x\t\t%d\t\t\t%d\n", test.filter.matchFlags, test.count, count);
        ASSERT_EQ(count, test.count);
    }


    NNT_ASSERT_RESULT_SUCCESS(g_Client.Finalize());
    ASSERT_NO_FATAL_FAILURE(Fx3TearDown());
} // NOLINT(impl/function_size)

/**
 * Make sure Initialize() / Finalize() doesn't leak handles
 */
TEST(HsBasic, ClientInitAndFinalize)
{
    // This is specified in desc and nect file
    const int HandleTableSize = 512;

    nn::usb::Host                 client;
    nn::usb::HostInterface        interface;
    nn::usb::HostEndpoint         endpoint;

    ASSERT_NO_FATAL_FAILURE(Fx3Setup());

    for (int i = 0; i < HandleTableSize; i++)
    {
        NNT_ASSERT_RESULT_SUCCESS(client.Initialize());

        for (int j = 0; j < g_IfCount; j++)
        {
            nn::usb::InterfaceProfile& profile = g_IfList[j].ifProfile;

            if (profile.ifDesc.bInterfaceNumber != FX3_INTERFACE_NUMBER_BULK_2)
            {
                continue;
            }

            NNT_ASSERT_RESULT_SUCCESS(
                interface.Initialize(&client, profile.handle)
            );

            for (int k = 0; k < nn::usb::UsbLimitMaxEndpointPairCount; k++)
            {
                nn::usb::UsbEndpointDescriptor& epDesc = profile.epInDesc[k];

                if (epDesc.bEndpointAddress)
                {
                    NNT_ASSERT_RESULT_SUCCESS(
                        endpoint.Initialize(&interface, &epDesc, 4, 1024 * 32)
                    );
                    break;
                }
            }
            ASSERT_TRUE(endpoint.IsInitialized());

            NNT_ASSERT_RESULT_SUCCESS(
                endpoint.CreateSmmuSpace(g_TxBuffer[0], sizeof(g_TxBuffer[0]))
            );
            NNT_ASSERT_RESULT_SUCCESS(
                endpoint.ShareReportRing(g_TxReport, sizeof(g_TxReport))
            );

            break;
        }
        ASSERT_TRUE(interface.IsInitialized());

        NNT_ASSERT_RESULT_SUCCESS(endpoint.Finalize());
        NNT_ASSERT_RESULT_SUCCESS(interface.Finalize());
        NNT_ASSERT_RESULT_SUCCESS(client.Finalize());

        // Don't drive CPU load to 100% continuously, so to avoid stressing unrelated system processes
        if(i % 5 == 0)
        {
            nn::os::SleepThread( nn::TimeSpan::FromMilliSeconds(10) );
        }
    }

    Fx3TearDown();
}

/**
 * Make sure DMA buffer alignment requirement is enforced
 */
TEST(HsBasic, BufferAlignment)
{
    size_t bytesTransferred;
    nn::usb::InterfaceProfile& profile = g_IfList[0].ifProfile;

    NN_USB_DMA_ALIGN char buffer[64];

    //nn::os::SleepThread(nn::TimeSpan::FromSeconds(5));
    ASSERT_NO_FATAL_FAILURE(Fx3Setup());

    NNT_ASSERT_RESULT_SUCCESS(g_Client.Initialize());
    NNT_ASSERT_RESULT_SUCCESS(g_Interface.Initialize(&g_Client, profile.handle));
    NNT_ASSERT_RESULT_SUCCESS(
        g_EpIn.Initialize(&g_Interface, &profile.epInDesc[2], 4, sizeof(buffer))
    );

    NNT_ASSERT_RESULT_FAILURE(
        nn::usb::ResultAlignmentError,
        g_EpIn.PostBuffer(&bytesTransferred, buffer + 1, sizeof(buffer))
    );

    NNT_ASSERT_RESULT_SUCCESS(g_EpIn.Finalize());
    NNT_ASSERT_RESULT_SUCCESS(g_Interface.Finalize());
    NNT_ASSERT_RESULT_SUCCESS(g_Client.Finalize());

    Fx3TearDown();
}

/**
 * Verify basic behavior of static sMMU map
 */
TEST(HsBasic, StaticSmmuMap)
{
    nn::usb::InterfaceProfile& profile = g_IfList[0].ifProfile;
    ASSERT_NO_FATAL_FAILURE(Fx3Setup());

    NNT_ASSERT_RESULT_SUCCESS(g_Client.Initialize());
    NNT_ASSERT_RESULT_SUCCESS(g_Interface.Initialize(&g_Client, profile.handle));
    NNT_ASSERT_RESULT_SUCCESS(
        g_EpIn.Initialize(&g_Interface, &profile.epInDesc[2], 1, 0)
    );

    // buffer alignment
    NNT_ASSERT_RESULT_FAILURE(
        nn::usb::ResultAlignmentError,
        g_EpIn.CreateSmmuSpace(g_RxBuffer[0] + 1, sizeof(g_RxBuffer[0]))
    );
    NNT_ASSERT_RESULT_FAILURE(
        nn::usb::ResultAlignmentError,
        g_EpIn.CreateSmmuSpace(g_RxBuffer[0], sizeof(g_RxBuffer[0]) - 1)
    );

    // this should succeed
    NNT_ASSERT_RESULT_SUCCESS(
        g_EpIn.CreateSmmuSpace(g_RxBuffer[0], sizeof(g_RxBuffer[0]))
    );

    // duplicated call
    NNT_ASSERT_RESULT_FAILURE(
        nn::usb::ResultOperationDenied,
        g_EpIn.CreateSmmuSpace(g_RxBuffer[0], sizeof(g_RxBuffer[0]))
    );

    NNT_ASSERT_RESULT_SUCCESS(g_EpIn.Finalize());
    NNT_ASSERT_RESULT_SUCCESS(g_Interface.Finalize());
    NNT_ASSERT_RESULT_SUCCESS(g_Client.Finalize());

    Fx3TearDown();
}

/**
 * Verify basic behavior of shared report ring
 */
TEST(HsBasic, SharedReportRing)
{
    nn::usb::InterfaceProfile& profile = g_IfList[0].ifProfile;
    ASSERT_NO_FATAL_FAILURE(Fx3Setup());

    NNT_ASSERT_RESULT_SUCCESS(g_Client.Initialize());
    NNT_ASSERT_RESULT_SUCCESS(g_Interface.Initialize(&g_Client, profile.handle));
    NNT_ASSERT_RESULT_SUCCESS(
        g_EpIn.Initialize(&g_Interface, &profile.epInDesc[2], 1, 0)
    );

    // buffer alignment
    NNT_ASSERT_RESULT_FAILURE(
        nn::usb::ResultAlignmentError,
        g_EpIn.ShareReportRing(g_RxReport + 1, sizeof(g_RxReport))
    );
    NNT_ASSERT_RESULT_FAILURE(
        nn::usb::ResultAlignmentError,
        g_EpIn.ShareReportRing(g_RxReport, 4096 - 1)
    );

    // this should succeed
    NNT_ASSERT_RESULT_SUCCESS(
        g_EpIn.ShareReportRing(g_RxReport, sizeof(g_RxReport))
    );

    // duplicated call
    NNT_ASSERT_RESULT_FAILURE(
        nn::usb::ResultOperationDenied,
        g_EpIn.ShareReportRing(g_RxReport, sizeof(g_RxReport))
    );

    NNT_ASSERT_RESULT_SUCCESS(g_EpIn.Finalize());
    NNT_ASSERT_RESULT_SUCCESS(g_Interface.Finalize());
    NNT_ASSERT_RESULT_SUCCESS(g_Client.Finalize());

    Fx3TearDown();
}

/**
 * Check the behavior when report ring is overrun
 *   - iteration 0: no shared report ring
 *   - iteration 1: shared report ring
 */
TEST(HsBasic, ReportRingOverrun)
{
    const uint32_t MaxXferCount = nn::usb::HsLimitMaxXferPerUrbCount;
    const uint32_t XferSize     = 1024;
    const uint32_t RetriveCount = MaxXferCount / 2;

    NN_STATIC_ASSERT(MaxXferCount > RetriveCount);
    NN_STATIC_ASSERT(XferSize * MaxXferCount <= FX3_MAX_TRANSFER_SIZE);

    nn::os::SystemEventType *inEvent;
    nn::os::SystemEventType *outEvent;
    nn::usb::InterfaceProfile& profile = g_IfList[0].ifProfile;

    uint32_t sizeArray[MaxXferCount];

    NN_USB_DMA_ALIGN Fx3TestDataRarameters dataParams = {
        .endpointAddress = FX3_ENDPOINT_BULK_2_OUT,
        .dataSeed        = 0,  // don't care for bulk loopback
        .entries         = MaxXferCount,
    };

    for (uint32_t i = 0; i < MaxXferCount; i++)
    {
        dataParams.params[i] = {XferSize, XferSize};
        sizeArray[i] = XferSize;
    }

    ASSERT_NO_FATAL_FAILURE(Fx3Setup());

    NNT_ASSERT_RESULT_SUCCESS(g_Client.Initialize());
    NNT_ASSERT_RESULT_SUCCESS(g_Interface.Initialize(&g_Client, profile.handle));
    NNT_ASSERT_RESULT_SUCCESS(
        g_EpIn.Initialize(&g_Interface, &profile.epInDesc[2], 1, 0)
    );
    NNT_ASSERT_RESULT_SUCCESS(
        g_EpOut.Initialize(&g_Interface, &profile.epOutDesc[2], 1, 0)
    );

    inEvent  = g_EpIn.GetCompletionEvent();
    outEvent = g_EpOut.GetCompletionEvent();

    for (int iter = 0; iter < 2; iter++)
    {
        if (iter == 0)
        {
            // Get Xfer Report with SF call
        }
        else
        {
            NN_LOG("Enable shared report ring\n");

            // Get Xfer Report with shared ring, bypass SF call
            NNT_ASSERT_RESULT_SUCCESS(
                g_EpIn.ShareReportRing(g_RxReport, sizeof(g_RxReport))
            );
            NNT_ASSERT_RESULT_SUCCESS(
                g_EpOut.ShareReportRing(g_TxReport, sizeof(g_TxReport))
            );
        }

        for (uint32_t round = 0; round < 5; round++)
        {
            size_t   bytesTransferred;
            uint32_t xferId;
            uint32_t reportCount = 0;
            nn::usb::XferReport report[MaxXferCount];

            NNT_ASSERT_RESULT_SUCCESS(
                g_Interface.ControlRequest(
                    &bytesTransferred,              // bytes transferred
                    (uint8_t*)&dataParams,          // buffer
                    0x40,                           // device to host | vendor | device
                    FX3_REQUEST_TRANSFER_DATA,      // bRequest
                    0,                              // wValue
                    0,                              // wIndex
                    sizeof(Fx3TestDataRarameters)   // wLength
                )
            );

            NNT_ASSERT_RESULT_SUCCESS(
                g_EpIn.BatchBufferAsync(
                    &xferId,
                    g_RxBuffer[0],
                    sizeArray,
                    MaxXferCount,
                    round,
                    nn::usb::SchedulePolicy_Absolute,
                    0
                )
            );
            NNT_ASSERT_RESULT_SUCCESS(
                g_EpOut.BatchBufferAsync(
                    &xferId,
                    g_TxBuffer[0],
                    sizeArray,
                    MaxXferCount,
                    round,
                    nn::usb::SchedulePolicy_Absolute,
                    0
                )
            );

            // Always clear EpIn ReportRing
            do {
                uint32_t count = 0;

                nn::os::WaitSystemEvent(inEvent);
                nn::os::ClearSystemEvent(inEvent);

                NNT_ASSERT_RESULT_SUCCESS(
                    g_EpIn.GetXferReport(&count, report, MaxXferCount)
                );

                for (uint32_t i = 0; i < count; i++)
                {
                    ASSERT_EQ(round, report[i].context);
                }

                reportCount += count;
            } while (reportCount != MaxXferCount);

            // EpOut is guaranteed to have all completed
            nn::os::WaitSystemEvent(outEvent);
            nn::os::ClearSystemEvent(outEvent);

            // Check EpOut ReportRing
            if (round == 0)
            {
                // Don't retrive the reports so Round 1 will overrun
            }
            else if (round == 1)
            {
                NNT_ASSERT_RESULT_SUCCESS(
                    g_EpOut.GetXferReport(&reportCount, report, MaxXferCount)
                );

                ASSERT_EQ(MaxXferCount, reportCount);

                // Those reports must be leftovers from last round
                for (uint32_t i = 0; i < reportCount; i++)
                {
                    ASSERT_EQ(round - 1, report[i].context);
                }

                // No more reports - round 1 reports are lost
                NNT_ASSERT_RESULT_SUCCESS(
                    g_EpOut.GetXferReport(&reportCount, report, MaxXferCount)
                );
                ASSERT_EQ(0, reportCount);
            }
            else if (round == 2)
            {
                // Retrive only part of the reports
                NNT_ASSERT_RESULT_SUCCESS(
                    g_EpOut.GetXferReport(&reportCount, report, RetriveCount)
                );

                ASSERT_EQ(RetriveCount, reportCount);

                // Those reports must be for current round
                for (uint32_t i = 0; i < reportCount; i++)
                {
                    ASSERT_EQ(round, report[i].context);
                }
            }
            else if (round == 3)
            {
                NNT_ASSERT_RESULT_SUCCESS(
                    g_EpOut.GetXferReport(&reportCount, report, MaxXferCount)
                );

                ASSERT_EQ(MaxXferCount, reportCount);

                // Those are leftovers from last round
                for (uint32_t i = 0; i < MaxXferCount - RetriveCount; i++)
                {
                    ASSERT_EQ(round - 1, report[i].context);
                }

                // Those are from current round
                for (uint32_t i = MaxXferCount - RetriveCount; i < MaxXferCount; i++)
                {
                    ASSERT_EQ(round, report[i].context);
                }
            }
            else if (round == 4)
            {
                NNT_ASSERT_RESULT_SUCCESS(
                    g_EpOut.GetXferReport(&reportCount, report, MaxXferCount)
                );

                ASSERT_EQ(MaxXferCount, reportCount);

                // Those are from current round
                for (uint32_t i = 0; i < MaxXferCount; i++)
                {
                    ASSERT_EQ(round, report[i].context);
                }
            }
            else
            {
                FAIL(); // should never reach here
            }
        }
    }

    NNT_ASSERT_RESULT_SUCCESS(g_EpOut.Finalize());
    NNT_ASSERT_RESULT_SUCCESS(g_EpIn.Finalize());
    NNT_ASSERT_RESULT_SUCCESS(g_Interface.Finalize());
    NNT_ASSERT_RESULT_SUCCESS(g_Client.Finalize());

    Fx3TearDown();
} // NOLINT(impl/function_size)

/**
 * Make sure premature completion (i.e. failure before entering TRB ring) is
 * handled properly. The reports must be in order.
 *   - iteration 0: no shared report ring
 *   - iteration 1: shared report ring
 */
TEST(HsBasic, PrematureCompletion)
{
    // test parameters
    const uint32_t MaxXferCount = 8;
    const uint32_t XferSize     = 1024 * 100;  // 2 TRB per xfer

    NN_STATIC_ASSERT(MaxXferCount <= 32);  // uint32_t pattern
    NN_STATIC_ASSERT(MaxXferCount <= nn::usb::HsLimitMaxUrbPerEndpointCount);
    NN_STATIC_ASSERT(XferSize <= FX3_MAX_TRANSFER_SIZE);
    NN_STATIC_ASSERT(MaxXferCount * XferSize <= nn::usb::HsLimitMaxUrbTransferSize);

    uint32_t reportCount;
    nn::usb::XferReport report[MaxXferCount];

    NN_USB_DMA_ALIGN Fx3TestDataRarameters dataParams = {
        .endpointAddress = FX3_ENDPOINT_BULK_2_OUT,
        .dataSeed        = 0,  // don't care for bulk loopback
    };

    nn::usb::InterfaceProfile& profile = g_IfList[0].ifProfile;

    ASSERT_NO_FATAL_FAILURE(Fx3Setup());

    NNT_ASSERT_RESULT_SUCCESS(g_Client.Initialize());
    NNT_ASSERT_RESULT_SUCCESS(g_Interface.Initialize(&g_Client, profile.handle));
    NNT_ASSERT_RESULT_SUCCESS(
        g_EpIn.Initialize(&g_Interface, &profile.epInDesc[2], MaxXferCount, XferSize)
    );
    NNT_ASSERT_RESULT_SUCCESS(
        g_EpOut.Initialize(&g_Interface, &profile.epOutDesc[2], MaxXferCount, XferSize)
    );

    for (int iter = 0; iter < 2; iter++)
    {
        if (iter == 0)
        {
            // Get Xfer Report with SF call
        }
        else
        {
            NN_LOG("Enable shared report ring\n");

            // Get Xfer Report with shared ring, bypass SF call
            NNT_ASSERT_RESULT_SUCCESS(
                g_EpIn.ShareReportRing(g_RxReport, sizeof(g_RxReport))
            );
            NNT_ASSERT_RESULT_SUCCESS(
                g_EpOut.ShareReportRing(g_TxReport, sizeof(g_TxReport))
            );
        }

        // bit pattern: 1 = good, 0 = bad
        for (uint32_t pattern = 0; pattern < (1 << MaxXferCount); pattern++)
        {
            uint32_t xferId;
            uint32_t xferCount = 0;
            uint32_t submitted = 0;
            size_t bytesTransferred;

            for (int i = 0; i < MaxXferCount; i++)
            {
                if (pattern & (1 << i))
                {
                    xferCount++;
                }
            }

            if (xferCount > 0)
            {
                dataParams.entries = xferCount;
                for (int i = 0; i < xferCount; i++)
                {
                    dataParams.params[i] = {XferSize, XferSize};
                }
            }

            NNT_ASSERT_RESULT_SUCCESS(
                g_Interface.ControlRequest(
                    &bytesTransferred,              // bytes transferred
                    (uint8_t*)&dataParams,          // buffer
                    0x40,                           // device to host | vendor | device
                    FX3_REQUEST_TRANSFER_DATA,      // bRequest
                    0,                              // wValue
                    0,                              // wIndex
                    sizeof(Fx3TestDataRarameters)   // wLength
                )
            );

            ASSERT_EQ(bytesTransferred, sizeof(Fx3TestDataRarameters));

            // post in xfer
            NN_LOG("IN : ");
            for (int i = 0; i < MaxXferCount; i++)
            {
                if (pattern & (1 << i))
                {
                    NN_LOG("Good -> ");
                    NNT_ASSERT_RESULT_SUCCESS(
                        g_EpIn.PostBufferAsync(&xferId, g_RxBuffer[submitted++], XferSize, i)
                    );
                }
                else
                {
                    NN_LOG("Bad -> ");
                    NNT_ASSERT_RESULT_SUCCESS(
                        g_EpIn.PostBufferAsync(&xferId, g_RxBuffer[0] + 1, XferSize, i)
                    );
                }
            }
            NN_LOG("Done\n");

            // post out xfer
            for (int i = 0; i < xferCount; i++)
            {
                g_EpOut.PostBufferAsync(&xferId, g_TxBuffer[i], XferSize, i);
            }

            // wait for 200ms for the xfer completion, should be more than enough
            nn::os::SleepThread( nn::TimeSpan::FromMilliSeconds(200) );

            // now get and check the IN reports
            NNT_ASSERT_RESULT_SUCCESS(
                g_EpIn.GetXferReport(&reportCount, report, MaxXferCount)
            );

            ASSERT_EQ(reportCount, MaxXferCount);

            for (int i = 0; i < reportCount; i++)
            {
                // make sure the reports are in order
                ASSERT_EQ(report[i].context, i);

                if (pattern & (1 << i))
                {
                    NNT_ASSERT_RESULT_SUCCESS(report[i].result);
                }
                else
                {
                    NNT_ASSERT_RESULT_FAILURE(nn::usb::ResultAlignmentError, report[i].result);
                }
            }

            // now get and check the OUT reports
            NNT_ASSERT_RESULT_SUCCESS(
                g_EpOut.GetXferReport(&reportCount, report, MaxXferCount)
            );

            ASSERT_EQ(reportCount, xferCount);

            for (int i = 0; i < reportCount; i++)
            {
                // make sure the reports are in order
                ASSERT_EQ(report[i].context, i);

                NNT_ASSERT_RESULT_SUCCESS(report[i].result);
            }
        }
    }

    NNT_ASSERT_RESULT_SUCCESS(g_EpOut.Finalize());
    NNT_ASSERT_RESULT_SUCCESS(g_EpIn.Finalize());
    NNT_ASSERT_RESULT_SUCCESS(g_Interface.Finalize());
    NNT_ASSERT_RESULT_SUCCESS(g_Client.Finalize());

    Fx3TearDown();
} // NOLINT(impl/function_size)

/**
 * Single thread loopback test
 */
TEST(HsBasic, BulkPipeline)
{
    nn::os::Tick tick0;
    nn::os::Tick tick1;
    nn::os::Tick ticks;

    uint32_t outSeed = 0xdeadbeef;
    uint32_t inSeed  = outSeed;

    NN_USB_DMA_ALIGN Fx3TestDataRarameters dataParams = {
        .endpointAddress = FX3_ENDPOINT_BULK_2_OUT,
        .dataSeed        = 0,  // don't care for bulk loopback
    };

    struct {
        uint32_t pipelineDepth;
        uint32_t xferSize;
        uint8_t  xferCount;
        bool     patternCheck;
    } params[] = {
        // with pattern check
        {
            1,
            FX3_MAX_TRANSFER_SIZE,
            FX3_TRANSFER_MAX_ENTRIES,
            true,
        },
        {
            2,
            FX3_MAX_TRANSFER_SIZE,
            FX3_TRANSFER_MAX_ENTRIES,
            true,
        },
        {
            4,
            FX3_MAX_TRANSFER_SIZE,
            FX3_TRANSFER_MAX_ENTRIES,
            true,
        },
        {
            7,
            FX3_MAX_TRANSFER_SIZE,
            FX3_TRANSFER_MAX_ENTRIES,
            true,
        },

        // without pattern check
        {
            1,
            FX3_MAX_TRANSFER_SIZE,
            FX3_TRANSFER_MAX_ENTRIES,
            false,
        },
        {
            2,
            FX3_MAX_TRANSFER_SIZE,
            FX3_TRANSFER_MAX_ENTRIES,
            false,
        },
        {
            4,
            FX3_MAX_TRANSFER_SIZE,
            FX3_TRANSFER_MAX_ENTRIES,
            false,
        },
        {
            7,
            FX3_MAX_TRANSFER_SIZE,
            FX3_TRANSFER_MAX_ENTRIES,
            false,
        },

        // 8 * FX3_MAX_TRANSFER_SIZE > HsLimitMaxUrbTransferSize
    };

    ASSERT_NO_FATAL_FAILURE(Fx3Setup());

    nn::usb::InterfaceProfile& profile = g_IfList[0].ifProfile;

    NNT_ASSERT_RESULT_SUCCESS(g_Client.Initialize());
    NNT_ASSERT_RESULT_SUCCESS(g_Interface.Initialize(&g_Client, profile.handle));

    for (auto& param : params)
    {
        uint32_t xferId;
        size_t   bytesTransferred;
        uint32_t inSubmittedCount  = 0;
        uint32_t inFinishedCount   = 0;
        uint32_t outSubmittedCount = 0;
        uint32_t outFinishedCount  = 0;
        nn::os::SystemEventType *inEvent;
        nn::os::SystemEventType *outEvent;

        NN_LOG("Pipeline %d 0x%x %d\n", param.pipelineDepth, param.xferSize, param.xferCount);

        NNT_ASSERT_RESULT_SUCCESS(
            g_EpIn.Initialize(&g_Interface, &profile.epInDesc[2], param.pipelineDepth, param.xferSize)
        );
        NNT_ASSERT_RESULT_SUCCESS(
            g_EpOut.Initialize(&g_Interface, &profile.epOutDesc[2], param.pipelineDepth, param.xferSize)
        );

        inEvent  = g_EpIn.GetCompletionEvent();
        outEvent = g_EpOut.GetCompletionEvent();

        // prepare
        dataParams.entries = param.xferCount;
        for (int i = 0; i < param.xferCount; i++)
        {
            dataParams.params[i] = {param.xferSize, param.xferSize};
        }

        NNT_ASSERT_RESULT_SUCCESS(
            g_Interface.ControlRequest(
                &bytesTransferred,              // bytes transferred
                (uint8_t*)&dataParams,          // buffer
                0x40,                           // device to host | vendor | device
                FX3_REQUEST_TRANSFER_DATA,      // bRequest
                0,                              // wValue
                0,                              // wIndex
                sizeof(Fx3TestDataRarameters)   // wLength
            )
        );

        ASSERT_EQ(bytesTransferred, sizeof(Fx3TestDataRarameters));

        // begin
        tick0 = nn::os::GetSystemTick();

        for (int i = 0; i < param.pipelineDepth; i++)
        {
            if (outSubmittedCount < param.xferCount)
            {
                if (param.patternCheck)
                {
                    Fx3MakeGaloisPattern(g_TxBuffer[i], param.xferSize, outSeed, &outSeed);
                }

                NNT_ASSERT_RESULT_SUCCESS(
                    g_EpOut.PostBufferAsync(&xferId,
                                          g_TxBuffer[i],
                                          param.xferSize,
                                          (uint64_t)g_TxBuffer[i])
                );

                outSubmittedCount++;
            }

            if (inSubmittedCount < param.xferCount)
            {
                NNT_ASSERT_RESULT_SUCCESS(
                    g_EpIn.PostBufferAsync(&xferId,
                                         g_RxBuffer[i],
                                         param.xferSize,
                                         (uint64_t)g_RxBuffer[i])
                );

                inSubmittedCount++;
            }
        }

        bool inDone  = false;
        bool outDone = false;
        while (!inDone || !outDone)
        {
            uint32_t reportCount;
            nn::usb::XferReport report[nn::usb::HsLimitMaxUrbPerEndpointCount];

            switch (nn::os::WaitAny(outEvent, inEvent))
            {
            case 0:  // out
                nn::os::ClearSystemEvent(outEvent);

                NNT_ASSERT_RESULT_SUCCESS(
                    g_EpOut.GetXferReport(&reportCount, report, param.pipelineDepth)
                );

                for (int i = 0; i < reportCount; i++)
                {
                    void *buffer = (void*)report[i].context;

                    NNT_ASSERT_RESULT_SUCCESS(report[i].result);
                    ASSERT_EQ(report[i].transferredSize, param.xferSize);

                    if (++outFinishedCount == param.xferCount)
                    {
                        ASSERT_EQ(i + 1, reportCount);
                        outDone = true;
                        break;
                    }

                    if (outSubmittedCount < param.xferCount)
                    {
                        if (param.patternCheck)
                        {
                            Fx3MakeGaloisPattern((uint8_t*)buffer, param.xferSize, outSeed, &outSeed);
                        }

                        NNT_ASSERT_RESULT_SUCCESS(
                            g_EpOut.PostBufferAsync(&xferId,
                                                  buffer,
                                                  param.xferSize,
                                                  (uint64_t)buffer)
                        );

                        outSubmittedCount++;
                    }
                }
                break;

            case 1:  // in
                nn::os::ClearSystemEvent(inEvent);

                NNT_ASSERT_RESULT_SUCCESS(
                    g_EpIn.GetXferReport(&reportCount, report, param.pipelineDepth)
                );

                for (int i = 0; i < reportCount; i++)
                {
                    void *buffer = (void*)report[i].context;

                    NNT_ASSERT_RESULT_SUCCESS(report[i].result);
                    ASSERT_EQ(report[i].transferredSize, param.xferSize);

                    if (param.patternCheck)
                    {
                        ASSERT_TRUE(
                            Fx3CheckGaloisPattern((uint8_t*)buffer, param.xferSize, inSeed, &inSeed)
                        );
                    }

                    if (++inFinishedCount == param.xferCount)
                    {
                        ASSERT_EQ(i + 1, reportCount);
                        inDone = true;
                        break;
                    }

                    if (inSubmittedCount < param.xferCount)
                    {
                        NNT_ASSERT_RESULT_SUCCESS(
                            g_EpIn.PostBufferAsync(&xferId,
                                                 buffer,
                                                 param.xferSize,
                                                 (uint64_t)buffer)
                        );

                        inSubmittedCount++;
                    }
                }
                break;

            default:
                NN_UNEXPECTED_DEFAULT;
                break;
            }
        }

        // end
        tick1 = nn::os::GetSystemTick();

        // statistic
        ticks = tick1 - tick0;
        uint64_t us = ticks.ToTimeSpan().GetMicroSeconds();
        uint32_t totalBytes = param.xferSize * param.xferCount * 2;

        FX3_LOG(
            "Elapsed time: %dus %f MB/s for 0x%x bytes loop back%s.\n",
            us,
            (float)(totalBytes) / ((float)us / (float)1000000) / (1024 * 1024),
            totalBytes,
            param.patternCheck ? " with pattern check" : ""
        );

        NN_LOG(
            "##teamcity[buildStatisticValue key='SS_Pipeline_D%d%s_MB/s' value='%f']\n",
            param.pipelineDepth,
            param.patternCheck ? "_PatternCheck" : "",
            (float)totalBytes / ((float)us / (float)1000000) / (1024 * 1024)
        );

        NNT_ASSERT_RESULT_SUCCESS(g_EpOut.Finalize());
        NNT_ASSERT_RESULT_SUCCESS(g_EpIn.Finalize());
    }

    NNT_ASSERT_RESULT_SUCCESS(g_Interface.Finalize());
    NNT_ASSERT_RESULT_SUCCESS(g_Client.Finalize());

    Fx3TearDown();
} // NOLINT(impl/function_size)

/**
 * Batch API with bulk xfer
 */
TEST(HsBasic, BulkBatch)
{
    struct {
        uint32_t       xferCount;
        const uint32_t sizeArray[8];
    } params[] = {
        {1, { 96 }},
        {1, { 94 }},
        {2, { 95, 96 }},
        {3, { 96, 97, 98 }},
        {4, { 96, 97, 98, 99 }},
        {8, { 96, 96, 96, 96 ,96, 96, 96, 96 }},
    };

    uint32_t outSeed = 0xdeadbeef;
    uint32_t inSeed  = outSeed;

    NN_USB_DMA_ALIGN Fx3TestDataRarameters dataParams = {
        .endpointAddress = FX3_ENDPOINT_BULK_2_OUT,
        .dataSeed        = 0,  // don't care for bulk loopback
    };

    ASSERT_NO_FATAL_FAILURE(Fx3Setup());

    nn::os::SystemEventType *inEvent;
    nn::os::SystemEventType *outEvent;
    nn::usb::InterfaceProfile& profile = g_IfList[0].ifProfile;

    uint32_t xferId;
    size_t   bytesTransferred;
    uint32_t reportCount;
    nn::usb::XferReport report[nn::usb::HsLimitMaxUrbPerEndpointCount];

    NNT_ASSERT_RESULT_SUCCESS(g_Client.Initialize());
    NNT_ASSERT_RESULT_SUCCESS(g_Interface.Initialize(&g_Client, profile.handle));

    NNT_ASSERT_RESULT_SUCCESS(
        g_EpIn.Initialize(
            &g_Interface, &profile.epInDesc[2],
            nn::usb::HsLimitMaxUrbPerEndpointCount, FX3_MAX_TRANSFER_SIZE
        )
    );
    NNT_ASSERT_RESULT_SUCCESS(
        g_EpOut.Initialize(
            &g_Interface, &profile.epOutDesc[2],
            nn::usb::HsLimitMaxUrbPerEndpointCount, FX3_MAX_TRANSFER_SIZE
        )
    );

    inEvent  = g_EpIn.GetCompletionEvent();
    outEvent = g_EpOut.GetCompletionEvent();

    // check the limit enforcement
    const uint32_t dummySizeArray[nn::usb::HsLimitMaxXferPerUrbCount + 1] = {0};
    NNT_ASSERT_RESULT_FAILURE(
        nn::usb::ResultInvalidParameter,
        g_EpIn.BatchBufferAsync(&xferId,
                              g_RxBuffer[0],
                              dummySizeArray,
                              nn::usb::HsLimitMaxXferPerUrbCount + 1,  // size error
                              0,
                              nn::usb::SchedulePolicy_Absolute,
                              0)
    );

    for (auto& param : params)
    {
        uint32_t totalSize = 0;

        // prepare
        NN_LOG("=== %d:", param.xferCount);
        dataParams.entries = param.xferCount;
        for (uint32_t i = 0; i < param.xferCount; i++)
        {
            NN_LOG(" %d", param.sizeArray[i]);
            dataParams.params[i] = {param.sizeArray[i], param.sizeArray[i]};
            totalSize += param.sizeArray[i];
        }
        NN_LOG("\n");

        NNT_ASSERT_RESULT_SUCCESS(
            g_Interface.ControlRequest(
                &bytesTransferred,              // bytes transferred
                (uint8_t*)&dataParams,          // buffer
                0x40,                           // device to host | vendor | device
                FX3_REQUEST_TRANSFER_DATA,      // bRequest
                0,                              // wValue
                0,                              // wIndex
                sizeof(Fx3TestDataRarameters)   // wLength
            )
        );

        ASSERT_EQ(bytesTransferred, sizeof(Fx3TestDataRarameters));

        // begin
        Fx3MakeGaloisPattern(g_TxBuffer[0], totalSize, outSeed, &outSeed);

        NNT_ASSERT_RESULT_SUCCESS(
            g_EpIn.BatchBufferAsync(&xferId,
                                  g_RxBuffer[0],
                                  param.sizeArray,
                                  param.xferCount,
                                  0,
                                  nn::usb::SchedulePolicy_Absolute,
                                  0)
        );

        NNT_ASSERT_RESULT_SUCCESS(
            g_EpOut.BatchBufferAsync(&xferId,
                                   g_TxBuffer[0],
                                   param.sizeArray,
                                   param.xferCount,
                                   0,
                                   nn::usb::SchedulePolicy_Absolute,
                                   0)
        );

        for (int i = 0; i < 2; i++)
        {
            switch (nn::os::WaitAny(outEvent, inEvent))
            {
            case 0: // out
                nn::os::ClearSystemEvent(outEvent);

                NNT_ASSERT_RESULT_SUCCESS(
                    g_EpOut.GetXferReport(&reportCount, report, param.xferCount)
                );

                ASSERT_EQ(reportCount, param.xferCount);

                for (int i = 0; i < reportCount; i++)
                {
                    NNT_ASSERT_RESULT_SUCCESS(report[i].result);
                    ASSERT_EQ(report[i].transferredSize, param.sizeArray[i]);
                }
                break;

            case 1: // in
                nn::os::ClearSystemEvent(inEvent);

                NNT_ASSERT_RESULT_SUCCESS(
                    g_EpIn.GetXferReport(&reportCount, report, param.xferCount)
                );

                ASSERT_EQ(reportCount, param.xferCount);

                for (int i = 0; i < reportCount; i++)
                {
                    NNT_ASSERT_RESULT_SUCCESS(report[i].result);
                    ASSERT_EQ(report[i].transferredSize, param.sizeArray[i]);
                }

                ASSERT_TRUE(
                    Fx3CheckGaloisPattern(g_RxBuffer[0], totalSize, inSeed, &inSeed)
                );
                break;

            default:
                NN_UNEXPECTED_DEFAULT;
                break;
            }
        }

    }

    NNT_ASSERT_RESULT_SUCCESS(g_EpOut.Finalize());
    NNT_ASSERT_RESULT_SUCCESS(g_EpIn.Finalize());

    NNT_ASSERT_RESULT_SUCCESS(g_Interface.Finalize());
    NNT_ASSERT_RESULT_SUCCESS(g_Client.Finalize());

    Fx3TearDown();
} // NOLINT(impl/function_size)

} // hs
} // usb
} // nnt
