﻿using System;
using System.Threading;
using System.Diagnostics;
using System.Management;
using System.IO.Ports;

namespace SerialPortExample
{
    enum ExitCode : int
    {
        Success,
        Failed,
    }

    class SerialPortExample
    {
        private const   int         OpenPortIdleDuration  = 120000; // miliseconds
        private const   int         Fx3AttachedTimeout    = 20;
        private const   string      SpecialExitPattern    = "Fx3MonExit";
        private const   string      FailedString          = "FAIL:";
        private const   string      Fx3VidPid             = "VID_04B4&PID_0007";
        private const   int         Fx3BaudRate           = 115200;
        private static  ExitCode    Fx3ExitCode           = ExitCode.Success;
        private static  SerialPort  Fx3Port               = null;

        public static int Main()
        {
            string instanceName = string.Empty;
            string portName = string.Empty;
            bool isFound = false;
            Thread readThread = new Thread(Read);

            // Get a list of serial port names.
            string[] ports = SerialPort.GetPortNames();

            Console.Write("Looking for FX3 Debug Monitor port");
            try
            {
                int comSearchCount = 0;
                // Find FX3 debug COM port
                while (!isFound)
                {
                    Console.Write(".");

                    ManagementObjectSearcher searcher = new ManagementObjectSearcher("root\\WMI", "SELECT * FROM MSSerial_PortName");
                    foreach (ManagementObject queryObj in searcher.Get())
                    {
                        instanceName = queryObj["InstanceName"].ToString();;
                        if (instanceName.Contains(Fx3VidPid))
                        {
                            portName = queryObj["PortName"].ToString();
                            Fx3Port = new SerialPort(portName, Fx3BaudRate, Parity.None, 8, StopBits.One);
                            Fx3Port.ReadTimeout = OpenPortIdleDuration;
                            isFound = true;
                            Console.WriteLine("\nFound FX3 monitor!");
                            break;
                        }
                    }

                    Thread.Sleep(1000);
                    if (comSearchCount > Fx3AttachedTimeout)
                    {
                        Fx3ExitCode = ExitCode.Failed;
                        return (int)Fx3ExitCode;
                    }

                    comSearchCount++;
                }
                Console.WriteLine();

                Fx3Port.Open();

                readThread.Start();

                readThread.Join();
            }
            catch(System.IO.IOException IOex)
            {
                Console.WriteLine("Device not found. Check FX3 device USB connection.");
                Console.WriteLine(IOex.Message);
            }
            catch(System.Exception ex)
            {
                Console.WriteLine(ex.Message);
            }
            finally
            {
                if (Fx3Port != null)
                {
                    Fx3Port.Close();
                    Console.WriteLine("*****Closing port.*****");
                }
            }

            return (int)Fx3ExitCode;
        }

        public static void Read()
        {
            bool isPortClosed = false;
            while (!isPortClosed)
            {
                try
                {
                    string dataLine = Fx3Port.ReadLine();
                    Console.WriteLine(dataLine);

                    // Close port if SpecialExitPattern is recieved for FX3 device.
                    if (dataLine.Contains(SpecialExitPattern))
                    {
                        Console.WriteLine("*****Special exit pattern ({0}) received. Closing port.*****", SpecialExitPattern);
                        isPortClosed = true;
                    }

                    if (dataLine.Contains(FailedString))
                    {
                        Fx3ExitCode = ExitCode.Failed;
                    }
                }
                catch (TimeoutException)
                {
                    Console.WriteLine("Reading timed out: {0}.\n", OpenPortIdleDuration);
                    Fx3ExitCode = ExitCode.Failed;
                    isPortClosed = true;
                }
                catch (System.IO.IOException IOex)
                {
                    Console.WriteLine("Device not found. Failed to receive data. Check FX3 device USB connection.");
                    Console.WriteLine(IOex.Message);
                    Fx3ExitCode = ExitCode.Failed;
                    isPortClosed = true;
                }

            }
        }
    }
}
