﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nnt.h>
#include <nn/nn_Log.h>
#include <nn/nn_Windows.h>
#include <cstdio>
#include <memory>
#include <tchar.h>
#include <winusb.h>
#pragma warning(disable: 4668)
#include <setupapi.h>
#include <vector>

#include "../../Common/UsbTestUtil.h"

// ドライバの GUID
const GUID TARGET_GUID = { 0x61A954B0, 0xB7F1, 0x4333,{ 0xAF, 0xBD, 0x6A, 0x41, 0x76, 0x9C, 0xE2, 0x66 } };

class StopWatch
{
public:
    void Start()
    {
        QueryPerformanceCounter(&m_Start);
    }
    void Stop()
    {
        QueryPerformanceCounter(&m_End);
    }

    int GetElapsedMicroseconds() const
    {
        LARGE_INTEGER freq;
        QueryPerformanceFrequency(&freq);
        return static_cast<int>((m_End.QuadPart - m_Start.QuadPart) * 1000 * 1000 / freq.QuadPart);
    }
private:
    LARGE_INTEGER m_Start;
    LARGE_INTEGER m_End;
};

class StackBuffer
{
private:
    void *m_Buffer;
    StackBuffer(const StackBuffer&);
    StackBuffer& operator=(const StackBuffer&);
public:
    explicit StackBuffer(ULONG size) : m_Buffer(NULL)
    {
        if ( size )
        {
            m_Buffer = malloc(size);
        }
    }
    ~StackBuffer()
    {
        if (m_Buffer)
        {
            free(m_Buffer);
        }
    }
    template<typename T> T GetBuffer()
    {
        return reinterpret_cast<T>(m_Buffer);
    }
};

struct UsbDeviceInfo
{
    HANDLE deviceHandle;
    WINUSB_INTERFACE_HANDLE winUsbHandle;
    USHORT interfaceIndex;
    unsigned char pipeBulkIn;
    unsigned char pipeBulkOut;
};

bool ProbeDevice(UsbDeviceInfo* pOutDeviceInfo)
{
    HANDLE deviceHandle = NULL;
    WINUSB_INTERFACE_HANDLE winUsbHandle;

    HDEVINFO deviceInfoHandle = SetupDiGetClassDevs(&TARGET_GUID, NULL, NULL, DIGCF_PRESENT | DIGCF_DEVICEINTERFACE);

    if (!deviceInfoHandle)
    {
        NN_LOG("[Host PC] device information was not found (%d).\n", GetLastError());
        return false;
    }

    SP_DEVICE_INTERFACE_DATA interfaceData = { 0 };
    interfaceData.cbSize = sizeof(SP_DEVICE_INTERFACE_DATA);

    DWORD dwIndex = 0;

    while (NN_STATIC_CONDITION(true))
    {
        if (SetupDiEnumDeviceInterfaces(deviceInfoHandle, NULL, &TARGET_GUID, dwIndex, &interfaceData))
        {
            break;
        }
        else
        {
            if (GetLastError() == ERROR_NO_MORE_ITEMS)
            {
                return false;
            }
        }

        dwIndex++;
    }

    ULONG interfaceDetailDataLength = 256;

    if (!SetupDiGetDeviceInterfaceDetail(deviceInfoHandle, &interfaceData, NULL, 0, &interfaceDetailDataLength,
        NULL))
    {
        if (GetLastError() != ERROR_INSUFFICIENT_BUFFER)
        {
            return false;
        }
    }

    ::std::unique_ptr<char[]> buffer(new char[interfaceDetailDataLength]);
    auto pInterfaceDetailData = reinterpret_cast<PSP_DEVICE_INTERFACE_DETAIL_DATA>(buffer.get());
    pInterfaceDetailData->cbSize = sizeof(SP_DEVICE_INTERFACE_DETAIL_DATA);

    if (!SetupDiGetDeviceInterfaceDetail(deviceInfoHandle, &interfaceData, pInterfaceDetailData,
        interfaceDetailDataLength, &interfaceDetailDataLength, NULL))
    {
        return false;
    }

    deviceHandle = CreateFile(pInterfaceDetailData->DevicePath, GENERIC_WRITE | GENERIC_READ,
        FILE_SHARE_WRITE | FILE_SHARE_READ, NULL, OPEN_EXISTING, FILE_ATTRIBUTE_NORMAL | FILE_FLAG_OVERLAPPED,
        NULL);

    if (deviceHandle == INVALID_HANDLE_VALUE)
    {
        return false;
    }

    if (!WinUsb_Initialize(deviceHandle, &winUsbHandle))
    {
        CloseHandle(deviceHandle);
        return false;
    }

    uint8_t pipeBulkIn = 0;
    uint8_t pipeBulkOut = 0;

    USB_INTERFACE_DESCRIPTOR interfaceDescriptor;
    WinUsb_QueryInterfaceSettings(winUsbHandle, 0, &interfaceDescriptor);
    for (int i = 0; i < interfaceDescriptor.bNumEndpoints; i++)
    {
        WINUSB_PIPE_INFORMATION pipeInfo;
        while (!WinUsb_QueryPipe(winUsbHandle, 0, static_cast<uint8_t>(i), &pipeInfo))
        {
            Sleep(10);
        }
        if (pipeInfo.PipeType == UsbdPipeTypeBulk && USB_ENDPOINT_DIRECTION_IN(pipeInfo.PipeId))
        {
            pipeBulkIn = pipeInfo.PipeId;
        }
        else if (pipeInfo.PipeType == UsbdPipeTypeBulk && USB_ENDPOINT_DIRECTION_OUT(pipeInfo.PipeId))
        {
            pipeBulkOut = pipeInfo.PipeId;
        }
    }

    if (!pipeBulkIn)
    {
        NN_LOG("[Host PC] BULK IN pipe was not found\n");
        return false;
    }

    if (!pipeBulkOut)
    {
        NN_LOG("[Host PC] BULK OUT pipe was not found\n");
        return false;
    }

    pOutDeviceInfo->interfaceIndex = interfaceDescriptor.bInterfaceNumber;
    pOutDeviceInfo->deviceHandle = deviceHandle;
    pOutDeviceInfo->winUsbHandle = winUsbHandle;
    pOutDeviceInfo->pipeBulkIn = pipeBulkIn;
    pOutDeviceInfo->pipeBulkOut = pipeBulkOut;

    return true;
}

uint32_t transferCycle = 1;

bool loop_back(UsbDeviceInfo* info, int size)
{
    NN_LOG("%s(%d)\n", __FUNCTION__, size);

    // prepare data
    std::vector<uint8_t> outBuffer(size);
    uint8_t* pOutBuffer = &outBuffer[0];
    MakeGaloisPattern(pOutBuffer, (uint32_t)size, transferCycle);
    if (!CheckGaloisPattern(pOutBuffer, (uint32_t)size, transferCycle))
    {
        NN_LOG("Error: OutBuffer data pattern check failed. Exit.");
        return false;
    }
    else
    {
        NN_LOG("transferCycle: %d\n", transferCycle);
    }

    std::vector<uint8_t> inBuffer(size, static_cast<unsigned char>(0xff));
    uint8_t* pInBuffer = &inBuffer[0];

    // send data
    StopWatch swSend;
    swSend.Start();
    ULONG lengthTransferred = 0;
    if (!WinUsb_WritePipe(info->winUsbHandle, info->pipeBulkOut, &outBuffer[0], size, &lengthTransferred, NULL))
    {
        NN_LOG("WinUsb_WritePipe failed (%d)", GetLastError());
        return false;
    }
    swSend.Stop();

    // recv data
    StopWatch swRecv;
    swRecv.Start();
    if (!WinUsb_ReadPipe(info->winUsbHandle, info->pipeBulkIn, &inBuffer[0], size, &lengthTransferred, NULL))
    {
        NN_LOG("WinUsb_ReadPipe failed (%d)", GetLastError());
        return false;
    }
    swRecv.Stop();

    // compare data
    bool isDataCorrect = CheckGaloisPattern(pInBuffer, (uint32_t)size, transferCycle++);
    if (!isDataCorrect)
    {
        NN_LOG("InBuffer data pattern check failed\n");
    }

    if (size < 4 * 1024 * 1024)
    {
        NN_LOG("[%02x", *pInBuffer);
        pInBuffer++;
        for (int i = 0; i < 16 && i < size; i++)
        {
            NN_LOG(", %02x", *pInBuffer);
            pInBuffer++;
        }
        NN_LOG("]\n");
    }

    {
        unsigned char *p_src = &outBuffer[0];
        unsigned char *p_dst = &inBuffer[0];
        int i;

        for (i = 0; i < size; i++)
        {
            if (*p_dst != *p_src)
            {
                //NN_LOG("error [%d] src %02x dst %02x\n", i, *p_src, *p_dst);
            }

            p_src++;
            p_dst++;
        }
    }

    // compute throughput
    {
        int argc    = nnt::GetHostArgc();
        char **argv = nnt::GetHostArgv();

        for (int i = 0; i < argc; i++)
        {
            NN_LOG("%d %s\n", i, argv[i]);
        }

        int usec_send = swSend.GetElapsedMicroseconds();
        int usec_recv = swRecv.GetElapsedMicroseconds();

        int usec_total = usec_send + usec_recv;

        NN_LOG("send %d bytes %d us %d Bps\n", size, usec_send, (size * (1000000 / usec_send)));
        NN_LOG("recv %d bytes %d us %d Bps\n", size, usec_recv, (size * (1000000 / usec_recv)));

        NN_LOG(
            "##teamcity[buildStatisticValue key='%s_Bandwidth(MB/s)_Loopback_%dB' value='%f']\n",
            argc >= 2 ? argv[1] : "",
            size,
            (float)(2 * size) / ((float)usec_total / (float)1000000) / (1024 * 1024)
            );
    }
    return isDataCorrect;
}

class UsbPcHostTest : public ::testing::TestWithParam<int>
{
private:
    UsbDeviceInfo info;
public:
    virtual void SetUp() override
    {
        NN_LOG("Searching for device");

        while (!ProbeDevice(&info))
        {
            Sleep(500);
            NN_LOG(".");
        }

        NN_LOG("\n");
    }
    virtual void TearDown() override
    {
        WinUsb_Free(info.winUsbHandle);
        CloseHandle(info.deviceHandle);
    }
    UsbDeviceInfo* GetDeviceInfo()
    {
        return &info;
    }
};

/*
    SIGLONTD-8148 - Test is DISABLED as of 8/31
    Windows tries to send 32 bytes then 0 when SDEV is connected to PC
    Might be Target Manager doing this automatically
    Possible Solution: Create unique VID/PID GUID for device stack test
*/
INSTANTIATE_TEST_CASE_P(BulkReadTests, UsbPcHostTest, testing::ValuesIn(testValues));

TEST_P(UsbPcHostTest, ConcurrentReadWrite)
{
    EXPECT_TRUE(loop_back(GetDeviceInfo(), GetParam()));
}

