﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/os.h>
#include <nn/nn_Log.h>
#include <nn/nn_TimeSpan.h>
#include <nn/nn_Result.h>
#include <nn/usb/usb_Device.h>
#include <nnt/gtest/gtest.h>

#include "../../Common/UsbDsInterface.h"
#include "../../Common/UsbTestUtil.h"

using namespace nn;
using namespace nn::usb;
using namespace nn::usb::test;

NN_USB_DMA_ALIGN uint8_t readData[4 * 1024 * 1024];
NN_USB_DMA_ALIGN uint8_t writeData[4 * 1024 * 1024];

uint32_t readTransferCycle = 1; // used to change data pattern every transfer

bool Read(size_t size, int interfaceNum)
{
    uint32_t bytesRead = 0;

    // prepare data
    MakeGaloisPattern(readData, (uint32_t)size, readTransferCycle++);

    NN_LOG("Started reading %d bytes from interface %d. \n", size, interfaceNum);
    //                                       (uint32_t *pOutBytesTransferred, void *buffer, uint32_t bytes)
    Result result = UsbDsInterface::m_EndPoints[interfaceNum][0].PostBuffer(&bytesRead, readData, size);
    if (result.IsSuccess())
    {
        NN_LOG("Succeed to read %d bytes\n", bytesRead);
        return true;
    }
    else
    {
        NN_LOG("Failed to read %d bytes\n", bytesRead);
        return false;
    }
}

uint32_t writeTransferCycle = 1; // used to change data pattern every transfer

bool Write(size_t size, int interfaceNum)
{
    uint32_t bytesWrote = 0;

    NN_LOG("Started writing %d bytes to interface %d. \n", size, interfaceNum);
    //                                       (uint32_t *pOutBytesTransferred, void *buffer, uint32_t bytes)
    Result result = UsbDsInterface::m_EndPoints[interfaceNum][1].PostBuffer(&bytesWrote, writeData, size);
    if (result.IsSuccess())
    {
        NN_LOG("Succeed to write %d bytes\n", bytesWrote);
        if(CheckGaloisPattern(writeData, (uint32_t)size, writeTransferCycle++))
        {
            return true;
        }
        else
        {
            NN_LOG("Write data pattern check failed.");
            return false;
        }
    }
    else
    {
        NN_LOG("Failed to write %d bytes\n", bytesWrote);
        return false;
    }
}

class UsbDeviceStackTest : public ::testing::TestWithParam<int>{};

INSTANTIATE_TEST_CASE_P(BulkReadTests, UsbDeviceStackTest, testing::ValuesIn(testValues));


TEST_P(UsbDeviceStackTest, BulkRead1)
{
    EXPECT_TRUE(Read(GetParam(), 0));
}

TEST_P(UsbDeviceStackTest, BulkRead2)
{
    EXPECT_TRUE(Read(GetParam(), 1));
}

TEST_P(UsbDeviceStackTest, BulkRead3)
{
    EXPECT_TRUE(Read(GetParam(), 2));
}

TEST_P(UsbDeviceStackTest, BulkRead4)
{
    EXPECT_TRUE(Read(GetParam(), 3));
}

TEST_P(UsbDeviceStackTest, BulkWrite1)
{
    EXPECT_TRUE(Write(GetParam(), 0));
}

TEST_P(UsbDeviceStackTest, BulkWrite2)
{
    EXPECT_TRUE(Write(GetParam(), 1));
}

TEST_P(UsbDeviceStackTest, BulkWrite3)
{
    EXPECT_TRUE(Write(GetParam(), 2));
}

TEST_P(UsbDeviceStackTest, BulkWrite4)
{
    EXPECT_TRUE(Write(GetParam(), 3));
}
