﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <string>
#include <sstream>
#include <iomanip>

#include <nn/nn_Assert.h>
#include <nn/nn_BitTypes.h>
#include <nn/nn_Log.h>
#include <nn/nn_Result.h>

#include <nn/os.h>
#include <nn/usb/pd/usb_PdResult.h>
#include <nn/usb/pd/usb_PdCradle.h>

#include "Draw.h"
#include "Input.h"
#include <nnt/usbPdUtil/testUsbPd_util.h>
#include "SceneGetCradleVdo.h"
#include "ToString.h"

namespace nnt { namespace usb { namespace pd {

    namespace {
        // 1フレームの時間 (msec)
        const int64_t FrameLength = 50;

        // API を呼ぶ間隔 (frame)
        const int64_t UpdateInterval = 20;

        bool g_IsLedOn = false;

        nn::TimeSpan g_ApiExecutionTime;
        nn::os::Tick g_SceneStart;

        nn::usb::pd::CradleSession g_Session;

        nn::Result g_SetCradleVdoResult;

        void Print() NN_NOEXCEPT
        {
            nnt::usb::pd::draw::Clear();
            std::stringstream ss;

            ss << "API Result: " << ToString(g_SetCradleVdoResult) << std::endl;
            ss << "API Execution Time: " << g_ApiExecutionTime.GetMicroSeconds() << " usec" << std::endl;
            ss << std::endl;

            ss << "LED: " << (g_IsLedOn ? "ON" : "OFF") << std::endl;
            ss << std::endl;

            float now = (nn::os::GetSystemTick() - g_SceneStart).ToTimeSpan().GetMilliSeconds() / 1000.0f;
            ss << "Time: " << std::fixed << std::setprecision(1) << now << std::endl;
            ss << std::endl;

            ss << "Press B key to go back." << std::endl;

            nnt::usb::pd::draw::Print(ss.str().c_str());
            nnt::usb::pd::draw::Draw();
        }

        void Update() NN_NOEXCEPT
        {
            nn::usb::pd::VdmLed vdmLed;
            if (g_IsLedOn)
            {
                vdmLed.Set<nn::usb::pd::VdmLed::Brightness>(128);
                vdmLed.Set<nn::usb::pd::VdmLed::OnTime>(50); // 点灯時間 (10ms 単位)
                vdmLed.Set<nn::usb::pd::VdmLed::OffTime>(0); // 消灯時間 (10ms 単位、0 以外を指定すると点滅)
                vdmLed.Set<nn::usb::pd::VdmLed::DelayTime>(0); // 点灯から消灯への移行時間 (10ms 単位)
            }
            else
            {
                vdmLed.Set<nn::usb::pd::VdmLed::Brightness>(0);
                vdmLed.Set<nn::usb::pd::VdmLed::OnTime>(0);
                vdmLed.Set<nn::usb::pd::VdmLed::OffTime>(0);
                vdmLed.Set<nn::usb::pd::VdmLed::DelayTime>(0);
            }

            auto start = nn::os::GetSystemTick();

            g_SetCradleVdoResult = nn::usb::pd::SetCradleVdo(&g_Session, &vdmLed, nn::usb::pd::CradleVdmCommand_Led);
            if (g_SetCradleVdoResult.IsFailure())
            {
                NN_LOG("[error] CradleVdmCommand_Led module:%d description:%d\n", g_SetCradleVdoResult.GetModule(), g_SetCradleVdoResult.GetDescription());
            }

            auto end = nn::os::GetSystemTick();
            g_ApiExecutionTime = (end - start).ToTimeSpan();
        }

    } // namespace

    void SceneTestLed(SceneResult *pOutResult)
    {
        g_SceneStart = nn::os::GetSystemTick();

        nn::usb::pd::OpenCradleSession( &g_Session );

        for(int64_t frame = 0;; frame++)
        {
            input::Update();
            Print();

            if (frame % UpdateInterval == 0)
            {
                // LED の点灯状態を変更
                g_IsLedOn = !g_IsLedOn;
                Update();
            }

            // キャンセルボタンが押されたらループを抜ける
            if (input::IsCancelPushed())
            {
                break;
            }

            nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(FrameLength));
        }

        nn::usb::pd::CloseCradleSession(&g_Session);

        pOutResult->nextSceneType = SceneType_Menu;
        pOutResult->nextSceneArg.menu.type = SceneMenuType_Top;
    }
}}}
