﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <sstream>

#include <nn/nn_Common.h>
#include <nn/os.h>

#include "Draw.h"
#include "Input.h"
#include "SceneMenu.h"
#include "ToString.h"

namespace nnt { namespace usb { namespace pd {
    namespace {
        // 1フレームの時間 (msec)
        const int64_t FrameLength = 50;

        std::vector<SceneResult> g_Results;
        std::vector<const char*> g_Names;

        // 状態取得 API テストの項目
        const State StateListForGetStatus[] = {
            State_NoConnection,
            State_OfficialAcAdaptor,
            State_AnkerMobileBattery,
            State_AnkerPowerPort,
            State_AppleCharger,
            State_GooglePixelCharger,
            State_GoogleDualPowerPort,
            State_RavpowerMobileBattery,
            State_RavpowerMobileBatteryAfterWakeByBatterySaving,
            State_CradleWithNoAcAdaptor,
            State_CradleWithOfficialAcAdaptor,
            State_CradleWithHdmiTvOn,
            State_CradleWithAppleCharger,
            State_CradleWithPc,
            State_CradleOverVoltage,
            State_CradleOverVoltageAfterContract,
            State_RelayBoxWithHdmiTvOn,

            // リレーボックスの異常系は重要性が低いので削除
            // State_RelayBoxWithNoAcAdaptor,
            // State_RelayBoxWithAppleCharger,
            // State_RelayBoxWithPc,

            State_TableDockWithNoAcAdaptor,
            State_TableDockWithHdmiTvOn,
            State_TableDockWithAppleCharger,
            State_TableDockWithGooglePixelCharger,
            State_TableDockWithPc,
            State_TableDockOverVoltage,
            State_TableDockOverVoltageAfterContract,

            State_Pc,
            State_Nx,
            State_OverVoltage,
            State_OverVoltageAfterContract,
            State_UsbEther,
            State_UsbEtherAfterSleepWithRavpowerMobileBattery,
        };

        // 状態遷移 API テストの項目
        // 状態取得 API テストから、NoticeEvent が発生しない項目を除いている
        const State StateListForGetNotice[] = {
            State_OfficialAcAdaptor,
            State_AppleCharger,
            State_CradleWithOfficialAcAdaptor,
            State_CradleWithHdmiTvOn,
            State_CradleWithAppleCharger,
            State_CradleWithPc,
            State_RelayBoxWithHdmiTvOn,
            State_Pc,
            State_OverVoltage,
            State_OverVoltageAfterContract,
        };

        // g_Results と g_Names の初期化
        void Init(SceneMenuType type)
        {
            g_Results.clear();
            g_Names.clear();

            switch(type)
            {
            case SceneMenuType_Top:
                {
                    SceneResult result;
                    result.nextSceneType = SceneType_Menu;
                    result.nextSceneArg.menu.type = SceneMenuType_GetStatus;

                    g_Results.push_back(result);
                    g_Names.push_back("Get status test");
                }

                {
                    SceneResult result;
                    result.nextSceneType = SceneType_Menu;
                    result.nextSceneArg.menu.type = SceneMenuType_GetNotice;

                    g_Results.push_back(result);
                    g_Names.push_back("Get notice test");
                }

                {
                    SceneResult result;
                    result.nextSceneType = SceneType_GetCradleVdo;

                    g_Results.push_back(result);
                    g_Names.push_back("Get cradle VDO test");
                }

                {
                    SceneResult result;
                    result.nextSceneType = SceneType_TestLed;

                    g_Results.push_back(result);
                    g_Names.push_back("Cradle LED test");
                }

                {
                    SceneResult result;
                    result.nextSceneType = SceneType_TestUsbHubReset;

                    g_Results.push_back(result);
                    g_Names.push_back("Cradle USB hub reset Test");
                }

                {
                    SceneResult result;
                    result.nextSceneType = SceneType_StateTransition;

                    g_Results.push_back(result);
                    g_Names.push_back("Multi-state transition Test");
                }
                break;
            case SceneMenuType_GetStatus:
                for (int i = 0; i < sizeof(StateListForGetStatus) / sizeof(State); i++)
                {
                    SceneResult result;
                    result.nextSceneType = SceneType_GetStatus;
                    result.nextSceneArg.getStatus.state = StateListForGetStatus[i];

                    g_Results.push_back(result);
                    g_Names.push_back(ToString(StateListForGetStatus[i]));
                }
                break;
            case SceneMenuType_GetNotice:
                for (int i = 0; i < sizeof(StateListForGetNotice) / sizeof(State); i++)
                {
                    SceneResult result;
                    result.nextSceneType = SceneType_GetNotice;
                    result.nextSceneArg.getNotice.state = StateListForGetNotice[i];

                    g_Results.push_back(result);
                    g_Names.push_back(ToString(StateListForGetNotice[i]));
                }
                break;
            default:
                NN_ABORT("Invalid SceneMenuType.\n");
            }
        }

        void Print(SceneMenuType type, int selectedIndex) NN_NOEXCEPT
        {
            nnt::usb::pd::draw::Clear();

            std::stringstream ss;

            // 表示文字数が多過ぎると nn::gfx::util::DebugFontTextWriter::Print() がハングするので調整する
            //（nnt::usb::pd::draw::Initialize() の NvMemorySize を増やしても改善せず）
            const int DisplayMax = 15;
            int start = 0;
            if ( selectedIndex >= DisplayMax )
            {
                start = selectedIndex - (DisplayMax - 1);
            }
            int end = start + DisplayMax;
            if ( end > g_Names.size() )
            {
                end = g_Names.size();
            }

            for (int i = start; i < end; i++)
            {
                if (i == selectedIndex)
                {
                    ss << "* ";
                }
                else
                {
                    ss << "  ";
                }

                ss << g_Names[i] << std::endl;
            }

            ss << std::endl;

            if (type != SceneMenuType_Top)
            {
                ss << "  " << selectedIndex + 1 << " / " << g_Names.size() << std::endl;
                ss << std::endl;
            }

            ss << "Press up/down button to move cursor, A key to choose." << std::endl;
            if (type != SceneMenuType_Top)
            {
                ss << "Remove all USB devices before starting tests." << std::endl;
            }
            ss << "Press B key to go back." << std::endl;

            nnt::usb::pd::draw::Print(ss.str().c_str());
            nnt::usb::pd::draw::Draw();
        }
    }

    void SceneMenu(SceneResult *pOutResult, const SceneMenuArg& arg)
    {
        Init(arg.type);

        // 妙なカーソル移動を防ぐために、シーン開始時に Update() を呼んでおく
        input::Update();

        int selectedIndex = 0;

        for(int64_t frame = 0;; frame++)
        {
            input::Update();
            Print(arg.type, selectedIndex);

            // 決定ボタンが押されたらループを抜ける
            if (input::IsOkPushed())
            {
                break;
            }

            // キャンセルボタンが押されたらループを抜ける
            if (input::IsCancelPushed())
            {
                break;
            }

            // カーソル移動
            if (input::IsUpPushed())
            {
                if (selectedIndex == 0)
                {
                    selectedIndex = g_Names.size() - 1;
                }
                else
                {
                    selectedIndex--;
                }
            }
            if (input::IsDownPushed())
            {
                if (selectedIndex == g_Names.size() - 1)
                {
                    selectedIndex = 0;
                }
                else
                {
                    selectedIndex++;
                }
            }

            nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(FrameLength));
        }

        *pOutResult = g_Results[selectedIndex];
        if (input::IsCancelPushed())
        {
            SceneResult result;
            result.nextSceneType = SceneType_Menu;
            result.nextSceneArg.menu.type = SceneMenuType_Top;
            *pOutResult = result;
        }
        return;
    }
}}}
