﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <string>
#include <sstream>
#include <iomanip>

#include <nn/nn_Assert.h>
#include <nn/nn_Result.h>

#include <nn/init.h>
#include <nn/os.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/usb/pd/usb_Pd.h>

#include "Draw.h"
#include "Input.h"
#include <nnt/usbPdUtil/testUsbPd_util.h>
#include "SceneGetStatus.h"
#include "ToString.h"

namespace nnt { namespace usb { namespace pd {

    namespace {
        // 1フレームの時間 (msec)
        const int64_t FrameLength = 50;

        nn::TimeSpan g_ApiExecutionTime;
        nn::os::Tick g_SceneStart;

        nn::usb::pd::Session g_PdSession;
        nn::usb::pd::Notice g_PdNotice;
        nn::usb::pd::Status g_PdStatus;

        State g_State;
        int g_NoticeCount;

        // g_PdNotice の値が適切かをテストする (挿入イベント用)
        bool TestPdNoticeForInsert() NN_NOEXCEPT
        {
            if (g_State == State_CradleWithPc ||
                g_State == State_CradleWithAppleCharger ||
                g_State == State_OverVoltage ||
                g_State == State_OverVoltageAfterContract ||
                g_State == State_CradleOverVoltage ||
                g_State == State_CradleOverVoltageAfterContract)
            {
                return g_PdNotice.IsErrorNotice();
            }
            else
            {
                return g_PdNotice.IsActiveNotice();
            }
        }

        // g_PdNotice の値が適切かをテストする (抜去イベント用)
        bool TestPdNoticeForRemove() NN_NOEXCEPT
        {
            if (g_State == State_OverVoltage ||
                g_State == State_OverVoltageAfterContract ||
                g_State == State_CradleOverVoltage ||
                g_State == State_CradleOverVoltageAfterContract)
            {
                return g_PdNotice.IsErrorNotice();
            }
            else
            {
                return g_PdNotice.IsActiveNotice();
            }
        }

        void Print() NN_NOEXCEPT
        {
            nnt::usb::pd::draw::Clear();
            std::stringstream ss;

            // ヘッダ表示
            ss << "Test Item: " << ToString(g_State) << std::endl;
            ss << "Notice Count: " << g_NoticeCount << std::endl;
            ss << std::endl;
            nnt::usb::pd::draw::Print(ss.str().c_str());
            ss.str("");

            if (g_NoticeCount > 0)
            {
                // テスト結果表示
                bool connected = g_PdStatus.IsActive();
                bool testResult = connected ? TestPdNoticeForInsert() : TestPdNoticeForRemove();
                if (testResult)
                {
                    nnt::usb::pd::draw::Print("Test Result: OK", nnt::usb::pd::draw::Color_Green);
                }
                else
                {
                    nnt::usb::pd::draw::Print("Test Result: NG", nnt::usb::pd::draw::Color_Red);
                }

                ss << "API Execution Time: " << g_ApiExecutionTime.GetMicroSeconds() << " usec" << std::endl;
                ss << std::endl;

                // 詳細表示
                ss << "IsActiveNotice: " << ToString(g_PdNotice.IsActiveNotice()) << std::endl;
                ss << "IsErrorNotice: " << ToString(g_PdNotice.IsErrorNotice()) << std::endl;
                ss << "IsDataRoleNotice: " << ToString(g_PdNotice.IsDataRoleNotice()) << std::endl;
                ss << "IsPowerRoleNotice: " << ToString(g_PdNotice.IsPowerRoleNotice()) << std::endl;
                ss << "IsDeviceNotice: " << ToString(g_PdNotice.IsDeviceNotice()) << std::endl;
                ss << "IsConsumerContractNotice: " << ToString(g_PdNotice.IsConsumerContractNotice()) << std::endl;
                ss << "IsProviderContractNotice: " << ToString(g_PdNotice.IsProviderContractNotice()) << std::endl;
                ss << "IsRequestNotice: " << ToString(g_PdNotice.IsRequestNotice()) << std::endl;
                ss << std::endl;
            }

            // フッタ表示
            float now = (nn::os::GetSystemTick() - g_SceneStart).ToTimeSpan().GetMilliSeconds() / 1000.0f;
            ss << "Time: " << std::fixed << std::setprecision(1) << now << std::endl;
            ss << std::endl;
            ss << "Press B key to go back." << std::endl;

            nnt::usb::pd::draw::Print(ss.str().c_str());
            nnt::usb::pd::draw::Draw();
        }

        void Update() NN_NOEXCEPT
        {
            // GetNotice
            auto start = nn::os::GetSystemTick();

            nn::usb::pd::GetNotice(&g_PdNotice, &g_PdSession);

            auto end = nn::os::GetSystemTick();
            g_ApiExecutionTime = (end - start).ToTimeSpan();

            // GetStatus
            nn::usb::pd::GetStatus(&g_PdStatus, &g_PdSession);
        }

    } // namespace

    void SceneGetNotice(SceneResult *pOutResult, const SceneGetNoticeArg& arg, nn::os::SystemEventType *pNoticeEvent)
    {
        g_SceneStart = nn::os::GetSystemTick();

        g_State = arg.state;

        g_NoticeCount = 0;

        nn::usb::pd::OpenSession(&g_PdSession);

        nn::usb::pd::BindNoticeEvent(pNoticeEvent, &g_PdSession);

        nn::os::ClearSystemEvent(pNoticeEvent);

        for(;;)
        {
            input::Update();
            Print();

            // Notice イベントが発生したら、テストを進める
            if (nn::os::TimedWaitSystemEvent(pNoticeEvent, nn::TimeSpan::FromMilliSeconds(FrameLength)))
            {
                nn::os::ClearSystemEvent(pNoticeEvent);
                Update();

                g_NoticeCount++;
            }

            // キャンセルボタンが押されたらループを抜ける
            if (input::IsCancelPushed())
            {
                break;
            }

            nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(FrameLength));
        }

        nn::usb::pd::UnbindNoticeEvent(&g_PdSession);

        nn::usb::pd::CloseSession(&g_PdSession);

        pOutResult->nextSceneType = SceneType_Menu;
        pOutResult->nextSceneArg.menu.type = SceneMenuType_GetNotice;
    }
}}}
