﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/init.h>
#include <nn/omm/omm_Api.h>
#include <nn/os.h>
#include <nn/psm/psm.h>
#include <nn/psm/psm_System.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/usb/pd/usb_Pd.h>
#include <nn/usb/pd/usb_PdCradle.h>
#include <nn/usb/usb_Host.h>
#include <nn/vi.private.h>

#include "Draw.h"
#include "Input.h"
#include "Scene.h"
#include "SceneGetStatus.h"
#include "SceneGetNotice.h"
#include "SceneGetCradleVdo.h"
#include "SceneMenu.h"
#include "SceneTestLed.h"
#include "SceneTestUsbHubReset.h"
#include "SceneStateTransition.h"

extern "C" void nninitStartup()
{
    const size_t MemoryHeapSize = 512 * 1024 * 1024;
    const size_t MallocHeapSize = 256 * 1024 * 1024;

    NN_ABORT_UNLESS_RESULT_SUCCESS(
        ::nn::os::SetMemoryHeapSize(MemoryHeapSize));

    uintptr_t address = uintptr_t();
    NN_ABORT_UNLESS_RESULT_SUCCESS(
        ::nn::os::AllocateMemoryBlock(&address, MallocHeapSize));

    ::nn::init::InitializeAllocator(
        reinterpret_cast<void*>(address), MallocHeapSize);
}

extern "C" void nnMain()
{
    // nn::usb::pd::GetNotice() 用イベント
    nn::os::SystemEventType noticeEvent;

    nn::omm::Initialize();

    nn::psm::Initialize();

    nn::vi::Initialize();

    nn::usb::pd::Initialize();
    nn::usb::pd::InitializeCradle();

    nn::usb::Host host;
    NN_ABORT_UNLESS_RESULT_SUCCESS(host.Initialize());

    nnt::usb::pd::draw::Initialize();
    nnt::usb::pd::input::Initialize();

    nnt::usb::pd::SceneResult result;
    nnt::usb::pd::SceneType type = nnt::usb::pd::SceneType_Menu;
    nnt::usb::pd::SceneArg arg;
    arg.menu.type = nnt::usb::pd::SceneMenuType_Top;

    for (;;)
    {
        switch(type)
        {
        case nnt::usb::pd::SceneType_GetNotice:
            nnt::usb::pd::SceneGetNotice(&result, arg.getNotice, &noticeEvent);
            break;
        case nnt::usb::pd::SceneType_GetStatus:
            nnt::usb::pd::SceneGetStatus(&result, arg.getStatus, &noticeEvent, &host);
            break;
        case nnt::usb::pd::SceneType_GetCradleVdo:
            nnt::usb::pd::SceneGetCradleVdo(&result);
            break;
        case nnt::usb::pd::SceneType_Menu:
            nnt::usb::pd::SceneMenu(&result, arg.menu);
            break;
        case nnt::usb::pd::SceneType_TestLed:
            nnt::usb::pd::SceneTestLed(&result);
            break;
        case nnt::usb::pd::SceneType_TestUsbHubReset:
            nnt::usb::pd::SceneTestUsbHubReset(&result, &host);
            break;
        case nnt::usb::pd::SceneType_StateTransition:
            nnt::usb::pd::SceneStateTransition(&result, &noticeEvent);
            break;
        default:
            NN_ABORT("Invalid scene type.");
        }

        type = result.nextSceneType;
        arg = result.nextSceneArg;
    }

    nnt::usb::pd::input::Finalize();
    nnt::usb::pd::draw::Finalize();

    host.Finalize();

    nn::usb::pd::FinalizeCradle();
    nn::usb::pd::Finalize();

    nn::vi::Finalize();

    nn::psm::Finalize();

    nn::omm::Finalize();
}
